#!/usr/bin/env python3

# Copyright (c) 2024 NETINT Technologies Inc.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

from pathlib import Path
from string import Template

import argparse
import getpass
import ipaddress
import json
import os
import re
import shlex
import shutil
import socket
import subprocess
import sys
import tarfile
import time
import urllib.request

description = "NETINT Bitstreams Edge Quick Installer Script v2.4.0"

docker = "docker"
empty = ""
etc = Path("/etc")
install = "install"
y = "y"


enable_iommu = False
passthrough_exists = True
iommu_exists = True
def is_iommu_enabled(machine):
   global iommu_exists  
   
   try:
       with open('/etc/default/grub', 'r') as f:
           grub_content = f.read()
           
       cmdline_linux = ""
       cmdline_default = ""
       any_iommu_param_found = False  
       
       for line in grub_content.splitlines():
           if line.startswith('GRUB_CMDLINE_LINUX='):
               cmdline_linux = line
           elif line.startswith('GRUB_CMDLINE_LINUX_DEFAULT='):
               cmdline_default = line
               
       combined_cmdline = cmdline_linux + cmdline_default
       
       def get_iommu_patterns(machine):
            if machine == "arm64":
                return {
                    "all": ["arm64.iommu=off", "arm-smmu.disable_bypass=0", "arm64.iommu=on", "arm-smmu.disable_bypass=1, iommu=on, iommu=off"],
                    "disable": ["arm64.iommu=off", "arm-smmu.disable_bypass=1, iommu=off"],
                    "enable": ["arm64.iommu=on", "arm-smmu.disable_bypass=0, iommu=on"]
                }
            return {
                "all": ["amd_iommu=off", "intel_iommu=off", "iommu=0", "iommu=off",
                        "amd_iommu=on", "intel_iommu=on", "iommu=1", "iommu=on"],
                "disable": ["amd_iommu=off", "intel_iommu=off", "iommu=0", "iommu=off"],
                "enable": ["amd_iommu=on", "intel_iommu=on", "iommu=1", "iommu=on"]
            }

       patterns = get_iommu_patterns(machine)
       any_iommu_param_found = any(p in combined_cmdline for p in patterns["all"])

       if any(p in combined_cmdline for p in patterns["disable"]):
            return False
       if any(p in combined_cmdline for p in patterns["enable"]):
            return True
       
       if not any_iommu_param_found:
           iommu_exists = False
           
       return False
       
   except Exception as e:
       print(f"Error reading grub file: {e}\n"
              "IOMMU status cannot be properly determined, please manually verify IOMMU is enabled")
       iommu_exists = False  
       return False


def is_iommu_passthrough_enabled(machine):
    global passthrough_exists
    try:
        with open('/etc/default/grub', 'r') as f:
            grub_content = f.read()
            
        cmdline_linux = ""
        cmdline_default = ""
        for line in grub_content.splitlines():
            if line.startswith('GRUB_CMDLINE_LINUX='):
                cmdline_linux = line
            elif line.startswith('GRUB_CMDLINE_LINUX_DEFAULT='):
                cmdline_default = line
                
        combined_cmdline = cmdline_linux + cmdline_default

        passthrough_strings = ["iommu.passthrough=1", "iommu=pt", "iommu.passthrough=on"]
        disable_strings = ["iommu.passthrough=0", "iommu.passthrough=off"]

        if any(s in combined_cmdline for s in disable_strings):
            return False
        if any(s in combined_cmdline for s in passthrough_strings):
            return True
        if not any(s in combined_cmdline for s in ["iommu.passthrough", "iommu=pt"]):
            passthrough_exists = False
            return False
                
        return False
        
    except Exception as e:
        print(f"Error reading grub file: {e}\n"
              "IOMMU passthrough status cannot be properly determined, please manually verify IOMMU passthrough is enabled")
        passthrough_exists = False
        return False


def get_current_boot_iommu_state(machine):
    try:
        with open('/proc/cmdline', 'r') as f:
            cmdline = f.read()
        
        if machine == "arm64":
            iommu_enabled = any(x in cmdline for x in ['arm64.iommu=on', 'arm-smmu.disable_bypass=0', 'iommu=1', 'iommu=on'])
            passthrough = any(x in cmdline for x in ['iommu=pt', 'iommu.passthrough=1', 'iommu.passthrough=on'])
        else:
            iommu_enabled = any(x in cmdline for x in ['intel_iommu=on', 'amd_iommu=on', 'iommu=1', 'iommu=on'])
            passthrough = any(x in cmdline for x in ['iommu=pt', 'iommu.passthrough=1', 'iommu.passthrough=on', 'intel_iommu=pt', "amd_iommu=pt"])
        
        return iommu_enabled and passthrough
    except Exception as e:
        print(f"Error reading cmdline file: {e}\n"
              "IOMMU and passthrough boot status cannot be properly determined")
        return False


def check_ports_availability():
    required_ports = {
        "ETCD": [2379, 2380],
        "Lms": [9094, 9095, 9101],
        "NVS": [9092, 9100],
        "Stream trans": [8057],
        "Gateway": [80, 443], 
        "Nginx": [8085],
        "MINIO": [9000, 9001],
        "Notification": [9096],
        "Mysql": [3306],
        "RabbitMQ": [5672, 15672, 25672, 61613, 61614, 1883],
        "Redis node": [6379],
        "Stream live": [1935, 8023, 8022, 10080],
    }

    def is_port_in_use(port: int) -> bool:
        """Checks if port is in use on both IPv4 and IPv6"""
        # Check IPv4
        try:
            with socket.socket(socket.AF_INET, socket.SOCK_STREAM) as sock:
                sock.settimeout(2)  # Add timeout to prevent hanging
                if sock.connect_ex(('127.0.0.1', port)) == 0:
                    return True
        except socket.error:
            pass
        # Check IPv6
        try:
            with socket.socket(socket.AF_INET6, socket.SOCK_STREAM) as sock:
                sock.settimeout(2)
                if sock.connect_ex(('::1', port)) == 0:
                    return True
        except socket.error:
            pass
        # Fall back to netstat
        if check_and_install("netstat"):
            try:
                result = run("netstat -tuln", subprocess.PIPE, True)
                return any(f":{port} " in line for line in result.stdout.splitlines())
            except subprocess.SubprocessError:
                pass
        return True  # Assume port is in use if check fails

    def get_process_info(port: int) -> str:
        """Gets process using port with multiple fallbacks"""
        # Try lsof first - most detailed info
        if check_and_install("lsof"):
            try:
                cmd = f"sudo lsof -i :{port}"
                result = run(cmd, stdout=subprocess.PIPE, universal_newlines=True)
                if result.stdout:
                    lines = result.stdout.strip().split('\n')
                    if len(lines) > 1:
                        process = lines[1].split()
                        if len(process) >= 2:
                            return f"{process[0]} (PID: {process[1]})"
            except (subprocess.SubprocessError, IndexError, subprocess.TimeoutExpired):
                pass
        # Try netstat as fallback
        if check_and_install("netstat"):
            try:
                result = subprocess.run(
                    f"sudo netstat -tulnp | grep ':{port} '",
                    shell=True,
                    check=True,
                    stdout=subprocess.PIPE,
                    universal_newlines=True
                )
                if result.stdout:
                    lines = result.stdout.strip().split('\n')
                    for line in lines:
                        parts = line.split()
                        if len(parts) >= 7:
                            return parts[6]
            except (subprocess.SubprocessError, IndexError, subprocess.TimeoutExpired):
                pass
        # Try ss as last resort
        if check_and_install("ss"):
            try:
                cmd = f"sudo ss -lptn sport = :{port}"
                result = subprocess.run(
                    cmd,
                    shell=True,
                    check=True,
                    stdout=subprocess.PIPE,
                    universal_newlines=True
                )
                if result.stdout:
                    lines = result.stdout.splitlines()
                    for line in lines:
                        if "users:" in line:
                            return line.split("users:")[-1].strip()
            except (subprocess.SubprocessError, IndexError, subprocess.TimeoutExpired):
                pass
                
        return "unknown process"

    conflicts = {}
    for service, ports in required_ports.items():
        for port in ports:
            if is_port_in_use(port):
                conflicts.setdefault(service, []).append(
                    (port, get_process_info(port))
                )

    if conflicts:
        print("\nWARNING: The following required ports are currently in use:")
        for service, port_info in conflicts.items():
            for port, process in port_info:
                print(f"{service} - Port {port} is in use by {process}")
        sys.exit("Please free these ports and run the installer again.")

    print("\nAll required ports are available.")


def check_docker_access():
    try:
        run("docker info", subprocess.PIPE, True)
        return True
    except subprocess.CalledProcessError:
        return False


def handle_docker_permissions():
    if check_docker_access():
        return True
        
    user = getpass.getuser()
    try:
        run("getent group docker", subprocess.PIPE, True)
        groups_output = run("groups", subprocess.PIPE, True).stdout
        if "docker" not in groups_output:
            print(f"\nAdding user {user} to Docker group...")
            run(f"sudo usermod -aG docker {user}")
            print("\nDocker access configured. Please log out, log back in, and rerun this script.")
            sys.exit(0)
    except subprocess.CalledProcessError:
        pass
        
    print("\nDocker access denied. Please ensure:")
    print("1. Docker is installed: sudo apt-get install docker-ce")
    print("2. Docker daemon is running: sudo systemctl start docker")
    print("3. You have logged out and back in after being added to docker group")
    sys.exit(1)


def check_and_install(*tools):
    for tool in tools:
        if shutil.which(tool):
            continue
        if action_or_package := executable_packages.get(tool, None):
            if callable(action_or_package):
                action_or_package()
                continue
            if yninput(f"Use apt-get to install {action_or_package}"):
                run(f"sudo apt-get install -y {action_or_package}")
                continue
        wprint(f"Missing {tool}! Skipping")
        return False
    return True


def check_scaling_governor(should_print=True):
    if should_print:
        print(
            f"\nChecking if scaling_governor is set to '{desired_value.rstrip()}' for all CPUs..."
        )
    for i in range(cpu_count):
        p = Path(f"/sys/devices/system/cpu/cpu{i}/cpufreq/scaling_governor")
        if not p.exists():
            print("WARNING: CPU settings cannot be set!")
            break
        with p.open() as cpu:
            if cpu.read() != desired_value:
                return False
    return True


def extract(f):
    print(f"Extracting {f}...")
    with tarfile.open(f) as tf:
        tf.extractall(release_directory)


def install_docker():
    # https://docs.docker.com/engine/install/ubuntu/#install-using-the-repository
    equals = "="
    id_key, ubuntu = "ID", "ubuntu"
    version_codename_key = "VERSION_CODENAME"
    linux = uname.sysname
    with open("/etc/os-release") as f:
        for line in f:
            items = line.rstrip().split(equals)
            if len(items) != 2:
                continue
            if items[0] == id_key and items[1] != ubuntu:
                wprint(
                    f"Skipped installing Docker due to unsupported {linux} distribution: {items[1]}"
                )
                return
            elif items[0] == version_codename_key:
                version_codename = items[1]
                break
    if not yninput("\nInstall Docker"):
        return
    print("Downloading and installing Docker (Uses sudo)")
    run("sudo apt-get update")
    run("sudo apt-get install -y ca-certificates")
    run(f"sudo {install} -m 0755 -d /etc/apt/keyrings")
    linux = linux.lower()
    gpg = etc / Path("apt/keyrings/docker.asc")
    if not gpg.exists():
        filename = None
        try:
            filename, _ = urllib.request.urlretrieve(
                f"https://download.docker.com/{linux}/{ubuntu}/gpg"
            )
        except Exception as e:
            wprint(f"Could not retrieve Docker's official GPG key: {e}")
        if filename:
            run(f"sudo {install} {filename} {gpg}")
            run(f"sudo chmod a+r {gpg}")
    local_apt_source_list = Path("docker.list")
    apt_source_list = etc.joinpath("apt/sources.list.d", local_apt_source_list)
    if not apt_source_list.exists():
        with local_apt_source_list.open("w") as f:
            f.write(
                f"""deb [arch={machine} signed-by=/etc/apt/keyrings/docker.asc] \
https://download.docker.com/{linux}/{ubuntu} \
{version_codename} stable"""
            )
        run(f"sudo {install} {local_apt_source_list} {apt_source_list}")
        local_apt_source_list.unlink()
    run("sudo apt-get update")
    run(
        "sudo apt-get install -y docker-ce docker-ce-cli containerd.io docker-buildx-plugin docker-compose-plugin"
    )
    if not check_docker_access():
        handle_docker_permissions()


def load_images(archive):
    for tar_archive in release_directory.glob("*.tar"):
        run(f"docker image load -i {tar_archive}")


def get_image_tags():
    image_tags = {}

    # Parse the output to extract image names and tags
    for line in run(
        "docker images --filter=reference='yyz1harbor01.netint.ca/bitstreams/*' --format '{{.Repository}}:{{.Tag}}'",
        subprocess.PIPE,
        True,
    ).stdout.split("\n"):
        if line:
            name_tag = line.split(":")
            image_name = name_tag[0].split("/")[-1]
            image_tag = name_tag[1]
            image_tags[image_name] = image_tag
    return image_tags


def run(s, stdout=None, universal_newlines=None, cwd=None):
    return subprocess.run(
        shlex.split(s),
        check=True,
        stdout=stdout,
        universal_newlines=universal_newlines,
        cwd=cwd,
    )


def stop_services():
    if not check_and_install(docker):
        return
    if not check_docker_access():
        handle_docker_permissions()
    container_ids = []
    try:
        for line in run("docker container ls", subprocess.PIPE, True).stdout.split("\n"):
            if "configs-" in line:
                container_ids.append(line.split()[0])
        if container_ids:
            print("\nStopping running Bitstreams containers...")
            run("docker stop " + " ".join(container_ids))
    except subprocess.CalledProcessError as e:
        print(f"Warning: Error stopping containers: {e}")


def wprint(s):
    print(f"WARNING: {s}!", file=sys.stderr)


def yninput(p):
    if args.y:
        return True
    return input(f"{p}? options: y/n [default: n] ") == y


def yninput2(p):
    if args.y:
        return True
    response = input(f"{p}? options: Y/n [default: Y] ").lower().strip()
    return response in ['y', 'yes', ''] or not response


executable_packages = {
    "docker": install_docker,
    "install": "coreutils",
    "nvme": "nvme-cli",
    "systool": "sysfsutils",
    "netstat": "net-tools",
    "lsof": "lsof",
    "ss": "iproute2"
}


def prepare_configs(image_tags):
    print("Image tags")
    for image_name, image_tag in image_tags.items():
        print(f"{image_name}:{image_tag}")

    try:
        dns = socket.gethostbyaddr(ipv4_address)[0].split(".")[0]
    except socket.herror as _:
        print("Warning: DNS lookup failed, falling back to IP address")
        dns = ipv4_address

    host_info = {
        "CURRENT_NODE_IP": ipv4_address,
        "Node_1_DNS": dns if args.use_dns_lookup else ipv4_address,
        "LMS_IMAGE_TAG": image_tags["lms"],
        "NVS_IMAGE_TAG": image_tags["nvs"],
        "STREAM_TRANS_IMAGE_TAG": image_tags["stream-trans"],
        "NOTIFICATION_IMAGE_TAG": image_tags["notification"],
        "STREAM_LIVE_SRS_IMAGE_TAG": image_tags["stream-live"],
        "GATEWAY_IMAGE_TAG": image_tags["gateway"],
        "EXT_PROTOCOL": args.protocol,
        "GATEWAY_PUBLIC_PORT": f":{args.gateway_port}" if args.gateway_port else empty,
        "SRT_PUBLIC_PORT": f"{args.srt_port}" if args.srt_port else empty,
        "RTMP_PUBLIC_PORT": f":{args.rtmp_port}" if args.rtmp_port else empty,
    }

    with open("non_cluster.json") as f:
        contents = f.read()
    with open("non_cluster_dirty.json", "w") as f:
        f.write(Template(contents).substitute(host_info))

    # Load the dictionary from non_cluster.json
    with open("non_cluster_dirty.json") as json_file:
        d = json.load(json_file)
    os.remove("non_cluster_dirty.json")

    for filepath in [
        ".env",
        "config/gateway/conf.d/api.conf",
        "config/gateway/lua/common_init.lua",
        "config/nginx/conf.d/default.conf",
        "config/stream-live/srs.conf",
        "config/etcd/config.yml",
        "config/lms/config.toml",
        "config/nvs/config.toml",
        "config/stream-trans/config.toml",
        "config/notification/config.toml",
    ]:
        file = configs_directory.joinpath(filepath)
        with file.open() as f:
            contents = f.read()
        with file.open("w") as f:
            f.write(Template(contents).substitute(d))

def T1M_configs():
    nvme = "nvme"
    if check_and_install(nvme):
        result = run(f"sudo nvme list", subprocess.PIPE, True)
        model_names = result.stdout.splitlines()

        quadra_t1m_exists = any(
            "QuadraT1M" in line for line in model_names if "Quadra" in line
        )
        # Reduce concurrent trans task count to 4 if any T1M is present
        if quadra_t1m_exists:
            # Modify the stream-trans config.toml file
            config_file = configs_directory.joinpath("config/stream-trans/config.toml")
            with config_file.open() as f:
                config_content = f.readlines()
            with config_file.open("w") as f:
                for config_line in config_content:
                    if "target_concurrent_trans_task_count" in config_line:
                        f.write("    target_concurrent_trans_task_count = 3\n")
                    else:
                        f.write(config_line)


uname = os.uname()
if uname.sysname != "Linux":
    sys.exit(f"{uname.sysname} not supported")

dpkg = "dpkg"
if check_and_install(dpkg):
    p = run(f"{dpkg} --print-architecture", subprocess.PIPE, True)
    machine = p.stdout.rstrip()
else:
    sys.exit(f"ERROR: {dpkg} not found!")

parser = argparse.ArgumentParser(description=description)
parser.epilog = (
   "Required ports:\n"
   "  ETCD: 2379, 2380\n" 
   "  Lms: 9094, 9095, 9101\n"
   "  NVS: 9092, 9100\n"
   "  Stream trans: 8057\n"
   "  Gateway: 80, 443\n"
   "  Nginx: 8085\n" 
   "  MINIO: 9000, 9001\n"
   "  Notification: 9096\n"
   "  Mysql: 3306\n"
   "  RabbitMQ: 5672, 15672, 25672, 61613, 61614, 1883\n"
   "  Redis node: 6379\n"
   "  Stream live 1935, 8023, 8022, 10080\n\n"
   "These ports must be available for Bitstreams services to start properly.\n"
   "Run: ./edge_quick_installer.py --help"
)
parser.add_argument(
    f"-{y}",
    action="store_true",
    help="answer yes to all yes/no prompts (supersedes all other arguments)",
)
parser.add_argument("ipv4Addr", type=ipaddress.IPv4Address)
for k, v in {"rtmp": 0, "srt": 10080, "gateway": 0}.items():
    parser.add_argument(
        f"--{k}-port", type=int, default=v, help=f"default: {v}. 0 means unspecified."
    )

for arg in [
    "overwrite-release-directory",
    "keep-release-directory",
    "firmware-upgrade",
    "skip-firmware-upgrade",
]:
    parser.add_argument(f"--{arg}", action="store_true")
parser.add_argument("--fail-if-no-iommu", action="store_true", 
                   help="Exit if IOMMU or passthrough not enabled instead of prompting")
parser.add_argument("protocol", choices=["http", "https"])
parser.add_argument("--use-dns-lookup", action="store_true", 
                   help="Use DNS lookup instead of IP address for region")

args = parser.parse_args()

print(description + "\n")


iommu_enabled = is_iommu_enabled(machine)
passthrough_enabled = is_iommu_passthrough_enabled(machine)
current_boot_enabled = get_current_boot_iommu_state(machine)

if not current_boot_enabled:
    if args.fail_if_no_iommu:
        issues = []
        if iommu_enabled and passthrough_enabled:
            sys.exit(f"ERROR: IOMMU and passthrough enabled but not active, need to manually reboot system for automated testing.")
        if not iommu_enabled:
            issues.append("IOMMU")
        if not passthrough_enabled:
            issues.append("passthrough")
        sys.exit(f"ERROR: {' and '.join(issues)} must be enabled for automated testing. Run script manually to enable them.")
    elif not iommu_enabled or not passthrough_enabled:
        if yninput2("IOMMU is not enabled. Enable IOMMU for improved performance"):
            enable_iommu = True
            print("Enabling IOMMU...\n")
            try:
                with open('/etc/default/grub', 'r') as f:
                    lines = f.readlines()
            except Exception as e:
                print("Error reading grub file, IOMMU and passthrough cannot be detected properly. Please enable both manually.")
                lines = None
                
            def update_params(line, enable_iommu=False, enable_passthrough=False):
                if not line.startswith(('GRUB_CMDLINE_LINUX=', 'GRUB_CMDLINE_LINUX_DEFAULT=')):
                    return line
                
                var_name, params = line.split('=', 1)
                params = params.strip().strip('"\'')
                
                new_params = []
                params_updated = False
                
                # Process existing parameters
                for param in params.split():
                    if machine == "arm64":
                        iommu_disable_params = {
                            'arm64.iommu=off': 'arm64.iommu=on',
                            'arm-smmu.disable_bypass=1': 'arm-smmu.disable_bypass=0', 
                            'iommu=0': 'iommu=1',
                            'iommu=off': 'iommu=on'
                        }
                        if enable_iommu and param in iommu_disable_params:
                            new_params.append(iommu_disable_params[param])
                            params_updated = True
                        elif enable_passthrough and param == 'iommu.passthrough=0':
                            new_params.append('iommu.passthrough=1')
                            params_updated = True
                        elif enable_passthrough and param == 'iommu.passthrough=off':
                            new_params.append('iommu.passthrough=on')
                            params_updated = True
                        else:
                            new_params.append(param)
                            
                    elif machine == "amd64":
                        if enable_iommu and param in ['amd_iommu=off', 'intel_iommu=off', 'iommu=0', 'iommu=off']:
                            new_params.append(param.replace('off','on').replace('0','1'))
                            params_updated = True
                        elif enable_passthrough and param == 'iommu.passthrough=0':
                            new_params.append('iommu.passthrough=1')
                            params_updated = True
                        elif enable_passthrough and param == 'iommu.passthrough=off':
                            new_params.append('iommu.passthrough=on')
                            params_updated = True
                        else:
                            new_params.append(param)
                            
                # Add missing parameters if needed
                if not params_updated:
                    if machine == "arm64":
                        if enable_iommu:
                            new_params.extend(['arm64.iommu=on', 'arm-smmu.disable_bypass=0'])
                        if enable_passthrough:
                            new_params.append('iommu.passthrough=1')
                    elif machine == "amd64":
                        if enable_iommu:
                            new_params.extend(['intel_iommu=on', 'amd_iommu=on', 'iommu=1', 'iommu=on'])
                        if enable_passthrough:
                            new_params.append('iommu=pt')

                # Add passthrough parameter if it doesn't exist and IOMMU is enabled
                if not passthrough_exists and enable_iommu and var_name == 'GRUB_CMDLINE_LINUX':
                    if machine == "arm64":
                        new_params.append('iommu.passthrough=1')
                    elif machine == "amd64":
                        new_params.append('iommu=pt')

                if not iommu_exists and enable_iommu and var_name == 'GRUB_CMDLINE_LINUX':
                    if machine == "arm64":
                        new_params.extend(['arm64.iommu=on', 'arm-smmu.disable_bypass=0'])
                    elif machine == "amd64":
                        new_params.extend(['intel_iommu=on', 'amd_iommu=on', 'iommu=1', 'iommu=on'])
                            
                return f'{var_name}="{" ".join(new_params)}"\n'

            if lines is not None:
                new_lines = [
                    update_params(line, 
                                enable_iommu=not iommu_enabled,
                                enable_passthrough=not passthrough_enabled)
                    for line in lines
                ]

                temp_grub = 'grub_temp'

                try:
                    with open(temp_grub, 'w') as f:
                        f.writelines(new_lines)
                    try:
                        run(f'sudo mv {temp_grub} /etc/default/grub')
                        run('sudo update-grub')
                    except subprocess.CalledProcessError as e:
                        print(f"Error updating GRUB configuration: {e}"
                            "IOMMU/passthrough not updated, please manually configure grub bootloader file to enable both")
                except IOError as e:
                    print(f"Error writing GRUB configuration: {e}\n"
                        "IOMMU/passthrough not updated, please manually configure grub bootloader file to enable both")

        else:
            print("Warning: Continuing without IOMMU. Server performance may be impacted")
            print("To enable IOMMU, either:\n"
                """1. manually edit /etc/default/grub, then run in terminal "sudo update-grub2" followed by "sudo reboot"\n"""
                "2. run this script again and select y to enable IOMMU")
    
    else:
        print("IOMMU and IOMMU passthrough enabled - pending reboot")
        print("Warning: IOMMU configuration exists but isn't active.")
        print("To enable IOMMU, either:\n"
              """1. Wait for reboot prompt at end of script\n"""
              """2. After script completion, run "sudo reboot" """)

else:
    print("IOMMU and IOMMU passthrough enabled and active")    

configs = "configs"
tar = ".tar"
tar_gz = tar + ".gz"

print("\nChecking for and stopping any running Bitstreams services...")
stop_services()
time.sleep(2)

print("\nChecking required ports availability...")
check_ports_availability()

configs_archive = Path(configs + tar)
if not configs_archive.exists():
    sys.exit("No configs archive found in current directory!")

version_pattern = "V(\d+).(\d+).(\d+)_\w+"
docker_images_regexp = re.compile(
    version_pattern + f"_docker_images_{machine}" + tar_gz
)
quadra_firmware_regexp = re.compile("Quadra_FW_" + version_pattern + tar_gz)

docker_images_archives = []
quadra_firmware_archives = []

for x in Path.cwd().iterdir():
    if n := docker_images_regexp.fullmatch(x.name):
        docker_images_archives.append(n)
    elif o := quadra_firmware_regexp.fullmatch(x.name):
        quadra_firmware_archives.append(o)

if not docker_images_archives:
    sys.exit("No docker images package archive found in current directory!")

docker_images_archives.sort(key=lambda m: m.string, reverse=True)
print("\nSelecting most recent versions of the package archives...")
docker_images_archive = docker_images_archives[0].string
print(f"Selected {docker_images_archive}")


release_directory = Path("release")
configs_directory = release_directory.joinpath(configs)

if release_directory.is_dir():
    if args.y or args.overwrite_release_directory:
        response = y
    elif args.keep_release_directory:
        response = empty
    else:
        response = (
            y if yninput(f"Overwrite the {release_directory} directory") else empty
        )
    if response == y:
        run(f"sudo rm -rf {release_directory}")
        extract(configs_archive)
        extract(docker_images_archive)
    elif configs_directory.exists():
        run(f"sudo rm -rf {configs_directory}")
        extract(configs_archive)
else:
    extract(configs_archive)
    extract(docker_images_archive)

ipv4_address = str(args.ipv4Addr)
if args.protocol == "https":
    run(f"bash gencert.sh {ipv4_address}", cwd=configs_directory.joinpath("cert"))

quadra_upgrade_script = None
if quadra_firmware_archives:
    if args.y or args.firmware_upgrade:
        response = y
    elif args.skip_firmware_upgrade:
        response = empty
    else:
        response = y if yninput("\nPerform firmware upgrade (Uses sudo)") else empty
    if response == y:
        quadra_firmware_archives.sort(key=lambda m: m.string, reverse=True)
        nvme = "nvme"
        if check_and_install(nvme):
            print("\nSelecting most recent version of the Quadra release archive...")
            quadra_firmware_archive, quadra_firmware_archive_version = (
                quadra_firmware_archives[0].string,
                quadra_firmware_archives[0].groups(),
            )
            print(f"Selected {quadra_firmware_archive}")
            extract(quadra_firmware_archive)
            quadra_firmware_folder_prefix = (
                f"Quadra_FW_V{'.'.join(quadra_firmware_archive_version)}"
            )
            for x in release_directory.iterdir():
                if x.name.startswith(quadra_firmware_folder_prefix):
                    quadra_upgrade_script = x.joinpath("quadra_auto_upgrade.sh")
                    run(f"sudo {quadra_upgrade_script.resolve()} -y")
                    break

ni_rsrc_update = "ni_rsrc_update"
if check_and_install(ni_rsrc_update):
    print("\nUninitializing NETINT Quadra devices... (Uses sudo)")
    run(f"sudo {ni_rsrc_update} -D")

cpu_count = os.cpu_count()
desired_value = "performance\n"
systemctl = "systemctl"

if not check_scaling_governor():
    systool = "systool"
    if check_and_install(systool, systemctl, install):
        print(
            f"Setting scaling_governor to '{desired_value.rstrip()}' for all CPUs. (Uses sudo)"
        )
        scaling_governor_regexp = re.compile(
            "devices/system/cpu/cpu\d+/cpufreq/scaling_governor"
        )
        local_sysfs_configuration_file = Path("sysfs.conf")
        sysfs_configuration_file = etc / local_sysfs_configuration_file
        with sysfs_configuration_file.open() as infile:
            with local_sysfs_configuration_file.open("w") as outfile:
                for line in infile:
                    if not scaling_governor_regexp.match(line):
                        outfile.write(line)
                for i in range(cpu_count):
                    outfile.write(
                        f"devices/system/cpu/cpu{i}/cpufreq/scaling_governor = "
                        + desired_value
                    )
        run(
            f"sudo {install} {local_sysfs_configuration_file} {sysfs_configuration_file}"
        )
        local_sysfs_configuration_file.unlink()
        run(f"sudo {systemctl} restart sysfsutils")
        if not check_scaling_governor(False):
            wprint("scaling_governor for all CPUs not set properly")

local_pam_limits_configuration_file = Path("limits.conf")
pam_limits_configuration_file = (
    etc / Path("security") / local_pam_limits_configuration_file
)
if check_and_install(install):
    print(f"\nUpdating {pam_limits_configuration_file} (Uses sudo)")
    octothorpe = "#"
    eof = "# End of file\n"
    wildcard, core, nofile = "*", "core", "nofile"
    with pam_limits_configuration_file.open() as infile:
        with local_pam_limits_configuration_file.open("w") as outfile:
            for line in infile:
                if line == eof:
                    continue
                elif line.startswith(octothorpe):
                    outfile.write(line)
                else:
                    fields = line.split()
                    if not len(fields) == 4 or not (
                        fields[0] == wildcard
                        and (fields[2] == nofile or fields[2] == core)
                    ):
                        outfile.write(line)
            outfile.write("* - nofile 65535\n* - core unlimited\n\n")
            outfile.write(eof)
    run(
        f"sudo {install} {local_pam_limits_configuration_file} {pam_limits_configuration_file}"
    )
    local_pam_limits_configuration_file.unlink()

sysctl = "sysctl"
local_sysctl_configuration_file = Path(sysctl + ".conf")
sysctl_configuration_file = etc / local_sysctl_configuration_file

if check_and_install(sysctl, install):
    print("\nConfiguring kernel parameters (Uses sudo)")
    octothorpe, semicolon = "#", ";"
    mem_default, mem_max = 16777216, 167772160
    token_values = {
        "vm.overcommit_memory": 1,
        "net.core.somaxconn": 32768,
        "net.core.rmem_max": mem_max,
        "net.core.rmem_default": mem_default,
        "net.core.wmem_max": mem_max,
        "net.core.wmem_default": mem_default,
        "net.core.netdev_max_backlog": 2000,
        "net.ipv4.tcp_rmem": "4096 262144 213675200",
        "net.ipv4.tcp_wmem": "4096 24576  213675200",
        "net.ipv4.tcp_max_syn_backlog": 8192,
        "kernel.core_pattern": "/var/core/core-%e-%p-%t",
    }
    with sysctl_configuration_file.open() as infile:
        with local_sysctl_configuration_file.open("w") as outfile:
            for line in infile:
                if line.startswith(octothorpe) or line.startswith(semicolon):
                    outfile.write(line)
                else:
                    should_write = True
                    for token in token_values.keys():
                        if token in line:
                            should_write = False
                            break
                    if should_write:
                        outfile.write(line)
            for k, v in token_values.items():
                outfile.write(f"{k} = {v}\n")
    run(f"sudo {install} {local_sysctl_configuration_file} {sysctl_configuration_file}")
    local_sysctl_configuration_file.unlink()
    core = Path("/var/core")
    if not core.is_dir():
        run(f"sudo mkdir {core}")
    run(f"sudo chmod 777 {core}")
    run(f"sudo {sysctl} -p")

if check_and_install(systemctl, install):
    print("\nDisabling Ubuntu apport (Uses sudo)")
    run(f"sudo {systemctl} stop apport")
    run(f"sudo {systemctl} disable apport")
    local_apport_configuration_file = Path("apport")
    apport_configuration_file = etc / Path("default") / local_apport_configuration_file
    enabled = "enabled"
    with apport_configuration_file.open() as infile:
        with local_apport_configuration_file.open("w") as outfile:
            for line in infile:
                if not line.startswith(enabled):
                    outfile.write(line)
            outfile.write("enabled=0\n")
    run(f"sudo {install} {local_apport_configuration_file} {apport_configuration_file}")
    local_apport_configuration_file.unlink()

for service in ["etcd", "mysql", "minio"]:
    print("\nCreating {} directory (Uses sudo)".format(service))
    run("sudo mkdir -p /var/bitstreams2/{}".format(service))
    run("sudo chmod 777 /var/bitstreams2/{}".format(service))

file = Path("/var/bitstreams2/mysql/grastate.dat")
print("\nCleaning {}".format(file))
if file.exists():
    run(f"sudo rm {file}")

gpasswd = "gpasswd"
groups = "groups"

if check_and_install(systemctl, docker, install, gpasswd, groups):
    print("\nConfiguring the Docker daemon (Uses sudo)")
    local_daemon_configuration_file = Path("daemon.json")
    daemon_configuration_file = etc / Path(docker) / local_daemon_configuration_file
    log_driver, log_driver_value = "log-driver", "json-file"
    log_opts, log_opts_value = "log-opts", {"max-size": "500m", "max-file": "10"}
    should_write = False
    if daemon_configuration_file.exists():
        with daemon_configuration_file.open() as f:
            j = json.load(f)
            if log_driver in j:
                if j[log_driver] != log_driver_value:
                    should_write = True
                    j[log_driver] = log_driver_value
            else:
                should_write = True
                j[log_driver] = log_driver_value
            if log_opts in j:
                if j[log_opts] != log_opts_value:
                    should_write = True
                    j[log_opts] = log_opts_value
            else:
                should_write = True
                j[log_opts] = log_opts_value
    else:
        should_write = True
        j = {log_driver: log_driver_value, log_opts: log_opts_value}
    if should_write:
        with local_daemon_configuration_file.open("w") as f:
            json.dump(j, f)
        run(
            f"sudo {install} {local_daemon_configuration_file} {daemon_configuration_file}"
        )
        local_daemon_configuration_file.unlink()
        run(f"sudo {systemctl} restart docker")
    run(f"sudo {systemctl} enable docker")
    p = run(groups, subprocess.PIPE, True)
    if docker not in p.stdout:
        run(f"sudo {gpasswd} -a {getpass.getuser()} docker")
        sys.exit("Please log out and log back in and then rerun this script.")

stop_services()

if check_and_install(docker):
    container_ids = ""
    for line in run("docker container ls", subprocess.PIPE, True).stdout.split("\n"):
        if "bitstream" in line:
            container_ids += f" {line.split()[0]}"
    if container_ids:
        print("\nRemoving existing Bitstreams containers...")
        run(f"docker container rm -f{container_ids}")
    image_ids = ""
    for line in run("docker image list", subprocess.PIPE, True).stdout.split("\n"):
        if "bitstream" in line:
            image_ids += f" {line.split()[2]}"
    if image_ids:
        print("\nRemoving existing Bitstreams images")
        run(f"docker image rm -f{image_ids}")
    print("\nLoading new Bitstreams images")
    load_images(docker_images_archive)

    image_tags = get_image_tags()
    prepare_configs(image_tags)
    T1M_configs()
    print("\nStarting Bitstreams services...")
    try:
        run("docker compose up -d --pull never", cwd=configs_directory)
    except subprocess.CalledProcessError:
        sys.exit("Failed to start Bitstreams services...")

if args.fail_if_no_iommu:
    print("\nCompleted")
elif enable_iommu or (iommu_enabled and passthrough_enabled and not current_boot_enabled):
    if yninput2("Installation completed. Reboot system to complete IOMMU configuration (recommended)"):
        print("Rebooting system...")
        run("sudo reboot")
    else:
        print("""Warning: IOMMU settings will not take effect until system is rebooted.\nTo reboot, please run: "sudo reboot" """)
else:
    print("\nCompleted.")
