#!/usr/bin/env bash

# Copyright (C) 2022 NETINT Technologies
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

SCRIPT_VERSION="v4.15"

# Global variables and default settings
SCRIPT_PATH=$(cd -- "$(dirname -- "${BASH_SOURCE[0]}")" 2>&1 > /dev/null; pwd)
CUSTOM_PATH="${SCRIPT_PATH}/FL_BIN"           # Arg for --path
TMP_WORKSPACE="/tmp"     # Path to folder for temporary files (/tmp/)
UPGRADE_LOG="${TMP_WORKSPACE}/netint_fw_upgrade_log.txt"  # Path to background upgrade processes (cold_upgrade(), unified_upgrade(), unified_upgrade_check()) log
FLAG_ERROR=false         # Error occured
FLAG_COLD_UPGRADE=false  # Prefer cold upgrade over warm upgrade
FLAG_ASSUME_YES=false    # Assume Yes to prompt
FLAG_SKIP_SAME_VER_UPGRADES=false # Skip upgrade for cards with same FWrev as FW binary file's FWrev
FLAG_UPGRADE_REINIT=false # Attempt to handle ni_rsrc re-initialization for unified upgraded cards
FLAG_LOG_TO_DMESG=false  # Write some high level logs to dmesg
FLAG_REBOOT_REQ_EXIT_CODE=false # Use exit code 3 when reboot is required no matter if expected
REBOOT_REQ=false         # Reboot required after script exit
FLAG_AVOID_SUDO=false    # Avoid using sudo
SUDO="sudo "             # Will be "sudo " if user is not root, else ""
MODEL_PATT="*"           # Filter cards to upgrade by model number based on this wildcard pattern
NI_RSRC_PERMISSION_PREFIX="" # Will be "sudo " if user does not have write access to /dev/shm/NI* files; else ""
PERSISTENT_CONFIG_TYPE="" # Allows user to override selection of auto-selected persistent config
if [ -t 0 ]; then
    TTY_SETTINGS=$(stty -g)  # Save initial tty settings to be recovered later to workaround Ubuntu 22 bug #1992025
fi

# Global arrays holding card info and sharing same index order
# Below associative arrays are all indexed by NVMe device path (eg. /dev/nvme0)
declare -a DEVICES=()        # List of NVMe device paths (eg. /dev/nvme0)
declare -A NODES=()          # List of NVMe block paths (eg. /dev/nvme0n1)
declare -A SERIAL_NUMS=()    # Card serial numbers
declare -A UPGRADE_TYPE=()   # Single letters for upgrade type to use: c-cold, s-skip, u-unified
declare -A MODEL_NUMS=()     # Card model numbers
declare -A FW_REVISION=()    # Card FW revs
declare -A FW_BIN=()         # Path to FW binary to use for card's upgrade
declare -A FW_BIN_FWREV=()   # FW binary's 8 character FW rev
declare -A UPGRADE_RESULT=() # Integers for result of upgrade: -1-skip, 0-good, 1- reset_error, 2-download_error, 3-burn_error, 4-activation_error, 5-fw_rev_check_error

# Configure variables for terminal color text
function setup_terminal_colors() {
    if [[ $SHELL =~ .*zsh ]]; then
        cRst="\x1b[0m"
        cRed="\x1b[31m"
        cGrn="\x1b[32m"
        cYlw="\x1b[33m"
        cBlu="\x1b[34m"
        cMag="\x1b[35m"
        cCyn="\x1b[36m"
    else
        cRst="\e[0m"
        cRed="\e[31m"
        cGrn="\e[32m"
        cYlw="\e[33m"
        cBlu="\e[34m"
        cMag="\e[35m"
        cCyn="\e[36m"
    fi
}

# Write a log message to multiple places at same time. Note, writing to dmesg
# only allowed if $FLAG_LOG_TO_DMESG is true
# $1 - log message text
# $2 - bitmask for log destination (0:stdout, 1:stderr, 2:dmesg, 3:upgrade_log.txt)
function log() {
    LOG_DEST=$2
    if (( (LOG_DEST & 0x1) == 0x1 )); then
        echo -e "$1"
    fi
    if (( (LOG_DEST & 0x2) == 0x2 )); then
        echo -e "$1" 1>&2
    fi
    if $FLAG_LOG_TO_DMESG && (( ((LOG_DEST & 0x4) == 0x4) || ((LOG_DEST & 0x1) == 0x1) || ((LOG_DEST & 0x2) == 0x2) )); then
        # Note: on some server/OS(ubuntu22) messages containing color codes are mistakenly converted by /dev/kmsg to zsh color codes on write. For now, just remove color codes from messages going to kmsg
        regex_workspace=$1
        regex='(.*)\\e\[[0-9;]+m(.*)'
        while [[ $regex_workspace =~ $regex ]]; do
          regex_workspace=${BASH_REMATCH[1]}${BASH_REMATCH[2]}
        done
        ${SUDO}sh -c "echo 'quadra_auto_upgrade.sh: ${regex_workspace}' > /dev/kmsg"
    fi
    # if $FLAG_LOG_TO_DMESG && (( (LOG_DEST & 0x4) == 0x4 )); then
        # ${SUDO}sh -c "echo 'quadra_auto_upgrade.sh: ${1}' > /dev/kmsg"
    # fi
    if (( (LOG_DEST & 0x8) == 0x8 )); then
        # Note: multiple processes will write to same $UPGRADE_LOG concurrently. This may have intra-line interleaving on NFS/HDFS
        echo -e "$1" >> $UPGRADE_LOG
    fi
}

# Prints usage information for the shell script
# $1 - if true, print full help text
function print_help_text() {
    echo "-------- quadra_auto_upgrade.sh --------"
    echo "Perform firmware upgrade to Quadra cards on system."
    echo "This script will automatically detect Quadra cards and select upgrade method."
    echo ""
    echo "Usage: ./quadra_auto_upgrade.sh [OPTION]"
    echo "Example: ./quadra_auto_upgrade.sh -p FL_BIN/nor_flash_v2.2.0_RC1.bin"
    echo ""
    echo "Options:"
    echo "-p, --path=PATH"
    echo "    Set PATH to firmware binary file (eg. nor_flash.bin) to use for upgrade; or,"
    echo "    set PATH to folder containing official release firmware binary files with"
    echo "    official name format (nor_flash_v*.*.*-Quadra*.bin) where file with largest"
    echo "    release version will be auto-selected."
    echo "    (Default: <quadra_auto_upgrade.sh path>/FL_BIN/)"
    echo "-c, --cold"
    echo "    Prefer cold upgrade to warm upgrade. If this option is not selected, warm"
    echo "    upgrade will be used when firmware loader component is same between firmware"
    echo "    binary file and card's current firmware."
    echo "-y, --yes"
    echo "    Automatically answer yes to all prompts. Requires password-less sudo access."
    echo "--skip_same_ver_upgrades"
    echo "    Skip upgrade for cards which have same firmware revision and firmware loader"
    echo "    component as the firmware binary file to use for firmware upgrade."
    echo "--reinit_after_upgrade"
    echo "    Handle ni_rsrc re-initialization (init_rsrc) for upgraded cards."
    echo "    Note, ni_rsrc_mon and ni_rsrc_update from compatible version of Libxcoder"
    echo "    must be installed on system."
    echo "--log_to_dmesg"
    echo "    Write log messages going to stdout/stderr to dmesg as well"
    echo "--reboot_req_exit_code"
    echo "    Use exit code 3 when reboot is required after cold-upgrade, or when error"
    echo "    occurs during upgrade FW activation."
    echo "-s, --avoid_sudo"
    echo "    Do not use sudo. Default behavior will use sudo unless user is 'root'."
    echo "-m, --model=PATTERN"
    echo "    Filter cards to upgrade by their model number based on wildcard PATTERN."
    echo "    (Default: *)"
    echo "-h, --help"
    echo "    Display short help text and exit."
    echo "--help_full"
    echo "    Display full text and exit."
    echo "-v, --version"
    echo "    Output version information and exit."
    echo "--persistent_config=CONFIG_TYPE"
    echo "    Set persistent configuration type to use (eg. 'A', 'B', or 'C') when "
    echo "    auto-selecting firmware binary."
    if $1; then
        echo ""
        echo "Description:"
        echo "quadra_auto_upgrade.sh performs automated FW upgrade for Netint Quadra hardware."
        echo "It detects availabe Quadra devices on system, can automatically select a viable"
        echo "release binary, perform FW upgrades, then produce a summary table of results."
        echo ""
        echo "FW upgrade is prevented when Netint device is running video operations as it"
        echo "could interfere with FW upgrade success."
        echo ""
        echo "The core FW upgrade process involves a few steps:"
        echo "    1. Reset:    Perform initial controller reset to ensure good state."
        echo "    2. Download: Write FW binary file to Netint device's DDR memory."
        echo "    3. Burn:     Instruct the Netint device to move the downloaded FW binary to"
        echo "                 NOR flash memory and use it after next reset. If burn fails,"
        echo "                 the previous working FW on the Netint device will still be in"
        echo "                 effect."
        echo "    4. Activate: Reset to activate the new FW binary. For cold upgrade the reset"
        echo "                 will be to power-cycle the card (hotplug, host reboot, etc.)."
        echo "                 User will need to trigger the power-cycle. For normal upgrade the"
        echo "                 reset will be an NVMe controller reset issued by this script."
        echo "    5. Check:    After upgrade, check new FW is being run. This script will"
        echo "                 check result of normal upgrade; but not cold upgrade, as its"
        echo "                 activation involves host reboot."
        echo "The core FW upgrade process uses the nvme-cli application to run its steps. The"
        echo "nvme-cli application requires root permissions (use of sudo) as it interacts"
        echo "with the kernel's NVMe driver."
        echo ""
        echo "Trouble-shooting:"
        echo ""
        echo "If FW burn fails due to 'firmware image specified for activation is invalid'"
        echo "this means that the signature of the FW binary does not match its contents,"
        echo "possibly due to file corruption. Try re-downloading the FW release from Netint."
        echo ""
        echo "If no Netint devices are detected in ni_rsrc_mon after FW upgrade, check"
        echo "ownership of Netint shared memory files (/dev/shm/NI*) and run ni_rsrc_mon with"
        echo "sudo if the shared memory files are owned by root user."
        echo "This script can be configured to re-init the shared memory files after normal"
        echo "upgrade. This will result in root user's ownership of the shared memory files if"
        echo "root user owned the shared memory files at the start of quadra_auto_upgrade.sh."
        echo "It is possible to change owner of shared memory files with 'chown'+'chgrp' or"
        echo "re-initializion: 'sudo ni_rsrc_update -D; init_rsrc'"
        echo ""
        echo "If no Netint devices are detected by quadra_auto_upgrade.sh check that the"
        echo "kernel's NVMe driver is loaded. On Linux, run 'sudo lsmod | grep nvme' and check"
        echo "for 'nvme' and 'nvme_core'. Use 'sudo modprobe -a nvme_core nvme' to start nvme"
        echo "driver."
    fi
    return 0
}

# parse a flag with an arg in or after it
# $1 flag pattern, $2 entire flag arg, $3 arg after flag arg
# return 1 if path is in second arg (separated by space), else return 0. Store path in $extract_arg_ret
function extract_arg() {
    unset extract_arg_ret
    # check valid arg flag
    if [ -n "$(printf "%s" ${2} | grep -Eoh "${1}")" ]; then
        # check if path string is connected by '=' or is in following arg
        if [ -n "$(echo "${2}" | grep -Eoh "${1}=")" ]; then
            arg_str=`printf "%s" "${2}" | grep -Poh "${1}=\K.+"`;
            # trim out leading and trailing quotation marks
            extract_arg_ret=`echo "${arg_str}" | sed -e 's/^\(["'\'']\)//' -e 's/\(["'\'']\)$//'`;
            return 0;
        elif [ -n "$(printf "%s" ${2} | grep -Eoh "^${1}$")" ]; then
            arg_str="${3}";
            # trim out leading and trailing quotation marks
            extract_arg_ret=`printf "%s" "${arg_str}" | sed -e 's/^\(["'\'']\)//' -e 's/\(["'\'']\)$//'`;
            return 1;
        else
            echo "Unknown option '$2', exiting";
            exit 1;
        fi
    else
        echo "Target flag '$1' not found in '$2', exiting"; exit 1;
    fi
}

# Parse Command line arguments
function parse_command_line_args() {
    while [ "$1" != "" ]; do
        case $1 in
            -p | -p=*)                   extract_arg "\-p" $1 $2; prev_rc=$?;
                                         if [ "$prev_rc" -eq 1 ]; then shift; fi
                                         CUSTOM_PATH=$extract_arg_ret
            ;;
            --path | --path=*)           extract_arg "\--path" $1 $2; prev_rc=$?;
                                         if [ "$prev_rc" -eq 1 ]; then shift; fi
                                         CUSTOM_PATH=$extract_arg_ret
            ;;
            -c | --cold)                 FLAG_COLD_UPGRADE=true
            ;;
            -y | --yes)                  FLAG_ASSUME_YES=true
            ;;
            --skip_same_ver_upgrades)    FLAG_SKIP_SAME_VER_UPGRADES=true
            ;;
            --reinit_after_upgrade)      FLAG_UPGRADE_REINIT=true
            ;;
            --log_to_dmesg)              FLAG_LOG_TO_DMESG=true
            ;;
            --reboot_req_exit_code)      FLAG_REBOOT_REQ_EXIT_CODE=true
            ;;
            -s | --avoid_sudo)           FLAG_AVOID_SUDO=true
            ;;
            -m | -m=*)                   extract_arg "\-m" $1 $2; prev_rc=$?;
                                         if [ "$prev_rc" -eq 1 ]; then shift; fi
                                         MODEL_PATT=$extract_arg_ret
            ;;
            --model | --model=*)         extract_arg "\--model" $1 $2; prev_rc=$?;
                                         if [ "$prev_rc" -eq 1 ]; then shift; fi
                                         MODEL_PATT=$extract_arg_ret
            ;;
            -h | --help)                 print_help_text false; exit 0
            ;;
            --help_full)                 print_help_text true; exit 0
            ;;
            -v | --version)              echo $SCRIPT_VERSION; exit 0
            ;;
            --persistent_config)         PERSISTENT_CONFIG_TYPE=${2}; shift
            ;;
            --persistent_config=*)       extract_arg "\--persistent_config" $1 $2; prev_rc=$?;
                                         if [ "$prev_rc" -eq 1 ]; then shift; fi
                                         PERSISTENT_CONFIG_TYPE=$extract_arg_ret
            ;;
            *)                           echo "quadra_auto_upgrade.sh: invalid option -- '${1}'" 1>&2;
                                         echo "Try './quadra_auto_upgrade.sh -h' for more information." 1>&2;
                                         exit 1
            ;;
        esac
        shift
    done
}

# Check if user is root. Set $SUDO accordingly
function set_sudo() {
    if [[ $(whoami) == "root" ]]; then
        SUDO=""
    else
        SUDO="sudo "
    fi
}

# Check for passwordless sudo access
# return 0 if true, 124 if false
function sudo_check() {
    timeout -k 1 1 sudo whoami &> /dev/null
    return $?
}

# Determine whether to use sudo for accessing ni_rsrc_mon and ni_rsrc_update
# Set global variable NI_RSRC_PERMISSION_PREFIX to "sudo " or ""
function get_ni_rsrc_permission_prefix() {
    if [ ! -f /dev/shm/NI_SHM_CODERS ]; then
        NI_RSRC_PERMISSION_PREFIX=""
    elif [ -w /dev/shm/NI_SHM_CODERS ]; then
        NI_RSRC_PERMISSION_PREFIX=""
    else
        NI_RSRC_PERMISSION_PREFIX=${SUDO}
    fi
}

# Present a yes/no question, read answer
# $1 - prompt to use. Note, ' Press [y/n]: ' will be auto-inserted
# return 1 if yes, 0 if no
function prompt_yn() {
    if ! $FLAG_ASSUME_YES; then
        while true; do
            read -p "${1} Press [y/n]: " -n  1 -r
            log "" 1
            if [[ $REPLY =~ ^[Yy1]$ ]]; then
                return 1
            elif [[ $REPLY =~ ^[Nn0]$ ]]; then
                return 0
            else
                log "${cYlw}Warning${cRst}: Unrecognized input. Please try again." 1
            fi
        done
    else
        # Assume yes case
        log "${1} Press [y/n]: y" 1
        return 1
    fi
}

# Check for user arg conflicts
# return 0 if no conflict, return 1 if conflicted
function check_arg_conflict() {
    if [[ -n "$PERSISTENT_CONFIG_TYPE" ]] &&  [[ -n ${CUSTOM_PATH} ]] && [[ -f ${CUSTOM_PATH} ]]; then
        # ERROR: Both PERSISTENT_CONFIG_TYPE and CUSTOM_PATH are defined
        log "${cRed}Error${cRst}: Cannot use --path to point to file when also using --persistent_config." 6
        return 1
    fi
    return 0
}

# Check nvme-cli is installed on system
# return 0 if installed, 1 if failed
function check_nvme_cli() {
    if (! [[ -x "$(command -v nvme)" ]]) && (! [[ -x "$(${SUDO}which nvme)" ]]); then
        # ERROR: nvme-cli not installed
        log "${cRed}Error${cRst}: NVMe-CLI is not installed. Please install it and try again!" 6
        return 1
    fi
    return 0
}

# Initialize upgrade script log file for individual card upgrade functions run
# in parallel (reset_device(), cold_upgrade(), unified_upgrade_check(), and unified_upgrade())
# return 0 if successful, 1 if failed
function create_upgrade_log() {
    ${SUDO}rm $UPGRADE_LOG &> /dev/null
    touch $UPGRADE_LOG
    return 0
}

# Update the global arrays $DEVICES, $NODES, $SERIAL_NUMS, $FW_REVISION, $MODEL_NUMS with
# latest information of card/device
# return 0 if successful, 1 if failed
function get_cards_info() {
    # Local arrays that will be updated and used
    UNSORTED_DEVICES=()
    UNSORTED_NODES=()
    UNSORTED_SERIAL_NUMS=()
    UNSORTED_MODEL_NUMS=()
    UNSORTED_FW_REVISION=()
    INDEX_SEQUENCE=()       # Index sequence for sorting

    # Reset global arrays for this function's output
    DEVICES=()
    NODES=()
    SERIAL_NUMS=()
    FW_REVISION=()
    MODEL_NUMS=()

    # Get the unsorted information of the cards/devices from sudo nvme list
    # From version 2.0, nvme data changes formats
    NVME_VERSION=(`nvme --version | grep -oP '^nvme version \K\d+\.\d+(\.\d+)?'`)
    if [[ $(printf "%s\n%s" "$NVME_VERSION" "2.0" | sort -V | head -n 1) == "2.0" ]]; then
        XCOD_DATA=(`${SUDO}nvme list | sed -e '1,2d' | grep -P "Quadra" | sed 's/\([^ ]*\)[ ]*\([^ ]*\)[ ]*\([^ ]*\)[ ]*\([^ ]*\).*\(........\)/\1 \3 \4 \5/'`)
    else
        XCOD_DATA=(`${SUDO}nvme list | sed -e '1,2d' | grep -P "Quadra" | sed 's/\([^ ]*\)[ ]*\([^ ]*\)[ ]*\([^ ]*\).*\(........\)/\1 \2 \3 \4/'`)
    fi
    # Check all devices found are physical functions
    PFs=`ls -l /sys/class/block/ | grep 00.0/nvme`
    for (( i=0; i<${#XCOD_DATA[@]}; i+=4 )); do
        NODE=$(echo ${XCOD_DATA[$i]} | grep -Poh '(?<=/dev/).*$' 2> /dev/null)
        if echo $PFs | grep -q $NODE; then
            # Get device path for a given block path
            if which udevadm &> /dev/null && \
               udevadm info -q path -n ${XCOD_DATA[$i]} &> /dev/null && \
               udevadm info -q path -n ${XCOD_DATA[$i]} 2> /dev/null | grep -Poh '(?<=/)nvme\d+(?=(/|$))' &> /dev/null; then
                # Check udevadm to find connection from block name to device name
                DEVID="/dev/$(udevadm info -q path -n ${XCOD_DATA[$i]} | grep -Poh '(?<=/)nvme\d+(?=(/|$))')"
            else
                # Guess device path as contraction of block path
                DEVID=`echo ${XCOD_DATA[$i]} | cut -c 6-`
            fi
            UNSORTED_DEVICES+=(${DEVID})
            UNSORTED_NODES+=(${XCOD_DATA[$i]})
            UNSORTED_SERIAL_NUMS+=(${XCOD_DATA[$i+1]})
            UNSORTED_MODEL_NUMS+=(${XCOD_DATA[$i+2]})
            UNSORTED_FW_REVISION+=(${XCOD_DATA[$i+3]})
        fi
    done

    # Set up indices for sorting
    INDEX_SEQUENCE=( $(IFS=$'\n'; sort -V <<< "${UNSORTED_DEVICES[*]}" | grep -Poh '(?<=/dev/nvme)\d+') )
    if [ -z $INDEX_SEQUENCE ]; then
        log "${cRed}Error${cRst}: No Quadra Transcoder device found" 6
        return 1
    fi

    # Sorts and updates $DEVICES, $NODES, $SERIAL_NUMS, $FW_REVISION, $MODEL_NUMS
    for INDEX in ${INDEX_SEQUENCE[@]}; do
        for DEV_IDX in ${!UNSORTED_DEVICES[@]}; do
            if [[ ${UNSORTED_DEVICES[$DEV_IDX]} == /dev/nvme${INDEX} ]]; then
                DEVICES+=(${UNSORTED_DEVICES[$DEV_IDX]})
                NODES[${UNSORTED_DEVICES[$DEV_IDX]}]=${UNSORTED_NODES[$DEV_IDX]}
                SERIAL_NUMS[${UNSORTED_DEVICES[$DEV_IDX]}]=${UNSORTED_SERIAL_NUMS[$DEV_IDX]}
                MODEL_NUMS[${UNSORTED_DEVICES[$DEV_IDX]}]=${UNSORTED_MODEL_NUMS[$DEV_IDX]}
                FW_REVISION[${UNSORTED_DEVICES[$DEV_IDX]}]=${UNSORTED_FW_REVISION[$DEV_IDX]}
            fi
        done
    done

    return 0
}

# Check for some simple errors after get_cards_info() but before start of upgrade()
function error_checking() {
    if [[ ${#DEVICES[@]} == 0 ]]; then
        # ERROR: no devices found
        log "${cRed}Error${cRst}: No Quadra Transcoder device found" 6
        return 1
    elif [[ ${#SERIAL_NUMS[@]} != ${#DEVICES[@]} ]]; then
        # ERROR: not all serial numbers for all cards retrieved
        log "${cRed}Error${cRst}: Not all cards have proper serial number format" 6
        return 1
    elif [ -n ${CUSTOM_PATH} ] && [ ! -f ${CUSTOM_PATH} ] && [ ! -d ${CUSTOM_PATH} ]; then
        # ERROR: invalid path specified for --path
        log "${cRed}Error${cRst}: Cannot find specified FW file or folder at ${CUSTOM_PATH}" 6
        return 1
    elif ${NI_RSRC_PERMISSION_PREFIX}which ni_rsrc_mon &> /dev/null && \
         ${NI_RSRC_PERMISSION_PREFIX}timeout -s SIGKILL 2 ni_rsrc_mon -S 2> /dev/null | \
         egrep -oh "^([0-9]+[ ]+){3}([0-9]+)[ ]+([0-9]+[ ]+){3}" | tr -s ' ' | \
         cut -d ' ' -f 4 | grep -q [^0]; then
        # ERROR: found running dec/enc instances
        log "${cRed}Error${cRst}: Please stop all transcoding processes before running firmware upgrade" 6
        return 1
    fi
    return 0
}

# Discover available FW binaries
# Select FW binary file to use for each card; set $FW_BIN, $FW_BIN_FWREV
# Select upgrade type depending on FW version and FL version; set $UPGRADE_TYPE
# return 0 if successful, 1 if failed
function set_upgrade_bin_and_type() {
    # Exclude cards based on model number wildcard pattern
    for DEVICE in ${DEVICES[@]}; do
        if [[ ${MODEL_NUMS[$DEVICE]} == $MODEL_PATT ]]; then
            # default upgrade type is unified
            UPGRADE_TYPE[$DEVICE]='u'
        else
            UPGRADE_TYPE[$DEVICE]='s'
        fi
    done

    # Determine FW binary file to use
    if [ -n ${CUSTOM_PATH} ] && [ -f ${CUSTOM_PATH} ]; then
        # user specified FW binary path
        FW_BIN_SIZE=`stat -c %s ${CUSTOM_PATH}`
        if [[ $FW_BIN_SIZE -lt 1048704 ]]; then  # 1048704 is offset to end of header of FW img in nor_flash.bin
            log "${cRed}Error${cRst}: Suspicious file size for FW file at ${CUSTOM_PATH}. File possibly corrupt" 6
            return 1
        fi
        CUSTOM_BIN_FWREV=`dd if=${CUSTOM_PATH} bs=1 skip=1048580 count=8 2>/dev/null` # 1048580 is offset for FW rev in header of FW img in nor_flash.bin
        PERSISTENT_CONFIG_BIN=$(dd if=${CUSTOM_PATH} bs=1 skip=16640 count=8 2>/dev/null | grep -Poh 'CONF000\K[A-Z]')
        # notify user if selected FW binary would change persistent config type
        for DEVICE in ${DEVICES[@]}; do
            if [[ ${UPGRADE_TYPE[$DEVICE]} == 's' ]]; then
                continue
            fi
            FW_BIN[$DEVICE]=$CUSTOM_PATH
            FW_BIN_FWREV[$DEVICE]=$CUSTOM_BIN_FWREV
            PERSISTENT=$(${SUDO}nvme admin-passthru ${DEVICE} --opcode 0xc2 -n 1 --read --cdw12=0x90 -l 0x100 2>&1 | grep -Poh 'CONF000\K[A-Z]')
            if [[ "$PERSISTENT_CONFIG_BIN" != "$PERSISTENT" ]] ; then
                log "${cYlw}Warning${cRst}: Changing persistent config for $DEVICE from $PERSISTENT to $PERSISTENT_CONFIG_BIN" 1
            fi
        done
    elif [ -n ${CUSTOM_PATH} ] && [ -d ${CUSTOM_PATH} ]; then
        # FW folder path is specified
        # auto select FW binary in <quadra_auto_upgrade.sh path>/FL_BIN/
        for DEVICE in ${DEVICES[@]}; do
            if [[ ${UPGRADE_TYPE[$DEVICE]} == 's' ]]; then
                continue
            fi
            # official releases must be named with format nor_flash_v*.*.*-Quadra*.bin for auto-discovery
            if [[ -z "$PERSISTENT_CONFIG_TYPE" ]] ; then
                PERSISTENT=$(${SUDO}nvme admin-passthru ${DEVICE} --opcode 0xc2 -n 1 --read --cdw12=0x90 -l 0x100 2>&1 | grep -Poh 'CONF000\K[A-Z]')
            else
                PERSISTENT=${PERSISTENT_CONFIG_TYPE}
            fi
            if [[ -z "$PERSISTENT" ]] ; then
                FW_BIN_PATH=`ls ${CUSTOM_PATH}/nor_flash_v*.*.*-Quadra.bin 2> /dev/null | sort -V | tail -n 1 | xargs -n 1 realpath 2> /dev/null`
            else
                if [[ "$PERSISTENT" == "D" ]]; then
                    FW_BIN_PATH=`ls ${CUSTOM_PATH}/nor_flash_v*.*.*-Quadra.bin 2> /dev/null | sort -V | tail -n 1 | xargs -n 1 realpath 2> /dev/null`
                else
                    FW_BIN_PATH=`ls ${CUSTOM_PATH}/nor_flash_v*.*.*-Quadra-${PERSISTENT}.bin 2> /dev/null | sort -V | tail -n 1 | xargs -n 1 realpath 2> /dev/null`
                fi
            fi
            # Check FW found
            if [ -z "$FW_BIN_PATH" ] ; then
                if [[ -z "$PERSISTENT" ]] ; then
                    log "${cRed}Error${cRst}: Cannot find a compatible FW file for $DEVICE at ${CUSTOM_PATH}/nor_flash_v*.*.*-Quadra.bin" 6
                else
                    log "${cRed}Error${cRst}: Cannot find a compatible FW file for $DEVICE at ${CUSTOM_PATH}/nor_flash_v*.*.*-Quadra-${PERSISTENT}.bin" 6
                fi
                return 1
            fi
            FW_BIN_SIZE=`stat -c %s ${FW_BIN_PATH}`
            if [[ $FW_BIN_SIZE -lt 1048704 ]]; then  # 1048704 is offset to end of header of FW img in nor_flash.bin
                log "${cRed}Error${cRst}: Suspicious file size for FW file at ${FW_BIN_PATH}. File possibly corrupt" 6
                return 1
            fi
            CUSTOM_BIN_FWREV=`dd if=${FW_BIN_PATH} bs=1 skip=1048580 count=8 2>/dev/null` # 1048580 is offset for FW rev in header of FW img in nor_flash.bin
            FW_BIN[$DEVICE]=$FW_BIN_PATH
            FW_BIN_FWREV[$DEVICE]=$CUSTOM_BIN_FWREV
        done
    else
        log "${cRed}Error${cRst}: Cannot find specified FW file or folder at ${CUSTOM_PATH}" 6
        return 1
    fi

    FL_VERSION_WITH_UNIFIED_UPGRADE=400
    FL_VERSION_FOR_PERSISTENT_CONFIG=450

    # Determine upgrade type for each card
    # Store output in UPGRADE_TYPE as single letters for upgrade type to use:
    for DEVICE in ${DEVICES[@]}; do
        if [[ ${UPGRADE_TYPE[$DEVICE]} == 's' ]]; then
            continue
        fi
        # compare card FL2 version with FW binary FL2 version
        CARD_FL_VERSION=`${SUDO}nvme get-log ${DEVICE} -i 194 -l 8 2>/dev/null | grep -Poh '(?<= ").{5}'`
        BIN_FL_VERSION=`dd if=${FW_BIN[$DEVICE]} bs=1 skip=131076 count=5 2>/dev/null`
        # if FL2 version is not of expected format, the input FW binary file may be corrupt
        if ! [[ $BIN_FL_VERSION =~ [0-9A-Z]\.[0-9A-Z]\.[0-9A-Z] ]]; then
            log "${cRed}Error${cRst}: Invalid firmware loader version format (${BIN_FL_VERSION}) for FW file at ${FW_BIN[$DEVICE]}. Possibly incorrect file format or corruption" 6
            return 1
        fi

        #Check for unified upgrade
        if [[ "${CARD_FL_VERSION//.}" -lt "${FL_VERSION_WITH_UNIFIED_UPGRADE}" ]] && [[ "${BIN_FL_VERSION//.}" -ge "${FL_VERSION_WITH_UNIFIED_UPGRADE}" ]]; then
            UPGRADE_TYPE[$DEVICE]='c'
        elif [[ "${CARD_FL_VERSION//.}" -lt "${FL_VERSION_WITH_UNIFIED_UPGRADE}" ]] && [[ "${BIN_FL_VERSION//.}" -lt "${FL_VERSION_WITH_UNIFIED_UPGRADE}" ]]; then
            UPGRADE_TYPE[$DEVICE]='c' # use cold upgrade since warm upgrade is deprecated
        elif [[ "${CARD_FL_VERSION//.}" -ge "${FL_VERSION_WITH_UNIFIED_UPGRADE}" ]] && [[ "${BIN_FL_VERSION//.}" -lt "${FL_VERSION_WITH_UNIFIED_UPGRADE}" ]]; then
            UPGRADE_TYPE[$DEVICE]='c'
        fi

        # compare FW rev on card and in binary
        if $FLAG_SKIP_SAME_VER_UPGRADES && [[ "${FW_REVISION[$DEVICE]}" == "${FW_BIN_FWREV[$DEVICE]}" ]]; then
            UPGRADE_TYPE[$DEVICE]='s'
        fi
        PERSISTENT=$(${SUDO}nvme admin-passthru ${DEVICE} --opcode 0xc2 -n 1 --read --cdw12=0x90 -l 0x100 2>&1 | grep -Poh 'CONF000\K[A-Z]')
        # If change in persistent config, use cold upgrade
        if [[ -n "$PERSISTENT_CONFIG_TYPE" ]]; then
            if [[ "$PERSISTENT_CONFIG_TYPE" != "$PERSISTENT" ]] && [[ "${CARD_FL_VERSION//.}" -ge "${FL_VERSION_FOR_PERSISTENT_CONFIG}" ]] ; then
                if [[ "$PERSISTENT_CONFIG_TYPE" == "D"  ]] && [[ -z "$PERSISTENT" ]]; then
                    if $FLAG_SKIP_SAME_VER_UPGRADES; then
                        UPGRADE_TYPE[$DEVICE]='s'
                    else
                        UPGRADE_TYPE[$DEVICE]='u'
                    fi
                else
                    UPGRADE_TYPE[$DEVICE]='c'
                fi
            elif [[ "$PERSISTENT_CONFIG_TYPE" != "$PERSISTENT" ]] && [[ "${CARD_FL_VERSION//.}" -lt "${FL_VERSION_FOR_PERSISTENT_CONFIG}" ]] ; then
                FW_BIN_PATH=`ls ${CUSTOM_PATH}/nor_flash_v*.*.*-Quadra-${PERSISTENT_CONFIG_TYPE}.bin 2> /dev/null | sort -V | tail -n 1 | xargs -n 1 realpath 2> /dev/null`
                PERSISTENT_CONFIG_KEYS=$(dd if=${FW_BIN_PATH} bs=1 skip=16640 count=256 2> /dev/null | tr -d '\000' 2>/dev/null)
                echo "$PERSISTENT_CONFIG_KEYS" > ${TMP_WORKSPACE}/netint_upgrade_pc_dict.txt
                UPGRADE_TYPE[$DEVICE]='u'
            fi
        else
            if [ -n ${CUSTOM_PATH} ] && [ -f ${CUSTOM_PATH} ]; then
                PERSISTENT_CONFIG_BIN=$(dd if=${CUSTOM_PATH} bs=1 skip=16640 count=8 2>/dev/null | grep -Poh 'CONF000\K[A-Z]')
            else
                PERSISTENT_CONFIG_BIN=$(dd if=${FW_BIN_PATH} bs=1 skip=16640 count=8 2>/dev/null | grep -Poh 'CONF000\K[A-Z]')
            fi
            if [[ "$PERSISTENT_CONFIG_BIN" != "$PERSISTENT" ]] && [[ "${CARD_FL_VERSION//.}" -ge "${FL_VERSION_FOR_PERSISTENT_CONFIG}" ]] ; then
                UPGRADE_TYPE[$DEVICE]='c'
            elif [[ "$PERSISTENT_CONFIG_BIN" != "$PERSISTENT" ]] && [[ "${CARD_FL_VERSION//.}" -lt "${FL_VERSION_FOR_PERSISTENT_CONFIG}" ]] ; then
                PERSISTENT_CONFIG_KEYS=$(dd if=${FW_BIN_PATH} bs=1 skip=16640 count=256 2> /dev/null | tr -d '\000' 2>/dev/null)
                echo "$PERSISTENT_CONFIG_KEYS" > ${TMP_WORKSPACE}/netint_upgrade_pc_dict.txt
                UPGRADE_TYPE[$DEVICE]='u'
            fi
        fi

        # Select cold upgrade whenever upgrading from FW release v4.9.6 AND persistent_config_type is 'E'
        if [[ "$PERSISTENT" == "E" ]] && [[ "${FW_REVISION[$DEVICE]:0:3}" == "496" ]] && [[ ${UPGRADE_TYPE[$DEVICE]} != 's' ]]; then
            UPGRADE_TYPE[$DEVICE]='c'
        fi

        # prefer user-selected cold-upgrade to unified-upgrade but not skip-upgrade
        if $FLAG_COLD_UPGRADE && [[ ${UPGRADE_TYPE[$DEVICE]} == 'u' ]] && [[ ${UPGRADE_TYPE[$DEVICE]} != 's' ]]; then
            UPGRADE_TYPE[$DEVICE]='c'
        fi
    done
    return 0
}

# List information about cards detected and the FW they will be upgraded to
# return 0 if successful, 1 if failed
function list_quadra_devices() {
    # State which binary is being used and the upgrade type
    log "Number of Quadra Transcoders found on $(hostname): ${#DEVICES[@]}" 1

    declare -A UNIQUE_FW_BIN_PATHS
    for FW_BIN_PATH in "${FW_BIN[@]}"; do UNIQUE_FW_BIN_PATHS["$FW_BIN_PATH"]=1; done
    for FW_BIN_PATH in "${!UNIQUE_FW_BIN_PATHS[@]}"; do
        log "FW upgrade file: $FW_BIN_PATH" 1
    done

    log "#   Device         Block              Serial Number        Card FW Rev File FW Rev Action" 1
    log "--- -------------- ------------------ -------------------- ----------- ----------- --------------" 1
    INDEX=0
    for DEVICE in ${DEVICES[@]}; do
        if [[ ${UPGRADE_TYPE[$DEVICE]} == 'u' ]]; then
            ACTION="upgrade"
        elif [[ ${UPGRADE_TYPE[$DEVICE]} == 'c' ]]; then
            ACTION="cold upgrade"
        elif [[ ${UPGRADE_TYPE[$DEVICE]} == 's' ]]; then
            ACTION="skip upgrade"
        fi
        log "$(printf "%-3s %-14s %-18s %-20s %-11s %-11s %-12s\n" "${INDEX}" "${DEVICE}" "${NODES[$DEVICE]}" "${SERIAL_NUMS[$DEVICE]}" "${FW_REVISION[$DEVICE]}" "${FW_BIN_FWREV[$DEVICE]}" "${ACTION}")" 1
        (( INDEX+=1 ))
    done

    return 0
}

# Reset device so that its read queue is cleared
# $1 - target nvme device path (eg. /dev/nvme0)
function reset_device() {
    log "${1} -- Initial reset of device." 8
    ${SUDO}nvme reset $1 &> /dev/null
    rc=$?
    log "${1} -- Return value: ${rc}" 8
    if [ $rc -ne 0 ]; then
        log "${1} -- Error: Initial reset failed!" 8
        return 1
    fi
    return 0
}

# $1 - target nvme device path (eg. /dev/nvme0)
# $2 - fw binary path (eg. FL_BIN/nor_flash.bin)
function download_burn() {
    DEVICE=$1
    BINARY=$2

    # Download FW binary onto the card
    log "${DEVICE} -- Downloading firmware image." 8
    OUTPUT=$(${SUDO}nvme fw-download ${DEVICE} -f ${BINARY} -x 65536 2>&1)
    log "${DEVICE} -- Return value: $?" 8
    log "${DEVICE} -- ${OUTPUT}" 8
    if ! [ "$OUTPUT" == "Firmware download success" ]; then
        log "${DEVICE} -- Error: Download failed!" 8
        return 1
    fi

    # Burn FW binary
    log "${DEVICE} -- Burning firmware binary." 8
    OUTPUT=$(${SUDO}nvme admin-passthru ${DEVICE} --opcode 0xc7 -t 180000 2>&1) # opcode 0xC7 is burn_nor
    rc=$?
    log "${DEVICE} -- Return value: ${rc}" 8
    log "${DEVICE} -- ${OUTPUT}" 8
    if [ $rc -eq 0 ] && [[ $OUTPUT =~ 00000000$ ]]; then
        log "${DEVICE} -- Upgrade complete!" 8
    else
        log "${DEVICE} -- Error: Commit failed!" 8
        return 2
    fi

    return 0
}

# Issue cold upgrade nvme commands to the selected card. Note, device still
# requires hard reset (eg. powercycle/hotplug) to activate new FW.
# $1 - target nvme device path (eg. /dev/nvme0)
# $2 - fw binary path (eg. FL_BIN/nor_flash.bin)
# return 0 if successful, non-0 if failed
function cold_upgrade() {
    DEVICE=$1
    BINARY=$2

    log "${DEVICE} -- Upgrading with $BINARY." 8

    download_burn "${DEVICE}" "${BINARY}"
    rc="$?"
    if [[ "${rc}" -ne 0 ]]; then
      return "${rc}"
    fi

    log "${DEVICE} -- Finished FW cold upgrade download and burn." 8
    return 0
}

# Issue unified upgrade nvme commands to the selected card and attempt to activate
# the new FW through NVMe soft reset.
# $1 - target nvme device path (eg. /dev/nvme0)
# $2 - fw binary path (eg. FL_BIN/nor_flash.bin)
# return 0 if successful, non-0 if failed
function unified_upgrade() {
    DEVICE=$1
    BINARY=$2

    log "${DEVICE} -- Upgrading with $BINARY." 8

    download_burn "${DEVICE}" "${BINARY}"
    rc="$?"
    if [[ "${rc}" -ne 0 ]]; then
      return "${rc}"
    fi

    # Reset card to activate new FW
    log "${DEVICE} -- Resetting device." 8
    OUTPUT=$(${SUDO}nvme reset ${DEVICE} 2>&1)
    rc=$?
    log "${DEVICE} -- Return value: ${rc}" 8
    log "${DEVICE} -- ${OUTPUT}" 8

    if [ $rc -ne 0 ]; then
        # log "${DEVICE} -- ${cRed}Error${cRst}: Reset failed!" 6
        log "${DEVICE} -- Error: Reset failed! Require reboot." 8
        return 3
    fi

    log "${DEVICE} -- Finished FW upgrade download, commit, and reset." 8
    return 0
}

# Check unified upgrade sucessfully activated new FW rev
# the new FW.
# $1 - target nvme device path (eg. /dev/nvme0)
# return 0 if successful, non-0 if failed
function unified_upgrade_check() {
    DEVICE=$1
    FW_REV=`${SUDO}nvme id-ctrl ${DEVICE} | grep -Poh 'fr +: \K.*'`
    CARD_LOADED_FL2_VERSION=`${SUDO}nvme get-log ${DEVICE} -i 194 -l 8 2>/dev/null | grep -Poh '(?<= ").{5}'`
    CARD_NOR_FLASH_FL2_VERSION=`${SUDO}nvme get-log ${DEVICE} -i 194 -l 16 -b 2>/dev/null | hexdump -e'"%_u"' | tail -n 1`
    CARD_NOR_FLASH_FL2_VERSION="${CARD_NOR_FLASH_FL2_VERSION%nul\*}"

    # Check that the current FW Rev on the card matches what is in image file used for upgrade
    if [[ ${FW_BIN_FWREV[$DEVICE]} != ${FW_REV} ]]; then
        log "${DEVICE} -- Error: Upgraded FW not properly activated! May require reboot." 8
        return 4
    fi
    if [[ "${CARD_LOADED_FL2_VERSION}" != "${CARD_NOR_FLASH_FL2_VERSION}" ]]; then
        log "${DEVICE} -- Last executed firmware loader version is v${CARD_LOADED_FL2_VERSION}" 1
        log "${DEVICE} -- Reboot to run firmware loader version v${CARD_NOR_FLASH_FL2_VERSION} (optional)" 1
    fi

    return 0
}

# Re-init resources for unified upgraded cards
# return 0 if successful, non-0 if failed
function unified_upgrade_reinit() {
    rc=0
    ${SUDO}rm $TMP_WORKSPACE/netint_upgrade_reinit_log.txt &> /dev/null
    touch $TMP_WORKSPACE/netint_upgrade_reinit_log.txt

    # Check ni_rsrc_* apps are accessible on $PATH
    if ! ${NI_RSRC_PERMISSION_PREFIX}which ni_rsrc_update &> /dev/null || \
       ! ${NI_RSRC_PERMISSION_PREFIX}which ni_rsrc_mon &> /dev/null; then
        log "${cYlw}Warning${cRst}: could not find ni_rsrc_update and ni_rsrc_mon for --reinit_after_upgrade" 1
        return 1
    fi

    for DEVICE in ${DEVICES[@]}; do
        if [[ ${UPGRADE_TYPE[$DEVICE]} == 'u' ]]; then
            echo "${NI_RSRC_PERMISSION_PREFIX}ni_rsrc_update -l debug -d ${DEVICE}" >> $TMP_WORKSPACE/netint_upgrade_reinit_log.txt 2>&1
            ${NI_RSRC_PERMISSION_PREFIX}ni_rsrc_update -l debug -d ${DEVICE} >> $TMP_WORKSPACE/netint_upgrade_reinit_log.txt 2>&1
            (( rc|=$? ))
            if [ $rc -ne 0 ]; then
                log "${DEVICE} -- Error: failed to remove device for --reinit_after_upgrade" 8
            fi
        fi
    done
    echo "${NI_RSRC_PERMISSION_PREFIX}ni_rsrc_mon -l debug" >> $TMP_WORKSPACE/netint_upgrade_reinit_log.txt 2>&1
    ${NI_RSRC_PERMISSION_PREFIX}ni_rsrc_mon -l debug  >> $TMP_WORKSPACE/netint_upgrade_reinit_log.txt 2>&1
    (( rc|=$? ))
    if [ $rc -ne 0 ]; then
        log "${cYlw}Warning${cRst}: failed to init resources for --reinit_after_upgrade" 1
    fi

    if [ $rc -ne 0 ]; then
        return 1
    else
        return 0
    fi
}

# Parse $UPGRADE_LOG for upgrade results of each card to update $UPGRADE_RESULT. Can be called
# multiple times and before completion of upgrade process
# Note: log message formats referenced here have to match where they are written
function check_upgrade_results() {
    for DEVICE in ${DEVICES[@]}; do
        if grep -q "${DEVICE} -- Finished FW" $UPGRADE_LOG; then
            UPGRADE_RESULT[${DEVICE}]=0
        fi
        if grep -q "${DEVICE} -- Error: Initial reset failed!" $UPGRADE_LOG; then
            UPGRADE_RESULT[${DEVICE}]=1
        elif grep -q "${DEVICE} -- Error: Download failed!" $UPGRADE_LOG; then
            UPGRADE_RESULT[${DEVICE}]=2
        elif grep -q "${DEVICE} -- Error: Burn failed!" $UPGRADE_LOG; then
            UPGRADE_RESULT[${DEVICE}]=3
        elif grep -q "${DEVICE} -- Error: Reset failed!" $UPGRADE_LOG; then
            UPGRADE_RESULT[${DEVICE}]=4
        elif grep -q "${DEVICE} -- Error: Upgraded FW not properly activated" $UPGRADE_LOG; then
            UPGRADE_RESULT[${DEVICE}]=5
        fi
    done
}

# Convert $UPGRADE_RESULT integer to summary string
function upgrade_result_code_to_message() {
    case $1 in
        -1) UPGRADE_RESULT_STR="Skipped same ver upgrade"
        ;;
        0) if [[ ${UPGRADE_TYPE[$DEVICE]} == 'c' ]]; then UPGRADE_RESULT_STR="Successful cold upgrade";
           elif [[ ${UPGRADE_TYPE[$DEVICE]} == 'u' ]]; then UPGRADE_RESULT_STR="Successful upgrade";
           else UPGRADE_RESULT_STR="Script error, unknown \$UPGRADE_TYPE"; fi
        ;;
        1) UPGRADE_RESULT_STR="Failed initial reset"
        ;;
        2) UPGRADE_RESULT_STR="Failed FW download"
        ;;
        3) UPGRADE_RESULT_STR="Failed FW burn"
        ;;
        4) UPGRADE_RESULT_STR="Failed FW activation"
        ;;
        5) UPGRADE_RESULT_STR="Failed activation check"
        ;;
        *) UPGRADE_RESULT_STR="Script error, unknown \$UPGRADE_RESULT"
        ;;
    esac
}

# List upgrade status after upgrade is complete and successful
function list_upgrade_status() {
    log "#   Device         Block              Serial Number        Card FW Rev Result" 1
    log "--- -------------- ------------------ -------------------- ----------- ------------------------" 1
    INDEX=0
    for DEVICE in ${DEVICES[@]}; do
        upgrade_result_code_to_message ${UPGRADE_RESULT[$DEVICE]}
        log "$(printf "%-3s %-14s %-18s %-20s %-11s %s\n" "${INDEX}" "${DEVICE}" "${NODES[$DEVICE]}" "${SERIAL_NUMS[$DEVICE]}" "${FW_REVISION[$DEVICE]}" "${UPGRADE_RESULT_STR}")" 1
        (( INDEX+=1 ))
    done
    return 0
}

# Perform upgrade with information from global arrays
# return 0 if successful, 1 if failed
function upgrade() {
    log "Starting to upgrade. This process may take 1 minute..." 1
    create_upgrade_log

    # Wait for FW reads caused by ni_rsrc_mon in error_checking() to complete lest they timeout
    PIDS1=()
    for DEVICE in ${DEVICES[@]}; do
        UPGRADE_RESULT[${DEVICE}]=0
        reset_device $DEVICE &
        PIDS1+=($!)
    done
    for PID in ${PIDS1[@]}; do
        wait $PID > /dev/null 2>&1
    done
    check_upgrade_results
    sleep 1

    # Perform upgrades in parallel
    PIDS2=() # Note, if this array were to have same name as PIDS1 array above,
             # there will be a print indentation bug on Ubuntu22 where every
             # line is indented by length of previous line.
    for DEVICE in ${DEVICES[@]}; do
        if [[ ${UPGRADE_RESULT[${DEVICE}]} != 0 ]]; then
            continue
        fi
        if [[ ${UPGRADE_TYPE[$DEVICE]} == 'u' ]]; then
            unified_upgrade ${DEVICE} ${FW_BIN[$DEVICE]} &
            PIDS2+=($!)
        elif [[ ${UPGRADE_TYPE[$DEVICE]} == 'c' ]]; then
            cold_upgrade ${DEVICE} ${FW_BIN[$DEVICE]} &
            PIDS2+=($!)
        elif [[ ${UPGRADE_TYPE[$DEVICE]} == 's' ]]; then
            UPGRADE_RESULT[${DEVICE}]=-1
        fi
    done
    for PID in ${PIDS2[@]}; do
        wait $PID > /dev/null 2>&1
    done
    check_upgrade_results

    # Check upgraded cards are sucessfully activated
    PIDS3=()
    for DEVICE in ${DEVICES[@]}; do
        if [[ ${UPGRADE_TYPE[$DEVICE]} == 'u' ]] && [[ ${UPGRADE_RESULT[${DEVICE}]} == 0 ]]; then
            unified_upgrade_check ${DEVICE} &
            PIDS3+=($!)
        fi
    done
    for PID in ${PIDS3[@]}; do
        wait $PID > /dev/null 2>&1
    done
    check_upgrade_results

    # For reduced confusion, in results table show sucessfully upgraded cards 'Card FW Rev' as ${FW_BIN_FWREV[$DEVICE]}
    for DEVICE in ${DEVICES[@]}; do
        if [[ ${UPGRADE_RESULT[${DEVICE}]} == 0 && ( ${UPGRADE_TYPE[$DEVICE]} == 'c' || ${UPGRADE_TYPE[$DEVICE]} == 'u' ) ]]; then
            FW_REVISION[$DEVICE]=${FW_BIN_FWREV[$DEVICE]}
        fi
    done

    # Determine whether reboot is required after script exit
    for DEVICE in ${DEVICES[@]}; do
        if [[ (${UPGRADE_RESULT[${DEVICE}]} == 0 && ${UPGRADE_TYPE[$DEVICE]} == 'c') || \
              (${UPGRADE_RESULT[${DEVICE}]} == 4 && ${UPGRADE_TYPE[$DEVICE]} == 'u') || \
              (${UPGRADE_RESULT[${DEVICE}]} == 5 && ${UPGRADE_TYPE[$DEVICE]} == 'u') ]]; then
            REBOOT_REQ=true
        fi
    done

    # Check if FW revision in nvme list matches card
    NVME_VERSION=(`nvme --version | grep -oP '^nvme version \K\d+\.\d+(\.\d+)?'`)
    if [[ $(printf "%s\n%s" "$NVME_VERSION" "2.0" | sort -V | head -n 1) == "2.0" ]]; then
        for DEVICE in ${DEVICES[@]}; do
            if [[ ${UPGRADE_RESULT[${DEVICE}]} == 0 && ( ${UPGRADE_TYPE[$DEVICE]} == 'c' || ${UPGRADE_TYPE[$DEVICE]} == 'u' ) ]]; then
                FW_REV=`${SUDO}nvme id-ctrl ${DEVICE} | grep -Poh 'fr +: \K.*'`
                NVME_LIST_FWREV=(`${SUDO}nvme list | sed -e '1,2d' | grep -P "${DEVICE}" | sed 's/\([^ ]*\)[ ]*\([^ ]*\)[ ]*\([^ ]*\)[ ]*\([^ ]*\).*\(........\)/\5/'`)
                if [[ ${NVME_LIST_FWREV} != ${FW_REV} ]]; then
                    log "${cYlw}Warning${cRst}: This system's combination of nvme-cli version and linux kernel version will cause FWrev to not update on nvme list after upgrade (linux 983a338b, nvme-cli #1991). Cards that have been successfully upgraded are ready for use. To fix visual bug, please reload NVMe driver or reboot system." 1
                    break
                fi
            fi
        done
    fi

    # Restore corrupted tty settings to recover from Ubuntu 22 bug #1992025
    if [ -t 0 ]; then
        stty $TTY_SETTINGS
    fi

    # Upgrade completed
    if grep -q "Error" $UPGRADE_LOG; then
        mv $UPGRADE_LOG ./$(basename ${UPGRADE_LOG})
        if [ $? -ne 0 ]; then
            log "${cRed}Error${cRst}: Firmware upgrade failed! See errors in ${UPGRADE_LOG} and try again." 6
        else
            log "${cRed}Error${cRst}: Firmware upgrade failed! See errors in $(basename ${UPGRADE_LOG}) and try again." 6
        fi
        list_upgrade_status
        return 1
    else
        log "Firmware upgrade complete!" 1
        list_upgrade_status
        if [[ "${UPGRADE_TYPE[*]}" =~ "c" ]]; then
            log "Cards that underwent cold upgrade require system reboot to activate new firmware loader and firmware." 1
        fi
        # Use ni_rsrc_update to remove unified upgraded cards and re-init resources
        if [[ "${UPGRADE_TYPE[*]}" =~ "u" ]] && $FLAG_UPGRADE_REINIT; then
            unified_upgrade_reinit
            if [ $? -eq 0 ]; then
                log "Cards that underwent upgrade ready for use." 1
            else
                log "${cYlw}Warning${cRst}: Cards that underwent upgrade have new FW activated but still require ni_rsrc initialization (${NI_RSRC_PERMISSION_PREFIX}init_rsrc)." 1
            fi
        elif [[ "${UPGRADE_TYPE[*]}" =~ "u" ]]; then
            log "Cards that underwent upgrade have new FW activated but still require ni_rsrc initialization (${NI_RSRC_PERMISSION_PREFIX}init_rsrc)." 1
        fi
        # rm $UPGRADE_LOG # No need to remove $UPGRADE_LOG if its only in /tmp/
        return 0
    fi
}

function exit_caused_by_script() {
    trap - EXIT
    # Restore corrupted tty settings to recover from Ubuntu 22 bug #1992025
    if [ -t 0 ]; then
        stty $TTY_SETTINGS
    fi
    log "quadra_auto_upgrade.sh finished" 1
    exit $1
}

function exit_caused_by_signal() {
    log "${cYlw}Warning${cRst}: quadra_auto_upgrade.sh received external signal to exit" 2
    exit_caused_by_script $1
}

####### MAIN #######
shopt -u nullglob
setup_terminal_colors           # Setup variables for terminal color printouts
parse_command_line_args "$@"    # Parse user args

log "Welcome to the Quadra Transcoder Firmware Upgrade Utility ${SCRIPT_VERSION}!" 1
trap exit_caused_by_signal EXIT

# Check if user is root, set $SUDO accordingly
if $FLAG_AVOID_SUDO; then
    SUDO=""
else
    set_sudo
fi

# Check for conditions to run without user intervention
if $FLAG_ASSUME_YES && [ -n ${SUDO} ]; then
    sudo_check
    if [ $? -ne 0 ]; then
        log "${cRed}Error${cRst}: need to configure passwordless use of sudo before using -y|--yes" 2
        exit_caused_by_script 1
    fi
fi

# Check if both PERSISTENT_CONFIG_TYPE and CUSTOM_PATH are defined
check_arg_conflict
if [ $? -ne 0 ]; then exit_caused_by_script 1; fi

# Set global variable $NI_RSRC_PERMISSION_PREFIX
get_ni_rsrc_permission_prefix

# check nvme-cli installed on system
check_nvme_cli
if [ $? -ne 0 ]; then exit_caused_by_script 1; fi

# Initialize and sort arrays with card attributes: $DEVICES, $NODES, $SERIAL_NUMS, $FW_REVISION, $MODEL_NUMS
get_cards_info
if [ $? -ne 0 ]; then exit_caused_by_script 1; fi

# Check for simple errors in environment
error_checking
if [ $? -ne 0 ]; then exit_caused_by_script 1; fi

# For each Quadra card, determine which kind of upgrade to use and select FW file to upgrade with
set_upgrade_bin_and_type
if [ $? -ne 0 ]; then exit_caused_by_script 1; fi

# List Quadra devices and corresponding serial numbers and auto-select binary files
list_quadra_devices
if [ $? -ne 0 ]; then exit_caused_by_script 1; fi

# Check if cards have the same FW Rev as their target binary and check if all cards are to be skipped for upgrade
INDEX=0
NUM_CARDS_SKIP_UPGRADE=0
IDX_CARDS_SAME_FW_AS_TARGET=()
for DEVICE in ${DEVICES[@]}; do
    if [[ ${UPGRADE_TYPE[$DEVICE]} == 's' ]]; then
        (( NUM_CARDS_SKIP_UPGRADE+=1 ))
        if [ ! -z "${FW_BIN_FWREV[$DEVICE]+x}" ] && [[ ${FW_BIN_FWREV[$DEVICE]} == ${FW_REVISION[$DEVICE]} ]]; then
            IDX_CARDS_SAME_FW_AS_TARGET+=($INDEX)
        fi
    fi
    (( INDEX+=1 ))
done
if [[ ${#IDX_CARDS_SAME_FW_AS_TARGET[@]} == ${#DEVICES[@]} ]]; then
    log "${cYlw}Warning${cRst}: All cards have the same FW as the target FW upgrade file" 1
    if $FLAG_SKIP_SAME_VER_UPGRADES; then
        exit_caused_by_script 0
    fi
elif [[ $NUM_CARDS_SKIP_UPGRADE == ${#DEVICES[@]} ]]; then
    log "${cYlw}Warning${cRst}: All cards are to be skipped for upgrade" 1
    exit_caused_by_script 0
elif [[ ${#IDX_CARDS_SAME_FW_AS_TARGET[@]} -gt 0 ]]; then
    log "${cYlw}Warning${cRst}: ${#IDX_CARDS_SAME_FW_AS_TARGET[@]} cards have the same FW as the target FW upgrade file" 1
    log "Indices of devices with same FW:" $( IFS="," ; echo "${!IDX_CARDS_SAME_FW_AS_TARGET[*]}" ) 1
fi

# Prompt for user to begin upgrade
prompt_yn "Do you want to upgrade ALL of the above Quadra Transcoder(s)?"
if [ $? -eq 0 ]; then
    log "Upgrade cancelled" 1
    exit_caused_by_script 0;
fi

# Perform upgrade
log "${cYlw}Warning${cRst}: User must not interrupt the upgrade process, doing so may lead to device malfunction!" 1
upgrade; upgrade_rc=$?

if [[ -f "${TMP_WORKSPACE}/netint_upgrade_pc_dict.txt" ]] ; then
    PERSISTENT_INFO=$(cat ${TMP_WORKSPACE}/netint_upgrade_pc_dict.txt)
    KEY_COUNT=$(echo "$PERSISTENT_INFO" | tr -cd '[:alnum:]' | wc -m | awk '{print $1/8}' | xargs printf "%x\n") 2> /dev/null
    dd if=/dev/zero of=${TMP_WORKSPACE}/netint_upgrade_pc_dict.txt count=1 bs=$(( 4096 - `stat --printf="%s" ${TMP_WORKSPACE}/netint_upgrade_pc_dict.txt` )) oflag=append conv=notrunc 2> /dev/null
    for DEVICE in ${DEVICES[@]}; do
        sudo nvme admin-passthru $DEVICE -o 0xc1 -n 1 --write --cdw12=0x91 --cdw13=0x$KEY_COUNT -l 4096 --input-file=${TMP_WORKSPACE}/netint_upgrade_pc_dict.txt 2> /dev/null
    done
    rm ${TMP_WORKSPACE}/netint_upgrade_pc_dict.txt
fi

if $FLAG_REBOOT_REQ_EXIT_CODE && $REBOOT_REQ; then
    exit_caused_by_script 3;    # Reboot required
elif [ $upgrade_rc -eq 0 ]; then
    exit_caused_by_script 0;    # Success
else
    exit_caused_by_script 1;    # Fail
fi
