# waiSessionTable.py : manaage session table.

from enum import Enum
import time

MAX_SESSIONS = 100  # Define the maximum number of sessions as a constant

class SessionStatus(Enum):
    CLOSED = 0
    OPENED = 1
    
class SessionError(Enum):
    NO_ERROR = 0
    NO_FREE_SLOT = -1
    INVALID_INDEX = -2
    SESSION_CLOSED = -3
    INVALID_SENDER_ID = -4

class SessionTable:
    start_index = 0  # Class variable to store the starting index for searching
    active_session_count = 0 # store number of active session

    def __init__(self):
        # Initialize an array to store session entries
        self.sessions = [
            {
                "sender_id": None,
                "alive_timestamp": 0,
                "status": SessionStatus.CLOSED,
                "is_eos_received": 0,
                "chunk_data_in_progress_count": 0,
                "sender_pid": None,
                "chunk_data": [],  # Initialize chunk_data as an empty list
                "latest_tokens": [],  # Initialize latest_tokens as an empty list
                "caption_buffer": {},  # Initialize caption_buffer as an empty dictionary
                "worker_chunk_data_send_count": 0,  # Initialize worker_chunk_data_send_count
                "client_caption_send_count": 0,  # Initialize client_caption_send_count
            } for _ in range(MAX_SESSIONS)
        ]

    def open_session(self, sender_id, sender_pid):
        total_entries = len(self.sessions)
        
        # Loop through the total number of entries in the session table
        for offset in range(total_entries):
            session_id = (self.start_index + offset) % total_entries
            entry = self.sessions[session_id]
            if entry["status"] == SessionStatus.CLOSED:
                # Update entry data
                entry["sender_id"] = sender_id
                entry["alive_timestamp"] = time.time()
                entry["status"] = SessionStatus.OPENED
                entry["sender_pid"] = sender_pid
                # Update the class variable start_index for the next search
                self.start_index = (session_id + 1) % total_entries
                self.active_session_count += 1
                return session_id, SessionError.NO_ERROR  # Return the session index and no error

        return -1, SessionError.NO_FREE_SLOT  # No free slot available, return error

    def _clear_table_entry(self, session_index):
        # No need to check if the session index is valid,
        # as this method is intended for internal use

        # Reset all items to default values
        self.sessions[session_index] = {
            "sender_id": None,
            "alive_timestamp": 0,
            "status": SessionStatus.CLOSED,
            "is_eos_received": 0,
            "chunk_data_in_progress_count": 0,
            "sender_pid": None,
            "chunk_data": [],  # Initialize chunk_data as an empty list
            "latest_tokens": [],  # Initialize latest_tokens as an empty list
            "caption_buffer": {},  # Initialize caption_buffer as an empty dictionary
            "worker_chunk_data_send_count": 0,  # Initialize worker_chunk_data_send_count
            "client_caption_send_count": 0,  # Initialize client_caption_send_count
        }
        return SessionError.NO_ERROR

    def close_session(self, session_index):
        # Check if the session index is valid
        if 0 <= session_index < MAX_SESSIONS:
            self.active_session_count -= 1
            # Call the _clear_table_entry method to reset all items
            return self._clear_table_entry(session_index)
        else:
            return SessionError.INVALID_INDEX

    def update_alive_timestamp(self, session_index):
        # Check if the session index is valid
        if 0 <= session_index < MAX_SESSIONS:
            entry = self.sessions[session_index]
            # Check if the session is opened
            if entry["status"] == SessionStatus.OPENED:
                # Update the alive timestamp
                entry["alive_timestamp"] = time.time()
                return SessionError.NO_ERROR
            else:
                return SessionError.SESSION_CLOSED
        else:
            return SessionError.INVALID_INDEX

    def get_sender_id(self, session_index):
        # Check if the session index is valid
        if 0 <= session_index < MAX_SESSIONS:
            entry = self.sessions[session_index]
            # Check if the session is opened
            if entry["status"] == SessionStatus.OPENED:
                return entry["sender_id"], SessionError.NO_ERROR
            else:
                return None, SessionError.SESSION_CLOSED
        else:
            return None, SessionError.INVALID_INDEX

    def get_alive_timestamp(self, session_index):
        # Check if the session index is valid
        if 0 <= session_index < MAX_SESSIONS:
            entry = self.sessions[session_index]
            # Check if the session is opened
            if entry["status"] == SessionStatus.OPENED:
                return entry["alive_timestamp"], SessionError.NO_ERROR
            else:
                return None, SessionError.SESSION_CLOSED
        else:
            return None, SessionError.INVALID_INDEX

    def get_status(self, session_index):
        # Check if the session index is valid
        if 0 <= session_index < MAX_SESSIONS:
            return self.sessions[session_index]["status"], SessionError.NO_ERROR
        else:
            return None, SessionError.INVALID_INDEX

    def get_is_eos_received(self, session_index):
        # Check if the session index is valid
        if 0 <= session_index < MAX_SESSIONS:
            return self.sessions[session_index]["is_eos_received"], SessionError.NO_ERROR
        else:
            return None, SessionError.INVALID_INDEX

    def set_is_eos_received(self, session_index, value):
        # Check if the session index is valid
        if 0 <= session_index < MAX_SESSIONS:
            self.sessions[session_index]["is_eos_received"] = value
            return SessionError.NO_ERROR
        else:
            return SessionError.INVALID_INDEX

    def get_chunk_data_in_progress_count(self, session_index):
        # Check if the session index is valid
        if 0 <= session_index < MAX_SESSIONS:
            return self.sessions[session_index]["chunk_data_in_progress_count"], SessionError.NO_ERROR
        else:
            return None, SessionError.INVALID_INDEX

    def increase_chunk_data_in_progress_count(self, session_index):
        # Check if the session index is valid
        if 0 <= session_index < MAX_SESSIONS:
            self.sessions[session_index]["chunk_data_in_progress_count"] += 1
            self.sessions[session_index]["alive_timestamp"] = time.time()
            return self.sessions[session_index]["chunk_data_in_progress_count"], SessionError.NO_ERROR
        else:
            return None, SessionError.INVALID_INDEX

    def decrease_chunk_data_in_progress_count(self, session_index):
        # Check if the session index is valid
        if 0 <= session_index < MAX_SESSIONS:
            # Ensure the count does not go below zero
            self.sessions[session_index]["chunk_data_in_progress_count"] = max(0, self.sessions[session_index]["chunk_data_in_progress_count"] - 1)
            return self.sessions[session_index]["chunk_data_in_progress_count"], SessionError.NO_ERROR
        else:
            return None, SessionError.INVALID_INDEX

    def get_sender_pid(self, session_index):
        # Check if the session index is valid
        if 0 <= session_index < MAX_SESSIONS:
            return self.sessions[session_index]["sender_pid"], SessionError.NO_ERROR
        else:
            return None, SessionError.INVALID_INDEX

    def set_sender_pid(self, session_index, value):
        # Check if the session index is valid
        if 0 <= session_index < MAX_SESSIONS:
            self.sessions[session_index]["sender_pid"] = value
            return SessionError.NO_ERROR
        else:
            return SessionError.INVALID_INDEX

    def push_chunk_data(self, session_index, chunk):
        # Check if the session index is valid
        if 0 <= session_index < MAX_SESSIONS:
            # Append the chunk to the chunk_data list
            self.sessions[session_index]["chunk_data"].append(chunk)
            return SessionError.NO_ERROR
        else:
            return SessionError.INVALID_INDEX

    def pop_chunk_data(self, session_index):
        # Check if the session index is valid
        if 0 <= session_index < MAX_SESSIONS:
            # Pop the first chunk from the chunk_data list
            if self.sessions[session_index]["chunk_data"]:
                return self.sessions[session_index]["chunk_data"].pop(0), SessionError.NO_ERROR
            else:
                return None, SessionError.NO_ERROR  # Return None if chunk_data is empty
        else:
            return None, SessionError.INVALID_INDEX

    def peek_chunk_data(self, session_index):
        # Check if the session index is valid
        if 0 <= session_index < MAX_SESSIONS:
            # Peek at the first chunk in the chunk_data list
            if self.sessions[session_index]["chunk_data"]:
                return self.sessions[session_index]["chunk_data"][0], SessionError.NO_ERROR
            else:
                return None, SessionError.NO_ERROR  # Return None if chunk_data is empty
        else:
            return None, SessionError.INVALID_INDEX

    def set_latest_tokens(self, session_index, tokens):
        # Check if the session index is valid
        if 0 <= session_index < MAX_SESSIONS:
            # Set the latest_tokens for the session
            self.sessions[session_index]["latest_tokens"] = tokens
            return SessionError.NO_ERROR
        else:
            return SessionError.INVALID_INDEX

    def get_latest_tokens(self, session_index):
        # Check if the session index is valid
        if 0 <= session_index < MAX_SESSIONS:
            # Get the latest_tokens for the session
            return self.sessions[session_index]["latest_tokens"], SessionError.NO_ERROR
        else:
            return None, SessionError.INVALID_INDEX

    def get_session_id_from_sender_id(self, sender_id):
        for session_id, entry in enumerate(self.sessions):
            if entry["sender_id"] == sender_id:
                return session_id, SessionError.NO_ERROR

        return None, SessionError.INVALID_SENDER_ID  # Sender ID not found
    def get_sender_id_from_pid_uid(self, sender_pid, sender_uid):
        # Convert sender_pid and sender_uid to hexadecimal strings
        sender_pid_hex = hex(sender_pid)[2:]  # [2:] to remove the '0x' prefix
        sender_uid_hex = hex(sender_uid)[2:]

        # Combine the hexadecimal strings with an underscore
        return f'{sender_pid_hex}_{sender_uid_hex}'

    def save_caption(self, session_index, chunk_index, chunk_caption):
        self.sessions[session_index]["caption_buffer"][chunk_index] = chunk_caption
    
    def pop_caption(self, session_index, chunk_index):
        return self.sessions[session_index]["caption_buffer"].pop(chunk_index, None)

    def get_next_chunk_index(self, session_index):
        caption_buffer = self.sessions[session_index]["caption_buffer"]
        if caption_buffer:
            return min(caption_buffer.keys())
        else:
            return None

    def get_caption_count(self, session_index):
        return len(self.sessions[session_index]["caption_buffer"])

    def increase_worker_chunk_data_send_count(self, session_index):
        self.sessions[session_index]["worker_chunk_data_send_count"] += 1
        return self.sessions[session_index]["worker_chunk_data_send_count"]

    def increase_client_caption_send_count(self, session_index):
        self.sessions[session_index]["client_caption_send_count"] += 1
        return self.sessions[session_index]["client_caption_send_count"]

    def get_worker_chunk_data_send_count(self, session_index):
        return self.sessions[session_index]["worker_chunk_data_send_count"]

    def get_client_caption_send_count(self, session_index):
        return self.sessions[session_index]["client_caption_send_count"]

    def get_active_session_count(self):
        return self.active_session_count


