# waiUtil.py : Whisper daemon unitity codes.

import pika
import sys
import time
import random
import numpy as np
import os
import logging

import waiMsg_pb2
import waiWorkerMsg_pb2
from waiConstants import FFMPEG_TIMEBASE
# Set this variable to True to enable logging.info statements, and False to disable them
enable_print_statements = False 

def delete_file(file_path):
     try:
          os.remove(file_path)
          if enable_print_statements:
            logging.info(f"File '{file_path}' deleted successfully.")
     except OSError as e:
          # Handle the case when the file doesn't exist or there's a permission error
          logging.error(f"Error deleting file '{file_path}': {e}")

def flush_queue(channel, queue_name):
    while True:
        method_frame, header_frame, body = channel.basic_get(queue=queue_name, auto_ack=True)
        if method_frame:
            # If there is a message, acknowledge and discard it
            logging.info(f"Flushing MQ: {queue_name}")
        else:
            # No more messages in the queue
            break

def delete_queue(queue_name):
    connection = pika.BlockingConnection(pika.ConnectionParameters('localhost'))
    channel = connection.channel()

    # Delete the queue
    channel.queue_delete(queue=queue_name)

    connection.close()

def get_sender_id_from_pid_uid(sender_pid, sender_uid):
    # Convert sender_pid and sender_uid to hexadecimal strings
    sender_pid_hex = hex(sender_pid)[2:]  # [2:] to remove the '0x' prefix
    sender_uid_hex = hex(sender_uid)[2:]

    # Combine the hexadecimal strings with an underscore
    return f'{sender_pid_hex}_{sender_uid_hex}'

wai_msg_string = ["OPEN_SESSION","CLOSE_SESSION","CHUNK","EOS"]

def print_wai_msg(header, wai_msg):
    if isinstance(wai_msg, waiMsg_pb2.waiCommandMsg):
        sender_id = get_sender_id_from_pid_uid(wai_msg.sender_pid, wai_msg.sender_uid)
        hdr = f"{header}: S={sender_id}, CMD_T={wai_msg_string[wai_msg.msg_type]}"
        if wai_msg.msg_type == waiMsg_pb2.waiMsgType.CHUNK:     
            logging.info(f"{hdr}, pts:{wai_msg.chunk_pts}")
        else:
            logging.info(hdr)
    elif isinstance(wai_msg, waiMsg_pb2.waiResponseMsg):
        if wai_msg.msg_type == waiMsg_pb2.waiMsgType.CHUNK:     
            #logging.info(f"{header} ({wai_msg.pts}, {wai_msg.len}) SUB:{wai_msg.caption_return}")
            logging.debug(f"Caption: {wai_msg.caption_return}, {header} ({wai_msg.pts}, {wai_msg.len})")
        else:
            logging.info(f"{header}: REST_T={wai_msg_string[wai_msg.msg_type]}")
    elif isinstance(wai_msg, waiWorkerMsg_pb2.waiWorkerCommandMsg):
        logging.info(f"{header}: S={wai_msg.session_id}, W_CMD, chunk_index:{wai_msg.chunk_index}, chunk_offset:{wai_msg.chunk_offset}, chunk_len:{len(wai_msg.chunk_data)/FFMPEG_TIMEBASE:.2f}s")
        logging.debug(f"\ttokens: {wai_msg.tokens}")
    elif isinstance(wai_msg, waiWorkerMsg_pb2.waiWorkerResponseMsg):
        logging.info(f"{header}: S={wai_msg.session_id}, W_RESP from worker_{wai_msg.worker_id} chunk_index:{wai_msg.chunk_index}, chunk_offset:{wai_msg.chunk_offset}, chunk_len:{wai_msg.chunk_len/FFMPEG_TIMEBASE:.2f}s")
        logging.debug(f"\ttokens: {wai_msg.tokens}")
        logging.debug(f"\tcaption:{wai_msg.caption_return}\n")
    else:
        logging.info(f"Unknown message type: {type(wai_msg)}")

def delete_old_log_files(log_directory, max_total_size):
    # Get a list of all files in the directory
    log_files = os.listdir(log_directory)

    # Sort log files by creation time (oldest first)
    log_files.sort(key=lambda x: os.path.getctime(os.path.join(log_directory, x)))

    total_size = sum(os.path.getsize(os.path.join(log_directory, f)) for f in log_files)
    
    logging.info(f'Current log directory size {total_size}')
    logging.debug(f'Current log files {log_files}')

    if total_size > max_total_size:
        # Delete oldest log files until total size is less than (max_total_size * 0.9)
        while total_size > (max_total_size * 0.9):
            if not log_files:
                break
            file_to_delete = log_files.pop(0)
            file_path = os.path.join(log_directory, file_to_delete)
            total_size -= os.path.getsize(file_path)
            logging.info(f'Removing {file_path}, total_size {total_size}')
            os.remove(file_path)

