# waiWorker.py : Whisper daemon worker code
import pika
import sys
import time
import logging
import os
from daemonize import Daemonize
from logging.handlers import TimedRotatingFileHandler
from logging.handlers import WatchedFileHandler


import waiUtil
import waiWhisperUtil
from waiConstants import WORKER_INBOX
from waiConstants import MODEL_WARMUP_FILE
from waiConstants import LOG_FILE_SIZE_LIMIT 
from waiConstants import LOG_DIR_SIZE_CHECK_INTERVAL

# Set this variable to True to enable print statements, and False to disable them
enable_print_statements = False
worker_inbox_name = f'{WORKER_INBOX}'

def create_connection():
    # Connection parameters
    parameters = pika.ConnectionParameters(
        host='172.17.0.1',
        port=5672,
        virtual_host='/',
        credentials=pika.credentials.PlainCredentials('ni_whisper', 'logan'),
        heartbeat=60,
        connection_attempts=3,
        retry_delay=5
    )

    # Create a connection
    connection = pika.BlockingConnection(parameters)
    return connection

def main(log_directory):
    worker_id = int(sys.argv[1])
    logging.info(f'worker_{worker_id} start....')
    
    waiWhisperUtil.loadDefaultWhisperModel(worker_id)
    waiWhisperUtil.warmUpWhisperModel(MODEL_WARMUP_FILE)
    waiWhisperUtil.setupConnection()

    # Create the connection outside of the loop
    #connection = create_connection()
    connection = pika.BlockingConnection(pika.ConnectionParameters('localhost'))
    # Create the channel outside of the loop
    channel = connection.channel()
    last_log_dir_size_check_time = 0

    try:
        while True:
            try:
                # delete old log files if necessary
                if worker_id == 0:
                    current_time = time.time()

                    # Check log dir size periodically
                    if current_time - last_log_dir_size_check_time >= LOG_DIR_SIZE_CHECK_INTERVAL:
                        # delete old log files if necessary
                        waiUtil.delete_old_log_files(log_directory, LOG_FILE_SIZE_LIMIT)  # Max total size in bytes (5G)
                        last_log_dir_size_check_time = current_time

                # Try to get a message from the queue
                method_frame, header_frame, body = channel.basic_get(queue=worker_inbox_name, auto_ack=True)

                if method_frame:
                    # Process the message if one is received
                    waiWhisperUtil.workerCallback(body)
                else:
                    # no chunk data waiting in the queue, we can sleep.
                    time.sleep(0.1)

            except (pika.exceptions.AMQPConnectionError, pika.exceptions.ChannelClosedByBroker) as e:
                logging.error(f"Worker_{worker_id}: {worker_inbox_name} Connection failed. Retrying in 2 seconds...")
                logging.error(f"Error details: {e}")
                time.sleep(2)
                # Attempt to reconnect
                connection = pika.BlockingConnection(pika.ConnectionParameters('localhost'))
                #connection = create_connection()
                channel = connection.channel()

    except KeyboardInterrupt:
        # Close the channel and connection on exit
        channel.close()
        connection.close()
        logging.info(f'Worker_{worker_id} stopped....')


if __name__ == '__main__':
    worker_id = int(sys.argv[1])

    # Define log directory path
    log_directory = os.path.join(os.getcwd(), 'logs')

    # Define log file path
    logfile = os.path.join(log_directory, f'waiWorker.log')
    with open(logfile, 'w') as f:
        pass
    # For parent host read
    os.chmod(logfile, 0o0644)

    # Daemonize the process
    pid = f"/var/run/waiWorker_{worker_id}.pid"
    action = "start"
    daemon = Daemonize(app=f"waiWorker_{worker_id}", pid=pid, action=action)
    daemon.start()

    # Redirect stdout and stderr to the log file
    sys.stdout = open(logfile, 'a')
    sys.stderr = open(logfile, 'a')

    # Initialize logger with TimedRotatingFileHandler for worker 0 only, other will use WatchedFileHandler
    if worker_id == 0:
        handler = TimedRotatingFileHandler(logfile, when="H", interval=1)
    else :
        handler = WatchedFileHandler(logfile)
    handler.setFormatter(logging.Formatter('%(asctime)s - %(levelname)s - %(message)s'))
    logger = logging.getLogger()
    logger.addHandler(handler)
    logger.setLevel(logging.INFO)

    # Set the desired logging level for Pika
    logging.getLogger('pika').setLevel(logging.WARNING)
    
    # Log message
    logger.info(f"Worker_{worker_id} Daemon started successfully")


    # Call the main function
    main(log_directory)
