#!/usr/bin/env python3

# This script will be called by docker HEALTHCHECK
# This script will return 0 when healthy, 1 when unhealthy.
# Prior to whisper model warm-up, report status as unhealthy.
# Write updates to /workspace/whisper/logs/waiAdmin.log

from datetime import datetime
import subprocess
import re
import os
from datetime import datetime, timedelta

def wais_procs_running():
    # check waiServer and waiWorker processes are present
    res = subprocess.run("ps aux | grep -v grep | grep -q 'waiServer.py'",
                         capture_output=True, text=True, shell=True)
    if res.returncode:
        return False
    res = subprocess.run("ps aux | grep -v grep | grep -q 'waiWorker.py'",
                         capture_output=True, text=True, shell=True)
    if res.returncode:
        return False
    return True

def find_last_log(file_path, re_str) -> str:
    # read waiServer.log or waiWorker.log in reverse to find last line with
    # occurence of a regex string
    res = subprocess.run("tac " + file_path + " | grep -Ph -m 1 '" + re_str +
                         "'", capture_output=True, text=True, shell=True)
    if not res.returncode:
        return res.stdout
    else:
        return None

def get_log_line_ts(log_str) -> datetime:
    match = re.search(r'\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2},\d{3}', log_str.split(' - ')[0])
    if match:
        return datetime.strptime(match.group(0) + "000", "%Y-%m-%d %H:%M:%S,%f")
    else:
        print("Error - failed to parse timestamp from line: " + log_str)
        return None

def wais_warmup_complete(waiWorker_indices, warmup_start_time):
    for i in waiWorker_indices:
        w_log = find_last_log("/workspace/whisper/logs/waiWorker.log",
                              "Worker_{} model warmup completed".format(i))
        if w_log:
            #warmup is faster now, since we don't need to dl the model file.
            if get_log_line_ts(w_log) > (warmup_start_time - timedelta(seconds=10)):
                continue
            else:
                return False
        else:
            return False
    return True

# MAIN -------------------------------------------------------------------------
script_name = os.path.basename(__file__)

log_prev_state = find_last_log("/workspace/whisper/logs/waiAdmin.log",
                               " state: ")
if not log_prev_state:
    print("Error - no state messages in waiAdmin.log yet")
    exit(1)

if "state: starting" in log_prev_state:
    print("Just started, not yet begining warm up")
    exit(1)
elif "state: warm up" in log_prev_state:
    if not wais_procs_running():
        print("Error, could not find waiServer and/or waiWorker processes")
        exit(1)

    log_docker_start = find_last_log("/workspace/whisper/logs/waiAdmin.log",
                                     " state: starting")
    if not log_prev_state:
        print("Error, could not find docker start time in waiAdmin.log")
        exit(1)

    log_wai_workers = find_last_log("/workspace/whisper/logs/waiAdmin.log",
                                    " started \d+ waiWorkers")
    if not log_prev_state:
        print("Error, could not find number of waiWorkers started in "
              "waiAdmin.log")
        exit(1)

    if get_log_line_ts(log_docker_start) <= \
       get_log_line_ts(log_wai_workers) <= \
       get_log_line_ts(log_prev_state):
        waiWorker_indices = range(int(re.search(r" started (\d+) waiWorkers",
                                                log_wai_workers).group(1)))
    else:
        print("Error, in warm up state but waiAdmin.log has unexpected "
              "messages order")
        exit(1)

    if wais_warmup_complete(waiWorker_indices, get_log_line_ts(log_prev_state)):
        with open("/workspace/whisper/logs/waiAdmin.log", 'a') as file:
            file.write(datetime.now().strftime("%Y-%m-%d %H:%M:%S,%f")[:-3] +
                       " - INFO - " + script_name + " state: healthy\n")
        print("Healthy state")
        exit(0)
    else:
        print("Still in warm up ({}s)"\
              .format((datetime.now() - get_log_line_ts(log_prev_state))\
                      .total_seconds()))
        exit(1)
elif "state: healthy" in log_prev_state:
    if not wais_procs_running():
        with open("/workspace/whisper/logs/waiAdmin.log", 'a') as file:
            file.write(datetime.now().strftime("%Y-%m-%d %H:%M:%S,%f")[:-3] +
                       " - INFO - " + script_name + " state: processes down\n")
        print("Error, could not find waiServer and/or waiWorker processes")
        exit(1)
    # TODO: query waiServer connection
    print("Healthy state")
    exit(0)
elif "state: processes down" in log_prev_state:
    if wais_procs_running():
        with open("/workspace/whisper/logs/waiAdmin.log", 'a') as file:
            file.write(datetime.now().strftime("%Y-%m-%d %H:%M:%S,%f")[:-3] +
                       " - INFO - " + script_name + " state: healthy\n")
        print("Healthy state")
        exit(0)
    else:
        print("Error, could not find waiServer and/or waiWorker processes")
        exit(1)
elif "state: start up failed" in log_prev_state:
    print("Error, docker start up failed")
    exit(1)
else:
    print("Error, unknown state. Add to waiHealthCheck.py")
    exit(1)

exit(1)
