#!/usr/bin/env python3

# Copyright (c) 2024 NETINT Technologies Inc.
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

from pathlib import Path
from string import Template
from typing import Dict

import argparse
import getpass
import ipaddress
import json
import os
import platform
import re
import shlex
import shutil
import socket
import subprocess
import sys
import tarfile
import time
import urllib.request
import zipfile


have_prereqs = True
for tool in ["docker", "docker compose", "groups", "install", "nvme", "sysctl", "systool"]:
    if tool == "docker compose":
        if shutil.which("docker") and subprocess.run(["docker", "compose", "version"], capture_output=True).returncode:
            have_prereqs = False
            print("\033[31m[ WARNING ]:\033[96m \033[0mDocker Compose v2\033[96m missing! Please install the Docker plugin.\033[0m", file=sys.stderr)
    elif not shutil.which(tool):
        have_prereqs = False
        print(f"\033[31m[ WARNING ]:\033[0m {tool}\033[96m missing! Please install the corresponding package.\033[0m", file=sys.stderr)
        if tool == "docker":
            print("\033[31m[ WARNING ]:\033[96m Please install the \033[0mDocker Compose v2\033[96m plugin as well.\033[0m", file=sys.stderr)
if not have_prereqs:
    sys.exit(1)

description = "NETINT Bitstreams Edge Quick Installer Script v2.9"

# Release-specific constants - update these for each new release 
INTERMEDIATE_VERSION = "v2.8.0"

docker = "docker"
empty = ""
etc = Path("/etc")
install = "install"
y = "y"
configs = "configs"
tar = ".tar"
tar_gz = tar + ".gz"
release_directory = Path("release")
configs_directory = release_directory.joinpath(configs)


enable_iommu = False
passthrough_exists = True
iommu_exists = True
def is_iommu_enabled(machine):
    global iommu_exists

    try:
        with open('/etc/default/grub', 'r') as f:
            grub_content = f.read()

        cmdline_linux = ""
        cmdline_default = ""
        any_iommu_param_found = False

        for line in grub_content.splitlines():
            if line.startswith('GRUB_CMDLINE_LINUX='):
                cmdline_linux = line
            elif line.startswith('GRUB_CMDLINE_LINUX_DEFAULT='):
                cmdline_default = line

        combined_cmdline = cmdline_linux + cmdline_default

        def get_iommu_patterns(machine):
            if machine == "arm64":
                return {
                    "all": ["arm64.iommu=off", "arm-smmu.disable_bypass=0", "arm64.iommu=on", "arm-smmu.disable_bypass=1, iommu=on, iommu=off"],
                    "disable": ["arm64.iommu=off", "arm-smmu.disable_bypass=1, iommu=off"],
                    "enable": ["arm64.iommu=on", "arm-smmu.disable_bypass=0, iommu=on"]
                }
            return {
                "all": ["amd_iommu=off", "intel_iommu=off", "iommu=0", "iommu=off",
                        "amd_iommu=on", "intel_iommu=on", "iommu=1", "iommu=on"],
                "disable": ["amd_iommu=off", "intel_iommu=off", "iommu=0", "iommu=off"],
                "enable": ["amd_iommu=on", "intel_iommu=on", "iommu=1", "iommu=on"]
            }

        patterns = get_iommu_patterns(machine)
        any_iommu_param_found = any(p in combined_cmdline for p in patterns["all"])

        if any(p in combined_cmdline for p in patterns["disable"]):
            return False
        if any(p in combined_cmdline for p in patterns["enable"]):
            return True

        if not any_iommu_param_found:
            iommu_exists = False

        return False

    except Exception as e:
        print(f"Error reading grub file: {e}\n"
              "IOMMU status cannot be properly determined, please manually verify IOMMU is enabled")
        iommu_exists = False
        return False


def is_iommu_passthrough_enabled(machine):
    global passthrough_exists
    try:
        with open('/etc/default/grub', 'r') as f:
            grub_content = f.read()
          
        cmdline_linux = ""
        cmdline_default = ""
        for line in grub_content.splitlines():
            if line.startswith('GRUB_CMDLINE_LINUX='):
                cmdline_linux = line
            elif line.startswith('GRUB_CMDLINE_LINUX_DEFAULT='):
                cmdline_default = line
                
        combined_cmdline = cmdline_linux + cmdline_default

        passthrough_strings = ["iommu.passthrough=1", "iommu=pt", "iommu.passthrough=on"]
        disable_strings = ["iommu.passthrough=0", "iommu.passthrough=off"]

        if any(s in combined_cmdline for s in disable_strings):
            return False
        if any(s in combined_cmdline for s in passthrough_strings):
            return True
        if not any(s in combined_cmdline for s in ["iommu.passthrough", "iommu=pt"]):
            passthrough_exists = False
            return False
                
        return False
        
    except Exception as e:
        print(f"Error reading grub file: {e}\n"
              "IOMMU passthrough status cannot be properly determined, please manually verify IOMMU passthrough is enabled")
        passthrough_exists = False
        return False


def get_current_boot_iommu_state(machine):
    try:
        with open('/proc/cmdline', 'r') as f:
            cmdline = f.read()

        if machine == "arm64":
            iommu_enabled = any(x in cmdline for x in ['arm64.iommu=on', 'arm-smmu.disable_bypass=0', 'iommu=1', 'iommu=on'])
            passthrough = any(x in cmdline for x in ['iommu=pt', 'iommu.passthrough=1', 'iommu.passthrough=on'])
        else:
            iommu_enabled = any(x in cmdline for x in ['intel_iommu=on', 'amd_iommu=on', 'iommu=1', 'iommu=on'])
            passthrough = any(x in cmdline for x in ['iommu=pt', 'iommu.passthrough=1', 'iommu.passthrough=on', 'intel_iommu=pt', "amd_iommu=pt"])

        return iommu_enabled and passthrough
    except Exception as e:
        print(f"Error reading cmdline file: {e}\n"
              "IOMMU and passthrough boot status cannot be properly determined")
        return False


def is_ubuntu24() -> bool:
    if hasattr(platform, 'freedesktop_os_release'):
        return platform.freedesktop_os_release().get("VERSION_ID") == "24.04"
    # If freedesktop_os_release is unavailable, assume that it's older than Ubuntu 24
    return False


def check_ports_availability():
    required_ports = {
        "ETCD": [2379, 2380],
        "Lms": [9094, 9095, 9101],
        "NVS": [9092, 9100],
        "Stream trans": [8057],
        "Gateway": [80, 443], 
        "Nginx": [8085],
        "MINIO": [9000, 9001],
        "Notification": [9096],
        "Mysql": [3306],
        "RabbitMQ": [5672, 15672, 25672, 61613, 61614, 1883],
        "Redis node": [6379],
        "Stream live": [1935, 8023, 8022, 10080],
    }

    def is_port_in_use(port: int) -> bool:
        with socket.socket(socket.AF_INET, socket.SOCK_STREAM) as sock:
            sock.settimeout(2)
            if sock.connect_ex(("localhost", port)) == 0:
                return True
        return False

    def get_process_info(port: int) -> str:
        """Gets process using port with multiple fallbacks"""
        # Try lsof first - most detailed info
        if shutil.which("lsof"):
            try:
                cmd = f"sudo lsof -i :{port}"
                result = run(cmd, stdout=subprocess.PIPE, universal_newlines=True)
                if result.stdout:
                    lines = result.stdout.strip().split('\n')
                    if len(lines) > 1:
                        process = lines[1].split()
                        if len(process) >= 2:
                            return f"{process[0]} (PID: {process[1]})"
            except (subprocess.SubprocessError, IndexError, subprocess.TimeoutExpired):
                pass
        # Try netstat as fallback
        if shutil.which("netstat"):
            try:
                result = subprocess.run(
                    f"sudo netstat -tulnp | grep ':{port} '",
                    shell=True,
                    check=True,
                    stdout=subprocess.PIPE,
                    universal_newlines=True
                )
                if result.stdout:
                    lines = result.stdout.strip().split('\n')
                    for line in lines:
                        parts = line.split()
                        if len(parts) >= 7:
                            return parts[6]
            except (subprocess.SubprocessError, IndexError, subprocess.TimeoutExpired):
                pass
        # Try ss as last resort
        if shutil.which("ss"):
            try:
                cmd = f"sudo ss -lptn sport = :{port}"
                result = subprocess.run(
                    cmd,
                    shell=True,
                    check=True,
                    stdout=subprocess.PIPE,
                    universal_newlines=True
                )
                if result.stdout:
                    lines = result.stdout.splitlines()
                    for line in lines:
                        if "users:" in line:
                            return line.split("users:")[-1].strip()
            except (subprocess.SubprocessError, IndexError, subprocess.TimeoutExpired):
                pass
                
        return "unknown process"

    conflicts = {}
    for service, ports in required_ports.items():
        for port in ports:
            if is_port_in_use(port):
                conflicts.setdefault(service, []).append(
                    (port, get_process_info(port))
                )

    if conflicts:
        print("\nWARNING: The following required ports are currently in use:")
        for service, port_info in conflicts.items():
            for port, process in port_info:
                print(f"{service} - Port {port} is in use by {process}")
        sys.exit("Please free these ports and run the installer again.")

    print("\nAll required ports are available.")


def check_docker_access():
    try:
        run("docker info", subprocess.PIPE, True)
        return True
    except subprocess.CalledProcessError:
        return False


def handle_docker_permissions():
    if check_docker_access():
        return True

    user = getpass.getuser()
    # Skip docker group membership check if running as root
    if user == "root":
        print("\nRunning as root - skipping Docker group membership check")
        return True
    
    try:
        run("getent group docker", subprocess.PIPE, True)
        groups_output = run("groups", subprocess.PIPE, True).stdout
        if "docker" not in groups_output:
            print(f"\nAdding user {user} to Docker group...")
            run(f"sudo usermod -aG docker {user}")
            print(
                "\nDocker access configured. Please log out, log back in, and rerun this script."
            )
            sys.exit(0)
    except subprocess.CalledProcessError:
        pass

    print("\nDocker access denied. Please ensure:")
    print("1. Docker is installed: sudo apt-get install docker-ce")
    print("2. Docker daemon is running: sudo systemctl start docker")
    print("3. You have logged out and back in after being added to docker group")
    sys.exit(1)


def check_scaling_governor(should_print=True):
    if should_print:
        print(
            f"\nChecking if scaling_governor is set to '{desired_value.rstrip()}' for all CPUs..."
        )
    for i in range(cpu_count):
        p = Path(f"/sys/devices/system/cpu/cpu{i}/cpufreq/scaling_governor")
        if not p.exists():
            print("WARNING: CPU settings cannot be set!")
            break
        with p.open() as cpu:
            if cpu.read() != desired_value:
                return False
    return True


def extract(f, dest_dir=release_directory):
    print(f"Extracting {f}...")
    if f.suffix == ".zip":
        with zipfile.ZipFile(f) as zf:
            zf.extractall(dest_dir)
    else:
        with tarfile.open(f) as tf:
            tf.extractall(dest_dir)


def load_images():
    for tar_archive in release_directory.glob("*.tar"):
        run(f"docker image load -i {tar_archive}")


def get_image_tags():
    image_tags = {}

    # Parse the output to extract image names and tags
    for line in run(
        "docker images --filter=reference='yyz1harbor01.netint.ca/bitstreams/*' --format '{{.Repository}}:{{.Tag}}'",
        subprocess.PIPE,
        True,
    ).stdout.splitlines():
        if line:
            name_tag = line.split(":")
            image_name = name_tag[0].split("/")[-1]
            image_tag = name_tag[1]
            image_tags[image_name] = image_tag
    return image_tags


def parse_release_version_from_tags():
    tag_pattern = re.compile(r"release_v(\d+)\.(\d+)\.(\d+)_RC\d+")
    image_tags = get_image_tags()
    for tag in image_tags.values():
        m = tag_pattern.fullmatch(tag)
        if m:
            return tuple(int(x) for x in m.groups())
    return None


def download_intermediate_release() -> Dict[str, Path]:
    """Download intermediate release and return paths to required files."""
    dest_dir = Path(f"intermediate_{INTERMEDIATE_VERSION}")
    dest_dir.mkdir(parents=True, exist_ok=True)
    zip_path = dest_dir / f"Bitstreams_Edge_{INTERMEDIATE_VERSION}_{arch_suffix}.zip"

    print(f"\nDownloading intermediate package {INTERMEDIATE_VERSION} from {INTERMEDIATE_URL}")
    try:
        urllib.request.urlretrieve(INTERMEDIATE_URL, zip_path)
    except Exception as e:
        raise Exception(f"Failed to download intermediate version {INTERMEDIATE_VERSION}: {e}")
    
    extract(zip_path, dest_dir)

    file_patterns = {
        'configs': 'configs.tar',
        'non_cluster': 'non_cluster.json',
        'docker_images': f"{INTERMEDIATE_VERSION.upper()}_*_docker_images_{machine}.tar.gz"
    }
    required_files = {}
    for key, pattern in file_patterns.items():
        matches = list(dest_dir.glob(f"**/{pattern}"))
        if not matches:
            raise FileNotFoundError(f"Missing {key} file ({pattern}) in {INTERMEDIATE_VERSION} package")
        required_files[key] = matches[0]
    print(f"Intermediate release {INTERMEDIATE_VERSION} downloaded and extracted successfully")
    return required_files


def run(s, stdout=None, universal_newlines=None, cwd=None):
    return subprocess.run(
        shlex.split(s),
        check=True,
        stdout=stdout,
        universal_newlines=universal_newlines,
        cwd=cwd,
    )


def stop_services():
    handle_docker_permissions()
    try:
        result = run("docker container ls -a --filter name=configs- --format '{{.ID}}'", subprocess.PIPE, True)
        container_ids = [cid for cid in result.stdout.splitlines() if cid]
        if container_ids:
            print("\nStopping and removing Bitstreams containers...")
            run("docker container rm -f " + " ".join(container_ids))
            time.sleep(2)
    except subprocess.CalledProcessError as e:
        print(f"Warning: Error stopping containers: {e}")


def wprint(s):
    print(f"WARNING: {s}!", file=sys.stderr)


def yninput(p):
    if args.y:
        return True
    return input(f"{p}? options: y/n [default: n] ") == y


def yninput2(p):
    if args.y:
        return True
    response = input(f"{p}? options: Y/n [default: Y] ").lower().strip()
    return response in ['y', 'yes', ''] or not response


def prepare_configs(image_tags, non_cluster_json_path: Path):
    print("Image tags")
    for image_name, image_tag in image_tags.items():
        print(f"{image_name}:{image_tag}")

    try:
        dns = socket.gethostbyaddr(ipv4_address)[0].split(".")[0]
    except socket.herror as _:
        print("Warning: DNS lookup failed, falling back to IP address")
        dns = ipv4_address

    host_info = {
        "CURRENT_NODE_IP": ipv4_address,
        "Node_1_DNS": dns if args.use_dns_lookup else ipv4_address,
        "LMS_IMAGE_TAG": image_tags["lms"],
        "NVS_IMAGE_TAG": image_tags["nvs"],
        "STREAM_TRANS_IMAGE_TAG": image_tags["stream-trans"],
        "NOTIFICATION_IMAGE_TAG": image_tags["notification"],
        "STREAM_LIVE_SRS_IMAGE_TAG": image_tags["stream-live"],
        "GATEWAY_IMAGE_TAG": image_tags["gateway"],
        "EXT_PROTOCOL": args.protocol,
        "GATEWAY_PUBLIC_PORT": f":{args.gateway_port}" if args.gateway_port else empty,
        "SRT_PUBLIC_PORT": f"{args.srt_port}" if args.srt_port else empty,
        "RTMP_PUBLIC_PORT": f":{args.rtmp_port}" if args.rtmp_port else empty,
    }

    with non_cluster_json_path.open() as f:
        contents = f.read()
    with open("non_cluster_dirty.json", "w") as f:
        f.write(Template(contents).substitute(host_info))

    # Load the dictionary from non_cluster.json
    with open("non_cluster_dirty.json") as json_file:
        d = json.load(json_file)
    os.remove("non_cluster_dirty.json")

    for filepath in [
        ".env",
        "config/gateway/conf.d/api.conf",
        "config/gateway/lua/common_init.lua",
        "config/nginx/conf.d/default.conf",
        "config/stream-live/srs.conf",
        "config/etcd/config.yml",
        "config/lms/config.toml",
        "config/nvs/config.toml",
        "config/stream-trans/config.toml",
        "config/notification/config.toml",
    ]:
        file = configs_directory.joinpath(filepath)
        with file.open() as f:
            contents = f.read()
        with file.open("w") as f:
            f.write(Template(contents).substitute(d))

    result = run("sudo nvme list -o json", subprocess.PIPE, True).stdout
    if result and any(device["ModelNumber"] == "QuadraT1M" for device in json.loads(result)["Devices"]):
        print("Reduce concurrent trans task count to 3 as the set up has T1M in use")
        config_file = configs_directory.joinpath("config/stream-trans/config.toml")
        with config_file.open() as f:
            config_content = f.readlines()
        with config_file.open("w") as f:
            for config_line in config_content:
                if "target_concurrent_trans_task_count" in config_line:
                    f.write("    target_concurrent_trans_task_count = 3\n")
                else:
                    f.write(config_line)


def install_from_archives(archive_files: Dict[str, Path], ipv4_address: str, force_overwrite: bool = False):
    configs_archive_path = archive_files['configs']
    docker_images_archive_path = archive_files['docker_images']
    non_cluster_json_path = archive_files['non_cluster']
    print(f"\nInstalling from:\n- Configs: {configs_archive_path}\n- Docker images: {docker_images_archive_path}")
    if release_directory.is_dir() and any(release_directory.glob("*.tar")):
        if force_overwrite or args.y or args.overwrite_release_directory:
            response = y
        elif args.keep_release_directory:
            response = empty
        else:
            response = (
                y if yninput(f"Overwrite the {release_directory} directory") else empty
            )
        if response == y:
            run(f"sudo rm -rf {release_directory}")
            extract(configs_archive_path)
            extract(docker_images_archive_path)
        elif configs_directory.exists():
            run(f"sudo rm -rf {configs_directory}")
            extract(configs_archive_path)
    else:
        extract(configs_archive_path)
        extract(docker_images_archive_path)

    if args.protocol == "https":
        run(f"bash gencert.sh {ipv4_address}", cwd=configs_directory.joinpath("cert"))

    stop_services()
    
    # Remove images using Docker filter
    try:
        result = run("docker images --filter=reference='yyz1harbor01.netint.ca/bitstreams/*' --format '{{.ID}}'", subprocess.PIPE, True)
        image_ids = [img_id for img_id in result.stdout.splitlines() if img_id]
        if image_ids:
            print("\nRemoving existing Bitstreams images")
            run("docker image rm -f " + " ".join(image_ids))
    except subprocess.CalledProcessError as e:
        print(f"Warning: Error removing images: {e}")
    
    print("\nLoading new Bitstreams images")
    load_images()

    image_tags = get_image_tags()
    prepare_configs(image_tags, non_cluster_json_path)

    print("\nStarting Bitstreams services...")
    try:
        run("docker compose up -d --pull never", cwd=configs_directory)
    except subprocess.CalledProcessError:
        sys.exit("Failed to start Bitstreams services...")


system = platform.system()
if system != "Linux":
    sys.exit(f"{system} not supported")

p = run("dpkg --print-architecture", subprocess.PIPE, True)
machine = p.stdout.rstrip()
if machine not in ("amd64", "arm64"):
    raise ValueError(f"Unsupported machine architecture: {machine}")

# Create intermediate release URL based on architecture
arch_suffix = {"amd64": "x86", "arm64": "arm"}[machine]
INTERMEDIATE_URL = f"https://releases.netint.com/edge/5F4HFJPAPWM39HP/Bitstreams_Edge_{INTERMEDIATE_VERSION}_{arch_suffix}.zip"

parser = argparse.ArgumentParser(description=description)
parser.epilog = (
   "Required ports:\n"
   "  ETCD: 2379, 2380\n" 
   "  Lms: 9094, 9095, 9101\n"
   "  NVS: 9092, 9100\n"
   "  Stream trans: 8057\n"
   "  Gateway: 80, 443\n"
   "  Nginx: 8085\n" 
   "  MINIO: 9000, 9001\n"
   "  Notification: 9096\n"
   "  Mysql: 3306\n"
   "  RabbitMQ: 5672, 15672, 25672, 61613, 61614, 1883\n"
   "  Redis node: 6379\n"
   "  Stream live 1935, 8023, 8022, 10080\n\n"
   "These ports must be available for Bitstreams services to start properly.\n"
   "Run: ./edge_quick_installer.py --help"
)
parser.add_argument(
    f"-{y}",
    action="store_true",
    help="answer yes to all yes/no prompts (supersedes all other arguments)",
)
parser.add_argument("ipv4Addr", type=ipaddress.IPv4Address)
for k, v in {"rtmp": 0, "srt": 10080, "gateway": 0}.items():
    parser.add_argument(
        f"--{k}-port", type=int, default=v, help=f"default: {v}. 0 means unspecified."
    )

for arg in [
    "overwrite-release-directory",
    "keep-release-directory",
    "skip-firmware-upgrade",
]:
    parser.add_argument(f"--{arg}", action="store_true")
parser.add_argument("--fail-if-no-iommu", action="store_true", 
                   help="Exit if IOMMU or passthrough not enabled instead of prompting")
parser.add_argument("protocol", choices=["http", "https"])
parser.add_argument("--use-dns-lookup", action="store_true", 
                   help="Use DNS lookup instead of IP address for region")

args = parser.parse_args()

print(description + "\n")

# Capture target node IP early for downstream functions
ipv4_address = str(args.ipv4Addr)


if is_ubuntu24():
    print("Detected Ubuntu 24.04 - proceeding with installation.")
else:
    print("\033[31m[ WARNING ]:\033[96m Bitstreams' target OS is Ubuntu 24.04 Server.\033[0m")
    if not yninput("Do you want to continue the installation anyway"):
        sys.exit("Installation aborted...")
    print("Continuing installation...\n")


iommu_enabled = is_iommu_enabled(machine)
passthrough_enabled = is_iommu_passthrough_enabled(machine)
current_boot_enabled = get_current_boot_iommu_state(machine)

if not current_boot_enabled:
    if args.fail_if_no_iommu:
        issues = []
        if iommu_enabled and passthrough_enabled:
            sys.exit("ERROR: IOMMU and passthrough enabled but not active, need to manually reboot system for automated testing.")
        if not iommu_enabled:
            issues.append("IOMMU")
        if not passthrough_enabled:
            issues.append("passthrough")
        sys.exit(f"ERROR: {' and '.join(issues)} must be enabled for automated testing. Run script manually to enable them.")
    elif not iommu_enabled or not passthrough_enabled:
        if yninput2("IOMMU is not enabled. Enable IOMMU for improved performance"):
            enable_iommu = True
            print("Enabling IOMMU...\n")
            try:
                with open('/etc/default/grub', 'r') as f:
                    lines = f.readlines()
            except Exception:
                print("Error reading grub file, IOMMU and passthrough cannot be detected properly. Please enable both manually.")
                lines = []
                
            def update_params(line, enable_iommu=False, enable_passthrough=False):
                if not line.startswith(('GRUB_CMDLINE_LINUX=', 'GRUB_CMDLINE_LINUX_DEFAULT=')):
                    return line
                
                var_name, params = line.split('=', 1)
                params = params.strip().strip('"\'')
                
                new_params = []
                params_updated = False
                
                # Process existing parameters
                for param in params.split():
                    if machine == "arm64":
                        iommu_disable_params = {
                            'arm64.iommu=off': 'arm64.iommu=on',
                            'arm-smmu.disable_bypass=1': 'arm-smmu.disable_bypass=0', 
                            'iommu=0': 'iommu=1',
                            'iommu=off': 'iommu=on'
                        }
                        if enable_iommu and param in iommu_disable_params:
                            new_params.append(iommu_disable_params[param])
                            params_updated = True
                        elif enable_passthrough and param == 'iommu.passthrough=0':
                            new_params.append('iommu.passthrough=1')
                            params_updated = True
                        elif enable_passthrough and param == 'iommu.passthrough=off':
                            new_params.append('iommu.passthrough=on')
                            params_updated = True
                        else:
                            new_params.append(param)
                            
                    elif machine == "amd64":
                        if enable_iommu and param in ['amd_iommu=off', 'intel_iommu=off', 'iommu=0', 'iommu=off']:
                            new_params.append(param.replace('off','on').replace('0','1'))
                            params_updated = True
                        elif enable_passthrough and param == 'iommu.passthrough=0':
                            new_params.append('iommu.passthrough=1')
                            params_updated = True
                        elif enable_passthrough and param == 'iommu.passthrough=off':
                            new_params.append('iommu.passthrough=on')
                            params_updated = True
                        else:
                            new_params.append(param)
                            
                # Add missing parameters if needed
                if not params_updated:
                    if machine == "arm64":
                        if enable_iommu:
                            new_params.extend(['arm64.iommu=on', 'arm-smmu.disable_bypass=0'])
                        if enable_passthrough:
                            new_params.append('iommu.passthrough=1')
                    elif machine == "amd64":
                        if enable_iommu:
                            new_params.extend(['intel_iommu=on', 'amd_iommu=on', 'iommu=1', 'iommu=on'])
                        if enable_passthrough:
                            new_params.append('iommu=pt')

                # Add passthrough parameter if it doesn't exist and IOMMU is enabled
                if not passthrough_exists and enable_iommu and var_name == 'GRUB_CMDLINE_LINUX':
                    if machine == "arm64":
                        new_params.append('iommu.passthrough=1')
                    elif machine == "amd64":
                        new_params.append('iommu=pt')

                if not iommu_exists and enable_iommu and var_name == 'GRUB_CMDLINE_LINUX':
                    if machine == "arm64":
                        new_params.extend(['arm64.iommu=on', 'arm-smmu.disable_bypass=0'])
                    elif machine == "amd64":
                        new_params.extend(['intel_iommu=on', 'amd_iommu=on', 'iommu=1', 'iommu=on'])

                return f'{var_name}="{" ".join(new_params)}"\n'

            if lines is not None:
                new_lines = [
                    update_params(line, 
                                enable_iommu=not iommu_enabled,
                                enable_passthrough=not passthrough_enabled)
                    for line in lines
                ]

                temp_grub = 'grub_temp'

                try:
                    with open(temp_grub, 'w') as f:
                        f.writelines(new_lines)
                    try:
                        run(f'sudo mv {temp_grub} /etc/default/grub')
                        run('sudo update-grub')
                    except subprocess.CalledProcessError as e:
                        print(f"Error updating GRUB configuration: {e}"
                            "IOMMU/passthrough not updated, please manually configure grub bootloader file to enable both")
                except IOError as e:
                    print(f"Error writing GRUB configuration: {e}\n"
                        "IOMMU/passthrough not updated, please manually configure grub bootloader file to enable both")

        else:
            print("Warning: Continuing without IOMMU. Server performance may be impacted")
            print("To enable IOMMU, either:\n"
                """1. manually edit /etc/default/grub, then run in terminal "sudo update-grub2" followed by "sudo reboot"\n"""
                "2. run this script again and select y to enable IOMMU")
    
    else:
        print("IOMMU and IOMMU passthrough enabled - pending reboot")
        print("Warning: IOMMU configuration exists but isn't active.")
        print("To enable IOMMU, either:\n"
              """1. Wait for reboot prompt at end of script\n"""
              """2. After script completion, run "sudo reboot" """)

else:
    print("IOMMU and IOMMU passthrough enabled and active")

print("\nChecking for any running Bitstreams services...")
stop_services()

print("\nChecking required ports availability...")
check_ports_availability()

configs_archive = Path(configs + tar)
if not configs_archive.exists():
    sys.exit("No configs archive found in current directory!")

version_pattern = r"V(\d+).(\d+).(\d+)_\w+"
docker_images_regexp = re.compile(
    version_pattern + f"_docker_images_{machine}" + re.escape(tar_gz)
)

docker_images_archives = []

for x in Path.cwd().iterdir():
    if n := docker_images_regexp.fullmatch(x.name):
        docker_images_archives.append(n)

if not docker_images_archives:
    sys.exit("No docker images package archive found in current directory!")

docker_images_archives.sort(key=lambda m: m.string, reverse=True)
print("\nSelecting most recent versions of the package archives...")
selected_match = docker_images_archives[0]
docker_images_archive = selected_match.string
target_version = tuple(int(x) for x in selected_match.groups())
print(f"Selected {docker_images_archive}")

# Version validation for v2.9.x installer - only supports specific upgrade paths
current_version = parse_release_version_from_tags()
intermediate_files = None

if current_version:
    current_major, current_minor, _ = current_version
    print(f"\nCurrent version: v{current_version[0]}.{current_version[1]}.{current_version[2]}")
    print(f"Target version: v{target_version[0]}.{target_version[1]}.{target_version[2]}")
    
    # Check if upgrade is supported
    if current_major != 2 or current_minor not in (7, 8, 9) or target_version[0] != 2 or target_version[1] != 9:
        sys.exit("Error: This v2.9.x installer only supports upgrades to v2.9.x from v2.7.x or v2.8.x")
    
    if current_minor == 7:
        # v2.7.x → v2.9.x (needs intermediate v2.8.0)
        print(f"Upgrade from v{current_version[0]}.{current_version[1]}.{current_version[2]} → {INTERMEDIATE_VERSION} → v{target_version[0]}.{target_version[1]}.{target_version[2]}")
        try:
            intermediate_files = download_intermediate_release()
        except Exception as e:
            sys.exit(f"Error: {e}\nCannot proceed without intermediate release {INTERMEDIATE_VERSION} for v2.7.x → v2.9.x upgrade")
    elif current_minor == 8:
        # v2.8.x → v2.9.x (direct upgrade)
        print(f"Direct upgrade: v{current_version[0]}.{current_version[1]}.{current_version[2]} → v{target_version[0]}.{target_version[1]}.{target_version[2]}")
else:
    print("\nNo existing version detected - performing fresh installation")

# Firmware upgrade and system tuning prior to deployment
needs_quadra_reboot = False # flag to reboot after firmware loader version changed
if not args.skip_firmware_upgrade:
    quadra_firmware_archive_prefix = "Quadra_FW_V"
    quadra_firmware_archive_version = "5.4.3"
    quadra_firmware_archive_rc_version = "RC1"
    quadra_firmware_archive = (
        quadra_firmware_archive_prefix
        + quadra_firmware_archive_version
        + "_"
        + quadra_firmware_archive_rc_version
        + tar_gz
    )
    has_quadra_firmware_archive = False
    for p in Path.cwd().iterdir():
        if p.name == quadra_firmware_archive:
            has_quadra_firmware_archive = True
            break

    if has_quadra_firmware_archive:
        extract(p)
        quadra_firmware_folder_prefix = (
            quadra_firmware_archive_prefix + quadra_firmware_archive_version
        )
        for directory in release_directory.iterdir():
            if directory.name.startswith(quadra_firmware_folder_prefix):
                print("")
                quadra_upgrade_script = directory.joinpath("quadra_auto_upgrade.sh")
                # Inform the user that firmware upgrade is starting. Captured output will appear
                # only after the script finishes.
                print("Starting Quadra firmware upgrade script... (this may take a few minutes)", flush=True)
                reboot_pattern = re.compile(
                    r"(Reboot to run firmware loader version v[\w\.]+ \(optional\))"
                )
                completion_pattern = re.compile(r"Firmware upgrade complete!")

                cmd = f"sudo {quadra_upgrade_script.resolve()} -y --skip_same_ver_upgrades"
                e_code, op = subprocess.getstatusoutput(cmd)
                if e_code:
                    sys.exit("Quadra firware upgrade (FAILURE)\n")
                print(f"{op}")

                if reboot_pattern.search(op or "") and completion_pattern.search(op or ""):
                    print("Quadra firmware upgrade completed – reboot required")
                    needs_quadra_reboot = True
                print("")
                break

cpu_count = os.cpu_count()
assert cpu_count
desired_value = "performance\n"
systemctl = "systemctl"

if not check_scaling_governor():
    print(
        f"Setting scaling_governor to '{desired_value.rstrip()}' for all CPUs. (Uses sudo)"
    )
    scaling_governor_regexp = re.compile(
        r"devices/system/cpu/cpu\d+/cpufreq/scaling_governor"
    )
    local_sysfs_configuration_file = Path("sysfs.conf")
    sysfs_configuration_file = etc / local_sysfs_configuration_file
    with sysfs_configuration_file.open() as infile:
        with local_sysfs_configuration_file.open("w") as outfile:
            for line in infile:
                if not scaling_governor_regexp.match(line):
                    outfile.write(line)
            for i in range(cpu_count):
                outfile.write(
                    f"devices/system/cpu/cpu{i}/cpufreq/scaling_governor = "
                    + desired_value
                )
    run(
        f"sudo {install} {local_sysfs_configuration_file} {sysfs_configuration_file}"
    )
    local_sysfs_configuration_file.unlink()
    run(f"sudo {systemctl} restart sysfsutils")
    if not check_scaling_governor(False):
        wprint("scaling_governor for all CPUs not set properly")

local_pam_limits_configuration_file = Path("limits.conf")
pam_limits_configuration_file = (
    etc / Path("security") / local_pam_limits_configuration_file
)
print(f"\nUpdating {pam_limits_configuration_file} (Uses sudo)")
octothorpe = "#"
eof = "# End of file\n"
wildcard, core, nofile = "*", "core", "nofile"
with pam_limits_configuration_file.open() as infile:
    with local_pam_limits_configuration_file.open("w") as outfile:
        for line in infile:
            if line == eof:
                continue
            elif line.startswith(octothorpe):
                outfile.write(line)
            else:
                fields = line.split()
                if not len(fields) == 4 or not (
                    fields[0] == wildcard
                    and (fields[2] == nofile or fields[2] == core)
                ):
                    outfile.write(line)
        outfile.write("* - nofile 65535\n* - core unlimited\n\n")
        outfile.write(eof)
run(
    f"sudo {install} {local_pam_limits_configuration_file} {pam_limits_configuration_file}"
)
local_pam_limits_configuration_file.unlink()

sysctl = "sysctl"
local_sysctl_configuration_file = Path(sysctl + ".conf")
sysctl_configuration_file = etc / local_sysctl_configuration_file

print("\nConfiguring kernel parameters (Uses sudo)")
octothorpe, semicolon = "#", ";"
mem_default, mem_max = 16777216, 167772160
token_values = {
    "vm.overcommit_memory": 1,
    "net.core.somaxconn": 32768,
    "net.core.rmem_max": mem_max,
    "net.core.rmem_default": mem_default,
    "net.core.wmem_max": mem_max,
    "net.core.wmem_default": mem_default,
    "net.core.netdev_max_backlog": 2000,
    "net.ipv4.tcp_rmem": "4096 262144 213675200",
    "net.ipv4.tcp_wmem": "4096 24576  213675200",
    "net.ipv4.tcp_max_syn_backlog": 8192,
    "kernel.core_pattern": "/var/core/core-%e-%p-%t",
}
with sysctl_configuration_file.open() as infile:
    with local_sysctl_configuration_file.open("w") as outfile:
        for line in infile:
            if line.startswith(octothorpe) or line.startswith(semicolon):
                outfile.write(line)
            else:
                should_write = True
                for token in token_values.keys():
                    if token in line:
                        should_write = False
                        break
                if should_write:
                    outfile.write(line)
        for a, b in token_values.items():
            outfile.write(f"{a} = {b}\n")
run(f"sudo {install} {local_sysctl_configuration_file} {sysctl_configuration_file}")
local_sysctl_configuration_file.unlink()
core_dir = Path("/var/core")
if not core_dir.is_dir():
    run(f"sudo mkdir {core_dir}")
run(f"sudo chmod 777 {core_dir}")
run(f"sudo {sysctl} -p")

print("\nDisabling Ubuntu apport (Uses sudo)")
run(f"sudo {systemctl} stop apport")
run(f"sudo {systemctl} disable apport")
local_apport_configuration_file = Path("apport")
apport_configuration_file = etc / Path("default") / local_apport_configuration_file
enabled = "enabled"
with apport_configuration_file.open() as infile:
    with local_apport_configuration_file.open("w") as outfile:
        for line in infile:
            if not line.startswith(enabled):
                outfile.write(line)
        outfile.write("enabled=0\n")
run(f"sudo {install} {local_apport_configuration_file} {apport_configuration_file}")
local_apport_configuration_file.unlink()

for service in ["etcd", "mysql", "minio", "mysql-keychain"]:
    print("\nCreating {} directory (Uses sudo)".format(service))
    run("sudo mkdir -p /var/bitstreams2/{}".format(service))
    run("sudo chmod 777 /var/bitstreams2/{}".format(service))

file = Path("/var/bitstreams2/mysql/grastate.dat")
print("\nCleaning {}".format(file))
if file.exists():
    run(f"sudo rm {file}")

groups = "groups"

print("\nConfiguring the Docker daemon (Uses sudo)")
local_daemon_configuration_file = Path("daemon.json")
daemon_configuration_file = etc / Path(docker) / local_daemon_configuration_file
log_driver, log_driver_value = "log-driver", "json-file"
log_opts, log_opts_value = "log-opts", {"max-size": "500m", "max-file": "10"}
should_write = False
if daemon_configuration_file.exists():
    with daemon_configuration_file.open() as f:
        j = json.load(f)
        if log_driver in j:
            if j[log_driver] != log_driver_value:
                should_write = True
                j[log_driver] = log_driver_value
        else:
            should_write = True
            j[log_driver] = log_driver_value
        if log_opts in j:
            if j[log_opts] != log_opts_value:
                should_write = True
                j[log_opts] = log_opts_value
        else:
            should_write = True
            j[log_opts] = log_opts_value
else:
    should_write = True
    j = {log_driver: log_driver_value, log_opts: log_opts_value}
if should_write:
    with local_daemon_configuration_file.open("w") as f:
        json.dump(j, f)
    run(
        f"sudo {install} {local_daemon_configuration_file} {daemon_configuration_file}"
    )
    local_daemon_configuration_file.unlink()
    run(f"sudo {systemctl} restart docker")
run(f"sudo {systemctl} enable docker")

if intermediate_files:
    print("\nInstalling intermediate release...")
    install_from_archives(intermediate_files, ipv4_address, force_overwrite=True)
    print("Intermediate release installation completed")

print("\nInstalling target release...")
# Force overwrite if this is a staged upgrade (intermediate was installed)
install_from_archives({
    'configs': configs_archive,
    'docker_images': Path(docker_images_archive),
    'non_cluster': Path("non_cluster.json"),
}, ipv4_address, force_overwrite=bool(intermediate_files))

if args.fail_if_no_iommu:
    print("\nCompleted")
elif enable_iommu or (iommu_enabled and passthrough_enabled and not current_boot_enabled):
    if yninput2("Installation completed. Reboot system to complete IOMMU configuration (recommended)"):
        print("Rebooting system...")
        run("sudo reboot")
    else:
        print("""Warning: IOMMU settings will not take effect until system is rebooted.\nTo reboot, please run: "sudo reboot" """)
else:
    print("\nCompleted.")

# Mandatory reboot after Quadra firmware loader upgrade
if needs_quadra_reboot:
    countdown_sec = 10
    print(
        f"\nQuadra firmware loader upgrade was detected. System will reboot in {countdown_sec} seconds..."
    )
    for i in range(countdown_sec, 0, -1):
        print(f"Rebooting in {i} seconds...", end="\r", flush=True)
        time.sleep(1)
    print("Rebooting system...")
    run("sudo reboot")
