/*
 * Copyright 2015 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#define LOG_TAG "NiQuadraOMXH264Encoder"

#include <utils/Log.h>

#include "OMX_Core.h"
#include "OMX_VideoExt.h"
#include "OMX_IndexExt.h"
#include "vidc_vendor_extensions.h"

VendorExtension::VendorExtension(OMX_INDEXTYPE id, const char *name, OMX_DIRTYPE dir,
        const ParamListBuilder& p)
    : mId(id),
      mName(name),
      mPortDir(dir),
      mParams(std::move(p.mParams)),
      mIsSet(false) {
}

// copy extension Info to OMX_CONFIG_ANDROID_VENDOR_EXTENSIONTYPE* struct passed
OMX_ERRORTYPE VendorExtension::copyInfoTo(
        OMX_CONFIG_ANDROID_VENDOR_EXTENSIONTYPE *ext) const {
    // Extension info
    strlcpy((char *)ext->cName, mName.c_str(), OMX_MAX_STRINGNAME_SIZE);
    ext->eDir = mPortDir;
    ext->nParamCount = paramCount();

    // Per-parameter info
    // Must be copied only if there are enough params to fill-in
    if (ext->nParamSizeUsed < ext->nParamCount) {
        return OMX_ErrorNone;
    }

    int i = 0;
    for (const Param& p : mParams) {
        strlcpy((char *)ext->param[i].cKey, p.name(), OMX_MAX_STRINGNAME_SIZE);
        ext->param[i].bSet = mIsSet ? OMX_TRUE : OMX_FALSE;
        ext->param[i].eValueType = p.type();
        ++i;
    }
    return OMX_ErrorNone;
}

bool VendorExtension::setParamInt32(
        OMX_CONFIG_ANDROID_VENDOR_EXTENSIONTYPE *ext, const char *paramKey,
        OMX_S32 setInt32) const {
    int paramIndex = indexOfParam(paramKey);
    if (!_isParamAccessTypeOK(ext, paramIndex, OMX_AndroidVendorValueInt32)) {
        return false;
    }
    ext->param[paramIndex].nInt32 = setInt32;
    return true;
}

bool VendorExtension::setParamInt64(
        OMX_CONFIG_ANDROID_VENDOR_EXTENSIONTYPE *ext, const char *paramKey,
        OMX_S64 setInt64) const {
    int paramIndex = indexOfParam(paramKey);
    if (!_isParamAccessTypeOK(ext, paramIndex, OMX_AndroidVendorValueInt64)) {
        return false;
    }
    ext->param[paramIndex].nInt64 = setInt64;
    return true;
}

bool VendorExtension::setParamString(
        OMX_CONFIG_ANDROID_VENDOR_EXTENSIONTYPE *ext, const char *paramKey,
        const char *setStr) const {
    int paramIndex = indexOfParam(paramKey);
    if (!_isParamAccessTypeOK(ext, paramIndex, OMX_AndroidVendorValueString)) {
        return false;
    }
    strlcpy((char *)ext->param[paramIndex].cString, setStr, OMX_MAX_STRINGVALUE_SIZE);
    return true;
}

bool VendorExtension::readParamInt32(
        OMX_CONFIG_ANDROID_VENDOR_EXTENSIONTYPE *ext, const char *paramKey,
        OMX_S32 *readInt32) const {
    int paramIndex = indexOfParam(paramKey);
    if (!_isParamAccessTypeOK(ext, paramIndex, OMX_AndroidVendorValueInt32)) {
        return false;
    }
    if (ext->param[paramIndex].bSet == OMX_TRUE) {
        *readInt32 = ext->param[paramIndex].nInt32;
        return true;
    }
    return false;
}

bool VendorExtension::readParamInt64(
        OMX_CONFIG_ANDROID_VENDOR_EXTENSIONTYPE *ext, const char *paramKey,
        OMX_S64 *readInt64) const {
    int paramIndex = indexOfParam(paramKey);
    if (!_isParamAccessTypeOK(ext, paramIndex, OMX_AndroidVendorValueInt64)) {
        return false;
    }
    if (ext->param[paramIndex].bSet == OMX_TRUE) {
        *readInt64 = ext->param[paramIndex].nInt64;
        return true;
    }
    return false;
}

bool VendorExtension::readParamString(
            OMX_CONFIG_ANDROID_VENDOR_EXTENSIONTYPE *ext, const char *paramKey,
            char *readStr) const {
    int paramIndex = indexOfParam(paramKey);
    if (!_isParamAccessTypeOK(ext, paramIndex, OMX_AndroidVendorValueString)) {
        return false;
    }
    if (ext->param[paramIndex].bSet == OMX_TRUE) {
        strlcpy(readStr,
                (const char *)ext->param[paramIndex].cString, OMX_MAX_STRINGVALUE_SIZE);
        return true;
    }
    return false;
}

// Checkers
OMX_ERRORTYPE VendorExtension::isConfigValid(
    OMX_CONFIG_ANDROID_VENDOR_EXTENSIONTYPE *ext) const {
    if (ext->nParamSizeUsed < ext->nParamCount) {
        ALOGE("allotted params(%u) < required(%u) for %s",
                ext->nParamSizeUsed, ext->nParamCount, mName.c_str());
        return OMX_ErrorBadParameter;
    }
    if (ext->nParamCount != paramCount()) {
        ALOGE("incorrect param count(%u) v/s required(%u) for %s",
                ext->nParamCount, paramCount(), mName.c_str());
        return OMX_ErrorBadParameter;
    }
    if (strncmp((char *)ext->cName, mName.c_str(), OMX_MAX_STRINGNAME_SIZE) != 0) {
        ALOGE("extension name mismatch(%s) v/s expected(%s)",
                (char *)ext->cName, mName.c_str());
        return OMX_ErrorBadParameter;
    }

    for (OMX_U32 i = 0; i < paramCount(); ++i) {
        if (!_isParamAccessOK(ext, i)) {
            ALOGI("_isParamAccessOK failed for %u", i);
            return OMX_ErrorBadParameter;
        }
    }

    return OMX_ErrorNone;
}

bool VendorExtension::isConfigKey(
         OMX_CONFIG_ANDROID_VENDOR_EXTENSIONTYPE *ext,
         const char *paramKey) const {
    bool retStatus = false;
    if (!strncmp((char *)ext->cName, paramKey, OMX_MAX_STRINGNAME_SIZE)) {
        retStatus = true;
    }

    return retStatus;
}

//static
const char* VendorExtension::typeString(OMX_ANDROID_VENDOR_VALUETYPE type) {
    switch (type) {
        case OMX_AndroidVendorValueInt32: return "Int32";
        case OMX_AndroidVendorValueInt64: return "Int64";
        case OMX_AndroidVendorValueString: return "String";
        default: return "InvalidType";
    }
}

std::string VendorExtension::debugString() const {
    std::string str = "vendor." + mName + "{";
    for (const Param& p : mParams) {
        str += "{ ";
        str += p.name();
        str += " : ";
        str += typeString(p.type());
        str += " },  ";
    }
    str += "}";
    return str;
}

bool VendorExtension::_isParamAccessTypeOK(
        OMX_CONFIG_ANDROID_VENDOR_EXTENSIONTYPE *ext, int paramIndex,
        OMX_ANDROID_VENDOR_VALUETYPE type) const {
    if (paramIndex < 0
            || paramIndex >= (int)ext->nParamSizeUsed
            || paramIndex >= (int)paramCount()) {
        ALOGE("Invalid Param index(%d) for %s (max=%u)",
                paramIndex, mName.c_str(), paramCount());
        return false;
    }
    if (type != mParams[paramIndex].type()) {
        ALOGE("Invalid Type for field(%s) for %s.%s (expected=%s)",
                typeString(type), mName.c_str(), mParams[paramIndex].name(),
                typeString(mParams[paramIndex].type()));
        return false;
    }
    return true;
}

bool VendorExtension::_isParamAccessOK(
        OMX_CONFIG_ANDROID_VENDOR_EXTENSIONTYPE *ext, int paramIndex) const {
    if (paramIndex < 0
            || paramIndex >= (int)ext->nParamSizeUsed
            || paramIndex >= (int)paramCount()) {
        ALOGE("Invalid Param index(%d) for %s (max=%u)",
                paramIndex, mName.c_str(), paramCount());
        return false;
    }
    if (ext->param[paramIndex].eValueType != mParams[paramIndex].type()) {
        ALOGE("Invalid Type for field(%s) for %s.%s (expected=%s)",
                typeString(ext->param[paramIndex].eValueType),
                mName.c_str(), mParams[paramIndex].name(),
                typeString(mParams[paramIndex].type()));
        return false;
    }
    if (strncmp((const char *)ext->param[paramIndex].cKey,
            mParams[paramIndex].name(), OMX_MAX_STRINGNAME_SIZE) != 0) {
        ALOGE("Invalid Key for field(%s) for %s.%s (expected=%s)",
                ext->param[paramIndex].cKey,
                mName.c_str(), mParams[paramIndex].name(),
                mParams[paramIndex].name());
        return false;
    }
    return true;
}

int VendorExtension::indexOfParam(const char *key) const {
    int i = 0;
    for (const Param& p : mParams) {
        if (!strncmp(key, p.name(), OMX_MAX_STRINGNAME_SIZE)) {
            return i;
        }
        ++i;
    }
    ALOGE("Failed to lookup param(%s) in extension(%s)",
            key, mName.c_str());
    return -1;
}

void VendorExtensionStore::dumpExtensions() const {
    ALOGV("Vendor extensions supported (%u)", size());
    for (const VendorExtension& v : mExt) {
        ALOGV("   %s", v.debugString().c_str());
    }
}
