/*
 * Copyright 2015 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#define LOG_TAG	"NiQuadraOMXH264Decoder"

#include "NiQuadraOMXDecoder.h"

#include "ni_rsrc_api.h"
#include "ni_av_codec.h"
#include "ni_util.h"
#include "ni_log.h"

#include "libyuv.h"

#include <string.h>
#include <cutils/properties.h>
#include <media/stagefright/foundation/ADebug.h>
#include <media/stagefright/MediaDefs.h>
#include <OMX_VideoExt.h>
#include <OMX_IndexExt.h>
#include <inttypes.h>

#include <media/hardware/HardwareAPI.h>
#include <hardware/gralloc.h>
#include <ui/GraphicBuffer.h>
#include <ui/PixelFormat.h>

namespace android {

#define AVERROR(e) (-(e))
#define PROP_XOCDER_LOG_LEVEL "xcoder_log_level"
#define PROP_DECODER_MAX_LOAD "ro.vmi.quadraDecodeMaxLoad"

typedef struct _xcoder_log_level {
  const char * level_str;
  ni_log_level_t level;
} xcoder_log_level;

static const xcoder_log_level log_level_list[] = {
  { "fatal", NI_LOG_FATAL },
  { "error", NI_LOG_ERROR },
  { "info", NI_LOG_INFO },
  { "debug", NI_LOG_DEBUG },
  { "trace", NI_LOG_TRACE }
};

static ni_log_level_t convert_ni_log_level(const char *log_level) {
  int i = 0;
  int tablesz = sizeof(log_level_list) / sizeof(xcoder_log_level);

  if (log_level) {
    for (i = 0; i < tablesz; i++) {
      if (!strcasecmp(log_level, log_level_list[i].level_str)) {
        return log_level_list[i].level;
      }
    }
  }

  ALOGE("Error: invalid xcoder log level is specified: %s", log_level);
  return NI_LOG_INVALID;
}

NiQuadraOMXDecoder::NiQuadraOMXDecoder(
    const char *name,
    const char *componentRole,
    OMX_VIDEO_CODINGTYPE codingType,
    const char *mimeType,
    const CodecProfileLevel *profileLevels,
    size_t numProfileLevels,
    const OMX_CALLBACKTYPE *callbacks,
    OMX_PTR appData,
    OMX_COMPONENTTYPE **component)
  : SoftVideoDecoderOMXComponent(
      name, componentRole, codingType,
      profileLevels, numProfileLevels,
      320 /* width */, 240 /* height */, callbacks,
      appData, component) {

  initPorts(
          1 /* numMinInputBuffers */, kNumBuffers, NI_INPUT_BUF_SIZE,
          1 /* numMinOutputBuffers */, kNumBuffers, mimeType);

  mTimeStart = mTimeCur = mTimeLast = systemTime();
  m_total_frame_decoded = 0;
  m_frame_decoded_one_second = 0;

  m_signalled_error = false;
  m_port_reset = false;
  m_need_to_flush = false;
  m_android_native_buffers = false;
  m_store_meta_in_buffers = false;
  m_allocate_native_handle = false;
  m_width = m_height = 0;
  m_ctx = NULL;
}

NiQuadraOMXDecoder::~NiQuadraOMXDecoder() {
  ALOGI("enter into ~NiQuadraOMXDecoder()");
  xcoder_decoder_close();

  if (m_ctx) {
    free (m_ctx);
    m_ctx = NULL;
  }
}

int NiQuadraOMXDecoder::xcoder_decoder_init() {
  ni_retcode_t ret = NI_RETCODE_SUCCESS;
  ni_session_context_t *api_ctx = &m_ctx->api_ctx;
  ni_xcoder_params_t *api_param = &m_ctx->api_param;
  ni_session_data_io_t *api_pkt = &m_ctx->api_pkt;
  ni_session_data_io_t *api_frame = &m_ctx->api_frame;

  m_width = outputBufferWidth();
  m_height = outputBufferHeight();

  ALOGI("enter into xcoder_decoder_init(), width:%d, height:%d, Width:%d, Height:%d",
  	m_width, m_height, mWidth, mHeight);

  if (m_width <= 0 || m_height <= 0) {
    ALOGE("Error: invalid input width or height");
    return AVERROR(EINVAL);
  }

  if (m_width > NI_MAX_RESOLUTION_WIDTH ||
      m_height > NI_MAX_RESOLUTION_HEIGHT ||
      m_width * m_height > NI_MAX_RESOLUTION_AREA) {
    ALOGE("Error XCoder resolution %dx%d not supported", m_width, m_height);
    return AVERROR(EINVAL);
  } else if (m_width < NI_MIN_RESOLUTION_WIDTH ||
             m_height < NI_MIN_RESOLUTION_HEIGHT) {
    ALOGE("Error XCoder resolution %dx%d not supported", m_width, m_height);
    return AVERROR(EINVAL);
  }

  if ((ret = ni_device_session_context_init(api_ctx)) < 0) {
    ALOGE("Error: xcoder init decoder context failure: %d", ret);
    return ret;
  }

  switch (m_codec_type) {
    case CODEC_FORMAT_H264:
      api_ctx->codec_format = NI_CODEC_FORMAT_H264;
      break;
    case CODEC_FORMAT_H265:
      api_ctx->codec_format = NI_CODEC_FORMAT_H265;
      break;
    default:
      ALOGE("Error: video codec format %d not supported", m_codec_type);
      return AVERROR(EINVAL);
  }

  api_ctx->src_bit_depth = 8;
  api_ctx->src_endian = NI_FRAME_LITTLE_ENDIAN;
  api_ctx->bit_depth_factor = 1;
  api_ctx->keep_alive_timeout = NI_DEFAULT_KEEP_ALIVE_TIMEOUT;
  ret = ni_decoder_init_default_params(api_param, 25, 1, 200000, m_width, m_height);
  if (ret < 0) {
    ALOGE("Error: set decoder params by default value failed: %d", ret);
    return ret;
  }

  if ((m_width * m_height) >= DIMENSION_2K_VIDEO) {
    char value[] = "1";
    ret = ni_decoder_params_set_value(api_param, "multicoreJointMode", value);
    if (ret < 0) {
      ALOGE("Error: set decoder params multicoreJointMode failed: %d", ret);
      return ret;
    }
  }

  api_ctx->p_session_config = api_param;
  ret = ni_device_session_open(api_ctx, NI_DEVICE_TYPE_DECODER);
  if (ret < 0) {
    ALOGE("Error: open decoder session failed: %d", ret);
    return ret;
  } else {
    ALOGI("open decoder session successful");
  }

  m_sos_send = false;
  m_eos_send = false;
  m_eos = false;
  m_is_closed = false;
  m_offset = 0LL;

  m_color_primaries = NI_COL_PRI_UNSPECIFIED;
  m_color_trc = NI_COL_TRC_UNSPECIFIED;
  m_color_space = NI_COL_SPC_UNSPECIFIED;
  m_full_range = 0;

  m_output_width = m_width;
  m_output_height = m_height;

  memset(api_pkt, 0, sizeof(ni_session_data_io_t));
  memset(api_frame, 0, sizeof(ni_session_data_io_t));

  return 0;
}

int NiQuadraOMXDecoder::xcoder_decoder_close() {
  if (!m_ctx) {
    return 0;
  }

  ni_retcode_t ret = NI_RETCODE_SUCCESS;
  ni_session_context_t *api_ctx = &m_ctx->api_ctx;

  if (m_is_closed) {
    return 0;
  }

  ALOGI("enter into xcoder_decoder_close()");

  ret = ni_device_session_close(api_ctx, m_eos, NI_DEVICE_TYPE_DECODER);
  if (ret < 0) {
    ALOGE("Failed to close Decode Session (status = %d)", ret);
  }

  ni_device_session_context_clear(api_ctx);

  ni_device_close(api_ctx->device_handle);
  ni_device_close(api_ctx->blk_io_handle);

  api_ctx->device_handle = NI_INVALID_DEVICE_HANDLE;
  api_ctx->blk_io_handle = NI_INVALID_DEVICE_HANDLE;

  m_is_closed = true;

  return 0;
}

int NiQuadraOMXDecoder::xcoder_decoder_reset() {
  int ret = 0;

  if (!m_ctx) {
    return 0;
  }
 
  ALOGI("enter into xcoder_decoder_reset()");

  if ((ret = xcoder_decoder_close()) < 0) {
    ALOGE("Error: xcoder_decode_close failed: %d", ret);
    return ret;
  }

  ni_packet_buffer_free(&(m_ctx->api_pkt.data.packet));

  if ((ret = xcoder_decoder_init()) < 0) {
    ALOGE("Error: xcoder_decode_init failed: %d", ret);
    return ret;
  }

  m_ctx->api_ctx.session_run_state = SESSION_RUN_STATE_RESETTING;

  return 0;
}

int NiQuadraOMXDecoder::xcoder_decoder_flush() {
  ALOGI("enter into xcoder_decoder_flush()");

  if (!m_ctx) {
    return 0;
  }

  int ret = 0;
  bool send_frame = false;
  ni_session_data_io_t *api_pkt = &m_ctx->api_pkt;
  ni_session_data_io_t *api_frame = &m_ctx->api_frame;

  //packet is NULL means eos
  if ((ret = xcoder_decoder_send(NULL, &send_frame)) < 0) {
    ALOGE("Error: send xcoder_decoder_send failed: %d", ret);
    return ret;
  }

  ni_packet_buffer_free(&(api_pkt->data.packet));

  //receive all output from vpu
  while (1) {
    bool get_frame = false;
    OMX_BUFFERHEADERTYPE outHeader;
    memset(&outHeader, 0, sizeof(OMX_BUFFERHEADERTYPE));

    ret = xcoder_decoder_receive(&outHeader, &get_frame);
    if (ret < 0) {
      ALOGE("Error: recv xcoder_decoder_receive failed: %d", ret);
      return ret;
    }

    if (get_frame) {
      ni_decoder_frame_buffer_free(&(api_frame->data.frame));
    }

    if (m_eos) {
      break;
    }
  }

  ret = ni_device_dec_session_flush(&m_ctx->api_ctx);
  if (ret < 0) {
    ALOGE("Error: ni_device_dec_session_flush failed: %d", ret);
    return ret;
  }

  memset(api_pkt, 0, sizeof(ni_session_data_io_t));
  memset(api_frame, 0, sizeof(ni_session_data_io_t));
  m_eos_send = false;
  m_eos = false;

  return 0;
}

//OMX_BUFFERHEADERTYPE --> ni_packet_t
int NiQuadraOMXDecoder::xcoder_decoder_send(
    OMX_BUFFERHEADERTYPE *inHeader,
    bool *send_frame) {
  ni_session_context_t *api_ctx = &m_ctx->api_ctx;
  ni_packet_t *xpkt = &(m_ctx->api_pkt.data.packet);
  int extra_prev_size = 0;
  int need_draining = 0;
  int new_packet = 0;
  int send_size = 0;
  int ret = 0;
  int sent = 0;

  ALOGV("enter into xcoder_decode_send()");

  //all inputs have been sent to vpu
  if (m_eos_send) {
    return 0;
  }

  if (NULL == inHeader) {
    need_draining = 1;
  }

  if (xpkt->data_len == 0) {
    memset(xpkt, 0, sizeof(ni_packet_t));

    xpkt->video_width = m_width;
    xpkt->video_height = m_height;
    xpkt->p_data = NULL;

    if (inHeader) {
      xpkt->pts = inHeader->nTimeStamp;
      //xpkt->dts = inHeader->nTimeStamp;
      xpkt->data_len = inHeader->nFilledLen;
    }

    if (xpkt->data_len + api_ctx->prev_size > 0) {
      ni_packet_buffer_alloc(xpkt, (xpkt->data_len + api_ctx->prev_size));
      if (!xpkt->p_data) {
        ret = -1;
        goto fail;
      }
    }

    new_packet = 1;
  } else {
    send_size = xpkt->data_len;
  }

  if (!m_sos_send) {
    xpkt->start_of_stream = 1;
    m_sos_send = true;
  }

  if (need_draining && !m_eos_send) {
    xpkt->end_of_stream = 1;
    xpkt->data_len = 0;

    if (new_packet) {
      extra_prev_size = api_ctx->prev_size;

      send_size = ni_packet_copy(xpkt->p_data, NULL, 0, api_ctx->p_leftover, &api_ctx->prev_size);

      //increment offset of data sent to decoder and save it
      xpkt->pos = (long long)m_offset;
      m_offset += extra_prev_size;
    }

    if (send_size < 0) {
      ALOGE("Failed to copy pkt (status = %d)", send_size);
      ret = -1;
      goto fail;
    }
    xpkt->data_len += extra_prev_size;

    sent = 0;
    if (xpkt->data_len > 0) {
      sent = ni_device_session_write(api_ctx, &(m_ctx->api_pkt), NI_DEVICE_TYPE_DECODER);
    }

    if (sent < 0) {
      if (NI_RETCODE_ERROR_VPU_RECOVERY == sent) {
        ret = xcoder_decoder_reset();
      } else {
        ret = sent;
      }
      goto fail;
    }

    ni_device_session_flush(api_ctx, NI_DEVICE_TYPE_DECODER);

    if (api_ctx->ready_to_close) {
      m_eos_send = true;
    }
  } else {
    if (new_packet) {
      extra_prev_size = api_ctx->prev_size;

      send_size = ni_packet_copy(xpkt->p_data, inHeader->pBuffer + inHeader->nOffset,
          inHeader->nFilledLen, api_ctx->p_leftover, &api_ctx->prev_size);

      //increment offset of data sent to decoder and save it
      xpkt->pos = (long long)m_offset;
      m_offset += (inHeader->nFilledLen + extra_prev_size);
    }

    if (send_size < 0) {
      ALOGE("Failed to copy pkt (status = %d)", send_size);
      ret = -1;
      goto fail;
    }
    xpkt->data_len += extra_prev_size;

    sent = 0;
    if (xpkt->data_len > 0) {
      sent = ni_device_session_write(api_ctx, &(m_ctx->api_pkt), NI_DEVICE_TYPE_DECODER);
    }

    if (sent < 0) {
      if (NI_RETCODE_ERROR_VPU_RECOVERY == sent) {
        ret = xcoder_decoder_reset();
      } else {
        ret = sent;
      }
      goto fail;
    }
  }

  if (xpkt->data_len == 0) {
    free(xpkt->p_custom_sei_set);
    xpkt->p_custom_sei_set = NULL;
  }

  if (sent != 0) {
    ni_packet_buffer_free(xpkt);
    *send_frame = true;
  }

  return sent;

fail:
  ni_packet_buffer_free(xpkt);
  free(xpkt->p_custom_sei_set);
  xpkt->p_custom_sei_set = NULL;

  return ret;
}

static int do_yuv420_to_rgba888_convert(uint8_t* yuv[],
                                        uint8_t* rgba,
                                        int yuv_stride,
                                        int yuv_width,
                                        int yuv_height,
                                        int rgb_stride) {
  ALOGV("enter into do_yuv420_to_rgba888_convert()");

  uint8_t*src_y = yuv[0];
  uint8_t*src_u = yuv[1];
  uint8_t*src_v = yuv[2];

  int src_stride_y = yuv_stride;
  int src_stride_u = ((yuv_stride / 2 + 127) / 128) * 128;
  int src_stride_v = src_stride_u;

  return libyuv::I420ToARGB(src_y, src_stride_y,
                     src_v, src_stride_v,
                     src_u, src_stride_u,
                     rgba, rgb_stride * 4,
                     yuv_width, yuv_height);
}

//ni_frame_t --> OMX_BUFFERHEADERTYPE
int NiQuadraOMXDecoder::retrieve_frame(OMX_BUFFERHEADERTYPE *outHeader) {
  ni_session_context_t *api_ctx = &m_ctx->api_ctx;
  ni_session_data_io_t *p_session_data = &m_ctx->api_frame;
  ni_frame_t *xfme = &(p_session_data->data.frame);
  int ret = 0;

  ALOGV("enter into retrieve_frame()");

  outHeader->nTimeStamp = xfme->pts;

  if (m_store_meta_in_buffers) {
    status_t res;
    uint8_t *rgba = NULL;
    struct VideoNativeMetadata *meta = NULL;

    meta = (struct VideoNativeMetadata *)(outHeader->pBuffer);
    if (!meta || meta->eType != kMetadataBufferTypeANWBuffer) {
      ALOGE("Error: meta eType isn't kMetadataBufferTypeANWBuffer : %d.", meta->eType);
      ret = -1;
      goto EXIT;
    }

    sp<GraphicBuffer> gbuf(GraphicBuffer::from(meta->pBuffer));
    res = gbuf->lock(GRALLOC_USAGE_SW_WRITE_OFTEN, (void**)(&rgba));
    if (res != 0) {
      ALOGE("Error: lock GraphicBuffer failed: %d.", res);
      ret = -1;
      goto EXIT;
    }

    int src_stride = api_ctx->active_video_width;
    int dst_stride = gbuf->getStride();
    int ret = do_yuv420_to_rgba888_convert(xfme->p_data,
                                           rgba,
                                           src_stride,
                                           xfme->video_width,
                                           xfme->video_height,
                                           dst_stride);
    if (ret < 0) {
      ALOGE("Error: do_yuv420_to_rgba888_convert failed");
      gbuf->unlock();
      ret = -1;
      goto EXIT;
    }

    outHeader->nFilledLen = sizeof(struct VideoNativeMetadata);
    gbuf->unlock();
  } else {
    uint32_t frame_size = 0;
    uint8_t *src, *dst;
    int i, j;

    for (i = 0; i < 3; i++) {
      src = xfme->p_data[i];

      int plane_height = api_ctx->active_video_height;
      int plane_width = api_ctx->active_video_width;
      int write_height = m_height;
      int write_width = m_width;

      // support for 8/10 bit depth
      // plane_width is the actual Y stride size
      write_width *= api_ctx->bit_depth_factor;

      if (i == 1 || i == 2) {
        plane_height /= 2;

        // U/V stride size is multiple of 128, following the calculation
        // in ni_decoder_frame_buffer_alloc
        plane_width = (((int) (api_ctx->actual_video_width) / 2 *
            api_ctx->bit_depth_factor + 127) / 128) * 128;

        write_height /= 2;
        write_width /= 2;
      }

      // apply the cropping windown in writing out the YUV frame
      // for now the windown is usually crop-left = crop-top = 0, and we
      // use this to simplify the cropping logic
      dst = outHeader->pBuffer + outHeader->nOffset + frame_size;

      for (j = 0; j < plane_height; j++) {
        memcpy (dst, src, write_width);
        src += plane_width;
        dst += write_width;
      }

      frame_size += (plane_height * write_width);
    }

    outHeader->nFilledLen = frame_size;
  }

EXIT:
  ni_decoder_frame_buffer_free(&(p_session_data->data.frame));
  return ret;
}

int NiQuadraOMXDecoder::xcoder_decoder_receive(
    OMX_BUFFERHEADERTYPE *outHeader,
    bool *get_frame) {
  ni_session_context_t *api_ctx = &m_ctx->api_ctx;
  ni_session_data_io_t *p_session_data = &m_ctx->api_frame;
  int alloc_mem, height, actual_width;
  int frame_planar;
  int ret = 0;

  if (m_eos || !outHeader) {
    return 0;
  }

  ALOGV("enter into xcoder_decoder_receive()");

  // if active video resolution has been obtained we just use it as it's the
  // exact size of frame to be returned, otherwise we use what we are told by
  // upper stream as the initial setting and it will be adjusted.
  height =
      (int)(api_ctx->active_video_height > 0 ? api_ctx->active_video_height
                                               : m_height);
  actual_width =
      (int)(api_ctx->actual_video_width > 0 ? api_ctx->actual_video_width
                                              : m_width);

  // allocate memory only after resolution is known (buffer pool set up)
  alloc_mem = (api_ctx->active_video_width > 0 && 
               api_ctx->active_video_height > 0 ? 1 : 0);

  frame_planar = NI_PIXEL_PLANAR_FORMAT_PLANAR;

  ret = ni_decoder_frame_buffer_alloc(
          api_ctx->dec_fme_buf_pool, &(p_session_data->data.frame), alloc_mem,
          actual_width, height, (m_codec_type == CODEC_FORMAT_H264),
          api_ctx->bit_depth_factor, frame_planar);
  if (ret != NI_RETCODE_SUCCESS) {
    ALOGE("in %s ni_decoder_frame_buffer_alloc failed:%d", __FUNCTION__, ret);
    return ret;
  }

  ret = ni_device_session_read(api_ctx, p_session_data, NI_DEVICE_TYPE_DECODER);
  if (ret == 0) {
    m_eos = p_session_data->data.frame.end_of_stream;
    if (m_eos) {
      outHeader->nFilledLen = 0;
    }
    ni_decoder_frame_buffer_free(&(p_session_data->data.frame));
  } else if (ret > 0) {
    m_eos = p_session_data->data.frame.end_of_stream;
    m_output_width = p_session_data->data.frame.video_width; // ppu auto crop reports wdith as cropped width
    m_output_height = p_session_data->data.frame.video_height; // ppu auto crop reports heigth as cropped height

    m_color_primaries = p_session_data->data.frame.color_primaries;
    m_color_trc = p_session_data->data.frame.color_trc;
    m_color_space = p_session_data->data.frame.color_space;
    m_full_range = p_session_data->data.frame.video_full_range_flag;

    *get_frame = true;
  } else {
    if (NI_RETCODE_ERROR_VPU_RECOVERY == ret) {
      ni_decoder_frame_buffer_free(&(p_session_data->data.frame));
      ret = xcoder_decoder_reset();
    }
  }

  return ret;
}

int NiQuadraOMXDecoder::xcoder_send_receive(
    OMX_BUFFERHEADERTYPE *inHeader,
    bool *send_frame,
    OMX_BUFFERHEADERTYPE *outHeader,
    bool *get_frame) {
  int ret = 0;

  ALOGV("enter into xcoder_send_receive()");

  ret = xcoder_decoder_send(inHeader, send_frame);
  if (ret < 0) {
    ALOGE("in %s send pkt error:%d", __func__, ret);
    return ret;
  }

  ret = xcoder_decoder_receive(outHeader, get_frame);
  if (ret < 0) {
    ALOGE("in %s receive frame error:%d", __func__, ret);
    return ret;
  }

  return 0;
}

status_t NiQuadraOMXDecoder::resetPlugin() {
  ALOGI("enter into resetPlugin()");

  m_in_flush = false;
  m_received_eos = false;

  return OK;
}

bool NiQuadraOMXDecoder::get_color_aspects() {
  int32_t primaries, transfer, coeffs;
  bool fullRange = false;

  ALOGV("enter into get_color_aspects()");

  primaries = m_color_primaries;
  transfer = m_color_trc;
  coeffs = m_color_space;

  if (m_full_range) {
    fullRange = true;
  }

  //convert iso color aspects to codec aspects
  ColorAspects colorAspects;
  ColorUtils::convertIsoColorAspectsToCodecAspects(
      primaries, transfer, coeffs, fullRange, colorAspects);

  // Update color aspects if necessary.
  if (colorAspectsDiffer(colorAspects, mBitstreamColorAspects)) {
    mBitstreamColorAspects = colorAspects;
    status_t err = handleColorAspectsChange();
    CHECK(err == OK);
  }

  return true;
}

void NiQuadraOMXDecoder::onReset() {
  ALOGI("enter into onReset()");

  if (m_need_to_flush) {
    m_need_to_flush = false;
  }

  SoftVideoDecoderOMXComponent::onReset();

  m_signalled_error = false;

  xcoder_decoder_reset();
  resetPlugin();
}

void NiQuadraOMXDecoder::onPortFlushCompleted(OMX_U32 portIndex) {
  ALOGI("enter into onPortFlushCompleted()");

  /* Once the output buffers are flushed, ignore any buffers that are held in decoder */
  if (kOutputPortIndex == portIndex) {
    m_need_to_flush = true;
  }
}

void NiQuadraOMXDecoder::onQueueFilled(OMX_U32 portIndex) {
  UNUSED(portIndex);

  if (m_signalled_error) {
    ALOGE("m_signalled_error is enable !!!");
    return;
  }

  if (mOutputPortSettingsChange != NONE) {
    ALOGE("mOutputPortSettingsChange is enable !!!");
    return;
  }

  ALOGV("enter into onQueueFilled()");

  if (! m_ctx) {
    m_ctx = (NiOMXDecCtx *)malloc(sizeof(NiOMXDecCtx));
    if (!m_ctx) {
      notify(OMX_EventError, OMX_ErrorUnsupportedSetting, 0, NULL);
      m_signalled_error = true;
      ALOGE("faile to malloc NiOMXDecCtx !!!");
      return;
    }

    memset(m_ctx, 0, sizeof(NiOMXDecCtx));

    ni_log_level_t log_level = NI_LOG_INVALID;
    char property[PROPERTY_VALUE_MAX] = {0};
    if (property_get(PROP_XOCDER_LOG_LEVEL, property, NULL)) {
      log_level = convert_ni_log_level(property);
      if (log_level != NI_LOG_INVALID) {
        ALOGI("set xcoder log level: %s", property);
        ni_log_set_level(log_level);
      }
    }

    if (xcoder_decoder_init() < 0) {
      ALOGE("Failed to initialize NETINT decoder");
      notify(OMX_EventError, OMX_ErrorUnsupportedSetting, 0, NULL);
      m_signalled_error = true;
      return;
    }

    /* Reset the plugin state */
    resetPlugin();
  }

  if ((outputBufferWidth() != m_width) || (outputBufferHeight() != m_height)) {
    m_width = outputBufferWidth();
    m_height = outputBufferHeight();
  }

  BufferInfo *inInfo = NULL;
  OMX_BUFFERHEADERTYPE *inHeader = NULL;

  List<BufferInfo *> &inQueue = getPortQueue(kInputPortIndex);
  List<BufferInfo *> &outQueue = getPortQueue(kOutputPortIndex);

  while (!outQueue.empty()) {
    BufferInfo *outInfo = NULL;
    OMX_BUFFERHEADERTYPE *outHeader = NULL;

    if (m_port_reset) {
      outInfo = *outQueue.begin();
      if (outInfo) {
        outHeader = outInfo->mHeader;
        outHeader->nFlags = 0;
        outHeader->nTimeStamp = 0;
        outHeader->nOffset = 0;

        if (retrieve_frame(outHeader) < 0) {
          notify(OMX_EventError, OMX_ErrorUnsupportedSetting, 0, NULL);
          m_signalled_error = true;
          return;
        }

        outInfo->mOwnedByUs = false;
        outQueue.erase(outQueue.begin());
        outInfo = NULL;
        notifyFillBufferDone(outHeader);
        outHeader = NULL;

        m_port_reset = false;

        if (outQueue.empty()) {
          return;
        }
      }
    }

    if (m_need_to_flush) {
      xcoder_decoder_flush();
      if (m_in_flush) {
        resetPlugin();
      }
      m_need_to_flush = false;
    }

    //note: when the previous input is not sent to vpu, don't obtain new input from inQueue
    if (!m_in_flush && (NULL == inHeader)) {
      if (!inQueue.empty()) {
        inInfo = *inQueue.begin();
        inHeader = inInfo->mHeader;
        if (inHeader == NULL) {
          inQueue.erase(inQueue.begin());
          inInfo->mOwnedByUs = false;
          continue;
        }
      } else {
        //inQueue is empty, quit
        break;
      }

      //check whether eos is received on input port
      if (inHeader->nFilledLen == 0) {
        inQueue.erase(inQueue.begin());
        inInfo->mOwnedByUs = false;
        notifyEmptyBufferDone(inHeader);

        if (!(inHeader->nFlags & OMX_BUFFERFLAG_EOS)) {
          return;
        }
 
        m_received_eos = true;
        m_in_flush = true;
        inHeader = NULL;

      } else if (inHeader->nFlags & OMX_BUFFERFLAG_EOS) {
        m_received_eos = true;
      }
    }

    outInfo = *outQueue.begin();
    if (outInfo) {
      outHeader = outInfo->mHeader;
      outHeader->nFlags = 0;
      outHeader->nTimeStamp = 0;
      outHeader->nOffset = 0;
    }

    bool send_frame = false;
    bool get_frame = false;
    if (xcoder_send_receive(inHeader, &send_frame, outHeader, &get_frame) < 0) {
      ALOGE("Failed to xcoder_send_receive!!!!!");
      notify(OMX_EventError, OMX_ErrorUnsupportedSetting, 0, NULL);
      m_signalled_error = true;
      return;
    }

    if (send_frame && inHeader != NULL) {
      inInfo->mOwnedByUs = false;
      inQueue.erase(inQueue.begin());
      inInfo = NULL;
      notifyEmptyBufferDone(inHeader);
      inHeader = NULL;
    }

    if (m_received_eos && !m_in_flush) {
      if (send_frame) {
        m_in_flush = true;
      }
    }

    if (get_frame) {
      get_color_aspects();

      if ((m_output_width != m_width) || (m_output_height != m_height)) {
        ALOGI("seq change from:%dx%d to %dx%d, ", m_width, m_height, m_output_width, m_output_height);
        m_width = m_output_width;
        m_height = m_output_height;
        handlePortSettingsChange(&m_port_reset, m_width, m_height);
        if (m_port_reset) {
          return;
        }
      } else if (mUpdateColorAspects) {
        ALOGI("notify ColorAspects change");
        notify(OMX_EventPortSettingsChanged, kOutputPortIndex,
            kDescribeColorAspectsIndex, NULL);
        mUpdateColorAspects = false;
        m_port_reset = true;
        return;
      }
    }

    if (get_frame || m_eos) {
      if (get_frame) {
        m_total_frame_decoded += 1;
        m_frame_decoded_one_second += 1;
        mTimeCur = systemTime();
        if (mTimeCur - mTimeLast >= ONE_SECOND_IN_NANOSECOND) {
          ALOGI("OMX decoder DEC_AVG_FPS:%.2f, DEC_CUR_FPS:%.2f",
              (m_total_frame_decoded / ((mTimeCur - mTimeStart) / (float)ONE_SECOND_IN_NANOSECOND)),
              (m_frame_decoded_one_second / ((mTimeCur - mTimeLast) / (float)ONE_SECOND_IN_NANOSECOND)));
          m_frame_decoded_one_second = 0;
          mTimeLast = mTimeCur;
        }

        if (retrieve_frame(outHeader) < 0) {
          notify(OMX_EventError, OMX_ErrorUnsupportedSetting, 0, NULL);
          m_signalled_error = true;
          return;
        }
      }

      if (m_eos) {	
        outHeader->nFlags |= OMX_BUFFERFLAG_EOS;
      }

      outInfo->mOwnedByUs = false;
      outQueue.erase(outQueue.begin());
      outInfo = NULL;
      notifyFillBufferDone(outHeader);
      outHeader = NULL;

      if (m_eos) {
        resetPlugin();
      }
    }
  }
}

void NiQuadraOMXDecoder::updatePortDefinitions(
            bool updateCrop, bool updateInputSize) {
  ALOGV("enter into updatePortDefinitions()");

  OMX_PARAM_PORTDEFINITIONTYPE *outDef = &editPortInfo(kOutputPortIndex)->mDef;
  outDef->format.video.nFrameWidth = outputBufferWidth();
  outDef->format.video.nFrameHeight = outputBufferHeight();
  outDef->format.video.eColorFormat = mOutputFormat;
  outDef->format.video.nSliceHeight = outDef->format.video.nFrameHeight;

  int32_t bpp = (mOutputFormat == OMX_COLOR_FormatYUV420Planar16) ? 2 : 1;
  outDef->format.video.nStride = outDef->format.video.nFrameWidth * bpp;

  if (m_store_meta_in_buffers) {
    outDef->nBufferSize = sizeof(struct VideoNativeMetadata);
  } else {
    outDef->nBufferSize =
        (outDef->format.video.nStride * outDef->format.video.nSliceHeight * 3) / 2;
  }

  OMX_PARAM_PORTDEFINITIONTYPE *inDef = &editPortInfo(kInputPortIndex)->mDef;
  inDef->format.video.nFrameWidth = mWidth;
  inDef->format.video.nFrameHeight = mHeight;
  // input port is compressed, hence it has no stride
  inDef->format.video.nStride = 0;
  inDef->format.video.nSliceHeight = 0;

  // when output format changes, input buffer size does not actually change
  if (updateInputSize) {
    inDef->nBufferSize = ni_max(
            outDef->nBufferSize,
            ni_max(NI_INPUT_BUF_SIZE, inDef->nBufferSize));
  }

  if (updateCrop) {
    mCropLeft = 0;
    mCropTop = 0;
    mCropWidth = mWidth;
    mCropHeight = mHeight;
  }
}

OMX_ERRORTYPE NiQuadraOMXDecoder::internalGetParameter(
        OMX_INDEXTYPE index, OMX_PTR param) {
  // Include extension index OMX_INDEXEXTTYPE.
  const int32_t indexFull = index;

  ALOGV("enter into internalGetParameter()");

  switch (indexFull) {
    case OMX_IndexParamVideoPortFormat:
    {
      OMX_VIDEO_PARAM_PORTFORMATTYPE *formatParams =
          (OMX_VIDEO_PARAM_PORTFORMATTYPE *)param;

      if (!isValidOMXParam(formatParams)) {
        return OMX_ErrorBadParameter;
      }

      if (formatParams->nPortIndex > kMaxPortIndex) {
        return OMX_ErrorBadPortIndex;
      }

      if (formatParams->nPortIndex == kInputPortIndex) {
        formatParams->eCompressionFormat = OMX_VIDEO_CodingAVC;
        formatParams->eColorFormat = OMX_COLOR_FormatUnused;
        formatParams->xFramerate = 0;
      } else {
        if (formatParams->nIndex == 0) {
          formatParams->eColorFormat = (OMX_COLOR_FORMATTYPE)HAL_PIXEL_FORMAT_RGBA_8888;
          formatParams->eCompressionFormat = OMX_VIDEO_CodingUnused;
        } else if (formatParams->nIndex == 1) {
          formatParams->eColorFormat = OMX_COLOR_FormatYUV420Planar;
          formatParams->eCompressionFormat = OMX_VIDEO_CodingUnused;
        } else {
          return OMX_ErrorNoMore;
        }
      }

      return OMX_ErrorNone;
    }

    case OMX_IndexParamVideoAndroidRequiresSwRenderer:
    {
      OMX_PARAM_U32TYPE *swRender = (OMX_PARAM_U32TYPE *)param;

      if (swRender->nPortIndex == kOutputPortIndex) {
#ifdef USE_GPU_RENDER
        swRender->nU32 = 0;
#else
        swRender->nU32 = 1;
#endif
      } else {
        return OMX_ErrorBadPortIndex;
      }

      return OMX_ErrorNone;
    }

    case kGetAndroidNativeBufferUsageExtensionIndex:
    {
      GetAndroidNativeBufferUsageParams *nativeBuffersUsage =
              (GetAndroidNativeBufferUsageParams *)param;

      if (nativeBuffersUsage->nPortIndex == kOutputPortIndex) {
        nativeBuffersUsage->nUsage = GRALLOC_USAGE_SW_WRITE_OFTEN;
      }

      return OMX_ErrorNone;
    }

    default:
      return SoftVideoDecoderOMXComponent::internalGetParameter(index, param);
  }
}

OMX_ERRORTYPE NiQuadraOMXDecoder::internalSetParameter(
        OMX_INDEXTYPE index, OMX_PTR param) {
  // Include extension index OMX_INDEXEXTTYPE.
  const int32_t indexFull = index;

  ALOGV("enter into internalSetParameter()");

  switch (indexFull) {
    case OMX_IndexParamVideoPortFormat:
    {
      OMX_VIDEO_PARAM_PORTFORMATTYPE *formatParams =
          (OMX_VIDEO_PARAM_PORTFORMATTYPE *)param;

      if (!isValidOMXParam(formatParams)) {
        return OMX_ErrorBadParameter;
      }

      if (formatParams->nPortIndex > kMaxPortIndex) {
        return OMX_ErrorBadPortIndex;
      }

      if (formatParams->nPortIndex == kInputPortIndex) {
        if (formatParams->eCompressionFormat != OMX_VIDEO_CodingAVC ||
            formatParams->eColorFormat != OMX_COLOR_FormatUnused) {
          return OMX_ErrorUnsupportedSetting;
        }
      } else {
        if (formatParams->eCompressionFormat != OMX_VIDEO_CodingUnused) {
          return OMX_ErrorUnsupportedSetting;
        }

        if ((formatParams->eColorFormat != OMX_COLOR_FormatYUV420Planar) &&
            (formatParams->eColorFormat != ((OMX_COLOR_FORMATTYPE)HAL_PIXEL_FORMAT_RGBA_8888))) {
          ALOGI("unsupported video format : %d", formatParams->eColorFormat);
          return OMX_ErrorUnsupportedSetting;
        }

        mOutputFormat = formatParams->eColorFormat;
        updatePortDefinitions(false, false);
      }

      return OMX_ErrorNone;
    }

    case kEnableAndroidNativeHandleExtensionIndex:
    {
      m_android_native_buffers =
          (bool)(((EnableAndroidNativeBuffersParams *)param)->enable);

      return OMX_ErrorNone;
    }

    case kAllocateNativeHandleExtensionIndex:
    {
      m_allocate_native_handle =
          (bool)(((AllocateNativeHandleParams *)param)->enable);

      return OMX_ErrorNone;
    }

    case kStoreANWBufferInMetadataExtensionIndex:
    {
      bool enable =
          (bool)(((StoreMetaDataInBuffersParams *)param)->bStoreMetaData);

      if (enable != m_store_meta_in_buffers) {
        m_store_meta_in_buffers = enable;
        updatePortDefinitions(false, false);
      }

      return OMX_ErrorNone;
    }

    case kStoreMetaDataInBuffersExtensionIndex:
    {
      bool enable =
          (bool)(((StoreMetaDataInBuffersParams *)param)->bStoreMetaData);

      if (enable != m_store_meta_in_buffers) {
        m_store_meta_in_buffers = enable;
        updatePortDefinitions(false, false);
      }

      return OMX_ErrorNone;
    }

    case kUseAndroidNativeBufferExtensionIndex:
    {
      return OMX_ErrorUnsupportedIndex;
    }

    default:
      return SoftVideoDecoderOMXComponent::internalSetParameter(index, param);
  }
}

OMX_ERRORTYPE NiQuadraOMXDecoder::getExtensionIndex(
        const char *name, OMX_INDEXTYPE *index) {
  ALOGV("enter into getExtensionIndex(), index: %s", name);

  if (!strcmp(name, "OMX.google.android.index.enableAndroidNativeBuffers")) {
    *(int32_t*)index = kEnableAndroidNativeHandleExtensionIndex;
    return OMX_ErrorNone;
  } else if (!strcmp(name, "OMX.google.android.index.allocateNativeHandle")) {
    *(int32_t*)index = kAllocateNativeHandleExtensionIndex;
    return OMX_ErrorNone;
  } else if (!strcmp(name, "OMX.google.android.index.storeANWBufferInMetadata")) {
    *(int32_t*)index = kStoreANWBufferInMetadataExtensionIndex;
    return OMX_ErrorNone;
  } else if (!strcmp(name, "OMX.google.android.index.storeMetaDataInBuffers")) {
    *(int32_t*)index = kStoreMetaDataInBuffersExtensionIndex;
    return OMX_ErrorNone;
  } else if (!strcmp(name, "OMX.google.android.index.useAndroidNativeBuffer2")) {
    *(int32_t*)index = kUseAndroidNativeBuffer2ExtensionIndex;
    return OMX_ErrorNone;
  } else if (!strcmp(name, "OMX.google.android.index.useAndroidNativeBuffer")) {
    *(int32_t*)index = kUseAndroidNativeBufferExtensionIndex;
    return OMX_ErrorNone;
  } else if (!strcmp(name, "OMX.google.android.index.getAndroidNativeBufferUsage")) {
    *(int32_t*)index = kGetAndroidNativeBufferUsageExtensionIndex;
    return OMX_ErrorNone;
  }

  return SoftVideoDecoderOMXComponent::getExtensionIndex(name, index);
}

int NiQuadraOMXDecoder::getColorAspectPreference() {
  return kPreferBitstream;
}

#define DEFAULT_DECODER_MAX_LOAD (80)

bool NiQuadraOMXDecoder::check_ni_decoder_is_available() {
  char property[PROPERTY_VALUE_MAX] = {0};
  int dec_max_load = DEFAULT_DECODER_MAX_LOAD;
  ni_retcode_t ret = NI_RETCODE_SUCCESS;
  ni_device_context_t *p_device_ctx = NULL;
  ni_session_context_t *p_ctx = NULL;
  ni_device_info_t *decoders = NULL;
  bool is_available = false;
  bool is_opened = false;
  int count = 0;

  if (property_get(PROP_DECODER_MAX_LOAD, property, NULL)) {
    int load = atoi(property);
    if (load > 0 && load < 100) {
      ALOGI("get property ro.vmi.quadraDecodeMaxLoad:%s", property);
      dec_max_load = load;
    }
  }

  decoders = (ni_device_info_t *)calloc(NI_MAX_DEVICE_CNT,
      sizeof(ni_device_info_t));
  if (!decoders) {
    ALOGE("%s: Failed to allocate ni_device_info_t", __FUNCTION__);
    return false;
  }

  p_ctx = ni_device_session_context_alloc_init();
  if(!p_ctx) {
    ALOGE("%s: Failed to allocate ni_session_context_t by "
           "ni_device_session_context_alloc_init()", __FUNCTION__);
    goto EXIT;
  }

  if (NI_RETCODE_SUCCESS ==
      ni_rsrc_list_devices(NI_DEVICE_TYPE_DECODER, decoders, &count)) {
    for (int i = 0; i < count; i++) {
      p_device_ctx = ni_rsrc_get_device_context(NI_DEVICE_TYPE_DECODER,
          decoders[i].module_id);
      if(!p_device_ctx) {
        ALOGE("%s: Failed to get %dth device context", __FUNCTION__, i);
        goto EXIT;
      }

      p_ctx->device_handle =
          ni_device_open(p_device_ctx->p_device_info->dev_name,
                         &p_ctx->max_nvme_io_size);
      if (NI_INVALID_DEVICE_HANDLE == p_ctx->device_handle) {
        ALOGE("ERROR: ni_device_open() failed for %s: %s",
            p_device_ctx->p_device_info->dev_name,
            strerror(NI_ERRNO));
        goto EXIT;
      } else {
        is_opened = true;
        p_ctx->blk_io_handle = p_ctx->device_handle;
        p_ctx->hw_id = p_device_ctx->p_device_info->hw_id;
      }

      ret = ni_device_session_query(p_ctx, NI_DEVICE_TYPE_DECODER);
      if (ret != NI_RETCODE_SUCCESS) {
        ALOGE("ERROR: ni_device_session_query() returned %d for decoder:%s:%d",
            ret,
            p_device_ctx->p_device_info->dev_name,
            p_device_ctx->p_device_info->hw_id);
        goto EXIT;
      }

      if (!p_ctx->load_query.total_contexts) {
        p_ctx->load_query.current_load = 0;
      }

      int load = ((p_ctx->load_query.total_contexts == 0 ||
          p_ctx->load_query.current_load > p_ctx->load_query.fw_load) ?
          p_ctx->load_query.current_load : p_ctx->load_query.fw_load);
      if (load < dec_max_load) {
        is_available = true;
        break;
      }

      if (is_opened) {
        ni_device_close(p_ctx->device_handle);
        p_ctx->device_handle = NI_INVALID_DEVICE_HANDLE;
        p_ctx->blk_io_handle = NI_INVALID_DEVICE_HANDLE;
        is_opened = false;
      }

      if (p_device_ctx) {
        ni_rsrc_free_device_context(p_device_ctx);
        p_device_ctx = NULL;
      }
    }
  }

EXIT:
  if (decoders) {
    free(decoders);
    decoders = NULL;
  }

  if (p_ctx) {
    if (is_opened) {
      ni_device_close(p_ctx->device_handle);
      p_ctx->device_handle = NI_INVALID_DEVICE_HANDLE;
      p_ctx->blk_io_handle = NI_INVALID_DEVICE_HANDLE;
      is_opened = false;
    }

    ni_device_session_context_clear(p_ctx);
    free(p_ctx);
    p_ctx = NULL;
  }

  if (p_device_ctx) {
    ni_rsrc_free_device_context(p_device_ctx);
    p_device_ctx = NULL;
  }

  return is_available;
}

} // namespace android

