/*
 * Copyright 2015 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#define LOG_TAG	"NiQuadraOMXH264Encoder"

#include <media/hardware/HardwareAPI.h>
#include <media/hardware/MetadataBufferType.h>
#include <media/stagefright/foundation/ADebug.h>
#include <media/stagefright/MediaDefs.h>
#include <media/stagefright/MediaErrors.h>
#include <OMX_IndexExt.h>
#include <OMX_VideoExt.h>
#include <OMX_Video.h>

#include <ui/Fence.h>
#include <ui/Rect.h>
#include <ui/GraphicBufferMapper.h>
#include <nativebase/nativebase.h>
#include <hardware/gralloc.h>
#include <ui/GraphicBuffer.h>

#include <cutils/properties.h>

#include "ni_rsrc_api.h"
#include "ni_av_codec.h"
#include "ni_util.h"
#include "ni_log.h"

#include "NiQuadraOMXEncoder.h"

namespace android {

#define AVERROR(e) (-(e))
#define NI_ALIGN(x,a) (((x)+(a)-1)&~((a)-1))

#define PROP_XOCDER_LOG_LEVEL "xcoder_log_level"

typedef struct _xcoder_log_level {
  const char * level_str;
  ni_log_level_t level;
} xcoder_log_level;

static const xcoder_log_level log_level_list[] = {
  { "fatal", NI_LOG_FATAL },
  { "error", NI_LOG_ERROR },
  { "info", NI_LOG_INFO },
  { "debug", NI_LOG_DEBUG },
  { "trace", NI_LOG_TRACE }
};

static ni_log_level_t convert_ni_log_level(const char *log_level) {
  int i = 0;
  int tablesz = sizeof(log_level_list) / sizeof(xcoder_log_level);

  if (log_level) {
    for (i = 0; i < tablesz; i++) {
      if (!strcasecmp(log_level, log_level_list[i].level_str)) {
        return log_level_list[i].level;
      }
    }
  }

  ALOGE("Error: invalid xcoder log level is specified: %s", log_level);
  return NI_LOG_INVALID;
}

NiQuadraOMXEncoder::NiQuadraOMXEncoder(
    const char *name,
    const char *componentRole,
    OMX_VIDEO_CODINGTYPE codingType,
    const char *mimeType,
    const CodecProfileLevel *profileLevels,
    size_t numProfileLevels,
    const OMX_CALLBACKTYPE *callbacks,
    OMX_PTR appData,
    OMX_COMPONENTTYPE **component)
  : SoftVideoEncoderOMXComponent(
      name, componentRole, codingType,
      profileLevels, numProfileLevels,
      176 /* width */, 144 /* height */, callbacks,
      appData, component) {

  initPorts(
          kNumBuffers, kNumBuffers, ((mWidth * mHeight * 3) >> 1),
          mimeType, 2);

  m_intra_period = -1;
  m_ltr_interval = -1;
  m_ltr_frame_ref_invalid = -1;
  m_max_frame_size = -1;
  m_crf = -1;
  m_crf2 = -1.0;
  m_vbvMaxRate = -1;
  m_vbvBufferSize = -1;
  m_max_frame_size_ratio = -1;
  memset(&m_ltr, 0, sizeof(ni_long_term_ref_t));
  memset(&m_min_max_qp, 0, sizeof(ni_rc_min_max_qp));

  m_signalled_error = false;
  m_is_initialized = false;
  m_input_eos = false;
  m_output_eos = false;
  m_send_eos = false;
  m_is_closed = false;
  m_ctx = NULL;

  memset(&m_sParamAVC, 0, sizeof(OMX_VIDEO_PARAM_AVCTYPE));
  m_sParamAVC.nSize = sizeof(OMX_VIDEO_PARAM_AVCTYPE);
  m_sParamAVC.nVersion.s.nVersionMajor = 1;
  m_sParamAVC.nVersion.s.nVersionMinor = 0;
  m_sParamAVC.nVersion.s.nRevision = 0;
  m_sParamAVC.nVersion.s.nStep = 0;
  m_sParamAVC.nPortIndex = (OMX_U32) kOutputPortIndex;
  m_sParamAVC.eProfile =  OMX_VIDEO_AVCProfileHigh;
  m_sParamAVC.eLevel =  OMX_VIDEO_AVCLevel4;

  memset(&m_sParamHEVC, 0, sizeof(OMX_VIDEO_PARAM_HEVCTYPE));
  m_sParamHEVC.nSize = sizeof(OMX_VIDEO_PARAM_HEVCTYPE);
  m_sParamHEVC.nVersion.s.nVersionMajor = 1;
  m_sParamHEVC.nVersion.s.nVersionMinor = 0;
  m_sParamHEVC.nVersion.s.nRevision = 0;
  m_sParamHEVC.nVersion.s.nStep = 0;
  m_sParamHEVC.nPortIndex = (OMX_U32) kOutputPortIndex;
  m_sParamHEVC.eProfile =  OMX_VIDEO_HEVCProfileMain;
  m_sParamHEVC.eLevel =  OMX_VIDEO_HEVCMainTierLevel1;

  memset(m_convert_buffers, 0, sizeof(m_convert_buffers));
  memset(m_convert_buffers_free, 0, sizeof(m_convert_buffers_free));
  m_cb_index = -1;

  init_vendor_extensions(m_vendor_extension_store);
  m_vendor_extension_store.dumpExtensions();
}

NiQuadraOMXEncoder::~NiQuadraOMXEncoder() {
  ALOGI("enter into ~NiQuadraOMXEncoder()");

  if (!m_ctx) {
    ALOGI("m_ctx isn't created, so no need to release");
    return;
  }

  xcoder_encode_close();
  m_xcoder_params.clear();
  free(m_ctx);
  m_ctx = NULL;

  List<BufferInfo *> &outQueue = getPortQueue(1);
  List<BufferInfo *> &inQueue = getPortQueue(0);
  CHECK(outQueue.empty());
  CHECK(inQueue.empty());
}

int NiQuadraOMXEncoder::xcoder_setup_encoder() {
  ni_session_context_t *api_ctx = nullptr;
  ni_xcoder_params_t *p_param = nullptr;
  ni_xcoder_params_t *pparams = nullptr;
  int ret = 0;

  ALOGI("enter into xcoder_setup_encoder()");

  api_ctx = &m_ctx->api_ctx;
  p_param = &m_ctx->api_param;

  m_ctx->started = 0;
  m_ctx->m_sps_pps_arrived = false;
  m_ctx->m_first_pkt_arrived = false;
  m_ctx->m_spsPpsHdr = nullptr;
  m_ctx->m_spsPpsHdrLen = 0;
  m_ctx->m_total_frames_received = 0;
  m_send_eos = false;

  memset(&m_ctx->api_fme, 0, sizeof(ni_session_data_io_t));
  memset(&m_ctx->api_pkt, 0, sizeof(ni_session_data_io_t));

  if ((ret = ni_device_session_context_init(api_ctx)) < 0) {
    ALOGE("Init NETINT encoder context failure : %d\n", ret);
    return ret;
  }

  api_ctx->codec_format = m_codec_type;

  ret = ni_encoder_init_default_params(p_param,
    25, 1, mBitrate, mWidth, mHeight, (ni_codec_format_t)api_ctx->codec_format);
  switch (ret) {
  case NI_RETCODE_PARAM_ERROR_WIDTH_TOO_BIG:
    ALOGE("Invalid Picture Width: too big\n");
    break;
  case NI_RETCODE_PARAM_ERROR_WIDTH_TOO_SMALL:
    ALOGE("Invalid Picture Width: too small\n");
    break;
  case NI_RETCODE_PARAM_ERROR_HEIGHT_TOO_BIG:
    ALOGE("Invalid Picture Height: too big\n");
    break;
  case NI_RETCODE_PARAM_ERROR_HEIGHT_TOO_SMALL:
    ALOGE("Invalid Picture Height: too small\n");
    break;
  case NI_RETCODE_PARAM_ERROR_AREA_TOO_BIG:
    ALOGE("Invalid Picture Width x Height: exceeds %d\n", NI_MAX_RESOLUTION_AREA);
    break;
  case NI_RETCODE_PARAM_ERROR_PIC_WIDTH:
    ALOGE("Invalid Picture Width\n");
    break;
  case NI_RETCODE_PARAM_ERROR_PIC_HEIGHT:
    ALOGE("Invalid Picture Height\n");
    break;
  default:
    if (ret < 0) {
      ALOGE("Init NETINT encoder params failure\n");
    }
    break;
  }

  if (ret < 0) {
    return AVERROR(EINVAL);
  }

  //todo: support multiple format
  switch (mColorFormat) {
  	case OMX_COLOR_FormatYUV420Planar:
      api_ctx->pixel_format = NI_PIX_FMT_YUV420P;
      api_ctx->bit_depth_factor = 1;
      break;
    case OMX_COLOR_FormatYUV420SemiPlanar:
      api_ctx->pixel_format = NI_PIX_FMT_NV12;
      api_ctx->bit_depth_factor = 1;
      break;
	default:
      api_ctx->pixel_format = NI_PIX_FMT_RGBA;
      api_ctx->bit_depth_factor = 4;
      break;
  }

  for (auto & param : m_xcoder_params) {
    const char *key = param.first.c_str();
    const char *value = param.second.c_str();
    ret = ni_encoder_params_set_value(p_param, key, value);
    if (ret < 0) {
      ALOGE("set encoder's initialization parameter %s:%s failure\n", key, value);
      return ret;
    } else {
      ALOGI("set encoder's initialization parameter %s:%s success\n", key, value);
    }
  }

  api_ctx->hw_id = -1;
  api_ctx->src_bit_depth = 8;
  api_ctx->src_endian = NI_FRAME_LITTLE_ENDIAN;
  api_ctx->keep_alive_timeout = NI_DEFAULT_KEEP_ALIVE_TIMEOUT;

  api_ctx->sei_hdr_content_light_level_info_len = 0;
  api_ctx->light_level_data_len = 0;
  api_ctx->sei_hdr_mastering_display_color_vol_len = 0;
  api_ctx->mdcv_max_min_lum_data_len = 0;
  api_ctx->p_master_display_meta_data = NULL;

  api_ctx->p_session_config = &m_ctx->api_param;
  pparams = (ni_xcoder_params_t *)api_ctx->p_session_config;

  //todo: support multiple format
#if 1
  switch (mColorFormat) {
    case OMX_COLOR_FormatYUV420SemiPlanar:
      pparams->cfg_enc_params.planar = NI_PIXEL_PLANAR_FORMAT_SEMIPLANAR;
      break;
	default:
      pparams->cfg_enc_params.planar = NI_PIXEL_PLANAR_FORMAT_PLANAR;
      break;
  }
#else
  pparams->cfg_enc_params.planar = NI_PIXEL_PLANAR_FORMAT_PLANAR;
#endif

  //ni_frame_buffer_alloc(&(m_ctx->api_fme.data.frame), mWidth,
  //    mHeight, 0, 0, api_ctx->bit_depth_factor, 1, 1);

  api_ctx->ori_width = mWidth;
  api_ctx->ori_height = mHeight;
  api_ctx->ori_bit_depth_factor = api_ctx->bit_depth_factor;
  api_ctx->ori_pix_fmt = api_ctx->pixel_format;

  api_ctx->ori_luma_linesize = 0;
  api_ctx->ori_chroma_linesize = 0; 

  return 0;
}

void NiQuadraOMXEncoder::init_vendor_extensions(VendorExtensionStore &store) {
  //initialization parameter
  ADD_EXTENSION("quadra-ext-enc-bitrate", QUADRA_IndexConfigVideoBitrate, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-profile", QUADRA_IndexConfigVideoProfile, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-level", QUADRA_IndexConfigVideoLevel, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-gopPresetIdx", QUADRA_IndexConfigVideoGopPresetIdx, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-lowDelay", QUADRA_IndexConfigVideoLowDelay, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-useLowDelayPocType", QUADRA_IndexConfigVideoUseLowDelayPocType, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-RcEnable", QUADRA_IndexConfigVideoRcEnable, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-cuLevelRCEnable", QUADRA_IndexConfigVideoCuLevelRCEnable, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-intraPeriod", QUADRA_IndexConfigVideoIntraPeriod, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-intraQP", QUADRA_IndexConfigVideoIntraQP, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-sliceMode", QUADRA_IndexConfigVideoSliceMode, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-sliceArg", QUADRA_IndexConfigVideoSliceArg, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-intraRefreshMode", QUADRA_IndexConfigVideoIntraRefreshMode, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-intraRefreshArg", QUADRA_IndexConfigVideoIntraRefreshArg, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-intraMbRefreshMode", QUADRA_IndexConfigVideoIntraMbRefreshMode, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-intraMbRefreshArg", QUADRA_IndexConfigVideoIntraMbRefreshArg, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-mbLevelRcEnable", QUADRA_IndexConfigVideoMbLevelRcEnable, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-crf", QUADRA_IndexConfigVideoCrf, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-crfFloat", QUADRA_IndexConfigVideoCrfFloat, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-rdoLevel", QUADRA_IndexConfigVideoRdoLevel, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-EnableRdoQuant", QUADRA_IndexConfigVideoEnableRdoQuant, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-lookAheadDepth", QUADRA_IndexConfigVideoLookAheadDepth, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-vbvBufferSize", QUADRA_IndexConfigVideoVbvBufferSize, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-vbvMaxRate", QUADRA_IndexConfigVideoVbvMaxRate, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-multicoreJointMode", QUADRA_IndexConfigVideoMulticoreJointMode, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-qlevel", QUADRA_IndexConfigVideoQlevel, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-chromaQpOffset", QUADRA_IndexConfigVideoChromaQpOffset, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-tolCtbRcInter", QUADRA_IndexConfigVideoTolCtbRcInter, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-tolCtbRcIntra", QUADRA_IndexConfigVideoTolCtbRcIntra, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-enable2PassGop", QUADRA_IndexConfigVideoEnable2PassGop, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-zeroCopyMode", QUADRA_IndexConfigVideoZeroCopyMode, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  //reconfigure parameter
  ADD_EXTENSION("quadra-ext-enc-reconfig-bitrate", QUADRA_IndexConfigVideoReconfigBitrate, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueInt32)

  ADD_EXTENSION("quadra-ext-enc-reconfig-intraPeriod", QUADRA_IndexConfigVideoReconfigIntraPeriod, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueInt32)

  ADD_EXTENSION("quadra-ext-enc-reconfig-force-idr-frame", QUADRA_IndexConfigVideoReconfigForceIDRFrame, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueInt32)

  ADD_EXTENSION("quadra-ext-enc-reconfig-ltr", QUADRA_IndexConfigVideoReconfigLTR, OMX_DirOutput)
  ADD_PARAM    ("use_cur_src_as_long_term_pic", OMX_AndroidVendorValueInt32)
  ADD_PARAM_END("use_long_term_ref", OMX_AndroidVendorValueInt32)

  ADD_EXTENSION("quadra-ext-enc-reconfig-ltr-interval", QUADRA_IndexConfigVideoReconfigLTRInterval, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueInt32)

  ADD_EXTENSION("quadra-ext-enc-reconfig-frame-ref-invalid", QUADRA_IndexConfigVideoReconfigFrameRefInvalid, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueInt32)

  ADD_EXTENSION("quadra-ext-enc-reconfig-max-frame-size", QUADRA_IndexConfigVideoReconfigMaxFrameSize, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueInt32)

  ADD_EXTENSION("quadra-ext-enc-reconfig-min-max-qp", QUADRA_IndexConfigVideoReconfigMinMaxQp, OMX_DirOutput)
  ADD_PARAM    ("minQpI", OMX_AndroidVendorValueInt32)
  ADD_PARAM    ("maxQpI", OMX_AndroidVendorValueInt32)
  ADD_PARAM    ("maxDeltaQp", OMX_AndroidVendorValueInt32)
  ADD_PARAM    ("minQpPB", OMX_AndroidVendorValueInt32)
  ADD_PARAM_END("maxQpPB", OMX_AndroidVendorValueInt32)

  ADD_EXTENSION("quadra-ext-enc-reconfig-crf", QUADRA_IndexConfigVideoReconfigCRF, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueInt32)

  ADD_EXTENSION("quadra-ext-enc-reconfig-crf2", QUADRA_IndexConfigVideoReconfigCRF2, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueString)

  ADD_EXTENSION("quadra-ext-enc-reconfig-vbv", QUADRA_IndexConfigVideoReconfigVbv, OMX_DirOutput)
  ADD_PARAM    ("vbvMaxRate", OMX_AndroidVendorValueInt32)
  ADD_PARAM_END("vbvBufferSize", OMX_AndroidVendorValueInt32)

  ADD_EXTENSION("quadra-ext-enc-reconfig-max-frame-size-ratio", QUADRA_IndexConfigVideoReconfigMaxFrameSizeRatio, OMX_DirOutput)
  ADD_PARAM_END("value", OMX_AndroidVendorValueInt32)
}

OMX_ERRORTYPE NiQuadraOMXEncoder::get_vendor_extension_config(
        OMX_CONFIG_ANDROID_VENDOR_EXTENSIONTYPE *ext) {
  if (ext->nIndex >= m_vendor_extension_store.size()) {
    return OMX_ErrorNoMore;
  }

  const VendorExtension& vExt = m_vendor_extension_store[ext->nIndex];
  ALOGV("VendorExt: getConfig: index=%u (%s)", ext->nIndex, vExt.name());

  vExt.copyInfoTo(ext);
  if (ext->nParamSizeUsed < vExt.paramCount()) {
    // this happens during initial getConfig to query only extension-name and param-count
    return OMX_ErrorNone;
  }

  return OMX_ErrorNone;
}

OMX_ERRORTYPE NiQuadraOMXEncoder::set_vendor_extension_config(
        OMX_CONFIG_ANDROID_VENDOR_EXTENSIONTYPE *ext) {
  if (ext->nIndex >= m_vendor_extension_store.size()) {
    return OMX_ErrorBadParameter;
  }

  const VendorExtension& vExt = m_vendor_extension_store[ext->nIndex];
  ALOGV("VendorExt: setConfig: index=%u (%s)", ext->nIndex, vExt.name());

  OMX_ERRORTYPE err = OMX_ErrorNone;
  err = vExt.isConfigValid(ext);
  if (err != OMX_ErrorNone) {
    return err;
  }

  vExt.set();

  bool valueSet = false;
  switch ((OMX_U32)vExt.extensionIndex()) {
    //initialization parameter
    case QUADRA_IndexConfigVideoBitrate:
    case QUADRA_IndexConfigVideoProfile:
    case QUADRA_IndexConfigVideoLevel:
    case QUADRA_IndexConfigVideoGopPresetIdx:
    case QUADRA_IndexConfigVideoLowDelay:
    case QUADRA_IndexConfigVideoUseLowDelayPocType:
    case QUADRA_IndexConfigVideoRcEnable:
    case QUADRA_IndexConfigVideoCuLevelRCEnable:
    case QUADRA_IndexConfigVideoIntraPeriod:
    case QUADRA_IndexConfigVideoIntraQP:
    case QUADRA_IndexConfigVideoSliceMode:
    case QUADRA_IndexConfigVideoSliceArg:
    case QUADRA_IndexConfigVideoIntraRefreshMode:
    case QUADRA_IndexConfigVideoIntraRefreshArg:
    case QUADRA_IndexConfigVideoIntraMbRefreshMode:
    case QUADRA_IndexConfigVideoIntraMbRefreshArg:
    case QUADRA_IndexConfigVideoMbLevelRcEnable:
    case QUADRA_IndexConfigVideoCrf:
    case QUADRA_IndexConfigVideoCrfFloat:
    case QUADRA_IndexConfigVideoRdoLevel:
    case QUADRA_IndexConfigVideoEnableRdoQuant:
    case QUADRA_IndexConfigVideoLookAheadDepth:
    case QUADRA_IndexConfigVideoVbvBufferSize:
    case QUADRA_IndexConfigVideoVbvMaxRate:
    case QUADRA_IndexConfigVideoMulticoreJointMode:
    case QUADRA_IndexConfigVideoQlevel:
    case QUADRA_IndexConfigVideoChromaQpOffset:
    case QUADRA_IndexConfigVideoTolCtbRcInter:
    case QUADRA_IndexConfigVideoTolCtbRcIntra:
    case QUADRA_IndexConfigVideoEnable2PassGop:
    case QUADRA_IndexConfigVideoZeroCopyMode:
    {
      if (m_is_initialized) {
        ALOGE("configure initialization parameter but xcoder session is initialized yet");
        return OMX_ErrorIncorrectStateOperation;
      }

      char value[OMX_MAX_STRINGVALUE_SIZE];
      valueSet |= vExt.readParamString(ext, "value", value);
      if (!valueSet) {
        break;
      }

      std::string key(vExt.name() + strlen("quadra-ext-enc-"));
      m_xcoder_params[key] = value;
      ALOGI("VendorExt setConfig: name[%s], value[%s]", vExt.name(), value);

      if ((OMX_U32)vExt.extensionIndex() == QUADRA_IndexConfigVideoBitrate) {
        char *endptr;
        long bitrate = strtol(value, &endptr, 10);
        if (endptr == value || *endptr != '\0' || bitrate < 0 || bitrate > INT_MAX) {
          ALOGE("Invalid bitrate value: %s", value);
          return OMX_ErrorBadParameter;
        }
        mBitrate = (OMX_U32)bitrate;
      }

      break;
    }

    //reconfigure parameter
    case QUADRA_IndexConfigVideoReconfigBitrate:
    {
      int32_t bitrate = 0;
      valueSet |= vExt.readParamInt32(ext, "value", (OMX_S32 *)(&bitrate));
      if (!valueSet) {
        break;
      }

      mBitrate = bitrate;
      err = xcoder_reconfig_bitrate();
      break;
    }
    case QUADRA_IndexConfigVideoReconfigIntraPeriod:
    {
      int32_t intraPeriod = 0;
      valueSet |= vExt.readParamInt32(ext, "value", (OMX_S32 *)(&intraPeriod));
      if (!valueSet) {
        break;
      }

      m_intra_period = intraPeriod;
      err = xcoder_reconfig_intraprd();
      break;
    }
    case QUADRA_IndexConfigVideoReconfigForceIDRFrame:
    {
      int32_t force_idr_frame = 0;
      valueSet |= vExt.readParamInt32(ext, "value", (OMX_S32 *)(&force_idr_frame));
      if (!valueSet) {
        break;
      }

      if (0 == force_idr_frame) {
        return OMX_ErrorBadParameter;
      }

      err = xcoder_force_idr_frame();
      break;
    }
    case QUADRA_IndexConfigVideoReconfigLTR:
    {
      int32_t use_cur_src_as_long_term_pic = 0;
      int32_t use_long_term_ref = 0;
      valueSet |= vExt.readParamInt32(ext, "use_cur_src_as_long_term_pic", (OMX_S32 *)(&use_cur_src_as_long_term_pic));
      valueSet &= vExt.readParamInt32(ext, "use_long_term_ref", (OMX_S32 *)(&use_long_term_ref));
      if (!valueSet) {
        break;
      }

      m_ltr.use_cur_src_as_long_term_pic = (uint8_t)use_cur_src_as_long_term_pic;
      m_ltr.use_long_term_ref = (uint8_t)use_long_term_ref;
      err = xcoder_reconfig_ltr();
      break;
    }
    case QUADRA_IndexConfigVideoReconfigLTRInterval:
    {
      int32_t ltr_interval = 0;
      valueSet |= vExt.readParamInt32(ext, "value", (OMX_S32 *)(&ltr_interval));
      if (!valueSet) {
        break;
      }

      m_ltr_interval = ltr_interval;
      err = xcoder_reconfig_ltr_interval();
      break;
    }
    case QUADRA_IndexConfigVideoReconfigFrameRefInvalid:
    {
      int32_t frame_num = 0;
      valueSet |= vExt.readParamInt32(ext, "value", (OMX_S32 *)(&frame_num));
      if (!valueSet) {
        break;
      }

      m_ltr_frame_ref_invalid = frame_num;
      err = xcoder_set_frame_ref_invalid();
      break;
    }
    case QUADRA_IndexConfigVideoReconfigMaxFrameSize:
    {
      int32_t max_frame_size = 0;
      valueSet |= vExt.readParamInt32(ext, "value", (OMX_S32 *)(&max_frame_size));
      if (!valueSet) {
        break;
      }

      m_max_frame_size = max_frame_size;
      err = xcoder_reconfig_max_frame_size();
      break;
    }
    case QUADRA_IndexConfigVideoReconfigMinMaxQp:
    {
      int32_t minQpI = 0;
      int32_t maxQpI = 0;
      int32_t maxDeltaQp = 0;
      int32_t minQpPB = 0;
      int32_t maxQpPB = 0;
      valueSet |= vExt.readParamInt32(ext, "minQpI", (OMX_S32 *)(&minQpI));
      valueSet &= vExt.readParamInt32(ext, "maxQpI", (OMX_S32 *)(&maxQpI));
      valueSet &= vExt.readParamInt32(ext, "maxDeltaQp", (OMX_S32 *)(&maxDeltaQp));
      valueSet &= vExt.readParamInt32(ext, "minQpPB", (OMX_S32 *)(&minQpPB));
      valueSet &= vExt.readParamInt32(ext, "maxQpPB", (OMX_S32 *)(&maxQpPB));

      if (!valueSet) {
        break;
      }

      m_min_max_qp.minQpI = minQpI;
      m_min_max_qp.maxQpI = maxQpI;
      m_min_max_qp.maxDeltaQp = maxDeltaQp;
      m_min_max_qp.minQpPB = minQpPB;
      m_min_max_qp.maxQpPB = maxQpPB;
      err = xcoder_reconfig_min_max_qp();
      break;
    }
    case QUADRA_IndexConfigVideoReconfigCRF:
    {
      int32_t crf = 0;
      valueSet |= vExt.readParamInt32(ext, "value", (OMX_S32 *)(&crf));
      if (!valueSet) {
        break;
      }

      m_crf = crf;
      err = xcoder_reconfig_crf();
      break;
    }
    case QUADRA_IndexConfigVideoReconfigCRF2:
    {
      float crf2 = -1.0;
      char value[OMX_MAX_STRINGVALUE_SIZE];
      valueSet |= vExt.readParamString(ext, "value", value);
      if (!valueSet) {
        break;
      }

      crf2 = strtof(value, NULL);
      m_crf2 = crf2;
      err = xcoder_reconfig_crf2();
      break;
    }
    case QUADRA_IndexConfigVideoReconfigVbv:
    {
      int32_t vbvMaxRate = 0;
      int32_t vbvBufferSize = 0;
      valueSet |= vExt.readParamInt32(ext, "vbvMaxRate", (OMX_S32 *)(&vbvMaxRate));
      valueSet &= vExt.readParamInt32(ext, "vbvBufferSize", (OMX_S32 *)(&vbvBufferSize));
      if (!valueSet) {
        break;
      }

      m_vbvMaxRate = vbvMaxRate;
      m_vbvBufferSize = vbvBufferSize;
      err = xcoder_reconfig_vbv();
      break;
    }
    case QUADRA_IndexConfigVideoReconfigMaxFrameSizeRatio:
    {
      int32_t max_frame_size_ratio = 0;
      valueSet |= vExt.readParamInt32(ext, "value", (OMX_S32 *)(&max_frame_size_ratio));
      if (!valueSet) {
        break;
      }

      m_max_frame_size_ratio = max_frame_size_ratio;
      err = xcoder_reconfig_max_frame_size_ratio();
      break;
    }
    default:
    {
      return OMX_ErrorNotImplemented;
    }
  }

  return err;
}

int NiQuadraOMXEncoder::xcoder_encode_init() {
  ni_session_context_t *api_ctx = nullptr;
  ni_xcoder_params_t *p_param = nullptr;
  int src_stride[NI_MAX_NUM_DATA_POINTERS] = {0};
  int linesize_aligned = mWidth;
  int height_aligned = mHeight;
  int ret = 0;

  ALOGI("enter into xcoder_encode_init()");

  ret = xcoder_setup_encoder();
  if (ret < 0) {
    ALOGE("Error : xcoder_setup_encoder failed (%d)", ret);
    return ret; 
  }

  ALOGI("xcoder_setup_encoder success");

  api_ctx = &m_ctx->api_ctx;
  p_param = (ni_xcoder_params_t *)api_ctx->p_session_config;

  if (linesize_aligned < NI_MIN_WIDTH) {
    p_param->cfg_enc_params.conf_win_right += (NI_MIN_WIDTH - mWidth) / 2 * 2;
    linesize_aligned = NI_MIN_WIDTH;
  } else {
    linesize_aligned = NI_ALIGN(mWidth, 2);
    p_param->cfg_enc_params.conf_win_right += (linesize_aligned - mWidth) / 2 * 2;
  }
  p_param->source_width = linesize_aligned;

  if (height_aligned < NI_MIN_HEIGHT) {
    p_param->cfg_enc_params.conf_win_bottom += (NI_MIN_HEIGHT - mHeight) / 2 * 2;
    height_aligned = NI_MIN_HEIGHT;
  } else {
    height_aligned = NI_ALIGN(mHeight, 2);
    p_param->cfg_enc_params.conf_win_bottom += (height_aligned - mHeight) / 2 * 2;
  }
  p_param->source_height = height_aligned;

  ALOGI("linesize_aligned : %d, bit_depth_factor : %d", linesize_aligned, api_ctx->bit_depth_factor);
  //todo : support mutiply format
  switch (api_ctx->pixel_format) {
  	case NI_PIX_FMT_YUV420P:
      src_stride[0] = linesize_aligned * api_ctx->bit_depth_factor;
      src_stride[1] = src_stride[0] / 2;
      src_stride[2] = src_stride[0] / 2;
      ALOGI("pixel_format NI_PIX_FMT_YUV420P, src_stride[0]:%d, src_stride[1]:%d, src_stride[2] :%d", src_stride[0], src_stride[1] , src_stride[2]);
      break;
    case NI_PIX_FMT_NV12:
      src_stride[0] = linesize_aligned * api_ctx->bit_depth_factor;
      src_stride[1] = src_stride[0];
      src_stride[2] = 0;
      ALOGI("pixel_format NI_PIX_FMT_NV12, src_stride[0]:%d, src_stride[1]:%d, src_stride[2] :%d", src_stride[0], src_stride[1] , src_stride[2]);
      break;
	default:
      src_stride[0] = linesize_aligned * api_ctx->bit_depth_factor;
      src_stride[1] = 0;
      src_stride[2] = 0;
      ALOGI("pixel_format NI_PIX_FMT_RGBA, src_stride[0]:%d, src_stride[1]:%d, src_stride[2] :%d", src_stride[0], src_stride[1] , src_stride[2]);
      break;
  }

  ret = ni_encoder_frame_zerocopy_check(api_ctx, p_param, mWidth, mHeight,
    (const int *)src_stride, true);
  if (ret == NI_RETCODE_SUCCESS) {
    ALOGI("ni_encoder_frame_zerocopy_check success");
  }

  //init encoder session
  ret = ni_device_session_open(api_ctx, NI_DEVICE_TYPE_ENCODER);
  switch (ret) {
  case NI_RETCODE_SUCCESS:
    ALOGI("XCoder %s.%d (inst: %d) opened successfully\n",
        api_ctx->dev_xcoder_name, api_ctx->hw_id, api_ctx->session_id);
    break;
  case NI_RETCODE_INVALID_PARAM:
    ALOGE("Failed to open encoder (status = %d), invalid parameter values\n", ret);
    break;
  default:
    ALOGE("Failed to open encoder (status = %d), resource unavailable\n", ret);
    break;
  }
  if (ret < 0) {
    ALOGE("Failed to open NETINT encoder (status = %d)\n", ret);
    return ret;
  }

  m_ctx->dev_enc_idx = api_ctx->hw_id;

  if (mInputDataIsMeta) {
    for (size_t i = 0; i < MAX_CONVERSION_BUFFERS; i++) {
      if (m_convert_buffers[i] != nullptr) {
        free(m_convert_buffers[i]);
        m_convert_buffers[i] = nullptr;
      }

      m_convert_buffers[i] = (uint8_t *)malloc(linesize_aligned * mHeight * 4);
      if (m_convert_buffers[i] == nullptr) {
        ALOGE("Allocating conversion buffer failed.");
        return OMX_ErrorUndefined;
      }

      m_convert_buffers_free[i] = 1;
    }
  }

  m_is_initialized = true;

  ALOGI("xcoder_setup_encoder success");
  return 0;
}

bool NiQuadraOMXEncoder::extractGraphicBuffer(
    OMX_BUFFERHEADERTYPE *inHeader,
    uint8_t *conversionBuffer) {
  uint8_t *source = NULL;
  if (!inHeader || !conversionBuffer) {
    return false;
  }

  ALOGV("enter into extractGraphicBuffer()");

  source = inHeader->pBuffer + inHeader->nOffset;
  MetadataBufferType bufferType = *(MetadataBufferType *)source;
  bool usingANWBuffer = (bufferType == kMetadataBufferTypeANWBuffer);
  if (!usingANWBuffer && bufferType != kMetadataBufferTypeGrallocSource) {
    ALOGE("Unsupported metadata type (%d)", bufferType);
    return false;
  }

  buffer_handle_t handle;
  size_t srcStride;
  if (usingANWBuffer) {
    VideoNativeMetadata &nativeMeta = *(VideoNativeMetadata *)source;
    ANativeWindowBuffer *buffer = nativeMeta.pBuffer;
    handle = buffer->handle;
    srcStride = buffer->stride;

    if (nativeMeta.nFenceFd >= 0) {
      sp<Fence> fence = new Fence(nativeMeta.nFenceFd);
      nativeMeta.nFenceFd = -1;
      status_t err = fence->wait(kFenceTimeoutMs);
      if (err != OK) {
        ALOGE("Timed out waiting on input fence");
        return false;
      }
    }
  } else {
    VideoGrallocMetadata &grallocMeta = *(VideoGrallocMetadata *)(source);
    handle = grallocMeta.pHandle;
    srcStride = mWidth;
  }

  auto & mapper = GraphicBufferMapper::get();
  void *bits = NULL;
  status_t res = mapper.lock(
                    handle,
                    GRALLOC_USAGE_SW_READ_OFTEN | GRALLOC_USAGE_SW_WRITE_NEVER,
                    Rect(mWidth, mHeight), &bits);
  if (res != OK) {
    ALOGE("Unable to lock image buffer %p for access", handle);
    return false;
  }

  ALOGV("extractGraphicBuffer success, addr:%p, width:%d, height:%d",
      bits, mWidth, mHeight);

  uint8_t *src = (uint8_t *)bits;
  uint8_t *dst = conversionBuffer;
  for (int i = 0; i < mHeight; i++) {
    memcpy(dst, src, mWidth * 4);
    src += srcStride * 4;
    dst += mWidth * 4;
  }

  if (mapper.unlock(handle) != OK) {
    ALOGE("Unable to unlock image buffer %p for access", handle);
  }

  ALOGV("extractGraphicBuffer success");
  return true;
}

int NiQuadraOMXEncoder::xcoder_send_frame(
    OMX_BUFFERHEADERTYPE *inHeader, bool *send_frame) {
  ni_session_context_t *api_ctx = nullptr;
  ni_session_data_io_t *api_fme = nullptr;
  ni_xcoder_params_t *p_param = nullptr;
  int linesize_aligned = NI_ALIGN(mWidth, 2);
  int src_stride[NI_MAX_NUM_DATA_POINTERS]       = {0};
  int height_aligned[NI_MAX_NUM_DATA_POINTERS]   = {0};
  int dst_stride[NI_MAX_NUM_DATA_POINTERS]       = {0};
  int src_height[NI_MAX_NUM_DATA_POINTERS]       = {0};
  const uint8_t *p_src[NI_MAX_NUM_DATA_POINTERS] = {0};
  ni_frame_t dec_frame;
  uint8_t mdcv_data[NI_MAX_SEI_DATA];
  uint8_t cll_data[NI_MAX_SEI_DATA];
  uint8_t cc_data[NI_MAX_SEI_DATA];
  uint8_t udu_data[NI_MAX_SEI_DATA];
  uint8_t hdrp_data[NI_MAX_SEI_DATA];
  uint8_t *source = NULL;
  bool alignment_2pass_wa = false;
  bool isnv12frame = false;
  int need_to_copy = 1;
  int sent = 0;
  int ret = 0;

  api_ctx = &m_ctx->api_ctx;
  api_fme = &m_ctx->api_fme;
  p_param = (ni_xcoder_params_t *)api_ctx->p_session_config;
  memset(&dec_frame, 0, sizeof(ni_frame_t));

  ALOGV("enter into xcoder_send_frame()");

  if (m_send_eos) {
    return 0;
  }

  isnv12frame =
     (api_ctx->pixel_format == NI_PIX_FMT_NV12) ? true : false;

  alignment_2pass_wa = ((p_param->cfg_enc_params.lookAheadDepth ||
                         p_param->cfg_enc_params.crf >= 0 ||
                         p_param->cfg_enc_params.crfFloat >= 0) &&
                        (m_codec_type == NI_CODEC_FORMAT_H265 ||
                         m_codec_type == NI_CODEC_FORMAT_AV1));

  if (inHeader) {
    source = inHeader->pBuffer + inHeader->nOffset;
  }

  api_fme->data.frame.start_of_stream = 0;

  if (! m_ctx->started) {
    api_fme->data.frame.start_of_stream = 1;	
    m_ctx->started = 1;
  }

  api_fme->data.frame.end_of_stream = 0;

  api_fme->data.frame.preferred_characteristics_data_len = 0;
  api_fme->data.frame.force_key_frame = 0;
  api_fme->data.frame.use_cur_src_as_long_term_pic = 0;
  api_fme->data.frame.use_long_term_ref = 0;
  api_fme->data.frame.sei_total_len = 0;
  api_fme->data.frame.sei_cc_offset = 0;
  api_fme->data.frame.sei_cc_len = 0;
  api_fme->data.frame.sei_hdr_mastering_display_color_vol_offset = 0;
  api_fme->data.frame.sei_hdr_mastering_display_color_vol_len = 0;
  api_fme->data.frame.sei_hdr_content_light_level_info_offset = 0;
  api_fme->data.frame.sei_hdr_content_light_level_info_len = 0;
  api_fme->data.frame.sei_hdr_plus_offset = 0;
  api_fme->data.frame.sei_hdr_plus_len = 0;
  api_fme->data.frame.roi_len = 0;
  api_fme->data.frame.reconf_len = 0;
  api_fme->data.frame.force_pic_qp = 0;

  if (inHeader) {
    memset(api_ctx->enc_change_params, 0, sizeof(ni_encoder_change_params_t));
    api_fme->data.frame.extra_data_len = NI_APP_ENC_FRAME_META_DATA_SIZE;

    api_fme->data.frame.ni_pict_type = PIC_TYPE_I;

    ni_enc_prep_aux_data(api_ctx, &api_fme->data.frame,
      &dec_frame, (ni_codec_format_t)api_ctx->codec_format, 1, mdcv_data,
      cll_data, cc_data, udu_data, hdrp_data);

    if (api_fme->data.frame.reconf_len) {
      api_fme->data.frame.extra_data_len += sizeof(ni_encoder_change_params_t);
    }

    api_fme->data.frame.pts = inHeader->nTimeStamp;
    //api_fme->data.frame.dts = inHeader->nTimeStamp;

    api_fme->data.frame.video_width = mWidth;
    api_fme->data.frame.video_height = mHeight;

    switch (api_ctx->pixel_format) {
  	case NI_PIX_FMT_YUV420P:
      src_stride[0] = linesize_aligned * api_ctx->bit_depth_factor;
      src_stride[1] = src_stride[0] / 2;
      src_stride[2] = src_stride[0] / 2;
      src_height[0] = mHeight;
      src_height[1] = mHeight / 2;
      src_height[2] = mHeight / 2;
      break;
    case NI_PIX_FMT_NV12:
      src_stride[0] = linesize_aligned * api_ctx->bit_depth_factor;
      src_stride[1] = src_stride[0];
      src_stride[2] = 0;
      src_height[0] = mHeight;
      src_height[1] = mHeight / 2;
      src_height[2] = 0;
      break;
    case NI_PIX_FMT_RGBA:
    case NI_PIX_FMT_BGRA:
    case NI_PIX_FMT_ABGR:
    case NI_PIX_FMT_ARGB:
    case NI_PIX_FMT_BGR0:
      src_stride[0] = linesize_aligned * api_ctx->bit_depth_factor;
      src_stride[1] = 0;
      src_stride[2] = 0;
      src_height[0] = mHeight;
      src_height[1] = 0;
      src_height[2] = 0;
      alignment_2pass_wa = false;
      break;
    default:
      ALOGE("Error: Invalid pixel format : %d", api_ctx->pixel_format);
      return OMX_ErrorUndefined;
    }

    if (p_param->luma_linesize) {
      alignment_2pass_wa = false;
    }

    ni_get_min_frame_dim(mWidth, mHeight, (ni_pix_fmt_t)api_ctx->pixel_format,
        dst_stride, height_aligned);

    if (alignment_2pass_wa && !mInputDataIsMeta) {
      if (isnv12frame) {
         // for 2-pass encode output mismatch WA, need to extend (and
         // pad) CbCr plane height, because 1st pass assume input 32
         // align
         height_aligned[1] = NI_ALIGN(height_aligned[0], 32) / 2;
       } else {
        // for 2-pass encode output mismatch WA, need to extend (and
        // pad) Cr plane height, because 1st pass assume input 32 align
        height_aligned[2] = NI_ALIGN(height_aligned[0], 32) / 2;
      }
    }

    if (mInputDataIsMeta) {
      uint8_t *conversionBuffer = NULL;
      size_t i = 0;
      for (; i < MAX_CONVERSION_BUFFERS; i++) {
        if (m_convert_buffers_free[i]) {
          m_convert_buffers_free[i] = 0;
          conversionBuffer = m_convert_buffers[i];
          break;
        }
      }

      if (NULL == conversionBuffer) {
        ALOGE("No free buffers to hold conversion data");
        return OMX_ErrorUndefined;
      }

      if (! extractGraphicBuffer(inHeader, conversionBuffer)) {
        ALOGE("Error: extractGraphicBuffer failed");

        m_convert_buffers_free[i] = 1;
        m_cb_index = -1;

        return OMX_ErrorUndefined;
      }

      m_cb_index = i;

      p_src[0] = conversionBuffer;
      p_src[1] = p_src[2] = nullptr; 
    } else {
      switch (api_ctx->pixel_format) {
        case NI_PIX_FMT_YUV420P:
        case NI_PIX_FMT_NV12:
          p_src[0] = source;
          p_src[1] = p_src[0] + src_stride[0] * src_height[0];
          p_src[2] = p_src[1] + src_stride[1] * src_height[1];
          break;
        case NI_PIX_FMT_RGBA:
        case NI_PIX_FMT_BGRA:
        case NI_PIX_FMT_ABGR:
        case NI_PIX_FMT_ARGB:
        case NI_PIX_FMT_BGR0:
          p_src[0] = source;
          p_src[1] = p_src[2] = nullptr;
          break;
        default:
          ALOGE("Error: Invalid pixel format : %d", api_ctx->pixel_format);
          return OMX_ErrorUndefined;
      }
    }

    // check input resolution zero copy compatible or not
    ret = ni_encoder_frame_zerocopy_check(api_ctx, p_param, mWidth, mHeight,
      (const int *)src_stride, false);
    if (ret == NI_RETCODE_SUCCESS) {
      ALOGV("It's zero copy compatible");
      need_to_copy = 0;
      ret = ni_encoder_frame_zerocopy_buffer_alloc(
            &(api_fme->data.frame), mWidth,
            mHeight, (const int *)src_stride, p_src,
            (int)api_fme->data.frame.extra_data_len);
      if (ret < 0) {
        return AVERROR(ENOMEM);
      }
    } else {
      if (p_param->luma_linesize && p_param->chroma_linesize) {
        dst_stride[0] = p_param->luma_linesize;
        dst_stride[1] = p_param->chroma_linesize;
        dst_stride[2] = isnv12frame ? 0 : p_param->chroma_linesize;
      }

      ni_encoder_sw_frame_buffer_alloc(
          !isnv12frame, &(api_fme->data.frame), mWidth,
          height_aligned[0], dst_stride, (m_codec_type == NI_CODEC_FORMAT_H264),
          (int)api_fme->data.frame.extra_data_len, alignment_2pass_wa);
    }

    if (!api_fme->data.frame.p_data[0]) {
      return AVERROR(ENOMEM);
    }

    if (need_to_copy) {
      ni_copy_frame_data(
          (uint8_t **)(api_fme->data.frame.p_data),
          (uint8_t **)p_src, mWidth, mHeight, api_ctx->bit_depth_factor,
          (ni_pix_fmt_t)api_ctx->pixel_format, p_param->cfg_enc_params.conf_win_right,
          dst_stride, height_aligned, src_stride, src_height);
    }

    //copy auxiliary data to ni_frame_t
    ni_enc_copy_aux_data(api_ctx, &api_fme->data.frame, &dec_frame,
        (ni_codec_format_t)api_ctx->codec_format, mdcv_data, cll_data,
        cc_data, udu_data, hdrp_data, 0, isnv12frame);
  } else {
    api_fme->data.frame.end_of_stream = 1;
    m_send_eos = true;
    ALOGI("send eos to vpu\n");
  }

  sent = ni_device_session_write(api_ctx, api_fme, NI_DEVICE_TYPE_ENCODER);
  if (sent < 0) {
    if (NI_RETCODE_ERROR_VPU_RECOVERY == sent) {
      ALOGE("NETINT encoder is in NI_RETCODE_ERROR_VPU_RECOVERY\n");
      xcoder_encode_reset();
      ret = AVERROR(EAGAIN);
    } else {
      ALOGE("send frame data to NETINT encoder failed:%d\n", sent);
      ret = AVERROR(EIO);
    }
  } else if (sent == 0) {
    if (!m_send_eos) {
      ret = AVERROR(EAGAIN);
    }
  } else if (sent > 0) {
    *send_frame = true;
    ret = sent;
  }

  if (*send_frame && mInputDataIsMeta) {
    m_convert_buffers_free[m_cb_index] = 1;
    m_cb_index = -1;

    ALOGV("send frame to vpu success\n");
  }

  return ret;
}

int NiQuadraOMXEncoder::xcoder_receive_packet(
    OMX_BUFFERHEADERTYPE *outHeader, bool *get_pkt) {
  ni_session_context_t *api_ctx = nullptr;
  ni_session_data_io_t *api_pkt = nullptr;
  ni_xcoder_params_t *pparam = nullptr;
  ni_packet_t *xpkt = nullptr;
  uint8_t *p_dst = nullptr;
  int recv = 0;
  int ret = 0;

  ALOGV("enter into xcoder_receive_packet()");

  if (!outHeader) {
    return -1;
  }

  if (m_output_eos) {
    return 0;
  }

  api_ctx = &m_ctx->api_ctx;
  api_pkt = &m_ctx->api_pkt;
  xpkt = &api_pkt->data.packet;
  pparam = (ni_xcoder_params_t *)api_ctx->p_session_config;
  p_dst = outHeader->pBuffer + outHeader->nOffset;

  if (ni_packet_buffer_alloc(xpkt, NI_MAX_TX_SZ)) {
    return AVERROR(ENOMEM);
  }

  while (1) {
    recv = ni_device_session_read(api_ctx, api_pkt,
        NI_DEVICE_TYPE_ENCODER);
    if (recv <= 0) {
      if (xpkt->end_of_stream) {
        ALOGI("Received the last pkt, end_of_stream : 1, pkt len : 0\n");
        outHeader->nFlags |= OMX_BUFFERFLAG_EOS;
        m_output_eos = true;
        *get_pkt = true;
        break;
      }

      if (NI_RETCODE_ERROR_VPU_RECOVERY == recv) {
        ALOGE("NETINT encoder is in NI_RETCODE_ERROR_VPU_RECOVERY\n");
        xcoder_encode_reset();
        ret = AVERROR(EAGAIN);
        break;
      }
      ret = AVERROR(EAGAIN);
      break;
    } else {
      int meta_size = api_ctx->meta_size;

      if (! m_ctx->m_sps_pps_arrived) {
        ALOGI("First received sps/pps medta data\n");

        m_ctx->m_spsPpsHdrLen = xpkt->data_len - meta_size;
        m_ctx->m_spsPpsHdr = (uint8_t *)malloc(m_ctx->m_spsPpsHdrLen);
        memcpy(m_ctx->m_spsPpsHdr, (uint8_t *)xpkt->p_data + meta_size,
                m_ctx->m_spsPpsHdrLen);

        m_ctx->m_sps_pps_arrived = true;
        continue;
      }

      m_ctx->m_total_frames_received++;
      ALOGV("Read %" PRIu64 "th frame from vpu success\n", m_ctx->m_total_frames_received);

      if (xpkt->frame_type == 3) {
        ret = AVERROR(EAGAIN);
        break;
      }

      if (xpkt->frame_type == 0) {
        outHeader->nFlags |= OMX_BUFFERFLAG_SYNCFRAME;
      }

      if (!m_ctx->m_first_pkt_arrived) {
        outHeader->nFlags |= OMX_BUFFERFLAG_CODECCONFIG;
        memcpy(p_dst, m_ctx->m_spsPpsHdr, m_ctx->m_spsPpsHdrLen);
        outHeader->nFilledLen = m_ctx->m_spsPpsHdrLen;
        p_dst += m_ctx->m_spsPpsHdrLen;

        m_ctx->m_first_pkt_arrived = true;
	  }

      memcpy(p_dst, (uint8_t *)xpkt->p_data + meta_size,
              xpkt->data_len - meta_size);
      outHeader->nFilledLen += xpkt->data_len - meta_size;
      outHeader->nTimeStamp = xpkt->pts;

      if (xpkt->end_of_stream) {
        ALOGI("Received the last pkt, end_of_stream : 1, pkt len : %d\n", outHeader->nFilledLen);
        outHeader->nFlags |= OMX_BUFFERFLAG_EOS;
        m_output_eos = true;
      }

      *get_pkt = true;
      break;
    }
  }

  return ret;
}

int NiQuadraOMXEncoder::xcoder_send_receive(
    OMX_BUFFERHEADERTYPE *inHeader, bool *send_frame,
    OMX_BUFFERHEADERTYPE *outHeader, bool *get_pkt) {
  int ret = 0;
  ni_session_data_io_t *api_fme = &m_ctx->api_fme;

  ALOGV("enter into xcoder_send_receive()");

  ret = xcoder_send_frame(inHeader, send_frame);
  ni_frame_buffer_free(&api_fme->data.frame);
  if (ret < 0) {
    ALOGE("Error : xcoder_send_frame error : %d", ret);
    return ret;
  }

  ret = xcoder_receive_packet(outHeader, get_pkt);
  if ((ret < 0) && (ret != AVERROR(EAGAIN))) {
    ALOGE("Error : xcoder_receive_packet error : %d", ret);
    return ret;
  }

  return 0;
}

int NiQuadraOMXEncoder::xcoder_encode_close() {
  ni_session_context_t *api_ctx = NULL;

  ALOGI("enter into xcoder_encode_close()");

  if (m_is_closed) {
    return 0;
  }

  api_ctx = &m_ctx->api_ctx;
  ni_device_session_close(api_ctx, 0, NI_DEVICE_TYPE_ENCODER);
  ni_device_close(api_ctx->device_handle);
  ni_device_close(api_ctx->blk_io_handle);

  ni_packet_buffer_free(&(m_ctx->api_pkt.data.packet));
  ni_frame_buffer_free(&(m_ctx->api_fme.data.frame));

  ni_device_session_context_clear(api_ctx);

  if (m_ctx->m_spsPpsHdr) {
    free(m_ctx->m_spsPpsHdr);
    m_ctx->m_spsPpsHdr = NULL;
  }

  m_ctx->started = 0;
  m_send_eos = false;
  m_is_closed = true;
  m_is_initialized = false;
  memset(m_ctx, 0, sizeof(NiOMXEncCtx));

  if (mInputDataIsMeta) {
    for (size_t i = 0; i < MAX_CONVERSION_BUFFERS; i++) {
      if (m_convert_buffers[i]) {
        free(m_convert_buffers[i]);
        m_convert_buffers[i] = nullptr;
      }
    }
  }
 
  return 0;
}

int NiQuadraOMXEncoder::xcoder_encode_reset() {
  ALOGI("enter into xcoder_encode_reset()");

  xcoder_encode_close();
  return xcoder_encode_init();
}

OMX_ERRORTYPE NiQuadraOMXEncoder::xcoder_reconfig_bitrate() {
  ALOGI("enter into %s", __func__);
  ALOGI("xcoder config bitrate: %d", mBitrate);

  if (!m_is_initialized) {
    m_xcoder_params["bitrate"] = mBitrate;
    return OMX_ErrorNone;
  }

  int ret = ni_reconfig_bitrate(&m_ctx->api_ctx, mBitrate);
  if (ret < 0) {
    ALOGE("xcoder reconfig bitrate: %d error: %d", mBitrate, ret);
    if (ret == NI_RETCODE_INVALID_PARAM) {
      return OMX_ErrorBadParameter;
    } else {
      return OMX_ErrorUndefined;
    }
  }
  return OMX_ErrorNone;
}

OMX_ERRORTYPE NiQuadraOMXEncoder::xcoder_reconfig_intraprd() {
  ALOGI("enter into %s", __func__);
  ALOGI("xcoder config intraPeriod: %d", m_intra_period);

  if (!m_is_initialized) {
    m_xcoder_params["intraPeriod"] = m_intra_period;
    return OMX_ErrorNone;
  }

  int ret = ni_reconfig_intraprd(&m_ctx->api_ctx, m_intra_period);
  if (ret < 0) {
    ALOGE("xcoder reconfig intraPeriod: %d error: %d", m_intra_period, ret);
    if (ret == NI_RETCODE_INVALID_PARAM) {
      return OMX_ErrorBadParameter;
    } else {
      return OMX_ErrorUndefined;
    }
  }
  return OMX_ErrorNone;
}

OMX_ERRORTYPE NiQuadraOMXEncoder::xcoder_force_idr_frame() {
  ALOGI("enter into %s", __func__);

  if (!m_is_initialized) {
    return OMX_ErrorNone;
  }

  int ret = ni_force_idr_frame_type(&m_ctx->api_ctx);
  if (ret < 0) {
    ALOGE("xcoder force IDR frame error: %d", ret);
    if (ret == NI_RETCODE_INVALID_PARAM) {
      return OMX_ErrorBadParameter;
    } else {
      return OMX_ErrorUndefined;
    }
  }
  return OMX_ErrorNone;
}

OMX_ERRORTYPE NiQuadraOMXEncoder::xcoder_reconfig_ltr() {
  ALOGI("enter into %s", __func__);
  ALOGI("xcoder reconfig LTR: [%d, %d]", m_ltr.use_cur_src_as_long_term_pic,
        m_ltr.use_long_term_ref);

  if (!m_is_initialized) {
    ALOGE("xcoder reconfig LTR but xcoder session isn't initialized yet");
    return OMX_ErrorIncorrectStateOperation;
  }

  int ret = ni_set_ltr(&m_ctx->api_ctx, &m_ltr);
  if (ret < 0) {
    ALOGE("xcoder reconfig LTR error: %d", ret);
    if (ret == NI_RETCODE_INVALID_PARAM) {
      return OMX_ErrorBadParameter;
    } else {
      return OMX_ErrorUndefined;
    }
  }
  return OMX_ErrorNone;
}

OMX_ERRORTYPE NiQuadraOMXEncoder::xcoder_reconfig_ltr_interval() {
  ALOGI("enter into %s", __func__);
  ALOGI("xcoder reconfig LTR interval: %d", m_ltr_interval);

  if (!m_is_initialized) {
    ALOGE("xcoder reconfig LTR interval but xcoder session isn't initialized yet");
    return OMX_ErrorIncorrectStateOperation;
  }

  int ret = ni_set_ltr_interval(&m_ctx->api_ctx, m_ltr_interval);
  if (ret < 0) {
    ALOGE("xcoder reconfig LTR interval error: %d", ret);
    if (ret == NI_RETCODE_INVALID_PARAM) {
      return OMX_ErrorBadParameter;
    } else {
      return OMX_ErrorUndefined;
    }
  }
  return OMX_ErrorNone;
}

OMX_ERRORTYPE NiQuadraOMXEncoder::xcoder_set_frame_ref_invalid() {
  ALOGI("enter into %s", __func__);
  ALOGI("xcoder set frame: %d ref invalid", m_ltr_frame_ref_invalid);

  if (!m_is_initialized) {
    ALOGE("xcoder set frame: %d ref invalid but xcoder session isn't initialized yet",
            m_ltr_frame_ref_invalid);
    return OMX_ErrorIncorrectStateOperation;
  }

  int ret = ni_set_frame_ref_invalid(&m_ctx->api_ctx, m_ltr_frame_ref_invalid);
  if (ret < 0) {
    ALOGI("xcoder set frame: %d ref invalid error: %d", m_ltr_frame_ref_invalid, ret);
    if (ret == NI_RETCODE_INVALID_PARAM) {
      return OMX_ErrorBadParameter;
    } else {
      return OMX_ErrorUndefined;
    }
  }
  return OMX_ErrorNone;
}

OMX_ERRORTYPE NiQuadraOMXEncoder::xcoder_reconfig_max_frame_size() {
  ALOGI("enter into %s", __func__);
  ALOGI("xcoder reconfig max frame size: %d", m_max_frame_size);

  if (!m_is_initialized) {
    ALOGE("xcoder reconfig max frame size: %d but xcoder session isn't initialized yet",
            m_max_frame_size);
    return OMX_ErrorIncorrectStateOperation;
  }

  int ret = ni_reconfig_max_frame_size(&m_ctx->api_ctx, m_max_frame_size);
  if (ret < 0) {
    ALOGE("xcoder reconfig max frame size error: %d", ret);
    if (ret == NI_RETCODE_INVALID_PARAM) {
      return OMX_ErrorBadParameter;
    } else {
      return OMX_ErrorUndefined;
    }
  }
  return OMX_ErrorNone;
}

OMX_ERRORTYPE NiQuadraOMXEncoder::xcoder_reconfig_min_max_qp() {
  ALOGI("enter into %s", __func__);
  ALOGI("xcoder reconfig min-max-qp: [%d:%d:%d:%d:%d]", m_min_max_qp.minQpI,
          m_min_max_qp.maxQpI, m_min_max_qp.maxDeltaQp,\
          m_min_max_qp.minQpPB, m_min_max_qp.maxQpPB);

  if (!m_is_initialized) {
    ALOGE("xcoder reconfig min-max-qp but xcoder session isn't initialized yet");
    return OMX_ErrorIncorrectStateOperation;
  }

  int ret = ni_reconfig_min_max_qp(&m_ctx->api_ctx, &m_min_max_qp);
  if (ret < 0) {
    ALOGE("xcoder reconfig min-max-qp error: %d", ret);
    if (ret == NI_RETCODE_INVALID_PARAM) {
      return OMX_ErrorBadParameter;
    } else {
      return OMX_ErrorUndefined;
    }
  }
  return OMX_ErrorNone;
}

OMX_ERRORTYPE NiQuadraOMXEncoder::xcoder_reconfig_crf() {
  ALOGI("enter into %s", __func__);
  ALOGI("xcoder reconfig CRF: %d", m_crf);

  if (!m_is_initialized) {
    ALOGE("xcoder reconfig CRF but xcoder session isn't initialized yet");
    return OMX_ErrorIncorrectStateOperation;
  }

  int ret = ni_reconfig_crf(&m_ctx->api_ctx, m_crf);
  if (ret < 0) {
    ALOGE("xcoder reconfig CRF error: %d", ret);
    if (ret == NI_RETCODE_INVALID_PARAM) {
      return OMX_ErrorBadParameter;
    } else {
      return OMX_ErrorUndefined;
    }
  }
  return OMX_ErrorNone;
}

OMX_ERRORTYPE NiQuadraOMXEncoder::xcoder_reconfig_crf2() {
  ALOGI("enter into %s", __func__);
  ALOGI("xcoder reconfig CRF2: %f", m_crf2);

  if (!m_is_initialized) {
    ALOGE("xcoder reconfig CRF2 but xcoder session isn't initialized yet");
    return OMX_ErrorIncorrectStateOperation;
  }

  int ret = ni_reconfig_crf2(&m_ctx->api_ctx, m_crf2);
  if (ret < 0) {
    ALOGE("xcoder reconfig CRF2 error: %d", ret);
    if (ret == NI_RETCODE_INVALID_PARAM) {
      return OMX_ErrorBadParameter;
    } else {
      return OMX_ErrorUndefined;
    }
  }
  return OMX_ErrorNone;
}

OMX_ERRORTYPE NiQuadraOMXEncoder::xcoder_reconfig_vbv() {
  ALOGI("enter into %s", __func__);
  ALOGI("xcoder reconfig VBV: [%d, %d]", m_vbvMaxRate, m_vbvBufferSize);

  if (!m_is_initialized) {
    ALOGE("xcoder reconfig VBV but xcoder session isn't initialized yet");
    return OMX_ErrorIncorrectStateOperation;
  }

  int ret = ni_reconfig_vbv_value(&m_ctx->api_ctx, m_vbvMaxRate, m_vbvBufferSize);
  if (ret < 0) {
    ALOGE("xcoder reconfig VBV error: %d", ret);
    if (ret == NI_RETCODE_INVALID_PARAM) {
      return OMX_ErrorBadParameter;
    } else {
      return OMX_ErrorUndefined;
    }
  }
  return OMX_ErrorNone;
}

OMX_ERRORTYPE NiQuadraOMXEncoder::xcoder_reconfig_max_frame_size_ratio() {
  ALOGI("enter into %s", __func__);
  ALOGI("xcoder reconfig max frame size ratio: %d", m_max_frame_size_ratio);

  if (!m_is_initialized) {
    ALOGE("xcoder reconfig max frame size ratio but xcoder session isn't initialized yet");
    return OMX_ErrorIncorrectStateOperation;
  }

  int ret = ni_reconfig_max_frame_size_ratio(&m_ctx->api_ctx, m_max_frame_size_ratio);
  if (ret < 0) {
    ALOGE("xcoder reconfig max frame size ratio error: %d", ret);
    if (ret == NI_RETCODE_INVALID_PARAM) {
      return OMX_ErrorBadParameter;
    } else {
      return OMX_ErrorUndefined;
    }
  }

  return OMX_ErrorNone;
}

OMX_ERRORTYPE NiQuadraOMXEncoder::internalGetParameter(OMX_INDEXTYPE index, OMX_PTR params) {
    const int32_t indexFull = index;

    switch (indexFull) {
        case OMX_IndexParamVideoHevc:
        {
            OMX_VIDEO_PARAM_HEVCTYPE* pParam =
                (OMX_VIDEO_PARAM_HEVCTYPE*)params;
            memcpy(pParam, &m_sParamHEVC, sizeof(m_sParamHEVC));
            return OMX_ErrorNone;
        }

        case OMX_IndexParamVideoBitrate:
        {
            OMX_VIDEO_PARAM_BITRATETYPE *bitRate =
                (OMX_VIDEO_PARAM_BITRATETYPE *)params;

            if (!isValidOMXParam(bitRate)) {
                return OMX_ErrorBadParameter;
            }

            if (bitRate->nPortIndex != 1) {
                return OMX_ErrorUndefined;
            }

            bitRate->eControlRate = OMX_Video_ControlRateVariable;
            bitRate->nTargetBitrate = mBitrate;
            return OMX_ErrorNone;
        }

        case OMX_IndexParamVideoAvc:
        {
            OMX_VIDEO_PARAM_AVCTYPE *avcParams = (OMX_VIDEO_PARAM_AVCTYPE *)params;

            if (!isValidOMXParam(avcParams)) {
                return OMX_ErrorBadParameter;
            }

            if (avcParams->nPortIndex != 1) {
                return OMX_ErrorUndefined;
            }

            OMX_VIDEO_AVCLEVELTYPE omxLevel = OMX_VIDEO_AVCLevel41;
            // TODO: maintain profile
            avcParams->eProfile = (OMX_VIDEO_AVCPROFILETYPE)OMX_VIDEO_AVCProfileConstrainedBaseline;
            avcParams->eLevel = omxLevel;
            avcParams->nRefFrames = 1;
            avcParams->bUseHadamard = OMX_TRUE;
            avcParams->nAllowedPictureTypes = (OMX_VIDEO_PictureTypeI
                    | OMX_VIDEO_PictureTypeP | OMX_VIDEO_PictureTypeB);
            avcParams->nRefIdx10ActiveMinus1 = 0;
            avcParams->nRefIdx11ActiveMinus1 = 0;
            avcParams->bWeightedPPrediction = OMX_FALSE;
            avcParams->bconstIpred = OMX_FALSE;
            avcParams->bDirect8x8Inference = OMX_FALSE;
            avcParams->bDirectSpatialTemporal = OMX_FALSE;
            avcParams->nCabacInitIdc = 0;
            return OMX_ErrorNone;
        }

        default:
            return SoftVideoEncoderOMXComponent::internalGetParameter(index, params);
    }
}

OMX_ERRORTYPE NiQuadraOMXEncoder::internalSetParameter(OMX_INDEXTYPE index, const OMX_PTR params) {
    int32_t indexFull = index;

    switch (indexFull) {
        case OMX_IndexParamVideoHevc:
        {
            OMX_VIDEO_PARAM_HEVCTYPE *hevcType =
                (OMX_VIDEO_PARAM_HEVCTYPE*)params;

            if (!isValidOMXParam(hevcType)) {
                return OMX_ErrorBadParameter;
            }

            if (hevcType->nPortIndex != 1) {
                return OMX_ErrorUndefined;
            }

            memcpy(&m_sParamHEVC, hevcType, sizeof(m_sParamHEVC));
            return OMX_ErrorNone;
        }

        case OMX_IndexParamVideoBitrate:
        {
            OMX_VIDEO_PARAM_BITRATETYPE *bitRate =
                (OMX_VIDEO_PARAM_BITRATETYPE *)params;

            if (!isValidOMXParam(bitRate)) {
                return OMX_ErrorBadParameter;
            }

            mBitrate = bitRate->nTargetBitrate;
            return OMX_ErrorNone;
        }

        case OMX_IndexParamVideoAvc:
        {
            OMX_VIDEO_PARAM_AVCTYPE *avcType = (OMX_VIDEO_PARAM_AVCTYPE *)params;

            if (!isValidOMXParam(avcType)) {
                return OMX_ErrorBadParameter;
            }

            if (avcType->nPortIndex != 1) {
                return OMX_ErrorUndefined;
            }

            memcpy(&m_sParamAVC, avcType, sizeof(m_sParamAVC));
            return OMX_ErrorNone;
        }

        default:
            return SoftVideoEncoderOMXComponent::internalSetParameter(index, params);
    }
}

OMX_ERRORTYPE NiQuadraOMXEncoder::getConfig(
        OMX_INDEXTYPE index, OMX_PTR _params) {
    switch ((int)index) {
        case OMX_IndexConfigAndroidIntraRefresh:
        {
            OMX_VIDEO_CONFIG_ANDROID_INTRAREFRESHTYPE *intraRefreshParams =
                (OMX_VIDEO_CONFIG_ANDROID_INTRAREFRESHTYPE *)_params;

            if (!isValidOMXParam(intraRefreshParams)) {
                return OMX_ErrorBadParameter;
            }

            if (intraRefreshParams->nPortIndex != kOutputPortIndex) {
                return OMX_ErrorUndefined;
            }

            intraRefreshParams->nRefreshPeriod = m_intra_period;
            return OMX_ErrorNone;
        }

        case OMX_IndexConfigAndroidVendorExtension:
        {
           OMX_CONFIG_ANDROID_VENDOR_EXTENSIONTYPE *ext =
               reinterpret_cast<OMX_CONFIG_ANDROID_VENDOR_EXTENSIONTYPE *>(_params);

           if (!isValidOMXParam(ext)) {
             return OMX_ErrorBadParameter;
           }
           return get_vendor_extension_config(ext);
        }

        default:
            return SoftVideoEncoderOMXComponent::getConfig(index, _params);
    }
}

OMX_ERRORTYPE NiQuadraOMXEncoder::internalSetConfig(
        OMX_INDEXTYPE index, const OMX_PTR _params, bool *frameConfig) {
    switch ((int)index) {
        case OMX_IndexConfigVideoIntraVOPRefresh:
        {
            OMX_CONFIG_INTRAREFRESHVOPTYPE *params =
                (OMX_CONFIG_INTRAREFRESHVOPTYPE *)_params;

            if (!isValidOMXParam(params)) {
                return OMX_ErrorBadParameter;
            }

            if (params->nPortIndex != kOutputPortIndex) {
                return OMX_ErrorBadPortIndex;
            }

            if (params->IntraRefreshVOP) {
                int ret = 0;
                ret = xcoder_force_idr_frame();
                if (ret < 0) {
                  if (ret == NI_RETCODE_INVALID_PARAM) {
                    return OMX_ErrorBadParameter;
                  } else {
                    return OMX_ErrorUndefined;
                  }
                }
            }

            return OMX_ErrorNone;
        }

        case OMX_IndexConfigVideoBitrate:
        {
            int ret = 0;
            OMX_VIDEO_CONFIG_BITRATETYPE *params =
                (OMX_VIDEO_CONFIG_BITRATETYPE *)_params;

            if (!isValidOMXParam(params)) {
                return OMX_ErrorBadParameter;
            }

            if (params->nPortIndex != kOutputPortIndex) {
                return OMX_ErrorBadPortIndex;
            }

            mBitrate = params->nEncodeBitrate;
            ret = xcoder_reconfig_bitrate();
            if (ret < 0) {
              if (ret == NI_RETCODE_INVALID_PARAM) {
                return OMX_ErrorBadParameter;
              } else {
                return OMX_ErrorUndefined;
              }
            }

            return OMX_ErrorNone;
        }

        case OMX_IndexConfigAndroidIntraRefresh:
        {
            int ret = 0;
            const OMX_VIDEO_CONFIG_ANDROID_INTRAREFRESHTYPE *intraRefreshParams =
                (const OMX_VIDEO_CONFIG_ANDROID_INTRAREFRESHTYPE *)_params;

            if (!isValidOMXParam(intraRefreshParams)) {
                return OMX_ErrorBadParameter;
            }

            if (intraRefreshParams->nPortIndex != kOutputPortIndex) {
                return OMX_ErrorUndefined;
            }

            m_intra_period = intraRefreshParams->nRefreshPeriod;
            ret = xcoder_reconfig_intraprd();
            if (ret < 0) {
              if (ret == NI_RETCODE_INVALID_PARAM) {
                return OMX_ErrorBadParameter;
              } else {
                return OMX_ErrorUndefined;
              }
            }

            return OMX_ErrorNone;
        }

        case OMX_IndexConfigAndroidVendorExtension:
        {
            OMX_CONFIG_ANDROID_VENDOR_EXTENSIONTYPE *ext =
                reinterpret_cast<OMX_CONFIG_ANDROID_VENDOR_EXTENSIONTYPE *>(_params);

            if (!isValidOMXParam(ext)) {
                return OMX_ErrorBadParameter;
            }

            return set_vendor_extension_config(ext);
        }

        default:
            return SimpleSoftOMXComponent::internalSetConfig(index, _params, frameConfig);
    }
}

void NiQuadraOMXEncoder::onQueueFilled(OMX_U32 portIndex) {
  UNUSED(portIndex);

  ALOGV("enter into onQueueFilled()");

  // Initialize encoder
  if (! m_ctx) {
    ni_log_level_t log_level = NI_LOG_INVALID;
    char property[PROPERTY_VALUE_MAX] = {0};
    if (property_get(PROP_XOCDER_LOG_LEVEL, property, NULL)) {
      log_level = convert_ni_log_level(property);
      if (log_level != NI_LOG_INVALID) {
        ALOGI("set xcoder log level: %s", property);
        ni_log_set_level(log_level);
      }
    }

    m_ctx = (NiOMXEncCtx *)calloc(1, sizeof(NiOMXEncCtx));
    if (!m_ctx) {
      ALOGE("Failed to initialize encoder");
      notify(OMX_EventError, OMX_ErrorUndefined, 0, NULL);
      return;
    }

    //Initialize encoder session
    if (xcoder_encode_init() < 0) {
      ALOGE("Failed to initialize encoder");
      notify(OMX_EventError, OMX_ErrorUndefined, 0, NULL);
      return;
	}

	ALOGI("succeed to initialize encoder");
  }

  if (m_signalled_error) {
    return;
  }

  List<BufferInfo *> &inQueue = getPortQueue(0);
  List<BufferInfo *> &outQueue = getPortQueue(1);

  while(!m_output_eos && !outQueue.empty()) {
    //OMX_ERRORTYPE error;
    BufferInfo *inputBufferInfo;
    OMX_BUFFERHEADERTYPE *inHeader;

    BufferInfo *outputBufferInfo = *outQueue.begin();
    OMX_BUFFERHEADERTYPE *outHeader = outputBufferInfo->mHeader;

    if (m_input_eos) {
      inHeader = NULL;
      inputBufferInfo = NULL;
    } else if (!inQueue.empty()) {
      inputBufferInfo = *inQueue.begin();
      inHeader = inputBufferInfo->mHeader;
    } else {
      return;
    }

    if (inHeader != NULL) {
      outHeader->nFlags = inHeader->nFlags;
    }

    if ((inHeader != NULL)
        && (inHeader->nFlags & OMX_BUFFERFLAG_EOS)) {
      ALOGI("Recv OMX_BUFFERFLAG_EOS, input is eos, input len:%d\n", inHeader->nFilledLen);
      m_input_eos = true;
    }

    outHeader->nTimeStamp = 0;
    outHeader->nFlags = 0;
    outHeader->nFilledLen = 0;
    outHeader->nOffset = 0;

    OMX_BUFFERHEADERTYPE *tmpInHeader = NULL;
    if (!m_input_eos) {
      tmpInHeader = inHeader;
    }

    bool send_frame = false;
    bool get_pkt = false;
    if (xcoder_send_receive(tmpInHeader, &send_frame,
        outHeader, &get_pkt) < 0) {
      ALOGI("Error : xcoder_send_receive failed\n");
      notify(OMX_EventError, OMX_ErrorUnsupportedSetting, 0, NULL);
      m_signalled_error = true;
      return;
    }

    ALOGV("xcoder_send_receive success\n");

    //notify input
    if (send_frame && inHeader != NULL) {
      ALOGV("xcoder_send_receive send success, notifyEmptyBufferDone\n");
      inputBufferInfo->mOwnedByUs = false;

      inQueue.erase(inQueue.begin());
      notifyEmptyBufferDone(inHeader);
    }

    //notify output
    if (get_pkt && (outHeader->nFilledLen >= 0)) {
      ALOGV("xcoder_send_receive recv success, notifyFillBufferDone\n");
      outputBufferInfo->mOwnedByUs = false;

      outQueue.erase(outQueue.begin());
      notifyFillBufferDone(outHeader);
    }
  }

  return;
}

void NiQuadraOMXEncoder::onReset() {
  ALOGI("enter into onReset()");

  SoftVideoEncoderOMXComponent::onReset();

  if (xcoder_encode_close() < 0) {
    ALOGW("releaseEncoder failed");
  }
}

} // namespace android

