#include <stdio.h>
#include <string.h>

extern "C" {
#include <libavformat/avformat.h>
#include <libavcodec/avcodec.h>
}

#include <media/NdkMediaCodec.h>
#include <media/NdkMediaExtractor.h>
#include <media/NdkMediaFormat.h>

#define LOG_TAG "netint_decoder_test"

int main(int argc, char **argv) {
  if (argc != 3) {
    fprintf(stderr, "Usage: %s <input_file> <output_file>\n", argv[0]);
    fprintf(stderr, "Note: The file suffix must be h264/h265 for <input_file>\n");
    fprintf(stderr, "Note: The file suffix must be yuv for <output_file>\n");
    return -1;
  }

  AVFormatContext *fmt_ctx = NULL;
  AVStream *video_stream = NULL;
  const char *in_file_name = NULL;
  const char *out_file_name = NULL;
  AVPacket pkt;
  AMediaCodec* codec = NULL;
  AMediaFormat* format = NULL;
  int video_stream_index = -1;
  const char *format_str = NULL;
  FILE *outputFp = NULL;
  media_status_t status;

  int ret;
  int pkt_num = 0;
  int frame_num = 0;
  bool input_eos = false;
  bool output_eos = false;
  bool flush = false;

  in_file_name = argv[1];
  out_file_name = argv[2];

  outputFp = fopen(out_file_name, "w+");
  if (!outputFp) {
    printf("open output file failed!!!\n");
    return -1;
  }

  if (avformat_open_input(&fmt_ctx, in_file_name, NULL, NULL) < 0) {
    fprintf(stderr, "Could not open source file %s\n", in_file_name);
    goto END;
  }

  if (avformat_find_stream_info(fmt_ctx, NULL) < 0) {
    fprintf(stderr, "Could not find stream information\n");
    goto END;
  }

  for (int i = 0; i < fmt_ctx->nb_streams; i++) {
    if (fmt_ctx->streams[i]->codecpar->codec_type == AVMEDIA_TYPE_VIDEO) {
      video_stream_index = i;
      break;
    }
  }
  video_stream = fmt_ctx->streams[video_stream_index];

  //create MediaCodec's codec
  switch (video_stream->codecpar->codec_id) {
    case AV_CODEC_ID_H264:
      codec = AMediaCodec_createDecoderByType("video/avc");
      printf("video codec_id: AV_CODEC_ID_H264\n");
      break;
    case AV_CODEC_ID_HEVC:
      codec = AMediaCodec_createDecoderByType("video/hevc");
      printf("video codec_id: AV_CODEC_ID_HEVC\n");
      break;
    default:
      fprintf(stderr, "Netint decoder does not support this codec.");
      goto END;
  }

  //configure MediaCodec's codec
  format = AMediaFormat_new();

  if (AV_CODEC_ID_H264 == video_stream->codecpar->codec_id) {
    AMediaFormat_setString(format, "mime", "video/avc");
  } else {
    AMediaFormat_setString(format, "mime", "video/hevc");
  }

  printf("video info, width:%d, height:%d\n", video_stream->codecpar->width, video_stream->codecpar->height);

  AMediaFormat_setInt32(format, "width", video_stream->codecpar->width);
  AMediaFormat_setInt32(format, "height", video_stream->codecpar->height);

  switch (video_stream->codecpar->format) {
  case AV_PIX_FMT_YUV420P:
    AMediaFormat_setInt32(format, "color-format", 19);
    break;
  default:
    fprintf(stderr, "Netint decoder does not support this format.");
    goto END;
  }

  AMediaFormat_setInt32(format, "bitrate", 3000000);
  AMediaFormat_setInt32(format, "frame-rate", 25);
  AMediaFormat_setInt32(format, "bitrate-mode",1);

  format_str = AMediaFormat_toString(format);
  fprintf(stderr, "Input format: %s\n", format_str);

  status = AMediaCodec_configure(codec, format, NULL, NULL, 0);
  if (status != AMEDIA_OK) {
    fprintf(stderr, "It's not lucky, AMediaCodec_configure failed %d\n", status);
    goto END;
  } else {
    fprintf(stderr, "It's lucky, AMediaCodec_configure success\n");
  }

  //start decode for MediaCodec's codec
  AMediaCodec_start(codec);

  av_init_packet(&pkt);
  pkt.data = NULL;
  pkt.size = 0;

  while (1) {
    //read pkt
    //pkt.size > 0 means the pkt isn't sent
    if (pkt.size == 0) {
      if (!input_eos) {
        ret = av_read_frame(fmt_ctx, &pkt);
        if (ret < 0 && ret != AVERROR_EOF) {
          fprintf(stderr, "It's not lucky, some error occur : %d\n", ret);
          break;
        } else if (ret == AVERROR_EOF) {
          input_eos = true;
          printf("input eos!!!\n");
        }

        if (pkt.stream_index != video_stream_index) {
          av_packet_unref(&pkt);
          continue;
        }
      }
    } else {
      printf("the pkt isn't sent, continue to send it!!!\n");
    }

    {
      //get input buff index, and fill buffer
      ssize_t inIdx = AMediaCodec_dequeueInputBuffer(codec, 1000);
      if (inIdx >= 0) {
        if (input_eos) {
          printf("send input eos to AMediaCodec!!!\n");
          AMediaCodec_queueInputBuffer(codec, inIdx, 0, 0, 0, AMEDIACODEC_BUFFER_FLAG_END_OF_STREAM);
          flush = true;
        } else {
          size_t bufSize;
          uint8_t* buf = AMediaCodec_getInputBuffer(codec, inIdx, &bufSize);
          if (buf && pkt.size <= bufSize) {
            memcpy(buf, pkt.data, pkt.size);
            AMediaCodec_queueInputBuffer(codec, inIdx, 0, pkt.size, 0, 0);
  
            pkt_num++;
            printf("Send %dth encoded pkt to MediaCodec's codec!!!\n", pkt_num);
          }

          av_packet_unref(&pkt);
        }
      }
    }

    {
      //get output from decoder
      AMediaCodecBufferInfo info;
      ssize_t outIdx = AMediaCodec_dequeueOutputBuffer(codec, &info, 1000);
      if (outIdx >= 0) {
        if (info.flags & AMEDIACODEC_BUFFER_FLAG_END_OF_STREAM) {
          printf("output is eos!!!\n");
          output_eos = true;
        }
  
        if (info.size > 0) {
          size_t bufSize;
          uint8_t *buf = AMediaCodec_getOutputBuffer(codec, outIdx, &bufSize);
          if (buf) {
            fwrite(buf, 1, info.size, outputFp);
          }
  
          frame_num++;
          printf("Received %dth encoded pkt from MediaCodec's codec!!!\n", frame_num);
        }
  
        AMediaCodec_releaseOutputBuffer(codec, outIdx, true);
      }
    }

    if (flush || output_eos) {
      break;
    }
  }

  while (flush && !output_eos) {
    printf("Recv input eos, and try to recv decoder frames left!!!\n");

    //get output from decoder
    AMediaCodecBufferInfo info;
    ssize_t outIdx = AMediaCodec_dequeueOutputBuffer(codec, &info, 1000);
    if (outIdx >= 0) {
      if (info.flags & AMEDIACODEC_BUFFER_FLAG_END_OF_STREAM) {
        printf("output is eos!!!\n");
        output_eos = true;
      }

      if (info.size > 0) {
        size_t bufSize;
        uint8_t *buf = AMediaCodec_getOutputBuffer(codec, outIdx, &bufSize);
        if (buf) {
          fwrite(buf, 1, info.size, outputFp);
        }

        frame_num++;
        printf("Received %dth encoded pkt from MediaCodec's codec!!!\n", frame_num);
      }

      AMediaCodec_releaseOutputBuffer(codec, outIdx, true);
    }
  }

END:

  if (outputFp) {
    fclose(outputFp);
    outputFp = nullptr;
  }

  if (fmt_ctx) {
    avformat_close_input(&fmt_ctx);
    fmt_ctx = nullptr;
  }

  if (format) {
    AMediaFormat_delete(format);
    format = nullptr;
  }

  if (codec) {
    AMediaCodec_stop(codec);
    AMediaCodec_delete(codec);
    codec = nullptr;
  }

  printf("total send pkt:%d, recv frame:%d !!!\n", pkt_num, frame_num);

  return 0;
}

