#!/usr/bin/env bash

ffmpeg_path="";
android_arch=x86_64;
android_api=28;
debug=false;

# parse a flag with an arg in or after it
# $1 flag pattern, $2 entire flag arg, $3 arg after flag arg
# return 1 if path is in second arg (separated by space), else return 0. Store path in $extract_arg_ret
extract_arg () {
    unset extract_arg_ret
    # check valid arg flag
    if [ -n "$(printf "%s" ${2} | grep -Eoh "${1}")" ]; then
        # check if path string is connected by '=' or is in following arg
        if [ -n "$(echo "${2}" | grep -Eoh "${1}=")" ]; then
            arg_str=`printf "%s" "${2}" | grep -Poh "${1}=\K.+"`;
            # trim out leading and trailing quotation marks
            extract_arg_ret=`echo "${arg_str}" | sed -e 's/^\(["'\'']\)//' -e 's/\(["'\'']\)$//'`;
            return 0;
        elif [ -n "$(printf "%s" ${2} | grep -Eoh "^${1}$")" ]; then
            arg_str="${3}";
            # trim out leading and trailing quotation marks
            extract_arg_ret=`printf "%s" "${arg_str}" | sed -e 's/^\(["'\'']\)//' -e 's/\(["'\'']\)$//'`;
            return 1;
        else
            echo "Unknown option '$2', exiting";
            exit 1;
        fi
    else
        echo "Target flag '$1' not found in '$2', exiting"; exit 1;
    fi
}

while [ "$1" != "" ]; do
    case $1 in
        -h | --help) echo "Usage: ./build.sh --ffmpeg_path [OPTION]";
                     echo "build MediaCodec demo";
                     echo "Example: ./build.sh --ffmpeg_path=xxx";
                     echo;
                     echo "Options:";
                     echo "-h, --help                  display this help and exit.";
                     echo "-g, --gdb                   compile for GDB.";
                     echo "--ffmpeg_path \"<path>\"      the path where the ffmpeg build, demo is depent on ffmpeg build";
                     echo "--android_arch \"<arch>\"     cross compile CPU arch when compiling for --android. [arm,arm64,x86,x86_64(default)]";
                     echo "--android_api \"<api>\"       android api number when compiling for --android. [28(default), 29, 30, and so on]";
                     exit 0
        ;;
        --ffmpeg_path | --ffmpeg_path=*)    extract_arg "\-\-ffmpeg_path" "$1" "$2"; prev_rc=$?;
                                            if [ "$prev_rc" -eq 1 ]; then shift; fi
                                            ffmpeg_path=$extract_arg_ret
        ;;
        --android_arch | --android_arch=*)  extract_arg "\-\-android_arch" "$1" "$2"; prev_rc=$?;
                                            if [ "$prev_rc" -eq 1 ]; then shift; fi
                                            android_arch=$extract_arg_ret
        ;;
        --android_api | --android_api=*)    extract_arg "\-\-android_api" "$1" "$2"; prev_rc=$?;
                                            if [ "$prev_rc" -eq 1 ]; then shift; fi
                                            android_api=$extract_arg_ret
        ;;
        -g | --gdb)      debug=true
        ;;
        *)               echo "Usage: ./build.sh --ffmpeg_path [OPTION]...";
                         echo "Try './build.sh --help' for more information"; exit 1
        ;;
    esac
    shift
done

if [ -z ${ANDROID_NDK_ROOT} ]; then
  echo "You must set ANDROID_NDK_ROOT environment variable"
  echo "Please download NDK r20b from https://developer.android.com/ndk/downloads/older_releases"
  exit -1
fi

if [ "${ffmpeg_path}" = "" ]; then
  echo "You must enter the path where the ffmpeg build"
  echo "Example: ./build.sh --ffmpeg_path=~/FFmpegXcoder/FFmpeg-n4.3.1"
  exit -1
fi

if [ "${android_arch}" = "arm" ]; then
    ARCH=armeabi-v7a
elif [ "${android_arch}" = "arm64" ]; then
    ARCH=arm64-v8a
elif [ "${android_arch}" = "x86" ]; then
    ARCH=x86
elif [ "${android_arch}" = "x86_64" ]; then
    ARCH=x86_64
elif [ "${android_arch}" = "" ]; then
    ARCH=x86_64
else
    echo "Error - unknown option for --android_arch. Select from: [arm,arm64,x86,x86_64]"
    exit -1
fi

enable_debug=off
if $debug; then
  enable_debug=on
fi

cmake_toolchain=${ANDROID_NDK_ROOT}/build/cmake/android.toolchain.cmake
android_platform=android-${android_api}

include_path=${ANDROID_NDK_ROOT}/sysroot/usr/include/
lib_path=${ANDROID_NDK_ROOT}/platforms/${android_platform}/arch-${ARCH}/usr/lib64/

ff_include_path=${ffmpeg_path}/android/${android_arch}/include
ff_lib_path=${ffmpeg_path}/android/${android_arch}/lib

cd ${ANDROID_NDK_ROOT}
ndk_version=$(grep "Pkg.Revision" source.properties | cut -d' ' -f3 | cut -d'.' -f1)
echo "ndk_version is :${ndk_version}"
cd -

rm build -rf
mkdir build

cd build

echo "cmake -DCMAKE_TOOLCHAIN_FILE=${cmake_toolchain} -DANDROID_ABI=${ARCH} -DANDROID_NDK=${ANDROID_NDK_ROOT} -DANDROID_PLATFORM=${android_platform} -DINCLUDE_PATH=${include_path} -DLIB_PATH=${lib_path} -DFF_INCLUDE_PATH=${ff_include_path} -DFF_LIB_PATH=${ff_lib_path} -DENABLE_DEBUG_BUILD=${enable_debug} .."

cmake -DCMAKE_TOOLCHAIN_FILE=${cmake_toolchain} -DANDROID_ABI=${ARCH} -DANDROID_NDK=${ANDROID_NDK_ROOT} -DANDROID_PLATFORM=${android_platform} -DINCLUDE_PATH=${include_path} -DLIB_PATH=${lib_path} -DFF_INCLUDE_PATH=${ff_include_path} -DFF_LIB_PATH=${ff_lib_path} -DENABLE_DEBUG_BUILD=${enable_debug} -DNDK_VERSION=${ndk_version} ..

if [ $? != 0 ]; then
  echo -e "\e[31mcmake failed. Exiting...\e[0m"
  exit 1
else
  make
  RC=$?
fi

exit $RC

