#include <stdio.h>
#include <string.h>

extern "C" {
#include <libavformat/avformat.h>
#include <libavcodec/avcodec.h>
}

#include <media/NdkMediaCodec.h>
#include <media/NdkMediaExtractor.h>
#include <media/NdkMediaFormat.h>

/* Send one packet to decoder, attempt to read all available decoded frames from decoder,
   saving YUV to file. If run with with *pkt=NULL, flush the decoder of remaining frames
   until decoder reports EOF */

static bool is_send_eos = false;

static int decode(AVCodecContext *dec_ctx, AVFrame *frame, AVPacket *pkt, int *get_frame)
{
  int ret = 0;

  *get_frame = 0;

  // Send a frame packet to decoder
  if (!is_send_eos) {
    ret = avcodec_send_packet(dec_ctx, pkt);
    if (ret < 0) {
      if (ret == AVERROR_EOF) {
        is_send_eos = true;
        ret = 0;
      } else if (ret != AVERROR(EAGAIN)) {
        fprintf(stderr, "Error(%d) sending a packet for decoding\n", ret);
        return ret;
      }
    }
  }

  /* Run this loop if a packet was sucessfully sent during avcodec_send_packet() OR
	 no packet was sent because this function is in flushing mode OR
	 a frame was sucessfully received by avcodec_receive_frame() */
  while (ret >= 0 || ! pkt) {
    // Attempt to retreive a decoded frame
    ret = avcodec_receive_frame(dec_ctx, frame);
    // Failed to get a frame because decoder output buffer is empty
    if (ret == AVERROR(EAGAIN))
    {
      // If function not in flushing mode
      if (pkt)
        return ret; // Exit this function to send next packet
      else
        continue; // Loop in this function to flush decoder
    }
    // Decoder reports EOF, flushing is complete
    else if (ret == AVERROR_EOF)
    {
      fprintf(stderr, "decoder EOF reached\n");
      return ret;
    }
    // Decoding error occured, possibly bad packet in input
    else if (ret < 0)
    {
      fprintf(stderr, "Error during decoding. RC=%d\n", ret);
      exit(1);
    }

    *get_frame = 1;
    break;
  }

  return 0;
}

int enc_param_get_key_value(char *p_str, char *key, char *value)
{
    if (!p_str || !key || !value)
    {
        return 1;
    }

    char *p = strchr(p_str, '=');
    if (!p)
    {
        return 1;
    } else
    {
        *p = '\0';
        key[0] = '\0';
        value[0] = '\0';
        strcpy(key, p_str);
        strcpy(value, p + 1);
        return 0;
    }
}

int main(int argc, char **argv) {
  if (argc < 3) {
    fprintf(stderr, "Usage: %s <input_file> <output_file> <xcoder-params=\"xxx\">\n", argv[0]);
    fprintf(stderr, "Note: The file suffix must be H264/h265 for <output_file>\n");
    fprintf(stderr, "      <xcoder-params=\"xxx\"> are optional, not necessary\n");
    fprintf(stderr, "eg: %s /data/media/Dinner_1920x1080p30_300.h264 /data/media/test.h265 xcoder-params=\"gopPresetIdx=9:intraPeriod=120:RcEnable=1\"\n", argv[0]);
    exit(0);
  }

  const char *in_file_name = NULL;
  const char *out_file_name = NULL;
  char enc_xcoder_params[2048] = {0};
  FILE *outputFp = NULL;
  AVFormatContext *fmt_ctx = NULL;
  AVStream *video_stream = NULL;
  AVCodecContext *codec_ctx= NULL;
  const AVCodec *dec_codec = NULL;
  AVFrame *frame = NULL;
  AVPacket pkt;
  AMediaCodec *codec = NULL;
  AMediaFormat* format = NULL;
  int video_stream_index = -1;
  const char *format_str = NULL;
  media_status_t status;

  int ret = 0;
  int frame_num = 0;
  int pkt_num = 0;
  bool is_need_read = true;
  bool read_eos = false;
  bool input_eos = false;
  bool output_eos = false;
  bool flush = false;

  in_file_name = argv[1];
  out_file_name = argv[2];

  bool out_as_h264_encode = false;
  bool out_as_h265_encode = false;
  if (out_file_name) {
    if (strlen(out_file_name) <= 3) {
      fprintf(stderr, "The output file's name is invalid\n");
      exit(1);
    }

    const char *out_file_suffix = out_file_name + strlen(out_file_name) - 3;
    if (!strcmp(out_file_suffix, "264")) {
      out_as_h264_encode = true;
    } else if (!strcmp(out_file_suffix, "265")) {
      out_as_h265_encode = true;
    } else {
      fprintf(stderr, "The output file's suffix must be H264/H265\n");
      exit(1);
    }
  }

  if (argc >= 4) {
    char *xcoder_params = argv[3];
    int prefix_len = strlen("xcoder-params=");
    memcpy(enc_xcoder_params, xcoder_params + prefix_len, strlen(argv[3]) - prefix_len);
  }

  // Allocate frame buffer
  frame = av_frame_alloc();
  if (!frame)
  {
    fprintf(stderr, "Could not allocate video frame\n");
    exit(1);
  }
  
  outputFp = fopen(out_file_name, "w");
  if (!outputFp) {
    fprintf(stderr, "Could not open %s\n", out_file_name);
    goto END;
  }

  if (avformat_open_input(&fmt_ctx, in_file_name, NULL, NULL) < 0) {
    fprintf(stderr, "Could not open source file %s\n", in_file_name);
    goto END;
  }

  if (avformat_find_stream_info(fmt_ctx, NULL) < 0) {
    fprintf(stderr, "Could not find stream information\n");
    goto END;
  }

  for (int i = 0; i < fmt_ctx->nb_streams; i++) {
    if (fmt_ctx->streams[i]->codecpar->codec_type == AVMEDIA_TYPE_VIDEO) {
      video_stream_index = i;
      break;
    }
  }
  video_stream = fmt_ctx->streams[video_stream_index];

  if (out_as_h264_encode) {
    codec = AMediaCodec_createEncoderByType("video/avc");
    printf("choose H264 encoder for output file\n");
  } else if (out_as_h265_encode) {
    codec = AMediaCodec_createEncoderByType("video/hevc");
    printf("choose H265 encoder for output file\n");
  }

  dec_codec = avcodec_find_decoder(video_stream->codecpar->codec_id);
  if (!dec_codec) {
    fprintf(stderr, "Dec Codec not found\n");
    goto END;
  }

  codec_ctx = avcodec_alloc_context3(dec_codec);
  if (!codec_ctx) {
    fprintf(stderr, "Failed to allocate the codec context\n");
    goto END;
  }

  // Copy codec parameters from input stream to output codec context
  if (avcodec_parameters_to_context(codec_ctx, video_stream->codecpar) < 0) {
    fprintf(stderr, "Failed to copy codec parameters to decoder context\n");
    goto END;
  }

  if (avcodec_open2(codec_ctx, dec_codec, NULL) < 0) {
    fprintf(stderr, "Failed to open codec\n");
    goto END;
  }

  format = AMediaFormat_new();

  if (out_as_h264_encode) {
    AMediaFormat_setString(format, "mime", "video/avc");
  } else if (out_as_h265_encode) {
    AMediaFormat_setString(format, "mime", "video/hevc");
  }

  printf("video info, width:%d, height:%d\n", video_stream->codecpar->width, video_stream->codecpar->height);

  AMediaFormat_setInt32(format, "width", video_stream->codecpar->width);
  AMediaFormat_setInt32(format, "height", video_stream->codecpar->height);

  switch (video_stream->codecpar->format) {
  case AV_PIX_FMT_YUV420P:
    AMediaFormat_setInt32(format, "color-format", 19);
    fprintf(stderr, "It's AV_PIX_FMT_YUV420P.\n");
    break;
  default:
    fprintf(stderr, "Netint encoder does not support this format.\n");
    goto END;
  }

  //standard encode parameters in Android MediaCodec
  //the following 3 parameters are required by MediaCodec
  AMediaFormat_setInt32(format, "bitrate", 200000);
  AMediaFormat_setInt32(format, "frame-rate", 25);
  AMediaFormat_setInt32(format, "i-frame-interval", 20);

  /* Quadra's customized encode parameters
   * The parameter setting method:
   *   1.If a parameter to be set has only one field (ending in "value"), such as "quadra-ext-enc-gopPresetIdx",
   *     set the parameter as follows:
   *       AMediaFormat_setString(format, "vendor.quadra-ext-enc-gopPresetIdx.value", "9");
   *
   *   2.If a parameter to be set contains multiple fields, such as "quadra-ext-enc-reconfig-vbv",
   *     set the parameter as follows:
   *       AMediaFormat_setInt32(format, "vendor.quadra-ext-enc-reconfig-vbv.vbvMaxRate.value",  2000000);
   *       AMediaFormat_setInt32(format, "vendor.quadra-ext-enc-reconfig-vbv.vbvBufferSize.value",  2000);
   *
   * NOTE: For "AMediaFormat_setInt32", "AMediaFormat_setInt64" and "AMediaFormat_setString", the three api is need to be
   *       selected according to the specific filed type.
   */
  //AMediaFormat_setString(format, "vendor.quadra-ext-enc-gopPresetIdx.value", "9");
  //AMediaFormat_setString(format, "vendor.quadra-ext-enc-RcEnable.value", "1");
  //AMediaFormat_setString(format, "vendor.quadra-ext-enc-lowDelay.value", "1");
  AMediaFormat_setString(format, "vendor.quadra-ext-enc-bitrate.value", "200000");

  //set Quadra's customized encode parameters
  {
    char key[64], value[64];
    char *curr = enc_xcoder_params, *colon_pos;
    int ret = 0;

    while (*curr)
    {
      colon_pos = strchr(curr, ':');
      if (colon_pos)
      {
        *colon_pos = '\0';
      }

      if (strlen(curr) > sizeof(key) + sizeof(value) - 1 ||
          enc_param_get_key_value(curr, key, value))
      {
        fprintf(stderr,
              "Error: xcoder-params p_config key/value not "
              "retrieved: %s\n",
              curr);
        ret = -1;
        break;
      }

      char enc_param_key[128]= {0};
      sprintf(enc_param_key, "vendor.quadra-ext-enc-%s.value", key);
      AMediaFormat_setString(format, enc_param_key, value);

      if (colon_pos)
      {
        curr = colon_pos + 1;
      }
      else
      {
        curr += strlen(curr);
      }
    }
  }

  format_str = AMediaFormat_toString(format);
  fprintf(stderr, "Input format: %s\n", format_str);

  status = AMediaCodec_configure(codec, format, NULL, NULL, 1);
  if (status != AMEDIA_OK) {
    fprintf(stderr, "It's not lucky, AMediaCodec_configure failed %d\n", status);
    goto END;
  } else {
    fprintf(stderr, "It's lucky, AMediaCodec_configure success\n");
  }

  AMediaCodec_start(codec);

  av_init_packet(&pkt);
  pkt.data = NULL;
  pkt.size = 0;
  while (1) {
    //read video pkt and decode
    while (is_need_read && !input_eos) {
      if (!read_eos) {
        ret = av_read_frame(fmt_ctx, &pkt);
        if (ret < 0 && ret != AVERROR_EOF) {
          fprintf(stderr, "It's not lucky, some error occur : %d\n", ret);
          goto END;
        } else if (ret == AVERROR_EOF) {
          fprintf(stderr, "It's read_eos\n");
          read_eos = true;
        }
      }

      int get_frame = 0;
      if ((pkt.stream_index != video_stream_index) && !read_eos) {
        av_packet_unref(&pkt);
        continue;
      }

      if (!read_eos) {
        ret = decode(codec_ctx, frame, &pkt, &get_frame);
        av_packet_unref(&pkt);
      } else {
        ret = decode(codec_ctx, frame, NULL, &get_frame);
      }

      if (ret == AVERROR_EOF) {
        input_eos = true;
      }

      if (get_frame) {
        is_need_read = false;
        break;
      }
    }

    //send frame to MediaCodec's codec
    if (!flush) {
	  //get input buff index, and fill buffer
	  ssize_t inIdx = AMediaCodec_dequeueInputBuffer(codec, 1000);
	  if (inIdx >= 0) {
		if (input_eos) {
          printf("input eos!!!\n");
          AMediaCodec_queueInputBuffer(codec, inIdx, 0, 0, 0, AMEDIACODEC_BUFFER_FLAG_END_OF_STREAM);
          flush = true;
		} else {
		  size_t bufSize;
		  uint8_t *buf = AMediaCodec_getInputBuffer(codec, inIdx, &bufSize);
		  if (buf) {
            int i;
            uint8_t *tmp_dst = buf;
            uint8_t *tmp_src = frame->data[0];
            for (i = 0; i < frame->height; i++) {
              memcpy(tmp_dst, tmp_src, frame->width);
              tmp_src += frame->linesize[0];
              tmp_dst += frame->width;
            }

            tmp_src = frame->data[1];
            for (i = 0; i < frame->height / 2; i++) {
              memcpy(tmp_dst, tmp_src, frame->width / 2);
              tmp_src += frame->linesize[1];
              tmp_dst += frame->width / 2;
            }

            tmp_src = frame->data[2];
            for (i = 0; i < frame->height / 2; i++) {
              memcpy(tmp_dst, tmp_src, frame->width / 2);
              tmp_src += frame->linesize[2];
              tmp_dst += frame->width / 2;
            }

            uint64_t presentationTimeUs;
            presentationTimeUs = frame_num * (1000000 / 25);
            AMediaCodec_queueInputBuffer(codec, inIdx, 0, (tmp_dst - buf), presentationTimeUs, 0);

            //the current frame is sent successfully, so read the next frame
            is_need_read = true;
            frame_num++;
            printf("Send %dth decoded frame to MediaCodec's codec!!!\n", frame_num);
          }
        }
      }

      //get output from encoder
      AMediaCodecBufferInfo info;
      ssize_t outIdx = AMediaCodec_dequeueOutputBuffer(codec, &info, 1000);
      if (outIdx >= 0) {
        if (info.flags & AMEDIACODEC_BUFFER_FLAG_END_OF_STREAM) {
          printf("output is eos!!!\n");
          output_eos = true;
        }

        if (info.size > 0) {
          if (info.flags & AMEDIACODEC_BUFFER_FLAG_CODEC_CONFIG) {
            printf("Received CODEC_CONFIG!!!\n");
          }

          pkt_num++;
          printf("Received %dth encoded pkt from MediaCodec's codec!!!\n", pkt_num);

          size_t bufSize;
          uint8_t *buf = AMediaCodec_getOutputBuffer(codec, outIdx, &bufSize);
          if (buf) {
            fwrite(buf, 1, info.size, outputFp);
          }
        }

        AMediaCodec_releaseOutputBuffer(codec, outIdx, true);
      }
    }

/*
    //If encode parameters are set dynamically during encoding, "AMediaCodec_setParameters" must be called
    //reconfigure vbv
    if (frame_num == reconfigure_vbv_frame_num) {
      printf("reconfigure vbv!!!\n");
      AMediaFormat_delete(format);
      format = AMediaFormat_new();
      //use Quadra's customized encode parameters to reconfigure
      AMediaFormat_setInt32(format, "vendor.quadra-ext-enc-reconfig-vbv.vbvMaxRate.value",  vbvMaxRate);
      AMediaFormat_setInt32(format, "vendor.quadra-ext-enc-reconfig-vbv.vbvBufferSize.value",  vbvBufferSize);
      AMediaCodec_setParameters(codec, format);
    }
*/

    if (flush || output_eos) {
      printf("It's in flush state!!!\n");
      break;
    }
  }

  printf("Recv input eos, and try to recv encoded pkts left!!!\n");

  //get output from encoder
  while (flush && !output_eos) {
    AMediaCodecBufferInfo info;
    ssize_t outIdx = AMediaCodec_dequeueOutputBuffer(codec, &info, 1000);
    if (outIdx >= 0) {
      if (info.flags & AMEDIACODEC_BUFFER_FLAG_END_OF_STREAM) {
        printf("output is eos!!!\n");
        output_eos = true;
      }

      if (info.size > 0) {
        pkt_num++;
        printf("It's in flush state, and Received %dth encoded pkt from MediaCodec's codec!!!\n", pkt_num);

        size_t bufSize;
        uint8_t *buf = AMediaCodec_getOutputBuffer(codec, outIdx, &bufSize);
        if (buf) {
          fwrite(buf, 1, info.size, outputFp);
        }
      }

      AMediaCodec_releaseOutputBuffer(codec, outIdx, true);
    }
  }

END:

  if (outputFp) {
    fclose(outputFp);
    outputFp = nullptr;
  }

  if (fmt_ctx) {
    avformat_close_input(&fmt_ctx);
    fmt_ctx = nullptr;
  }

  if (codec_ctx) {
    avcodec_free_context(&codec_ctx);
    codec_ctx = nullptr;
  }

  if (frame) {
    av_frame_free(&frame);
    frame = nullptr;
  }

  if (format) {
    AMediaFormat_delete(format);
    format = nullptr;
  }

  if (codec) {
    AMediaCodec_stop(codec);
    AMediaCodec_delete(codec);
    codec = nullptr;
  }

  printf("total send frame:%d, recv pkt:%d !!!\n", frame_num, pkt_num);

  return 0;
}

