/*******************************************************************************
 *
 * Copyright (C) 2023 NETINT Technologies
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 ******************************************************************************/

/*!*****************************************************************************
 *  \file   gstniquadrahwframe.c *
 *  \brief  Implement of NetInt Quadra hardware frame.
 ******************************************************************************/

#include "gstniquadrahwframe.h"

static gboolean
gst_ni_hwframe_meta_init (GstNiHWFrameMeta * ni_hwframe_meta,
    gpointer params, GstBuffer * buffer)
{
  ni_hwframe_meta->p_frame_ctx = NULL;

  return TRUE;
}

static void
gst_ni_hwframe_meta_free (GstNiHWFrameMeta * ni_hwframe_meta,
    GstBuffer * buffer)
{
  if (ni_hwframe_meta->p_frame_ctx) {
    gst_ni_hw_frame_context_unref (ni_hwframe_meta->p_frame_ctx);
  }
}

static gboolean
gst_ni_hwframe_meta_transform (GstBuffer * dest, GstMeta * meta,
    GstBuffer * buffer, GQuark type, gpointer data)
{
  GstNiHWFrameMeta *smeta = NULL;
  GstNiHWFrameMeta *dmeta = NULL;

  smeta = (GstNiHWFrameMeta *) meta;

  if (GST_META_TRANSFORM_IS_COPY (type)) {
    GstMetaTransformCopy *copy = data;

    if (!copy->region) {
      dmeta = (GstNiHWFrameMeta *) gst_buffer_add_meta (dest,
          GST_NI_HWFRAME_META_INFO, NULL);

      if (!dmeta)
        return FALSE;

      dmeta->p_frame_ctx = gst_ni_hw_frame_context_ref (smeta->p_frame_ctx);
    } else {
      /* It makes no sense to copy part of the HW frame instead of all of it */
      /* so return FALSE, if transform type is not supported */
      return FALSE;
    }
  }

  return TRUE;
}

GType
gst_ni_hwframe_meta_api_get_type (void)
{
  static GType type;
  static const gchar *tags[] = { "memory", NULL };

  if (g_once_init_enter (&type)) {
    GType _type = gst_meta_api_type_register ("GstNiHWFrameMetaAPI", tags);
    g_once_init_leave (&type, _type);
  }
  return type;
}

const GstMetaInfo *
gst_ni_hwframe_meta_get_info (void)
{
  static const GstMetaInfo *ni_hwframe_meta_info = NULL;

  if (g_once_init_enter ((GstMetaInfo **) & ni_hwframe_meta_info)) {
    const GstMetaInfo *meta = gst_meta_register (GST_NI_HWFRAME_META_API_TYPE,
        "GstNiHWFrameMeta", sizeof (GstNiHWFrameMeta),
        (GstMetaInitFunction) gst_ni_hwframe_meta_init,
        (GstMetaFreeFunction) gst_ni_hwframe_meta_free,
        (GstMetaTransformFunction) gst_ni_hwframe_meta_transform);
    g_once_init_leave ((GstMetaInfo **) & ni_hwframe_meta_info,
        (GstMetaInfo *) meta);
  }

  return ni_hwframe_meta_info;
}

/* define mini object for GstNiFrameContext*/
GST_DEFINE_MINI_OBJECT_TYPE (GstNiFrameContext, gst_ni_hw_frame_context);

static void
gst_ni_hw_frame_context_free (GstMiniObject * mini_obj)
{
  GstNiFrameContext *ctx = (GstNiFrameContext *) mini_obj;

  if (ctx->ni_surface) {
    ni_hwframe_buffer_recycle (ctx->ni_surface, ctx->ni_surface->device_handle);
  }

  g_free (ctx->ni_surface);
  g_slice_free (GstNiFrameContext, ctx);
}

GstNiFrameContext *
gst_ni_hw_frame_context_new (ni_session_context_t * p_session,
    niFrameSurface1_t * ni_surface, gint dev_idx)
{
  GstNiFrameContext *ctx = NULL;

  ctx = g_slice_new0 (GstNiFrameContext);
  if (!ctx) {
    return NULL;
  }

  gst_mini_object_init (GST_MINI_OBJECT_CAST (ctx), 0,
      GST_TYPE_NI_HW_FRAME_CONTEXT,
      NULL, NULL, (GstMiniObjectFreeFunction) gst_ni_hw_frame_context_free);

  ctx->ni_surface = g_malloc0 (sizeof (niFrameSurface1_t));
  if (!ctx->ni_surface) {
    return NULL;
  }

  memcpy (ctx->ni_surface, ni_surface, sizeof (niFrameSurface1_t));
  ctx->p_session = p_session;
  ctx->dev_idx = dev_idx;

  return ctx;
}

/*
 * When there is more than one metadata of type GST_NI_HWFRAME_META_INFO
 * in a GstBuffer, this API can be used to set the metadata ID.
 * It is not allowed to call this API after calling gst_buffer_add_ni_hwframe_meta.
 */
void
gst_ni_hw_frame_context_set_id (GstNiFrameContext * ctx, gint id)
{
  ctx->id = id;
}

void
gst_buffer_add_ni_hwframe_meta (GstBuffer * buffer,
    GstNiFrameContext * frame_ctx)
{
  GstNiHWFrameMeta *meta;

  g_return_if_fail (GST_IS_BUFFER (buffer));

  meta = (GstNiHWFrameMeta *) gst_buffer_add_meta (buffer,
      GST_NI_HWFRAME_META_INFO, NULL);

  if (meta) {
    meta->p_frame_ctx = gst_ni_hw_frame_context_ref (frame_ctx);
  }
}
