/*******************************************************************************
 *
 * Copyright (C) 2023 NETINT Technologies
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 ******************************************************************************/

/*!*****************************************************************************
 *  \file   gstniquadramerge.c
 *
 *  \brief  Implement of NetInt Quadra hardware merge filter.
 ******************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gst/gst.h>
#include <gst/video/video.h>

#include "niquadra.h"
#include "ni_device_api.h"
#include "gstniquadrahwframe.h"
#include "gstniquadrautils.h"

enum
{
  PROP_0,
  PROP_WIDTH,
  PROP_HEIGHT,
  PROP_FLITERBLIT,
  PROP_KEEP_ALIVE_TIMEOUT,
  PROP_LAST
};

typedef struct _GstNIQuadraMerge
{
  GstElement element;

  GstPad *sinkpad, *srcpad;

  gint in_width, in_height;
  GstVideoFormat format;

  gint out_width, out_height;
  guint keep_alive_timeout;

  gboolean initialized;
  ni_session_context_t api_ctx;
  ni_session_data_io_t api_dst_frame;
  ni_scaler_params_t params;
  guint extra_frames;
  gint downstream_card;
} GstNIQuadraMerge;

typedef struct _GstNIQuadraMergeClass
{
  GstElementClass parent_class;
} GstNIQuadraMergeClass;

#define GST_TYPE_NIQUADRAMERGE \
  (gst_niquadramerge_get_type())
#define GST_NIQUADRAMERGE(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_NIQUADRAMERGE,GstNIQuadraMerge))
#define GST_NIQUADRAMERGE_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_NIQUADRAMERGE,GstNIQuadraMerge))
#define GST_IS_NIQUADRAMERGE(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_NIQUADRAMERGE))
#define GST_IS_NIQUADRAMERGE_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_NIQUADRAMERGE))

GType gst_niquadramerge_get_type (void);

G_DEFINE_TYPE (GstNIQuadraMerge, gst_niquadramerge, GST_TYPE_ELEMENT);


#define FORMATS " { I420, NV12, ARGB, RGBA, ABGR, BGRA, "\
                "   I420_10LE, NV16, BGRx, NV12_10LE32 } "

static GstStaticPadTemplate src_factory = GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_MAKE (FORMATS))
    );

static GstStaticPadTemplate sink_factory = GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_MAKE (FORMATS))
    );

static GstFlowReturn gst_niquadramerge_chain (GstPad * pad, GstObject * parent,
    GstBuffer * inbuf);


static gboolean
gst_niquadramerge_sink_setcaps (GstPad * pad, GstObject * parent,
    GstCaps * caps)
{
  GstNIQuadraMerge *filter;
  GstVideoInfo info;
  GstCaps *src_caps;
  gboolean ret, gotit = FALSE;
  GstQuery *query;
  guint i;
  GType gtype;
  const GstStructure *params = NULL;

  filter = GST_NIQUADRAMERGE (parent);

  if (!gst_video_info_from_caps (&info, caps)) {
    return FALSE;
  }


  /* Query the downstream element for proposed allocation */
  query = gst_query_new_allocation (caps, TRUE);

  if (gst_pad_peer_query (filter->srcpad, query) == TRUE) {
    /* Search for allocation metadata */
    for (i = 0; i < gst_query_get_n_allocation_metas (query); i++) {
      gtype = gst_query_parse_nth_allocation_meta (query, i, &params);
      if (gtype == GST_VIDEO_META_API_TYPE) {
        if (params && (strcmp (gst_structure_get_name (params),
                    NI_PREALLOCATE_STRUCTURE_NAME) == 0)) {

          gotit = gst_structure_get_uint (params, NI_VIDEO_META_BUFCNT,
              &filter->extra_frames);
          if (gotit == FALSE)
            GST_ERROR_OBJECT (filter, "Did not find buffercnt\n");

          gotit = gst_structure_get_int (params, NI_VIDEO_META_CARDNO,
              &filter->downstream_card);
          if (gotit == FALSE)
            GST_ERROR_OBJECT (filter, "Did not find cardno\n");

          break;
        }
      }
    }
  }

  if (info.colorimetry.range == GST_VIDEO_COLOR_RANGE_0_255) {
    GST_DEBUG_OBJECT (filter,
        "WARNING: Full color range input, limited color output\n");
  }
  info.colorimetry.range = GST_VIDEO_COLOR_RANGE_16_235;

  gst_query_unref (query);

  filter->format = info.finfo->format;
  filter->in_width = info.width;
  filter->in_height = info.height;

  if (filter->out_width == 0)
    filter->out_width = filter->in_width;
  if (filter->out_height == 0)
    filter->out_height = filter->in_height;

  info.width = filter->out_width = NIALIGN (filter->out_width, 2);
  info.height = filter->out_height = NIALIGN (filter->out_height, 2);

  src_caps = gst_video_info_to_caps (&info);

  gst_caps_set_simple (src_caps, "hw_pix_fmt", G_TYPE_INT, PIX_FMT_NI_QUADRA,
      NULL);

  ret = gst_pad_set_caps (filter->srcpad, src_caps);
  gst_caps_unref (src_caps);

  return ret;
}

static gboolean
gst_niquadramerge_sink_event (GstPad * pad, GstObject * parent,
    GstEvent * event)
{
  GstNIQuadraMerge *filter = GST_NIQUADRAMERGE (parent);
  gboolean ret = FALSE;

  switch (GST_EVENT_TYPE (event)) {
    case GST_EVENT_CAPS:
    {
      GstCaps *caps;
      gst_event_parse_caps (event, &caps);
      ret = gst_niquadramerge_sink_setcaps (pad, parent, caps);
      gst_event_unref (event);
      break;
    }
    default:
      ret = gst_pad_push_event (filter->srcpad, event);
      break;
  }

  return ret;
}

static void
gst_niquadramerge_init (GstNIQuadraMerge * filter)
{
  filter->sinkpad = gst_pad_new_from_static_template (&sink_factory, "sink");
  gst_pad_set_event_function (filter->sinkpad, gst_niquadramerge_sink_event);
  gst_pad_set_chain_function (filter->sinkpad, gst_niquadramerge_chain);
  gst_element_add_pad (GST_ELEMENT (filter), filter->sinkpad);

  filter->srcpad = gst_pad_new_from_static_template (&src_factory, "src");
  gst_element_add_pad (GST_ELEMENT (filter), filter->srcpad);

  filter->out_width = 0;
  filter->out_height = 0;
  filter->params.filterblit = 0;

  filter->initialized = FALSE;
  filter->downstream_card = -1;
  filter->extra_frames = 0;
}

static void
gst_niquadramerge_dispose (GObject * obj)
{
  GstNIQuadraMerge *filter = GST_NIQUADRAMERGE (obj);

  if (filter->api_dst_frame.data.frame.p_buffer) {
    ni_frame_buffer_free (&filter->api_dst_frame.data.frame);
  }

  if (filter->api_ctx.session_id != NI_INVALID_SESSION_ID) {
    GST_DEBUG_OBJECT (filter, "libxcoder scale free context");
    ni_device_session_close (&filter->api_ctx, 1, NI_DEVICE_TYPE_SCALER);
  }
  ni_session_context_t *p_ctx = &filter->api_ctx;
  if (p_ctx) {
    if (p_ctx->device_handle != NI_INVALID_DEVICE_HANDLE) {
      ni_device_close (p_ctx->device_handle);
      p_ctx->device_handle = NI_INVALID_DEVICE_HANDLE;
    }
    if (p_ctx->blk_io_handle != NI_INVALID_DEVICE_HANDLE) {
      ni_device_close (p_ctx->blk_io_handle);
      p_ctx->blk_io_handle = NI_INVALID_DEVICE_HANDLE;
    }
  }
  ni_device_session_context_clear (&filter->api_ctx);

  G_OBJECT_CLASS (gst_niquadramerge_parent_class)->dispose (obj);
}

static GstFlowReturn
gst_niquadramerge_chain (GstPad * pad, GstObject * parent, GstBuffer * inbuf)
{
  GstNIQuadraMerge *filter;
  GstBuffer *outbuf;
  gpointer state = NULL;
  GstMeta *meta = NULL;
  const GstMetaInfo *info = NULL;
  niFrameSurface1_t *frame_0_surface = NULL;
  niFrameSurface1_t *frame_1_surface = NULL;
  GstNiHWFrameMeta *hwFrame1 = NULL;
  ni_session_data_io_t *p_session_data;
  int cardno = NI_INVALID_HWID;
  int gc620_pixfmt;
  int flags;
  ni_pix_fmt_t niPixFmt;
  int retval = 0;
  GstFlowReturn flow_ret = GST_FLOW_OK;

  filter = GST_NIQUADRAMERGE (parent);

  info = GST_NI_HWFRAME_META_INFO;
  while ((meta = gst_buffer_iterate_meta (inbuf, &state))) {
    if (meta->info->api == info->api) {
      GstNiHWFrameMeta *hwFrame = (GstNiHWFrameMeta *) meta;
      if (0 == hwFrame->p_frame_ctx->id) {
        frame_0_surface = hwFrame->p_frame_ctx->ni_surface;
      } else if (1 == hwFrame->p_frame_ctx->id) {
        frame_1_surface = hwFrame->p_frame_ctx->ni_surface;
        hwFrame1 = hwFrame;
      }
      cardno = hwFrame->p_frame_ctx->dev_idx;
    }
  }

  if (!frame_0_surface || !frame_1_surface) {
    flow_ret = GST_FLOW_ERROR;
    goto EXIT;
  }

  niPixFmt = convertGstVideoFormatToNIPix (filter->format);
  gc620_pixfmt = convertNIPixToGC620Format (niPixFmt);

  if (!filter->initialized) {
    if (!frame_0_surface || !frame_1_surface) {
      flow_ret = GST_FLOW_ERROR;
      GST_ERROR_OBJECT (filter, "gstniquadramerge frame_surdace is null");
      goto EXIT;
    }
    retval = ni_device_session_context_init (&filter->api_ctx);
    if (retval < 0) {
      flow_ret = GST_FLOW_ERROR;
      goto EXIT;
    }

    filter->api_ctx.session_id = NI_INVALID_SESSION_ID;
    filter->api_ctx.device_handle = NI_INVALID_DEVICE_HANDLE;
    filter->api_ctx.blk_io_handle = NI_INVALID_DEVICE_HANDLE;
    filter->api_ctx.hw_id = cardno;
    filter->api_ctx.device_type = NI_DEVICE_TYPE_SCALER;
    filter->api_ctx.scaler_operation = NI_SCALER_OPCODE_MERGE;
    filter->api_ctx.keep_alive_timeout = filter->keep_alive_timeout;
    filter->api_ctx.isP2P = 0;

    retval = ni_device_session_open (&filter->api_ctx, NI_DEVICE_TYPE_SCALER);
    if (retval < 0) {
      flow_ret = GST_FLOW_ERROR;
      if (filter->api_dst_frame.data.frame.p_buffer) {
        ni_frame_buffer_free (&filter->api_dst_frame.data.frame);
      }
      ni_device_session_close (&filter->api_ctx, 1, NI_DEVICE_TYPE_SCALER);
      ni_device_session_context_clear (&filter->api_ctx);
      goto EXIT;
    }

    if (filter->params.filterblit) {
      retval = ni_scaler_set_params (&filter->api_ctx, &(filter->params));
      if (retval < 0) {
        flow_ret = GST_FLOW_ERROR;
        goto EXIT;
      }
    }

    int pool_size = DEFAULT_NI_FILTER_POOL_SIZE;
    // If the downstream element is on a different card,
    // Allocate extra frames suggested by the downstream element
    if (filter->api_ctx.hw_id != filter->downstream_card) {
      pool_size += filter->extra_frames;
      GST_INFO_OBJECT (filter,
          "Increase frame pool by %d", filter->extra_frames);
    }

    int options = NI_SCALER_FLAG_IO | NI_SCALER_FLAG_PC;

    /* Allocate a pool of frames by the scaler */
    /* *INDENT-OFF* */
    retval = ni_device_alloc_frame (&filter->api_ctx,
      NI_ALIGN (filter->out_width, 2),
      NI_ALIGN (filter->out_height, 2),
      gc620_pixfmt, options,
      0,                      // rec width
      0,                      // rec height
      0,                      // rec X pos
      0,                      // rec Y pos
      pool_size,              // rgba color/pool size
      0,                      // frame index
      NI_DEVICE_TYPE_SCALER);
    /* *INDENT-ON* */
    if (retval < 0) {
      flow_ret = GST_FLOW_ERROR;
      goto EXIT;
    }

    filter->initialized = TRUE;
  }

  retval = ni_frame_buffer_alloc_hwenc (&filter->api_dst_frame.data.frame,
      filter->out_width, filter->out_height, 0);
  if (retval != NI_RETCODE_SUCCESS) {
    flow_ret = GST_FLOW_ERROR;
    goto EXIT;
  }

  /*
   * Allocate device input frame. This call won't actually allocate a frame,
   * but sends the incoming hardware frame index to the scaler manager
   */
  /* *INDENT-OFF* */
  retval = ni_device_alloc_frame (&filter->api_ctx,
      NI_ALIGN (filter->in_width, 2),
      NI_ALIGN (filter->in_height, 2),
      gc620_pixfmt,
      (frame_0_surface && frame_1_surface->encoding_type == 2) ? NI_SCALER_FLAG_CMP : 0,
      NI_ALIGN (filter->in_width, 2),  // src rectangle width
      NI_ALIGN (filter->in_height, 2),  // src rectangle height
      0,                        // src rectangle x = 0
      0,                        // src rectangle y = 0
      frame_0_surface->ui32nodeAddress,
      frame_0_surface->ui16FrameIdx, NI_DEVICE_TYPE_SCALER);
  /* *INDENT-ON* */
  if (retval != NI_RETCODE_SUCCESS) {
    flow_ret = GST_FLOW_ERROR;
    goto EXIT;
  }

  flags = NI_SCALER_FLAG_IO;
  flags |= (frame_1_surface
      && frame_1_surface->encoding_type == 2) ? NI_SCALER_FLAG_CMP : 0;
  /* Allocate device destination frame. This will acquire a frame from the pool */
  /* *INDENT-OFF* */
  retval = ni_device_alloc_frame (&filter->api_ctx,
      NI_ALIGN (filter->out_width, 2),
      NI_ALIGN (filter->out_height, 2),
      gc620_pixfmt,
      flags,    // output frame
      NI_ALIGN (filter->out_width, 2),  // dst rectangle width
      NI_ALIGN (filter->out_height, 2), // dst rectangle height
      0,                        // dst rectangle x
      0,                        // dst rectangle y
      frame_1_surface->ui32nodeAddress, // rgba color
      frame_1_surface->ui16FrameIdx, NI_DEVICE_TYPE_SCALER);
  /* *INDENT-ON* */
  if (retval != NI_RETCODE_SUCCESS) {
    flow_ret = GST_FLOW_ERROR;
    goto EXIT;
  }

  /* Set the new frame index */
  retval = ni_device_session_read_hwdesc (&filter->api_ctx,
      &filter->api_dst_frame, NI_DEVICE_TYPE_SCALER);
  if (retval != NI_RETCODE_SUCCESS) {
    flow_ret = GST_FLOW_ERROR;
    goto EXIT;
  }

  p_session_data = &filter->api_dst_frame;
  ni_frame_buffer_free (&p_session_data->data.frame);

  frame_1_surface->ui16width = filter->out_width;
  frame_1_surface->ui16height = filter->out_height;

  ni_set_bit_depth_and_encoding_type (&frame_1_surface->bit_depth,
      &frame_1_surface->encoding_type, niPixFmt);

  outbuf = gst_buffer_new_and_alloc (0);
  gst_buffer_add_ni_hwframe_meta (outbuf, hwFrame1->p_frame_ctx);

  outbuf = gst_buffer_make_writable (outbuf);
  gst_buffer_copy_into (outbuf, inbuf,
      GST_BUFFER_COPY_TIMESTAMPS | GST_BUFFER_COPY_META, 0, -1);
  gst_buffer_unref (inbuf);

  flow_ret = gst_pad_push (filter->srcpad, outbuf);
  return flow_ret;

EXIT:
  if (inbuf) {
    gst_buffer_unref (inbuf);
  }

  return flow_ret;
}

static void
gst_niquadramerge_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstNIQuadraMerge *self;

  g_return_if_fail (GST_IS_NIQUADRAMERGE (object));
  self = GST_NIQUADRAMERGE (object);

  switch (prop_id) {
    case PROP_WIDTH:
      self->out_width = g_value_get_int (value);
      break;
    case PROP_HEIGHT:
      self->out_height = g_value_get_int (value);
      break;
    case PROP_FLITERBLIT:
      self->params.filterblit = g_value_get_int (value);
      break;
    case PROP_KEEP_ALIVE_TIMEOUT:
      self->keep_alive_timeout = g_value_get_uint (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (self, prop_id, pspec);
  }
}

static void
gst_niquadramerge_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec)
{
  GstNIQuadraMerge *self;

  g_return_if_fail (GST_IS_NIQUADRAMERGE (object));
  self = GST_NIQUADRAMERGE (object);

  switch (prop_id) {
    case PROP_WIDTH:
      g_value_set_int (value, self->out_width);
      break;
    case PROP_HEIGHT:
      g_value_set_int (value, self->out_height);
      break;
    case PROP_FLITERBLIT:
      g_value_set_int (value, self->params.filterblit);
      break;
    case PROP_KEEP_ALIVE_TIMEOUT:
      g_value_set_uint (value, self->keep_alive_timeout);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (self, prop_id, pspec);
  }
}

static void
gst_niquadramerge_class_init (GstNIQuadraMergeClass * klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;
  GstElementClass *element_class = GST_ELEMENT_CLASS (klass);

  gobject_class->set_property = gst_niquadramerge_set_property;
  gobject_class->get_property = gst_niquadramerge_get_property;

  g_object_class_install_property (gobject_class, PROP_WIDTH,
      g_param_spec_int ("width", "Width",
          "Width of the image output", 0, G_MAXINT, 0,
          G_PARAM_CONSTRUCT | G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));

  g_object_class_install_property (gobject_class, PROP_HEIGHT,
      g_param_spec_int ("height", "Height",
          "Height of the image output", 0, G_MAXINT, 0,
          G_PARAM_CONSTRUCT | G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));
  g_object_class_install_property (gobject_class, PROP_FLITERBLIT,
      g_param_spec_int ("filterblit", "FILTERBLIT",
          "set filterblit", 0, 2, 0,
          G_PARAM_CONSTRUCT | G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));

  g_object_class_install_property (gobject_class, PROP_KEEP_ALIVE_TIMEOUT,
      g_param_spec_uint ("keep-alive-timeout", "Keep-alive-timeout",
          "Specify a custom session keep alive timeout in seconds",
          NI_MIN_KEEP_ALIVE_TIMEOUT, NI_MAX_KEEP_ALIVE_TIMEOUT,
          NI_DEFAULT_KEEP_ALIVE_TIMEOUT,
          G_PARAM_CONSTRUCT | G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));

  gst_element_class_add_static_pad_template (element_class, &src_factory);
  gst_element_class_add_static_pad_template (element_class, &sink_factory);

  gst_element_class_set_static_metadata (element_class,
      "NETINT Quadra MERGE filter", "Filter/Effect/Video/NIMerge",
      "MERGE Netint Quadra", "Minglong Zhang<minglong.zhang@netint.cn>");

  gobject_class->dispose = gst_niquadramerge_dispose;
}

gboolean
gst_niquadramerge_register (GstPlugin * plugin)
{
  return gst_element_register (plugin, "niquadramerge", GST_RANK_NONE,
      GST_TYPE_NIQUADRAMERGE);
}
