/*******************************************************************************
 *
 * Copyright (C) 2023 NETINT Technologies
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 ******************************************************************************/

/*!*****************************************************************************
 *  \file   gstniquadrapad.c
 *
 *  \brief  Implement of NetInt Quadra hardware pad filter.
 ******************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gst/gst.h>
#include <gst/video/video.h>
#include <gst/base/gstbasetransform.h>

#include "niquadra.h"
#include "ni_device_api.h"
#include "gstniquadrahwframe.h"
#include "gstniquadrautils.h"
#include "gstniquadrabaseconvert.h"

enum
{
  PROP_0,
  PROP_X,
  PROP_Y,
  PROP_COLOR,
  PROP_ASPECT,
  PROP_IS_P2P,
  PROP_LAST
};

#define DEFAULT_COLOR "black"

typedef struct _GstNIQuadraPad
{
  GstNiQuadraBaseConvert parent;

  gint x, y;
  gchar *color;
  gchar *aspect_str;
  gboolean is_p2p;
  guint shift;

  gst_rational aspect;
  uint8_t rgba_color[4];

  ni_session_data_io_t api_dst_frame;
} GstNIQuadraPad;

typedef struct _GstNIQuadraPadClass
{
  GstNiQuadraBaseConvertClass parent_class;
} GstNIQuadraPadClass;

#define GST_TYPE_NIQUADRAPAD \
  (gst_niquadrapad_get_type())
#define GST_NIQUADRAPAD(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_NIQUADRAPAD,GstNIQuadraPad))
#define GST_NIQUADRAPAD_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_NIQUADRAPAD,GstNIQuadraPad))
#define GST_IS_NIQUADRAPAD(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_NIQUADRAPAD))
#define GST_IS_NIQUADRAPAD_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_NIQUADRAPAD))

#define gst_niquadrapad_parent_class parent_class

GType gst_niquadrapad_get_type (void);

G_DEFINE_TYPE (GstNIQuadraPad, gst_niquadrapad,
    GST_TYPE_NI_QUADRA_BASE_CONVERT);


#define FORMATS " { I420, NV12, ARGB, RGBA, ABGR, BGRA, "\
                "   I420_10LE, NV16, BGRx, NV12_10LE32 } "

static GstStaticPadTemplate src_factory = GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_MAKE (FORMATS))
    );

static GstStaticPadTemplate sink_factory = GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_MAKE (FORMATS))
    );

static int
gst_draw_round_to_sub (unsigned int shift, int sub_dir, int round_dir,
    int value)
{
  if (!shift)
    return value;
  if (round_dir >= 0)
    value += round_dir ? (1 << shift) - 1 : 1 << (shift - 1);
  return (value >> shift) << shift;
}

static GstCaps *
gst_niquadrapad_fixate_caps (GstBaseTransform * base,
    GstPadDirection direction, GstCaps * caps, GstCaps * othercaps)
{
  GstNIQuadraPad *self = GST_NIQUADRAPAD (base);
  GstNiQuadraBaseConvert *parent = GST_NI_QUADRA_BASE_CONVERT (base);
  GstVideoInfo info;
  GstVideoFormat in_pixfmt;

  GST_DEBUG_OBJECT (base,
      "trying to fixate othercaps %" GST_PTR_FORMAT " based on caps %"
      GST_PTR_FORMAT, othercaps, caps);

  othercaps = remove_structures_from_caps (othercaps, "video/x-raw", 1);

  if (direction == GST_PAD_SINK) {
    GstStructure *outs;
    gint w = 0, h = 0;
    gboolean is_downstream_configured = FALSE;

    if (!gst_video_info_from_caps (&info, caps)) {
      return NULL;
    }

    in_pixfmt = info.finfo->format;
    if (in_pixfmt == GST_VIDEO_FORMAT_I420
        || in_pixfmt == GST_VIDEO_FORMAT_NV12
        || in_pixfmt == GST_VIDEO_FORMAT_I420_10LE
        || in_pixfmt == GST_VIDEO_FORMAT_P010_10LE) {
      self->shift = 1;
    } else {
      self->shift = 0;
    }

    outs = gst_caps_get_structure (othercaps, 0);
    if ((gst_structure_get_int (outs, "width", &w)) ||
        (gst_structure_get_int (outs, "height", &h))) {
      is_downstream_configured = TRUE;
    }

    if (!is_downstream_configured) {
      GstStructure *ins;
      const gchar *in_format;
      gint in_w, in_h;
      gint out_w, out_h;

      ins = gst_caps_get_structure (caps, 0);
      in_format = gst_structure_get_string (ins, "format");
      gst_structure_get_int (ins, "width", &in_w);
      gst_structure_get_int (ins, "height", &in_h);

      if (self->color && gst_parse_color (self->rgba_color,
              self->color, -1) < 0) {
        return NULL;
      }

      if (self->aspect_str && gst_parse_aspect (self->aspect_str,
              &self->aspect) < 0) {
        return NULL;
      }

      if (parent->width) {
        out_w = parent->width;
      } else {
        out_w = in_w;
      }

      if (parent->height) {
        out_h = parent->height;
      } else {
        out_h = in_h;
      }

      gst_rational in_aspect = { info.par_n, info.par_d };

      gst_rational adjusted_aspect = self->aspect;

      if (adjusted_aspect.num && adjusted_aspect.den) {
        adjusted_aspect = gst_div_q (adjusted_aspect, in_aspect);
        if (out_h < gst_rescale (out_w, adjusted_aspect.den,
                adjusted_aspect.num)) {
          out_h = gst_rescale (out_w, adjusted_aspect.den, adjusted_aspect.num);
        } else {
          out_w = gst_rescale (out_h, adjusted_aspect.num, adjusted_aspect.den);
        }
      }

      out_w = gst_draw_round_to_sub (self->shift, 0, -1, out_w);
      out_h = gst_draw_round_to_sub (self->shift, 1, -1, out_h);
      self->x = gst_draw_round_to_sub (self->shift, 0, -1, self->x);
      self->y = gst_draw_round_to_sub (self->shift, 1, -1, self->y);
      in_w = gst_draw_round_to_sub (self->shift, 0, -1, in_w);
      in_h = gst_draw_round_to_sub (self->shift, 1, -1, in_h);

      gst_structure_set (ins, "width", G_TYPE_INT, in_w, NULL);
      gst_structure_set (ins, "height", G_TYPE_INT, in_h, NULL);

      gst_structure_set (outs, "width", G_TYPE_INT, out_w, NULL);
      gst_structure_set (outs, "height", G_TYPE_INT, out_h, NULL);

      gst_structure_set (outs, "format", G_TYPE_STRING, in_format, NULL);
    }
  }

  othercaps = GST_NI_QUADRA_BASE_CONVERT_CLASS
      (parent_class)->fixate_size (base, direction, caps, othercaps);
  if (gst_caps_get_size (othercaps) == 1) {
    gint i;
    const gchar *format_fields[] = { "format", "colorimetry", "chroma-site" };
    GstStructure *format_struct = gst_caps_get_structure (caps, 0);
    GstStructure *fixated_struct;

    othercaps = gst_caps_make_writable (othercaps);
    fixated_struct = gst_caps_get_structure (othercaps, 0);

    for (i = 0; i < G_N_ELEMENTS (format_fields); i++) {
      if (gst_structure_has_field (format_struct, format_fields[i])) {
        gst_structure_set (fixated_struct, format_fields[i], G_TYPE_STRING,
            gst_structure_get_string (format_struct, format_fields[i]), NULL);
      } else {
        gst_structure_remove_field (fixated_struct, format_fields[i]);
      }
    }

    if (direction == GST_PAD_SINK) {
      if (gst_structure_has_field (fixated_struct, "format")) {
        const gchar *color;
        gchar *colorimetry_str;
        GstVideoColorimetry colorimetry;
        GstVideoFormat fmt =
            gst_video_format_from_string (gst_structure_get_string
            (fixated_struct, "format"));
        switch (fmt) {
          case GST_VIDEO_FORMAT_RGB:
          case GST_VIDEO_FORMAT_RGBA:
          case GST_VIDEO_FORMAT_ARGB:
          case GST_VIDEO_FORMAT_ABGR:
          case GST_VIDEO_FORMAT_GBRA:
          case GST_VIDEO_FORMAT_BGRx:
            color = gst_structure_get_string (fixated_struct, "colorimetry");
            gst_video_colorimetry_from_string (&colorimetry, color);

            colorimetry.matrix = GST_VIDEO_COLOR_MATRIX_RGB;
            colorimetry_str = gst_video_colorimetry_to_string (&colorimetry);
            if (colorimetry_str != NULL) {
              gst_structure_set (fixated_struct, "colorimetry", G_TYPE_STRING,
                  colorimetry_str, NULL);
              g_free (colorimetry_str);
            }
            break;

          default:
            break;
        }
      }

      if (gst_structure_has_field (fixated_struct, "colorimetry")) {
        const gchar *color;
        gchar *colorimetry_str;
        GstVideoColorimetry colorimetry;
        if ((color = gst_structure_get_string (fixated_struct, "colorimetry"))) {
          gst_video_colorimetry_from_string (&colorimetry, color);
        }

        if (colorimetry.range != GST_VIDEO_COLOR_RANGE_16_235) {
          GST_DEBUG_OBJECT (base,
              "WARNING: Full color range input, limited color output\n");
          colorimetry.range = GST_VIDEO_COLOR_RANGE_16_235;
          colorimetry_str = gst_video_colorimetry_to_string (&colorimetry);
          if (colorimetry_str != NULL) {
            gst_structure_set (fixated_struct, "colorimetry", G_TYPE_STRING,
                colorimetry_str, NULL);
            g_free (colorimetry_str);
          }
        }
      }
    }
  }

  return othercaps;
}

static GstFlowReturn
gst_niquadrapad_transform (GstBaseTransform * trans,
    GstBuffer * inbuf, GstBuffer * outbuf)
{
  GstNIQuadraPad *self = GST_NIQUADRAPAD (trans);
  GstNiQuadraBaseConvert *parent = GST_NI_QUADRA_BASE_CONVERT (trans);
  ni_session_data_io_t *p_session_data = &self->api_dst_frame;
  GstNiHWFrameMeta *hwFrameMeta = NULL;
  niFrameSurface1_t *frameSurface1 = NULL;
  niFrameSurface1_t *new_frame_surface = NULL;
  int gc620_pixfmt;
  ni_pix_fmt_t niPixFmt;
  int pool_size = parent->hwframe_pool_size;
  int retval = 0;

  if (parent->is_skip) {
    return GST_FLOW_OK;
  }

  memset (p_session_data, 0, sizeof (ni_session_data_io_t));

  hwFrameMeta = (GstNiHWFrameMeta *) gst_buffer_get_meta (inbuf,
      GST_NI_HWFRAME_META_API_TYPE);
  if (hwFrameMeta == NULL) {
    GST_ERROR_OBJECT (trans,
        "Impossible to convert between the formats supported by the filter");
    return GST_FLOW_ERROR;
  }

  frameSurface1 = hwFrameMeta->p_frame_ctx->ni_surface;

  if (!parent->initialized) {
    ni_device_session_context_init (&parent->api_ctx);
    parent->api_ctx.session_id = NI_INVALID_SESSION_ID;
    parent->api_ctx.device_handle = NI_INVALID_DEVICE_HANDLE;
    parent->api_ctx.blk_io_handle = NI_INVALID_DEVICE_HANDLE;
    parent->api_ctx.hw_id = hwFrameMeta->p_frame_ctx->dev_idx;
    parent->api_ctx.device_type = NI_DEVICE_TYPE_SCALER;
    parent->api_ctx.scaler_operation = NI_SCALER_OPCODE_PAD;
    parent->api_ctx.keep_alive_timeout = parent->keep_alive_timeout;
    parent->api_ctx.isP2P = self->is_p2p;

    retval = ni_device_session_open (&parent->api_ctx, NI_DEVICE_TYPE_SCALER);
    if (retval < 0) {
      GST_ERROR_OBJECT (trans, "Open crop device session error\n");
      ni_device_session_close (&parent->api_ctx, 1, NI_DEVICE_TYPE_SCALER);
      ni_device_session_context_clear (&parent->api_ctx);
      return GST_FLOW_ERROR;
    } else {
      GST_DEBUG_OBJECT (trans,
          "XCoder %s.%d (inst: %d) opened successfully\n",
          parent->api_ctx.dev_xcoder_name, parent->api_ctx.hw_id,
          parent->api_ctx.session_id);
    }

    memset (&self->api_dst_frame, 0, sizeof (ni_session_data_io_t));

    if (parent->api_ctx.isP2P) {
      pool_size = 1;
    }

    niPixFmt = convertGstVideoFormatToNIPix (parent->out_fmt);
    gc620_pixfmt = convertNIPixToGC620Format (niPixFmt);


    int options = NI_SCALER_FLAG_IO | NI_SCALER_FLAG_PC;
    if (self->is_p2p) {
      options |= NI_SCALER_FLAG_P2;
    }
    /* Allocate a pool of frames by the scaler */
    /* *INDENT-OFF* */
    retval = ni_device_alloc_frame (&parent->api_ctx,
        NI_ALIGN (parent->out_width, 2),
        NI_ALIGN (parent->out_height, 2),
        gc620_pixfmt,
        options,
        0,                      // rec width
        0,                      // rec height
        0,                      // rec X pos
        0,                      // rec Y pos
        pool_size,              // rgba color/pool size
        0,                      // frame index
        NI_DEVICE_TYPE_SCALER);
    /* *INDENT-ON* */
    if (retval < 0) {
      GST_ERROR_OBJECT (trans, "Init frame pool error\n");
      gst_buffer_unref (inbuf);
      ni_device_session_context_clear (&parent->api_ctx);
      return GST_FLOW_ERROR;
    }

    parent->initialized = TRUE;
  }

  retval = ni_frame_buffer_alloc_hwenc (&p_session_data->data.frame,
      parent->out_width, parent->out_height, 0);
  if (retval != NI_RETCODE_SUCCESS) {
    GST_ERROR_OBJECT (trans, "Can't assign input frame %d\n", retval);
    gst_buffer_unref (inbuf);
    return GST_FLOW_ERROR;
  }

  niPixFmt = convertGstVideoFormatToNIPix (parent->in_fmt);
  gc620_pixfmt = convertNIPixToGC620Format (niPixFmt);

  /*
   * Allocate device input frame. This call won't actually allocate a frame,
   * but sends the incoming hardware frame index to the scaler manager
   */
  /* *INDENT-OFF* */
  retval = ni_device_alloc_frame (&parent->api_ctx,
    NI_ALIGN (parent->in_width, 2),
    NI_ALIGN (parent->in_height, 2),
    gc620_pixfmt,
    0,                        // input frame
    parent->in_width,         // src rectangle width
    parent->in_height,        // src rectangle height
    0,                        // src rectangle x = 0
    0,                        // src rectangle y = 0
    0, frameSurface1->ui16FrameIdx, NI_DEVICE_TYPE_SCALER);
  /* *INDENT-ON* */

  if (retval != NI_RETCODE_SUCCESS) {
    GST_ERROR_OBJECT (trans, "Can't allocate device output frame %d", retval);
    gst_buffer_unref (inbuf);
    ni_frame_buffer_free (&p_session_data->data.frame);
    return GST_FLOW_ERROR;
  }

  uint32_t ui32RgbaColor;
  /* Scaler uses BGRA color, or ARGB in little-endian */
  ui32RgbaColor = (self->rgba_color[3] << 24) | (self->rgba_color[0] << 16)
      | (self->rgba_color[1] << 8) | self->rgba_color[2];
  /* Allocate hardware device destination frame. This acquires a frame from the pool */
  /* *INDENT-OFF* */
  retval = ni_device_alloc_frame (&parent->api_ctx,
    NI_ALIGN (parent->out_width, 2),
    NI_ALIGN (parent->out_height, 2),
    gc620_pixfmt,
    NI_SCALER_FLAG_IO,        // output frame
    parent->in_width,         // dst rectangle width
    parent->in_height,        // dst rectangle height
    self->x,                  // dst rectangle x
    self->y,                  // dst rectangle y
    ui32RgbaColor,            // rgba color
    -1, NI_DEVICE_TYPE_SCALER);
  /* *INDENT-ON* */

  if (retval != NI_RETCODE_SUCCESS) {
    GST_ERROR_OBJECT (trans, "Can't allocate device output frame %d", retval);
    gst_buffer_unref (inbuf);
    ni_frame_buffer_free (&p_session_data->data.frame);
    return GST_FLOW_ERROR;
  }

  /* Set the new frame index */
  retval = ni_device_session_read_hwdesc (&parent->api_ctx, p_session_data,
      NI_DEVICE_TYPE_SCALER);
  if (retval != NI_RETCODE_SUCCESS) {
    GST_ERROR_OBJECT (trans, "Can't acquire output frame %d\n", retval);
    gst_buffer_unref (inbuf);
    ni_frame_buffer_free (&p_session_data->data.frame);
    return GST_FLOW_ERROR;
  }

  new_frame_surface =
      (niFrameSurface1_t *) p_session_data->data.frame.p_data[3];
  new_frame_surface->dma_buf_fd = 0;
  new_frame_surface->ui32nodeAddress = 0;
  new_frame_surface->ui16width = parent->out_width;
  new_frame_surface->ui16height = parent->out_height;

  gst_set_bit_depth_and_encoding_type (&new_frame_surface->bit_depth,
      &new_frame_surface->encoding_type, parent->out_fmt);

  GstNiFrameContext *hwFrame =
      gst_ni_hw_frame_context_new (&parent->api_ctx, new_frame_surface,
      parent->api_ctx.hw_id);
  gst_buffer_add_ni_hwframe_meta (outbuf, hwFrame);
  gst_ni_hw_frame_context_unref (hwFrame);
  ni_frame_buffer_free (&p_session_data->data.frame);

  return GST_FLOW_OK;
}

static void
gst_niquadrapad_finalize (GObject * obj)
{
  GstNIQuadraPad *self = GST_NIQUADRAPAD (obj);
  GstBaseTransform *base = GST_BASE_TRANSFORM (self);
  GstNiQuadraBaseConvert *parent = GST_NI_QUADRA_BASE_CONVERT (obj);

  if (self->api_dst_frame.data.frame.p_buffer) {
    ni_frame_buffer_free (&self->api_dst_frame.data.frame);
  }

  if (parent->initialized) {
    if (parent->api_ctx.session_id != NI_INVALID_SESSION_ID) {
      GST_DEBUG_OBJECT (base, "libxcoder pad free context\n");
      ni_device_session_close (&parent->api_ctx, 1, NI_DEVICE_TYPE_SCALER);
    }

    ni_session_context_t *p_ctx = &parent->api_ctx;
    if (p_ctx) {
      if (p_ctx->device_handle != NI_INVALID_DEVICE_HANDLE) {
        ni_device_close (p_ctx->device_handle);
        p_ctx->device_handle = NI_INVALID_DEVICE_HANDLE;
      }
      if (p_ctx->blk_io_handle != NI_INVALID_DEVICE_HANDLE) {
        ni_device_close (p_ctx->blk_io_handle);
        p_ctx->blk_io_handle = NI_INVALID_DEVICE_HANDLE;
      }
    }
    ni_device_session_context_clear (&parent->api_ctx);

    parent->initialized = FALSE;
  }

  if (self->color) {
    g_free (self->color);
  }

  if (self->aspect_str) {
    g_free (self->aspect_str);
  }

  G_OBJECT_CLASS (gst_niquadrapad_parent_class)->finalize (obj);
}

static void
gst_niquadrapad_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstNIQuadraPad *self;

  g_return_if_fail (GST_IS_NIQUADRAPAD (object));
  self = GST_NIQUADRAPAD (object);

  switch (prop_id) {
    case PROP_X:
      self->x = g_value_get_int (value);
      break;
    case PROP_Y:
      self->y = g_value_get_int (value);
      break;
    case PROP_COLOR:
      g_free (self->color);
      self->color = g_value_dup_string (value);
      break;
    case PROP_ASPECT:
      g_free (self->aspect_str);
      self->aspect_str = g_value_dup_string (value);
      break;
    case PROP_IS_P2P:
      self->is_p2p = g_value_get_boolean (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (self, prop_id, pspec);
  }
}

static void
gst_niquadrapad_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec)
{
  GstNIQuadraPad *self;

  g_return_if_fail (GST_IS_NIQUADRAPAD (object));
  self = GST_NIQUADRAPAD (object);

  switch (prop_id) {
    case PROP_X:
      g_value_set_int (value, self->x);
      break;
    case PROP_Y:
      g_value_set_int (value, self->y);
      break;
    case PROP_COLOR:
      g_value_take_string (value, g_strdup (self->color));
      break;
    case PROP_ASPECT:
      g_value_take_string (value, g_strdup (self->aspect_str));
      break;
    case PROP_IS_P2P:
      g_value_set_boolean (value, self->is_p2p);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (self, prop_id, pspec);
  }
}

static void
gst_niquadrapad_class_init (GstNIQuadraPadClass * klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;
  GstElementClass *element_class = GST_ELEMENT_CLASS (klass);
  GstBaseTransformClass *trans_class = GST_BASE_TRANSFORM_CLASS (klass);

  gobject_class->set_property = gst_niquadrapad_set_property;
  gobject_class->get_property = gst_niquadrapad_get_property;
  gobject_class->finalize = gst_niquadrapad_finalize;

  g_object_class_install_property (gobject_class, PROP_X,
      g_param_spec_int ("x", "X",
          "X position of the padded area", 0, G_MAXINT, 0,
          G_PARAM_CONSTRUCT | G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));

  g_object_class_install_property (gobject_class, PROP_Y,
      g_param_spec_int ("y", "Y",
          "Y position of the padded area", 0, G_MAXINT, 0,
          G_PARAM_CONSTRUCT | G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));

  g_object_class_install_property (gobject_class, PROP_COLOR,
      g_param_spec_string ("color", "Color",
          "Color of the padded area", DEFAULT_COLOR,
          G_PARAM_CONSTRUCT | G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));

  g_object_class_install_property (gobject_class, PROP_ASPECT,
      g_param_spec_string ("aspect", "Aspect",
          "Pad to fit an aspect instead of a resolution", NULL,
          G_PARAM_CONSTRUCT | G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));

  g_object_class_install_property (gobject_class, PROP_IS_P2P,
      g_param_spec_boolean ("is-p2p", "Is-p2p",
          "enable p2p transfer", FALSE, G_PARAM_CONSTRUCT | G_PARAM_READWRITE));

  gst_element_class_add_static_pad_template (element_class, &src_factory);
  gst_element_class_add_static_pad_template (element_class, &sink_factory);

  gst_element_class_set_static_metadata (element_class,
      "NETINT Quadra PAD filter", "Filter/Effect/Video/NIPad",
      "PAD Netint Quadra", "Leo Liu<leo.liu@netint.cn>");

  trans_class->fixate_caps = gst_niquadrapad_fixate_caps;
  trans_class->transform = gst_niquadrapad_transform;
}

static void
gst_niquadrapad_init (GstNIQuadraPad * filter)
{
  filter->x = -1;
  filter->y = -1;
  filter->is_p2p = FALSE;
}

gboolean
gst_niquadrapad_register (GstPlugin * plugin)
{
  return gst_element_register (plugin, "niquadrapad", GST_RANK_NONE,
      GST_TYPE_NIQUADRAPAD);
}
