/*******************************************************************************
 *
 * Copyright (C) 2023 NETINT Technologies
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 *
 ******************************************************************************/

/*!*****************************************************************************
 *  \file   GstNiQuadraRotate.c
 *
 *  \brief  Implement of NetInt Quadra hardware rotate filter.
 ******************************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <gst/gst.h>
#include <gst/video/video.h>
#include "niquadra.h"
#include "ni_device_api.h"
#include "gstniquadrahwframe.h"
#include "gstniquadrautils.h"

enum
{
  PROP_0,
  GST_NIQUADRA_DEC_PROP_TIMEOUT,
  PROP_W,
  PROP_H,
  PROP_P2P,
  PROP_COLOR,
  PROP_ANGLE,
  PROP_AUTO_SKIP,
  PROP_LAST
};

#define BUFFER_WIDTH_PIXEL_ALIGNMENT 16

typedef struct _GstNiQuadraRotate
{
  GstElement element;

  GstPad *sinkpad, *srcpad;

  gint in_width, in_height;
  gint out_width, out_height;
  GstVideoFormat format;

  ni_session_context_t api_ctx;
  ni_session_data_io_t api_dst_frame;
  ni_frame_config_t output_frame_config;

  int initialized;
  guint keep_alive_timeout;     /* keep alive timeout setting */
  bool is_p2p;
  bool auto_skip;
  bool skip_filter;
  int angle;
  char *fillcolor_str;
  uint8_t fillcolor[4];
  bool fillcolor_enable;
  guint extra_frames;
  gint downstream_card;
} GstNiQuadraRotate;

typedef struct _GstNiQuadraRotateClass
{
  GstElementClass parent_class;
} GstNiQuadraRotateClass;

#define GST_TYPE_NIQUADRAROTATE \
  (gst_niquadrarotate_get_type())
#define GST_NIQUADRAROTATE(obj) \
  (G_TYPE_CHECK_INSTANCE_CAST((obj),GST_TYPE_NIQUADRAROTATE,GstNiQuadraRotate))
#define GST_NIQUADRAROTATE_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_CAST((klass),GST_TYPE_NIQUADRAROTATE,GstNiQuadraRotate))
#define GST_IS_NIQUADRAROTATE(obj) \
  (G_TYPE_CHECK_INSTANCE_TYPE((obj),GST_TYPE_NIQUADRAROTATE))
#define GST_IS_NIQUADRAROTATE_CLASS(klass) \
  (G_TYPE_CHECK_CLASS_TYPE((klass),GST_TYPE_NIQUADRAROTATE))

GType gst_niquadrarotateniquadra_get_type (void);

G_DEFINE_TYPE (GstNiQuadraRotate, gst_niquadrarotate, GST_TYPE_ELEMENT);

static GstStaticPadTemplate src_factory = GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_MAKE
        ("{ I420, YUY2, UYVY, NV12, ARGB, RGBA, ABGR, BGRA, I420_10LE, P010_10LE, NV16, BGRx, NV12_10LE32 }"))
    );

static GstStaticPadTemplate sink_factory = GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (GST_VIDEO_CAPS_MAKE
        ("{ I420, YUY2, UYVY, NV12, ARGB, RGBA, ABGR, BGRA, I420_10LE, P010_10LE, NV16, BGRx, NV12_10LE32 }"))
    );

static GstFlowReturn gst_niquadra_rotate_chain (GstPad * pad,
    GstObject * parent, GstBuffer * inbuf);

static gboolean
gst_niquadra_rotate_sink_setcaps (GstPad * pad, GstObject * parent,
    GstCaps * caps)
{
  GstNiQuadraRotate *filter = GST_NIQUADRAROTATE (parent);
  GstStructure *structure = gst_caps_get_structure (caps, 0);
  GstCaps *src_caps;
  gboolean ret, gotit = FALSE;
  GstVideoInfo info;
  GstQuery *query;
  guint i;
  GType gtype;
  const GstStructure *params = NULL;
  if (!gst_structure_get_int (structure, "width", &filter->in_width))
    return FALSE;
  if (!gst_structure_get_int (structure, "height", &filter->in_height))
    return FALSE;

  if (!gst_video_info_from_caps (&info, caps))
    return FALSE;

  if (info.colorimetry.range == GST_VIDEO_COLOR_RANGE_0_255) {
    GST_DEBUG_OBJECT (filter,
        "WARNING: Full color range input, limited color output\n");
  }
  info.colorimetry.range = GST_VIDEO_COLOR_RANGE_16_235;

  /* Query the downstream element for proposed allocation */
  query = gst_query_new_allocation (caps, TRUE);

  if (gst_pad_peer_query (filter->srcpad, query) == TRUE) {
    /* Search for allocation metadata */
    for (i = 0; i < gst_query_get_n_allocation_metas (query); i++) {
      gtype = gst_query_parse_nth_allocation_meta (query, i, &params);
      if (gtype == GST_VIDEO_META_API_TYPE) {
        if (params && (strcmp (gst_structure_get_name (params),
                    NI_PREALLOCATE_STRUCTURE_NAME) == 0)) {

          gotit = gst_structure_get_uint (params, NI_VIDEO_META_BUFCNT,
              &filter->extra_frames);
          if (gotit == FALSE)
            GST_ERROR_OBJECT (filter, "Did not find buffercnt\n");

          gotit = gst_structure_get_int (params, NI_VIDEO_META_CARDNO,
              &filter->downstream_card);
          if (gotit == FALSE)
            GST_ERROR_OBJECT (filter, "Did not find cardno\n");

          break;
        }
      }
    }
  }

  gst_query_unref (query);

  filter->format = info.finfo->format;
  if (filter->out_width == 0)
    filter->out_width = filter->in_width;
  if (filter->out_height == 0)
    filter->out_height = filter->in_height;
  info.width = filter->out_width = NI_ALIGN (filter->out_width, 2);
  info.height = filter->out_height = NI_ALIGN (filter->out_height, 2);
  src_caps = gst_video_info_to_caps (&info);

  if (filter->auto_skip && !filter->is_p2p && filter->angle == 0 &&
      filter->out_width == filter->in_width
      && filter->out_height == filter->in_height) {
    filter->skip_filter = true;
  }

  gst_caps_set_simple (src_caps, "hw_pix_fmt", G_TYPE_INT, PIX_FMT_NI_QUADRA,
      NULL);

  ret = gst_pad_set_caps (filter->srcpad, src_caps);
  gst_caps_unref (src_caps);

  return ret;
}

static gboolean
gst_niquadra_rotate_sink_event (GstPad * pad, GstObject * parent,
    GstEvent * event)
{
  GstNiQuadraRotate *filter = GST_NIQUADRAROTATE (parent);
  gboolean ret = FALSE;

  switch (GST_EVENT_TYPE (event)) {
    case GST_EVENT_CAPS:{
      GstCaps *caps;

      gst_event_parse_caps (event, &caps);
      ret = gst_niquadra_rotate_sink_setcaps (pad, parent, caps);
      gst_event_unref (event);
      break;
    }
    default:
      ret = gst_pad_push_event (filter->srcpad, event);
      break;
  }

  return ret;
}

static void
gst_niquadrarotate_init (GstNiQuadraRotate * filter)
{
  filter->sinkpad = gst_pad_new_from_static_template (&sink_factory, "sink");
  gst_pad_set_event_function (filter->sinkpad, gst_niquadra_rotate_sink_event);
  gst_pad_set_chain_function (filter->sinkpad, gst_niquadra_rotate_chain);
  gst_element_add_pad (GST_ELEMENT (filter), filter->sinkpad);

  filter->srcpad = gst_pad_new_from_static_template (&src_factory, "src");
  gst_element_add_pad (GST_ELEMENT (filter), filter->srcpad);

  filter->downstream_card = -1;
  filter->extra_frames = 0;
  filter->skip_filter = FALSE;
}

static void
gst_niquadrarotate_dispose (GObject * obj)
{
  GstNiQuadraRotate *filter = GST_NIQUADRAROTATE (obj);
  if (filter->fillcolor_str)
    g_free (filter->fillcolor_str);
  if (filter->api_dst_frame.data.frame.p_buffer) {
    ni_frame_buffer_free (&filter->api_dst_frame.data.frame);
  }
  if (filter->initialized) {
    if (filter->api_ctx.session_id != NI_INVALID_SESSION_ID) {
      GST_DEBUG_OBJECT (filter, "libxcoder rotate free context\n");
      ni_device_session_close (&filter->api_ctx, 1, NI_DEVICE_TYPE_SCALER);
    }
    ni_session_context_t *p_ctx = &filter->api_ctx;
    if (p_ctx) {
      if (p_ctx->device_handle != NI_INVALID_DEVICE_HANDLE) {
        ni_device_close (p_ctx->device_handle);
        p_ctx->device_handle = NI_INVALID_DEVICE_HANDLE;
      }
      if (p_ctx->blk_io_handle != NI_INVALID_DEVICE_HANDLE) {
        ni_device_close (p_ctx->blk_io_handle);
        p_ctx->blk_io_handle = NI_INVALID_DEVICE_HANDLE;
      }
    }
    ni_device_session_context_clear (&filter->api_ctx);
  }

  G_OBJECT_CLASS (gst_niquadrarotate_parent_class)->dispose (obj);
}

static GstFlowReturn
gst_niquadra_rotate_chain (GstPad * pad, GstObject * parent, GstBuffer * inbuf)
{
  GstNiQuadraRotate *filter = GST_NIQUADRAROTATE (parent);
  GstBuffer *outbuf = NULL;
  GstFlowReturn flow_ret = GST_FLOW_OK;
  int ret = -1;
  niFrameSurface1_t *new_frame_surface;
  ni_session_data_io_t *p_session_data = &filter->api_dst_frame;
  int gc620_pixfmt;
  ni_pix_fmt_t niPixFmt;
  ni_frame_config_t input_frame_config = { 0 };
  int rgba_color = 255;         /* black opaque */
  int aligned_picture_width, rotated_picture_width, rotated_picture_height;

  if (filter->skip_filter) {
    return gst_pad_push (filter->srcpad, inbuf);
  }

  memset (p_session_data, 0, sizeof (ni_session_data_io_t));

  GstNiHWFrameMeta *hwFrameMeta =
      (GstNiHWFrameMeta *) gst_buffer_get_meta (inbuf,
      GST_NI_HWFRAME_META_API_TYPE);
  if (hwFrameMeta == NULL) {
    GST_ERROR_OBJECT (filter,
        "Impossible to convert between the formats supported by the filter");
    if (inbuf) {
      gst_buffer_unref (inbuf);
    }
    return GST_FLOW_ERROR;
  }

  niFrameSurface1_t *frame_surface;
  frame_surface = hwFrameMeta->p_frame_ctx->ni_surface;

  if (!filter->initialized) {
    if (filter->format == GST_VIDEO_FORMAT_BGRP) {
      GST_ERROR_OBJECT (filter, "format BGRP not supported");
      gst_buffer_unref (inbuf);
      return GST_FLOW_ERROR;
    }
    if (!strcmp (filter->fillcolor_str, "none")) {
      filter->fillcolor_enable = false;
    } else if (gst_parse_color (filter->fillcolor, filter->fillcolor_str,
            -1) >= 0) {
      filter->fillcolor_enable = true;
    } else {
      GST_ERROR_OBJECT (filter,
          "Don't support the color input %s", filter->fillcolor_str);
      gst_buffer_unref (inbuf);
      return GST_FLOW_ERROR;
    }

    ni_device_session_context_init (&filter->api_ctx);
    filter->api_ctx.session_id = NI_INVALID_SESSION_ID;
    filter->api_ctx.device_handle = NI_INVALID_DEVICE_HANDLE;
    filter->api_ctx.blk_io_handle = NI_INVALID_DEVICE_HANDLE;
    filter->api_ctx.hw_id = hwFrameMeta->p_frame_ctx->dev_idx;
    filter->api_ctx.device_type = NI_DEVICE_TYPE_SCALER;
    filter->api_ctx.scaler_operation = NI_SCALER_OPCODE_ROTATE;
    filter->api_ctx.keep_alive_timeout = filter->keep_alive_timeout;
    filter->api_ctx.isP2P = filter->is_p2p;

    ret = ni_device_session_open (&filter->api_ctx, NI_DEVICE_TYPE_SCALER);
    if (ret < 0) {
      GST_ERROR_OBJECT (filter,
          "Can't open device session on card %d", filter->api_ctx.hw_id);
      gst_buffer_unref (inbuf);
      ni_device_session_close (&filter->api_ctx, 1, NI_DEVICE_TYPE_SCALER);
      ni_device_session_context_clear (&filter->api_ctx);
      return GST_FLOW_ERROR;
    } else {
      GST_DEBUG_OBJECT (filter,
          "XCoder %s.%d (inst: %d) opened successfully",
          filter->api_ctx.dev_xcoder_name, filter->api_ctx.hw_id,
          filter->api_ctx.session_id);
    }
    memset (&filter->api_dst_frame, 0, sizeof (ni_session_data_io_t));

    int options;
    options = NI_SCALER_FLAG_IO | NI_SCALER_FLAG_PC;
    if (filter->api_ctx.isP2P) {
      options |= NI_SCALER_FLAG_P2;
    }
    int pool_size = 4;
    if (filter->api_ctx.isP2P) {
      pool_size = 1;
    } else {
      // If the downstream element is on a different card,
      // Allocate extra frames suggested by the downstream element
      if (filter->api_ctx.hw_id != filter->downstream_card) {
        pool_size += filter->extra_frames;
        GST_INFO_OBJECT (filter,
            "Increase frame pool by %d", filter->extra_frames);
      }
    }

    niPixFmt = convertGstVideoFormatToNIPix (filter->format);
    gc620_pixfmt = convertNIPixToGC620Format (niPixFmt);

    /* Allocate a pool of frames by the scaler */
    /* *INDENT-OFF* */
    ret = ni_device_alloc_frame (&filter->api_ctx,
        NI_ALIGN (filter->out_width, 2),
        NI_ALIGN (filter->out_height, 2),
        gc620_pixfmt,
        options,
        0,                      // rec width
        0,                      // rec height
        0,                      // rec X pos
        0,                      // rec Y pos
        pool_size,              // rgba color/pool size
        0,                      // frame index
        NI_DEVICE_TYPE_SCALER);
    /* *INDENT-ON* */
    if (ret < 0) {
      GST_ERROR_OBJECT (filter, "Init frame pool error");
      ni_device_session_context_clear (&filter->api_ctx);
      gst_buffer_unref (inbuf);
      return GST_FLOW_ERROR;
    }
    filter->initialized = 1;
  }

  ret = ni_frame_buffer_alloc_hwenc (&p_session_data->data.frame,
      filter->in_width, filter->in_height, 0);
  if (ret != NI_RETCODE_SUCCESS) {
    GST_ERROR_OBJECT (filter,
        "ni rotate filter frame buffer alloc hwenc failed with %d", ret);
    gst_buffer_unref (inbuf);
    return GST_FLOW_ERROR;
  }

  niPixFmt = convertGstVideoFormatToNIPix (filter->format);
  gc620_pixfmt = convertNIPixToGC620Format (niPixFmt);

  input_frame_config.picture_format = gc620_pixfmt;

  input_frame_config.rgba_color = frame_surface->ui32nodeAddress;
  input_frame_config.frame_index = frame_surface->ui16FrameIdx;

  aligned_picture_width =
      NI_ALIGN (filter->in_width, BUFFER_WIDTH_PIXEL_ALIGNMENT);

  if (filter->angle == 0) {
    input_frame_config.picture_width = filter->in_width;
    input_frame_config.picture_height = filter->in_height;

    input_frame_config.rectangle_width =
        ni_min (filter->out_width, filter->in_width);
    input_frame_config.rectangle_height =
        ni_min (filter->out_height, filter->in_height);

    rotated_picture_width = filter->in_width;
    rotated_picture_height = filter->in_height;
  } else if ((filter->angle == -270) || (filter->angle == 90)) {
    input_frame_config.orientation = 1;
    input_frame_config.picture_width = aligned_picture_width;
    input_frame_config.picture_height = filter->in_height;

    input_frame_config.rectangle_width =
        ni_min (filter->out_width, filter->in_height);
    input_frame_config.rectangle_height =
        ni_min (filter->out_height, filter->in_width);

    rotated_picture_width = filter->in_height;
    rotated_picture_height = aligned_picture_width;
  } else if ((filter->angle == -180) || (filter->angle == 180)) {
    input_frame_config.orientation = 2;
    input_frame_config.picture_width = aligned_picture_width;
    input_frame_config.picture_height = filter->in_height;

    input_frame_config.rectangle_width =
        ni_min (filter->out_width, filter->in_width);
    input_frame_config.rectangle_height =
        ni_min (filter->out_height, filter->in_height);

    rotated_picture_width = aligned_picture_width;
    rotated_picture_height = filter->in_height;
  } else if ((filter->angle == -90) || (filter->angle == 270)) {
    input_frame_config.orientation = 3;
    input_frame_config.picture_width = aligned_picture_width;
    input_frame_config.picture_height = filter->in_height;

    input_frame_config.rectangle_width =
        ni_min (filter->out_width, filter->in_height);
    input_frame_config.rectangle_height =
        ni_min (filter->out_height, filter->in_width);

    rotated_picture_width = filter->in_height;
    rotated_picture_height = aligned_picture_width;
  } else {
    GST_ERROR_OBJECT (filter,
        "ni rotate filter does not support rotation of %d radians",
        filter->angle);
    gst_buffer_unref (inbuf);
    return GST_FLOW_ERROR;
  }

  input_frame_config.rectangle_x =
      (rotated_picture_width > input_frame_config.rectangle_width) ?
      (rotated_picture_width / 2) -
      (input_frame_config.rectangle_width / 2) : 0;
  input_frame_config.rectangle_y = (rotated_picture_height >
      input_frame_config.rectangle_height) ? (rotated_picture_height / 2) -
      (input_frame_config.rectangle_height / 2) : 0;
  if (aligned_picture_width - filter->in_width) {
    switch (input_frame_config.orientation) {
      case 1:                  // 90°
        input_frame_config.rectangle_y =
            (filter->in_width > input_frame_config.rectangle_height) ?
            (filter->in_width / 2) -
            (input_frame_config.rectangle_height / 2) : 0;
        break;
      case 2:                  // 180°
        input_frame_config.rectangle_x =
            aligned_picture_width - filter->in_width +
            ((filter->in_width > input_frame_config.rectangle_width) ?
            (filter->in_width / 2) -
            (input_frame_config.rectangle_width / 2) : 0);
        break;
      case 3:                  // 270°
        input_frame_config.rectangle_y =
            aligned_picture_width - filter->in_width +
            ((filter->in_width > input_frame_config.rectangle_height) ?
            (filter->in_width / 2) -
            (input_frame_config.rectangle_height / 2) : 0);
        break;
    }
  }

  ret = ni_device_config_frame (&filter->api_ctx, &input_frame_config);
  if (ret != NI_RETCODE_SUCCESS) {
    GST_ERROR_OBJECT (filter,
        "ni rotate filter device config input frame failed with %d", ret);
    gst_buffer_unref (inbuf);
    return GST_FLOW_ERROR;
  }

  if (filter->fillcolor_enable) {
    rgba_color = (filter->fillcolor[3] << 24) |
        (filter->fillcolor[0] << 16) |
        (filter->fillcolor[1] << 8) | filter->fillcolor[2];
  }

  ni_frame_config_t *pconfig = &filter->output_frame_config;
  pconfig->picture_width = filter->out_width;
  pconfig->picture_height = filter->out_height;
  pconfig->rectangle_width = input_frame_config.rectangle_width;
  pconfig->rectangle_height = input_frame_config.rectangle_height;
  pconfig->rectangle_x = (pconfig->picture_width > pconfig->rectangle_width) ?
      (pconfig->picture_width / 2) - (pconfig->rectangle_width / 2) : 0;
  pconfig->rectangle_y = (pconfig->picture_height > pconfig->rectangle_height) ?
      (pconfig->picture_height / 2) - (pconfig->rectangle_height / 2) : 0;
  pconfig->rgba_color = rgba_color;

  /* *INDENT-OFF* */
  ret = ni_device_alloc_frame (&filter->api_ctx,
      pconfig->picture_width,
      pconfig->picture_height,
      gc620_pixfmt,
      NI_SCALER_FLAG_IO,
      pconfig->rectangle_width,
      pconfig->rectangle_height,
      pconfig->rectangle_x,
      pconfig->rectangle_y,
      pconfig->rgba_color,
      -1,
      NI_DEVICE_TYPE_SCALER);
  /* *INDENT-ON* */
  if (ret != NI_RETCODE_SUCCESS) {
    GST_ERROR_OBJECT (filter,
        "ni rotate filter device alloc output frame failed with %d", ret);
    gst_buffer_unref (inbuf);
    return GST_FLOW_ERROR;
  }

  /* Set the new frame index */
  ret = ni_device_session_read_hwdesc (&filter->api_ctx, p_session_data,
      NI_DEVICE_TYPE_SCALER);
  if (ret != NI_RETCODE_SUCCESS) {
    GST_ERROR_OBJECT (filter, "Can't acquire output frame %d", ret);
    gst_buffer_unref (inbuf);
    return GST_FLOW_ERROR;
  }

  new_frame_surface =
      (niFrameSurface1_t *) p_session_data->data.frame.p_data[3];
  new_frame_surface->dma_buf_fd = 0;
  new_frame_surface->ui32nodeAddress = 0;
  new_frame_surface->ui16width = filter->out_width;
  new_frame_surface->ui16height = filter->out_height;

  ni_set_bit_depth_and_encoding_type (&new_frame_surface->bit_depth,
      &new_frame_surface->encoding_type, niPixFmt);

  GstNiFrameContext *hwFrame =
      gst_ni_hw_frame_context_new (&filter->api_ctx, new_frame_surface,
      filter->api_ctx.hw_id);

  outbuf = gst_buffer_new_and_alloc (0);

  gst_buffer_add_ni_hwframe_meta (outbuf, hwFrame);

  gst_ni_hw_frame_context_unref (hwFrame);

  ni_frame_buffer_free (&p_session_data->data.frame);

  outbuf = gst_buffer_make_writable (outbuf);
  gst_buffer_copy_into (outbuf, inbuf,
      GST_BUFFER_COPY_TIMESTAMPS | GST_BUFFER_COPY_META, 0, -1);
  gst_buffer_unref (inbuf);

  flow_ret = gst_pad_push (filter->srcpad, outbuf);
  return flow_ret;
}

static void
gst_niquadrarotate_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstNiQuadraRotate *self;

  g_return_if_fail (GST_IS_NIQUADRAROTATE (object));
  self = GST_NIQUADRAROTATE (object);

  switch (prop_id) {
    case PROP_W:
      self->out_width = g_value_get_int (value);
      break;
    case PROP_H:
      self->out_height = g_value_get_int (value);
      break;
    case PROP_P2P:
      self->is_p2p = g_value_get_boolean (value);
      break;
    case PROP_COLOR:
      g_free (self->fillcolor_str);
      self->fillcolor_str = g_strdup (g_value_get_string (value));
      break;
    case GST_NIQUADRA_DEC_PROP_TIMEOUT:
      self->keep_alive_timeout = g_value_get_uint (value);
      break;
    case PROP_ANGLE:
      self->angle = g_value_get_int (value);
      break;
    case PROP_AUTO_SKIP:
      self->auto_skip = g_value_get_boolean (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (self, prop_id, pspec);
  }
}

static void
gst_niquadrarotate_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec)
{
  GstNiQuadraRotate *self;

  g_return_if_fail (GST_IS_NIQUADRAROTATE (object));
  self = GST_NIQUADRAROTATE (object);

  switch (prop_id) {
    case PROP_W:
      GST_OBJECT_LOCK (self);
      g_value_set_int (value, self->out_width);
      GST_OBJECT_UNLOCK (self);
      break;
    case PROP_H:
      GST_OBJECT_LOCK (self);
      g_value_set_int (value, self->out_height);
      GST_OBJECT_UNLOCK (self);
      break;
    case PROP_P2P:
      GST_OBJECT_LOCK (self);
      g_value_set_boolean (value, self->is_p2p);
      GST_OBJECT_UNLOCK (self);
      break;
    case PROP_COLOR:
      GST_OBJECT_LOCK (self);
      g_value_take_string (value, g_strdup (self->fillcolor_str));
      GST_OBJECT_UNLOCK (self);
      break;
    case GST_NIQUADRA_DEC_PROP_TIMEOUT:
      GST_OBJECT_LOCK (self);
      g_value_set_uint (value, self->keep_alive_timeout);
      GST_OBJECT_UNLOCK (self);
      break;
    case PROP_ANGLE:
      GST_OBJECT_LOCK (self);
      g_value_set_int (value, self->angle);
      GST_OBJECT_UNLOCK (self);
      break;
    case PROP_AUTO_SKIP:
      GST_OBJECT_LOCK (self);
      g_value_set_boolean (value, self->auto_skip);
      GST_OBJECT_UNLOCK (self);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (self, prop_id, pspec);
  }
}

static void
gst_niquadrarotate_class_init (GstNiQuadraRotateClass * klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;
  GstElementClass *element_class = GST_ELEMENT_CLASS (klass);

  gobject_class->set_property = gst_niquadrarotate_set_property;
  gobject_class->get_property = gst_niquadrarotate_get_property;

  g_object_class_install_property (gobject_class, PROP_H,
      g_param_spec_int ("h", "H",
          "set output height of the rotate", 0, G_MAXINT,
          0, G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS |
          GST_PARAM_MUTABLE_PLAYING | GST_PARAM_CONTROLLABLE));
  g_object_class_install_property (gobject_class, PROP_W,
      g_param_spec_int ("w", "W",
          "set output width of the rotate", 0, G_MAXINT,
          0, G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS |
          GST_PARAM_MUTABLE_PLAYING | GST_PARAM_CONTROLLABLE));
  g_object_class_install_property (gobject_class, PROP_P2P,
      g_param_spec_boolean ("is-p2p", "Is-P2P",
          "set p2p mode of the rotate", FALSE,
          G_PARAM_READWRITE | G_PARAM_STATIC_NAME |
          GST_PARAM_MUTABLE_PLAYING | GST_PARAM_CONTROLLABLE));
  g_object_class_install_property (gobject_class, PROP_COLOR,
      g_param_spec_string ("color", "Color",
          "Set color of the rotate", "black",
          G_PARAM_CONSTRUCT | G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));
  g_object_class_install_property (gobject_class, GST_NIQUADRA_DEC_PROP_TIMEOUT,
      g_param_spec_uint ("keep-alive-timeout", "TIMEOUT",
          "Specify a custom session keep alive timeout in seconds. Defult is 3.",
          NI_MIN_KEEP_ALIVE_TIMEOUT, NI_MAX_KEEP_ALIVE_TIMEOUT,
          NI_DEFAULT_KEEP_ALIVE_TIMEOUT,
          G_PARAM_CONSTRUCT | G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));
  g_object_class_install_property (gobject_class, PROP_ANGLE,
      g_param_spec_int ("angle", "Angle", "set angle of the rotate", G_MININT,
          G_MAXINT, 0,
          G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS | GST_PARAM_MUTABLE_PLAYING
          | GST_PARAM_CONTROLLABLE));
  g_object_class_install_property (gobject_class, PROP_AUTO_SKIP,
      g_param_spec_boolean ("auto-skip", "Auto-Skip",
          "skip the rotate filter when input and output of this filter are the same",
          FALSE,
          G_PARAM_READWRITE | G_PARAM_STATIC_NAME | GST_PARAM_MUTABLE_PLAYING |
          GST_PARAM_CONTROLLABLE));

  gst_element_class_add_static_pad_template (element_class, &src_factory);
  gst_element_class_add_static_pad_template (element_class, &sink_factory);

  gst_element_class_set_static_metadata (element_class,
      "NETINT Quadra ROTATE filter", "Filter/Effect/Video/NIRotate",
      "Rotate Netint Quadra", "Minglong Zhang<Minglong.Zhang@netint.cn>");

  gobject_class->dispose = gst_niquadrarotate_dispose;
}

gboolean
gst_niquadrarotate_register (GstPlugin * plugin)
{
  return gst_element_register (plugin, "niquadrarotate",
      GST_RANK_NONE, GST_TYPE_NIQUADRAROTATE);
}
