/*
 * Copyright (c) 2010 Nicolas George
 * Copyright (c) 2011 Stefano Sabatini
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * @file
 * API example for ocr and vqe and transcode
 * @example ni_ocr_vqe_transcoder.c
 *
 * @added by cube.sun@netint.ca
 * use multiple threads to run ocr and vqe and transcode
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>

#include <sys/time.h>

#include <libavcodec/avcodec.h>
#include <libavformat/avformat.h>
#include <libavfilter/buffersink.h>
#include <libavfilter/buffersrc.h>
#include <libswscale/swscale.h>
#include <libavutil/imgutils.h>
#include <libavutil/opt.h>
#include <libavutil/time.h>
#include <libavutil/hwcontext.h>
#include <pthread.h>
#include <stdint.h>

#include "ni_list_head.h"
#include "netint_network.h"
#include "ocr_postprocess.h"
#include "vqe_postprocess.h"

#include "ni_log.h"

#define NI_WAIT_AI_FRAME_NUM 2
#define MAX_BOX_NUMBER 8

static AVFormatContext *ifmt_ctx;
static AVCodecContext *dec_ctx;
static AVCodecContext *enc_ctx;
AVFilterContext *buffersink_ctx;
AVFilterContext *buffersrc_ctx;
AVFilterGraph *filter_graph;
static int video_stream_index = -1;
int init_enc = 0;
int frame_num = 0,end = 0,encode_num = 0;
int64_t timer_start;

struct frame {
    struct ni_list_head list;
    AVFrame frm;
    int eof;
};
struct codec_worker {
    bool is_picture;

    const char *input_file;
    const char *output_file;
    const char *decoder_name;
    const char *decoder_params;
    const char *encoder_name;
    const char *encoder_params;
    const char *filter_desc;

    int devid;
    int bitrate;

    pthread_t pthtimeid;

    bool enable_ocr;
    pthread_t ocr_pthid;
    const char *ocr_output;
    int ocr_select_frame;
    struct ni_list_head ocr_frm_list;
    pthread_mutex_t ocr_frm_lock;
    pthread_cond_t  ocr_frm_cond;
    int ocr_exit;

    bool enable_vqe;
    pthread_t vqe_pthid;
    const char *vqe_output;
    int vqe_select_frame;
    struct ni_list_head vqe_frm_list;
    pthread_mutex_t vqe_frm_lock;
    pthread_cond_t  vqe_frm_cond;
    int window_size;
    int vqe_gap;
    int vqe_exit;

    pthread_mutex_t ai_lock;
};

struct codec_worker *codec_workers = NULL;

struct ocr_network {
    NiNetworkContext *network_ctx;
    OCRModelCtx *model_ctx;
    OCRModel *model;
};

struct ocr_network *ocr_network;

struct vqe_network {
    NiNetworkContext *network_ctx;
    VQEModelCtx *model_ctx;
    VQEModel *model;
};

struct vqe_network *vqe_network;

NI_LIST_HEAD(ocr_network_frame_free_list);
NI_LIST_HEAD(ocr_network_frame_busy_list);
NI_LIST_HEAD(vqe_network_frame_free_list);
NI_LIST_HEAD(vqe_network_frame_busy_list);

struct network_frame {
    NiNetworkFrame frame;
    struct ni_list_head list;
    int eof;
};

struct network_frame *ocr_network_frames;
struct network_frame *vqe_network_frames;

struct crop_box *area_box;

int box_num = 0;

float alert_threshold = 0.8;

static int refind (const char *a, char b)
{
    int i, len = strlen(a);
    for (i = len - 1; i >= 0; i--) {
        if (a[i] == b) return i;
    }
    return -1;
}

static int check_netint_frame(struct crop_box *frame, int width, int height) {
    // keep area_box stay in frame
    if ((frame->x + frame->w <= width) && (frame->y + frame->h <= height)) {
        return 1;
    }
    if (frame->x > width || frame->y > height) {
        return -1;
    }
    if (frame->x + frame->w > width) {
        if (width - frame->w >= 0) {
            frame->x = width - frame->w;
        } else {
            return -1;
        }
    }
    if (frame->y + frame->h > height) {
        if (height - frame->h >= 0) {
            frame->y = height - frame->h;
        } else {
            return -1;
        }
    }
    return 1;
}

static int resize_crop(struct crop_box *frame, int width, int height) {
    if (frame->w == 0 || frame->h == 0) {
        return 1;
    }
    float ratio = (float)frame->h * 10 / frame->w;
    if (0.47 <= ratio && ratio <= 1) {
        return 1;
    }
    if (ratio < 0.47) {
        return -1;
    }
    // add padding
    int left = width - frame->x;
    int perfect_width = frame->h * 32 / 3;
    if (left >= perfect_width) {
        frame->w = perfect_width;
    } else {
        frame->w = left;
        ratio = (float)frame->h * 10 / frame->w;
        if (ratio > 1.88) {
            return -1;
        }
    }
    return 1;
}

static int init_ocr_network_frames(struct ocr_network *ocr_network,
        int model_width, int model_height, int model_format)
{
    int i, ret;

    ni_init_list_head(&ocr_network_frame_free_list);
    ni_init_list_head(&ocr_network_frame_busy_list);

    ocr_network_frames = (struct network_frame *)calloc(NUM_NETWORK_FRAME,
            sizeof(struct network_frame));
    if (ocr_network_frames == NULL) {
        return -1;
    }

    for (i = 0; i < NUM_NETWORK_FRAME; i++) {
        NiNetworkFrame *frame = &ocr_network_frames[i].frame;
        frame->scale_width = model_width;
        frame->scale_height = model_height;
        frame->scale_format = model_format;
        ni_init_list_head(&ocr_network_frames[i].list);
        ni_list_add_tail(&ocr_network_frames[i].list, &ocr_network_frame_free_list);
        ret = ni_ai_packet_buffer_alloc(&frame->api_packet.data.packet,
            &ocr_network->network_ctx->network_data);
        if (ret != NI_RETCODE_SUCCESS) {
            av_log(NULL, AV_LOG_ERROR, "failed to allocate packet\n");
            return NIERROR(ENOMEM);
        }
        ret = ni_frame_buffer_alloc_hwenc(&frame->api_frame.data.frame,
                                          frame->scale_width,
                                          frame->scale_height, 0);
        if (ret != NI_RETCODE_SUCCESS) {
            av_log(NULL, AV_LOG_ERROR, "failed to allocate frame\n");
            return NIERROR(ENOMEM);
        }
    }

    return 0;
}

static void free_ocr_network_frames()
{
    int i;
    if (!ni_list_empty(&ocr_network_frame_busy_list)) {
        av_log(NULL, AV_LOG_ERROR, "Network frames are still in use\n");
    }

    for (i = 0; i < NUM_NETWORK_FRAME; i++) {
        NiNetworkFrame *frame = &ocr_network_frames[i].frame;
        ni_frame_buffer_free(&frame->api_frame.data.frame);
        ni_packet_buffer_free(&frame->api_packet.data.packet);
    }
    if (ocr_network_frames != NULL) {
        free(ocr_network_frames);
        ocr_network_frames = NULL;
    }
}

static void destroy_ocr_network(struct ocr_network *network)
{
    if (network && network->model) {
        network->model->destroy_model(network->model_ctx);;
        free(network->model_ctx);
    }
}

static int init_vqe_network_frames(struct vqe_network *vqe_network,
        int model_width, int model_height, int model_format)
{
    int i, ret;

    ni_init_list_head(&vqe_network_frame_free_list);
    ni_init_list_head(&vqe_network_frame_busy_list);

    vqe_network_frames = (struct network_frame *)calloc(NUM_NETWORK_FRAME,
            sizeof(struct network_frame));
    if (vqe_network_frames == NULL) {
        return -1;
    }

    for (i = 0; i < NUM_NETWORK_FRAME; i++) {
        NiNetworkFrame *frame = &vqe_network_frames[i].frame;
        frame->scale_width = model_width;
        frame->scale_height = model_height;
        frame->scale_format = model_format;
        ni_init_list_head(&vqe_network_frames[i].list);
        ni_list_add_tail(&vqe_network_frames[i].list, &vqe_network_frame_free_list);
        ret = ni_ai_packet_buffer_alloc(&frame->api_packet.data.packet,
            &vqe_network->network_ctx->network_data);
        if (ret != NI_RETCODE_SUCCESS) {
            av_log(NULL, AV_LOG_ERROR, "failed to allocate packet\n");
            return NIERROR(ENOMEM);
        }
        ret = ni_frame_buffer_alloc_hwenc(&frame->api_frame.data.frame,
                                          frame->scale_width,
                                          frame->scale_height, 0);
        if (ret != NI_RETCODE_SUCCESS) {
            av_log(NULL, AV_LOG_ERROR, "failed to allocate frame\n");
            return NIERROR(ENOMEM);
        }
    }

    return 0;
}

static void free_vqe_network_frames()
{
    int i;
    if (!ni_list_empty(&vqe_network_frame_busy_list)) {
        av_log(NULL, AV_LOG_ERROR, "Network frames are still in use\n");
    }

    for (i = 0; i < NUM_NETWORK_FRAME; i++) {
        NiNetworkFrame *frame = &vqe_network_frames[i].frame;
        ni_frame_buffer_free(&frame->api_frame.data.frame);
        ni_packet_buffer_free(&frame->api_packet.data.packet);
    }

    if (vqe_network_frames != NULL) {
        free(vqe_network_frames);
        vqe_network_frames = NULL;
    }
}

static void destroy_vqe_network(struct vqe_network *network)
{
    if (network && network->model) {
        network->model->destroy_model(network->model_ctx);;
        free(network->model_ctx);
    }
}

/* Specific to a model */
static int create_ocr_network(struct ocr_network *network,
        int model_width, int model_height, const char *character_file)
{
    int ret = 0;
    OCRModel *model;
    OCRModelCtx *model_ctx;
    ni_network_data_t *network_data = &network->network_ctx->network_data;

    model_ctx = (OCRModelCtx *)calloc(1, sizeof(OCRModelCtx));
    if (model_ctx == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate ocr model\n");
        return NIERROR(ENOMEM);
    }
    network->model_ctx = model_ctx;

    model = &ocr;

    ret = model->create_model(model_ctx, network_data, model_width, model_height, character_file);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to initialize ocr model\n");
        goto out;
    }
    network->model = model;

    return ret;

out:
    if (model_ctx) {
      free(model_ctx);
      network->model_ctx = NULL;
    }

    return ret;
}

/* Specific to a model */
static int create_vqe_network(struct vqe_network *network,
        float alert_threshold, int window_size, int model_width, int model_height)
{
    int ret = 0;
    VQEModel *model;
    VQEModelCtx *model_ctx;
    ni_network_data_t *network_data = &network->network_ctx->network_data;

    model_ctx = (VQEModelCtx *)calloc(1, sizeof(VQEModelCtx));
    if (model_ctx == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate vqe model\n");
        return NIERROR(ENOMEM);
    }
    network->model_ctx = model_ctx;

    model = &vqe;

    ret = model->create_model(model_ctx, network_data,
            alert_threshold, window_size, model_width, model_height);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to initialize vqe model\n");
        goto out;
    }
    network->model = model;

    return ret;

out:
    if (model_ctx) {
      free(model_ctx);
      network->model_ctx = NULL;
    }

    return ret;
}


static int init_ocr_network_struct(int model_width, int model_height,
        int model_format, const char *nbg_file, const char *character_file)
{
    int ret;

    ocr_network = (struct ocr_network *)calloc(1, sizeof(struct ocr_network));
    if (ocr_network == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate ocr_network\n");
        return -1;
    }

    ret = ni_alloc_network_context(&ocr_network->network_ctx, true,
            0, 30, model_format, model_width, model_height, nbg_file);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate network context\n");
        goto fail_out;
    }

    ret = create_ocr_network(ocr_network, model_width, model_height, character_file);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to create ocr network\n");
        goto fail_out;
    }

    ret = init_ocr_network_frames(ocr_network, model_width, model_height, model_format);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to initialize network_frames\n");
        goto fail_out;
    }

    return 0;
fail_out:
    destroy_ocr_network(ocr_network);
    ni_cleanup_network_context(ocr_network->network_ctx, true);
    free(ocr_network);
    ocr_network = NULL;

    free_ocr_network_frames();
    return -1;
}

static int init_vqe_network_struct(int model_width, int model_height,
        int model_format, const char *nbg_file, int window_size)
{
    int ret;

    vqe_network = (struct vqe_network *)calloc(1, sizeof(struct vqe_network));
    if (vqe_network == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate vqe_network\n");
        return -1;
    }

    ret = ni_alloc_network_context(&vqe_network->network_ctx, true,
            0, 30, model_format, model_width, model_height, nbg_file);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate network context\n");
        goto fail_out;
    }

    ret = create_vqe_network(vqe_network, alert_threshold, window_size,
            model_width, model_height);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to create vqe network\n");
        goto fail_out;
    }

    ret = init_vqe_network_frames(vqe_network, model_width, model_height,
            model_format);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to initialize network_frames\n");
        goto fail_out;
    }

    return 0;
fail_out:
    destroy_vqe_network(vqe_network);
    ni_cleanup_network_context(vqe_network->network_ctx, true);
    free(vqe_network);
    vqe_network = NULL;

    free_vqe_network_frames();
    return -1;
}

static void free_network_struct()
{
    if (codec_workers->enable_ocr) {
        if (ocr_network) {
            destroy_ocr_network(ocr_network);
            ni_cleanup_network_context(ocr_network->network_ctx, true);
            free(ocr_network);
            ocr_network = NULL;
        }
        free_ocr_network_frames();
    }

    if (codec_workers->enable_vqe) {
        if (vqe_network) {
            destroy_vqe_network(vqe_network);
            ni_cleanup_network_context(vqe_network->network_ctx, true);
            free(vqe_network);
            vqe_network = NULL;
        }
        free_vqe_network_frames();
    }
}

static int ni_read_ocr(struct ocr_network *ocr_network, FILE *fp, int frame_count)
{
    int ret;

    ret = ocr_network->model->ni_get_detections(ocr_network->model_ctx);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to get ocr detection.\n");
        return ret;
    }
    char *txt = ocr_network->model_ctx->det_res->txt;
    float prob = ocr_network->model_ctx->det_res->prob;

    int cur_frame = 0;
    cur_frame = frame_count*codec_workers->ocr_select_frame;

    char result[512] = { 0 };
    snprintf(result, sizeof(result), "Predicts of frame%d, The pred txt is: \"%s\" The pred prob is: %f\n", cur_frame , txt, prob);
    if (fp) {
        fwrite(result, strlen(result), 1, fp);
    } else {
        av_log(NULL, AV_LOG_ERROR, "failed to get opened file.\n");
    }

    return ret;
}

static int ni_read_vqe(struct vqe_network *vqe_network, FILE *fp, int aiframe_number)
{
    int ret;

    ret = vqe_network->model->ni_get_detections(vqe_network->model_ctx);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to get vqe detection.\n");
        return ret;
    }

    char *status = vqe_network->model_ctx->det_res->txt;
    float conf = vqe_network->model_ctx->det_res->conf;
    if (status && fp && aiframe_number % codec_workers->window_size == 0) {
        char result[512] = { 0 };
        int cur_frame = aiframe_number / codec_workers->window_size * codec_workers->vqe_select_frame;
        snprintf(result, sizeof(result), "Predicts of frame%d Video Quality Estimation result %s, conf %f\n", cur_frame, status, conf);
        fwrite(result, strlen(result), 1, fp);
    } else if (!fp){
        av_log(NULL, AV_LOG_ERROR, "failed to open file.\n");
        return -1;
    }

    return ret;
}

static int open_input_file() {
    int i, ret;
    AVCodec *dec;
    if ((ret = avformat_open_input(&ifmt_ctx, codec_workers->input_file, NULL, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot open input file\n");
        return ret;
    }
    if ((ret = avformat_find_stream_info(ifmt_ctx, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot find stream information\n");
        return ret;
    }

    for (i = 0; i < ifmt_ctx->nb_streams; i++) {
        if (ifmt_ctx->streams[i]->codecpar->codec_type == AVMEDIA_TYPE_VIDEO) {
            video_stream_index = i;
            break;
        }
    }
    if (video_stream_index < 0) {
        av_log(NULL, AV_LOG_ERROR, "can not find video stream\n");
        return video_stream_index;
    }

    dec = avcodec_find_decoder_by_name(codec_workers->decoder_name);
    if (dec) {
        if (ifmt_ctx->streams[video_stream_index]->codecpar->codec_id != dec->id) {
            av_log(NULL, AV_LOG_ERROR, "codec does not match with stream id\n");
            return AVERROR_DECODER_NOT_FOUND;
        }
    }

    /* create decoding context */
    dec_ctx = avcodec_alloc_context3(dec);
    if (!dec_ctx){
        av_log(NULL, AV_LOG_ERROR, "Failed to allocate the decoder context for stream\n");
        return AVERROR(ENOMEM);
    }
    avcodec_parameters_to_context(dec_ctx, ifmt_ctx->streams[video_stream_index]->codecpar);
    if(codec_workers->devid >= 0)
    {
        char str_devid[4] = {0};
        snprintf(str_devid, sizeof(str_devid), "%d", codec_workers->devid);
        av_opt_set(dec_ctx->priv_data, "dec", str_devid, 0);
    }
    if(codec_workers->decoder_params)
        av_opt_set(dec_ctx->priv_data, "xcoder-params", codec_workers->decoder_params, 0);
    if (dec_ctx->codec_type == AVMEDIA_TYPE_VIDEO)
        dec_ctx->framerate = av_guess_frame_rate(ifmt_ctx, ifmt_ctx->streams[video_stream_index], NULL);

    ret = avcodec_parameters_from_context(ifmt_ctx->streams[video_stream_index]->codecpar, dec_ctx);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to copy decoder parameters from codec context for stream #");
        return ret;
    }

    /* init the video decoder */
    if ((ret = avcodec_open2(dec_ctx, dec, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot open video decoder\n");
        return ret;
    }
    return 0;
}

static int open_output_file(AVFrame *frame)
{
    int ret = 0;
    AVCodec *enc;

    // Find video encoder codec selected
    enc = avcodec_find_encoder_by_name(codec_workers->encoder_name);
    if (!enc)
    {
        av_log(NULL, AV_LOG_ERROR,"Codec '%s' not found\n", codec_workers->encoder_name);
        return AVERROR_ENCODER_NOT_FOUND;
    }

    // Allocate codec context for encoding
    enc_ctx = avcodec_alloc_context3(enc);
    if (!enc_ctx)
    {
        av_log(NULL, AV_LOG_ERROR, "Could not allocate video codec context\n");
        return AVERROR(ENOMEM);
    }

    if (codec_workers->bitrate)
    {
        enc_ctx->bit_rate = codec_workers->bitrate;
    }
    enc_ctx->height = frame->height;
    enc_ctx->width = frame->width;
    enc_ctx->sample_aspect_ratio = frame->sample_aspect_ratio;
    enc_ctx->pix_fmt = frame->format;
    if (frame->hw_frames_ctx)
    {
        AVHWFramesContext* pAVHFWCtx = (AVHWFramesContext *)frame->hw_frames_ctx->data;
        enc_ctx->sw_pix_fmt = pAVHFWCtx->sw_format;
    }

    /* video time_base can be set to whatever is handy and supported by encoder */
    enc_ctx->time_base = av_inv_q(dec_ctx->framerate);

    av_log(NULL, AV_LOG_INFO, "enc_ctx->sw_pix_fmt %d, pix_fmt %d\n",
        enc_ctx->sw_pix_fmt, enc_ctx->pix_fmt);

    if ((enc->id == AV_CODEC_ID_H264) || (enc->id == AV_CODEC_ID_H265) ||
       (enc->id == AV_CODEC_ID_MJPEG) || (enc->id == AV_CODEC_ID_AV1))
    {
        if(codec_workers->encoder_params)
        {
            av_opt_set(enc_ctx->priv_data, "xcoder-params", codec_workers->encoder_params, 0);
        }
        if(codec_workers->devid >= 0)
        {
            char str_devid[4] = {0};
            snprintf(str_devid, sizeof(str_devid), "%d", codec_workers->devid);
            av_opt_set(enc_ctx->priv_data, "enc", str_devid, 0);
        }
    }
    else
    {
        av_log(NULL, AV_LOG_ERROR, "codec id %d not supported.\n", enc->id);
        ret = -1;
        return ret;
    }

    // Open encoder
    ret = avcodec_open2(enc_ctx, enc, NULL);
    if (ret < 0)
    {
        av_log(NULL, AV_LOG_ERROR, "Could not open enc\n");
        return ret;
    }
    return 0;
}

static int encode_frame(AVFrame *frame, AVPacket *pkt, FILE *outfile)
{
    int ret;
    if (!init_enc && frame)
    {
        if((ret = open_output_file(frame)) < 0)
        {
            av_log(NULL, AV_LOG_ERROR, "init encoder failed ret = %d\n", ret);
            return ret;
        }
        init_enc = 1;
    }

    int attempt_cnt = 1;
    int re_send = 0;
    static int send_cnt = 0;
    static int recv_cnt = 0;

    // Send frame to encoder. If frame=NULL, this signals end of stream
    if (frame)
    {
        frame->pict_type = AV_PICTURE_TYPE_NONE;
        av_log(NULL, AV_LOG_DEBUG, "Send %d frame %3"PRId64", type=%d\n", ++send_cnt, frame->pts, frame->pict_type);
    } else
    {
        av_log(NULL, AV_LOG_DEBUG,"Send done, flush encoder\n");
    }

resend:
    ret = avcodec_send_frame(enc_ctx, frame);
    // Encoder is too busy to receive frame, try sending frame again
    if (ret == AVERROR(EAGAIN))
    {
        av_log(NULL, AV_LOG_DEBUG, "Resending frame. Try #%d.\n", ++attempt_cnt);
        ret = 0;
        re_send = 1;
    }
    // Frame send issue occured
    else if (ret < 0)
    {
        av_log(NULL, AV_LOG_ERROR, "Error sending frame for encoding. RC=%d\n", AVUNERROR(ret));
        return ret;
    }

    /* Run this loop if a frame was sucessfully sent during avcodec_send_frame() OR
        no frame was sent because this function is in flushing mode OR
        a frame was sucessfully received by avcodec_receive_packet() */
    while (ret >= 0 || ! frame )
    {
        av_log(NULL, AV_LOG_DEBUG, "Receiving packet\n");
        ret = avcodec_receive_packet(enc_ctx, pkt);
        if (frame == NULL)
            av_log(NULL, AV_LOG_DEBUG, "Draining packet ret=%x, pkt size=%5d\n", ret, pkt->size);

        // No packets in encoder output buffer but not end of stream either
        if (ret == AVERROR(EAGAIN))
        {
            // If function not in flushing mode
            if (frame)
            {
                if (re_send)
                {
                    re_send = 0;
                    goto resend;
                }
                return 0; // Exit this function to send next packet
            }
            else
            {
                continue; // Loop in this function to flush encoder
            }
        }
        // End of stream detected
        else if (ret == AVERROR_EOF)
        {
            av_log(NULL, AV_LOG_DEBUG, "Encoder EOF reached\n");
            return 0;
        }
        // Packet receive error occured
        else if (ret < 0)
        {
            av_log(NULL, AV_LOG_ERROR, "Error during encoding\n");
            return ret;
        }
        // Packet sucessfully received
        av_log(NULL, AV_LOG_DEBUG,"Received packet %d, pts=%"PRId64"\n", ++recv_cnt, pkt->pts);
        encode_num++;

        fwrite(pkt->data, 1, pkt->size, outfile);
        av_packet_unref(pkt);
    }
    return 0;
}

static void print_report(int is_last_report, int64_t timer_start,
        int64_t cur_time, unsigned long frame_number)
{
    static int64_t last_time = -1;
    float t;
    char buf[1024];
    float fps;

    if (!is_last_report) {
        if (last_time == -1) {
            last_time = cur_time;
            return;
        }
        if ((cur_time - last_time) < 500000)
            return;
        last_time = cur_time;
    }

    t = (cur_time - timer_start) / 1000000.0;

    fps = t > 1 ? frame_number / t : 0;

    const char end = is_last_report ? '\n' : '\r';

    buf[0] = '\0';
    snprintf(buf + strlen(buf), sizeof(buf) - strlen(buf), "frame=%5lu fps=%3.*f ",
            frame_number, (fps < 9.95), fps);
    pr_err("%s   %c", buf, end);
    fflush(stderr);
}

static void *printfps(void *data)
{
    while (1) {
        if (end == 3) {
            break;
        }
        usleep(1000);
        print_report(0, timer_start, av_gettime_relative(), encode_num);
    }
    print_report(1, timer_start, av_gettime_relative(), encode_num);
    return (void *)0;
}

static void *detectOCRframe(void *data)
{
    int i, ret = 0;
    struct frame *frame;
    AVFrame *avframe;
    struct network_frame *network_frame;
    int should_exit = 0;
    NiNetworkContext *network_ctx = ocr_network->network_ctx;
    int aiframe_number = 0;
    ni_session_data_io_t ai_input_frame;

    FILE *fp = NULL;
    if (codec_workers->ocr_output){
        fp = fopen(codec_workers->ocr_output, "w");
    } else {
        int pos_head = refind(codec_workers->input_file, '/');
        int pos_tail = refind(codec_workers->input_file, '.');
        char* substr;
        int len = pos_tail - pos_head - 1;
        if (len > 0){
            substr = malloc(len + 1);
            strncpy(substr, codec_workers->input_file+pos_head+1, len);
            substr[len] = '\0';
        } else {
            len = 0;
            substr = "";
        }
        char name[512] = {0};
        snprintf(name, sizeof(name), "%s_ocr_result.txt", substr);
        // char *name = "result/ocr_result.txt";
        fp = fopen(name, "w");
        if (len > 0) {
            free(substr);
        }
        substr = NULL;
    }

    while(!should_exit)
    {
        if (end == 3 && ni_list_empty(&codec_workers->ocr_frm_list) && ni_list_empty(&ocr_network_frame_busy_list))
            break;

        for (i = 0; i < box_num;) {
            pthread_mutex_lock(&codec_workers->ocr_frm_lock);
            if (ni_list_empty(&codec_workers->ocr_frm_list) && ni_list_empty(&ocr_network_frame_busy_list)) {
                pthread_cond_wait(&codec_workers->ocr_frm_cond,
                        &codec_workers->ocr_frm_lock);
            }

            if (!ni_list_empty(&codec_workers->ocr_frm_list) &&
                    !ni_list_empty(&ocr_network_frame_free_list)) {
                frame = ni_list_first_entry(&codec_workers->ocr_frm_list, struct frame, list);
                if (i == box_num - 1) {
                    ni_list_del(&frame->list);
                    if (ni_list_size(&codec_workers->ocr_frm_list) == NI_WAIT_AI_FRAME_NUM) {
                        pthread_cond_signal(&codec_workers->ocr_frm_cond);
                    }
                }
                pthread_mutex_unlock(&codec_workers->ocr_frm_lock);
                avframe = &frame->frm;
                if (frame->eof) {
                    av_log(NULL, AV_LOG_DEBUG, "detect ocr thread get eof\n");
                    should_exit = 1;
                    break;
                }
                network_frame = ni_list_first_entry(&ocr_network_frame_free_list,
                        struct network_frame, list);

                ret = check_netint_frame(&area_box[i], avframe->width, avframe->height);
                if (ret < 0) {
                    av_log(NULL, AV_LOG_ERROR, "Error while check crop area w: %d h: %d x: %d y: %d width %d height %d\n",
                                    area_box[i].w, area_box[i].h, area_box[i].x, area_box[i].y, avframe->width, avframe->height);
                    i++;
                    continue;
                }
                ret = resize_crop(&area_box[i], avframe->width, avframe->height);
                if (ret < 0) {
                    av_log(NULL, AV_LOG_ERROR, "crop area not resize at perfect size, the output may not correct\n");
                }
                if (!frame->eof) {
                    pthread_mutex_lock(&codec_workers->ai_lock);
                    ai_input_frame.data.frame.p_data[3] = avframe->data[3];
                    ret = ni_set_network_input(network_ctx, true, &ai_input_frame,
                            &area_box[i], avframe->width, avframe->height, &network_frame->frame, true);
                    pthread_mutex_unlock(&codec_workers->ai_lock);

                    if (ret != 0) {
                        av_log(NULL, AV_LOG_ERROR, "Error while feeding the ocr filtergraph\n");
                        should_exit = 1;
                        break;
                    }
                }
                network_frame->eof = frame->eof;

                /* del from free list, add to busy list */
                ni_list_del(&network_frame->list);
                ni_list_add_tail(&network_frame->list, &ocr_network_frame_busy_list);

                if (i == box_num - 1) {
                    av_frame_unref(&frame->frm);
                    free(frame);
                    aiframe_number += 1;
                }
                i++;
            } else {
                pthread_mutex_unlock(&codec_workers->ocr_frm_lock);
                break;
            }
        }

        /* need to get tensor until busy list empty */
        while (!ni_list_empty(&ocr_network_frame_busy_list)) {
            network_frame = ni_list_first_entry(&ocr_network_frame_busy_list, struct network_frame, list);
            if (!network_frame->eof) {
                pthread_mutex_lock(&codec_workers->ai_lock);
                /* pull filtered frames from the filtergraph */
                ret = ni_get_network_output(network_ctx, true, &network_frame->frame, false,
                        true, ocr_network->model_ctx->out_tensor);
                pthread_mutex_unlock(&codec_workers->ai_lock);
                if (ret != 0 && ret != NIERROR(EAGAIN)) {
                    av_log(NULL, AV_LOG_ERROR, "Error when getting output");
                    should_exit = 1;
                    break;
                } else if (ret != NIERROR(EAGAIN)) {
                    /* del from busy list, add to free list */
                    ni_list_del(&network_frame->list);
                    ni_list_add_tail(&network_frame->list, &ocr_network_frame_free_list);

                    ni_read_ocr(ocr_network, fp, aiframe_number);
                    //pr_log("AI frame number %d\n", aiframe_number);
                }
            } else {
                ni_list_del(&network_frame->list);
                ni_list_add_tail(&network_frame->list, &ocr_network_frame_free_list);
                should_exit = 1;
            }
        }
    }
    fclose(fp);
    av_log(NULL, AV_LOG_DEBUG, "detectOCR thread finish\n");
    codec_workers->ocr_exit = 1;
    return (void *)0;
}

static void *detectVQEframe(void *data)
{
    int i, ret = 0;
    struct frame *frame;
    AVFrame *avframe;
    struct network_frame *network_frame;
    int should_exit = 0;
    NiNetworkContext *network_ctx = vqe_network->network_ctx;
    int aiframe_number = 0;
    ni_session_data_io_t ai_input_frame;

    FILE *fp = NULL;
    if (codec_workers->vqe_output){
        fp = fopen(codec_workers->vqe_output, "w");
    } else {
        int pos_head = refind(codec_workers->input_file, '/');
        int pos_tail = refind(codec_workers->input_file, '.');
        char* substr;
        int len = pos_tail - pos_head - 1;
        if (len > 0){
            substr = malloc(len + 1);
            strncpy(substr, codec_workers->input_file+pos_head+1, len);
            substr[len] = '\0';
        } else {
            len = 0;
            substr = "";
        }
        char name[512] = {0};
        snprintf(name, sizeof(name), "%s_vqe_result.txt", substr);
        // char *name = "result/vqe_result.txt";
        fp = fopen(name, "w");
        if (len > 0) {
            free(substr);
        }
        substr = NULL;
    }
    while(!should_exit)
    {
        if (end == 3 && ni_list_empty(&codec_workers->vqe_frm_list) && ni_list_empty(&vqe_network_frame_busy_list))
            break;

        for (i = 0; i < 1;) {
            pthread_mutex_lock(&codec_workers->vqe_frm_lock);
            if (ni_list_empty(&codec_workers->vqe_frm_list) && ni_list_empty(&vqe_network_frame_busy_list)) {
                pthread_cond_wait(&codec_workers->vqe_frm_cond,
                        &codec_workers->vqe_frm_lock);
            }

            if (!ni_list_empty(&codec_workers->vqe_frm_list) &&
                    !ni_list_empty(&vqe_network_frame_free_list)) {
                frame = ni_list_first_entry(&codec_workers->vqe_frm_list, struct frame, list);
                ni_list_del(&frame->list);
                if (ni_list_size(&codec_workers->vqe_frm_list) == NI_WAIT_AI_FRAME_NUM) {
                    pthread_cond_signal(&codec_workers->vqe_frm_cond);
                }
                pthread_mutex_unlock(&codec_workers->vqe_frm_lock);
                avframe = &frame->frm;
                if (frame->eof) {
                    av_log(NULL, AV_LOG_DEBUG, "detect vqe thread get eof\n");
                    should_exit = 1;
                    break;
                }

                network_frame = ni_list_first_entry(&vqe_network_frame_free_list,
                        struct network_frame, list);

                if (!frame->eof) {
                    // printf("ni set netint input width %d height %d output width %d height %d\n", avframe->width, avframe->height, network_frame->frame.scale_width, network_frame->frame.scale_height);
                    pthread_mutex_lock(&codec_workers->ai_lock);
                    ai_input_frame.data.frame.p_data[3] = avframe->data[3];
                    ret = ni_set_network_input(network_ctx, true, &ai_input_frame,
                            NULL, avframe->width, avframe->height, &network_frame->frame, true);
                    pthread_mutex_unlock(&codec_workers->ai_lock);

                    if (ret != 0) {
                        av_log(NULL, AV_LOG_ERROR, "Error while feeding the vqe filtergraph\n");
                        should_exit = 1;
                        break;
                    }
                }
                network_frame->eof = frame->eof;
                ni_list_del(&network_frame->list);
                ni_list_add_tail(&network_frame->list, &vqe_network_frame_busy_list);

                av_frame_unref(&frame->frm);
                free(frame);
                aiframe_number += 1;
                i++;
            } else {
                pthread_mutex_unlock(&codec_workers->vqe_frm_lock);
                break;
            }
        }

        /* need to get tensor until busy list empty, ensure only one frame process in ai core,
        if not mutiply thread will block the ai core */
        while (!ni_list_empty(&vqe_network_frame_busy_list)) {
            network_frame = ni_list_first_entry(&vqe_network_frame_busy_list, struct network_frame, list);
            if (!network_frame->eof) {
                pthread_mutex_lock(&codec_workers->ai_lock);
                /* pull filtered frames from the filtergraph */
                ret = ni_get_network_output(network_ctx, true, &network_frame->frame, false,
                        true, vqe_network->model_ctx->out_tensor);
                pthread_mutex_unlock(&codec_workers->ai_lock);

                if (ret != 0 && ret != NIERROR(EAGAIN)) {
                    av_log(NULL, AV_LOG_ERROR, "Error when getting output");
                    should_exit = 1;
                    break;
                } else if (ret != NIERROR(EAGAIN)) {
                    ni_list_del(&network_frame->list);
                    ni_list_add_tail(&network_frame->list, &vqe_network_frame_free_list);

                    ni_read_vqe(vqe_network, fp, aiframe_number);
                    //pr_log("AI frame number %d\n", aiframe_number);
                }
            } else {
                ni_list_del(&network_frame->list);
                ni_list_add_tail(&network_frame->list, &vqe_network_frame_free_list);
                should_exit = 1;
            }
        }
    }
    fclose(fp);
    av_log(NULL, AV_LOG_DEBUG, "detectOCR thread finish\n");
    codec_workers->vqe_exit = 1;
    return (void *)0;
}

static void help_usage(void)
{
    printf("Usage: \n"
            "-i | --input             input file path.\n"
            "-o | --output            output file path.\n"
            "-d | --decoder           decoder name [h264_ni_quadra_dec | h265_ni_quadra_dec | jpeg_ni_quadra_dec].\n"
            "-D | --decoder_params    decoder parameters.\n"
            "-f | --filter_dsc        filter set params.\n"
            "-e | --encode_name       encoder name [h264_ni_quadra_enc | h265_ni_quadra_enc | jpeg_ni_quadra_enc].\n"
            "-E | --encode_params     encoder params.\n"
            "-l | --loglevel          available debug level: warning, info, debug, trace.\n"
            "-n | --devid             device id.\n"
            "-b | --bitrate           set bitrate.\n"
            "-O | --enable_ocr        enable ocr.\n"
            "-V | --enable_vqe        enable vqe.\n"
            "-s | --ocr_nbg_path      ocr model path.\n"
            "-p | --ocr_model_pixel   ocr model pixel format:-p WdithxHeight.\n"
            "-m | --ocr_select_frame  select frame to ocr detection.\n"
            "-c | --character_file    input character library\n"
            "-r | --area_box          crop roi box format:-r w:h:x:y,w:h:x:y.\n"
            "-t | --ocr_output        the path of txt store the detect result of ocr.\n"
            "-S | --vqe_nbg_path      vqe model path.\n"
            "-P | --vqe_model_pixel   vqe model pixel format:-p WdithxHeight.\n"
            "-M | --vqe_select_frame  select frame to vqe detection.\n"
            "-w | --window_size       Video Quality Estimation window size, require window_size < 120 default:20.\n"
            "-T | --vqe_output        the path of txt store the detect result of vqe.\n"
            "-h | --help              print this help information.\n");
}

void setup_loglevel(char *loglevel)
{
    if (loglevel) {
        if (!strcmp(loglevel, "warning")) {
            av_log_set_level(AV_LOG_WARNING);
        } else if (!strcmp(loglevel, "info")) {
            av_log_set_level(AV_LOG_INFO);
        } else if (!strcmp(loglevel, "debug")) {
            av_log_set_level(AV_LOG_DEBUG);
        } else if (!strcmp(loglevel, "trace")) {
            av_log_set_level(AV_LOG_TRACE);
        } else {
            av_log_set_level(AV_LOG_INFO);
        }
    } else {
        av_log_set_level(AV_LOG_INFO);
    }
}

static int init_ocr_shared_thread_data(struct codec_worker *worker)
{
    int ret;

    ret = pthread_mutex_init(&worker->ocr_frm_lock, NULL);
    if (ret) {
        return ret;
    }

    ret = pthread_cond_init(&worker->ocr_frm_cond, NULL);
    if (ret) {
        goto fail_init_frm_cond;
    }

    ni_init_list_head(&worker->ocr_frm_list);

    return 0;

fail_init_frm_cond:
    pthread_mutex_destroy(&worker->ocr_frm_lock);
    return ret;
}

static int init_vqe_shared_thread_data(struct codec_worker *worker)
{
    int ret;

    ret = pthread_mutex_init(&worker->vqe_frm_lock, NULL);
    if (ret) {
        return ret;
    }

    ret = pthread_cond_init(&worker->vqe_frm_cond, NULL);
    if (ret) {
        goto fail_init_frm_cond;
    }

    ni_init_list_head(&worker->vqe_frm_list);

    return 0;

fail_init_frm_cond:
    pthread_mutex_destroy(&worker->vqe_frm_lock);
    return ret;
}

static int init_ai_mutex_thread_data(struct codec_worker *worker) {

    int ret;

    ret = pthread_mutex_init(&worker->ai_lock, NULL);
    if (ret) {
        return ret;
    }
    return 0;
}

static void cleanup_shared_thread_data(struct codec_worker *worker)
{
    if (worker->enable_ocr) {
        if (!ni_list_empty(&worker->ocr_frm_list)) {
            av_log(NULL, AV_LOG_ERROR, "frame list is not empty\n");
        }

        pthread_mutex_destroy(&worker->ocr_frm_lock);
        pthread_cond_destroy(&worker->ocr_frm_cond);
    }

    if (worker->enable_vqe) {
        if (!ni_list_empty(&worker->vqe_frm_list)) {
            av_log(NULL, AV_LOG_ERROR, "frame list is not empty\n");
        }

        pthread_mutex_destroy(&worker->vqe_frm_lock);
        pthread_cond_destroy(&worker->vqe_frm_cond);
    }
}

static void add_to_ocr_list(struct frame *frm, struct codec_worker *worker) {
    pthread_mutex_lock(&worker->ocr_frm_lock);
    if (ni_list_empty(&worker->ocr_frm_list)) {
        ni_list_add_tail(&frm->list, &worker->ocr_frm_list);
        pthread_cond_signal(&worker->ocr_frm_cond);
    } else if (ni_list_size(&worker->ocr_frm_list) == NI_WAIT_AI_FRAME_NUM && !frm->eof) {
        ni_list_add_tail(&frm->list, &worker->ocr_frm_list);
        pthread_cond_wait(&codec_workers->ocr_frm_cond, &codec_workers->ocr_frm_lock);
    } else {
        ni_list_add_tail(&frm->list, &worker->ocr_frm_list);
    }
    pthread_mutex_unlock(&worker->ocr_frm_lock);
}

static void add_to_vqe_list(struct frame *frm, struct codec_worker *worker) {
    pthread_mutex_lock(&worker->vqe_frm_lock);
    if (ni_list_empty(&worker->vqe_frm_list)) {
        ni_list_add_tail(&frm->list, &worker->vqe_frm_list);
        pthread_cond_signal(&worker->vqe_frm_cond);
    } else if (ni_list_size(&worker->vqe_frm_list) == NI_WAIT_AI_FRAME_NUM && !frm->eof) {
        ni_list_add_tail(&frm->list, &worker->vqe_frm_list);
        pthread_cond_wait(&codec_workers->vqe_frm_cond, &codec_workers->vqe_frm_lock);
    } else {
        ni_list_add_tail(&frm->list, &worker->vqe_frm_list);
    }
    pthread_mutex_unlock(&worker->vqe_frm_lock);
}

static int create_ai_thread(struct codec_worker *worker)
{
    int ret;
    pthread_attr_t attr;

    ret = pthread_attr_init(&attr);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "fail to initialize attr: %d.\n", ret);
        return ret;
    }

    ret = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "fail to set attr detachstate: %d.\n", ret);
        goto end;
    }
    if (worker->enable_ocr) {
        ret = pthread_create(&worker->ocr_pthid, NULL, detectOCRframe, 0);
        if (ret != 0) {
            av_log(NULL, AV_LOG_ERROR, "creat thread failed!, %d\n", ret);
            goto end;
        }
    }
    if (worker->enable_vqe) {
        ret = pthread_create(&worker->vqe_pthid, NULL, detectVQEframe, 0);
        if (ret != 0) {
            av_log(NULL, AV_LOG_ERROR, "creat thread failed!, %d\n", ret);
            goto end;
        }
    }

end:
    if (0 != pthread_attr_destroy(&attr)) {
        av_log(NULL, AV_LOG_ERROR, "failed to destroy pthread attr\n");
    }
    return ret;
}

static int destroy_ai_thread(struct codec_worker *worker)
{
    int ret;

    if (worker->enable_ocr) {
        void *ocr_result;
        pthread_cond_signal(&worker->ocr_frm_cond);
        ret = pthread_join(worker->ocr_pthid, &ocr_result);
        if (ret != 0) {
            av_log(NULL, AV_LOG_ERROR, "Failed to join ai thread\n");
        }
        if ((int)((long)ocr_result)) {
            av_log(NULL, AV_LOG_ERROR, "Bad ai thread result: %d\n", (int)((long)ocr_result));
        }
    }
    if (worker->enable_vqe) {
        void *vqe_result;
        pthread_cond_signal(&worker->vqe_frm_cond);
        ret = pthread_join(worker->vqe_pthid, &vqe_result);
        if (ret != 0) {
            av_log(NULL, AV_LOG_ERROR, "Failed to join ai thread\n");
        }
        if ((int)((long)vqe_result)) {
            av_log(NULL, AV_LOG_ERROR, "Bad ai thread result: %d\n", (int)((long)vqe_result));
        }
    }

    return 0;
}

static int create_fps_thread(struct codec_worker *worker)
{
    int ret;
    pthread_attr_t attr;

    ret = pthread_attr_init(&attr);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "fail to initialize attr: %d.\n", ret);
        return ret;
    }

    ret = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "fail to set attr detachstate: %d.\n", ret);
        goto end;
    }

    ret = pthread_create(&worker->pthtimeid,NULL,printfps,0);
    if (ret !=0) {
        av_log(NULL, AV_LOG_ERROR, "creat thread failed!\n");
        goto end;
    }
end:
    if (0 != pthread_attr_destroy(&attr)) {
        av_log(NULL, AV_LOG_ERROR, "failed to destroy pthread attr\n");
    }
    return ret;
}

static int destroy_fps_thread(struct codec_worker *worker)
{
    int ret;
    void *result;

    ret = pthread_join(worker->pthtimeid, &result);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to join fps thread\n");
    }
    if ((int)((long)result)) {
        av_log(NULL, AV_LOG_ERROR, "Bad fps thread result: %d\n", (int)((long)result));
    }

    return ret;
}

int main(int argc, char **argv)
{
    int ret, i;
    AVPacket packet;
    AVPacket *enc_packet;
    AVFrame *frame;
    FILE *outfile = NULL;

    const char *ocr_nbg_file = "network_binary_ocr_512.nb";
    int ocr_model_width = 512;
    int ocr_model_height = 48;
    int ocr_model_format = GC620_RGB888_PLANAR;
    const char *character_file = "input/characters.txt";

    const char *vqe_nbg_file = "network_binary_vqe.nb";
    int vqe_model_width = 256;
    int vqe_model_height = 256;
    int vqe_model_format = GC620_RGB888_PLANAR;

    char *loglevel = NULL;
    int opt;
    int opt_index;
    bool need_encode = true;
    codec_workers = (struct codec_worker *)calloc(1, sizeof(struct codec_worker));
    if (codec_workers == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate codec workers.\n");
        return EXIT_FAILURE;
    }
    area_box = (struct crop_box *)calloc(MAX_BOX_NUMBER, sizeof(struct crop_box));
    if (area_box == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate roi box.\n");
        return EXIT_FAILURE;
    }
    for (i = 0; i < MAX_BOX_NUMBER; i++) {
        area_box[i].w = 0;
        area_box[i].h = 0;
        area_box[i].x = 0;
        area_box[i].y = 0;
    }

    codec_workers->is_picture = false;
    codec_workers->devid = 0;
    codec_workers->bitrate = 0;
    codec_workers->enable_ocr = false;
    codec_workers->enable_vqe = false;
    codec_workers->ocr_exit = 0;
    codec_workers->vqe_exit = 0;
    codec_workers->ocr_select_frame = 1;
    codec_workers->vqe_select_frame = 1;
    codec_workers->window_size = 20;

    // const char *opt_string = "n:x:i:d:o:c:e:p:v:b:m:f:s:k:r:h";
    const char *opt_string = "i:o:d:D:f:e:E:l:n:b:OVs:p:m:c:r:t:S:P:M:w:T:h";
    static struct option long_options[] = {
        {"input",            required_argument, NULL, 'i'},
        {"output",           required_argument, NULL, 'o'},
        {"decoder",          required_argument, NULL, 'd'},
        {"decoder-params",   required_argument, NULL, 'D'},
        {"filter_desc",      required_argument, NULL, 'f'},
        {"encode_name",      required_argument, NULL, 'e'},
        {"encode_params",    required_argument, NULL, 'E'},
        {"loglevel",         required_argument, NULL, 'l'},
        {"devid",            required_argument, NULL, 'n'},
        {"bitrate",          required_argument, NULL, 'b'},
        {"enable_ocr",       no_argument,       NULL, 'O'},
        {"enable_vqe",       no_argument,       NULL, 'V'},
        {"ocr_nbg_path",     required_argument, NULL, 's'},
        {"ocr_model_pixel",  required_argument, NULL, 'p'},
        {"ocr_select_frame", required_argument, NULL, 'm'},
        {"char_file",        required_argument, NULL, 'c'},
        {"area_box",         required_argument, NULL, 'r'},
        {"ocr_output",       required_argument, NULL, 't'},
        {"vqe_nbg_path",     required_argument, NULL, 'S'},
        {"vqe_model_pixel",  required_argument, NULL, 'P'},
        {"vqe_select_frame", required_argument, NULL, 'M'},
        {"window_size",      required_argument, NULL, 'w'},
        {"vqe_output",       required_argument, NULL, 'T'},
        {"help",             no_argument,       NULL, 'h'},
        { NULL,              0,                 NULL,  0 },
    };

    while ((opt = getopt_long(argc, argv, opt_string, long_options, &opt_index)) != -1) {
        switch (opt) {
            case 'i':
                codec_workers->input_file = optarg;
                break;
            case 'o':
                codec_workers->output_file = optarg;
                break;
            case 'd':
                codec_workers->decoder_name = optarg;
                if (0 == strcmp("jpeg_ni_quadra_dec", codec_workers->decoder_name)) {
                    codec_workers->is_picture = true;
                }
                break;
            case 'D':
                codec_workers->decoder_params = optarg;
                break;
            case 'f':
                codec_workers->filter_desc = optarg;
                break;
            case 'e':
                codec_workers->encoder_name = optarg;
                break;
            case 'E':
                codec_workers->encoder_params = optarg;
                break;
            case 'l':
                loglevel = optarg;
                break;
            case 'n':
                codec_workers->devid = atoi(optarg);
                break;
            case 'b':
                codec_workers->bitrate = atoi(optarg);
                break;
            case 'O':
                codec_workers->enable_ocr = true;
                break;
            case 'V':
                codec_workers->enable_vqe = true;
                break;
            case 's':
                ocr_nbg_file = optarg;
                break;
            case 'p':
            {
                char *ch;
                ocr_model_width = strtoul(optarg, &ch, 10);
                if (*ch != 'x') {
                    av_log(NULL, AV_LOG_ERROR, "invalid model_pixel\n");
                    return EXIT_FAILURE;
                }
                ocr_model_height = strtoul(ch + 1, NULL, 10);
                break;
            }
            case 'm':
                codec_workers->ocr_select_frame = atoi(optarg);
                break;
            case 'c':
                character_file = optarg;
                break;
            case 'r':
            {
                char *ch = ";";
                for (; *ch; box_num++) {
                    if (box_num) {
                        area_box[box_num].w = strtoul(ch + 1, &ch, 10);
                        area_box[box_num].w = area_box[box_num].w > 0 ? area_box[box_num].w : 0;
                        if (*ch != ':') {
                            av_log(NULL, AV_LOG_ERROR, "invalid roi box\n");
                            return EXIT_FAILURE;
                        }
                    } else {
                        area_box[box_num].w = strtoul(optarg, &ch, 10);
                        area_box[box_num].w = area_box[box_num].w > 0 ? area_box[box_num].w : 0;
                        if (*ch != ':') {
                            av_log(NULL, AV_LOG_ERROR, "invalid roi box\n");
                            return EXIT_FAILURE;
                        }
                    }

                    area_box[box_num].h = strtoul(ch + 1, &ch, 10);
                    area_box[box_num].h = area_box[box_num].h > 0 ? area_box[box_num].h : 0;
                    if (*ch != ':') {
                        av_log(NULL, AV_LOG_ERROR, "invalid roi box\n");
                        return EXIT_FAILURE;
                    }
                    area_box[box_num].x = strtoul(ch + 1, &ch, 10);
                    area_box[box_num].x = area_box[box_num].x > 0 ? area_box[box_num].x : 0;
                    if (*ch != ':') {
                        av_log(NULL, AV_LOG_ERROR, "invalid roi box\n");
                        return EXIT_FAILURE;
                    }
                    area_box[box_num].y = strtoul(ch + 1, &ch, 10);
                    area_box[box_num].y = area_box[box_num].y > 0 ? area_box[box_num].y : 0;
                    // printf("w:%d h:%d x:%d y:%d ch:%c\n", area_box[box_num].w, area_box[box_num].h, area_box[box_num].x, area_box[box_num].y, *ch);
                }
                break;
            }
            case 't':
            {
                codec_workers->ocr_output = optarg;
                break;
            }
            case 'S':
                vqe_nbg_file = optarg;
                break;
            case 'P':
            {
                char *ch;
                vqe_model_width = strtoul(optarg, &ch, 10);
                if (*ch != 'x') {
                    av_log(NULL, AV_LOG_ERROR, "invalid model_pixel\n");
                    return EXIT_FAILURE;
                }
                vqe_model_height = strtoul(ch + 1, NULL, 10);
                break;
            }
            case 'M':
                codec_workers->vqe_select_frame = atoi(optarg);
                break;
            case 'w':
                if (atoi(optarg) <= 120 && atoi(optarg) > 0) {
                    codec_workers->window_size = atoi(optarg);
                } else {
                    av_log(NULL, AV_LOG_ERROR, "invalid window size\n");
                }
                break;
            case 'T':
            {
                codec_workers->vqe_output = optarg;
                break;
            }
            case 'h':
                help_usage();
                return EXIT_SUCCESS;
            default:
                help_usage();
                return EXIT_FAILURE;
        }
    }

    if (loglevel) {
        setup_loglevel(loglevel);
    }
    // Register all formats, codecs, filters
#if LIBAVFORMAT_VERSION_MAJOR < 58
    av_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif
#if LIBAVCODEC_VERSION_MAJOR < 58
    avcodec_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif
#if LIBAVFILTER_VERSION_MAJOR < 7
    avfilter_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif

    if (!codec_workers->output_file && !codec_workers->encoder_name) {
        need_encode = false;
        av_log(NULL, AV_LOG_ERROR, "output file name and encoder name not specified, "
               "won't do encode in this situation.\n");
    }

    if (codec_workers->input_file == NULL) {
        av_log(NULL, AV_LOG_ERROR, "input file name must be specified.\n");
        return EXIT_FAILURE;
    }

    if (!codec_workers->decoder_name) {
        av_log(NULL, AV_LOG_ERROR, "decoder_name name must be specified.\n");
        return EXIT_FAILURE;
    }

    if (need_encode) {
        if (!codec_workers->output_file) {
            av_log(NULL, AV_LOG_ERROR, "output file name must be specified.\n");
            return EXIT_FAILURE;
        }

        if (!codec_workers->encoder_name) {
            av_log(NULL, AV_LOG_ERROR, "encoder_name name must be specified.\n");
            return EXIT_FAILURE;
        }

        outfile = fopen(codec_workers->output_file,"wb");
        if(!outfile)
        {
            av_log(NULL, AV_LOG_ERROR,"open file fail!\n");
            return EXIT_FAILURE;
        }
    }

    if (codec_workers->enable_ocr) {
        if (access(ocr_nbg_file, R_OK) != 0) {
            av_log(NULL, AV_LOG_ERROR, "cannot find nbg: %s\n", ocr_nbg_file);
            return EXIT_FAILURE;
        }
        if (box_num < 1) {
            av_log(NULL, AV_LOG_ERROR, "box number: %d, cannot get enough box\n", box_num);
            return EXIT_FAILURE;
        }
        if (access(character_file, R_OK) != 0) {
            av_log(NULL, AV_LOG_ERROR, "cannot find character_file: %s\n", character_file);
            return EXIT_FAILURE;
        }
        if(codec_workers->ocr_select_frame <= 0) {
            av_log(NULL, AV_LOG_ERROR, "select frame must be greater than zero.\n");
            return EXIT_FAILURE;
        }
        if (init_ocr_shared_thread_data(codec_workers) != 0) {
            av_log(NULL, AV_LOG_ERROR, "failed to initialize common data\n");
            return EXIT_FAILURE;
        }
    }

    if (codec_workers->enable_vqe) {
        if (access(vqe_nbg_file, R_OK) != 0) {
            av_log(NULL, AV_LOG_ERROR, "cannot find nbg: %s\n", vqe_nbg_file);
            return EXIT_FAILURE;
        }
        if(codec_workers->vqe_select_frame <= 0) {
            av_log(NULL, AV_LOG_ERROR, "select frame must be greater than zero.\n");
            return EXIT_FAILURE;
        } else {
            codec_workers->vqe_gap = codec_workers->vqe_select_frame > codec_workers->window_size ?
                                     codec_workers->vqe_select_frame - codec_workers->window_size : 0;
        }
        if (init_vqe_shared_thread_data(codec_workers) != 0) {
            av_log(NULL, AV_LOG_ERROR, "failed to initialize common data\n");
            return EXIT_FAILURE;
        }
        if (codec_workers->is_picture) codec_workers->window_size = 1;
    }

    if (codec_workers->enable_ocr || codec_workers->enable_vqe) {
        if (init_ai_mutex_thread_data(codec_workers) != 0) {
            av_log(NULL, AV_LOG_ERROR, "failed to initialize common data\n");
            return EXIT_FAILURE;
        }
    }

    if (codec_workers->enable_ocr) {
        ret = init_ocr_network_struct(ocr_model_width, ocr_model_height, ocr_model_format, ocr_nbg_file, character_file);
        if (ret != 0) {
            av_log(NULL, AV_LOG_ERROR, "failed to initialize network struct\n");
            goto end;
        }
    }

    if (codec_workers->enable_vqe) {
        ret = init_vqe_network_struct(vqe_model_width, vqe_model_height, vqe_model_format, vqe_nbg_file, codec_workers->window_size);
        if (ret != 0) {
            av_log(NULL, AV_LOG_ERROR, "failed to initialize network struct\n");
            goto end;
        }
    }

    // Allocate packet for encoder output
    enc_packet = av_packet_alloc();
    frame = av_frame_alloc();
    if (!enc_packet || !frame) {
        av_log(NULL, AV_LOG_ERROR,"Could not allocate pkt frame\n");
        return EXIT_FAILURE;
    }

    if ((ret = open_input_file()) < 0)
        goto end;

    timer_start = av_gettime_relative();

    if (codec_workers->enable_ocr || codec_workers->enable_vqe) {
        if ((ret = create_ai_thread(codec_workers)) != 0) {
            av_log(NULL, AV_LOG_ERROR, "Failed to create ai thread\n");
            goto end;
        }
    }
    if (need_encode) {
        if ((ret = create_fps_thread(codec_workers)) != 0) {
            av_log(NULL, AV_LOG_ERROR, "Failed to create fps thread\n");
            goto stop_ait;
        }
    }

    /* read all packets */
    while (1) {
        if (codec_workers->enable_ocr || codec_workers->enable_vqe) {
            if (!(codec_workers->enable_ocr ^ codec_workers->ocr_exit) && !(codec_workers->enable_vqe ^ codec_workers->vqe_exit)) {
                break;
            }
        }

        if ((ret = av_read_frame(ifmt_ctx, &packet)) < 0) {
            if(end == 2) {
                if (codec_workers->enable_ocr) {
                    struct frame *ocr_frm_eof = (struct frame *)calloc(1, sizeof(struct frame));
                    if (ocr_frm_eof == NULL) {
                        av_log(NULL, AV_LOG_ERROR, "Failed to allcate frame\n");
                        goto stop_fpst;
                    }
                    ocr_frm_eof->eof = 1;
                    av_log(NULL, AV_LOG_DEBUG, "main ocr get eof\n");
                    add_to_ocr_list(ocr_frm_eof, codec_workers);
                }

                if (codec_workers->enable_vqe) {
                    struct frame *vqe_frm_eof = (struct frame *)calloc(1, sizeof(struct frame));
                    if (vqe_frm_eof == NULL) {
                        av_log(NULL, AV_LOG_ERROR, "Failed to allcate frame\n");
                        goto stop_fpst;
                    }
                    vqe_frm_eof->eof = 1;
                    av_log(NULL, AV_LOG_DEBUG, "main vqe get eof\n");
                    add_to_vqe_list(vqe_frm_eof, codec_workers);
                }

                if (need_encode) {
                    ret = encode_frame(NULL, enc_packet, outfile);
                    if (ret != 0) {
                        av_log(NULL, AV_LOG_ERROR, "Error when encode frame: %x\n", ret);
                    }
                } else {
                    ret = 0;
                }

                break;
            }
            end = 1;
        }
        if (packet.stream_index == video_stream_index || end == 1) {
            if(!end) {
                ret = avcodec_send_packet(dec_ctx, &packet);
            }
            else {
                ret = avcodec_send_packet(dec_ctx, NULL);
            }

            if (ret < 0 && ret != AVERROR_EOF) {
                av_log(NULL, AV_LOG_ERROR, "Error while sending a packet to the decoder %d\n", ret);
                break;
            }
            while (ret >= 0 || end == 1) {
                ret = avcodec_receive_frame(dec_ctx, frame);
                if (ret == AVERROR(EAGAIN)) {
                    break;
                }
                else if(ret == AVERROR_EOF){
                    end = 2;
                    break;
                }
                else if (ret < 0) {
                    av_log(NULL, AV_LOG_ERROR, "Error while receiving a frame from the decoder\n");
                    goto stop_fpst;
                }
                if (!codec_workers->is_picture) {
                    frame->pts = frame->best_effort_timestamp;
                    frame_num++;
                }

                if (codec_workers->enable_ocr) {
                    if ((frame_num) % codec_workers->ocr_select_frame == 0 ||
                        codec_workers->is_picture) {
                        struct frame *ocr_frm = (struct frame *)calloc(1, sizeof(struct frame));
                        if (ocr_frm == NULL) {
                            av_log(NULL, AV_LOG_ERROR, "Failed to allcate frame\n");
                            goto stop_fpst;
                        }

                        ret = av_frame_ref(&ocr_frm->frm, frame);
                        if (ret < 0) {
                            free(ocr_frm);
                            av_log(NULL, AV_LOG_ERROR, "copy frame to filter failed\n");
                            goto stop_fpst;
                        }

                        add_to_ocr_list(ocr_frm, codec_workers);
                    }
                }

                if (codec_workers->enable_vqe) {
                    if (frame_num % codec_workers->vqe_select_frame == 0 ||
                        frame_num % codec_workers->vqe_select_frame > codec_workers->vqe_gap ||
                        codec_workers->is_picture) {
                        struct frame *vqe_frm = (struct frame *)calloc(1, sizeof(struct frame));
                        if (vqe_frm == NULL) {
                            av_log(NULL, AV_LOG_ERROR, "Failed to allcate frame\n");
                            goto stop_fpst;
                        }

                        ret = av_frame_ref(&vqe_frm->frm, frame);
                        if (ret < 0) {
                            free(vqe_frm);
                            av_log(NULL, AV_LOG_ERROR, "copy frame to filter failed ret = %d\n", ret);
                            goto stop_fpst;
                        }

                        add_to_vqe_list(vqe_frm, codec_workers);
                    }
                }

                if (need_encode) {
                    ret = encode_frame(frame, enc_packet, outfile);
                    if (ret < 0) {
                        av_log(NULL, AV_LOG_ERROR, "Error when encode frame: %x\n", ret);
                        goto stop_fpst;
                    }
                } else {
                    if (codec_workers->is_picture) {
                        end = 2;
                    }
                    break;
                }
                av_frame_unref(frame);
            }
        }
        av_packet_unref(&packet);
    }

stop_fpst:
    end = 3;
    if (need_encode) {
        destroy_fps_thread(codec_workers);
    }
stop_ait:
    destroy_ai_thread(codec_workers);
end:
    av_frame_free(&frame);
    av_packet_free(&enc_packet);
    avcodec_free_context(&dec_ctx);
    avformat_close_input(&ifmt_ctx);
    if (need_encode) {
        fclose(outfile);
        avcodec_free_context(&enc_ctx);
    }

    cleanup_shared_thread_data(codec_workers);
    free_network_struct();
    free(codec_workers);
    if (ret < 0 && ret != AVERROR_EOF) {
        av_log(NULL, AV_LOG_ERROR, "Error occurred: %s\n", av_err2str(ret));
        exit(1);
    }
    exit(0);
}
