/*
 * Copyright (c) 2010 Nicolas George
 * Copyright (c) 2011 Stefano Sabatini
 * Copyright (C) 2024 NETINT Technologies
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * @file
 * API example for roi
 * @example ni_roi_detect_move_311.c
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>

#include <sys/time.h>

#include <libavcodec/avcodec.h>
#include <libavformat/avformat.h>
#include <libavfilter/buffersink.h>
#include <libavfilter/buffersrc.h>
#include <libswscale/swscale.h>
#include <libavutil/imgutils.h>
#include <libavutil/opt.h>
#include <libavutil/time.h>
#include "libavutil/hwcontext.h"
#include "libavutil/hwcontext_ni_quad.h"
#include <pthread.h>
#include <stdint.h>

#include "ni_list_head.h"
#include "netint_network_311.h"
#include "yolo_postprocess_311.h"
#include "yolo_model.h"

#include "ni_log.h"

#define NI_TRANSCODE_FRAME_NUM 3
#define NI_SAME_CENTER_THRESH 2
#define NI_SAME_BORDER_THRESH 8

bool use_hwframe = false;
struct SwsContext *sw_scale_ctx = NULL;
AVFrame scaled_frame;
FILE *outfile = NULL;
AVFrame *frame;
static AVFormatContext *fmt_ctx;
static AVCodecContext *dec_ctx;
static AVCodecContext *enc_ctx;
static int video_stream_index = -1;
int frame_num = 0, end = 0, encode_num = 0;
int64_t timer_start;
bool stop_fps = false;
int draw_box = 0;
int init_filters = 0;
int init_encoder = 0;

typedef struct _filter_worker
{
    AVFilterContext *buffersink_ctx;
    AVFilterContext *buffersrc_ctx;
    AVFilterGraph *filter_graph;
}filter_worker;

filter_worker *drawbox_filter;
filter_worker *hwdl_filter;
AVFilterContext *drawbox_filter_ctx = NULL;

struct frame_req {
    struct ni_list_head list;
    AVFrame avframe;
    int eof;
};

struct frame_req *pframes = NULL;

// Only count previous 1000 frames time statistics
#define MAX_TIMES_NUMBER 1000

int log_index = 0;

struct codec_worker {
    const char *decoder_name;
    const char *input_file;
    const char *output_file;
    const char *decoder_params;
    const char *encoder_name;
    const char *encoder_params;
    const char *color;
    int yolo_version;
    int devid;
    int bitrate;

    const char *roi_output;

    pthread_t pthid;
    pthread_t pthtimeid;

    int ait_exit;

    struct ni_list_head frm_list;
    pthread_mutex_t frm_lock;
    pthread_cond_t  frm_cond;

    struct ni_list_head free_list;
    pthread_mutex_t free_lock;
    pthread_cond_t free_cond;
};

struct codec_worker *codec_workers = NULL;

struct roi_network {
    NiNetworkContext *network_ctx;
    YoloModelCtx *model_ctx;
    YoloModel *model;
};

struct roi_network *roi_network;

float obj_thresh = 0.25;
float nms_thresh = 0.45;
static char *roi_class[] = {"person", "bicycle", "car", "motorcycle", "airplane", "bus", "train", "truck", "boat",
                            "traffic light", "fire hydrant", "stop sign", "parking meter", "bench", "bird", "cat",
                            "dog", "horse", "sheep", "cow", "elephant", "bear", "zebra", "giraffe", "backpack",
                            "umbrella", "handbag", "tie", "suitcase", "frisbee", "skis", "snowboard", "sports ball",
                            "kite", "baseball bat", "baseball glove", "skateboard", "surfboard", "tennis racket",
                            "bottle", "wine glass", "cup", "fork", "knife", "spoon", "bowl", "banana", "apple",
                            "sandwich", "orange", "broccoli", "carrot", "hot dog", "pizza", "donut", "cake", "chair",
                            "couch", "potted plant", "bed", "dining table", "toilet", "tv", "laptop", "mouse", "remote",
                            "keyboard", "cell phone", "microwave", "oven", "toaster", "sink", "refrigerator", "book",
                            "clock", "vase", "scissors", "teddy bear", "hair drier", "toothbrush"};

//AVRational qp_offset = { 3, 5 };
//AVRational qp_offset = { -3 , 5 };
AVRational qp_offset = { 0 , 0 };

NI_LIST_HEAD(network_frame_free_list);
NI_LIST_HEAD(network_frame_busy_list);

struct network_frame {
    NiNetworkFrame frame;
    struct ni_list_head list;
    struct frame_req *frame_req;
};

struct network_frame *roi_network_frames;

static int refind (const char *a, char b)
{
    int i;
    int len = strlen(a);
    for (i = len - 1; i >= 0; i--) {
        if (a[i] == b) return i;
    }
    return -1;
}

static int init_network_frames(struct roi_network *roi_network,
        int model_width, int model_height, int model_format)
{
    int i, ret;

    ni_init_list_head(&network_frame_free_list);
    ni_init_list_head(&network_frame_busy_list);

    roi_network_frames = (struct network_frame *)calloc(NUM_NETWORK_FRAME,
            sizeof(struct network_frame));
    if (roi_network_frames == NULL) {
        return -1;
    }

    for (i = 0; i < NUM_NETWORK_FRAME; i++) {
        NiNetworkFrame *frame = &roi_network_frames[i].frame;
        frame->scale_width = model_width;
        frame->scale_height = model_height;
        frame->scale_format = model_format;
        ni_init_list_head(&roi_network_frames[i].list);
        ni_list_add_tail(&roi_network_frames[i].list, &network_frame_free_list);
        // malloc ni_session_data_io_t
        ret = ni_quadra_filter_drawtext_prep_frames(&frame->api_packet);
        if (ret < 0)
        {
            av_log(NULL, AV_LOG_ERROR, "cannot alloc packet\n");
            return NIERROR(ENOMEM);
        }
        ret = ni_quadra_filter_drawtext_prep_frames(&frame->api_frame);
        if (ret < 0)
        {
            av_log(NULL, AV_LOG_ERROR, "cannot alloc frame\n");
            return NIERROR(ENOMEM);
        }

        ret = ni_quadra_ai_packet_buffer_alloc(frame->api_packet,
            &roi_network->network_ctx->network_data);
        if (ret != 0) {
            av_log(NULL, AV_LOG_ERROR, "failed to allocate packet\n");
            return NIERROR(ENOMEM);
        }
        ret = ni_quadra_frame_buffer_alloc_hwenc(frame->api_frame,
                                          frame->scale_width,
                                          frame->scale_height, 0);
        if (ret != 0) {
            av_log(NULL, AV_LOG_ERROR, "failed to allocate frame\n");
            return NIERROR(ENOMEM);
        }
    }

    return 0;
}

static void free_network_frames()
{
    int i;
    if (!ni_list_empty(&network_frame_busy_list)) {
        av_log(NULL, AV_LOG_WARNING, "Network frames are still in use\n");
    }

    for (i = 0; i < NUM_NETWORK_FRAME; i++) {
        NiNetworkFrame *frame = &roi_network_frames[i].frame;
        ni_quadra_frame_buffer_free(frame->api_frame);
        free(frame->api_frame);
        ni_quadra_packet_buffer_free(frame->api_packet);
        free(frame->api_packet);
    }
    if (roi_network_frames != NULL) {
        free(roi_network_frames);
        roi_network_frames = NULL;
    }
}

static void destroy_roi_network(struct roi_network *network)
{
    if (network && network->model) {
        network->model->destroy_model(network->model_ctx);;
        free(network->model_ctx);
    }
}

/* Specific to a model */
static int create_roi_network(struct roi_network *network,
        float obj_thresh, float nms_thresh,
        int model_width, int model_height)
{
    int ret = 0;
    YoloModel *model;
    YoloModelCtx *model_ctx;
    ni_quadra_network_data_t *network_data = &network->network_ctx->network_data;

    model_ctx = (YoloModelCtx *)calloc(1, sizeof(YoloModelCtx));
    if (model_ctx == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate yolo model\n");
        return NIERROR(ENOMEM);
    }
    network->model_ctx = model_ctx;

    switch (codec_workers->yolo_version) {
    case YOLOV4:
      model = &yolov4;
      break;
    case YOLOV5:
      model = &yolov5;
      break;
    default:
      av_log(NULL, AV_LOG_ERROR, "no supportted yolo model!!!\n");
      goto out;
    }

    ret = model->create_model(model_ctx, network_data,
            obj_thresh, nms_thresh, model_width, model_height);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to initialize yolov4 model\n");
        goto out;
    }
    network->model = model;

    return ret;

out:
    if (model_ctx) {
      free(model_ctx);
      network->model_ctx = NULL;
    }

    return ret;
}

static int init_network_struct(AVFrame *frame, int model_width, int model_height,
        int model_format, const char *nbg_file)
{
    int ret;
    AVHWFramesContext *pAVHFWCtx;
    AVNIDeviceContext *pAVNIDevCtx;
    int device_handle = 0, blk_io_handle = 0;
    roi_network = (struct roi_network *)calloc(1, sizeof(struct roi_network));
    if (roi_network == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate roi_network\n");
        return -1;
    }
    if (use_hwframe){
        pAVHFWCtx   = (AVHWFramesContext *)frame->hw_frames_ctx->data;
        pAVNIDevCtx = (AVNIDeviceContext *)pAVHFWCtx->device_ctx->hwctx;
        device_handle = pAVNIDevCtx->cards[0];
        blk_io_handle = pAVNIDevCtx->cards[0];
    }

    ret = ni_alloc_network_context(&roi_network->network_ctx, use_hwframe,
            0, 30, model_format, model_width, model_height, nbg_file,
            device_handle, blk_io_handle);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate network context\n");
        goto fail_out;
    }

    ret = create_roi_network(roi_network, obj_thresh, nms_thresh,
            model_width, model_height);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to create roi network\n");
        goto fail_out;
    }

    ret = init_network_frames(roi_network, model_width, model_height,
            model_format);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to initialize network_frames\n");
        goto fail_out;
    }

    return 0;
fail_out:
    destroy_roi_network(roi_network);
    ni_cleanup_network_context(roi_network->network_ctx, use_hwframe);
    free_network_frames();
    free(roi_network);
    roi_network = NULL;
    return -1;
}

static void free_network_struct()
{
    if (roi_network) {
        destroy_roi_network(roi_network);
        ni_cleanup_network_context(roi_network->network_ctx, use_hwframe);
        free(roi_network);
        roi_network = NULL;
    }

    free_network_frames();
}

static int ni_read_roi(struct roi_network *roi_network, AVFrame *out, int frame_count)
{
    AVFrameSideData *sd;
    AVFrameSideData *sd_roi_extra;
    AVRegionOfInterest *roi;
    AVRegionOfInterestNetintExtra *roi_extra;
    struct roi_box *roi_box = NULL;
    int roi_num = 0;
    int reserve_roi_num = 0;
    int ret;
    int i, j;
    int width = out->width;
    int height = out->height;

    ret = roi_network->model->ni_get_boxes(roi_network->model_ctx, width, height, &roi_box, &roi_num);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to get roi.\n");
        return ret;
    }

    if (roi_num == 0) {
        av_log(NULL, AV_LOG_DEBUG, "no roi available\n");
        return 0;
    }
    if (0) {
        pr_err("frame %d roi num %d\n", frame_count, roi_num);
        for (i = 0; i < roi_num; i++) {
            pr_err("frame count %d roi %d: top %d, bottom %d, left %d, right %d, class %d name %s prob %f\n",
                frame_count, i, roi_box[i].top, roi_box[i].bottom, roi_box[i].left,
                roi_box[i].right, roi_box[i].class, roi_class[roi_box[i].class], roi_box[i].prob);
        }
    }
    for (i = 0; i < roi_num; i++) {
        if (roi_box[i].class == 0 || roi_box[i].class == 7) {
            reserve_roi_num++;
        }
    }
    if (reserve_roi_num == 0) {
        av_log(NULL, AV_LOG_DEBUG, "no reserve roi available\n");
        ret = 0;
        goto out;
    }
    sd = av_frame_new_side_data(out, AV_FRAME_DATA_REGIONS_OF_INTEREST,
                                (int)(reserve_roi_num * sizeof(AVRegionOfInterest)));
    sd_roi_extra = av_frame_new_side_data(
        out, AV_FRAME_DATA_NETINT_REGIONS_OF_INTEREST_EXTRA,
        (int)(reserve_roi_num * sizeof(AVRegionOfInterestNetintExtra)));
    if (!sd || !sd_roi_extra) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate roi sidedata\n");
        ret = NIERROR(ENOMEM);
        goto out;
    }

    roi = (AVRegionOfInterest *)sd->data;
    roi_extra = (AVRegionOfInterestNetintExtra *)sd_roi_extra->data;
    for (i = 0, j = 0; i < roi_num; i++) {
        if (roi_box[i].class == 0 || roi_box[i].class == 7) {
            roi[j].self_size = sizeof(*roi);
            roi[j].top       = roi_box[i].top;
            roi[j].bottom    = roi_box[i].bottom;
            roi[j].left      = roi_box[i].left;
            roi[j].right     = roi_box[i].right;
            roi[j].qoffset   = qp_offset;
            roi_extra[j].self_size = sizeof(*roi_extra);
            roi_extra[j].cls       = roi_box[i].class;
            roi_extra[j].prob      = roi_box[i].prob;
            pr_log("roi %d: top %d, bottom %d, left %d, right %d, class %d prob %f qpo %d/%d\n",
                   j, roi[j].top, roi[j].bottom, roi[j].left, roi[j].right,
                   roi_extra[j].cls, roi_extra[j].prob, roi[j].qoffset.num, roi[j].qoffset.den);
            j++;
        }
    }

out:
    free(roi_box);
    return ret;
}

static int open_input_file()
{
    int ret;
    AVCodec *dec;
    if ((ret = avformat_open_input(&fmt_ctx, codec_workers->input_file, NULL, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot open input file\n");
        return ret;
    }

    if ((ret = avformat_find_stream_info(fmt_ctx, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot find stream information\n");
        return ret;
    }

    dec = avcodec_find_decoder_by_name(codec_workers->decoder_name);
    if (dec) {
                if (fmt_ctx->streams[0]->codecpar->codec_id != dec->id) {
                    av_log(NULL, AV_LOG_ERROR, "codec does not match with stream id\n");
                    return AVERROR_DECODER_NOT_FOUND;
                }
            }

    video_stream_index = 0;

    /* create decoding context */
    dec_ctx = avcodec_alloc_context3(dec);
    if (!dec_ctx)
        return AVERROR(ENOMEM);

    avcodec_parameters_to_context(dec_ctx, fmt_ctx->streams[0]->codecpar);

    char str_devid[4] = {0};
    snprintf(str_devid, sizeof(str_devid), "%d", codec_workers->devid);
    av_opt_set(dec_ctx->priv_data, "dec", str_devid, 0);
    if(codec_workers->decoder_params)
        av_opt_set(dec_ctx->priv_data, "xcoder-params",codec_workers->decoder_params, 0);

    if (dec_ctx->codec_type == AVMEDIA_TYPE_VIDEO)
            dec_ctx->framerate = av_guess_frame_rate(fmt_ctx, fmt_ctx->streams[0], NULL);

    ret = avcodec_parameters_from_context(fmt_ctx->streams[0]->codecpar, dec_ctx);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to copy decoder parameters from codec context for stream #");
        return ret;
    }

    /* init the video decoder */
    if ((ret = avcodec_open2(dec_ctx, dec, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot open video decoder\n");
        return ret;
    }
    return 0;
}

static int open_output_file(AVFrame *frame)
{
    int ret = 0;
    AVCodec *enc;

    // Find video encoder codec selected
    enc = avcodec_find_encoder_by_name(codec_workers->encoder_name);
    if (!enc)
    {
        av_log(NULL, AV_LOG_ERROR,"Codec '%s' not found\n", codec_workers->encoder_name);
        return AVERROR_ENCODER_NOT_FOUND;
    }

    // Allocate codec context for encoding
    enc_ctx = avcodec_alloc_context3(enc);
    if (!enc_ctx)
    {
        av_log(NULL, AV_LOG_ERROR, "Could not allocate video codec context\n");
        return AVERROR(ENOMEM);
    }

    if (codec_workers->bitrate)
    {
        enc_ctx->bit_rate = codec_workers->bitrate;
    }
    enc_ctx->height = frame->height;
    enc_ctx->width = frame->width;
    enc_ctx->framerate = dec_ctx->framerate;
    enc_ctx->sample_aspect_ratio = dec_ctx->sample_aspect_ratio;
    enc_ctx->pix_fmt = frame->format;
    if (frame->hw_frames_ctx)
    {
        AVHWFramesContext* pAVHFWCtx = (AVHWFramesContext *)frame->hw_frames_ctx->data;
        enc_ctx->sw_pix_fmt = pAVHFWCtx->sw_format;
    }

    /* video time_base can be set to whatever is handy and supported by encoder */
    enc_ctx->time_base = av_inv_q(dec_ctx->framerate);

    av_log(NULL, AV_LOG_INFO, "dec_ctx->sw_pix_fmt %d, pix_fmt %d\n",
            dec_ctx->sw_pix_fmt, dec_ctx->pix_fmt);

    if ((enc->id == AV_CODEC_ID_H264) || (enc->id == AV_CODEC_ID_H265))
    {
        if(codec_workers->encoder_params)
        {
            av_opt_set(enc_ctx->priv_data, "xcoder-params", codec_workers->encoder_params, 0);
        }
        if(codec_workers->devid >= 0)
        {
            char str_devid[4] = {0};
            snprintf(str_devid, sizeof(str_devid), "%d", codec_workers->devid);
            av_opt_set(enc_ctx->priv_data, "enc", str_devid, 0);
        }
    }
    else
    {
        av_log(NULL, AV_LOG_ERROR, "codec id %d not supported.\n", enc->id);
        ret = -1;
        return ret;
    }

    // Open encoder
    ret = avcodec_open2(enc_ctx, enc, NULL);
    if (ret < 0)
    {
        av_log(NULL, AV_LOG_ERROR, "Could not open enc\n");
        return ret;
    }
    return 0;
}

static int encode_frame(AVFrame *frame, AVPacket *pkt, FILE *outfile)
{
    int ret;
    if (!init_encoder && frame)
    {
        if((ret = open_output_file(frame)) < 0)
        {
            av_log(NULL, AV_LOG_ERROR, "init encoder failed ret = %d\n", ret);
            return ret;
        }
        init_encoder = 1;
    }

    int attempt_cnt = 1;
    int re_send = 0;
    static int send_cnt = 0;
    static int recv_cnt = 0;

    // Send frame to encoder. If frame=NULL, this signals end of stream
    if (frame)
    {
        frame->pict_type = AV_PICTURE_TYPE_NONE;
        av_log(NULL, AV_LOG_DEBUG, "Send %d frame %3"PRId64", type=%d\n", ++send_cnt, frame->pts, frame->pict_type);
    } else
    {
        av_log(NULL, AV_LOG_DEBUG,"Send done, flush encoder\n");
    }

resend:
    ret = avcodec_send_frame(enc_ctx, frame);
    // Encoder is too busy to receive frame, try sending frame again
    if (ret == AVERROR(EAGAIN))
    {
        av_log(NULL, AV_LOG_DEBUG, "Resending frame. Try #%d.\n", ++attempt_cnt);
        ret = 0;
        re_send = 1;
    }
    // Frame send issue occured
    else if (ret < 0)
    {
        av_log(NULL, AV_LOG_ERROR, "Error sending frame for encoding. RC=%d\n", AVUNERROR(ret));
        return ret;
    }

    /* Run this loop if a frame was sucessfully sent during avcodec_send_frame() OR
        no frame was sent because this function is in flushing mode OR
        a frame was sucessfully received by avcodec_receive_packet() */
    while (ret >= 0 || ! frame )
    {
        av_log(NULL, AV_LOG_DEBUG, "Receiving packet\n");
        ret = avcodec_receive_packet(enc_ctx, pkt);
        if (frame == NULL)
            av_log(NULL, AV_LOG_DEBUG, "Draining packet ret=%x, pkt size=%5d\n", ret, pkt->size);

        // No packets in encoder output buffer but not end of stream either
        if (ret == AVERROR(EAGAIN))
        {
            // If function not in flushing mode
            if (frame)
            {
                if (re_send)
                {
                    re_send = 0;
                    goto resend;
                }
                return 0; // Exit this function to send next packet
            }
            else
            {
                continue; // Loop in this function to flush encoder
            }
        }
        // End of stream detected
        else if (ret == AVERROR_EOF)
        {
            av_log(NULL, AV_LOG_DEBUG, "Encoder EOF reached\n");
            return 0;
        }
        // Packet receive error occured
        else if (ret < 0)
        {
            av_log(NULL, AV_LOG_ERROR, "Error during encoding\n");
            return ret;
        }
        // Packet sucessfully received
        av_log(NULL, AV_LOG_DEBUG,"Received packet %d, pts=%"PRId64"\n", ++recv_cnt, pkt->pts);
        encode_num++;

        fwrite(pkt->data, 1, pkt->size, outfile);
        av_packet_unref(pkt);
    }
    return 0;
}

static int init_filter(const char *filters_desc, filter_worker *f, bool hwmode)
{
    char args[512] = { 0 };
    char name[32] = { 0 };
    int i, ret = 0;
    AVFilterInOut *inputs, *outputs, *cur;
    AVBufferSrcParameters *par;
    AVRational time_base = fmt_ctx->streams[video_stream_index]->time_base;
    enum AVPixelFormat input_fmt;
    if (hwmode) {
        input_fmt = dec_ctx->pix_fmt;
    } else {
        input_fmt = dec_ctx->sw_pix_fmt;
    }

    f->filter_graph = avfilter_graph_alloc();
    if (!f->filter_graph) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate filter graph\n");
        ret = AVERROR(ENOMEM);
        goto end;
    }

    ret = avfilter_graph_parse2(f->filter_graph, filters_desc, &inputs, &outputs);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to parse graph\n");
        goto end;
    }

    // link input
    cur = inputs, i = 0;
    snprintf(name, sizeof(name), "in_%d", i);
    snprintf(args, sizeof(args),
            "video_size=%dx%d:pix_fmt=%d:time_base=%d/%d:pixel_aspect=%d/%d:frame_rate=%d/%d",
            dec_ctx->width, dec_ctx->height, input_fmt, time_base.num, time_base.den,
            dec_ctx->sample_aspect_ratio.num, dec_ctx->sample_aspect_ratio.den,dec_ctx->framerate.num,dec_ctx->framerate.den);
    ret = avfilter_graph_create_filter(&f->buffersrc_ctx, avfilter_get_by_name("buffer"), name,
                                    args, NULL, f->filter_graph);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot create buffer source\n");
        goto end;
    }

    // decoder out=hw
    if (hwmode && dec_ctx->hw_frames_ctx != NULL) {
        av_log(NULL, AV_LOG_DEBUG, "hw mode filter\n");
        // Allocate a new AVBufferSrcParameters instance when decoder out=hw
        par = av_buffersrc_parameters_alloc();
        if (!par) {
            av_log(NULL, AV_LOG_ERROR, "cannot allocate hwdl buffersrc parameters\n");
            ret = AVERROR(ENOMEM);
            goto end;
        }
        memset(par, 0, sizeof(*par));
        // set format and hw_frames_ctx to AVBufferSrcParameters when out=hw
        par->format = AV_PIX_FMT_NONE;
        par->hw_frames_ctx = dec_ctx->hw_frames_ctx;
        av_log(NULL, AV_LOG_DEBUG, "hw_frames_ctx %p\n", par->hw_frames_ctx);
        // Initialize the buffersrc filter with the provided parameters
        ret = av_buffersrc_parameters_set(f->buffersrc_ctx, par);
        av_freep(&par);
        if (ret < 0)
            goto end;
    } else { // decoder out=sw
        av_log(NULL, AV_LOG_DEBUG, "sw mode filter\n");
    }

    ret = avfilter_link(f->buffersrc_ctx, 0, cur->filter_ctx, cur->pad_idx);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to link input filter\n");
        goto end;
    }

    cur = outputs, i = 0;
    snprintf(name, sizeof(name), "out_%d", i);
    ret = avfilter_graph_create_filter(&f->buffersink_ctx, avfilter_get_by_name("buffersink"),
                                        name, NULL, NULL, f->filter_graph);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to create output filter: %d\n", i);
        goto end;
    }

    // connect  dst (index i) pads to one of buffer sink (index 0) pad
    ret = avfilter_link(cur->filter_ctx, cur->pad_idx, f->buffersink_ctx, 0);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to link output filter: %d\n", i);
        goto end;
    }

    // configure and validate the filter graph
    ret = avfilter_graph_config(f->filter_graph, NULL);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "%s failed to config graph filter\n",
               __func__);
        goto end;
    } else {
        av_log(NULL, AV_LOG_DEBUG, "%s success config graph filter %s\n",
               __func__, filters_desc);
    }

end:
    avfilter_inout_free(&inputs);
    avfilter_inout_free(&outputs);
    return ret;
}

static void print_report(int is_last_report, int64_t timer_start,
        int64_t cur_time, unsigned long frame_number)
{
    static int64_t last_time = -1;
    float t;
    char buf[1024];
    float fps;

    if (!is_last_report) {
        if (last_time == -1) {
            last_time = cur_time;
            return;
        }
        if ((cur_time - last_time) < 500000)
            return;
        last_time = cur_time;
    }

    t = (cur_time - timer_start) / 1000000.0;

    fps = t > 1 ? frame_number / t : 0;

    const char end = is_last_report ? '\n' : '\r';

    buf[0] = '\0';
    snprintf(buf + strlen(buf), sizeof(buf) - strlen(buf), "frame=%5ld fps=%3.*f ",
            frame_number, (fps < 9.95), fps);
    pr_err("%s   %c", buf, end);
    fflush(stderr);
}

static void *printfps(void *data)
{
    while (1) {
        if (stop_fps == true) {
            break;
        }
        usleep(1000);
        print_report(0, timer_start, av_gettime_relative(), encode_num);
    }
    print_report(1, timer_start, av_gettime_relative(), encode_num);
    return (void *)0;
}

static int ni_recreate_ai_frame(void *ni_frame, AVFrame *frame)
{
    uint8_t **p_ni_data = ni_quadra_get_frame_data(ni_frame);
    uint8_t *p_data = p_ni_data[0];

    av_log(NULL, AV_LOG_DEBUG,
           "linesize %d/%d/%d, data %p/%p/%p, pixel %dx%d\n",
           frame->linesize[0], frame->linesize[1], frame->linesize[2],
           frame->data[0], frame->data[1], frame->data[2], frame->width,
           frame->height);

    if (frame->format == AV_PIX_FMT_RGB24) {
        /* RGB24 -> BGRP */
        uint8_t *r_data = p_data + frame->width * frame->height * 2;
        uint8_t *g_data = p_data + frame->width * frame->height * 1;
        uint8_t *b_data = p_data + frame->width * frame->height * 0;
        uint8_t *fdata  = frame->data[0];
        int x, y;

        av_log(NULL, AV_LOG_DEBUG,
               "%s(): rgb24 to bgrp, pix %dx%d, linesize %d\n", __func__,
               frame->width, frame->height, frame->linesize[0]);

        for (y = 0; y < frame->height; y++) {
            for (x = 0; x < frame->width; x++) {
                int fpos  = y * frame->linesize[0];
                int ppos  = y * frame->width;
                uint8_t r = fdata[fpos + x * 3 + 0];
                uint8_t g = fdata[fpos + x * 3 + 1];
                uint8_t b = fdata[fpos + x * 3 + 2];

                r_data[ppos + x] = r;
                g_data[ppos + x] = g;
                b_data[ppos + x] = b;
            }
        }
    } else {
        av_log(NULL, AV_LOG_ERROR, "cannot recreate frame: invalid frame format\n");
    }
    return 0;
}

static int generate_ai_frame(struct roi_network *roi_network, void *ai_frame,
        AVFrame *avframe, bool hwframe)
{
    int ret = 0;
    NiNetworkContext *network_ctx = roi_network->network_ctx;
    int retval;

    if (hwframe == false) {
        ret = sws_scale(sw_scale_ctx, (const uint8_t * const *)avframe->data,
                avframe->linesize, 0, avframe->height, scaled_frame.data, scaled_frame.linesize);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "cannot do sw scale: inframe data 0x%lx, linesize %d/%d/%d/%d, height %d\n",
                    (unsigned long)avframe->data, avframe->linesize[0], avframe->linesize[1],
                    avframe->linesize[2], avframe->linesize[3], avframe->height);
            return ret;
        }
        retval = ni_quadra_ai_frame_buffer_alloc(ai_frame, &network_ctx->network_data);
        if (retval != 0) {
            av_log(NULL, AV_LOG_ERROR, "cannot allocate sw ai frame buffer\n");
            return NIERROR(ENOMEM);
        }
        ret = ni_recreate_ai_frame(ai_frame, &scaled_frame);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "cannot recreate sw ai frame\n");
            return ret;
        }
    } else {
        uint8_t** p_data = ni_quadra_get_frame_data(ai_frame);
        p_data[3] = avframe->data[3];
    }

    return ret;
}

static int dlhw_frame(AVFrame *hwframe, AVFrame **filt_frame)
{
    int ret;
    AVFrame *output;

    output = av_frame_alloc();
    if (!output) {
        av_log(NULL, AV_LOG_ERROR, "cannot allocate output filter frame\n");
        return NIERROR(ENOMEM);
    }

    ret = av_buffersrc_add_frame_flags(hwdl_filter->buffersrc_ctx, hwframe, AV_BUFFERSRC_FLAG_KEEP_REF);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "cannot add frame to hwdl buffer src\n");
        goto fail_out;
    }

    do {
        ret = av_buffersink_get_frame(hwdl_filter->buffersink_ctx, output);
        if (ret == AVERROR(EAGAIN)) {
            continue;
        } else if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "cannot get frame from hwdl buffer sink\n");
            goto fail_out;
        } else {
            break;
        }
    } while (1);

    *filt_frame = output;
    return 0;

fail_out:
    av_frame_free(&output);
    return ret;
}

static int check_movement(AVRegionOfInterest *last_roi, int last_roi_count, AVRegionOfInterest cur_roi,
                          AVRegionOfInterestNetintExtra *last_roi_extra, AVRegionOfInterestNetintExtra cur_roi_extra)
{
    int i;
    for (i = 0; i < last_roi_count; i++) {
        if (last_roi_extra[i].cls == cur_roi_extra.cls) {
            if (abs(cur_roi.left - last_roi[i].left) < NI_SAME_BORDER_THRESH &&
                abs(cur_roi.right - last_roi[i].right) < NI_SAME_BORDER_THRESH &&
                abs(cur_roi.top - last_roi[i].top) < NI_SAME_BORDER_THRESH &&
                abs(cur_roi.bottom - last_roi[i].bottom) < NI_SAME_BORDER_THRESH &&
                abs((last_roi[i].left + last_roi[i].right) / 2 - (cur_roi.left + cur_roi.right) / 2) < NI_SAME_CENTER_THRESH &&
                abs((last_roi[i].top + last_roi[i].bottom) / 2 - (cur_roi.top + cur_roi.bottom) / 2) < NI_SAME_CENTER_THRESH) {
                    return 1;
                }
        }
    }
    return 0;
}

static int draw_roi_box(AVFrame *inframe, AVFrame **outframe, AVRegionOfInterest roi, AVRegionOfInterestNetintExtra roi_extra)
{
    char drawbox_option[32];
    char *color;
    int n, ret;
    int x, y, w, h;
    float prob;

    int class = roi_extra.cls;
    if (class == 0) {
        color = "Blue";
    } else {
        color = "Red";
    }
    prob = roi_extra.prob;
    x = roi.left;
    y = roi.top;
    w = roi.right - roi.left;
    h = roi.bottom - roi.top;

    av_log(NULL, AV_LOG_DEBUG, "x %d, y %d, w %d, h %d calss %s prob %f\n",
            x, y, w, h, roi_class[class], prob);

    n = snprintf(drawbox_option, sizeof(drawbox_option), "%d", x); drawbox_option[n] = '\0';
    av_opt_set(drawbox_filter_ctx->priv, "x", drawbox_option, 0);

    n = snprintf(drawbox_option, sizeof(drawbox_option), "%d", y); drawbox_option[n] = '\0';
    av_opt_set(drawbox_filter_ctx->priv, "y", drawbox_option, 0);

    n = snprintf(drawbox_option, sizeof(drawbox_option), "%d", w); drawbox_option[n] = '\0';
    av_opt_set(drawbox_filter_ctx->priv, "w", drawbox_option, 0);

    n = snprintf(drawbox_option, sizeof(drawbox_option), "%d", h); drawbox_option[n] = '\0';
    av_opt_set(drawbox_filter_ctx->priv, "h", drawbox_option, 0);

    n = snprintf(drawbox_option, sizeof(drawbox_option), "%s", color); drawbox_option[n] = '\0';
    av_opt_set(drawbox_filter_ctx->priv, "color", drawbox_option, 0);

    ret = avfilter_graph_send_command(drawbox_filter->filter_graph, "drawbox", "color", color, NULL, 0, 0);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "cannot send drawbox filter command\n");
        return ret;
    }

    ret = av_buffersrc_add_frame_flags(drawbox_filter->buffersrc_ctx, inframe, AV_BUFFERSRC_FLAG_KEEP_REF);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "cannot add frame to drawbox buffer src %d\n", ret);
        return ret;
    }

    do {
        ret = av_buffersink_get_frame(drawbox_filter->buffersink_ctx, *outframe);
        if (ret == AVERROR(EAGAIN)) {
            continue;
        } else if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "cannot get frame from drawbox buffer sink %d\n",ret);
            return ret;
        } else {
            break;
        }
    } while (1);
    return 0;
}

static int process_roi(AVFrame *frame, AVFrame **filt_frame,
                       AVRegionOfInterest **last_roi,
                       AVRegionOfInterestNetintExtra **last_roi_extra,
                       int *last_roi_count,
                       FILE *fp)
{
    int i, num, ret;
    AVFrameSideData *sd = av_frame_get_side_data(frame, AV_FRAME_DATA_REGIONS_OF_INTEREST);
    AVFrameSideData *sd_roi_extra = av_frame_get_side_data(
        frame, AV_FRAME_DATA_NETINT_REGIONS_OF_INTEREST_EXTRA);
    AVFrame *input = NULL;
    AVFrame *output = NULL;
    static int filt_cnt = 0;
    int detected = 0;

    av_log(NULL, AV_LOG_DEBUG, "Filt %d frame pts %3"PRId64"\n",++filt_cnt, frame->pts);

    ret = dlhw_frame(frame, &input);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "cannot download hwframe\n");
        goto fail_out;
    }

    if (!sd || !sd_roi_extra || sd->size == 0 || sd_roi_extra->size == 0) {
        *filt_frame = input;
        if (*filt_frame == NULL) {
            av_log(NULL, AV_LOG_ERROR, "cannot clone frame\n");
            return NIERROR(ENOMEM);
        }
        av_log(NULL, AV_LOG_INFO, "no roi area in frame %d\n", filt_cnt);
        return 0;
    }

    AVRegionOfInterest *roi = (AVRegionOfInterest *)sd->data;
    AVRegionOfInterestNetintExtra *roi_extra =
    (AVRegionOfInterestNetintExtra *)sd_roi_extra->data;
    if ((sd->size % roi->self_size) ||
        (sd_roi_extra->size % roi_extra->self_size) ||
        (sd->size / roi->self_size !=
         sd_roi_extra->size / roi_extra->self_size)) {
        av_log(NULL, AV_LOG_ERROR, "invalid roi side data\n");
        ret = NIERROR(EINVAL);
        goto fail_out;
    }
    num = sd->size / roi->self_size;

    char result[2048] = { 0 };
    snprintf(result, sizeof(result), "Predicts of frame %d\n", filt_cnt);

    for (i = 0; i < num; i++) {
        if (check_movement(*last_roi, *last_roi_count, roi[i],
                           *last_roi_extra, roi_extra[i])) {
            continue;
        }
        detected++;

        if (draw_box) {
            output = av_frame_alloc();
            if (!output) {
                av_log(NULL, AV_LOG_ERROR, "cannot allocate output filter frame\n");
                return NIERROR(ENOMEM);
            }

            ret = draw_roi_box(input, &output, roi[i], roi_extra[i]);
            if (ret < 0) {
                av_log(NULL, AV_LOG_ERROR, "draw %d roi box failed\n", i);
                goto fail_out;
            }

            av_frame_free(&input);
            input = output;
        }
        snprintf(result + strlen(result), sizeof(result) - strlen(result),
                 "   type:%s,left:%d,right:%d,top:%d,bottom:%d,prob:%f\n",
                 roi_class[roi_extra[i].cls], roi[i].left, roi[i].right, roi[i].top,
                 roi[i].bottom, roi_extra[i].prob);
    }

    if (detected) {
        if (fp) {
            fwrite(result, strlen(result), 1, fp);
        } else {
            av_log(NULL, AV_LOG_ERROR, "failed to get opened file.\n");
        }
    }

    *filt_frame = input;
    free(*last_roi);
    free(*last_roi_extra);
    AVRegionOfInterest* cur_roi = NULL;
    AVRegionOfInterestNetintExtra *cur_roi_extra = NULL;
    if (num > 0) {
        cur_roi = av_malloc((int)(num * sizeof(AVRegionOfInterest)));
        cur_roi_extra = av_malloc((int)(num * sizeof(AVRegionOfInterestNetintExtra)));
        for (i = 0; i < num; i++) {
            (cur_roi[i]).self_size = roi[i].self_size;
            (cur_roi[i]).top       = roi[i].top;
            (cur_roi[i]).bottom    = roi[i].bottom;
            (cur_roi[i]).left      = roi[i].left;
            (cur_roi[i]).right     = roi[i].right;
            (cur_roi[i]).qoffset   = qp_offset;
            cur_roi_extra[i].self_size = roi_extra[i].self_size;
            cur_roi_extra[i].cls       = roi_extra[i].cls;
            cur_roi_extra[i].prob      = roi_extra[i].prob;
        }
    }
    *last_roi = cur_roi;
    *last_roi_extra = cur_roi_extra;
    *last_roi_count = num;

    return detected;
fail_out:
    av_frame_free(&input);
    av_frame_free(&output);
    return ret;
}

static void free_frame_req(struct frame_req *frame_req)
{
    if (!frame_req) {
        return;
    }
    pthread_mutex_lock(&codec_workers->free_lock);
    if (ni_list_empty(&codec_workers->free_list)) {
        ni_list_add_tail(&frame_req->list, &codec_workers->free_list);
        pthread_cond_signal(&codec_workers->free_cond);
    } else {
        ni_list_add_tail(&frame_req->list, &codec_workers->free_list);
    }
    pthread_mutex_unlock(&codec_workers->free_lock);
}

static void *printAIframe(void *data)
{
    int ret = 0;
//    struct timeval begin_time,end_time;
    struct frame_req *frame_req = NULL;
    AVFrame *avframe = NULL;
    struct network_frame *network_frame = NULL;
    int should_exit = 0;
    NiNetworkContext *network_ctx = roi_network->network_ctx;
    int aiframe_number = 0;
    AVPacket enc_packet = {0};
    void *ai_input_frame = NULL;
    AVFrame *processed_frame = NULL;
    AVRegionOfInterest *last_roi = NULL;
    AVRegionOfInterestNetintExtra *last_roi_extra = NULL;
    int last_roi_count = 0;

    FILE *fp = NULL;
    if (codec_workers->roi_output) {
        fp = fopen(codec_workers->roi_output, "w");
    } else {
        int pos_head = refind(codec_workers->input_file, '/');
        int pos_tail = refind(codec_workers->input_file, '.');
        char* substr;
        int len = pos_tail - pos_head - 1;
        if (len > 0){
            substr = malloc(len + 1);
            strncpy(substr, codec_workers->input_file+pos_head+1, len);
            substr[len] = '\0';
        } else {
            len = 0;
            substr = "";
        }
        char name[512] = {0};
        snprintf(name, sizeof(name), "%s_roi_result.txt", substr);
        // char *name = "result/roi_result.txt";
        fp = fopen(name, "w");
        if (len > 0) {
            free(substr);
        }
        substr = NULL;
    }

    ret = ni_quadra_filter_drawtext_prep_frames(&ai_input_frame);
    if (ret < 0)
    {
        av_log(NULL, AV_LOG_ERROR, "cannot alloc ai frame\n");
        return NIERROR(ENOMEM);
    }

    while(!should_exit)
    {
        if (end == 3 && ni_list_empty(&codec_workers->frm_list) && ni_list_empty(&network_frame_busy_list))
            break;

        pthread_mutex_lock(&codec_workers->frm_lock);
        if (ni_list_empty(&codec_workers->frm_list) && ni_list_empty(&network_frame_busy_list)) {
            pthread_cond_wait(&codec_workers->frm_cond,
                    &codec_workers->frm_lock);
        }

        // send frame to ai
        if ((!ni_list_empty(&codec_workers->frm_list)) &&
                !ni_list_empty(&network_frame_free_list)) {
            {
                frame_req = ni_list_first_entry(&codec_workers->frm_list, struct frame_req, list);
                ni_list_del(&frame_req->list);
                pthread_mutex_unlock(&codec_workers->frm_lock);

                avframe = &frame_req->avframe;

                if (!frame_req->eof) {
                    ret = generate_ai_frame(roi_network, ai_input_frame, avframe, use_hwframe);
                    if (ret < 0) {
                        av_log(NULL, AV_LOG_ERROR, "cannot generate ai frame\n");
                        should_exit = 1;
                        break;
                    }
                }
            }

            network_frame = ni_list_first_entry(&network_frame_free_list,
                        struct network_frame, list);
            if (!frame_req->eof) {
                ret = ni_set_network_input(network_ctx, use_hwframe, ai_input_frame, NULL,
                        avframe->width, avframe->height, &network_frame->frame, true);
                if (ret != 0 && ret != NIERROR(EAGAIN)) {
                    av_log(NULL, AV_LOG_ERROR, "Error while feeding the ai\n");
                    should_exit = 1;
                    break;
                }
            } else {
                ret = 0;
            }
            if (ret == 0) {
                network_frame->frame_req = frame_req;
                ni_list_del(&network_frame->list);
                ni_list_add_tail(&network_frame->list, &network_frame_busy_list);
            }
        } else {
            pthread_mutex_unlock(&codec_workers->frm_lock);
        }

        // get packet from ai
        if (!ni_list_empty(&network_frame_busy_list)) {
            network_frame = ni_list_first_entry(&network_frame_busy_list, struct network_frame, list);
            if (!network_frame->frame_req->eof) {
                /* pull filtered frames from the filtergraph */
                ret = ni_get_network_output(network_ctx, use_hwframe, &network_frame->frame, false,
                        true, roi_network->model_ctx->out_tensor);
                if (ret != 0 && ret != NIERROR(EAGAIN)) {
                    av_log(NULL, AV_LOG_ERROR, "Error when getting output\n");
                    should_exit = 1;
                    break;
                } else if (ret != NIERROR(EAGAIN)) {
                    frame_req = network_frame->frame_req;
                    ni_list_del(&network_frame->list);
                    ni_list_add_tail(&network_frame->list, &network_frame_free_list);

                    avframe = &frame_req->avframe;
                    ret = ni_read_roi(roi_network, avframe, aiframe_number);
                    if (ret < 0) {
                        av_log(NULL, AV_LOG_ERROR, "read roi failed\n");
                    }
                    aiframe_number++;
                    //pr_log("AI frame number %d\n", aiframe_number);

                    ret = process_roi(avframe, &processed_frame, &last_roi,
                                      &last_roi_extra, &last_roi_count, fp);
                    if (ret < 0) {
                        av_log(NULL, AV_LOG_ERROR, "cannot draw roi\n");
                        should_exit = 1;
                        break;
                    }
                    av_frame_unref(avframe);

                    ret = encode_frame(processed_frame, &enc_packet, outfile);
                    if (ret != 0) {
                        av_log(NULL, AV_LOG_ERROR, "Error when encode frame: %x\n", ret);
                        break;
                    }
                    av_frame_free(&processed_frame);

                    free_frame_req(frame_req);
                    frame_req = NULL;
                }
            } else {
                frame_req = network_frame->frame_req;
                ni_list_del(&network_frame->list);
                ni_list_add_tail(&network_frame->list, &network_frame_free_list);

                ret = encode_frame(NULL, &enc_packet, outfile);
                if (ret != 0) {
                    av_log(NULL, AV_LOG_ERROR, "Error when flushing encode frame: %x\n", ret);
                    break;
                }
                free_frame_req(frame_req);
                frame_req = NULL;
            }
        }
    }
    free(last_roi);
    free(last_roi_extra);
    fclose(fp);
    ni_quadra_frame_buffer_free(ai_input_frame);
    free(ai_input_frame);

    // avoid main thread block
    free_frame_req(frame_req);
    frame_req = NULL;
    while (!ni_list_empty(&codec_workers->frm_list)) {
        pthread_mutex_lock(&codec_workers->frm_lock);
        frame_req = ni_list_first_entry(&codec_workers->frm_list, struct frame_req, list);
        ni_list_del(&frame_req->list);
        pthread_mutex_unlock(&codec_workers->frm_lock);
        free_frame_req(frame_req);
        frame_req = NULL;
    }
    codec_workers->ait_exit = 1;
    return (void *)0;
}
static void help_usage(void)
{
    printf("Usage: \n"
            "-y | --yolo_version      yolo version [4|5].\n"
            "-s | --nbg_path          path of model.\n"
            "-k | --model_pixel       pixel of model input, 416x416 for yolov4 640x640 for yolov5.\n"
            "-i | --input             input file path.\n"
            "-o | --output_file       output file path.\n"
            "-d | --decoder           decoder name.\n"
            "-x | --decoder_params    decoder parameters.\n"
            "-n | --devid             device id.\n"
            "-v | --loglevel          available debug level: warning, info, debug, trace.\n"
            "-f | --filter_dsc        filter set params.\n"
            "-b | --bitrate           set bitrate.\n"
            "-e | --encode_name       encode name [h264_ni_dec | h265_ni_dec].\n"
            "-p | --encode_params     encode params.\n"
            "-t | --roi_output        the path of txt store the detect result of roi.\n"
            "-D | --drawbox           enable to draw box.\n"
            "-h | --help              print this help information.\n");
}

void setup_loglevel(char *loglevel)
{
    if (loglevel) {
        if (!strcmp(loglevel, "error")) {
            av_log_set_level(AV_LOG_ERROR);
        } else if (!strcmp(loglevel, "warning")) {
            av_log_set_level(AV_LOG_WARNING);
        } else if (!strcmp(loglevel, "info")) {
            av_log_set_level(AV_LOG_INFO);
        } else if (!strcmp(loglevel, "debug")) {
            av_log_set_level(AV_LOG_DEBUG);
        } else if (!strcmp(loglevel, "trace")) {
            av_log_set_level(AV_LOG_TRACE);
        } else {
            av_log_set_level(AV_LOG_INFO);
        }
    } else {
        av_log_set_level(AV_LOG_INFO);
    }
}

static int init_shared_thread_data(struct codec_worker *worker)
{
    int ret;
    int i;

    /* available list notification */
    ret = pthread_mutex_init(&worker->frm_lock, NULL);
    if (ret) {
        return ret;
    }

    ret = pthread_cond_init(&worker->frm_cond, NULL);
    if (ret) {
        goto fail_init_frm_cond;
    }

    ni_init_list_head(&worker->frm_list);

    /* free list notification */
    ret = pthread_mutex_init(&worker->free_lock, NULL);
    if (ret) {
        goto fail_init_free_lock;
    }

    ret = pthread_cond_init(&worker->free_cond, NULL);
    if (ret) {
        goto fail_init_free_cond;
    }

    ni_init_list_head(&worker->free_list);

    pframes = (struct frame_req *)calloc(NI_TRANSCODE_FRAME_NUM, sizeof(struct frame_req));
    if (!pframes) {
        goto fail_alloc_frames;
    }

    for (i = 0; i < NI_TRANSCODE_FRAME_NUM; i++) {
        ni_list_add_tail(&pframes[i].list, &worker->free_list);
    }

    return 0;
fail_alloc_frames:
    pthread_cond_destroy(&worker->free_cond);
fail_init_free_cond:
    pthread_mutex_destroy(&worker->free_lock);
fail_init_free_lock:
    pthread_cond_destroy(&worker->frm_cond);
fail_init_frm_cond:
    pthread_mutex_destroy(&worker->frm_lock);
    return ret;
}

static void cleanup_shared_thread_data(struct codec_worker *worker)
{
    if (!ni_list_empty(&worker->frm_list)) {
        av_log(NULL, AV_LOG_ERROR, "frame list is not empty\n");
    }

    free(pframes);

    pthread_mutex_destroy(&worker->frm_lock);
    pthread_cond_destroy(&worker->frm_cond);

    pthread_mutex_destroy(&worker->free_lock);
    pthread_cond_destroy(&worker->free_cond);
}

static int create_ai_thread(struct codec_worker *worker)
{
    int ret;
    pthread_attr_t attr;

    ret = pthread_attr_init(&attr);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "fail to initialize attr: %d.\n", ret);
        return ret;
    }

    ret = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "fail to set attr detachstate: %d.\n", ret);
        goto end;
    }

    ret = pthread_create(&worker->pthid, NULL, printAIframe, 0);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "creat thread failed!, %d\n", ret);
        goto end;
    }
end:
    if (0 != pthread_attr_destroy(&attr)) {
        av_log(NULL, AV_LOG_ERROR, "failed to destroy pthread attr\n");
    }
    return ret;
}

static int destroy_ai_thread(struct codec_worker *worker)
{
    int ret;
    void *result;
    av_log(NULL, AV_LOG_INFO, "try to stop ai thread\n");
    pthread_cond_signal(&worker->frm_cond);
    ret = pthread_join(worker->pthid, &result);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to join ai thread\n");
    }
    if ((int)((long)result)) {
        av_log(NULL, AV_LOG_ERROR, "Bad ai thread result: %d\n", (int)((long)result));
    }

    return 0;
}

static int create_fps_thread(struct codec_worker *worker)
{
    int ret;
    pthread_attr_t attr;

    ret = pthread_attr_init(&attr);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "fail to initialize attr: %d.\n", ret);
        return ret;
    }

    ret = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "fail to set attr detachstate: %d.\n", ret);
        goto end;
    }

    ret = pthread_create(&worker->pthtimeid,NULL,printfps,0);
    if (ret !=0) {
        av_log(NULL, AV_LOG_ERROR, "creat thread failed!\n");
    }
end:
    if (0 != pthread_attr_destroy(&attr)) {
        av_log(NULL, AV_LOG_ERROR, "failed to destroy pthread attr\n");
    }
    return ret;
}

static int destroy_fps_thread(struct codec_worker *worker)
{
    int ret;
    void *result;
    av_log(NULL, AV_LOG_INFO, "try to stop fps thread\n");
    ret = pthread_join(worker->pthtimeid, &result);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to join fps thread\n");
    }
    if ((int)((long)result)) {
        av_log(NULL, AV_LOG_ERROR, "Bad fps thread result: %d\n", (int)((long)result));
    }

    return ret;
}

static struct frame_req *get_free_frame_req()
{
    struct frame_req *frame_req;
wait_free_frame:
    pthread_mutex_lock(&codec_workers->free_lock);
    if (ni_list_empty(&codec_workers->free_list)) {
        pthread_cond_wait(&codec_workers->free_cond, &codec_workers->free_lock);
    }
    if (ni_list_empty(&codec_workers->free_list)) {
        pthread_mutex_unlock(&codec_workers->free_lock);
        goto wait_free_frame;
    }
    frame_req = ni_list_first_entry(&codec_workers->free_list, struct frame_req, list);
    ni_list_del(&frame_req->list);
    pthread_mutex_unlock(&codec_workers->free_lock);

    return frame_req;
}

static void put_busy_frame_req(struct frame_req *frame_req)
{
    pthread_mutex_lock(&codec_workers->frm_lock);
    if (ni_list_empty(&codec_workers->frm_list)) {
        ni_list_add_tail(&frame_req->list, &codec_workers->frm_list);
        pthread_cond_signal(&codec_workers->frm_cond);
    } else {
        ni_list_add_tail(&frame_req->list, &codec_workers->frm_list);
    }
    pthread_mutex_unlock(&codec_workers->frm_lock);
}

int main(int argc, char **argv)
{
    int i, ret;
    AVPacket packet;

    const char *nbg_file = "../../AI_benchmarking_examples/network_binary_yolov5s_improved.nb";
    int model_width = 640;
    int model_height = 640;
    int model_format = 0x10C;//GC620_RGB888_PLANAR

    struct frame_req *frame_req;

    char *loglevel = NULL;
    int opt;
    int opt_index;
    char *filter_dsc;
    codec_workers = (struct codec_worker *)calloc(1, sizeof(struct codec_worker));
    if (codec_workers == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate codec workers.\n");
        return EXIT_FAILURE;
    }
    codec_workers->bitrate = 0;
    codec_workers->yolo_version = 5;
    const char *opt_string = "y:n:x:i:d:o:e:p:v:b:t:Df:l:s:k:h";
    static struct option long_options[] = {
        {"yolo_version",   required_argument, NULL, 'y'},
        {"input",          required_argument, NULL, 'i'},
        {"output",         required_argument, NULL, 'o'},
        {"decoder",        required_argument, NULL, 'd'},
        {"decoder_params", required_argument, NULL, 'x'},
        {"devid",          required_argument, NULL, 'n'},
        {"loglevel",       required_argument, NULL, 'v'},
        {"filter_dsc",     required_argument, NULL, 'f'},
        {"bitrate",        required_argument, NULL, 'b'},
        {"encode_name",    required_argument, NULL, 'e'},
        {"encode_params",  required_argument, NULL, 'p'},
        {"roi_output",     required_argument, NULL, 't'},
        {"drawbox",        no_argument,       NULL, 'D'},
        {"log",            required_argument, NULL, 'l'},
        {"nbg_path",       required_argument, NULL, 's'},
        {"model_pixel",    required_argument, NULL, 'k'},
        {"help",           no_argument,       NULL, 'h'},
        { NULL,            0,                 NULL,  0 },
    };

    while ((opt = getopt_long(argc, argv, opt_string, long_options, &opt_index)) != -1) {
        switch (opt) {

            case 'd':
                codec_workers->decoder_name = optarg;
                break;
            case 'i':
                codec_workers->input_file = optarg;
                break;
            case 'o':
                codec_workers->output_file = optarg;
                break;
            case 'n':
                codec_workers->devid = atoi(optarg);
                break;
            case 'x':
                codec_workers->decoder_params = optarg;
                if (strcmp(codec_workers->decoder_params, "out=hw") == 0) {
                    use_hwframe = true;
                    printf("set use_hwframe %d\n", use_hwframe);
                }
                break;
            case 'v':
                loglevel = optarg;
                break;
            case 'f':
                filter_dsc = optarg;
                break;
            case 'b':
                codec_workers->bitrate = atoi(optarg);
                break;
            case 'e':
                codec_workers->encoder_name = optarg;
                break;
            case 't':
                codec_workers->roi_output = optarg;
                break;
            case 'D':
                draw_box = 1;
                break;
            case 'p':
                codec_workers->encoder_params = optarg;
                break;
            case 'l':
                log_index = atoi(optarg);
                break;
            case 's':
                nbg_file = optarg;
                break;
            case 'y':
                codec_workers->yolo_version = atoi(optarg);
                break;
            case 'k':
            {
                char *ch;
                model_width = strtoul(optarg, &ch, 10);
                if (*ch != 'x') {
                    av_log(NULL, AV_LOG_ERROR, "invalid model_pixel\n");
                    return EXIT_FAILURE;
                }
                model_height = strtoul(ch + 1, NULL, 10);
                break;
            }
            case 'h':
                help_usage();
                return EXIT_SUCCESS;
            default:
                help_usage();
                return EXIT_FAILURE;
        }
    }
    setup_loglevel(loglevel);

    // Register all formats, codecs, filters
#if LIBAVFORMAT_VERSION_MAJOR < 58
    av_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif
#if LIBAVCODEC_VERSION_MAJOR < 58
    avcodec_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif
#if LIBAVFILTER_VERSION_MAJOR < 7
    avfilter_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif

    if (access(nbg_file, R_OK) != 0) {
        av_log(NULL, AV_LOG_ERROR, "cannot find nbg: %s\n", nbg_file);
        return EXIT_FAILURE;
    }

    if (codec_workers->input_file == NULL) {
        av_log(NULL, AV_LOG_ERROR, "input file name must be specified.\n");
        return EXIT_FAILURE;
    }

    if (codec_workers->output_file == NULL) {
        av_log(NULL, AV_LOG_ERROR, "output file name must be specified.\n");
        return EXIT_FAILURE;
    }

    if(!filter_dsc)
    {
        av_log(NULL, AV_LOG_ERROR, "filter desc must be specified.\n");
        return EXIT_FAILURE;
    }

    if (!codec_workers->decoder_name || !codec_workers->encoder_name) {
        av_log(NULL, AV_LOG_ERROR, "decoder_name name must be specified.\n");
        return EXIT_FAILURE;
    }

    if (init_shared_thread_data(codec_workers) != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to initialize common data\n");
        return EXIT_FAILURE;
    }

    outfile = fopen(codec_workers->output_file,"wb");
    if(!outfile)
    {
        av_log(NULL, AV_LOG_ERROR, "open file fail!\n");
        exit(1);
    }
    // Allocate packet for encoder output
    frame = av_frame_alloc();
    if (!frame) {
        av_log(NULL, AV_LOG_ERROR, "Could not allocate pkt frame");
        exit(1);
    }

    // ret = init_network_struct(model_width, model_height, model_format, nbg_file);
    // if (ret != 0) {
    //     av_log(NULL, AV_LOG_ERROR, "failed to initialize network struct\n");
    //     goto end;
    // }

    if ((ret = open_input_file()) < 0)
        goto end;

    if (draw_box) {
        drawbox_filter = malloc(sizeof(filter_worker));
        drawbox_filter->buffersink_ctx = NULL;
        drawbox_filter->buffersrc_ctx = NULL;
        drawbox_filter->filter_graph = NULL;
    }

    hwdl_filter = malloc(sizeof(filter_worker));
    hwdl_filter->buffersink_ctx = NULL;
    hwdl_filter->buffersrc_ctx = NULL;
    hwdl_filter->filter_graph = NULL;

    if (use_hwframe == false) {
        memset(&scaled_frame, 0, sizeof(AVFrame));
        scaled_frame.width  = model_width;
        scaled_frame.height = model_height;
        scaled_frame.format = AV_PIX_FMT_RGB24;

        if (av_frame_get_buffer(&scaled_frame, 32)) {
            av_log(NULL, AV_LOG_ERROR, "cannot allocate scaled frame buffer\n");
            goto end;
        }

        sw_scale_ctx = sws_getContext(dec_ctx->width, dec_ctx->height, AV_PIX_FMT_YUV420P,
                scaled_frame.width, scaled_frame.height, scaled_frame.format,
                SWS_BICUBIC, NULL, NULL, NULL);
        if (!sw_scale_ctx) {
            av_log(NULL, AV_LOG_ERROR, "cannot create sw scale context for scaling\n");
            goto end;
        }
    }

    timer_start = av_gettime_relative();

    if ((ret = create_fps_thread(codec_workers)) != 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to create fps thread\n");
        goto end;
    }

    // if ((ret = create_ai_thread(codec_workers)) != 0) {
    //     av_log(NULL, AV_LOG_ERROR, "Failed to create ai thread\n");
    //     goto stop_fpst;
    // }

    /* read all packets */
    while (1) {
        if (codec_workers->ait_exit) {
            break;
        }
        if ((ret = av_read_frame(fmt_ctx, &packet)) < 0)
        {
            if(end == 2)
            {
                frame_req = get_free_frame_req();
                frame_req->eof = 1;
                put_busy_frame_req(frame_req);
                break;
            }
            end = 1;
        }
        if (packet.stream_index == video_stream_index || end == 1) {
            if(!end)
            {
                ret = avcodec_send_packet(dec_ctx, &packet);
            }
            else{
                ret = avcodec_send_packet(dec_ctx, NULL);
            }

            if (ret < 0 && ret != AVERROR_EOF) {
                av_log(NULL, AV_LOG_ERROR, "Error while sending a packet to the decoder\n");
                break;
            }
            while (ret >= 0 || end == 1) {
                ret = avcodec_receive_frame(dec_ctx, frame);
                if (ret == AVERROR(EAGAIN)) {
                    break;
                }
                else if(ret == AVERROR_EOF){
                    end = 2;
                    break;
                }
                else if (ret < 0) {
                    av_log(NULL, AV_LOG_ERROR, "Error while receiving a frame from the decoder\n");
                    goto stop_fpst;
                }

                if (!init_filters) {
                    ret = init_network_struct(frame, model_width, model_height, model_format, nbg_file);
                    if (ret != 0) {
                        av_log(NULL, AV_LOG_ERROR, "failed to initialize network struct\n");
                        goto end;
                    }
                    if ((ret = create_ai_thread(codec_workers)) != 0) {
                        av_log(NULL, AV_LOG_ERROR, "Failed to create ai thread\n");
                        goto stop_fpst;
                    }
                    if (draw_box) {
                        if ((ret = init_filter("drawbox", drawbox_filter, false)) < 0) {
                            av_log(NULL, AV_LOG_ERROR, "cannot initialize drawbox filter\n");
                            goto stop_ait;
                        }

                        for (i = 0; i < drawbox_filter->filter_graph->nb_filters; i++) {
                            if (strstr(drawbox_filter->filter_graph->filters[i]->name, "drawbox") != NULL) {
                                drawbox_filter_ctx = drawbox_filter->filter_graph->filters[i];
                                break;
                            }
                        }

                        if (drawbox_filter_ctx == NULL) {
                            av_log(NULL, AV_LOG_ERROR, "cannot find valid drawbox filter\n");
                            ret = NIERROR(EINVAL);
                            goto stop_ait;
                        }
                    }

                    const char *hwdl_desc = "[in]hwdownload,format=yuv420p[out]";
                    if ((ret = init_filter(hwdl_desc, hwdl_filter, true)) < 0) {
                        av_log(NULL, AV_LOG_ERROR, "cannot initialize hwdl filter\n");
                        goto stop_ait;
                    }
                    init_filters = 1;
                }
                frame->pts = frame->best_effort_timestamp;
                frame_num++;

                av_log(NULL, AV_LOG_DEBUG, "decoded frame_num %d\n", frame_num);

                frame_req = get_free_frame_req();
                frame_req->eof = 0;
                ret = av_frame_ref(&frame_req->avframe, frame);
                if (ret < 0) {
                    av_log(NULL, AV_LOG_ERROR, "copy frame to filter failed\n");
                    goto stop_ait;
                }
                av_log(NULL, AV_LOG_DEBUG, "generate frame %d\n",
                        ((niquadraFrameSurface1_t *)frame->data[3])->ui16FrameIdx);
                put_busy_frame_req(frame_req);
                av_frame_unref(frame);
            }
        }
        av_packet_unref(&packet);
    }

stop_ait:
    end = 3;
    destroy_ai_thread(codec_workers);
stop_fpst:
    stop_fps = true;
    destroy_fps_thread(codec_workers);
end:
    avcodec_free_context(&dec_ctx);
    avcodec_free_context(&enc_ctx);
    avformat_close_input(&fmt_ctx);
    av_frame_free(&frame);
    fclose(outfile);

    cleanup_shared_thread_data(codec_workers);
    free(codec_workers);
    free_network_struct();

    av_frame_unref(&scaled_frame);
    sws_freeContext(sw_scale_ctx);

    if (draw_box) {
        avfilter_graph_free(&drawbox_filter->filter_graph);
        free(drawbox_filter);
    }
    avfilter_graph_free(&hwdl_filter->filter_graph);
    free(hwdl_filter);
    if (ret < 0 && ret != AVERROR_EOF) {
        av_log(NULL, AV_LOG_ERROR, "Error occurred: %s\n", av_err2str(ret));
        exit(1);
    }
    exit(0);
}
