/*
 * Copyright (c) 2010 Nicolas George
 * Copyright (c) 2011 Stefano Sabatini
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * @file
 * API example for scene classify
 * @example ni_scene_classify.c
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <sys/time.h>

#include <libavcodec/avcodec.h>
#include <libavformat/avformat.h>
#include <libavfilter/buffersink.h>
#include <libavfilter/buffersrc.h>
#include <libswscale/swscale.h>
#include <libavutil/imgutils.h>
#include <libavutil/opt.h>
#include <libavutil/time.h>
#include <pthread.h>
#include <stdint.h>

#include "ni_list_head.h"
#include "scene_classify_postprocess.h"

#define NI_TRANSCODE_FRAME_NUM 3

bool use_hwframe = false;
static AVFormatContext *ifmt_ctx;
static AVCodecContext *dec_ctx;
static AVCodecContext *enc_ctx;
static int video_stream_index = -1;
int frame_num = 0, end = 0, encode_num = 0;
int64_t timer_start;
bool stop_fps = false;
int init_encoder = 0;

typedef struct _filter_worker
{
    AVFilterContext *buffersink_ctx;
    AVFilterContext *buffersrc_ctx;
    AVFilterGraph *filter_graph;
} filter_worker;

typedef struct _frame_req {
    struct ni_list_head list;
    AVFrame avframe;
    int eof;
} frame_req;

typedef struct codec_worker {
    const char *decoder_name;
    const char *input_file;
    const char *output_file;
    const char *decoder_params;
    const char *encoder_name;
    const char *encoder_params;
    int devid;
    const char *res_output;

    pthread_t pthid;
    pthread_t pthtimeid;

    int ait_exit;
    int model_width;
    int model_height;

    struct ni_list_head frm_list;
    pthread_mutex_t frm_lock;
    pthread_cond_t  frm_cond;

    struct ni_list_head free_list;
    pthread_mutex_t free_lock;
    pthread_cond_t free_cond;
} codec_worker;
codec_worker *codec_workers = NULL;

typedef struct sc_network {
    NiNetworkContext *network_ctx;
    SCModelCtx *model_ctx;
    SCModel *model;
} sc_network;
sc_network *p_sc_network;

static char *classes[] = {"airfield", "airplane_cabin", "airport_terminal", "alcove", "alley", "amphitheater", "amusement_arcade", "amusement_park",
"apartment_building-outdoor", "aquarium", "aqueduct", "arcade", "arch", "archaelogical_excavation", "archive", "arena-hockey","arena-performance",
"arena-rodeo", "army_base", "art_gallery", "art_school", "art_studio", "artists_loft", "assembly_line", "athletic_field-outdoor", "atrium-public",
"attic", "auditorium", "auto_factory", "auto_showroom", "badlands", "bakery-shop", "balcony-exterior", "balcony-interior", "ball_pit", "ballroom",
"bamboo_forest", "bank_vault", "banquet_hall", "bar", "barn", "barndoor", "baseball_field", "basement", "basketball_court-indoor", "bathroom",
"bazaar-indoor", "bazaar-outdoor", "beach", "beach_house", "beauty_salon", "bedchamber", "bedroom", "beer_garden", "beer_hall", "berth",
"biology_laboratory", "boardwalk", "boat_deck", "boathouse", "bookstore", "booth-indoor", "botanical_garden", "bow_window-indoor", "bowling_alley",
"boxing_ring", "bridge", "building_facade", "bullring", "burial_chamber", "bus_interior", "bus_station-indoor", "butchers_shop", "butte", "cabin-outdoor",
"cafeteria", "campsite", "campus", "canal-natural", "canal-urban", "candy_store", "canyon", "car_interior", "carrousel", "castle","catacomb", "cemetery",
"chalet", "chemistry_lab", "childs_room", "church-indoor", "church-outdoor", "classroom", "clean_room", "cliff", "closet", "clothing_store", "coast",
"cockpit", "coffee_shop", "computer_room", "conference_center", "conference_room", "construction_site", "corn_field", "corral", "corridor", "cottage",
"courthouse", "courtyard", "creek", "crevasse", "crosswalk", "dam", "delicatessen", "department_store", "desert-sand", "desert-vegetation", "desert_road",
"diner-outdoor", "dining_hall", "dining_room", "discotheque", "doorway-outdoor", "dorm_room", "downtown", "dressing_room", "driveway", "drugstore",
"elevator-door", "elevator_lobby", "elevator_shaft", "embassy", "engine_room", "entrance_hall", "escalator-indoor", "excavation", "fabric_store",
"farm", "fastfood_restaurant", "field-cultivated", "field-wild", "field_road", "fire_escape", "fire_station", "fishpond", "flea_market-indoor",
"florist_shop-indoor", "food_court", "football_field", "forest-broadleaf", "forest_path", "forest_road", "formal_garden", "fountain", "galley",
"garage-indoor", "garage-outdoor", "gas_station", "gazebo-exterior", "general_store-indoor", "general_store-outdoor", "gift_shop", "glacier",
"golf_course", "greenhouse-indoor", "greenhouse-outdoor", "grotto", "gymnasium-indoor", "hangar-indoor", "hangar-outdoor", "harbor",
"hardware_store", "hayfield", "heliport", "highway", "home_office", "home_theater", "hospital", "hospital_room", "hot_spring", "hotel-outdoor",
"hotel_room", "house", "hunting_lodge-outdoor", "ice_cream_parlor", "ice_floe", "ice_shelf", "ice_skating_rink-indoor", "ice_skating_rink-outdoor",
"iceberg", "igloo", "industrial_area", "inn-outdoor", "islet","jacuzzi-indoor", "jail_cell", "japanese_garden", "jewelry_shop", "junkyard", "kasbah",
"kennel-outdoor", "kindergarden_classroom", "kitchen", "lagoon", "lake-natural", "landfill", "landing_deck", "laundromat", "lawn", "lecture_room",
"legislative_chamber", "library-indoor", "library-outdoor", "lighthouse", "living_room", "loading_dock", "lobby", "lock_chamber", "locker_room", "mansion",
"manufactured_home", "market-indoor", "market-outdoor", "marsh", "martial_arts_gym", "mausoleum", "medina", "mezzanine", "moat-water", "mosque-outdoor",
"motel", "mountain", "mountain_path", "mountain_snowy", "movie_theater-indoor", "museum-indoor", "museum-outdoor", "music_studio", "natural_history_museum",
"nursery", "nursing_home", "oast_house", "ocean", "office", "office_building", "office_cubicles", "oilrig", "operating_room", "orchard", "orchestra_pit",
"pagoda", "palace", "pantry", "park", "parking_garage-indoor", "parking_garage-outdoor", "parking_lot", "pasture", "patio", "pavilion", "pet_shop", "pharmacy",
"phone_booth", "physics_laboratory", "picnic_area", "pier", "pizzeria", "playground", "playroom", "plaza", "pond", "porch", "promenade", "pub-indoor",
"racecourse", "raceway", "raft", "railroad_track", "rainforest", "reception", "recreation_room", "repair_shop", "residential_neighborhood", "restaurant",
"restaurant_kitchen", "restaurant_patio", "rice_paddy", "river", "rock_arch", "roof_garden", "rope_bridge", "ruin", "runway", "sandbox", "sauna",
"schoolhouse", "science_museum", "server_room", "shed", "shoe_shop", "shopfront", "shopping_mall-indoor", "shower", "ski_resort", "ski_slope", "sky",
"skyscraper", "slum", "snowfield", "soccer_field", "stable", "stadium-baseball", "stadium-football", "stadium-soccer", "stage-indoor", "stage-outdoor",
"staircase", "storage_room", "street", "subway_station-platform", "supermarket", "sushi_bar", "swamp", "swimming_hole", "swimming_pool-indoor",
"swimming_pool-outdoor", "synagogue-outdoor", "television_room", "television_studio", "temple-asia", "throne_room", "ticket_booth", "topiary_garden",
"tower", "toyshop", "train_interior", "train_station-platform", "tree_farm", "tree_house", "trench", "tundra", "underwater-ocean_deep", "utility_room",
"valley", "vegetable_garden", "veterinarians_office", "viaduct", "village", "vineyard", "volcano", "volleyball_court-outdoor", "waiting_room",
"water_park", "water_tower", "waterfall", "watering_hole", "wave", "wet_bar", "wheat_field", "wind_farm", "windmill", "yard", "youth_hostel", "zen_garden"};

static char *class_name_to_category[][26] = {
{"arena-hockey", "arena-performance", "athletic_field-outdoor", "volleyball_court-outdoor", "baseball_field", "basketball_court-indoor", "bowling_alley",
"boxing_ring", "football_field", "golf_course", "gymnasium-indoor", "martial_arts_gym","racecourse", "raceway", "raft", "ski_resort", "ski_slope",
"snowfield", "soccer_field", "stadium-baseball", "stadium-football", "stadium-soccer", "swimming_pool-indoor", "swimming_pool-outdoor",
"ice_skating_rink-indoor","ice_skating_rink-outdoor"},
{"music_studio", "television_studio","stage-indoor","home_theater","computer_room","home_office","reception","office", "office_cubicles"},
{"auditorium", "classroom", "conference_center", "conference_room", "legislative_chamber", "movie_theater-indoor","stage-outdoor","lecture_room"}};

static int class_id_to_category[][26] = {
{15, 16, 24, 351, 42, 44, 64, 65, 149, 164, 168, 225, 275, 276, 277, 304, 305, 309, 310, 312, 313, 314, 325, 326, 188, 189},
{238, 329, 315, 177, 100, 176, 280, 244, 246, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1},
{27, 92, 101, 102, 211, 235, 316, 210, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1}};

static char *category[] = {"stadiums", "tv_studios", "auditoriums", "others"};


NI_LIST_HEAD(network_frame_free_list);
NI_LIST_HEAD(network_frame_busy_list);

struct network_frame {
    NiNetworkFrame frame;
    struct ni_list_head list;
    frame_req *p_frame_req;
};
frame_req *pframes = NULL;

struct network_frame *sc_network_frames;

static int refind (const char *a, char b) {
    int i, len = strlen(a);
    for (i = len - 1; i >= 0; i--) {
        if (a[i] == b) return i;
    }
    return -1;
}

static int init_shared_thread_data(struct codec_worker *worker) {
    int ret, i;

    /* available list notification */
    ret = pthread_mutex_init(&worker->frm_lock, NULL);
    if (ret) {
        return ret;
    }

    ret = pthread_cond_init(&worker->frm_cond, NULL);
    if (ret) {
        goto fail_init_frm_cond;
    }

    ni_init_list_head(&worker->frm_list);

    /* free list notification */
    ret = pthread_mutex_init(&worker->free_lock, NULL);
    if (ret) {
        goto fail_init_free_lock;
    }

    ret = pthread_cond_init(&worker->free_cond, NULL);
    if (ret) {
        goto fail_init_free_cond;
    }

    ni_init_list_head(&worker->free_list);

    pframes = (frame_req *)calloc(NI_TRANSCODE_FRAME_NUM, sizeof(frame_req));
    if (!pframes) {
        goto fail_alloc_frames;
    }

    for (i = 0; i < NI_TRANSCODE_FRAME_NUM; i++) {
        ni_list_add_tail(&pframes[i].list, &worker->free_list);
    }

    return 0;
fail_alloc_frames:
    pthread_cond_destroy(&worker->free_cond);
fail_init_free_cond:
    pthread_mutex_destroy(&worker->free_lock);
fail_init_free_lock:
    pthread_cond_destroy(&worker->frm_cond);
fail_init_frm_cond:
    pthread_mutex_destroy(&worker->frm_lock);
    return ret;
}

static void cleanup_shared_thread_data(struct codec_worker *worker) {
    if (!ni_list_empty(&worker->frm_list)) {
        av_log(NULL, AV_LOG_ERROR, "frame list is not empty\n");
    }
    free(pframes);
    pthread_mutex_destroy(&worker->frm_lock);
    pthread_cond_destroy(&worker->frm_cond);

    pthread_mutex_destroy(&worker->free_lock);
    pthread_cond_destroy(&worker->free_cond);
}

static int init_network_frames(sc_network *p_sc_network, int model_width, int model_height, int model_format) {
    int i, ret;

    ni_init_list_head(&network_frame_free_list);
    ni_init_list_head(&network_frame_busy_list);

    sc_network_frames = (struct network_frame *)calloc(NUM_NETWORK_FRAME,
            sizeof(struct network_frame));
    if (sc_network_frames == NULL) {
        return -1;
    }

    for (i = 0; i < NUM_NETWORK_FRAME; i++) {
        NiNetworkFrame *frame = &sc_network_frames[i].frame;
        frame->scale_width = model_width;
        frame->scale_height = model_height;
        frame->scale_format = model_format;
        ni_init_list_head(&sc_network_frames[i].list);
        ni_list_add_tail(&sc_network_frames[i].list, &network_frame_free_list);
        ret = ni_ai_packet_buffer_alloc(&frame->api_packet.data.packet,
            &p_sc_network->network_ctx->network_data);
        if (ret != NI_RETCODE_SUCCESS) {
            av_log(NULL, AV_LOG_ERROR, "failed to allocate packet\n");
            return NIERROR(ENOMEM);
        }
        ret = ni_frame_buffer_alloc_hwenc(&frame->api_frame.data.frame,
                                          frame->scale_width,
                                          frame->scale_height, 0);
        if (ret != NI_RETCODE_SUCCESS) {
            av_log(NULL, AV_LOG_ERROR, "failed to allocate frame\n");
            return NIERROR(ENOMEM);
        }
    }

    return 0;
}

static void free_network_frames() {
    int i;
    if (!ni_list_empty(&network_frame_busy_list)) {
        av_log(NULL, AV_LOG_WARNING, "Network frames are still in use\n");
    }

    for (i = 0; i < NUM_NETWORK_FRAME; i++) {
        NiNetworkFrame *frame = &sc_network_frames[i].frame;
        ni_frame_buffer_free(&frame->api_frame.data.frame);
        ni_packet_buffer_free(&frame->api_packet.data.packet);
    }
    if (sc_network_frames != NULL) {
        free(sc_network_frames);
        sc_network_frames = NULL;
    }
}

static void destroy_network(sc_network *network) {
    if (network && network->model) {
        network->model->destroy_model(network->model_ctx);;
        free(network->model_ctx);
    }
}

/* Specific to a model */
static int create_network(sc_network *network, int model_width, int model_height) {
    int ret = 0;
    SCModel *model;
    SCModelCtx *model_ctx;
    ni_network_data_t *network_data = &network->network_ctx->network_data;

    model_ctx = (SCModelCtx *)calloc(1, sizeof(SCModelCtx));
    if (model_ctx == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate yolo model\n");
        return NIERROR(ENOMEM);
    }
    network->model_ctx = model_ctx;
    model = &sc;

    ret = model->create_model(model_ctx, network_data, model_width, model_height);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to initialize yolov4 model\n");
        goto out;
    }
    network->model = model;

    return ret;

out:
    if (model_ctx) {
      free(model_ctx);
      network->model_ctx = NULL;
    }

    return ret;
}

static int init_network_struct(int model_width, int model_height, int model_format,
                               const char *nbg_file, int use_hwframe, int devid) {
    int ret;

    p_sc_network = (sc_network *)calloc(1, sizeof(sc_network));
    if (p_sc_network == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate network\n");
        return -1;
    }

    ret = ni_alloc_network_context(&p_sc_network->network_ctx, use_hwframe,
            devid, 30, model_format, model_width, model_height, nbg_file);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate network context\n");
        goto fail_out;
    }

    ret = create_network(p_sc_network, model_width, model_height);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to create network\n");
        goto fail_out;
    }

    ret = init_network_frames(p_sc_network, model_width, model_height,
            model_format);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to initialize network_frames\n");
        goto fail_out;
    }

    return 0;
fail_out:
    destroy_network(p_sc_network);
    ni_cleanup_network_context(p_sc_network->network_ctx, use_hwframe);
    free_network_frames();
    free(p_sc_network);
    p_sc_network = NULL;
    return -1;
}

static void free_network_struct(sc_network *network) {
    if (network) {
        destroy_network(network);
        ni_cleanup_network_context(network->network_ctx, use_hwframe);
        free(network);
    }
    free_network_frames();
}

static int open_input_file() {
    int ret, i;
    const AVCodec *dec;
    if ((ret = avformat_open_input(&ifmt_ctx, codec_workers->input_file, NULL, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot open input file\n");
        return ret;
    }

    if ((ret = avformat_find_stream_info(ifmt_ctx, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot find stream information\n");
        return ret;
    }
    for (i = 0; i < ifmt_ctx->nb_streams; i++) {
        if (ifmt_ctx->streams[i]->codecpar->codec_type == AVMEDIA_TYPE_VIDEO) {
            video_stream_index = i;
            break;
        }
    }
    if (video_stream_index < 0) {
        av_log(NULL, AV_LOG_ERROR, "can not find video stream\n");
        return video_stream_index;
    }

    dec = avcodec_find_decoder_by_name(codec_workers->decoder_name);
    if (dec) {
        if (ifmt_ctx->streams[0]->codecpar->codec_id != dec->id) {
            av_log(NULL, AV_LOG_ERROR, "codec does not match with stream id\n");
            return AVERROR_DECODER_NOT_FOUND;
        }
    }

    /* create decoding context */
    dec_ctx = avcodec_alloc_context3(dec);
    if (!dec_ctx)
        return AVERROR(ENOMEM);

    avcodec_parameters_to_context(dec_ctx, ifmt_ctx->streams[0]->codecpar);

    char str_devid[4] = {0};
    snprintf(str_devid, sizeof(str_devid), "%d", codec_workers->devid);
    av_opt_set(dec_ctx->priv_data, "dec", str_devid, 0);
    if(codec_workers->decoder_params)
        av_opt_set(dec_ctx->priv_data, "xcoder-params",codec_workers->decoder_params, 0);

    if (dec_ctx->codec_type == AVMEDIA_TYPE_VIDEO)
            dec_ctx->framerate = av_guess_frame_rate(ifmt_ctx, ifmt_ctx->streams[0], NULL);

    ret = avcodec_parameters_from_context(ifmt_ctx->streams[0]->codecpar, dec_ctx);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to copy decoder parameters from codec context for stream #");
        return ret;
    }

    /* init the video decoder */
    if ((ret = avcodec_open2(dec_ctx, dec, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot open video decoder\n");
        return ret;
    }
    return 0;
}

static int open_output_file(AVFrame *frame) {
    int ret = 0;
    const AVCodec *enc;

    // Find video encoder codec selected
    enc = avcodec_find_encoder_by_name(codec_workers->encoder_name);
    if (!enc) {
        av_log(NULL, AV_LOG_ERROR,"Codec %s not found\n", codec_workers->encoder_name);
        return AVERROR_ENCODER_NOT_FOUND;
    }

    // Allocate codec context for encoding
    enc_ctx = avcodec_alloc_context3(enc);
    if (!enc_ctx) {
        av_log(NULL, AV_LOG_ERROR, "Could not allocate video codec context\n");
        return AVERROR(ENOMEM);
    }

    enc_ctx->height = frame->height;
    enc_ctx->width = frame->width;
    enc_ctx->framerate = dec_ctx->framerate;
    enc_ctx->sample_aspect_ratio = dec_ctx->sample_aspect_ratio;
    enc_ctx->pix_fmt = frame->format;
    if (frame->hw_frames_ctx) {
        AVHWFramesContext* pAVHFWCtx = (AVHWFramesContext *)frame->hw_frames_ctx->data;
        enc_ctx->sw_pix_fmt = pAVHFWCtx->sw_format;
    }

    /* video time_base can be set to whatever is handy and supported by encoder */
    enc_ctx->time_base = av_inv_q(dec_ctx->framerate);

    av_log(NULL, AV_LOG_INFO, "dec_ctx->sw_pix_fmt %d, pix_fmt %d\n",
            dec_ctx->sw_pix_fmt, dec_ctx->pix_fmt);

    if ((enc->id == AV_CODEC_ID_H264) || (enc->id == AV_CODEC_ID_H265)) {
        if(codec_workers->encoder_params) {
            av_opt_set(enc_ctx->priv_data, "xcoder-params", codec_workers->encoder_params, 0);
        }
        if(codec_workers->devid >= 0) {
            char str_devid[4] = {0};
            snprintf(str_devid, sizeof(str_devid), "%d", codec_workers->devid);
            av_opt_set(enc_ctx->priv_data, "enc", str_devid, 0);
        }
    } else {
        av_log(NULL, AV_LOG_ERROR, "codec id %d not supported.\n", enc->id);
        ret = -1;
        return ret;
    }

    // Open encoder
    ret = avcodec_open2(enc_ctx, enc, NULL);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Could not open enc\n");
        return ret;
    }
    return 0;
}

static int encode_frame(AVFrame *frame, AVPacket *pkt, FILE *outfile)
{
    int ret;
    if (!init_encoder && frame)
    {
        if((ret = open_output_file(frame)) < 0)
        {
            av_log(NULL, AV_LOG_ERROR, "init encoder failed ret = %d\n", ret);
            return ret;
        }
        init_encoder = 1;
    }

    int attempt_cnt = 1;
    int re_send = 0;
    static int send_cnt = 0;
    static int recv_cnt = 0;

    // Send frame to encoder. If frame=NULL, this signals end of stream
    if (frame) {
        frame->pict_type = AV_PICTURE_TYPE_NONE;
        av_log(NULL, AV_LOG_DEBUG, "Send %d frame %3"PRId64", type=%d\n", ++send_cnt, frame->pts, frame->pict_type);
    } else {
        av_log(NULL, AV_LOG_DEBUG,"Send done, flush encoder\n");
    }

resend:
    ret = avcodec_send_frame(enc_ctx, frame);
    // Encoder is too busy to receive frame, try sending frame again
    if (ret == AVERROR(EAGAIN)) {
        av_log(NULL, AV_LOG_DEBUG, "Resending frame. Try #%d.\n", ++attempt_cnt);
        ret = 0;
        re_send = 1;
    }
    // Frame send issue occured
    else if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Error sending frame for encoding. RC=%d\n", AVUNERROR(ret));
        return ret;
    }

    /* Run this loop if a frame was sucessfully sent during avcodec_send_frame() OR
        no frame was sent because this function is in flushing mode OR
        a frame was sucessfully received by avcodec_receive_packet() */
    while (ret >= 0 || ! frame ) {
        av_log(NULL, AV_LOG_DEBUG, "Receiving packet\n");
        ret = avcodec_receive_packet(enc_ctx, pkt);
        if (frame == NULL)
            av_log(NULL, AV_LOG_DEBUG, "Draining packet ret=%x, pkt size=%5d\n", ret, pkt->size);

        // No packets in encoder output buffer but not end of stream either
        if (ret == AVERROR(EAGAIN))  {
            // If function not in flushing mode
            if (frame) {
                if (re_send) {
                    re_send = 0;
                    goto resend;
                }
                return 0; // Exit this function to send next packet
            } else {
                continue; // Loop in this function to flush encoder
            }
        }
        // End of stream detected
        else if (ret == AVERROR_EOF) {
            av_log(NULL, AV_LOG_DEBUG, "Encoder EOF reached\n");
            return 0;
        }
        // Packet receive error occured
        else if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "Error during encoding\n");
            return ret;
        }
        // Packet sucessfully received
        av_log(NULL, AV_LOG_DEBUG,"Received packet %d, pts=%"PRId64"\n", ++recv_cnt, pkt->pts);
        encode_num++;

        fwrite(pkt->data, 1, pkt->size, outfile);
        av_packet_unref(pkt);
    }
    return 0;
}

static void print_report(int is_last_report, int64_t timer_start,
        int64_t cur_time, unsigned long frame_number) {
    static int64_t last_time = -1;
    float t;
    char buf[1024];
    float fps;

    if (!is_last_report) {
        if (last_time == -1) {
            last_time = cur_time;
            return;
        }
        if ((cur_time - last_time) < 500000)
            return;
        last_time = cur_time;
    }

    t = (cur_time - timer_start) / 1000000.0;

    fps = t > 1 ? frame_number / t : 0;

    const char end = is_last_report ? '\n' : '\r';

    buf[0] = '\0';
    snprintf(buf + strlen(buf), sizeof(buf) - strlen(buf), "frame=%5ld fps=%3.*f ",
            frame_number, (fps < 9.95), fps);
    fprintf(stderr, "%s   %c", buf, end);
}

static void *printfps(void *data) {
    while (1) {
        if (stop_fps == true) {
            break;
        }
        usleep(1000);
        print_report(0, timer_start, av_gettime_relative(), encode_num);
    }
    print_report(1, timer_start, av_gettime_relative(), encode_num);
    return (void *)0;
}


static int ni_read_ai(struct sc_network *p_sc_network, FILE *fp, int aiframe_number, AVFrame* frame) {
    int ret, i, j;
    AVFrameSideData *sd;
    AVNetintGeneralSideData *ni_general;
    void *cur_data;
    ret = p_sc_network->model->ni_get_detections(p_sc_network->model_ctx);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to get vqe detection.\n");
        return ret;
    }

    int index = p_sc_network->model_ctx->det_res->index;
    float prob = p_sc_network->model_ctx->det_res->prob;
    // set default category to other
    int category_index = 3;
    for (i = 0; i < 3; i++) {
        for (j = 0; j < 26; j++) {
            if (index == class_id_to_category[i][j]) {
                category_index = i;
            }
        }
    }
    if (fp) {
        char result[128] = { 0 };
        snprintf(result, sizeof(result), "Predicts of frame %d class \"%s\" category \"%s\" prob %f\n",
                 aiframe_number, classes[index], category[category_index], prob);
        // av_log(NULL, AV_LOG_INFO, "result %s", result);
        fwrite(result, strlen(result), 1, fp);
    } else {
        av_log(NULL, AV_LOG_ERROR, "failed to open file.\n");
        return -1;
    }

    sd = av_frame_new_side_data(frame, AV_FRAME_DATA_NETINT_GENERAL_SIDE_DATA,
                                sizeof(AVNetintGeneralSideData));
    if (!sd) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate general sidedata\n");
    }
    else {
        ni_general = (AVNetintGeneralSideData *)sd->data;
        ni_general->count = 0;
        ni_general->type[ni_general->count] = NI_FRAME_AUX_DATA_CATEGORY_CLASSIFY;
        ni_general->size[ni_general->count] = sizeof(ni_category_classify_t);
        cur_data = (ni_category_classify_t *)&(ni_general->data[ni_general->count]);
        ((ni_category_classify_t *)cur_data)->class_type = index;
        ((ni_category_classify_t *)cur_data)->category = category_index;
        ((ni_category_classify_t *)cur_data)->prob = prob;
        ni_general->count++;
    }

    return ret;
}

static int ni_recreate_ai_frame(ni_frame_t *ni_frame, AVFrame *frame) {
    uint8_t *p_data = ni_frame->p_data[0];

    av_log(NULL, AV_LOG_DEBUG,
           "linesize %d/%d/%d, data %p/%p/%p, pixel %dx%d\n",
           frame->linesize[0], frame->linesize[1], frame->linesize[2],
           frame->data[0], frame->data[1], frame->data[2], frame->width,
           frame->height);

    if (frame->format == AV_PIX_FMT_RGB24) {
        /* RGB24 -> BGRP */
        uint8_t *r_data = p_data + frame->width * frame->height * 2;
        uint8_t *g_data = p_data + frame->width * frame->height * 1;
        uint8_t *b_data = p_data + frame->width * frame->height * 0;
        uint8_t *fdata  = frame->data[0];
        int x, y;

        av_log(NULL, AV_LOG_DEBUG,
               "%s(): rgb24 to bgrp, pix %dx%d, linesize %d\n", __func__,
               frame->width, frame->height, frame->linesize[0]);

        for (y = 0; y < frame->height; y++) {
            for (x = 0; x < frame->width; x++) {
                int fpos  = y * frame->linesize[0];
                int ppos  = y * frame->width;
                uint8_t r = fdata[fpos + x * 3 + 0];
                uint8_t g = fdata[fpos + x * 3 + 1];
                uint8_t b = fdata[fpos + x * 3 + 2];

                r_data[ppos + x] = r;
                g_data[ppos + x] = g;
                b_data[ppos + x] = b;
            }
        }
    } else {
        av_log(NULL, AV_LOG_ERROR, "cannot recreate frame: invalid frame format\n");
    }
    return 0;
}

static int generate_ai_frame(struct sc_network *p_sc_network, ni_session_data_io_t *ai_frame,
        AVFrame *avframe, struct SwsContext *sw_scale_ctx, AVFrame *scaled_frame) {
    int ret = 0;
    NiNetworkContext *network_ctx = p_sc_network->network_ctx;
    ni_retcode_t retval;
    ret = sws_scale(sw_scale_ctx, (const uint8_t * const *)avframe->data,
            avframe->linesize, 0, avframe->height, scaled_frame->data, scaled_frame->linesize);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "cannot do sw scale: inframe data 0x%lx, linesize %d/%d/%d/%d, height %d\n",
                (unsigned long)avframe->data, avframe->linesize[0], avframe->linesize[1],
                avframe->linesize[2], avframe->linesize[3], avframe->height);
        return ret;
    }
    retval = ni_ai_frame_buffer_alloc(&ai_frame->data.frame, &network_ctx->network_data);
    if (retval != NI_RETCODE_SUCCESS) {
        av_log(NULL, AV_LOG_ERROR, "cannot allocate sw ai frame buffer\n");
        return NIERROR(ENOMEM);
    }
    ret = ni_recreate_ai_frame(&ai_frame->data.frame, scaled_frame);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "cannot recreate sw ai frame\n");
        return ret;
    }
    return ret;
}

static void free_frame_req(frame_req *p_frame_req)
{
    if (!p_frame_req) {
        return;
    }
    pthread_mutex_lock(&codec_workers->free_lock);
    if (ni_list_empty(&codec_workers->free_list)) {
        ni_list_add_tail(&p_frame_req->list, &codec_workers->free_list);
        pthread_cond_signal(&codec_workers->free_cond);
    } else {
        ni_list_add_tail(&p_frame_req->list, &codec_workers->free_list);
    }
    pthread_mutex_unlock(&codec_workers->free_lock);
}

static void *printAIframe(void *data) {
    int ret = 0;
    frame_req *p_frame_req = NULL;
    AVFrame *avframe = NULL;
    struct network_frame *network_frame = NULL;
    struct network_frame *pending_network_frame = NULL;
    int should_exit = 0;
    NiNetworkContext *network_ctx = p_sc_network->network_ctx;
    int aiframe_number = 0;
    AVPacket enc_packet = {0};
    ni_session_data_io_t ai_input_frame = {0};
    AVFrame *scaled_frame = NULL;
    struct SwsContext *sw_scale_ctx = NULL;

    FILE *outfile = NULL;
    outfile = fopen(codec_workers->output_file,"wb");
    if(!outfile) {
        av_log(NULL, AV_LOG_ERROR, "open output file fail!\n");
        goto end;
    }

    FILE *fp = NULL;
    if (codec_workers->res_output) {
        fp = fopen(codec_workers->res_output, "wb");
    } else {
        int pos_head = refind(codec_workers->input_file, '/');
        int pos_tail = refind(codec_workers->input_file, '.');
        char* substr;
        int len = pos_tail - pos_head - 1;
        if (len > 0){
            substr = malloc(len + 1);
            strncpy(substr, codec_workers->input_file+pos_head+1, len);
            substr[len] = '\0';
        } else {
            len = 0;
            substr = "";
        }
        char name[512] = {0};
        snprintf(name, sizeof(name), "%s_result.txt", substr);
        fp = fopen(name, "wb");
        if (len > 0) {
            free(substr);
        }
        substr = NULL;
    }
    if (!fp) {
        av_log(NULL, AV_LOG_ERROR, "open text file fail!\n");
        goto end;
    }

    if (use_hwframe == false) {
        scaled_frame = av_frame_alloc();
        scaled_frame->width  = codec_workers->model_width;
        scaled_frame->height = codec_workers->model_height;
        scaled_frame->format = AV_PIX_FMT_RGB24;

        if (av_frame_get_buffer(scaled_frame, 32)) {
            av_log(NULL, AV_LOG_ERROR, "cannot allocate scaled frame buffer\n");
            goto end;
        }

        sw_scale_ctx = sws_getContext(dec_ctx->width, dec_ctx->height, AV_PIX_FMT_YUV420P,
                scaled_frame->width, scaled_frame->height, scaled_frame->format,
                SWS_BICUBIC, NULL, NULL, NULL);
        if (!sw_scale_ctx) {
            av_log(NULL, AV_LOG_ERROR, "cannot create sw scale context for scaling\n");
            goto end;
        }
    }

    while(!should_exit)
    {
        if (end == 3 && ni_list_empty(&codec_workers->frm_list) && ni_list_empty(&network_frame_busy_list) && !pending_network_frame)
            break;

        if (pending_network_frame) {
            ret = ni_set_network_input(network_ctx, use_hwframe, &ai_input_frame, NULL,
                    avframe->width, avframe->height, &pending_network_frame->frame, true);
           if (ret == 0) {
                ni_list_add_tail(&pending_network_frame->list, &network_frame_busy_list);
                pending_network_frame = NULL;
            } else if (ret != NIERROR(EAGAIN)) {
                av_log(NULL, AV_LOG_ERROR, "Error while feeding the ai\n");
                should_exit = 1;
                break;
            }
        } else {
            pthread_mutex_lock(&codec_workers->frm_lock);
            if (ni_list_empty(&codec_workers->frm_list) && ni_list_empty(&network_frame_busy_list)) {
                pthread_cond_wait(&codec_workers->frm_cond,
                        &codec_workers->frm_lock);
            }

            // send frame to ai
            if ((!ni_list_empty(&codec_workers->frm_list)) &&
                    !ni_list_empty(&network_frame_free_list)) {
                {
                    p_frame_req = ni_list_first_entry(&codec_workers->frm_list, frame_req, list);
                    ni_list_del(&p_frame_req->list);
                    pthread_mutex_unlock(&codec_workers->frm_lock);

                    avframe = &p_frame_req->avframe;

                    if (!p_frame_req->eof) {
                        if (!use_hwframe) {
                            ret = generate_ai_frame(p_sc_network, &ai_input_frame, avframe, sw_scale_ctx, scaled_frame);
                            if (ret < 0) {
                                av_log(NULL, AV_LOG_ERROR, "cannot generate ai frame\n");
                                should_exit = 1;
                                break;
                            }
                        } else {
                            ai_input_frame.data.frame.p_data[3] = avframe->data[3];
                        }
                    }
                }

                network_frame = ni_list_first_entry(&network_frame_free_list,
                            struct network_frame, list);
                ni_list_del(&network_frame->list);
                if (!p_frame_req->eof) {
                    network_frame->p_frame_req = p_frame_req;
                    ret = ni_set_network_input(network_ctx, use_hwframe, &ai_input_frame, NULL,
                            avframe->width, avframe->height, &network_frame->frame, true);
                    if (ret == 0) {
                        ni_list_add_tail(&network_frame->list, &network_frame_busy_list);
                    } else if (ret == NIERROR(EAGAIN)) {
                        pending_network_frame = network_frame;
                    } else {
                        av_log(NULL, AV_LOG_ERROR, "Error while feeding the ai\n");
                        should_exit = 1;
                        break;
                    }
                } else {
                    network_frame->p_frame_req = p_frame_req;
                    ni_list_add_tail(&network_frame->list, &network_frame_busy_list);
                }
                if (ret == 0) {
                    network_frame->p_frame_req = p_frame_req;

                }
            } else {
                pthread_mutex_unlock(&codec_workers->frm_lock);
            }
        }

        // get packet from ai
        if (!ni_list_empty(&network_frame_busy_list)) {
            network_frame = ni_list_first_entry(&network_frame_busy_list, struct network_frame, list);
            if (!network_frame->p_frame_req->eof) {
                /* pull filtered frames from the filtergraph */
                ret = ni_get_network_output(network_ctx, use_hwframe, &network_frame->frame, false,
                        true, p_sc_network->model_ctx->out_tensor);
                if (ret != 0 && ret != NIERROR(EAGAIN)) {
                    av_log(NULL, AV_LOG_ERROR, "Error when getting output\n");
                    should_exit = 1;
                    break;
                } else if (ret != NIERROR(EAGAIN)) {
                    p_frame_req = network_frame->p_frame_req;
                    ni_list_del(&network_frame->list);
                    ni_list_add_tail(&network_frame->list, &network_frame_free_list);

                    avframe = &p_frame_req->avframe;

                    ret = ni_read_ai(p_sc_network, fp, aiframe_number, avframe);
                    if (ret < 0) {
                        av_log(NULL, AV_LOG_ERROR, "read ai failed\n");
                    }
                    aiframe_number++;
                    //pr_log("AI frame number %d\n", aiframe_number);

                    ret = encode_frame(avframe, &enc_packet, outfile);
                    if (ret != 0) {
                        av_log(NULL, AV_LOG_ERROR, "Error when encode frame: %x\n", ret);
                        break;
                    }
                    av_frame_unref(avframe);

                    free_frame_req(p_frame_req);
                    p_frame_req = NULL;
                }
            } else {
                p_frame_req = network_frame->p_frame_req;
                ni_list_del(&network_frame->list);
                ni_list_add_tail(&network_frame->list, &network_frame_free_list);

                ret = encode_frame(NULL, &enc_packet, outfile);
                if (ret != 0) {
                    av_log(NULL, AV_LOG_ERROR, "Error when flushing encode frame: %x\n", ret);
                    break;
                }
                free_frame_req(p_frame_req);
                p_frame_req = NULL;
            }
        }
    }

end:
    fclose(outfile);
    fclose(fp);
    if (scaled_frame)
        av_frame_free(&scaled_frame);
    if (sw_scale_ctx)
        sws_freeContext(sw_scale_ctx);

    // avoid main thread block
    free_frame_req(p_frame_req);
    p_frame_req = NULL;
    while (!ni_list_empty(&codec_workers->frm_list)) {
        pthread_mutex_lock(&codec_workers->frm_lock);
        p_frame_req = ni_list_first_entry(&codec_workers->frm_list, frame_req, list);
        ni_list_del(&p_frame_req->list);
        pthread_mutex_unlock(&codec_workers->frm_lock);
        free_frame_req(p_frame_req);
        p_frame_req = NULL;
    }
    codec_workers->ait_exit = 1;
    return (void *)0;
}

static void help_usage(void) {
    printf("Usage: \n"
            "-s | --nbg_path          path of model.\n"
            "-i | --input             input file path.\n"
            "-o | --output_file       output file path.\n"
            "-d | --decoder           decoder name.\n"
            "-x | --decoder_params    decoder parameters.\n"
            "-n | --devid             device id.\n"
            "-v | --loglevel          available debug level: warning, info, debug, trace.\n"
            "-e | --encode_name       encode name.\n"
            "-p | --encode_params     encode params.\n"
            "-t | --res_output        the path of txt store the detect result.\n"
            "-h | --help              print this help information.\n");
}

void setup_loglevel(char *loglevel) {
    if (loglevel) {
        if (!strcmp(loglevel, "error")) {
            av_log_set_level(AV_LOG_ERROR);
        } else if (!strcmp(loglevel, "warning")) {
            av_log_set_level(AV_LOG_WARNING);
        } else if (!strcmp(loglevel, "info")) {
            av_log_set_level(AV_LOG_INFO);
        } else if (!strcmp(loglevel, "debug")) {
            av_log_set_level(AV_LOG_DEBUG);
        } else if (!strcmp(loglevel, "trace")) {
            av_log_set_level(AV_LOG_TRACE);
        } else {
            av_log_set_level(AV_LOG_INFO);
        }
    } else {
        av_log_set_level(AV_LOG_INFO);
    }
}

static int create_ai_thread(struct codec_worker *worker) {
    int ret;
    pthread_attr_t attr;

    ret = pthread_attr_init(&attr);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "fail to initialize attr: %d.\n", ret);
        return ret;
    }

    ret = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "fail to set attr detachstate: %d.\n", ret);
        goto end;
    }

    ret = pthread_create(&worker->pthid, NULL, printAIframe, 0);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "creat thread failed!, %d\n", ret);
        goto end;
    }
end:
    if (0 != pthread_attr_destroy(&attr)) {
        av_log(NULL, AV_LOG_ERROR, "failed to destroy pthread attr\n");
    }
    return ret;
}

static int destroy_ai_thread(struct codec_worker *worker) {
    int ret;
    void *result;
    av_log(NULL, AV_LOG_INFO, "try to stop ai thread\n");
    pthread_cond_signal(&worker->frm_cond);
    ret = pthread_join(worker->pthid, &result);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to join ai thread\n");
    }
    if ((int)((long)result)) {
        av_log(NULL, AV_LOG_ERROR, "Bad ai thread result: %d\n", (int)((long)result));
    }
    return 0;
}

static int create_fps_thread(struct codec_worker *worker) {
    int ret;
    pthread_attr_t attr;

    ret = pthread_attr_init(&attr);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "fail to initialize attr: %d.\n", ret);
        return ret;
    }

    ret = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "fail to set attr detachstate: %d.\n", ret);
        goto end;
    }

    ret = pthread_create(&worker->pthtimeid,NULL,printfps,0);
    if (ret !=0) {
        av_log(NULL, AV_LOG_ERROR, "creat thread failed!\n");
    }
end:
    if (0 != pthread_attr_destroy(&attr)) {
        av_log(NULL, AV_LOG_ERROR, "failed to destroy pthread attr\n");
    }
    return ret;
}

static int destroy_fps_thread(struct codec_worker *worker) {
    int ret;
    void *result;
    av_log(NULL, AV_LOG_INFO, "try to stop fps thread\n");
    ret = pthread_join(worker->pthtimeid, &result);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to join fps thread\n");
    }
    if ((int)((long)result)) {
        av_log(NULL, AV_LOG_ERROR, "Bad fps thread result: %d\n", (int)((long)result));
    }

    return ret;
}

static frame_req *get_free_frame_req() {
    frame_req *p_frame_req;
    pthread_mutex_lock(&codec_workers->free_lock);
    while (ni_list_empty(&codec_workers->free_list)) {
        pthread_cond_wait(&codec_workers->free_cond, &codec_workers->free_lock);
    }
    p_frame_req = ni_list_first_entry(&codec_workers->free_list, frame_req, list);
    ni_list_del(&p_frame_req->list);
    pthread_mutex_unlock(&codec_workers->free_lock);

    return p_frame_req;
}

static void put_busy_frame_req(frame_req *p_frame_req) {
    pthread_mutex_lock(&codec_workers->frm_lock);
    if (ni_list_empty(&codec_workers->frm_list)) {
        ni_list_add_tail(&p_frame_req->list, &codec_workers->frm_list);
        pthread_cond_signal(&codec_workers->frm_cond);
    } else {
        ni_list_add_tail(&p_frame_req->list, &codec_workers->frm_list);
    }
    pthread_mutex_unlock(&codec_workers->frm_lock);
}

int main(int argc, char **argv) {
    int i, ret;
    const char *nbg_file = "../../AI_benchmarking_examples/scene_classification.nb";
    int model_width = 256;
    int model_height = 256;
    int model_format = GC620_RGB888_PLANAR;
    AVPacket *dec_packet = NULL;
    frame_req *frame_req = NULL;
    AVFrame *dec_frame = NULL;

    char *loglevel = NULL;
    codec_workers = (struct codec_worker *)calloc(1, sizeof(struct codec_worker));
    if (codec_workers == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate codec workers.\n");
        return EXIT_FAILURE;
    }
    codec_workers->model_width = model_width;
    codec_workers->model_height = model_height;
    int opt, opt_index;
    const char *opt_string = "i:o:d:x:e:p:s:t:n:v:h";
    static struct option long_options[] = {
        {"input",          required_argument, NULL, 'i'},
        {"output",         required_argument, NULL, 'o'},
        {"decoder",        required_argument, NULL, 'd'},
        {"decoder_params", required_argument, NULL, 'x'},
        {"encode_name",    required_argument, NULL, 'e'},
        {"encode_params",  required_argument, NULL, 'p'},
        {"nbg_path",       required_argument, NULL, 's'},
        {"res_output",     required_argument, NULL, 't'},
        {"devid",          required_argument, NULL, 'n'},
        {"loglevel",       required_argument, NULL, 'v'},
        {"help",           no_argument,       NULL, 'h'},
        { NULL,            0,                 NULL,  0 },
    };

    while ((opt = getopt_long(argc, argv, opt_string, long_options, &opt_index)) != -1) {
        switch (opt) {
            case 'i':
                codec_workers->input_file = optarg;
                break;
            case 'o':
                codec_workers->output_file = optarg;
                break;
            case 'd':
                codec_workers->decoder_name = optarg;
                break;
            case 'x':
                codec_workers->decoder_params = optarg;
                if (strcmp(codec_workers->decoder_params, "out=hw") == 0) {
                    use_hwframe = true;
                    printf("set use_hwframe %d\n", use_hwframe);
                }
                break;
            case 'e':
                codec_workers->encoder_name = optarg;
                break;
            case 'p':
                codec_workers->encoder_params = optarg;
                break;
            case 's':
                nbg_file = optarg;
                break;
            case 't':
                codec_workers->res_output = optarg;
                break;
            case 'n':
                codec_workers->devid = atoi(optarg);
                break;
            case 'v':
                loglevel = optarg;
                break;
            case 'h':
                help_usage();
                return EXIT_SUCCESS;
            default:
                help_usage();
                return EXIT_FAILURE;
        }
    }
    setup_loglevel(loglevel);

// Register all formats, codecs, filters
#if LIBAVFORMAT_VERSION_MAJOR < 58
    av_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif
#if LIBAVCODEC_VERSION_MAJOR < 58
    avcodec_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif
#if LIBAVFILTER_VERSION_MAJOR < 7
    avfilter_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif

    if (access(nbg_file, R_OK) != 0) {
        av_log(NULL, AV_LOG_ERROR, "cannot find nbg: %s\n", nbg_file);
        return EXIT_FAILURE;
    }
    if (codec_workers->input_file == NULL) {
        av_log(NULL, AV_LOG_ERROR, "input file name must be specified.\n");
        return EXIT_FAILURE;
    }
    if (codec_workers->output_file == NULL) {
        av_log(NULL, AV_LOG_ERROR, "output file name must be specified.\n");
        return EXIT_FAILURE;
    }
    if (!codec_workers->decoder_name || !codec_workers->encoder_name) {
        av_log(NULL, AV_LOG_ERROR, "decoder_name name must be specified.\n");
        return EXIT_FAILURE;
    }
    if (init_shared_thread_data(codec_workers) != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to initialize common data\n");
        return EXIT_FAILURE;
    }

    ret = init_network_struct(model_width, model_height, model_format, nbg_file,
                              use_hwframe, codec_workers->devid);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to initialize network struct\n");
        return EXIT_FAILURE;
    }

    if ((ret = open_input_file()) < 0)
        goto end;

    dec_frame = av_frame_alloc();
    dec_packet = av_packet_alloc();
    if (!dec_frame || !dec_packet) {
        av_log(NULL, AV_LOG_ERROR, "Could not allocate dec_frame or dec_packet\n");
        goto end;
    }

    timer_start = av_gettime_relative();

    if ((ret = create_fps_thread(codec_workers)) != 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to create fps thread\n");
        goto end;
    }

    if ((ret = create_ai_thread(codec_workers)) != 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to create ai thread\n");
        goto stop_fpst;
    }

    /* read all packets */
    while (1) {
        if (codec_workers->ait_exit) {
            break;
        }
        if ((ret = av_read_frame(ifmt_ctx, dec_packet)) < 0) {
            if(end == 2) {
                frame_req = get_free_frame_req();
                frame_req->eof = 1;
                put_busy_frame_req(frame_req);
                break;
            }
            end = 1;
        }
        if (dec_packet->stream_index == video_stream_index || end == 1) {
            if(!end) {
                ret = avcodec_send_packet(dec_ctx, dec_packet);
            } else{
                ret = avcodec_send_packet(dec_ctx, NULL);
            }

            if (ret < 0 && ret != AVERROR_EOF) {
                av_log(NULL, AV_LOG_ERROR, "Error while sending a packet to the decoder\n");
                break;
            }
            while (ret >= 0 || end == 1) {
                ret = avcodec_receive_frame(dec_ctx, dec_frame);
                if (ret == AVERROR(EAGAIN)) {
                    break;
                } else if(ret == AVERROR_EOF){
                    end = 2;
                    break;
                } else if (ret < 0) {
                    av_log(NULL, AV_LOG_ERROR, "Error while receiving a frame from the decoder\n");
                    goto stop_ait;
                }

                dec_frame->pts = dec_frame->best_effort_timestamp;
                frame_num++;

                av_log(NULL, AV_LOG_DEBUG, "decoded frame_num %d\n", frame_num);

                frame_req = get_free_frame_req();
                frame_req->eof = 0;
                ret = av_frame_ref(&frame_req->avframe, dec_frame);
                if (ret < 0) {
                    av_log(NULL, AV_LOG_ERROR, "copy frame to filter failed\n");
                    goto stop_ait;
                }
                put_busy_frame_req(frame_req);
                av_frame_unref(dec_frame);
            }
        }
        av_packet_unref(dec_packet);
    }

stop_ait:
    av_log(NULL, AV_LOG_INFO, "main thread end\n");
    end = 3;
    destroy_ai_thread(codec_workers);
stop_fpst:
    stop_fps = true;
    destroy_fps_thread(codec_workers);
end:
    avcodec_free_context(&dec_ctx);
    avcodec_free_context(&enc_ctx);
    avformat_close_input(&ifmt_ctx);
    av_frame_free(&dec_frame);
    av_packet_free(&dec_packet);

    cleanup_shared_thread_data(codec_workers);
    free(codec_workers);
    codec_workers = NULL;
    free_network_struct(p_sc_network);
    p_sc_network = NULL;

    if (ret < 0 && ret != AVERROR_EOF) {
        av_log(NULL, AV_LOG_ERROR, "Error occurred: %s\n", av_err2str(ret));
        exit(1);
    }
    return 0;
}
