#!/usr/bin/env bash
dest_path="./FFmpeg"

# parse a flag with an arg in or after it
# $1 flag pattern, $2 entire flag arg, $3 arg after flag arg
# return 1 if path is in second arg (separated by space), else return 0. Store path in $extract_arg_ret
extract_arg () {
    unset extract_arg_ret
    # check valid arg flag
    if [ -n "$(printf "%s" ${2} | grep -Eoh "${1}")" ]; then
        # check if path string is connected by '=' or is in following arg
        if [ -n "$(echo "${2}" | grep -Eoh "${1}=")" ]; then
            arg_str=`printf "%s" "${2}" | grep -Poh "${1}=\K.+"`;
            # trim out leading and trailing quotation marks
            extract_arg_ret=`echo "${arg_str}" | sed -e 's/^\(["'\'']\)//' -e 's/\(["'\'']\)$//'`;
            return 0;
        elif [ -n "$(printf "%s" ${2} | grep -Eoh "^${1}$")" ]; then
            arg_str="${3}";
            # trim out leading and trailing quotation marks
            extract_arg_ret=`printf "%s" "${arg_str}" | sed -e 's/^\(["'\'']\)//' -e 's/\(["'\'']\)$//'`;
            return 1;
        else
            echo "Unknown option '$2', exiting";
            exit 1;
        fi
    else
        echo "Target flag '$1' not found in '$2', exiting"; exit 1;
    fi
}

while [ "$1" != "" ]; do
    case $1 in
        -h | --help)   echo "Usage: ./install_ffmpeg_headers.sh [OPTION]";
                       echo "Copy all ffmpeg header files marked by lib*/Makefile to a destination folder."
                       echo "The files will be organized into sub-folders based on library type. This is"
                       echo "similar to installation of headers by Makefile on Linux into "
                       echo "/usr/local/includes/lib*/*.h"
                       echo;
                       echo "Example: ./install_ffmpeg_headers.sh --ff_path ../../FFmpeg-n4.2.1"
                       echo "         --dest_path ./3rd_libs/FFmpeg";
                       echo;
                       echo "Options:";
                       echo "-h, --help     display this help and exit";
                       echo "--ff_path      path to FFmpeg base folder";
                       echo "--dest_path    path to install folder (default: ./FFmpeg)";
                       exit 0
        ;;
        --ff_path | --ff_path=*)        extract_arg "\-\-ff_path" "$1" "$2"; prev_rc=$?;
                                        if [ "$prev_rc" -eq 1 ]; then shift; fi
                                        ff_path=$extract_arg_ret
        ;;
        --dest_path | --dest_path=*)    extract_arg "\-\-dest_path" "$1" "$2"; prev_rc=$?;
                                        if [ "$prev_rc" -eq 1 ]; then shift; fi
                                        dest_path=$extract_arg_ret
        ;;
        *)             echo "Usage: ./install_ffmpeg_headers.sh [OPTION]...";
                       echo "Try './install_ffmpeg_headers.sh --help' for more information";
                       exit 1
        ;;
    esac
    shift
done

if [ -z $ff_path ]; then
    echo "Error: no --ff_path provided"
    exit 1
fi

mkdir $dest_path
for subdir in $(ls ${ff_path}/lib*/Makefile | grep -Po "lib\w+"); do
    grep -Pzo "\nHEADERS =(\s+(\w+\.h)\s+\\\\\n)+" ${ff_path}/${subdir}/Makefile > temp_headers.txt
    headers_arr=($(grep -Pao "\w+\.h" temp_headers.txt))
    rm temp_headers.txt
    if [ ${#headers_arr[@]} -eq 0 ]; then
        continue
    fi
    mkdir ${dest_path}/${subdir}/
    for header in ${headers_arr[@]}; do
        echo cp ${ff_path}/${subdir}/${header} ${dest_path}/${subdir}/
        cp ${ff_path}/${subdir}/${header} ${dest_path}/${subdir}/
    done
done
