/*
* Copyright (c) 2001 Fabrice Bellard
*
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to deal
* in the Software without restriction, including without limitation the rights
* to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
* copies of the Software, and to permit persons to whom the Software is
* furnished to do so, subject to the following conditions:
*
* The above copyright notice and this permission notice shall be included in
* all copies or substantial portions of the Software.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
* THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
* OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
* THE SOFTWARE.
*/

/**
 * @file
 * API example for encode and AI
 * @example ni_encode_ai_test.c
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <pthread.h>
#include <getopt.h>
#include <libavcodec/avcodec.h>
#include <libavutil/opt.h>
#include <libavutil/imgutils.h>
#include <libavutil/time.h>
#include <libavutil/frame.h>
#include <libavfilter/avfilter.h>
#include <libavformat/avformat.h>
#include <libavfilter/buffersink.h>
#include <libavfilter/buffersrc.h>
#include <libswscale/swscale.h>


AVCodec *dec;
AVFormatContext *ifmt_ctx;

AVFilterContext *buffersink_ctx;
AVFilterContext *buffersrc_ctx;
AVFilterGraph *filter_graph;

AVFilterContext *over_buffersink_ctx;
AVFilterContext *over_buffersrc_ctx;
AVFilterGraph *over_filter_graph;


AVFrame *frame;
AVFrame *frame1;
AVFrame *filt_frame;
AVFrame *filt_frame1;
int bitrate = 1024000;
int fps = 30;
int frame_num = 0,end = 0,encode_num = 0;
int64_t timer_start;
int should_exit = 0;
int lock_frame = 0;
int begin_roiset = 0;
int roi_x = 0,roi_y = 0;
struct codec_worker {
    const char *encoder_name;
    const char *pic_name;
    const char *input_file;
    const char *output_file;
    const char *xcoder_params;
    const char *filter_dsc;
    const AVCodec *codec;
    AVCodecContext *codec_ctx;
    int input_format;
    int select_frame;
    int devid;
    int enc_idx;
    int src_height;
    int src_width;
    int dst_height;
    int dst_width;
    int pic_height;
    int pic_width;
};

struct codec_worker *codec_work = NULL;

void setup_loglevel(char *loglevel)
{
    int i;
    const struct { const char *name; int level; } log_levels[] =
    {
        { "quiet"  , AV_LOG_QUIET   },
        { "panic"  , AV_LOG_PANIC   },
        { "fatal"  , AV_LOG_FATAL   },
        { "error"  , AV_LOG_ERROR   },
        { "warning", AV_LOG_WARNING },
        { "info"   , AV_LOG_INFO    },
        { "verbose", AV_LOG_VERBOSE },
        { "debug"  , AV_LOG_DEBUG   },
        { "trace"  , AV_LOG_TRACE   },
    };

    for (i = 0; i < (sizeof(log_levels) / sizeof((log_levels)[0])); i++)
    {
        if (!strcmp(log_levels[i].name, loglevel))
        {
            av_log_set_level(log_levels[i].level);
        }
    }
}

static int init_filters(int fps)
{
    char args[512];
    int ret = 0;
    const AVFilter *buffersrc  = avfilter_get_by_name("buffer");
    const AVFilter *buffersink = avfilter_get_by_name("buffersink");
    AVFilterInOut *outputs = avfilter_inout_alloc();
    AVFilterInOut *inputs  = avfilter_inout_alloc();
    //AVRational time_base = codec_ctx->time_base;

    enum AVPixelFormat pix_fmts[] = { AV_PIX_FMT_NI_QUAD,AV_PIX_FMT_BGRA,AV_PIX_FMT_YUV420P,AV_PIX_FMT_NONE };

    filter_graph = avfilter_graph_alloc();
    if (!outputs || !inputs || !filter_graph) {
        ret = AVERROR(ENOMEM);
        goto end;
    }
    /* buffer video source: the decoded frames from the decoder will be inserted here. */
    snprintf(args, sizeof(args),
            "video_size=%dx%d:pix_fmt=%d:time_base=1/%d:pixel_aspect=0/1",
            codec_work->src_width, codec_work->src_height, codec_work->input_format,
            fps);

    ret = avfilter_graph_create_filter(&buffersrc_ctx, buffersrc, "input",
                                        args, NULL, filter_graph);
    if (ret < 0) {
    av_log(NULL, AV_LOG_ERROR, "Cannot create buffer source\n");
    goto end;
    }

    /* buffer video sink: to terminate the filter chain. */
    ret = avfilter_graph_create_filter(&buffersink_ctx, buffersink, "output",
                                        NULL, NULL, filter_graph);

    if (ret < 0) {
    av_log(NULL, AV_LOG_ERROR, "Cannot create buffer sink\n");
    goto end;
    }

    ret = av_opt_set_int_list(buffersink_ctx, "pix_fmts", pix_fmts,
                            AV_PIX_FMT_NONE, AV_OPT_SEARCH_CHILDREN);

    if (ret < 0) {
    av_log(NULL, AV_LOG_ERROR, "Cannot set output pixel format\n");
    goto end;
    }

    /*
    * Set the endpoints for the filter graph. The filter_graph will
    * be linked to the graph described by filters_descr.
    */

    /*
    * The buffer source output must be connected to the input pad of
    * the first filter described by filters_descr; since the first
    * filter input label is not specified, it is set to "in" by
    * default.
    */
    outputs->name       = av_strdup("input");
    outputs->filter_ctx = buffersrc_ctx;
    outputs->pad_idx    = 0;
    outputs->next       = NULL;

    /*
    * The buffer sink input must be connected to the output pad of
    * the last filter described by filters_descr; since the last
    * filter output label is not specified, it is set to "out" by
    * default.
    */
    inputs->name       = av_strdup("output");
    inputs->filter_ctx = buffersink_ctx;
    inputs->pad_idx    = 0;
    inputs->next       = NULL;

    if ((ret = avfilter_graph_parse_ptr(filter_graph, codec_work->filter_dsc,
                            &inputs, &outputs, NULL)) < 0)
    {
        goto end;
    }

    if ((ret = avfilter_graph_config(filter_graph, NULL)) < 0)
    {
        goto end;
    }

end:
    avfilter_inout_free(&inputs);
    avfilter_inout_free(&outputs);

    return ret;
}

static int init_overlay_filters(int fps,char *filter_dsc)
{
    char args[512];
    int ret = 0;
    int i;
    const AVFilter *buffersrc  = avfilter_get_by_name("buffer");
    const AVFilter *buffersink = avfilter_get_by_name("buffersink");
    AVFilterInOut *outputs = avfilter_inout_alloc();
    AVFilterInOut *inputs  = avfilter_inout_alloc();

    enum AVPixelFormat pix_fmts[] = { AV_PIX_FMT_NI_QUAD,AV_PIX_FMT_BGRA,AV_PIX_FMT_YUV420P,AV_PIX_FMT_NONE };

    over_filter_graph = avfilter_graph_alloc();
    if (!outputs || !inputs || !over_filter_graph) {
        ret = AVERROR(ENOMEM);
        goto end;
    }
    /* buffer video source: the decoded frames from the decoder will be inserted here. */
    snprintf(args, sizeof(args),
            "video_size=%dx%d:pix_fmt=%d:time_base=1/%d:pixel_aspect=0/1",
            codec_work->src_width, codec_work->src_height, codec_work->input_format,
            fps);

    ret = avfilter_graph_create_filter(&over_buffersrc_ctx, buffersrc, "in2",
                                        args, NULL, over_filter_graph);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot create buffer source\n");
        goto end;
    }

    /* buffer video sink: to terminate the filter chain. */
    ret = avfilter_graph_create_filter(&over_buffersink_ctx, buffersink, "out",
                                        NULL, NULL, over_filter_graph);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot create buffer sink\n");
        goto end;
    }
    ret = av_opt_set_int_list(over_buffersink_ctx, "pix_fmts", pix_fmts,
                            AV_PIX_FMT_NONE, AV_OPT_SEARCH_CHILDREN);

    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot set output pixel format\n");
        goto end;
    }

    /*
    * Set the endpoints for the filter graph. The filter_graph will
    * be linked to the graph described by filters_descr.
    */

    /*
    * The buffer source output must be connected to the input pad of
    * the first filter described by filters_descr; since the first
    * filter input label is not specified, it is set to "in" by
    * default.
    */
    outputs->name       = av_strdup("in2");
    outputs->filter_ctx = over_buffersrc_ctx;
    outputs->pad_idx    = 0;
    outputs->next       = NULL;

    /*
    * The buffer sink input must be connected to the output pad of
    * the last filter described by filters_descr; since the last
    * filter output label is not specified, it is set to "out" by
    * default.
    */
    inputs->name       = av_strdup("out");
    inputs->filter_ctx = over_buffersink_ctx;
    inputs->pad_idx    = 0;
    inputs->next       = NULL;

    if ((ret = avfilter_graph_parse_ptr(over_filter_graph, filter_dsc,
                            &inputs, &outputs, NULL)) < 0)
    {
        goto end;
    }

    if ((ret = avfilter_graph_config(over_filter_graph, NULL)) < 0)
    {
        goto end;
    }
    for (i = 0; i < over_filter_graph->nb_filters; i++) //find overlay filter
    {
        AVFilterContext* filter_ctxn = over_filter_graph->filters[i];
        printf("filter_ctxn->name = %s\n",filter_ctxn->name);
    }

end:
    avfilter_inout_free(&inputs);
    avfilter_inout_free(&outputs);

    return ret;
}

static void init_encode(int fps,int bitrate)
{
    int rc;
    const AVCodec *codec;
    AVCodecContext *codec_ctx;

#if LIBAVCODEC_VERSION_MAJOR < 58
    // Register all codecs
    avcodec_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif

    // Find video encoder codec selected
    codec = avcodec_find_encoder_by_name(codec_work->encoder_name);
    if (!codec)
    {
        av_log(NULL, AV_LOG_ERROR, "Codec '%s' not found\n", codec_work->encoder_name);
        exit(1);
    }

    // Allocate codec context for encoding
    codec_ctx = avcodec_alloc_context3(codec);
    if (!codec_ctx)
    {
        av_log(NULL, AV_LOG_ERROR, "Could not allocate video codec context\n");
        exit(1);
    }

    codec_ctx->bit_rate = bitrate;
    codec_ctx->width = codec_work->dst_width; // resolution must be divisible by 2
    codec_ctx->height = codec_work->dst_height;
    codec_ctx->time_base = (AVRational){1, fps};
    codec_ctx->framerate = (AVRational){fps, 1};
    codec_ctx->pix_fmt = AV_PIX_FMT_NI_QUAD;
    codec_ctx->sw_pix_fmt = AV_PIX_FMT_YUV420P;

    if ((codec->id == AV_CODEC_ID_H264) || (codec->id == AV_CODEC_ID_H265))
    {
        av_opt_set(codec_ctx->priv_data, "xcoder-params", codec_work->xcoder_params, 0);
        if (codec_work->enc_idx >= 0)
            av_opt_set_int(codec_ctx->priv_data, "enc", codec_work->enc_idx, 0);
    }
    else
    {
        av_log(NULL, AV_LOG_ERROR, "codec id %d not supported.\n", codec->id);
        exit(1);
    }

    // Open encoder
    rc = avcodec_open2(codec_ctx, codec, NULL);
    if (rc < 0)
    {
        av_log(NULL, AV_LOG_ERROR, "Could not open codec: %s\n", av_err2str(rc));
        exit(1);
    }

    codec_work->codec = codec;
    codec_work->codec_ctx = codec_ctx;
}

/* Send one frame to encoder, attempt to read all available encoder packets from encoder,
writing packets to output stream. If run with with *frame=NULL, flush the encoder of
remaining packets until encoder reports EOF */
static void encode(AVCodecContext *enc_ctx, AVFrame *frame, AVPacket *pkt,
                FILE *outfile)
{
    int rc;
    int attempt_cnt = 1;
    static int send_cnt = 0;
    static int recv_cnt = 0;
    // Send frame to encoder. If frame=NULL, this signals end of stream
    if (frame)
    av_log(NULL, AV_LOG_DEBUG,"Send %d frame pts %3"PRId64"\n",++send_cnt, frame->pts);
    else
    av_log(NULL, AV_LOG_DEBUG,"Send done, flush encoder\n");

    do
    {
        rc = avcodec_send_frame(enc_ctx, frame);
        // Encoder is too busy to receive frame, try sending frame again
        if (rc == AVERROR(EAGAIN))
        {
            av_log(NULL, AV_LOG_DEBUG, "Resending frame. Try #%d.\n", ++attempt_cnt);
            continue;
        }
        // Frame send issue occured
        else if (rc < 0)
        {
            av_log(NULL, AV_LOG_ERROR, "Error sending frame for encoding. RC=%d\n", AVUNERROR(rc));
            avcodec_free_context(&enc_ctx);
            exit(1);
        }
        // Frame sent successfully
        else
            break;
    }
    while (1);

    /* Run this loop if a frame was sucessfully sent during avcodec_send_frame() OR
        no frame was sent because this function is in flushing mode OR
        a frame was sucessfully received by avcodec_receive_packet() */

    while (rc >= 0 || ! frame )
    {
        av_log(NULL, AV_LOG_DEBUG, "Receiving packet\n");

        rc = avcodec_receive_packet(enc_ctx, pkt);

        if (frame == NULL)
            av_log(NULL, AV_LOG_DEBUG, "Draining packet rc=%x, pkt size=%5d\n", rc, pkt->size);

        // No packets in encoder output buffer but not end of stream either
        if (rc == AVERROR(EAGAIN))
        {
            // If function not in flushing mode
            if (frame)
            {
                return ;// Exit this function to send next packet
            }
            else
                continue; // Loop in this function to flush encoder
        }
        // End of stream detected
        else if (rc == AVERROR_EOF)
        {
            av_log(NULL, AV_LOG_DEBUG, "Encoder EOF reached\n");
            return;
        }
        // Packet receive error occured
        else if (rc < 0)
        {
            av_log(NULL, AV_LOG_ERROR, "Error during encoding\n");
            avcodec_free_context(&enc_ctx);
            exit(1);
        }
        // Packet sucessfully received
        recv_cnt++;
        av_log(NULL, AV_LOG_DEBUG,"Received packet #%d, pts=\033[K%3"PRId64"\n", recv_cnt, pkt->pts);

        av_log(NULL, AV_LOG_DEBUG,"Write %d packet pts %3"PRId64" (size=%5d)\n",recv_cnt,
                pkt->pts, pkt->size);
        encode_num++;
        fwrite(pkt->data, 1, pkt->size, outfile);
        av_packet_unref(pkt);
    }
}

static void print_report(int is_last_report, int64_t timer_start,
    int64_t cur_time, unsigned long frame_number)
{
    static int64_t last_time = -1;
    float t;
    char buf[1024];
    float fps;

    if (!is_last_report) {
        if (last_time == -1) {
            last_time = cur_time;
            return;
        }
        if ((cur_time - last_time) < 500000)
            return;
        last_time = cur_time;
    }

    t = (cur_time - timer_start) / 1000000.0;

    fps = t > 1 ? frame_number / t : 0;

    const char end = is_last_report ? '\n' : '\r';

    buf[0] = '\0';
    snprintf(buf + strlen(buf), sizeof(buf) - strlen(buf), "frame=%5ld fps=%3.*f ",
            frame_number, (fps < 9.95), fps);
    fprintf(stderr, "%s   %c", buf, end);
    fflush(stderr);
}

static void *printAIframe(void *arg)
{
    int ret = 0,flag=0;

    end = 2;
    while(1)
    {
        if(end == 1)
            break;

        if(lock_frame == 1 && flag == 0)
        {
            if ((ret = av_buffersrc_add_frame_flags(buffersrc_ctx, frame1, AV_BUFFERSRC_FLAG_KEEP_REF)) < 0) {
                av_log(NULL, AV_LOG_ERROR, "Error2 while feeding the filtergraph\n");
                break;
            }
            flag = 1;
            /* pull filtered frames from the filtergraph */
            while (1) {
                ret = av_buffersink_get_frame(buffersink_ctx, filt_frame);

                if (ret == AVERROR(EAGAIN))
                {
                    break;
                }
                else if(ret == AVERROR_EOF)
                {
                    end = 3;
                    should_exit++;
                    return NULL;
                }
                else if (ret < 0)
                {
                    end = 3;
                    should_exit++;
                    return NULL;
                }

                if(filt_frame->side_data && (*filt_frame->side_data)->data)
                {
                    AVRegionOfInterest *roi;
                    roi = (AVRegionOfInterest*)(*filt_frame->side_data)->data;
                    //   printf(" top %d, bottom %d, left %d, right %d, qpo %d/%d,size = %d\n",
                    //           roi[0].top, roi[0].bottom, roi[0].left, roi[0].right,
                    //           roi[0].qoffset.num, roi[0].qoffset.den,roi[0].self_size);
                    roi_x = (roi[0].left + roi[0].right)/2 - codec_work->pic_width/2;
                    roi_y = (roi[0].top + roi[0].bottom)/2 - codec_work->pic_height/2;
                    if(roi_x < 0)
                        roi_x = 0;
                    if(roi_y < 0)
                        roi_y = 0;
                    roi = NULL;
                    begin_roiset = 1;
                }

                av_frame_unref(filt_frame);
                lock_frame = 0;
                flag = 0;
            }
            av_frame_unref(frame1);
        }
        usleep(100);
    }
    should_exit++;
    end = 3;

    return NULL;
}

static void *encode_work(void *arg)
{
    FILE *input_fp, *output_fp;
    int i, rc;
    AVPacket *pkt;
    unsigned long frame_cnt;
    unsigned int read_size;
    unsigned long inputfile_size;

    input_fp = fopen(codec_work->input_file, "rb");
    if (input_fp == NULL)
    {
        av_log(NULL, AV_LOG_ERROR, "Could not to open inputfile: %s\n", codec_work->input_file);
        exit (1);
    }
    // Check resolution of input file
    fseek(input_fp, 0, SEEK_END);
    inputfile_size = ftell(input_fp);
    if(codec_work->input_format == AV_PIX_FMT_BGRA)
    {
        if (inputfile_size % (codec_work->src_width*codec_work->src_height*4) != 0)
        {
            av_log(NULL, AV_LOG_ERROR, "Size of inputfile is not integer multiple of resolution.\n"
                    "Either input file has partial frames, or input resolution is wrong.\n");
            exit(1);
        }
        frame_cnt = inputfile_size / (codec_work->src_width*codec_work->src_height*4);

    }
    else if(codec_work->input_format == AV_PIX_FMT_YUV420P)
    {
        if (inputfile_size % (codec_work->src_width*codec_work->src_height*3/2) != 0)
        {
            av_log(NULL, AV_LOG_ERROR, "Size of inputfile is not integer multiple of resolution.\n"
                    "Either input file has partial frames, or input resolution is wrong.\n");
            exit(1);
        }
        frame_cnt = inputfile_size / (codec_work->src_width*codec_work->src_height*3/2);
    }
    av_log(NULL, AV_LOG_INFO, "inputfile size=%lu, number of frames = %lu.\n", inputfile_size, frame_cnt);
    fseek(input_fp, 0, SEEK_SET);

    char flit1_dsc[256] = {0};
    snprintf(flit1_dsc, sizeof(flit1_dsc),"movie=%s,ni_quadra_hwupload=0[logo];[in2]ni_quadra_hwupload=0,ni_quadra_scale=%d:%d,ni_quadra_scale=format=0[main];[main][logo]ni_quadra_overlay=10:10[out]",\
            codec_work->pic_name,codec_work->dst_width,codec_work->dst_height);

    init_overlay_filters(fps,flit1_dsc);
    init_filters(fps);
    init_encode(fps,bitrate);

    // Open outputfile
    output_fp = fopen(codec_work->output_file, "wb");
    if (!output_fp)
    {
    av_log(NULL, AV_LOG_ERROR, "Could not open outputfile: %s\n", codec_work->output_file);
    exit(1);
    }

    // Allocate packet for encoder output
    pkt = av_packet_alloc();
    if (!pkt)
    {
        av_log(NULL, AV_LOG_ERROR, "Could not allocate AVPacket\n");
        exit(1);
    }
    // Allocate frame object

    frame1 = av_frame_alloc();
    filt_frame = av_frame_alloc();
    filt_frame1 = av_frame_alloc();
    if ( !frame1 || !filt_frame || !filt_frame1)
    {
        av_log(NULL, AV_LOG_ERROR, "Could not allocate AVFrame\n");
        exit(1);
    }

    // Repeatedly loop input if selected by user
    fseek(input_fp, 0, SEEK_SET);
    timer_start = av_gettime_relative();
    for (i = 0; i < frame_cnt; i++)
    {
        frame = av_frame_alloc();
        if(!frame)
        {
            av_log(NULL, AV_LOG_ERROR,"frame alloc failed\n");
            exit(1);
        }

        frame->format = codec_work->input_format;
        frame->width  = codec_work->src_width;
        frame->height = codec_work->src_height;
        // Create frame buffer with 32B alignment
        rc = av_frame_get_buffer(frame, 32);
        if (rc < 0)
        {
            av_log(NULL, AV_LOG_ERROR, "Could not allocate the AVFrame buffers\n");
            exit(1);
        }

        // Make sure the frame data is writable
        rc = av_frame_make_writable(frame);
        if (rc < 0)
        {
            av_log(NULL, AV_LOG_ERROR, "av_frame_make_writable() error.\n");
            exit(1);
        }
        if(codec_work->input_format == AV_PIX_FMT_BGRA) //read bgra
        {
            read_size = fread(frame->data[0], 1,codec_work->src_width*codec_work->src_height*4,input_fp);
            if (read_size != codec_work->src_width*codec_work->src_height*4) {
            av_log(NULL, AV_LOG_ERROR, "Failed to read BGRA. read_size=%u.\n", read_size);
            exit(1);
            }
        }
        else if(codec_work->input_format == AV_PIX_FMT_YUV420P) //read yuv
        {
            int j;
            // Read data for Y into frame buffer
            for (j = 0; j < codec_work->src_height; j++)
            {
                read_size = fread(&frame->data[0][0] + j * frame->linesize[0], codec_work->src_width, 1, input_fp);
                if (read_size != 1) {
                    av_log(NULL, AV_LOG_ERROR, "Failed to read Y. read_size=%u.\n", read_size);
                    exit(1);
                }
            }
            // Read data for U into frame buffer
            for (j = 0; j < codec_work->src_height / 2; j++)
            {
                read_size = fread(&frame->data[1][0] + j * frame->linesize[1], codec_work->src_width / 2, 1, input_fp);
                if (read_size != 1) {
                    av_log(NULL, AV_LOG_ERROR, "Failed to read U. read_size=%u.\n", read_size);
                    exit(1);
                }
            }
            // Read data for V into frame buffer
            for (j = 0; j < codec_work->src_height / 2; j++)
            {
                read_size = fread(&frame->data[2][0] + j * frame->linesize[2], codec_work->src_width / 2, 1, input_fp);
                if (read_size != 1)
                {
                    av_log(NULL, AV_LOG_ERROR, "Failed to read V. read_size=%u.\n", read_size);
                    exit(1);
                }
            }
        }

        frame->pts = i;

        if(frame_num%codec_work->select_frame == 0 && end != 3 && lock_frame == 0)
        {
            rc = av_frame_ref(frame1, frame);
            if (rc < 0) {
                av_frame_free(&frame1);
                return NULL;
            }
            lock_frame = 1;
        }
        if(begin_roiset == 1)
        {
            //use roi data (x,y) to reset overlay parameters
            char str1[20];
            char str2[20];
            snprintf(str1,sizeof(str1),"%d",roi_x);
            snprintf(str2,sizeof(str2),"%d",roi_y);
            AVFilterContext* filter_ctx1 = over_filter_graph->filters[7];
            av_opt_set(filter_ctx1->priv, "x", str1, 0 );
            av_opt_set(filter_ctx1->priv, "y", str2, 0 );
            begin_roiset = 0;
        }

        if ((rc = av_buffersrc_add_frame_flags(over_buffersrc_ctx, frame, AV_BUFFERSRC_FLAG_KEEP_REF)) < 0) {
            av_log(NULL, AV_LOG_ERROR, "Error1 while feeding the filtergraph\n");
            break;
        }
        frame_num++;
        /* pull filtered frames from the filtergraph */
        while (1) {
            rc = av_buffersink_get_frame(over_buffersink_ctx, filt_frame1);
            if (rc == AVERROR(EAGAIN))
            {
                break;
            }
            else if(rc == AVERROR_EOF)
            {
                break;
            }
            else if (rc < 0)
                break;

            encode(codec_work->codec_ctx,filt_frame1,pkt,output_fp);
            av_frame_unref(filt_frame1);
        }
        av_frame_unref(frame);
    }
    // Flush encoder
    encode(codec_work->codec_ctx, NULL, pkt, output_fp);
    end = 1;
    fclose(output_fp);
    fclose(input_fp);
    avcodec_free_context(&codec_work->codec_ctx);
    avfilter_graph_free(&over_filter_graph);
    avfilter_graph_free(&filter_graph);
    av_frame_free(&frame);
    av_packet_free(&pkt);
    should_exit++;

    return NULL;
}
static void help_usage(void)
{
    printf("This example program takes in yuv420p or bgra input and upload swframes to\n"
           "hwframes. Then select 1 frame every X frames to do ROI. To visualize the target\n"
           "of ROI, an overlay image may be applied. Note, the filter must use names 'input'\n"
           "and 'output'.\n"
           "\n"
           "Usage example:\n"
           "  ./ni_encode_ai_test -s 1920x1080 -v yvu420p -r 25 -i vidyo4_720p_60fps.yuv \\\n"
           "  -p box_190x290.png -c h264_ni_quadra_enc -e 0 -x \"RcEnable=1:bitrate=2000000\" \\\n"
           "  -m 2 -f \"[input]ni_quadra_hwupload=0,ni_quadra_roi=nb=./yolov5_1face_642F05EA.nb:qpoffset=-0.6[output]\" \\\n"
           "  -k 1280x720 -o vidyo4_720p60_overlay_roi.h264\n"
           "\n"
           "Usage:\n"
           "  -s | --src_size        Size of input yuv in [width]x[height].\n"
           "  -k | --dst_size        Size of output video in [width]x[height].\n"
           "  -x | --xcoder_param    Xcoder parameters. See Integration and Programing Guide for details.\n"
           "  -i | --input           Input yuv file.\n"
           "  -o | --output          Output video file.\n"
           "  -c | --encode_name     Codec name. [h264_ni_quadra_enc, h265_ni_quadra_enc].\n"
           "  -r | --fps             FPS of output.\n"
           "  -b | --bitrate         Bitrate in bits/second.\n"
           "  -e | --enc             Encoder device index to use. See ni_rsrc_mon.\n"
           "  -f | --filter_dsc      Use filter. This must use name 'input' and 'output'.\n"
           "  -m | --select_frame    frame period to re-acquire target for overlay and ROI.\n"
           "  -p | --picture_name    Picture name. Use it to overlay.\n"
           "  -l | --picture_size    Picture size in [width]x[height]\n"
           "  -v | --input_format    Input_format. (eg. bgra, yuv420p)\n"
           "  -g | --log_level       Log level for messages. [NONE, FATAL, ERROR, INFO, DEBUG, TRACE]\n"
           "  -h | --help            Help info.\n");
}

int main(int argc, char **argv)
{
    char *n;      // used for parsing width and height from --size
    // Input arg handling
    int opt;
    int opt_index;
    char *input_format;
    char *loglevel = NULL;
    pthread_t pthencode;
    pthread_t pthid;
    const char *opt_string = "s:k:x:i:o:c:r:b:e:f:g:m:p:l:v:h";
    static struct option long_options[] =
    {
        {"src_size",           required_argument, NULL, 's'},
        {"dst_size",           required_argument, NULL, 'k'},
        {"xcoder_params",      required_argument, NULL, 'x'},
        {"inputfile",          required_argument, NULL, 'i'},
        {"outputfile",         required_argument, NULL, 'o'},
        {"codec",              required_argument, NULL, 'c'},
        {"fps",                required_argument, NULL, 'r'},
        {"bitrate",            required_argument, NULL, 'b'},
        {"encode_name",        required_argument, NULL, 'e'},
        {"filter_dsc",         required_argument, NULL, 'f'},
        {"log_level",          required_argument, NULL, 'g'},
        {"select_frame",       required_argument, NULL, 'm'},
        {"pic_name",           required_argument, NULL, 'p'},
        {"pic_size",           required_argument, NULL, 'l'},
        {"input_format",       required_argument, NULL, 'v'},
        {"help",               no_argument,       NULL, 'h'},
        {NULL,                 0,                 NULL,   0},
    };

    codec_work = malloc(sizeof(struct codec_worker));
    if (codec_work == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate codec workers.\n");
        exit(1);
    }

    memset(codec_work, 0, sizeof(struct codec_worker));
    codec_work->enc_idx = -1;

    while ((opt = getopt_long(argc, argv, opt_string, long_options, &opt_index)) != -1)
    {
        switch (opt)
        {
            case 's':
                codec_work->src_width = (int) strtol(optarg, &n, 10);
                if (*n != 'x')
                {
                    av_log(NULL, AV_LOG_ERROR, "input resolution/size arg not of format [width]x[height]: %s\n", optarg);
                    exit(1);
                }
                codec_work->src_height = atoi(n + 1);
                break;
            case 'k':
                codec_work->dst_width = (int) strtol(optarg, &n, 10);
                if (*n != 'x')
                {
                    av_log(NULL, AV_LOG_ERROR, "input resolution/size arg not of format [width]x[height]: %s\n", optarg);
                    exit(1);
                }
                codec_work->dst_height = atoi(n + 1);
                break;
            case 'x':
                codec_work->xcoder_params = optarg;
                break;
            case 'i':
                codec_work->input_file = optarg;
                break;
            case 'o':
                codec_work->output_file = optarg;
                break;
            case 'c':
                codec_work->encoder_name = optarg;
                break;
            case 'e':
                codec_work->enc_idx = atoi(optarg);
                break;
            case 'b':
                bitrate = atoi(optarg);
                break;
            case 'r':
                fps = atoi(optarg);
                break;
            case 'f':
                codec_work->filter_dsc = optarg;
                break;
            case 'm':
                codec_work->select_frame = atoi(optarg);
                break;
            case 'p':
                codec_work->pic_name = optarg;
                break;
            case 'l':
                codec_work->pic_width = (int) strtol(optarg, &n, 10);
                if (*n != 'x')
                {
                    av_log(NULL, AV_LOG_ERROR, "input resolution/size arg not of format [width]x[height]: %s\n", optarg);
                    exit(1);
                }
                codec_work->pic_height = atoi(n + 1);
                break;
            case 'v':
                input_format = optarg;
                break;
            case 'g':
                loglevel = optarg;
                break;
            case 'h':
                help_usage();
                exit(0);
            default:
                help_usage();
                exit(1);
        }
    }

    // Input arg checking
    if (!codec_work->xcoder_params)
    {
        help_usage();
        exit(1);
    }
    av_log(NULL, AV_LOG_ERROR, "xcoder_params: %s.\n", codec_work->xcoder_params);
    if(!input_format)
    {
        help_usage();
        exit(1);
    }
    if(strcmp(input_format,"bgra") == 0)
    codec_work->input_format = AV_PIX_FMT_BGRA;
    else if(strcmp(input_format,"yuv420p") == 0)
    codec_work->input_format = AV_PIX_FMT_YUV420P;

    if (!codec_work->input_file || !codec_work->output_file || !codec_work->encoder_name
        || !codec_work->pic_name|| !codec_work->filter_dsc)
    {
        av_log(NULL, AV_LOG_ERROR, "Required args (inputfile, outputfile, encoder_name, pic_name, filter_dsc) not set.\n");
        exit(1);
    }
    if(codec_work->select_frame <= 0)
    {
        av_log(NULL, AV_LOG_ERROR, "Required args (select_frame) not set or must set greater than zero.\n");
    }
    av_log(NULL, AV_LOG_INFO, "inputfile:%s, outputfile:%s, codec:%s, src_width:%u, src_height:%u, dst_width:%u, dst_height:%u,enc_idx=%d.\n",\
        codec_work->input_file,codec_work->output_file, codec_work->encoder_name,\
        codec_work->src_width, codec_work->src_height, codec_work->dst_width, codec_work->dst_height,codec_work->enc_idx);

    if(loglevel)
    {
        setup_loglevel(loglevel);
    }

    if (pthread_create(&pthencode, NULL, encode_work, 0)!=0)
    {
        av_log(NULL, AV_LOG_ERROR,"creat thread failed!\n");
    }

    if (pthread_create(&pthid,NULL,printAIframe,0)!=0)
    {
        av_log(NULL, AV_LOG_ERROR,"creat thread failed!\n");
    }

    while(1)
    {
        if(should_exit >= 2)
            break;
        usleep(1000);
        print_report(0, timer_start, av_gettime_relative(), encode_num);
    }
    print_report(0, timer_start, av_gettime_relative(), encode_num);
    av_log(NULL,AV_LOG_INFO,"Encode done! This demo successfully quit.\n");

    return 0;
}
