/*
 * Copyright (c) 2001 Fabrice Bellard
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * @file
 * video encoding with libavcodec API example
 *
 * @example encode_video.c
 *
 * @added by zheng.lv@netint.ca
 * use multiple threads to run YUV encoding.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#ifdef __linux__
#include <sys/ioctl.h>
#endif
#include <sys/time.h>
#include <getopt.h>
#include <pthread.h>
#include <signal.h>

#include <libavcodec/avcodec.h>

#include <libavutil/opt.h>
#include <libavutil/imgutils.h>
#include <libavutil/time.h>

#define NI_CODEC_TIMING
//#define VERBOSE_DEBUG

enum {
    DEBUG_LEVEL_DEBUG,
    DEBUG_LEVEL_INFO,
    DEBUG_LEVEL_WARN,
    DEBUG_LEVEL_ERROR,
};

#define pr_err(fmt, ...) \
do {\
    fprintf(stderr, fmt, ##__VA_ARGS__); \
} while (0)

#define pr_warn(fmt, ...) \
do {\
    fprintf(stdout, fmt, ##__VA_ARGS__); \
} while (0)

#define pr_info(fmt, ...) \
do { \
    fprintf(stdout, fmt, ##__VA_ARGS__); \
} while (0)

#define pr_dbg(fmt, ...) \
do { \
    if (debug_level <= DEBUG_LEVEL_DEBUG) { \
        fprintf(stdout, fmt, ##__VA_ARGS__); \
    } \
} while (0)

int debug_level = DEBUG_LEVEL_INFO;
char ns_id[4] = { 0 };
char dev_id[4] = { 0 };
int print_stat = 1;
int print_time = 1;
FILE *flog = NULL;
#define LOG_BUF_SIZE 8192
void *logbuf;
int last_fps = 0;
int log_time = 0;
int mute = 0;
int rate_limit = 0;
int64_t start = 0;

int actual_threads = 0;
int sync_mode = 0;
int global_stop = 0;

struct encode_worker {
    pthread_t tid;
    const AVCodec *codec;
    int index;
    int should_stop;
    const char *filename;
    const char *output_dir;
    const char *logname;
    int bitrate;
    int fps;
    unsigned int width;
    unsigned int height;
    const char *codec_params;
    int loops;
    int ns_id;
    int dev_id;
    unsigned long frame_number;
    void *common;
};

struct common {
    pthread_mutex_t lock;
    pthread_cond_t ready_cond;
    int ready_num;
    int exit_num;
    int total_threads;

    int done;
    unsigned long ongoing;
    pthread_mutex_t sync_lock;
    pthread_cond_t sync_cond;
};

struct encode_worker *encode_workers;

static void print_report(int is_last_report, int64_t timer_start,
        int64_t cur_time, unsigned long frame_number)
{
    static int64_t last_time = -1;
    float t;
    char buf[1024];
    float fps;

    if (!print_stat)
        return;

    if (!is_last_report) {
        if (last_time == -1) {
            last_time = cur_time;
            return;
        }
        if ((cur_time - last_time) < 500000)
            return;
        last_time = cur_time;
    }

    t = (cur_time - timer_start) / 1000000.0;

    fps = t > 1 ? frame_number / t : 0;
    if (print_stat && !mute) {
        const char end = is_last_report ? '\n' : '\r';

        buf[0] = '\0';
        snprintf(buf + strlen(buf), sizeof(buf) - strlen(buf), "frame=%5ld fps=%3.*f ",
                frame_number, (fps < 9.95), fps);
        fprintf(stderr, "%s   %c", buf, end);
        fflush(stderr);
    }
}

static int probe_send_frame(AVCodecContext *enc_ctx)
{
#ifdef XCODER_LINUX_CUSTOM_DRIVER
    int ret;

    enc_ctx->opaque = (void *)1;
    ret = avcodec_send_frame(enc_ctx, (AVFrame *)~0);
    enc_ctx->opaque = (void *)0;

    return ret;
#else
    return 0;
#endif
}

static int __encode(AVCodecContext *enc_ctx, AVFrame *frame,
        AVPacket *pkt, FILE *outfile)
{
    int ret;

    /* send the frame to the encoder */
    if (frame)
        pr_dbg("Send frame %3"PRId64"\n", frame->pts);

    do {
        ret = probe_send_frame(enc_ctx);
        if (ret == AVERROR(EAGAIN)) {
            continue;
        } else if (ret < 0) {
            pr_err("Error probing encode ready status.\n");
            return ret;
        }

        ret = avcodec_send_frame(enc_ctx, frame);
        if (ret == AVERROR(EAGAIN)) {
            continue;
        } else if (ret < 0) {
            pr_err("Error sending a frame for encoding\n");
            return ret;
        } else
            break;
    } while (1);

    while (ret >= 0) {
        pr_dbg("receive packet.\n");
        ret = avcodec_receive_packet(enc_ctx, pkt);
        if (ret == AVERROR(EAGAIN) || ret == AVERROR_EOF) {
            return ret;
        } else if (ret < 0) {
            fprintf(stderr, "Error during encoding\n");
            return ret;
        }

        pr_dbg("Write packet %3"PRId64" (size=%5d)\n", pkt->pts, pkt->size);
        fwrite(pkt->data, 1, pkt->size, outfile);
        av_packet_unref(pkt);
    }

    return 0;
}

static void *encode_worker_func(void *thread_data)
{
    struct encode_worker *worker = (struct encode_worker *)thread_data;
    const AVCodec *codec = worker->codec;
    AVCodecContext *c = NULL;
    AVPacket *pkt = NULL;
    AVFrame *frame = NULL;
    unsigned int frame_size;
    unsigned int read_size;
    unsigned long input_filesize;
    char name[256] = { 0 };
    FILE *input = NULL;
    FILE *output = NULL;
    int fps = worker->fps;
    int loop_count = worker->loops;
    int ret;
    int l, i, y;
    struct common *common = (struct common *)worker->common;
    char str_devid[8] = { 0 };

    av_log(NULL, AV_LOG_INFO, "encode_worker %d enter.\n", worker->index);

    pthread_mutex_lock(&common->lock);
    common->ready_num++;
    if (common->ready_num >= common->total_threads) {
        pthread_cond_signal(&common->ready_cond);
    }
    pthread_mutex_unlock(&common->lock);

    input = fopen(worker->filename, "r");
    if (input == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to open file: %s.\n", strerror(errno));
        ret = AVERROR(errno);
        goto end;
    }

    fseek(input, 0, SEEK_END);
    input_filesize = ftell(input);
    frame_size = input_filesize / (worker->width * worker->height * 3 / 2);
    av_log(NULL, AV_LOG_DEBUG, "inputfile size=%lu, frame size=%u.\n",
            input_filesize, frame_size);

    if (codec->id == AV_CODEC_ID_H264) {
        snprintf(name, sizeof(name) - 1, "%s/output-%d.h264", worker->output_dir, worker->index);
    } else if (codec->id == AV_CODEC_ID_H265) {
        snprintf(name, sizeof(name) - 1, "%s/output-%d.h265", worker->output_dir, worker->index);
#ifdef AV1_SUPPORTED // AV1 not supported in 3.1.1
    } else if (codec->id == AV_CODEC_ID_AV1) {
        snprintf(name, sizeof(name) - 1, "%s/output-%d.data", worker->output_dir, worker->index);
#endif
    } else {
        av_log(NULL, AV_LOG_ERROR, "Invalid codec id %s.\n", avcodec_get_name(codec->id));
        ret = AVERROR(EINVAL);
        goto end;
    }

    output = fopen(name, "wb");
    if (!output) {
        av_log(NULL, AV_LOG_ERROR, "failed to open output file: %s.\n", strerror(errno));
        ret = AVERROR(errno);
        goto end;
    }

    c = avcodec_alloc_context3(codec);
    if (!c) {
        av_log(NULL, AV_LOG_ERROR, "Could not allocate video codec context\n");
        ret = AVERROR(ENOMEM);
        goto end;
    }

    /* put sample parameters */
    c->bit_rate = worker->bitrate;
    /* resolution must be a multiple of two */
    c->width = worker->width;
    c->height = worker->height;
    /* frames per second */
    c->time_base = (AVRational){1, fps};
    c->framerate = (AVRational){fps, 1};

    /* emit one intra frame every ten frames
     * check frame pict_type before passing frame
     * to encoder, if frame->pict_type is AV_PICTURE_TYPE_I
     * then gop_size is ignored and the output of encoder
     * will always be I frame irrespective to gop_size
     */
    c->gop_size = 10;
    c->max_b_frames = 1;
    c->pix_fmt = AV_PIX_FMT_YUV420P;

    if (!strncmp(codec->name, "h264_ni", strlen("h264_ni")) ||
        !strncmp(codec->name, "h265_ni", strlen("h265_ni"))
#ifdef AV1_SUPPORTED // AV1 not supported in 3.1.1
        || !strncmp(codec->name, "av1_ni", strlen("av1_ni"))
#endif
    ){
        /*
         * for example:
         * gopPresetIdx=6:lowDelay=1:intraPeriod=120:RcEnable=1:bitrate=4000000
         */
        if (worker->codec_params) {
            av_opt_set(c->priv_data, "xcoder-params", worker->codec_params, 0);
        }
        snprintf(str_devid, sizeof(str_devid), "%d", worker->dev_id);
        av_opt_set(c->priv_data, "enc", str_devid, 0);
    } else {
        av_log(NULL, AV_LOG_ERROR, "codec %s not supported.\n", codec->name);
        ret = AVERROR(EINVAL);
        goto end;
    }

    /* open it */
    ret = avcodec_open2(c, codec, NULL);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Could not open codec: %s\n",
                av_err2str(ret));
        goto end;
    }

    pkt = av_packet_alloc();
    if (!pkt) {
        ret = AVERROR(ENOMEM);
        goto end;
    }

    frame = av_frame_alloc();
    if (!frame) {
        av_log(NULL, AV_LOG_ERROR, "Could not allocate video frame\n");
        ret = AVERROR(ENOMEM);
        goto end;
    }

    frame->format = c->pix_fmt;
    frame->width  = c->width;
    frame->height = c->height;

    ret = av_frame_get_buffer(frame, 32);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Could not allocate the video frame data\n");
        goto end;
    }

    av_log(NULL, AV_LOG_DEBUG, "linesize: 0=%d, 1=%d, 2=%d.\n",
            frame->linesize[0], frame->linesize[1], frame->linesize[2]);

    for (l = 0; l < loop_count; l++) {
        fseek(input, 0, SEEK_SET);
        for (i = 0; i < frame_size; i++) {
//            if (rate_limit) {
//                while (1) {
//                    cur_time = av_gettime_relative();
//                    if ((dts + 20000) > (cur_time - timer_start)) {
//                        av_usleep(10000);
//                        continue;
//                    } else
//                        break;
//                }
//            }

            if (sync_mode) {
                pthread_mutex_lock(&common->sync_lock);
                common->done++;
                common->ongoing |= (1 << worker->index);
                if (common->done == common->total_threads) {
                    av_log(NULL, AV_LOG_TRACE, "%d: reset %lu.\n", worker->index, worker->frame_number);
                    common->done = 0;
                    common->ongoing = 0;
                    pthread_cond_broadcast(&common->sync_cond);
                } else {
                    while ((common->ongoing & (1 << worker->index)) && !worker->should_stop) {
                        av_log(NULL, AV_LOG_TRACE, "%d: wait %lu.\n", worker->index, worker->frame_number);
                        pthread_cond_wait(&common->sync_cond, &common->sync_lock);
                    }
                }
                av_log(NULL, AV_LOG_TRACE, "%d: wake %lu.\n", worker->index, worker->frame_number);
                pthread_mutex_unlock(&common->sync_lock);
            }

            if (worker->should_stop) {
                goto flush;
            }

            /* make sure the frame data is writable */
            ret = av_frame_make_writable(frame);
            if (ret < 0) {
                av_log(NULL, AV_LOG_ERROR, "frame make writable error.\n");
                goto end;
            }

            for (y = 0; y < c->height; y++) {
                read_size = fread(&frame->data[0][0] + y * frame->linesize[0], c->width, 1, input);
                if (read_size != 1) {
                    av_log(NULL, AV_LOG_ERROR, "y: read_size=%u.\n", read_size);
                    ret = AVERROR(errno);
                    goto end;
                }
            }

            for (y = 0; y < c->height / 2; y++) {
                read_size = fread(&frame->data[1][0] + y * frame->linesize[1], c->width / 2, 1, input);
                if (read_size != 1) {
                    av_log(NULL, AV_LOG_ERROR, "u: read_size=%u.\n", read_size);
                    ret = AVERROR(errno);
                    goto end;
                }
            }

            for (y = 0; y < c->height / 2; y++) {
                read_size = fread(&frame->data[2][0] + y * frame->linesize[2], c->width / 2, 1, input);
                if (read_size != 1) {
                    av_log(NULL, AV_LOG_ERROR, "v: read_size=%u.\n", read_size);
                    ret = AVERROR(errno);
                    goto end;
                }
            }

            frame->pts = l * frame_size + i;
            worker->frame_number = frame->pts + 1;  // begin frame count at 1

            /* encode the image */
            ret = __encode(c, frame, pkt, output);
            if (ret == AVERROR(EAGAIN)) {
                continue;
            } else if (ret == AVERROR_EOF) {
//                av_log(NULL, AV_LOG_INFO, "%s@%d: EOF.\n", __func__, __LINE__);
                goto end;
            } else if (ret < 0) {
                goto end;
            }
        }
    }

flush:
    /* flush the encoder */
    ret = __encode(c, NULL, pkt, output);
    if (ret == AVERROR_EOF) {
//        av_log(NULL, AV_LOG_INFO, "%s@%d: EOF.\n", __func__, __LINE__);
        ret = 0;
    }

    /* add sequence end code to have a real MPEG file */
//    fwrite(endcode, 1, sizeof(endcode), f);
end:
//    av_log(NULL, AV_LOG_INFO, "encode_worker %d exit: ret=0x%x.\n",
//            worker->index, -ret);

    if (input) {
        fclose(input);
    }
    if (output) {
        fclose(output);
    }
    if (c) {
        avcodec_free_context(&c);
    }
    if (frame) {
        av_frame_free(&frame);
    }
    if (pkt) {
        av_packet_free(&pkt);
    }

    pthread_mutex_lock(&common->lock);
    common->exit_num++;
    pthread_mutex_unlock(&common->lock);

    return (void *)((long)ret);
}

static void sigint_handler(int signo)
{
    global_stop = 1;
    av_log(NULL, AV_LOG_INFO, "%s().\n", __func__);
}

static inline int curr_total_frames(void)
{
    int i;
    unsigned long total_frames = 0;

    for (i = 0; i < actual_threads; i++) {
        total_frames += encode_workers[i].frame_number;
    }

    return total_frames;
}

static void help_usage(void)
{
    printf("Usage: \n"
            "-p | --pixel               [width]x[height] of yuv.\n"
            "-x | --xcoder_params       xcoder parameters.\n"
            "-i | --inputfile           input yuv file.\n"
            "-o | --output_dir          output directory path.\n"
            "-c | --codec               codec name.\n"
            "-t | --threads             number of threads.\n"
            "-b | --bitrate             bitrate.\n"
            "-n | --nsid                ns id.\n"
            "-d | --devid               dev id.\n"
            "-s | --loop                run loop count.\n"
            "-y | --sync                sync mode test.\n"
            "-h | --help                help info.\n");
}

static struct common *alloc_common(void)
{
    struct common *common;
    int ret;

    common = malloc(sizeof(struct common));
    if (common == NULL) {
        return NULL;
    }

    memset(common, 0, sizeof(struct common));

    ret = pthread_mutex_init(&common->lock, NULL);
    if (ret) {
        goto fail_init_lock;
    }

    ret = pthread_cond_init(&common->ready_cond, NULL);
    if (ret) {
        goto fail_init_ready_cond;
    }

    ret = pthread_mutex_init(&common->sync_lock, NULL);
    if (ret) {
        goto fail_init_sync_lock;
    }

    ret = pthread_cond_init(&common->sync_cond, NULL);
    if (ret) {
        goto fail_init_sync_cond;
    }

    return common;

fail_init_sync_cond:
    pthread_mutex_destroy(&common->sync_lock);
fail_init_sync_lock:
    pthread_cond_destroy(&common->ready_cond);
fail_init_ready_cond:
    pthread_mutex_destroy(&common->lock);
fail_init_lock:
    free(common);
    return NULL;
}

static void free_common(struct common *common)
{
    if (common) {
        pthread_cond_destroy(&common->sync_cond);
        pthread_mutex_destroy(&common->sync_lock);
        pthread_mutex_destroy(&common->lock);
        pthread_cond_destroy(&common->ready_cond);
        free(common);
    }
}

int main(int argc, char **argv)
{
    const char *codec_name = NULL;
    const char *output_dir = NULL;
    const AVCodec *codec;

    char *xcoder_params = NULL;
    unsigned int video_width = ~0;
    unsigned int video_height = ~0;
    int bitrate = 4000000;
    int fps = 30;
    unsigned int loop_count = 1;
    char *logfile = NULL;
    const char *input_file;
    int ns = 1;
    int devid = 0;
    int i;
    int ret;
    char *n;
    int state = EXIT_SUCCESS;
    int threads = 1;
    pthread_attr_t attr;
    struct common *common = NULL;
    int64_t timer_start;

    int opt;
    int opt_index;
    const char *opt_string = "yt:s:r:b:d:n:p:x:i:o:c:g:mefh";
    static struct option long_options[] = {
        {"pixel",         required_argument, NULL, 'p'},
        {"xcoder_params", required_argument, NULL, 'x'},
        {"input_file",    required_argument, NULL, 'i'},
        {"output_dir",    required_argument, NULL, 'o'},
        {"codec",         required_argument, NULL, 'c'},
        {"fps",           required_argument, NULL, 'r'},
        {"bitrate",       required_argument, NULL, 'b'},
        {"nsid",          required_argument, NULL, 'n'},
        {"devid",         required_argument, NULL, 'd'},
        {"loop",          required_argument, NULL, 's'},
        {"log",           required_argument, NULL, 'g'},
        {"threads",       required_argument, NULL, 't'},
        {"mute",          no_argument,       NULL, 'm'},
        {"limit",         no_argument,       NULL, 'e'},
        {"pr_fps",        no_argument,       NULL, 'f'},
        {"sync",          no_argument,       NULL, 'y'},
        {"help",          no_argument,       NULL, 'h'},
        {NULL,            0,                 NULL, 0},
    };

    while ((opt = getopt_long(argc, argv, opt_string, long_options, &opt_index)) != -1) {
        switch (opt) {
            case 'p':
                video_width = strtoul(optarg, &n, 10);
                if (*n != 'x')
                {
                    help_usage();
                    return -1;
                }
                video_height = strtoul(n + 1, NULL, 10);
                break;
            case 'x':
                xcoder_params = optarg;
                break;
            case 'i':
                input_file = optarg;
                break;
            case 'o':
                output_dir = optarg;
                break;
            case 'c':
                codec_name = optarg;
                break;
            case 'n':
                ns = atoi(optarg);
                break;
            case 'd':
                devid = atoi(optarg);
                break;
            case 'b':
                bitrate = atoi(optarg);
                break;
            case 'r':
                fps = atoi(optarg);
                break;
            case 's':
                loop_count = strtoul(optarg, NULL, 10);
                break;
            case 'g':
                log_time = 1;
                logfile = optarg;
                break;
            case 'm':
                mute = 1;
                break;
            case 'e':
                rate_limit = 1;
                break;
            case 'f':
                print_stat = 1;
                print_time = 0;
                break;
            case 't':
                threads = atoi(optarg);
                break;
            case 'y':
                sync_mode = 1;
                break;
            case 'h':
                help_usage();
                return 0;
            default:
                help_usage();
                return -1;
        }
    }

    if ((input_file == NULL) || (codec_name == NULL) || \
                (video_width == ~0) || (video_height == ~0)) {
        help_usage();
        exit(1);
    }

    if (output_dir == NULL) {
        output_dir = ".\0";
    }

    av_log_set_level(AV_LOG_INFO);

    if (sync_mode) {
        if ((threads <= 0) || (threads > sizeof(unsigned long))) {
            av_log(NULL, AV_LOG_ERROR, "invalid threads number %d.\n", threads);
            exit(1);
        }
    } else {
        if ((threads <= 0) || (threads > 256)) {
            av_log(NULL, AV_LOG_ERROR, "invalid threads number %d.\n", threads);
            exit(1);
        }
    }

    av_log(NULL, AV_LOG_INFO, "sync mode: %d.\n", sync_mode);
    av_log(NULL, AV_LOG_INFO, "threads: %d.\n", threads);
    av_log(NULL, AV_LOG_INFO, "input file: %s.\n", input_file);
    av_log(NULL, AV_LOG_INFO, "codec name: %s.\n", codec_name);
    av_log(NULL, AV_LOG_INFO, "video width: %u.\n", video_width);
    av_log(NULL, AV_LOG_INFO, "video height: %u.\n", video_height);
    av_log(NULL, AV_LOG_INFO, "namespace id: %d.\n", ns);
    av_log(NULL, AV_LOG_INFO, "device id: %d.\n", devid);
    av_log(NULL, AV_LOG_INFO, "loop count: %u.\n", loop_count);
    if (xcoder_params) {
        av_log(NULL, AV_LOG_INFO, "xcoder params: %s.\n", xcoder_params);
    }

    if (log_time) {
        av_log(NULL, AV_LOG_INFO, "log file: %s.\n", logfile);
    }

    if (access(input_file, R_OK) != 0) {
        av_log(NULL, AV_LOG_ERROR, "cannot access %s: %s.\n", input_file,
                strerror(errno));
        state = EXIT_FAILURE;
        goto end;
    }

#if LIBAVCODEC_VERSION_MAJOR < 58
    // Register all codecs
    avcodec_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif

    /* find netint encoder */
    codec = avcodec_find_encoder_by_name(codec_name);
    if (!codec) {
        av_log(NULL, AV_LOG_ERROR, "Codec '%s' not found\n", codec_name);
        state = EXIT_FAILURE;
        goto end;
    }

    av_log(NULL, AV_LOG_INFO, "Codec %s.\n", codec->name);

    common = alloc_common();
    if (common == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate common data.\n");
        state = EXIT_FAILURE;
        goto end;
    }

    common->total_threads = threads;

    ret = pthread_attr_init(&attr);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "fail to initialize attr: %s.\n",
                strerror(errno));
        state = EXIT_FAILURE;
        goto end;
    }

    ret = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "fail to set attr detachstate: %s.\n",
                strerror(errno));
        state = EXIT_FAILURE;
        goto end;
    }

    encode_workers = malloc(sizeof(struct encode_worker) * threads);
    if (encode_workers == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate encode workers.\n");
        state = EXIT_FAILURE;
        goto end;
    }
    memset(encode_workers, 0, sizeof(struct encode_worker) * threads);

    for (i = 0; i < threads; i++) {
        struct encode_worker *worker = &encode_workers[i];

        worker->index = i;
        worker->bitrate = bitrate;
        worker->width = video_width;
        worker->height = video_height;
        worker->fps = fps;
        worker->codec = codec;
        worker->filename = input_file;
        worker->output_dir = output_dir;
        worker->logname = logfile;
        worker->dev_id = devid;
        worker->ns_id = ns;
        worker->common = common;
        worker->loops = loop_count;
        worker->codec_params = xcoder_params;

        av_log(NULL, AV_LOG_INFO, "create thread %d, bitrate=%d, "
                "width=%u, height=%u, fps=%d, input=%s, output_dir=%s, loops=%d.\n",
                worker->index, worker->bitrate, worker->width, worker->height,
                worker->fps, worker->filename, worker->output_dir, worker->loops);

        ret = pthread_create(&worker->tid, &attr, encode_worker_func, worker);
        if (ret) {
            av_log(NULL, AV_LOG_ERROR, "failed to create pthread for %d: %s.\n",
                    i, strerror(errno));
            state = EXIT_FAILURE;
            goto end;
        }
        actual_threads++;
    }

    ret = pthread_attr_destroy(&attr);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "failed to destroy attr: %s.\n", strerror(ret));
        state = EXIT_FAILURE;
        goto end;
    }

    pthread_mutex_lock(&common->lock);
    while (common->ready_num < common->total_threads)
        pthread_cond_wait(&common->ready_cond, &common->lock);
    pthread_mutex_unlock(&common->lock);

    if (SIG_ERR == signal(SIGINT, sigint_handler)) {
        av_log(NULL, AV_LOG_ERROR, "failed to set signal.\n");
        ret = EXIT_FAILURE;
        goto end;
    }

    timer_start = av_gettime_relative();
    while (!global_stop) {
        if (common->exit_num == actual_threads) {
            break;
        }
        sleep(1);
        print_report(0, timer_start, av_gettime_relative(), curr_total_frames());
    }
    print_report(1, timer_start, av_gettime_relative(), curr_total_frames());

    av_log(NULL, AV_LOG_INFO, "main thread is going to exit.\n");

end:
    if (encode_workers) {
        for (i = 0; i < actual_threads; i++) {
            struct encode_worker *worker = &encode_workers[i];
            worker->should_stop = 1;
        }

        if (sync_mode) {
            pthread_mutex_lock(&common->sync_lock);
            pthread_cond_broadcast(&common->sync_cond);
            pthread_mutex_unlock(&common->sync_lock);
        }

        for (i = 0; i < actual_threads; i++) {
            void *result;
            struct encode_worker *worker = &encode_workers[i];

            ret = pthread_join(worker->tid, &result);
            if (ret) {
                av_log(NULL, AV_LOG_ERROR, "failed to join pthread for %d: %s.\n",
                        worker->index, strerror(errno));
                state = EXIT_FAILURE;
            }

            if ((long)result != 0) {
                av_log(NULL, AV_LOG_ERROR, "pthread %d result=0x%lx.\n",
                        worker->index, (long)result);
                state = EXIT_FAILURE;
            }
        }
        free(encode_workers);
    }

    free_common(common);

    av_log(NULL, AV_LOG_INFO, "EXIT.. state=0x%x.\n", state);

    return state;
}
