/*
 * Copyright (c) 2010 Nicolas George
 * Copyright (c) 2011 Stefano Sabatini
 * Copyright (c) 2014 Andrey Utkin
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * @file
 * API example for demuxing, decoding, filtering, encoding and muxing
 * @example transcoding.c
 *
 * @added by zheng.lv@netint.ca
 * use multiple threads to run transcoding.
 */

#include <stdlib.h>
#include <unistd.h>
#include <getopt.h>
#include <pthread.h>
#include <signal.h>
#include <assert.h>

#include <libavcodec/avcodec.h>
#include <libavformat/avformat.h>
#include <libavfilter/buffersink.h>
#include <libavfilter/buffersrc.h>
#include <libavutil/opt.h>
#include <libavutil/pixdesc.h>
#include <libavutil/time.h>
#include <libavutil/avassert.h>
#include <libavutil/channel_layout.h>
#include <libavutil/hwcontext.h>

#define AVFRAME_BLOCKING_COUNT 100

struct list_node {
    struct list_node *next;
    unsigned long data[0];
};

struct list_link {
    struct list_node *head;
    struct list_node *tail;
};

struct frame_info {
    int stream_index;
    int is_hw_frame;
    AVFrame *frame;
};

typedef struct StreamContext {
    AVCodecContext *dec_ctx;
    AVCodecContext *enc_ctx;
    AVRational enc_time_base;
    int64_t max_duration;
    int64_t total_duration;
    int64_t last_duration;
    int64_t max_pts;
    int64_t min_pts;
    int64_t last_encoded_pts;
    int64_t last_encoded_dts;
    int64_t last_decoded_pts;
    int64_t last_decoded_dts;
    int decoding_needed;
    int stream_copy;
    int disabled;
} StreamContext;

typedef struct FilteringContext {
    AVFilterContext *buffersink_ctx;
    AVFilterContext *buffersrc_ctx;
    AVFilterGraph *filter_graph;
} FilteringContext;

struct common {
    pthread_mutex_t lock;
    pthread_cond_t ready_cond;
    int ready_num;
    int exit_num;
    int total_threads;
};

struct codec_worker {
    int index;
    pthread_t tid;

    pthread_mutex_t frame_lock;
    pthread_cond_t consume_cond;
    pthread_cond_t product_cond;

    pthread_mutex_t launch_encoder_lock;
    pthread_cond_t lanuch_encoder_cond;
    pthread_mutex_t remux_lock;

    struct list_link frame_link;
    unsigned int filt_frames;

    unsigned long total_decoder_frames;
    unsigned long total_encoder_frames;

    AVFormatContext *ifmt_ctx;
    FilteringContext *filter_ctx;
    AVFormatContext *ofmt_ctx;
    StreamContext *stream_ctx;

    int should_exit;
    int force_exit;

    pthread_t encode_tid;

    // timestamp (pts) of last frame of this stream; this is usually set to
    // the last frame pts of existing streams when adding this as a new decoder
    // stream, and is used for the new stream starting pts to be in sync with
    // other streams in progress
    int64_t last_decoded_pts;
    int encode_exited;

    const char *encoder_name;
    const char *decoder_name;
    const char *input_file;
    const char *output_dir;
    /* for output file name */
    const char *prefix;
    const char *suffix;
    const char *encoder_params;
    const char *decoder_params;
    const char *filter_params;//only count video frames
    int bitrate;
    int devid;
    unsigned int loop;
    int nb_streams;
    int bypass_filter;
    int audio_filter;
    int repeat_header;
    int force_source_keyframe;
    int rescale_ts;
    int audio_disable;
    int video_stream_num;
    int *inited_encoder_ctx;
    int count_inited_encoder_ctx;
    int decoder_launch_encoder;
    int encoder_launched;
    char *output_file;
    struct common *common;
    AVFrame *decoded_frame;
    AVStream **out_streams;
};

struct codec_worker *codec_workers = NULL;
int active_codec_workers = 0;

int global_stop = 0;
int print_stat = 1;

static inline int __avframe_blocking_count(int is_hw_frame)
{
#define NI_MAX_FILTER_POOL_SIZE 4
    return is_hw_frame ? NI_MAX_FILTER_POOL_SIZE - 1 : 100;
}

static struct list_node *alloc_list_node(unsigned int size)
{
    struct list_node *node;

    node = malloc(sizeof(struct list_node) + size);
    if (node) {
        node->next = NULL;
    }

    return node;
}

static void free_list_node(struct list_node *node)
{
    if (node) {
        free(node);
    }
}

static void add_list_tail(struct list_node *new, struct list_link *link)
{
    if (link->head == NULL) {
        link->head = new;
        link->tail = new;
    } else {
        link->tail->next = new;
        link->tail = new;
    }
    link->tail->next = NULL;
}

static struct list_node *pick_list_head(struct list_link *link)
{
    struct list_node *first = NULL;

    if (link->head) {
        first = link->head;
        link->head = link->head->next;
    }

    return first;
}

static inline int list_empty(struct list_link *link)
{
    return (link->head == NULL);
}

static inline void *list_node_data(struct list_node *node)
{
    return node->data;
}

static inline struct list_node *this_list_node(void *data)
{
    return (struct list_node *)data - 1;
}

static struct frame_info *alloc_frame_info(void)
{
    struct list_node *node;

    node = alloc_list_node(sizeof(struct frame_info));
    if (node) {
        return (struct frame_info *)node->data;
    }

    return NULL;
}

static void free_frame_info(struct frame_info *frame_info)
{
    if (frame_info) {
        free_list_node(this_list_node(frame_info));
    }
}

static void print_report(int is_last_report, int64_t timer_start,
                         int64_t cur_time, unsigned long frame_number)
{
    static int64_t last_time = -1;
    float t;
    char buf[1024];
    float fps;

    if (!print_stat)
        return;

    if (!is_last_report) {
        if (last_time == -1) {
            last_time = cur_time;
            return;
        }
        if ((cur_time - last_time) < 500000)
            return;
        last_time = cur_time;
    }

    t = (cur_time - timer_start) / 1000000.0;

    fps = t > 1 ? frame_number / t : 0;
    if (print_stat) {
        const char end = is_last_report ? '\n' : '\r';

        buf[0] = '\0';
        snprintf(buf + strlen(buf), sizeof(buf) - strlen(buf), "frame=%5lu fps=%3.*f ",
                frame_number, (fps < 9.95), fps);
        if (AV_LOG_INFO > av_log_get_level()) {
            fprintf(stderr, "%s   %c", buf, end);
        } else {
            av_log(NULL, AV_LOG_INFO, "%s   %c", buf, end);
        }
        fflush(stderr);
    }
}

static int open_input_file(struct codec_worker *codec_worker,
                           const char *codec_name, const char *input_file)
{
    int ret;
    unsigned int i;
    AVFormatContext *ifmt_ctx = NULL;
    StreamContext *stream_ctx = NULL;
    int * inited_encoder_ctx = NULL;

    if ((ret = avformat_open_input(&ifmt_ctx, input_file, NULL, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot open input file\n");
        return ret;
    }

    if ((ret = avformat_find_stream_info(ifmt_ctx, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot find stream information\n");
        return ret;
    }

    stream_ctx = av_calloc(ifmt_ctx->nb_streams, sizeof(*stream_ctx));
    if (!stream_ctx)
        return AVERROR(ENOMEM);

    inited_encoder_ctx = av_calloc(ifmt_ctx->nb_streams,sizeof(int));
    if(!inited_encoder_ctx)
    {
        return AVERROR(ENOMEM);
    }

    codec_worker->inited_encoder_ctx = inited_encoder_ctx;

    for (i = 0; i < ifmt_ctx->nb_streams; i++) {
        AVStream *stream = ifmt_ctx->streams[i];
        AVCodecContext *codec_ctx;
        const AVCodec *dec;

        if ((stream->codecpar->codec_id == AV_CODEC_ID_H264 ||
            stream->codecpar->codec_id == AV_CODEC_ID_H265) && codec_name) {
            dec = avcodec_find_decoder_by_name(codec_name);
            if (dec) {
                if (stream->codecpar->codec_id != dec->id) {
                    av_log(NULL, AV_LOG_ERROR, "codec does not match with stream id\n");
                    return AVERROR_DECODER_NOT_FOUND;
                }
            }
        } else {
            dec = avcodec_find_decoder(stream->codecpar->codec_id);
        }

        if (!dec) {
            av_log(NULL, AV_LOG_ERROR, "Failed to find decoder for stream #%u\n", i);
            return AVERROR_DECODER_NOT_FOUND;
        }

        codec_ctx = avcodec_alloc_context3(dec);
        if (!codec_ctx) {
            av_log(NULL, AV_LOG_ERROR, "Failed to allocate the decoder context for stream #%u\n", i);
            return AVERROR(ENOMEM);
        }
        ret = avcodec_parameters_to_context(codec_ctx, stream->codecpar);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "Failed to copy decoder parameters to input decoder context "
                   "for stream #%u\n", i);
            return ret;
        }

        codec_ctx->pkt_timebase = stream->time_base;

        av_log(NULL, AV_LOG_DEBUG, "#%d: stream.time_base=%d/%d, avg_frame_rate=%d/%d.\n",
                i, stream->time_base.num, stream->time_base.den,
                stream->avg_frame_rate.num, stream->avg_frame_rate.den);

        if (!strncmp(dec->name, "h264_ni", strlen("h264_ni")) ||
            !strncmp(dec->name, "h265_ni", strlen("h265_ni"))) {
            char str_devid[4] = {0};
            snprintf(str_devid, sizeof(str_devid), "%d", codec_worker->devid);
            av_opt_set(codec_ctx->priv_data, "dec", str_devid, 0);
            av_opt_set(codec_ctx->priv_data, "xcoder-params", codec_worker->decoder_params, 0);
            av_opt_set(codec_ctx->priv_data, "keep_alive_timeout", "3", 0);
        }

        if (codec_ctx->codec_type == AVMEDIA_TYPE_VIDEO)
            codec_ctx->framerate = av_guess_frame_rate(ifmt_ctx, stream, NULL);

        ret = avcodec_parameters_from_context(stream->codecpar, codec_ctx);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "Failed to copy decoder parameters from codec context for stream #%u", i);
            return ret;
        }

        /* Reencode video & audio and remux subtitles etc. */
        if (codec_ctx->codec_type == AVMEDIA_TYPE_VIDEO ||
                codec_ctx->codec_type == AVMEDIA_TYPE_AUDIO) {
            /* Open decoder */
            ret = avcodec_open2(codec_ctx, dec, NULL);
            if (ret < 0) {
                av_log(NULL, AV_LOG_ERROR, "Failed to open decoder for stream #%u\n", i);
                return ret;
            }
            if(codec_ctx->codec_type == AVMEDIA_TYPE_VIDEO)
            {
                codec_worker->video_stream_num = i;
            }
        }
        stream_ctx[i].dec_ctx = codec_ctx;
        stream_ctx[i].last_decoded_pts = AV_NOPTS_VALUE;
        stream_ctx[i].last_decoded_pts = AV_NOPTS_VALUE;
    }

    av_dump_format(ifmt_ctx, 0, input_file, 0);
    codec_worker->ifmt_ctx = ifmt_ctx;
    codec_worker->stream_ctx = stream_ctx;
    codec_worker->nb_streams = ifmt_ctx->nb_streams;
    return 0;
}

static int reopen_input_file(struct codec_worker *codec_worker,
                             const char *codec_name, const char *input_file)
{
    int ret;
    unsigned int i;
    AVFormatContext *ifmt_ctx = NULL;
    StreamContext *stream_ctx = NULL;

    if (codec_worker->ifmt_ctx) {
        for (i = 0; i < codec_worker->ifmt_ctx->nb_streams; i++) {
            avcodec_free_context(&codec_worker->stream_ctx[i].dec_ctx);
            codec_worker->stream_ctx[i].dec_ctx = NULL;
        }
        avformat_close_input(&codec_worker->ifmt_ctx);
        codec_worker->ifmt_ctx = NULL;
    }

    if ((ret = avformat_open_input(&ifmt_ctx, input_file, NULL, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot open input file\n");
        return ret;
    }

    if ((ret = avformat_find_stream_info(ifmt_ctx, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot find stream information\n");
        return ret;
    }

    stream_ctx = codec_worker->stream_ctx;

    for (i = 0; i < ifmt_ctx->nb_streams; i++) {
        AVStream *stream = ifmt_ctx->streams[i];
        AVCodecContext *codec_ctx;
        const AVCodec *dec;

        if ((stream->codecpar->codec_id == AV_CODEC_ID_H264 ||
            stream->codecpar->codec_id == AV_CODEC_ID_H265) && codec_name) {
            dec = avcodec_find_decoder_by_name(codec_name);
            if (dec) {
                if (stream->codecpar->codec_id != dec->id) {
                    av_log(NULL, AV_LOG_ERROR, "codec does not match with stream id\n");
                    return AVERROR_DECODER_NOT_FOUND;
                }
            }
        } else {
            dec = avcodec_find_decoder(stream->codecpar->codec_id);
        }

        if (!dec) {
            av_log(NULL, AV_LOG_ERROR, "Failed to find decoder for stream #%u\n", i);
            return AVERROR_DECODER_NOT_FOUND;
        }

        codec_ctx = avcodec_alloc_context3(dec);
        if (!codec_ctx) {
            av_log(NULL, AV_LOG_ERROR, "Failed to allocate the decoder context for stream #%u\n", i);
            return AVERROR(ENOMEM);
        }
        ret = avcodec_parameters_to_context(codec_ctx, stream->codecpar);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "Failed to copy decoder parameters to input decoder context "
                   "for stream #%u\n", i);
            return ret;
        }

        av_log(NULL, AV_LOG_INFO, "#%d: stream.time_base=%d/%d, avg_frame_rate=%d/%d.\n",
                i, stream->time_base.num, stream->time_base.den,
                stream->avg_frame_rate.num, stream->avg_frame_rate.den);

        if (!strncmp(dec->name, "h264_ni", strlen("h264_ni")) ||
            !strncmp(dec->name, "h265_ni", strlen("h265_ni"))) {
            char str_devid[4] = {0};
            snprintf(str_devid, sizeof(str_devid), "%d", codec_worker->devid);
            av_opt_set(codec_ctx->priv_data, "dec", str_devid, 0);
            av_opt_set(codec_ctx->priv_data, "xcoder-params", codec_worker->decoder_params, 0);
            av_opt_set(codec_ctx->priv_data, "keep_alive_timeout", "3", 0);
        }

        if (codec_ctx->codec_type == AVMEDIA_TYPE_VIDEO)
            codec_ctx->framerate = av_guess_frame_rate(ifmt_ctx, stream, NULL);

        ret = avcodec_parameters_from_context(stream->codecpar, codec_ctx);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "Failed to copy decoder parameters from codec context for stream #%u", i);
            return ret;
        }

        /* Reencode video & audio and remux subtitles etc. */
        if (codec_ctx->codec_type == AVMEDIA_TYPE_VIDEO ||
                codec_ctx->codec_type == AVMEDIA_TYPE_AUDIO) {
            /* Open decoder */
            ret = avcodec_open2(codec_ctx, dec, NULL);
            if (ret < 0) {
                av_log(NULL, AV_LOG_ERROR, "Failed to open decoder for stream #%u\n", i);
                return ret;
            }
        }
        stream_ctx[i].dec_ctx = codec_ctx;
        stream_ctx[i].last_decoded_pts = AV_NOPTS_VALUE;
        stream_ctx[i].last_decoded_pts = AV_NOPTS_VALUE;
    }

    av_dump_format(ifmt_ctx, 0, input_file, 0);
    codec_worker->ifmt_ctx = ifmt_ctx;
    return 0;
}

/* Literally the NI encoder seession is opened in the send frame function. So
 * when there is no decoded or filtered frame sent to the encoder, the NI
 * encoder is not opened actually. It is no good calling avcodec_close function
 * in such moment. So we apply the need_to_reopen flag to judge whether it
 * needs to call avcodec alloc/free function according to the number of encoded
 * frames.
 *
 * close and reopen encoder
 */
static int reset_encoder(struct codec_worker *codec_worker, const char *codec_name,
                         int stream_index, const AVFrame *filt_frame)
{
    int ret;
    AVCodecContext *dec_ctx, *enc_ctx;
    const AVCodec *encoder;
    StreamContext *stream_ctx = codec_worker->stream_ctx;
    AVFormatContext *ofmt_ctx = codec_worker->ofmt_ctx;
    enc_ctx = stream_ctx[stream_index].enc_ctx;
    dec_ctx = stream_ctx[stream_index].dec_ctx;

    avcodec_free_context(&enc_ctx);

    if (dec_ctx->codec_type == AVMEDIA_TYPE_AUDIO && codec_worker->audio_disable) {
        return AVERROR(EINVAL);
    }

    if (dec_ctx->codec_type == AVMEDIA_TYPE_VIDEO
            || dec_ctx->codec_type == AVMEDIA_TYPE_AUDIO) {
        /* in this example, we choose transcoding to same codec */
        if (dec_ctx->codec_type == AVMEDIA_TYPE_VIDEO && codec_name) {
            encoder = avcodec_find_encoder_by_name(codec_name);
        } else {
            encoder = avcodec_find_encoder(dec_ctx->codec_id);
        }
        if (!encoder) {
            av_log(NULL, AV_LOG_FATAL, "Necessary encoder not found\n");
            return AVERROR_INVALIDDATA;
        }
        enc_ctx = avcodec_alloc_context3(encoder);
        if (!enc_ctx) {
            av_log(NULL, AV_LOG_FATAL, "Failed to allocate the encoder context\n");
            return AVERROR(ENOMEM);
        }

        /* In this example, we transcode to same properties (picture size,
         * sample rate etc.). These properties can be changed for output
         * streams easily using filters */
        if (dec_ctx->codec_type == AVMEDIA_TYPE_VIDEO) {
            enc_ctx->width = filt_frame->width;
            enc_ctx->height = filt_frame->height;
            enc_ctx->sample_aspect_ratio = dec_ctx->sample_aspect_ratio;
            if (filt_frame->hw_frames_ctx) {
                AVHWFramesContext* pAVHFWCtx = (AVHWFramesContext *)filt_frame->hw_frames_ctx->data;
                enc_ctx->pix_fmt = pAVHFWCtx->format;
                enc_ctx->sw_pix_fmt = pAVHFWCtx->sw_format;
            } else if (encoder->pix_fmts) {
                /* take first format from list of supported formats */
                enc_ctx->pix_fmt = encoder->pix_fmts[0];
            } else {
                enc_ctx->pix_fmt = dec_ctx->pix_fmt;
            }
            /* video time_base can be set to whatever is handy and supported by encoder */
            enc_ctx->time_base = stream_ctx[stream_index].enc_time_base;

            if (!strncmp(encoder->name, "h264_ni", strlen("h264_ni")) ||
                !strncmp(encoder->name, "h265_ni", strlen("h265_ni"))
#ifdef AV1_SUPPORTED // AV1 not supported in 3.1.1
                || !strncmp(encoder->name, "av1_ni", strlen("av1_ni"))
#endif
            ) {
                char str_devid[4] = {0};

                snprintf(str_devid, sizeof(str_devid), "%d", codec_worker->devid);
                av_opt_set(enc_ctx->priv_data, "enc", str_devid, 0);

                if (codec_worker->encoder_params) {
                    av_opt_set(enc_ctx->priv_data, "xcoder-params",
                            codec_worker->encoder_params, 0);
                }
            }
        } else {
            enc_ctx->sample_rate = dec_ctx->sample_rate;
#if LIBAVCODEC_VERSION_MAJOR >= 60 //6.1
            enc_ctx->ch_layout = dec_ctx->ch_layout;
            //enc_ctx->ch_layout.nb_channels = enc_ctx->ch_layout.nb_channels;
#else
            enc_ctx->channel_layout = dec_ctx->channel_layout;
            enc_ctx->channels = av_get_channel_layout_nb_channels(enc_ctx->channel_layout);
#endif
            /* take first format from list of supported formats */
            enc_ctx->sample_fmt = encoder->sample_fmts[0];
            enc_ctx->time_base = (AVRational){1, enc_ctx->sample_rate};
        }

        if ((ofmt_ctx->oformat->flags & AVFMT_GLOBALHEADER) && !codec_worker->repeat_header)
            enc_ctx->flags |= AV_CODEC_FLAG_GLOBAL_HEADER;

        /* Third parameter can be used to pass settings to encoder */
        ret = avcodec_open2(enc_ctx, encoder, NULL);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "Cannot open video encoder for stream #%u\n", stream_index);
            return ret;
        }

        stream_ctx[stream_index].enc_ctx = enc_ctx;
    }
    return 0;
}



static int open_output_file2(struct codec_worker *codec_worker)
{
    int ret = 0;
    AVFormatContext *ofmt_ctx = codec_worker->ofmt_ctx;
    char *output_file = codec_worker->output_file;
    av_dump_format(ofmt_ctx, 0, output_file, 1);

    if (!(ofmt_ctx->oformat->flags & AVFMT_NOFILE)) {
        ret = avio_open(&ofmt_ctx->pb, output_file, AVIO_FLAG_WRITE);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "Could not open output file '%s'", output_file);
            return ret;
        }
    }

    /* init muxer, write output file header */
    ret = avformat_write_header(ofmt_ctx, NULL);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Error occurred when opening output file\n");
        return ret;
    }
    return 0;
}

/* codec_name is video codec name. */
static int set_output_file(struct codec_worker *codec_worker, const char *codec_name)
{
    AVFormatContext *ofmt_ctx = NULL;
    char output_file[256] = { 0 };
    unsigned int i;
    AVFormatContext *ifmt_ctx = codec_worker->ifmt_ctx;
    AVStream **out_streams = NULL;

    if (strcmp(codec_worker->suffix, "null") == 0) {
        av_log(NULL, AV_LOG_DEBUG, "setting output file as null.\n");
        snprintf(output_file, sizeof(output_file), "null");
    }
    else {
        snprintf(output_file, sizeof(output_file) - 1, "%s/output-%d.%s", codec_worker->output_dir,
            codec_worker->index, codec_worker->suffix);
    }

    if (!strcmp(output_file, "null")) {
        avformat_alloc_output_context2(&ofmt_ctx, NULL, "null", NULL);
    } else {
        /* Note: The file extension string should be in output_file here for
                 avformat_alloc_output_context2() to auto-detect output format */
        avformat_alloc_output_context2(&ofmt_ctx, NULL, NULL, output_file);
    }
    if (!ofmt_ctx) {
        av_log(NULL, AV_LOG_ERROR, "Could not create output context\n");
        return AVERROR_UNKNOWN;
    }

    // Allocate array to store output stream pointers
    out_streams = av_calloc(ifmt_ctx->nb_streams, sizeof(AVStream*));
    if (!out_streams) {
        av_log(NULL, AV_LOG_ERROR, "Could not allocate output stream pointer array\n");
        avformat_free_context(ofmt_ctx);
        return AVERROR(ENOMEM);
    }

    // Create all output streams in input order
    for (i = 0; i < ifmt_ctx->nb_streams; i++) {
        AVStream *out_stream = avformat_new_stream(ofmt_ctx, NULL);
        if (!out_stream) {
            av_log(NULL, AV_LOG_ERROR, "Failed allocating output stream for input stream %u\n", i);
            av_freep(&out_streams);
            avformat_free_context(ofmt_ctx);
            return AVERROR_UNKNOWN;
        }
        out_streams[i] = out_stream;
    }

    strcpy(codec_worker->output_file, output_file);
    codec_worker->ofmt_ctx = ofmt_ctx;
    codec_worker->nb_streams = ifmt_ctx->nb_streams;
    codec_worker->out_streams = out_streams;
    return 0;
}

static int init_filter(FilteringContext* fctx, AVCodecContext *dec_ctx,
                       AVStream *in_stream, const char *filter_spec, int *audio_filter)
{
    char args[512] = {0};
    int ret = 0;
    const AVFilter *buffersrc = NULL;
    const AVFilter *buffersink = NULL;
    AVFilterContext *buffersrc_ctx = NULL;
    AVFilterContext *buffersink_ctx = NULL;
    AVFilterInOut *outputs;
    AVFilterInOut *inputs;
    AVFilterGraph *filter_graph = avfilter_graph_alloc();

    enum AVPixelFormat pix_fmts[] = {AV_PIX_FMT_NI_QUAD,AV_PIX_FMT_BGRA,AV_PIX_FMT_BGRP,AV_PIX_FMT_YUV420P,AV_PIX_FMT_NONE};

    if (!filter_graph) {
        ret = AVERROR(ENOMEM);
        goto end;
    }

    ret = avfilter_graph_parse2(filter_graph, filter_spec, &inputs, &outputs);
    if (ret < 0)
        goto end;

    if (dec_ctx->codec_type == AVMEDIA_TYPE_VIDEO) {
        buffersrc = avfilter_get_by_name("buffer");
        buffersink = avfilter_get_by_name("buffersink");
        if (!buffersrc || !buffersink) {
            av_log(NULL, AV_LOG_ERROR, "filtering source or sink element not found\n");
            ret = AVERROR_UNKNOWN;
            goto end;
        }

        snprintf(args, sizeof(args),
                "video_size=%dx%d:pix_fmt=%d:time_base=%d/%d:pixel_aspect=%d/%d",
                dec_ctx->width, dec_ctx->height, dec_ctx->pix_fmt,
                dec_ctx->pkt_timebase.num, dec_ctx->pkt_timebase.den,
                dec_ctx->sample_aspect_ratio.num,
                dec_ctx->sample_aspect_ratio.den);

        ret = avfilter_graph_create_filter(&buffersrc_ctx, buffersrc, "in",
                args, NULL, filter_graph);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "Cannot create buffer source\n");
            goto end;
        }

        if (dec_ctx->hw_frames_ctx != NULL) {
            AVBufferSrcParameters *par = av_buffersrc_parameters_alloc();
            if (!par)
                return AVERROR(ENOMEM);
            memset(par, 0, sizeof(*par));
            par->format = AV_PIX_FMT_NONE;
            par->hw_frames_ctx = dec_ctx->hw_frames_ctx;
#if LIBAVCODEC_VERSION_MAJOR  >= 61 // 7.1
            par->color_range = dec_ctx->color_range;
            par->color_space = dec_ctx->colorspace;
#endif
            ret = av_buffersrc_parameters_set(buffersrc_ctx, par);
            if (ret < 0)
                goto end;
            av_freep(&par);
        }

        ret = avfilter_graph_create_filter(&buffersink_ctx, buffersink, "out",
                NULL, NULL, filter_graph);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "Cannot create buffer sink\n");
            goto end;
        }


        ret = av_opt_set_int_list(buffersink_ctx, "pix_fmts", pix_fmts,
                              AV_PIX_FMT_NONE, AV_OPT_SEARCH_CHILDREN);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "Cannot set output pixel format\n");
            goto end;
        }
    } else if (dec_ctx->codec_type == AVMEDIA_TYPE_AUDIO) {
        buffersrc = avfilter_get_by_name("abuffer");
        buffersink = avfilter_get_by_name("abuffersink");
        if (!buffersrc || !buffersink) {
            av_log(NULL, AV_LOG_ERROR, "filtering source or sink element not found\n");
            ret = AVERROR_UNKNOWN;
            goto end;
        }


#if LIBAVCODEC_VERSION_MAJOR >= 60 //6.1
        if (av_channel_layout_check(&dec_ctx->ch_layout))
#else
        if (!dec_ctx->channel_layout)
#endif
        {
#if LIBAVCODEC_VERSION_MAJOR >= 60 //6.1
            av_channel_layout_default(&dec_ctx->ch_layout, dec_ctx->ch_layout.nb_channels);
#else
            dec_ctx->channel_layout = av_get_default_channel_layout(dec_ctx->channels);
#endif
        }

#if LIBAVCODEC_VERSION_MAJOR >= 60 //6.1
        char layout_str[64] = {0};
        av_channel_layout_describe(&dec_ctx->ch_layout, layout_str, sizeof(layout_str));
        snprintf(args, sizeof(args),
                "time_base=%d/%d:sample_rate=%d:sample_fmt=%s:channel_layout=%s",
                dec_ctx->pkt_timebase.num, dec_ctx->pkt_timebase.den, dec_ctx->sample_rate,
                av_get_sample_fmt_name(dec_ctx->sample_fmt),
                layout_str);
#else
        snprintf(args, sizeof(args),
                "time_base=%d/%d:sample_rate=%d:sample_fmt=%s:channel_layout=0x%"PRIx64,
                dec_ctx->pkt_timebase.num, dec_ctx->pkt_timebase.den, dec_ctx->sample_rate,
                av_get_sample_fmt_name(dec_ctx->sample_fmt),
                dec_ctx->channel_layout);
#endif

        ret = avfilter_graph_create_filter(&buffersrc_ctx, buffersrc, "in",
                args, NULL, filter_graph);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "Cannot create audio buffer source\n");
            goto end;
        }

        ret = avfilter_graph_create_filter(&buffersink_ctx, buffersink, "out",
                NULL, NULL, filter_graph);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "Cannot create audio buffer sink\n");
            goto end;
        }

        ret = av_opt_set_bin(buffersink_ctx, "sample_fmts",
                (uint8_t*)&dec_ctx->sample_fmt, sizeof(dec_ctx->sample_fmt),
                AV_OPT_SEARCH_CHILDREN);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "Cannot set output sample format\n");
            goto end;
        }

#if LIBAVCODEC_VERSION_MAJOR >= 60 //6.1
        ret = av_opt_set(buffersink_ctx, "ch_layouts",
                         layout_str, AV_OPT_SEARCH_CHILDREN);
#else
        ret = av_opt_set_bin(buffersink_ctx, "channel_layouts",
                (uint8_t*)&dec_ctx->channel_layout,
                sizeof(dec_ctx->channel_layout), AV_OPT_SEARCH_CHILDREN);
#endif
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "Cannot set output channel layout\n");
            goto end;
        }

        ret = av_opt_set_bin(buffersink_ctx, "sample_rates",
                (uint8_t*)&dec_ctx->sample_rate, sizeof(dec_ctx->sample_rate),
                AV_OPT_SEARCH_CHILDREN);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "Cannot set output sample rate\n");
            goto end;
        }
    } else {
        ret = AVERROR_UNKNOWN;
        goto end;
    }

    ret = avfilter_link(buffersrc_ctx, 0, inputs->filter_ctx, inputs->pad_idx);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to link input filter\n");
        goto end;
    }

    ret = avfilter_link(outputs->filter_ctx, outputs->pad_idx, buffersink_ctx, 0);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to link output filter\n");
        goto end;
    }

    if ((ret = avfilter_graph_config(filter_graph, NULL)) < 0)
        goto end;

    if (dec_ctx->codec_type == AVMEDIA_TYPE_AUDIO) {
        switch (dec_ctx->codec_id) {
            case AV_CODEC_ID_AAC:
                av_buffersink_set_frame_size(buffersink_ctx, 1024);
                *audio_filter = 1;
                break;

            case AV_CODEC_ID_ADPCM_SWF:
                av_buffersink_set_frame_size(buffersink_ctx, 4096);
                *audio_filter = 1;
                break;

            default:
                break;
        }
    }

    /* Fill FilteringContext */
    fctx->buffersrc_ctx = buffersrc_ctx;
    fctx->buffersink_ctx = buffersink_ctx;
    fctx->filter_graph = filter_graph;

end:
    avfilter_inout_free(&inputs);
    avfilter_inout_free(&outputs);

    return ret;
}

static int init_filters(struct codec_worker *codec_worker)
{
    const char *filter_spec;
    unsigned int i;
    int ret;
    AVFormatContext *ifmt_ctx = codec_worker->ifmt_ctx;
    StreamContext *stream_ctx = codec_worker->stream_ctx;
    int *audio_filter = &codec_worker->audio_filter;
    FilteringContext *filter_ctx;

    filter_ctx = av_malloc_array(ifmt_ctx->nb_streams, sizeof(*filter_ctx));
    if (!filter_ctx)
        return AVERROR(ENOMEM);

    for (i = 0; i < ifmt_ctx->nb_streams; i++) {
        filter_ctx[i].buffersrc_ctx  = NULL;
        filter_ctx[i].buffersink_ctx = NULL;
        filter_ctx[i].filter_graph   = NULL;
        if (!(ifmt_ctx->streams[i]->codecpar->codec_type == AVMEDIA_TYPE_AUDIO
                || ifmt_ctx->streams[i]->codecpar->codec_type == AVMEDIA_TYPE_VIDEO))
            continue;

        if (stream_ctx[i].stream_copy || stream_ctx[i].disabled)
            continue;

        if (ifmt_ctx->streams[i]->codecpar->codec_type == AVMEDIA_TYPE_VIDEO)
            filter_spec = codec_worker->filter_params != NULL && strlen(codec_worker->filter_params) ?
                          codec_worker->filter_params : "null";
        else
            filter_spec = "anull"; /* passthrough (dummy) filter for audio */
        ret = init_filter(&filter_ctx[i], stream_ctx[i].dec_ctx, ifmt_ctx->streams[i],
                filter_spec, audio_filter);
        if (ret)
            return ret;
    }

    codec_worker->filter_ctx = filter_ctx;
    return 0;
}

static int encode_write_frame(struct codec_worker *codec_worker, AVFrame *filt_frame,
                              unsigned int stream_index, int *got_frame)
{
    int ret;
    int got_frame_local;
    AVPacket enc_pkt = { 0 };
    StreamContext *stream_ctx = codec_worker->stream_ctx;
    AVFormatContext* ofmt_ctx = codec_worker->ofmt_ctx;

    if (!got_frame)
        got_frame = &got_frame_local;

    av_log(NULL, AV_LOG_DEBUG, "#%d: Encoding frame\n", stream_index);
    /* encode filtered frame */
    enc_pkt.data = NULL;
    enc_pkt.size = 0;

    ret = avcodec_send_frame(stream_ctx[stream_index].enc_ctx, filt_frame);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "#%d: fail to send frame.\n", stream_index);
        return ret;
    }
    if (filt_frame) {
        av_log(NULL, AV_LOG_DEBUG, "mux: -> pts=%ld,dts=%ld\n", enc_pkt.pts, enc_pkt.dts);
        av_frame_free(&filt_frame);
    }
    else {
        av_log(NULL, AV_LOG_DEBUG, "mux: -> flush encoder: pts=%ld,dts=%ld\n", enc_pkt.pts, enc_pkt.dts);
    }

    while (1) {
        ret = avcodec_receive_packet(stream_ctx[stream_index].enc_ctx, &enc_pkt);
        if (ret >= 0) {
            if (enc_pkt.size && enc_pkt.data) {
                *got_frame = 1;
            }
        } else if (ret == AVERROR(EAGAIN) || ret == AVERROR_EOF) {
            *got_frame = 0;
            ret = 0;
        } else {
            av_log(NULL, AV_LOG_ERROR, "#%d: fail to receive packet 0x%x\n", stream_index, ret);
        }

        if (ret < 0)
            break;
        if (!(*got_frame))
            break;

        if ((stream_ctx[stream_index].last_encoded_pts != AV_NOPTS_VALUE) &&
                (enc_pkt.pts == stream_ctx[stream_index].last_encoded_pts)) {
            av_log(NULL, AV_LOG_ERROR, "encoder: same pts!!! pts=%ld,last_pts=%ld\n", enc_pkt.pts,
                    stream_ctx[stream_index].last_encoded_pts);
        }
        if ((stream_ctx[stream_index].last_encoded_dts != AV_NOPTS_VALUE) &&
                (enc_pkt.dts <= stream_ctx[stream_index].last_encoded_dts)) {
            av_log(NULL, AV_LOG_ERROR, "encoder: Non-monotonically increasing dts!!! dts=%ld,last_dts=%ld\n",
                    enc_pkt.dts, stream_ctx[stream_index].last_encoded_dts);
        }

        stream_ctx[stream_index].last_encoded_pts = enc_pkt.pts;
        stream_ctx[stream_index].last_encoded_dts = enc_pkt.dts;

        /* prepare packet for muxing */
        enc_pkt.stream_index = stream_index;
        if (codec_worker->rescale_ts) {
            av_packet_rescale_ts(&enc_pkt,
                    stream_ctx[stream_index].enc_ctx->time_base,
                    ofmt_ctx->streams[stream_index]->time_base);
        }
        av_log(NULL, AV_LOG_DEBUG, "#%d: mux: <- pts=%ld,dts=%ld\n", stream_index, enc_pkt.pts, enc_pkt.dts);
        /* mux encoded frame */
        pthread_mutex_lock(&codec_worker->remux_lock);
        ret = av_interleaved_write_frame(ofmt_ctx, &enc_pkt);
        pthread_mutex_unlock(&codec_worker->remux_lock);
        av_packet_unref(&enc_pkt);
        if (ret < 0)
            break;
    }

    return ret;
}

static int init_encoder_ctx(struct codec_worker *codec_worker,AVFrame *frame,
                int stream_index,const char *codec_name,AVFormatContext *ofmt_ctx)
{
    int ret = 0;
    int stream_copy = 0;
    // unsigned int i = 0;
    const AVCodec *encoder;
    AVCodecContext *decoder_ctx = NULL,*encoder_ctx = NULL;
    AVStream *out_stream,*in_stream;
    StreamContext *stream_ctx = codec_worker->stream_ctx;
    AVFormatContext *ifmt_ctx = codec_worker->ifmt_ctx;
    if((codec_worker->inited_encoder_ctx)[stream_index])
    {
        return 0;
    }
    in_stream = ifmt_ctx->streams[stream_index];
    decoder_ctx = stream_ctx[stream_index].dec_ctx;
    stream_ctx[stream_index].last_encoded_pts = AV_NOPTS_VALUE;
    stream_ctx[stream_index].last_encoded_dts = AV_NOPTS_VALUE;
    out_stream = codec_worker->out_streams[stream_index];
    if (!out_stream) {
        av_log(NULL, AV_LOG_ERROR, "Output stream pointer is NULL for stream %d\n", stream_index);
        return AVERROR_UNKNOWN;
    }
    if (decoder_ctx->codec_type == AVMEDIA_TYPE_AUDIO || decoder_ctx->codec_type == AVMEDIA_TYPE_VIDEO)
    {
        if(decoder_ctx->codec_type == AVMEDIA_TYPE_AUDIO)
        {
            if(codec_worker->audio_disable)
            {
                ++(codec_worker->count_inited_encoder_ctx);
                codec_worker->inited_encoder_ctx[stream_index]=1;
                stream_ctx[stream_index].disabled = 1;
                return 0;
            }
            encoder = avcodec_find_encoder(decoder_ctx->codec_id);
            if (!encoder) {
                stream_copy = 1;
            }
        }
        else
        {
             /* in this example, we choose transcoding to same codec */
            if (!codec_name) {
                encoder = avcodec_find_encoder(decoder_ctx->codec_id);
            } else if (!strcmp(codec_name, "copy")) {
                stream_copy = 1;
                encoder = NULL;
            } else {
                encoder = avcodec_find_encoder_by_name(codec_name);
            }
        }
        if (!stream_copy && !encoder)
        {
            av_log(NULL, AV_LOG_FATAL, "Necessary encoder not found\n");
            return AVERROR_INVALIDDATA;
        }

        encoder_ctx = avcodec_alloc_context3(encoder);

        if ((ofmt_ctx->oformat->flags & AVFMT_GLOBALHEADER) && !codec_worker->repeat_header)
        {
            encoder_ctx->flags |= AV_CODEC_FLAG_GLOBAL_HEADER;
        }



        if (!stream_copy)
        {
        /* In this example, we transcode to same properties (picture size,
            * sample rate etc.). These properties can be changed for output
            * streams easily using filters */
            if (encoder_ctx->codec_type == AVMEDIA_TYPE_VIDEO)
            {
                encoder_ctx->bit_rate = codec_worker->bitrate;
                encoder_ctx->height = frame->height;
                encoder_ctx->width = frame->width;
                encoder_ctx->sample_aspect_ratio = frame->sample_aspect_ratio;
                encoder_ctx->pix_fmt = frame->format;
                if (frame->hw_frames_ctx)
                {
                    AVHWFramesContext* pAVHFWCtx = (AVHWFramesContext*)frame->hw_frames_ctx->data;
                    encoder_ctx->sw_pix_fmt = pAVHFWCtx->sw_format;
                }

                /* video time_base can be set to whatever is handy and supported by encoder */
                encoder_ctx->time_base = av_inv_q(decoder_ctx->framerate);

                if (encoder)
                {
                    out_stream->codecpar->codec_id = encoder->id;
                    if (!strncmp(encoder->name, "h264_ni", strlen("h264_ni")) ||
                        !strncmp(encoder->name, "h265_ni", strlen("h265_ni"))
                        #ifdef AV1_SUPPORTED // AV1 not supported in 3.1.1
                                        || !strncmp(encoder->name, "av1_ni", strlen("av1_ni"))
                        #endif
                    )
                    {
                        char str_devid[4] = {0};

                        snprintf(str_devid, sizeof(str_devid), "%d", codec_worker->devid);
                        av_opt_set(encoder_ctx->priv_data, "enc", str_devid, 0);

                        if (codec_worker->encoder_params) {
                            av_opt_set(encoder_ctx->priv_data, "xcoder-params",
                                    codec_worker->encoder_params, 0);
                        }
                    }
                }
            }
            else
            {
                encoder_ctx->sample_rate = decoder_ctx->sample_rate;
#if LIBAVCODEC_VERSION_MAJOR >= 60 //6.1
                encoder_ctx->ch_layout = decoder_ctx->ch_layout;
                // encoder_ctx->ch_layout.nb_channels = encoder_ctx->ch_layout.nb_channels;
#else
                encoder_ctx->channel_layout = decoder_ctx->channel_layout;
                encoder_ctx->channels = av_get_channel_layout_nb_channels(encoder_ctx->channel_layout);
#endif
                encoder_ctx->time_base = (AVRational){1, encoder_ctx->sample_rate};
                if (encoder) {
                    out_stream->codecpar->codec_id = encoder->id;
                    /* take first format from list of supported formats */
                    encoder_ctx->sample_fmt = encoder->sample_fmts[0];
                }
            }

            if (ofmt_ctx->oformat->flags & AVFMT_GLOBALHEADER)
                encoder_ctx->flags |= AV_CODEC_FLAG_GLOBAL_HEADER;

            /* Third parameter can be used to pass settings to encoder */
            ret = avcodec_open2(encoder_ctx, encoder, NULL);
            if (ret < 0)
            {
                av_log(NULL, AV_LOG_ERROR, "Cannot open video encoder for stream #%u\n", stream_index);
                return ret;
            }

            ret = avcodec_parameters_from_context(out_stream->codecpar, encoder_ctx);
            if (ret < 0)
            {
                av_log(NULL, AV_LOG_ERROR, "Failed to copy encoder parameters to output stream #%u\n", stream_index);
                return ret;
            }
        }
        else
        {
            ret = avcodec_parameters_to_context(encoder_ctx, in_stream->codecpar);
            if (ret < 0) {
                av_log(NULL, AV_LOG_ERROR, "Failed to copy encoder parameters to encoder context for stream #%u\n", stream_index);
                return ret;
            }

            ret = avcodec_parameters_copy(out_stream->codecpar, in_stream->codecpar);
            if (ret < 0) {
                av_log(NULL, AV_LOG_ERROR, "Copying parameters for stream #%u failed\n", stream_index);
                return ret;
            }
        }
        if (!codec_worker->rescale_ts)
        {
            out_stream->time_base = in_stream->time_base;
        }
        else
        {
            out_stream->time_base = encoder_ctx->time_base;
        }
        av_log(NULL, AV_LOG_INFO, "out_stream->time_base = %d/%d\n", out_stream->time_base.num, out_stream->time_base.den);

        stream_ctx[stream_index].enc_ctx = encoder_ctx;
        stream_ctx[stream_index].enc_time_base = encoder_ctx->time_base;
        stream_ctx[stream_index].stream_copy = stream_copy;
        stream_ctx[stream_index].decoding_needed = !stream_copy;
    }
    else if (decoder_ctx->codec_type == AVMEDIA_TYPE_UNKNOWN)
    {
        av_log(NULL, AV_LOG_FATAL, "Elementary stream #%d is of unknown type, cannot proceed\n", stream_index);
        return AVERROR_INVALIDDATA;
    }
    else
    {
        /* if this stream must be remuxed */
        ret = avcodec_parameters_copy(out_stream->codecpar, in_stream->codecpar);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "Copying parameters for stream #%u failed\n", stream_index);
            return ret;
        }
        out_stream->time_base = in_stream->time_base;
    }
    (codec_worker->inited_encoder_ctx)[stream_index] = 1;
    ++(codec_worker->count_inited_encoder_ctx);
    return ret;
}

static int send_encode_frame(struct codec_worker *codec_worker,
                             AVFrame *filt_frame, unsigned int stream_index)
{
    int ret = 0;
    struct frame_info *frame_info;

    frame_info = alloc_frame_info();
    if (frame_info == NULL) {
        ret = AVERROR(ENOMEM);
        av_frame_free(&filt_frame);
        return ret;
    }

    av_log(NULL, AV_LOG_DEBUG, "send frame: -> pts=%ld, dts=%ld\n", filt_frame->pts, filt_frame->pkt_dts);

    if (codec_worker->rescale_ts) {
        if (filt_frame->pts != AV_NOPTS_VALUE) {
            FilteringContext *filter_ctx = codec_worker->filter_ctx;
            int64_t start_time = 0;
            StreamContext *stream_ctx = codec_worker->stream_ctx;

            AVRational filter_tb = av_buffersink_get_time_base(filter_ctx[stream_index].buffersink_ctx);

            filt_frame->pts =
                av_rescale_q(filt_frame->pts, filter_tb, stream_ctx[stream_index].enc_time_base) -
                av_rescale_q(start_time, AV_TIME_BASE_Q, stream_ctx[stream_index].enc_time_base);
        }
    }

    frame_info->frame = filt_frame;
    frame_info->stream_index = stream_index;
    frame_info->is_hw_frame = filt_frame->hw_frames_ctx != NULL;
    if(stream_index == codec_worker->video_stream_num)
    {
        codec_worker->total_decoder_frames++;
    }
    av_log(NULL, AV_LOG_DEBUG, "#%d: decoder: frame_info=%p,frame=%p,"
            "stream_index=%d,total=%lu.\n", stream_index,
            frame_info, frame_info->frame, frame_info->stream_index,
            codec_worker->total_decoder_frames);

    av_log(NULL, AV_LOG_DEBUG, "send frame: <- pts=%ld, dts=%ld\n", filt_frame->pts, filt_frame->pkt_dts);

    if(!codec_worker->decoder_launch_encoder)
    {
        pthread_mutex_lock(&codec_worker->launch_encoder_lock);
        ret = init_encoder_ctx(codec_worker,filt_frame,stream_index,codec_worker->encoder_name,codec_worker->ofmt_ctx);
        if(ret < 0)
        {
            av_log(NULL,AV_LOG_ERROR,"fail to init encoder context.\n");
            codec_worker->force_exit = 1;
            pthread_cond_signal(&codec_worker->lanuch_encoder_cond);
            pthread_mutex_unlock(&codec_worker->launch_encoder_lock);
            codec_worker->decoder_launch_encoder = 1;
            return ret;
        }
        if(codec_worker->count_inited_encoder_ctx == codec_worker->nb_streams)
        {
            pthread_cond_signal(&codec_worker->lanuch_encoder_cond);
            codec_worker->decoder_launch_encoder = 1;
        }
        pthread_mutex_unlock(&codec_worker->launch_encoder_lock);
    }


    pthread_mutex_lock(&codec_worker->frame_lock);
    add_list_tail(this_list_node(frame_info), &codec_worker->frame_link);
    pthread_cond_signal(&codec_worker->consume_cond);
    codec_worker->filt_frames++;
    if (codec_worker->filt_frames >= __avframe_blocking_count(frame_info->is_hw_frame)) {
        do {
            pthread_cond_wait(&codec_worker->product_cond,
                    &codec_worker->frame_lock);
            if ((codec_worker->filt_frames < __avframe_blocking_count(frame_info->is_hw_frame)) ||
                    codec_worker->should_exit) {
                break;
            }
        } while (1);
    }
    pthread_mutex_unlock(&codec_worker->frame_lock);

    return ret;
}

static int filter_encode_write_frame(struct codec_worker *codec_worker,
                                     AVFrame *frame, unsigned int stream_index)
{
    int ret = 0;
    AVFrame *filt_frame;
    FilteringContext *filter_ctx = codec_worker->filter_ctx;

    if (frame) {
        av_log(NULL, AV_LOG_DEBUG, "#%d: Pushing decoded frame to filters: pts=%lu, dts=%lu.\n",
                stream_index, frame->pts, frame->pkt_dts);
    }

    if (((codec_worker->bypass_filter) && (codec_worker->ifmt_ctx->streams[stream_index]->codecpar->codec_type == AVMEDIA_TYPE_VIDEO)) ||
        ((!codec_worker->audio_filter) && (codec_worker->ifmt_ctx->streams[stream_index]->codecpar->codec_type == AVMEDIA_TYPE_AUDIO))) {
        if (frame) {
            filt_frame = av_frame_alloc();
            if (!filt_frame) {
                ret = AVERROR(ENOMEM);
                return ret;
            }
            av_frame_move_ref(filt_frame, frame);

            ret = send_encode_frame(codec_worker, filt_frame, stream_index);
            if (ret < 0) {
                av_log(NULL, AV_LOG_ERROR, "#%d: failed to send encode frame: %d\n", stream_index, ret);
                return ret;
            }
        }
    } else {
        /* push the decoded frame into the filtergraph */
        ret = av_buffersrc_add_frame_flags(filter_ctx[stream_index].buffersrc_ctx,
                frame, 0);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "#%d: Error while feeding the filtergraph, ret=%d.\n",
                    stream_index, ret);
            return ret;
        }

        /* pull filtered frames from the filtergraph */
        while (!codec_worker->should_exit) {
            filt_frame = av_frame_alloc();
            if (!filt_frame) {
                ret = AVERROR(ENOMEM);
                return ret;
            }

            av_log(NULL, AV_LOG_DEBUG, "#%d: Pulling filtered frame from filters\n", stream_index);
            ret = av_buffersink_get_frame(filter_ctx[stream_index].buffersink_ctx,
                    filt_frame);
            if (ret < 0) {
                /* if no more frames for output - returns AVERROR(EAGAIN)
                 * if flushed and no more frames for output - returns AVERROR_EOF
                 * rewrite retcode to 0 to show it as normal procedure completion
                 */
                if (ret == AVERROR(EAGAIN) || ret == AVERROR_EOF)
                    ret = 0;
                av_frame_free(&filt_frame);
                break;
            }

            filt_frame->pict_type = AV_PICTURE_TYPE_NONE;

            ret = send_encode_frame(codec_worker, filt_frame, stream_index);
            if (ret < 0) {
                av_log(NULL, AV_LOG_ERROR, "#%d: failed to send encode frame: %d\n",
                        stream_index, ret);
                break;
            }
        }
    }

    return ret;
}

static int flush_encoder(struct codec_worker *codec_worker, unsigned int stream_index)
{
    int ret;
    int got_frame;
    StreamContext *stream_ctx = codec_worker->stream_ctx;

    if (!(stream_ctx[stream_index].enc_ctx->codec->capabilities &
                AV_CODEC_CAP_DELAY))
        return 0;

    do {
        av_log(NULL, AV_LOG_INFO, "Flushing stream #%u encoder\n", stream_index);
        ret = encode_write_frame(codec_worker, NULL, stream_index, &got_frame);
        if (ret < 0)
            break;
        if (!got_frame)
            return 0;
    } while (0);
    return ret;
}

static void *encoder_thread(void *thread_data)
{
    struct frame_info *frame_info;
    AVFrame *filt_frame;
    int stream_index;
    struct codec_worker *codec_worker = (struct codec_worker *)thread_data;
    FilteringContext *filter_ctx = codec_worker->filter_ctx;
    StreamContext *stream_ctx = codec_worker->stream_ctx;
    AVCodecContext *enc_ctx;
    int nb_streams = codec_worker->nb_streams;
    int ret = 0;
    int i;
    if(!codec_worker->encoder_launched)
    {
        pthread_mutex_lock(&codec_worker->launch_encoder_lock);
        do{
            if(codec_worker->force_exit)
            {
                av_log(NULL, AV_LOG_WARNING, "encoder force exit before open output file\n");
                pthread_mutex_unlock(&codec_worker->launch_encoder_lock);
                goto end;
            }
            if(codec_worker->count_inited_encoder_ctx < codec_worker->nb_streams)
            {
                pthread_cond_wait(&codec_worker->lanuch_encoder_cond,&codec_worker->launch_encoder_lock);
            }
            else
            {
                ret = open_output_file2(codec_worker);
                pthread_mutex_unlock(&codec_worker->launch_encoder_lock);
                if(ret < 0)
                {
                    av_log(NULL, AV_LOG_ERROR, "Open output failed\n");
                    goto end;
                }
                break;
            }
        }while(1);

        codec_worker->encoder_launched = 1;


    }

    while (!codec_worker->force_exit) {
        pthread_mutex_lock(&codec_worker->frame_lock);
        if (list_empty(&codec_worker->frame_link)) {
            if (!codec_worker->should_exit) {
                pthread_cond_wait(&codec_worker->consume_cond, &codec_worker->frame_lock);
            } else {
                pthread_mutex_unlock(&codec_worker->frame_lock);
                break;
            }
        }

        if (list_empty(&codec_worker->frame_link)) {
            pthread_mutex_unlock(&codec_worker->frame_lock);
            continue;
        }

        frame_info = list_node_data(pick_list_head(&codec_worker->frame_link));
        assert(frame_info != NULL);
        codec_worker->filt_frames--;
        if (codec_worker->filt_frames < __avframe_blocking_count(frame_info->is_hw_frame)) {
            pthread_cond_signal(&codec_worker->product_cond);
        }
        pthread_mutex_unlock(&codec_worker->frame_lock);

        filt_frame = frame_info->frame;
        stream_index = frame_info->stream_index;
        enc_ctx = stream_ctx[stream_index].enc_ctx;
        //if encoder not yet started, skip checking resolution change
        if (enc_ctx && AVMEDIA_TYPE_VIDEO == enc_ctx->codec_type) {
            if (filt_frame->width != enc_ctx->width ||
                    filt_frame->height != enc_ctx->height) {
                av_log(NULL, AV_LOG_INFO, "stream %d resolution: %dx%d->%dx%d\n", stream_index,
                        enc_ctx->width, enc_ctx->height, filt_frame->width,
                        filt_frame->height);
                if (filter_ctx[stream_index].filter_graph) {
                    ret = flush_encoder(codec_worker, stream_index);
                    if (ret < 0) {
                        av_log(NULL, AV_LOG_ERROR, "#%d: Flushing encoder error during resolution change\n",
                                stream_index);
                        av_frame_free(&filt_frame);
                        free_frame_info(frame_info);
                        goto end;
                    }
                }

                ret = reset_encoder(codec_worker, codec_worker->encoder_name, stream_index, filt_frame);
                if (ret < 0) {
                    av_log(NULL, AV_LOG_ERROR, "#%d: failed to reopen output file.\n", stream_index);
                    av_frame_free(&filt_frame);
                    free_frame_info(frame_info);
                    goto end;
                }
            }
        }

        if ((codec_worker->force_source_keyframe) && (filt_frame->key_frame == 1)) {
            filt_frame->pict_type = AV_PICTURE_TYPE_I;
        } else {
            filt_frame->pict_type = AV_PICTURE_TYPE_NONE;
        }
        codec_worker->total_encoder_frames++;
//        frame_info->frame->pts = frame_info->frame->pkt_dts;
//        if (frame_info->frame->pkt_dts == AV_NOPTS_VALUE) {
//            frame_info->frame->pkt_dts = frame_info->frame->pts;
//        }
        av_log(NULL, AV_LOG_DEBUG, "#%d: encoder: frame_info=%p,frame=%p,stream=%d,"
                "total=%lu,pts=%lu,dts=%lu.\n", stream_index,
                frame_info, frame_info->frame, frame_info->stream_index,
                codec_worker->total_encoder_frames, frame_info->frame->pts,
                frame_info->frame->pkt_dts);
        ret = encode_write_frame(codec_worker, filt_frame, stream_index, NULL);
        free_frame_info(frame_info);
        if (ret < 0) {
            goto end;
        }
    }

    /* flush filters and encoders */
    for (i = 0; i < nb_streams; i++) {
        /* flush filter */
        if (!filter_ctx[i].filter_graph)
            continue;

        /* flush encoder */
        ret = flush_encoder(codec_worker, i);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "#%d: Flushing encoder failed\n", i);
            goto end;
        }
    }
    av_write_trailer(codec_worker->ofmt_ctx);

end:
    av_log(NULL, AV_LOG_INFO, "encoder exit: ret=0x%x.\n", -ret);
    codec_worker->encode_exited = 1;
    return (void *)((long)ret);
}

static int __decode(struct codec_worker *codec_worker, AVPacket *packet, int stream_index)
{
    int ret = 0;
    StreamContext *stream_ctx = codec_worker->stream_ctx;

    AVFrame *frame;
    AVPacket avpkt;

    if (packet) {
        avpkt = *packet;
//            av_packet_rescale_ts(&avpkt,
//                    ifmt_ctx->streams[stream_index]->time_base,
//                    stream_ctx[stream_index].dec_ctx->time_base);
    }

    ret = avcodec_send_packet(stream_ctx[stream_index].dec_ctx, packet ? &avpkt : NULL);
    if (ret < 0 && ret != AVERROR_EOF) {
        av_log(NULL, AV_LOG_ERROR, "#%d: failed to send packet.\n", stream_index);
        goto end;
    }

    frame = codec_worker->decoded_frame;
    while (ret >= 0 || !packet) {
        ret = avcodec_receive_frame(stream_ctx[stream_index].dec_ctx, frame);
        if (ret < 0) {
            if (ret != AVERROR(EAGAIN)) {
                if (ret != AVERROR_EOF) {
                    av_log(NULL, AV_LOG_ERROR, "failed to receive frame\n");
                }
            } else {
                ret = 0;
            }
            goto end;
        }
        if (codec_worker->last_decoded_pts) {
            frame->best_effort_timestamp =
#if LIBAVCODEC_VERSION_MAJOR >= 60 //6.1
                codec_worker->last_decoded_pts + frame->duration;
#else
                codec_worker->last_decoded_pts + frame->pkt_duration;
#endif
            av_log(NULL, AV_LOG_DEBUG, "%s %d NEW STREAM pts adjust to %ld\n",
                __func__, codec_worker->index, frame->best_effort_timestamp);
            codec_worker->last_decoded_pts = frame->best_effort_timestamp;
        }
        frame->pts = frame->best_effort_timestamp;
#if LIBAVCODEC_VERSION_MAJOR >= 60 //6.1
        if (stream_ctx[stream_index].max_duration < frame->duration) {
            stream_ctx[stream_index].max_duration = frame->duration;
        }
#else
        if (stream_ctx[stream_index].max_duration < frame->pkt_duration) {
            stream_ctx[stream_index].max_duration = frame->pkt_duration;
        }

#endif
        if (frame->pts != AV_NOPTS_VALUE) {
            if (stream_ctx[stream_index].max_pts < frame->pts) {
                stream_ctx[stream_index].max_pts = frame->pts;
            }
            if (stream_ctx[stream_index].min_pts > frame->pts) {
                stream_ctx[stream_index].min_pts = frame->pts;
            }
        } else {
#if LIBAVCODEC_VERSION_MAJOR >= 60 //6.1
            frame->pts = stream_ctx[stream_index].last_decoded_pts == AV_NOPTS_VALUE ? 0 :
                         stream_ctx[stream_index].last_decoded_pts + frame->duration;
#else
            frame->pts = stream_ctx[stream_index].last_decoded_pts + frame->pkt_duration;
#endif
            av_log(NULL, AV_LOG_DEBUG, "%s frame->pts == AV_NOPTS_VALUE, adjust to %ld\n", __func__, frame->pts);
        }

        if (packet) {
            av_log(NULL, AV_LOG_DEBUG, "decoded frame: stream=%d, pts=%ld, dts=%ld, "
                    "duration=%ld, best_effort=%ld\n",
                    stream_index, frame->pts, frame->pkt_dts,
#if LIBAVCODEC_VERSION_MAJOR >= 60 //6.1
                    frame->duration,
#else
                    frame->pkt_duration,
#endif
                    frame->best_effort_timestamp);
        } else {
            av_log(NULL, AV_LOG_DEBUG, "flush decoded frame: stream=%d, pts=%ld, dts=%ld, "
                    "duration=%ld, best_effort=%ld\n",
                    stream_index, frame->pts, frame->pkt_dts,
#if LIBAVCODEC_VERSION_MAJOR >= 60 //6.1
                    frame->duration,
#else
                    frame->pkt_duration,
#endif
                    frame->best_effort_timestamp);
        }

        if (stream_index == 0 &&
            stream_ctx[stream_index].last_decoded_pts != AV_NOPTS_VALUE &&
            frame->pts == stream_ctx[stream_index].last_decoded_pts) {
            av_log(NULL, AV_LOG_ERROR, "flush decoder: same pts!!!\n");
        }

        if (stream_index == 0 &&
            stream_ctx[stream_index].last_decoded_dts != AV_NOPTS_VALUE &&
            frame->pkt_dts == stream_ctx[stream_index].last_decoded_dts) {
            av_log(NULL, AV_LOG_ERROR, "flush decoder: same dts!!!\n");
        }

#if LIBAVCODEC_VERSION_MAJOR >= 60 //6.1
        stream_ctx[stream_index].total_duration += frame->duration;
#else
        stream_ctx[stream_index].total_duration += frame->pkt_duration;
#endif
        stream_ctx[stream_index].last_decoded_pts = frame->pts;
        stream_ctx[stream_index].last_decoded_dts = frame->pkt_dts;

        ret = filter_encode_write_frame(codec_worker, frame, stream_index);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "filter encode write frame error.\n");
            goto end;
        }
    }
end:
    return ret;
}

static int decoder_thread_run(struct codec_worker *codec_worker)
{
    AVPacket packet = { 0 };
    AVFormatContext *ifmt_ctx = codec_worker->ifmt_ctx;
//    FilteringContext *filter_ctx = codec_worker->filter_ctx;
    StreamContext *stream_ctx = codec_worker->stream_ctx;
    int ret;
    int i;

    do {
        while (!codec_worker->should_exit && !codec_worker->encode_exited) {
            ret = av_read_frame(ifmt_ctx, &packet);
            if (ret < 0) {
                if (ret == AVERROR_EOF) {
                    int err_ret = 0;

                    for (i = 0; i < ifmt_ctx->nb_streams; i++) {
                        do {
                            ret = __decode(codec_worker, NULL, i);
                        } while (ret >= 0);
                        avcodec_flush_buffers(stream_ctx[i].dec_ctx);
                        if (ret < 0 && ret != AVERROR_EOF) {
                            av_log(NULL, AV_LOG_ERROR, "flush decoder: stream=%d error\n", i);
                            err_ret = ret;
                        }
                    }

                    if (err_ret < 0) {
                        goto end;
                    }
                } else {
                    goto end;
                }
            }

            av_log(NULL, AV_LOG_DEBUG, "Demuxer gave frame of stream_index %u\n",
                    packet.stream_index);
            ret = __decode(codec_worker, &packet, packet.stream_index);
            av_packet_unref(&packet);

            if (ret < 0) {
                if (ret == AVERROR_EOF) {
                    ret = 0;
                    break;
                } else {
                    goto end;
                }
            }
        }

        for (i = 0; i < ifmt_ctx->nb_streams; i++) {
            av_log(NULL, AV_LOG_INFO, "flush decoder: stream=%d,total_duration=%ld,max_duration=%ld.\n",
                    i, stream_ctx[i].total_duration, stream_ctx[i].max_duration);
            stream_ctx[i].last_duration = stream_ctx[i].total_duration;
            stream_ctx[i].last_duration += stream_ctx[i].max_duration;
        }

        codec_worker->loop--;
        if (codec_worker->loop == 0) {
            break;
        }

        av_log(NULL, AV_LOG_INFO, "redo loop=%u.\n", codec_worker->loop);

        ret = reopen_input_file(codec_worker, codec_worker->decoder_name,
                codec_worker->input_file);
        if (ret < 0) {
            av_log(NULL, AV_LOG_INFO, "failed to reopen input file.\n");
            break;
        }
        ifmt_ctx = codec_worker->ifmt_ctx;
    } while (!codec_worker->should_exit && !codec_worker->encode_exited);

end:
    av_log(NULL, AV_LOG_INFO, "end of decoder thread.\n");
    return ret;
}

static int init_codec_worker(struct codec_worker *codec_worker)
{
    int ret;

    char *output_file = NULL;

    codec_worker->frame_link.head = NULL;
    codec_worker->frame_link.tail = NULL;


    output_file = av_calloc(256,sizeof(char));
    if(!output_file)
    {
        return AVERROR(ENOMEM);
    }

    codec_worker->output_file = output_file;

    ret = pthread_mutex_init(&codec_worker->frame_lock, NULL);
    if (ret) {
        goto fail_init_frame_lock;
    }

    ret = pthread_cond_init(&codec_worker->consume_cond, NULL);
    if (ret) {
        goto fail_init_consume_cond;
    }

    ret = pthread_cond_init(&codec_worker->product_cond, NULL);
    if (ret) {
        goto fail_init_product_cond;
    }

    ret = pthread_mutex_init(&codec_worker->launch_encoder_lock,NULL);
    if(ret)
    {
        goto fail_init_launch_encoder_lock;
    }

    ret = pthread_cond_init(&codec_worker->lanuch_encoder_cond,NULL);
    if(ret)
    {
        goto fail_init_launch_encoder_cond;
    }

    ret = pthread_mutex_init(&codec_worker->remux_lock, NULL);
    if (ret) {
        goto fail_init_remux_lock;
    }

    codec_worker->decoder_launch_encoder = 0;
    codec_worker->encoder_launched = 0;
    codec_worker->inited_encoder_ctx = 0;

    return 0;

fail_init_remux_lock:
    pthread_cond_destroy(&codec_worker->lanuch_encoder_cond);
fail_init_launch_encoder_cond:
    pthread_mutex_destroy(&codec_worker->launch_encoder_lock);
fail_init_launch_encoder_lock:
    pthread_cond_destroy(&codec_worker->product_cond);
fail_init_product_cond:
    pthread_cond_destroy(&codec_worker->consume_cond);
fail_init_consume_cond:
    pthread_mutex_destroy(&codec_worker->frame_lock);
fail_init_frame_lock:

    return ret;
}

static void cleanup_codec_worker(struct codec_worker *worker)
{
    if (worker) {
        pthread_mutex_destroy(&worker->launch_encoder_lock);
        pthread_cond_destroy(&worker->lanuch_encoder_cond);
        pthread_mutex_destroy(&worker->remux_lock);
        pthread_cond_destroy(&worker->product_cond);
        pthread_cond_destroy(&worker->consume_cond);
        pthread_mutex_destroy(&worker->frame_lock);
    }
}

static void *codec_worker_thread_run(void *thread_data)
{
    struct codec_worker *codec_worker = (struct codec_worker *)thread_data;
    pthread_attr_t attr;
    void *result;
    int i;
    int ret;

    pthread_mutex_lock(&codec_worker->common->lock);
    codec_worker->common->ready_num++;
    if (codec_worker->common->ready_num >= codec_worker->common->total_threads) {
        pthread_cond_signal(&codec_worker->common->ready_cond);
    }
    pthread_mutex_unlock(&codec_worker->common->lock);

    ret = open_input_file(codec_worker, codec_worker->decoder_name,
            codec_worker->input_file);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "fail to open input file.\n");
        goto end;
    }

    ret = set_output_file(codec_worker, codec_worker->encoder_name);
    if(ret < 0)
    {
        av_log(NULL, AV_LOG_ERROR, "Could not set output context\n");
        goto end;
    }

    ret = init_filters(codec_worker);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "fail to initialize filters.\n");
        goto end;
    }

    ret = pthread_attr_init(&attr);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "fail to initialize attr: %s.\n",
                strerror(ret));
        goto end;
    }

    ret = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "fail to set attr detachstate: %s.\n",
                strerror(ret));
        pthread_attr_destroy(&attr);
        goto end;
    }

    ret = pthread_create(&codec_worker->encode_tid, &attr, &encoder_thread,
            codec_worker);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "failed to create encoder thread: %s.\n",
                strerror(ret));
        pthread_attr_destroy(&attr);
        goto end;
    }

    ret = pthread_attr_destroy(&attr);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "failed to destroy pthread attr.\n");
        goto end;
    }

    ret = decoder_thread_run(codec_worker);
    pthread_mutex_lock(&codec_worker->frame_lock);
    if (!codec_worker->should_exit) {
        codec_worker->should_exit = 1;
    }
    pthread_cond_signal(&codec_worker->consume_cond);
    pthread_mutex_unlock(&codec_worker->frame_lock);
    if (pthread_join(codec_worker->encode_tid, &result)) {
        av_log(NULL, AV_LOG_ERROR, "failed to join encode thread.\n");
        if ((long)result) {
            ret = (int)((long)result);
        }
    }

    av_log(NULL, AV_LOG_INFO, "decoder normal shutdown.\n");

end:
    if (codec_worker->ifmt_ctx) {
        for (i = 0; i < codec_worker->ifmt_ctx->nb_streams; i++) {
            if (codec_worker->ofmt_ctx && codec_worker->ofmt_ctx->nb_streams > i &&
                    codec_worker->ofmt_ctx->streams[i] &&
                    codec_worker->stream_ctx[i].enc_ctx) {
                avcodec_free_context(&codec_worker->stream_ctx[i].enc_ctx);
            }
            if (codec_worker->filter_ctx && codec_worker->filter_ctx[i].filter_graph) {
                avfilter_graph_free(&codec_worker->filter_ctx[i].filter_graph);
            }
            if (codec_worker->decoded_frame) {
                av_frame_free(&codec_worker->decoded_frame);
                codec_worker->decoded_frame = NULL;
            }
            avcodec_free_context(&codec_worker->stream_ctx[i].dec_ctx);
        }
    }

    if (codec_worker->filter_ctx) {
        av_free(codec_worker->filter_ctx);
    }

    if (codec_worker->stream_ctx) {
        av_free(codec_worker->stream_ctx);
    }

    if (codec_worker->ifmt_ctx) {
        avformat_close_input(&codec_worker->ifmt_ctx);
    }

    if (codec_worker->ofmt_ctx) {
        if (!(codec_worker->ofmt_ctx->oformat->flags & AVFMT_NOFILE)) {
            avio_closep(&codec_worker->ofmt_ctx->pb);
        }
        avformat_free_context(codec_worker->ofmt_ctx);
    }

    if(codec_worker->inited_encoder_ctx)
    {
        av_freep(&codec_worker->inited_encoder_ctx);
    }

    if(codec_worker->output_file)
    {
        av_freep(&codec_worker->output_file);
    }

    av_log(NULL, AV_LOG_INFO, "decoder exit: ret=0x%x.\n", -ret);
    pthread_mutex_lock(&codec_worker->common->lock);
    codec_worker->common->exit_num++;
    pthread_mutex_unlock(&codec_worker->common->lock);

    return (void *)((long)ret);
}

static void sigint_handler(int signo)
{
    global_stop = 1;
    av_log(NULL, AV_LOG_INFO, "%s().\n", __func__);
}

static struct common *alloc_common(void)
{
    struct common *common;
    int ret;

    common = malloc(sizeof(struct common));
    if (common == NULL) {
        return NULL;
    }

    memset(common, 0, sizeof(struct common));

    ret = pthread_mutex_init(&common->lock, NULL);
    if (ret) {
        goto fail_init_lock;
    }

    ret = pthread_cond_init(&common->ready_cond, NULL);
    if (ret) {
        goto fail_init_ready_cond;
    }

    return common;

fail_init_ready_cond:
    pthread_mutex_destroy(&common->lock);
fail_init_lock:
    free(common);
    return NULL;
}

static void free_common(struct common *common)
{
    if (common) {
        pthread_mutex_destroy(&common->lock);
        pthread_cond_destroy(&common->ready_cond);
        free(common);
    }
}

static inline int curr_total_frames(void)
{
    int i;
    unsigned long total_frames = 0;

    for (i = 0; i < active_codec_workers; i++) {
        total_frames += codec_workers[i].total_encoder_frames;
    }

    return total_frames;
}

static void help_usage(void)
{
    printf("Usage: \n"
            "-x | --encoder-params    encoder parameters.\n"
            "-i | --input             input file path.\n"
            "-o | --output_dir        output directory path.\n"
            "-e | --encoder           encoder name.\n"
            "-d | --decoder           decoder name.\n"
            "-p | --decoder-params    decoder parameters.\n"
            "-t | --threads           number of codec threads.\n"
            "-b | --bitrate           bitrate.\n"
            "-n | --devid             device id.\n"
            "-s | --suffix            file extension of output. Use 'null' for no output.\n"
            "-l | --loop              numbers of loop to go.\n"
            "-f | --filter            decoded frames passing through filters.\n"
            "-r | --repeat-header     force repeating headers for IDR frames.\n"
            "-v | --loglevel          available debug level: warning, info, debug, trace.\n"
            "-k | --force-keyframe    encode a keyframe whenever input source has a keyframe.\n"
            "-c | --rescale-ts        rescale timestamp and timebase to default timebase of file type in --suffix.\n"
            "-a | --disable-audio     disable audio streams.\n"
            "-h | --help              print this help information.\n");
}

void setup_loglevel(char *loglevel)
{
    if (loglevel) {
        if (!strcmp(loglevel, "warning")) {
            av_log_set_level(AV_LOG_WARNING);
        } else if (!strcmp(loglevel, "info")) {
            av_log_set_level(AV_LOG_INFO);
        } else if (!strcmp(loglevel, "debug")) {
            av_log_set_level(AV_LOG_DEBUG);
        } else if (!strcmp(loglevel, "trace")) {
            av_log_set_level(AV_LOG_TRACE);
        } else {
            av_log_set_level(AV_LOG_INFO);
        }
    } else {
        av_log_set_level(AV_LOG_INFO);
    }
}

int main(int argc, char **argv)
{
    int ret;
    int bitrate = 4000000;
    char *input_file = NULL;
    char *output_dir = NULL;
    char *encoder_name = NULL;
    char *decoder_name = NULL;
    char *decoder_params = NULL;
    struct common *common = NULL;
    void *result;
    int threads = 1;
    int state = EXIT_SUCCESS;
    char *suffix = NULL;
    char *encoder_params = NULL;
    char *filter_params = NULL;
    int devid = 0;
    unsigned int loop = 1;
    pthread_attr_t attr;
    int i;
    int64_t timer_start;
    int opt;
    int opt_index;
    int filtering = 0;
    int repeat_header = 0;
    char *loglevel = NULL;
    int force_source_keyframe = 0;
    int rescale_ts = 0;
    int audio_disable = 0;
    const char *opt_string = "l:n:x:s:t:i:e:d:b:o:p:v:f:rkach";
    static struct option long_options[] = {
        {"input",          required_argument, NULL, 'i'},
        {"output_dir",     required_argument, NULL, 'o'},
        {"encoder",        required_argument, NULL, 'e'},
        {"decoder",        required_argument, NULL, 'd'},
        {"decoder-params", required_argument, NULL, 'p'},
        {"bitrate",        required_argument, NULL, 'b'},
        {"threads",        required_argument, NULL, 't'},
        {"suffix",         required_argument, NULL, 's'},
        {"encoder-params", required_argument, NULL, 'x'},
        {"devid",          required_argument, NULL, 'n'},
        {"loop",           required_argument, NULL, 'l'},
        {"filter",         required_argument, NULL, 'f'},
        {"repeat-header",  no_argument,       NULL, 'r'},
        {"loglevel",       required_argument, NULL, 'v'},
        {"force-keyframe", no_argument,       NULL, 'k'},
        {"rescale-ts",     no_argument,       NULL, 'c'},
        {"disable-audio",  no_argument,       NULL, 'a'},
        {"help",           no_argument,       NULL, 'h'},
        { NULL,            0,                 NULL,  0 },
    };

    while ((opt = getopt_long(argc, argv, opt_string, long_options, &opt_index)) != -1) {
        switch (opt) {
            case 'e':
                encoder_name = optarg;
                break;
            case 'd':
                decoder_name = optarg;
                break;
            case 'p':
                decoder_params = optarg;
                break;
            case 'i':
                input_file = optarg;
                break;
            case 'o':
                output_dir = optarg;
                break;
            case 't':
                threads = atoi(optarg);
                break;
            case 'b':
                bitrate = atoi(optarg);
                break;
            case 's':
                suffix = optarg;
                break;
            case 'n':
                devid = atoi(optarg);
                break;
            case 'x':
                encoder_params = optarg;
                break;
            case 'l':
                loop = strtoul(optarg, NULL, 10);
                break;
            case 'f':
                filtering = 1;
                filter_params = optarg;
                break;
            case 'r':
                repeat_header = 1;
                break;
            case 'v':
                loglevel = optarg;
                break;
            case 'k':
                force_source_keyframe = 1;
                break;
            case 'c':
                rescale_ts = 1;
                break;
            case 'a':
                audio_disable = 1;
                break;
            case 'h':
                help_usage();
                return EXIT_SUCCESS;
            default:
                help_usage();
                return EXIT_FAILURE;
        }
    }

    setup_loglevel(loglevel);

    if (input_file == NULL) {
        av_log(NULL, AV_LOG_ERROR, "input file name must be specified.\n");
        return EXIT_FAILURE;
    }

    if (output_dir == NULL) {
        output_dir = ".\0";
    }

    if (suffix == NULL) {
        suffix = "null\0";
    }

    if (threads <= 0 && threads > 256) {
        av_log(NULL, AV_LOG_ERROR, "invalid number of thread worker.\n");
        return EXIT_FAILURE;
    }

    if (encoder_name) {
        av_log(NULL, AV_LOG_INFO, "encoder_name: %s.\n", encoder_name);
    }
    if (decoder_name) {
        av_log(NULL, AV_LOG_INFO, "decoder_name: %s.\n", decoder_name);
    }

    codec_workers = malloc(sizeof(struct codec_worker) * threads);
    if (codec_workers == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate codec workers.\n");
        state = EXIT_FAILURE;
        goto end;
    }

    memset(codec_workers, 0, sizeof(struct codec_worker) * threads);

    common = alloc_common();
    if (common == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate common data.\n");
        state = EXIT_FAILURE;
        goto end;
    }

    common->total_threads = threads;

    // Register all formats, codecs, filters
#if LIBAVFORMAT_VERSION_MAJOR < 58
    av_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif
#if LIBAVCODEC_VERSION_MAJOR < 58
    avcodec_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif
#if LIBAVFILTER_VERSION_MAJOR < 7
    avfilter_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif

    ret = pthread_attr_init(&attr);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "fail to initialize attr: %s.\n", strerror(ret));
        state = EXIT_FAILURE;
        goto end;
    }

    ret = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "fail to set attr detachstate: %s.\n", strerror(ret));
        pthread_attr_destroy(&attr);
        state = EXIT_FAILURE;
        goto end;
    }

    for (i = 0; i < threads; i++) {
        struct codec_worker *codec_worker = &codec_workers[i];

        codec_worker->index = i;
        codec_worker->input_file = input_file;
        codec_worker->output_dir = output_dir;
        codec_worker->bitrate = bitrate;
        codec_worker->decoder_name = decoder_name;
        codec_worker->encoder_name = encoder_name;
        codec_worker->common = common;
        codec_worker->suffix = suffix;
        codec_worker->devid  = devid;
        codec_worker->encoder_params = encoder_params;
        codec_worker->decoder_params = decoder_params;
        codec_worker->filter_params = filter_params;
        codec_worker->loop = loop;
        codec_worker->bypass_filter = !filtering;
        codec_worker->audio_filter = 0;
        codec_worker->repeat_header = repeat_header;
        codec_worker->force_source_keyframe = force_source_keyframe;
        codec_worker->rescale_ts = rescale_ts;
        codec_worker->audio_disable = audio_disable;
        codec_worker->video_stream_num = -1;

        codec_worker->decoded_frame = av_frame_alloc();
        if (!codec_worker->decoded_frame) {
            av_log(NULL, AV_LOG_ERROR, "failed to allocate decoded frame for codec worker %d\n", i);
            goto end;
        }

        ret = init_codec_worker(codec_worker);
        if (ret) {
            av_log(NULL, AV_LOG_ERROR, "failed init codec worker %d.\n", i);
            pthread_attr_destroy(&attr);
            state = EXIT_FAILURE;
            goto end;
        }

        ret = pthread_create(&codec_worker->tid, &attr, &codec_worker_thread_run, codec_worker);
        if (ret) {
            av_log(NULL, AV_LOG_ERROR, "failed to create codec thread %d: %s.\n", i, strerror(ret));
            pthread_attr_destroy(&attr);
            cleanup_codec_worker(codec_worker);
            state = EXIT_FAILURE;
            goto end;
        }

        active_codec_workers++;
    }

    ret = pthread_attr_destroy(&attr);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "failed to destroy attr: %s.\n", strerror(ret));
        state = EXIT_FAILURE;
        goto end;
    }

    pthread_mutex_lock(&common->lock);
    while (common->ready_num < common->total_threads)
        pthread_cond_wait(&common->ready_cond, &common->lock);
    pthread_mutex_unlock(&common->lock);

    if (SIG_ERR == signal(SIGINT, sigint_handler)) {
        av_log(NULL, AV_LOG_ERROR, "failed to set signal.\n");
        state = EXIT_FAILURE;
        goto end;
    }

    timer_start = av_gettime_relative();
    while (global_stop == 0) {
        if (common->exit_num == active_codec_workers) {
            break;
        }
        sleep(1);

        print_report(0, timer_start, av_gettime_relative(), curr_total_frames());
    }
    print_report(1, timer_start, av_gettime_relative(), curr_total_frames());

    av_log(NULL, AV_LOG_INFO, "main thread is going to exit.\n");

end:
    for (i = 0; i < active_codec_workers; i++) {
        struct codec_worker *codec_worker = &codec_workers[i];

        av_log(NULL, AV_LOG_INFO, "pthread %d ready to exit.\n", codec_worker->index);

        pthread_mutex_lock(&codec_worker->frame_lock);
        codec_worker->should_exit = 1;
        pthread_cond_signal(&codec_worker->product_cond);
        pthread_mutex_unlock(&codec_worker->frame_lock);
        if (pthread_join(codec_worker->tid, &result) == 0) {
            if ((long)result != 0) {
                state = EXIT_FAILURE;
            }
        }

        if (!list_empty(&codec_worker->frame_link)) {
            av_log(NULL, AV_LOG_ERROR, "frame list no empty.\n");
            state = EXIT_FAILURE;
        }

        cleanup_codec_worker(codec_worker);
    }

    if (state < 0)
        av_log(NULL, AV_LOG_ERROR, "Error occurred: %s\n", av_err2str(ret));

    free_common(common);
    free(codec_workers);

    av_log(NULL, AV_LOG_INFO, "EXIT.. state=0x%x.\n", state);
    return state ? 1 : 0;
}
