/*
 * Copyright (c) 2001 Fabrice Bellard
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * @file
 * video encoding with libavcodec API example
 *
 * @example encode_video.c
 *
 * @added by leo.ma@netint.cn
 * use multiple threads to run YUV encoding.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#ifdef __linux__
#include <sys/ioctl.h>
#endif
#include <sys/time.h>
#include <getopt.h>
#include <pthread.h>
#include <signal.h>

#include <libavcodec/avcodec.h>
#include <libavfilter/buffersink.h>
#include <libavfilter/buffersrc.h>
#include <libavformat/avformat.h>
#include <libavutil/avassert.h>
#include <libavutil/opt.h>
#include <libavutil/imgutils.h>
#include <libavutil/time.h>

#define NI_CODEC_TIMING
//#define VERBOSE_DEBUG

enum {
    DEBUG_LEVEL_DEBUG,
    DEBUG_LEVEL_INFO,
    DEBUG_LEVEL_WARN,
    DEBUG_LEVEL_ERROR,
};

#define pr_err(fmt, ...) \
do {\
    fprintf(stderr, fmt, ##__VA_ARGS__); \
} while (0)

#define pr_warn(fmt, ...) \
do {\
    fprintf(stdout, fmt, ##__VA_ARGS__); \
} while (0)

#define pr_info(fmt, ...) \
do { \
    fprintf(stdout, fmt, ##__VA_ARGS__); \
} while (0)

#define pr_dbg(fmt, ...) \
do { \
    if (debug_level <= DEBUG_LEVEL_DEBUG) { \
        fprintf(stdout, fmt, ##__VA_ARGS__); \
    } \
} while (0)

#define AVFRAME_BLOCKING_COUNT 100
#define LOG_BUF_SIZE 8192

#define container_of(ptr, type, member) \
    ((type *)((char *)(ptr) - (size_t)&(((type *)0)->member)))

#define list_entry(ptr, type, member) \
    container_of(ptr, type, member)

#define list_first_entry(ptr, type, field) \
    list_entry((ptr)->next, type, field)

struct list_head {
    struct list_head *next, *prev;
};

typedef struct frame_node {
    struct list_head link;
    AVFrame *frame;
} FrameNode;

struct codec_worker;

typedef struct stream_context {
    pthread_t tid;
    int index;
    int should_exit;
    int filt_frames;
    struct list_head frame_list;
    struct codec_worker *worker;
    pthread_cond_t consume_cond;
    pthread_cond_t product_cond;
    pthread_mutex_t frame_lock;
    //const AVCodec *codec;
} StreamContext;

struct codec_worker {
    pthread_t tid;
    int index;
    StreamContext stream_ctxes[2];
    AVFilterGraph *filter_graph;
    AVFilterContext *bufsrc_ctx;
    AVFilterContext *bufsink_ctxes[2];
    int should_stop;
    const char *filename;
    const char *output_dir;
    const char *logname;
    const char *codec_name;
    const char *codec_params;
    AVRational sar;
    int bitrate;
    int fps;
    int mode;
    unsigned int width;
    unsigned int height;
    int loops;
    int dev_id;
    unsigned long frame_number;
    unsigned long total_encoded_frame_number;
    void *common;
};

struct common {
    pthread_mutex_t lock;
    pthread_cond_t ready_cond;
    int ready_num;
    int exit_num;
    int total_threads;
    int done;
    unsigned long ongoing;
    pthread_mutex_t sync_lock;
    pthread_cond_t sync_cond;
};

static int debug_level = DEBUG_LEVEL_INFO;
static int print_stat = 1;
static int log_time = 0;
static int mute = 0;
static int rate_limit = 0;
static int actual_threads = 0;
static int sync_mode = 0;
static int global_stop = 0;
static struct codec_worker *codec_workers;

static inline void INIT_LIST_HEAD(struct list_head *list)
{
    list->next = list->prev = list;
}

static inline int list_empty(const struct list_head *head)
{
    return head->next == head;
}

static inline void __list_add(struct list_head *new, struct list_head *prev, struct list_head *next)
{
    next->prev = new;
    new->next = next;
    new->prev = prev;
    prev->next = new;
}

static inline void list_add(struct list_head *_new, struct list_head *head)
{
    __list_add(_new, head, head->next);
}

static inline void list_add_tail(struct list_head *_new, struct list_head *head)
{
    __list_add(_new, head->prev, head);
}

static inline void __list_del(struct list_head *entry)
{
    entry->next->prev = entry->prev;
    entry->prev->next = entry->next;
}

static inline void list_del(struct list_head *entry)
{
    __list_del(entry);
    entry->next = entry->prev = NULL;
}

static void print_report(int is_last_report, int64_t timer_start, int64_t cur_time)
{
    static int64_t last_time = -1;
    float t;
    char buf[1024];
    float fps;
    int i;
    unsigned long frame_number = 0;

    if (!print_stat)
        return;

    if (!is_last_report) {
        if (last_time == -1) {
            last_time = cur_time;
            return;
        }
        if ((cur_time - last_time) < 500000)
            return;
        last_time = cur_time;
    }

    t = (cur_time - timer_start) / 1000000.0;

    for (i = 0; i < actual_threads; i++) {
        frame_number += codec_workers[i].total_encoded_frame_number;
    }

    fps = t > 1 ? frame_number / t : 0;
    if (print_stat && !mute) {
        const char end = is_last_report ? '\n' : '\r';

        buf[0] = '\0';
        snprintf(buf + strlen(buf), sizeof(buf) - strlen(buf), "frame=%5ld fps=%3.*f ",
                frame_number, (fps < 9.95), fps);
        fprintf(stderr, "%s   %c", buf, end);
        fflush(stderr);
    }
}

static int probe_send_frame(AVCodecContext *enc_ctx)
{
#ifdef XCODER_LINUX_CUSTOM_DRIVER
    int ret;

    enc_ctx->opaque = (void *)1;
    ret = avcodec_send_frame(enc_ctx, (AVFrame *)~0);
    enc_ctx->opaque = (void *)0;

    return ret;
#else
    return 0;
#endif
}

static int __encode(AVCodecContext *enc_ctx, AVFrame *frame,
                    AVPacket *pkt, FILE *outfile)
{
    int ret;

    do {
        ret = probe_send_frame(enc_ctx);
        if (ret == AVERROR(EAGAIN)) {
            continue;
        } else if (ret < 0) {
            pr_err("Error probing encode ready status.\n");
            return ret;
        }

        ret = avcodec_send_frame(enc_ctx, frame);
        if (ret == AVERROR(EAGAIN)) {
            continue;
        } else if (ret < 0) {
            pr_err("Error sending a frame for encoding\n");
            return ret;
        } else {
            break;
        }
    } while (1);

    /* send the frame to the encoder */
    if (frame) {
        pr_dbg("Send frame %3"PRId64"\n", frame->pts);
        av_frame_free(&frame);
    } else {
        pr_info("Flush encoder\n");
    }

    while (ret >= 0) {
        pr_dbg("receive packet.\n");
        ret = avcodec_receive_packet(enc_ctx, pkt);
        if (ret == AVERROR(EAGAIN) || ret == AVERROR_EOF) {
            return ret;
        } else if (ret < 0) {
            fprintf(stderr, "Error during encoding\n");
            return ret;
        }

        pr_dbg("Write packet %3"PRId64" (size=%5d)\n", pkt->pts, pkt->size);
        if (outfile) {
            fwrite(pkt->data, 1, pkt->size, outfile);
        }
        av_packet_unref(pkt);
    }

    return 0;
}

static void *encode_routine(void *arg)
{
    StreamContext *stream_ctx = (StreamContext *)arg;
    struct codec_worker *worker = stream_ctx->worker;
    const AVCodec *codec;
    AVCodecContext *enc_ctx = NULL;
    FILE *output = NULL;
    char name[256] = { 0 };
    char str_devid[8] = { 0 };
    FrameNode *fn;
    AVPacket pkt = { 0 };
    int fps = worker->fps;
    int ret;

    av_log(NULL, AV_LOG_DEBUG, "encode worker %d enter.\n", stream_ctx->index);

    /* find netint encoder */
    codec = avcodec_find_encoder_by_name(worker->codec_name);
    if (!codec) {
        av_log(NULL, AV_LOG_ERROR, "Codec %s not found\n", worker->codec_name);
        goto end;
    }

    av_log(NULL, AV_LOG_DEBUG, "Codec %s.\n", codec->name);

    if (codec->id == AV_CODEC_ID_H264) {
        snprintf(name, sizeof(name) - 1, "%s/output%d-%d.h264", worker->output_dir, worker->index, stream_ctx->index);
    } else if (codec->id == AV_CODEC_ID_H265) {
        snprintf(name, sizeof(name) - 1, "%s/output%d-%d.h265", worker->output_dir, worker->index, stream_ctx->index);
#ifdef AV1_SUPPORTED // AV1 not supported in 3.1.1
    } else if (codec->id == AV_CODEC_ID_AV1) {
        snprintf(name, sizeof(name) - 1, "%s/output%d-%d.data", worker->output_dir, worker->index, stream_ctx->index);
#endif
    } else {
        av_log(NULL, AV_LOG_ERROR, "Invalid codec id %s.\n", avcodec_get_name(codec->id));
        ret = AVERROR(EINVAL);
        goto end;
    }

    output = fopen(name, "wb");
    if (!output) {
        av_log(NULL, AV_LOG_ERROR, "failed to open output file: %s.\n", strerror(errno));
        ret = AVERROR(errno);
        goto end;
    }

    enc_ctx = avcodec_alloc_context3(codec);
    if (!enc_ctx) {
        av_log(NULL, AV_LOG_ERROR, "Could not allocate video codec context\n");
        ret = AVERROR(ENOMEM);
        goto end;
    }

    /* put sample parameters */
    enc_ctx->bit_rate = worker->bitrate;
    /* resolution must be a multiple of two */
    enc_ctx->width = worker->width;
    enc_ctx->height = worker->height;
    /* frames per second */
    enc_ctx->time_base = (AVRational){1, fps};
    enc_ctx->framerate = (AVRational){fps, 1};

    /* emit one intra frame every ten frames
     * check frame pict_type before passing frame
     * to encoder, if frame->pict_type is AV_PICTURE_TYPE_I
     * then gop_size is ignored and the output of encoder
     * will always be I frame irrespective to gop_size
     */
    enc_ctx->gop_size = 10;
    enc_ctx->max_b_frames = 1;
    enc_ctx->pix_fmt = AV_PIX_FMT_YUV420P;

    if (!strncmp(codec->name, "h264_ni", strlen("h264_ni")) ||
        !strncmp(codec->name, "h265_ni", strlen("h265_ni"))
#ifdef AV1_SUPPORTED // AV1 not supported in 3.1.1
        || !strncmp(codec->name, "av1_ni", strlen("av1_ni"))
#endif
    ) {
        /*
         * for example:
         * gopPresetIdx=6:lowDelay=1:intraPeriod=120:RcEnable=1:bitrate=4000000
         */
        if (worker->codec_params) {
            av_opt_set(enc_ctx->priv_data, "xcoder-params", worker->codec_params, 0);
        }
        snprintf(str_devid, sizeof(str_devid), "%d", worker->dev_id);
        av_opt_set(enc_ctx->priv_data, "enc", str_devid, 0);
    } else {
        av_log(NULL, AV_LOG_ERROR, "codec %s not supported.\n", codec->name);
        ret = AVERROR(EINVAL);
        goto end;
    }

    /* open it */
    ret = avcodec_open2(enc_ctx, codec, NULL);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Could not open codec: %s\n",
               av_err2str(ret));
        goto end;
    }

    while (1) {
        pthread_mutex_lock(&stream_ctx->frame_lock);

        while (!stream_ctx->should_exit && list_empty(&stream_ctx->frame_list)) {
            // Wait for available frames to encode
            pthread_cond_wait(&stream_ctx->consume_cond, &stream_ctx->frame_lock);
        }

        if (list_empty(&stream_ctx->frame_list) && stream_ctx->should_exit) {
            // End of all frame encoding
            pthread_mutex_unlock(&stream_ctx->frame_lock);
            break;
        }

        // Fetch the frame to encode
        fn = list_first_entry(&stream_ctx->frame_list, FrameNode, link);
        av_assert0(fn != NULL);
        list_del(&fn->link);

        stream_ctx->filt_frames--;
        if (stream_ctx->filt_frames < AVFRAME_BLOCKING_COUNT) {
            pthread_cond_signal(&stream_ctx->product_cond);
        }

        pthread_mutex_unlock(&stream_ctx->frame_lock);

        /* encode the image */
        ret = __encode(enc_ctx, fn->frame, &pkt, output);
        if (ret == AVERROR_EOF) {
            ret = 0;
            free(fn);
            break;
        } else if (ret < 0) {
            if (ret != AVERROR(EAGAIN)) {
                goto end;
            }
        }

        free(fn);
        worker->total_encoded_frame_number++;
    }

end:
    av_log(NULL, AV_LOG_INFO, "stream context %d exit: ret=0x%x.\n",
           stream_ctx->index, -ret);

    if (output) {
        fclose(output);
    }
    if (enc_ctx) {
        avcodec_free_context(&enc_ctx);
    }

    return (void *)((long)ret);
}

static int input_file_open(struct codec_worker *worker, const char *filename,
                           enum AVPixelFormat pix_fmt)
{
    int i, ret, video_stream_index;
    char video_size[16] = { 0 };
#if LIBAVFORMAT_VERSION_MAJOR >= 59
    const
#endif
    AVInputFormat *file_iformat;
    AVDictionary *iformat_opts = NULL;
    AVFormatContext *ifmt_ctx = NULL;
    AVStream *st;

    file_iformat = av_find_input_format("rawvideo");
    if (!file_iformat) {
        av_log(NULL, AV_LOG_ERROR, "failed to find input format\n");
        return -1;
    }

    snprintf(video_size, sizeof(video_size), "%dx%d", worker->width, worker->height);
    av_dict_set(&iformat_opts, "video_size", video_size, 0);
    av_dict_set(&iformat_opts, "pixel_format", av_get_pix_fmt_name(pix_fmt), 0);
    av_dict_set(&iformat_opts, "scan_all_fmt", NULL, AV_DICT_MATCH_CASE);

    ret = avformat_open_input(&ifmt_ctx, filename, file_iformat, &iformat_opts);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to open input file\n");
        return ret;
    }

    ret = avformat_find_stream_info(ifmt_ctx, NULL);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot find stream information\n");
        return ret;
    }

    for (i = 0; i < ifmt_ctx->nb_streams; i++) {
      if (ifmt_ctx->streams[i]->codecpar->codec_type == AVMEDIA_TYPE_VIDEO) {
        video_stream_index = i;
        break;
      }
    }
    st = ifmt_ctx->streams[video_stream_index];

    worker->sar = st->codecpar->sample_aspect_ratio;

    avformat_close_input(&ifmt_ctx);

    return 0;
}

static int filter_graph_open(struct codec_worker *worker)
{
    int i, ret;
    int width = worker->width;
    int height = worker->height;
    AVRational sar = worker->sar;
    AVFilterInOut *inputs, *outputs, *cur;
    char args[256] = { 0 };
    char filt_desc[256] = { 0 };

    worker->filter_graph = avfilter_graph_alloc();
    if (worker->filter_graph == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate filter graph\n");
        goto out;
    }

    snprintf(filt_desc, sizeof(filt_desc), "ni_quadra_yuv444to420=mode=%d", worker->mode);
    ret = avfilter_graph_parse2(worker->filter_graph, filt_desc, &inputs, &outputs);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to parse graph\n");
        goto out;
    }

    snprintf(args, sizeof(args), "video_size=%dx%d:pix_fmt=%d:time_base=%d/%d:pixel_aspect=%d/%d",
             width, height, AV_PIX_FMT_YUV444P, 1, 30, sar.num, sar.den);
    av_log(NULL, AV_LOG_INFO, "input filter args: %s\n", args);
    ret = avfilter_graph_create_filter(&worker->bufsrc_ctx,
                                       avfilter_get_by_name("buffer"), "in_0",
                                       args, NULL, worker->filter_graph);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to create input filter bufsrc\n");
        goto out;
    }

    ret = avfilter_link(worker->bufsrc_ctx, 0, inputs->filter_ctx, inputs->pad_idx);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to link input filter bufsrc\n");
        goto out;
    }

    for (cur = outputs, i = 0; cur; cur = cur->next, i++) {
        char name[256] = { 0 };

        snprintf(name, sizeof(name), "out_%d", i);
        ret = avfilter_graph_create_filter(&worker->bufsink_ctxes[i],
                                           avfilter_get_by_name("buffersink"),
                                           name, NULL, NULL, worker->filter_graph);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "failed to create output filter: %d", i);
            goto out;
        }

        ret = avfilter_link(cur->filter_ctx, cur->pad_idx, worker->bufsink_ctxes[i], 0);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "failed to link output filter: %d\n", i);
            goto out;
        }
    }

    ret = avfilter_graph_config(worker->filter_graph, NULL);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to config graph filter\n");
        goto out;
    }

out:
    avfilter_inout_free(&inputs);
    avfilter_inout_free(&outputs);
    return ret;
}

static void frame_dispatch(StreamContext *stream_ctx, AVFrame *frame)
{
    FrameNode *fn;

    fn = (FrameNode *) malloc(sizeof(*fn));
    av_assert0(fn);
    fn->frame = frame;

    pthread_mutex_lock(&stream_ctx->frame_lock);
    list_add_tail(&fn->link, &stream_ctx->frame_list);
    pthread_cond_signal(&stream_ctx->consume_cond);
    stream_ctx->filt_frames++;
    if (stream_ctx->filt_frames >= AVFRAME_BLOCKING_COUNT) {
        do {
            pthread_cond_wait(&stream_ctx->product_cond, &stream_ctx->frame_lock);
            if ((stream_ctx->filt_frames < AVFRAME_BLOCKING_COUNT) ||
                    stream_ctx->should_exit) {
                break;
            }
        } while (1);
    }
    pthread_mutex_unlock(&stream_ctx->frame_lock);
}

static int filter_frame(struct codec_worker *worker, AVFrame *frame)
{
    int i, ret;

    if (!frame) {
        for (i = 0; i < 2; i++) {
            frame_dispatch(&worker->stream_ctxes[i], NULL);
        }
        return 0;
    }

    av_log(NULL, AV_LOG_DEBUG, "frame pixel=%dx%d, crop=%ld/%ld/%ld/%ld linsize %d/%d/%d\n",
           frame->width, frame->height, frame->crop_top,
           frame->crop_bottom, frame->crop_left, frame->crop_right,
           frame->linesize[0], frame->linesize[1], frame->linesize[2]);

    av_log(NULL, AV_LOG_DEBUG, "ref count %d/%d/%d\n",
           frame->buf[0] ? av_buffer_get_ref_count(frame->buf[0]) : -1,
           frame->buf[1] ? av_buffer_get_ref_count(frame->buf[1]) : -1,
           frame->buf[2] ? av_buffer_get_ref_count(frame->buf[2]) : -1);

    frame->pict_type = AV_PICTURE_TYPE_NONE;
    av_log(NULL, AV_LOG_DEBUG, "decode frame %3"PRId64", type=%d\n", frame->pts, frame->pict_type);

    ret = av_buffersrc_add_frame_flags(worker->bufsrc_ctx, frame, AV_BUFFERSRC_FLAG_KEEP_REF);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to add frame to buffersrc\n");
        return ret;
    }

    for (i = 0; i < 2; i++) {
        while (1) {
            AVFrame *filt_frame = av_frame_alloc();
            if (!filt_frame) {
                return AVERROR(ENOMEM);
            }

            ret = av_buffersink_get_frame(worker->bufsink_ctxes[i], filt_frame);
            if (ret == AVERROR(EAGAIN) || ret == AVERROR_EOF) {
                ret = 0;
                av_frame_free(&filt_frame);
                break;
            } else if (ret < 0) {
                av_log(NULL, AV_LOG_ERROR, "failed to get frame from buffersink\n");
                av_frame_free(&filt_frame);
                break;
            }

            av_log(NULL, AV_LOG_DEBUG, "filt frame pixel=%dx%d, crop=%ld/%ld/%ld/%ld, linesize=%d/%d/%d\n",
                   filt_frame->width, filt_frame->height, filt_frame->crop_top,
                   filt_frame->crop_bottom, filt_frame->crop_left, filt_frame->crop_right,
                   filt_frame->linesize[0], filt_frame->linesize[1], filt_frame->linesize[2]);

            av_log(NULL, AV_LOG_DEBUG, "filt ref count %d/%d/%d\n",
                   filt_frame->buf[0] ? av_buffer_get_ref_count(filt_frame->buf[0]) : -1,
                   filt_frame->buf[1] ? av_buffer_get_ref_count(filt_frame->buf[1]) : -1,
                   filt_frame->buf[2] ? av_buffer_get_ref_count(filt_frame->buf[2]) : -1);

            frame_dispatch(&worker->stream_ctxes[i], filt_frame);
        }
    }

    return ret;
}

static void *filter_routine(void *arg)
{
    struct codec_worker *worker = (struct codec_worker *)arg;
    AVFrame *frame = NULL;
    unsigned int frame_size;
    unsigned int read_size;
    unsigned long input_filesize;
    FILE *input = NULL;
    int loop_count = worker->loops;
    int ret;
    int l, i, x, y;
    struct common *common = (struct common *)worker->common;

    av_log(NULL, AV_LOG_DEBUG, "filter worker %d enter.\n", worker->index);

    ret = input_file_open(worker, worker->filename, AV_PIX_FMT_YUV444P);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Could not obtain the stream format\n");
        goto end;
    }

    ret = filter_graph_open(worker);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Could not open filter graph\n");
        goto end;
    }

    input = fopen(worker->filename, "rb");
    if (input == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to open file: %s.\n", strerror(errno));
        ret = AVERROR(errno);
        goto end;
    }

    fseek(input, 0, SEEK_END);
    input_filesize = ftell(input);
    frame_size = input_filesize / (worker->width * worker->height * 3);
    av_log(NULL, AV_LOG_DEBUG, "inputfile size=%lu, frame size=%u.\n",
           input_filesize, frame_size);

    frame = av_frame_alloc();
    if (!frame) {
        av_log(NULL, AV_LOG_ERROR, "Could not allocate video frame\n");
        ret = AVERROR(ENOMEM);
        goto end;
    }

    frame->format = AV_PIX_FMT_YUV444P;
    frame->width  = worker->width;
    frame->height = worker->height;

    ret = av_frame_get_buffer(frame, 32);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Could not allocate the video frame data\n");
        goto end;
    }

    av_log(NULL, AV_LOG_DEBUG, "linesize: 0=%d, 1=%d, 2=%d.\n",
           frame->linesize[0], frame->linesize[1], frame->linesize[2]);

    for (i = 0; i < 2; i++) {
        StreamContext *stream_ctx = &worker->stream_ctxes[i];
        ret = pthread_create(&stream_ctx->tid, NULL, encode_routine, stream_ctx);
        if (ret) {
            av_log(NULL, AV_LOG_ERROR, "failed to create pthread for %d: %s.\n",
                   i, strerror(errno));
            goto end;
        }
    }

    // Tell the main thread to move on.
    pthread_mutex_lock(&common->lock);
    common->ready_num++;
    if (common->ready_num >= common->total_threads) {
        pthread_cond_signal(&common->ready_cond);
    }
    pthread_mutex_unlock(&common->lock);

    // Read YUV frames
    for (l = 0; l < loop_count; l++) {
        fseek(input, 0, SEEK_SET);
        for (i = 0; i < frame_size; i++) {
//            if (rate_limit) {
//                while (1) {
//                    cur_time = av_gettime_relative();
//                    if ((dts + 20000) > (cur_time - timer_start)) {
//                        av_usleep(10000);
//                        continue;
//                    } else
//                        break;
//                }
//            }

            if (sync_mode) {
                pthread_mutex_lock(&common->sync_lock);
                common->done++;
                common->ongoing |= (1 << worker->index);
                if (common->done == common->total_threads) {
                    av_log(NULL, AV_LOG_TRACE, "%d: reset %lu.\n", worker->index, worker->frame_number);
                    common->done = 0;
                    common->ongoing = 0;
                    pthread_cond_broadcast(&common->sync_cond);
                } else {
                    while ((common->ongoing & (1 << worker->index)) && !worker->should_stop) {
                        av_log(NULL, AV_LOG_TRACE, "%d: wait %lu.\n", worker->index, worker->frame_number);
                        pthread_cond_wait(&common->sync_cond, &common->sync_lock);
                    }
                }
                av_log(NULL, AV_LOG_TRACE, "%d: wake %lu.\n", worker->index, worker->frame_number);
                pthread_mutex_unlock(&common->sync_lock);
            }

            if (worker->should_stop) {
                goto end;
            }

            /* make sure the frame data is writable */
            ret = av_frame_make_writable(frame);
            if (ret < 0) {
                av_log(NULL, AV_LOG_ERROR, "frame make writable error.\n");
                goto end;
            }

            for (x = 0; x < 3; x++) {
                for (y = 0; y < worker->height; y++) {
                    read_size = fread(frame->data[x] + y * frame->linesize[x], worker->width, 1, input);
                    if (read_size != 1) {
                        av_log(NULL, AV_LOG_ERROR, "y: read_size=%u.\n", read_size);
                        ret = AVERROR(errno);
                        goto end;
                    }
                }
            }

            frame->pts = l * frame_size + i;
            worker->frame_number = frame->pts + 1;  // begin frame count at 1

            // YUV444p to YUV420p
            ret = filter_frame(worker, frame);
            if (ret < 0) {
                goto end;
            }
        }
    }

    // Send NULL frame to flush
    ret = filter_frame(worker, NULL);
    if (ret < 0) {
        goto end;
    }

end:
    // Tell encoding threads to exit.
    for (i = 0; i < 2; i++) {
        StreamContext *stream_ctx = &worker->stream_ctxes[i];
        pthread_mutex_lock(&stream_ctx->frame_lock);
        stream_ctx->should_exit = 1;
        pthread_cond_signal(&stream_ctx->consume_cond);
        pthread_mutex_unlock(&stream_ctx->frame_lock);
    }

    // Wait for encoding threads termination.
    for (i = 0; i < 2; i++) {
        void *result;
        StreamContext *stream_ctx = &worker->stream_ctxes[i];

        ret = pthread_join(stream_ctx->tid, &result);
        if (ret) {
            av_log(NULL, AV_LOG_ERROR, "failed to join pthread for %d: %s.\n",
                   stream_ctx->index, strerror(errno));
            return (void *)((long)ret);
        }

        if ((long)result != 0) {
            av_log(NULL, AV_LOG_ERROR, "pthread %d result=0x%lx.\n",
                   stream_ctx->index, (long)result);
            return (void *)((long)ret);
        }
    }

    // Release resources.
    if (input) {
        fclose(input);
    }
    if (frame) {
        av_frame_free(&frame);
    }
    if (worker->filter_graph) {
        avfilter_graph_free(&worker->filter_graph);
    }
    pthread_mutex_lock(&common->lock);
    common->exit_num++;
    pthread_mutex_unlock(&common->lock);

    return (void *)((long)ret);
}

static void sigint_handler(int signo)
{
    global_stop = 1;
    av_log(NULL, AV_LOG_INFO, "%s().\n", __func__);
}

static void help_usage(void)
{
    printf("Usage: \n"
            "-p | --pixel               [width]x[height] of yuv.\n"
            "-x | --xcoder_params       xcoder parameters.\n"
            "-i | --input_file          input yuv file.\n"
            "-o | --output_dir          output directory path.\n"
            "-c | --codec               codec name.\n"
            "-m | --mode                mode 0 have better PSNR and 1 can decode as 420.\n"
            "-t | --threads             number of threads.\n"
            "-b | --bitrate             bitrate.\n"
            "-d | --devid               dev id.\n"
            "-s | --loop                run loop count.\n"
            "-y | --sync                sync mode test.\n"
            "-h | --help                help info.\n");
}

static struct common *alloc_common(void)
{
    struct common *common;
    int ret;

    common = malloc(sizeof(struct common));
    if (common == NULL) {
        return NULL;
    }

    memset(common, 0, sizeof(struct common));

    ret = pthread_mutex_init(&common->lock, NULL);
    if (ret) {
        goto fail_init_lock;
    }

    ret = pthread_cond_init(&common->ready_cond, NULL);
    if (ret) {
        goto fail_init_ready_cond;
    }

    ret = pthread_mutex_init(&common->sync_lock, NULL);
    if (ret) {
        goto fail_init_sync_lock;
    }

    ret = pthread_cond_init(&common->sync_cond, NULL);
    if (ret) {
        goto fail_init_sync_cond;
    }

    return common;

fail_init_sync_cond:
    pthread_mutex_destroy(&common->sync_lock);
fail_init_sync_lock:
    pthread_cond_destroy(&common->ready_cond);
fail_init_ready_cond:
    pthread_mutex_destroy(&common->lock);
fail_init_lock:
    free(common);
    return NULL;
}

static void free_common(struct common *common)
{
    if (common) {
        pthread_cond_destroy(&common->sync_cond);
        pthread_mutex_destroy(&common->sync_lock);
        pthread_mutex_destroy(&common->lock);
        pthread_cond_destroy(&common->ready_cond);
        free(common);
    }
}

static int stream_context_init(struct codec_worker *worker)
{
    int i, ret = 0;

    for (i = 0; i < 2; i++) {
        StreamContext *stream_ctx = &worker->stream_ctxes[i];
        stream_ctx->index = i;
        stream_ctx->worker = worker;
        INIT_LIST_HEAD(&stream_ctx->frame_list);

        ret = pthread_mutex_init(&stream_ctx->frame_lock, NULL);
        if (ret) {
            break;
        }

        ret = pthread_cond_init(&stream_ctx->consume_cond, NULL);
        if (ret) {
            break;
        }

        ret = pthread_cond_init(&stream_ctx->product_cond, NULL);
        if (ret) {
            break;
        }
    }

    return ret;
}

static void stream_context_cleanup(struct codec_worker *worker)
{
    int i;

    for (i = 0; i < 2; i++) {
        StreamContext *stream_ctx = &worker->stream_ctxes[i];
        pthread_cond_destroy(&stream_ctx->product_cond);
        pthread_cond_destroy(&stream_ctx->consume_cond);
        pthread_mutex_destroy(&stream_ctx->frame_lock);
    }
}

int main(int argc, char **argv)
{
    const char *codec_name = NULL;
    const char *output_dir = NULL;
    char *xcoder_params = NULL;
    unsigned int video_width = ~0;
    unsigned int video_height = ~0;
    int bitrate = 4000000;
    int fps = 30;
    unsigned int loop_count = 1;
    char *logfile = NULL;
    const char *input_file;
    int devid = 0;
    int i;
    int ret;
    int mode = 0;
    char *n;
    int state = EXIT_SUCCESS;
    int threads = 1;
    struct common *common = NULL;
    int64_t timer_start;
    int opt;
    int opt_index;
    const char *opt_string = "yt:s:r:b:d:p:x:i:o:c:g:m:uefh";
    static struct option long_options[] = {
        {"pixel",         required_argument, NULL, 'p'},
        {"xcoder_params", required_argument, NULL, 'x'},
        {"input_file",    required_argument, NULL, 'i'},
        {"output_dir",    required_argument, NULL, 'o'},
        {"codec",         required_argument, NULL, 'c'},
        {"mode",          required_argument, NULL, 'm'},
        {"fps",           required_argument, NULL, 'r'},
        {"bitrate",       required_argument, NULL, 'b'},
        {"devid",         required_argument, NULL, 'd'},
        {"loop",          required_argument, NULL, 's'},
        {"log",           required_argument, NULL, 'g'},
        {"threads",       required_argument, NULL, 't'},
        {"mute",          no_argument,       NULL, 'm'},
        {"limit",         no_argument,       NULL, 'e'},
        {"pr_fps",        no_argument,       NULL, 'f'},
        {"sync",          no_argument,       NULL, 'y'},
        {"help",          no_argument,       NULL, 'h'},
        {NULL,            0,                 NULL, 0},
    };

    while ((opt = getopt_long(argc, argv, opt_string, long_options, &opt_index)) != -1) {
        switch (opt) {
            case 'p':
                video_width = strtoul(optarg, &n, 10);
                if (*n != 'x') {
                    help_usage();
                    return -1;
                }
                video_height = strtoul(n + 1, NULL, 10);
                break;
            case 'x':
                xcoder_params = optarg;
                break;
            case 'i':
                input_file = optarg;
                break;
            case 'o':
                output_dir = optarg;
                break;
            case 'c':
                codec_name = optarg;
                break;
            case 'm':
                mode = atoi(optarg);
                break;
            case 'd':
                devid = atoi(optarg);
                break;
            case 'b':
                bitrate = atoi(optarg);
                break;
            case 'r':
                fps = atoi(optarg);
                break;
            case 's':
                loop_count = strtoul(optarg, NULL, 10);
                break;
            case 'g':
                log_time = 1;
                logfile = optarg;
                break;
            case 'u':
                mute = 1;
                break;
            case 'e':
                rate_limit = 1;
                break;
            case 'f':
                print_stat = 1;
                break;
            case 't':
                threads = atoi(optarg);
                break;
            case 'y':
                sync_mode = 1;
                break;
            case 'h':
                help_usage();
                return 0;
            default:
                help_usage();
                return -1;
        }
    }

    if (input_file == NULL || codec_name == NULL ||
        video_width == ~0 || video_height == ~0) {
        help_usage();
        exit(1);
    }

    if (output_dir == NULL) {
        output_dir = ".";
    }

    av_log_set_level(AV_LOG_INFO);

    if (sync_mode) {
        if ((threads <= 0) || (threads > sizeof(unsigned long))) {
            av_log(NULL, AV_LOG_ERROR, "invalid threads number %d.\n", threads);
            exit(1);
        }
    } else {
        if ((threads <= 0) || (threads > 256)) {
            av_log(NULL, AV_LOG_ERROR, "invalid threads number %d.\n", threads);
            exit(1);
        }
    }

    av_log(NULL, AV_LOG_INFO, "yuv444p filter mode: %d.\n", mode);
    av_log(NULL, AV_LOG_INFO, "sync mode: %d.\n", sync_mode);
    av_log(NULL, AV_LOG_INFO, "threads: %d.\n", threads);
    av_log(NULL, AV_LOG_INFO, "input file: %s.\n", input_file);
    av_log(NULL, AV_LOG_INFO, "codec name: %s.\n", codec_name);
    av_log(NULL, AV_LOG_INFO, "video width: %u.\n", video_width);
    av_log(NULL, AV_LOG_INFO, "video height: %u.\n", video_height);
    av_log(NULL, AV_LOG_INFO, "device id: %d.\n", devid);
    av_log(NULL, AV_LOG_INFO, "loop count: %u.\n", loop_count);
    if (xcoder_params) {
        av_log(NULL, AV_LOG_INFO, "xcoder params: %s.\n", xcoder_params);
    }

    if (log_time) {
        av_log(NULL, AV_LOG_INFO, "log file: %s.\n", logfile);
    }

    if (access(input_file, R_OK) != 0) {
        av_log(NULL, AV_LOG_ERROR, "cannot access %s: %s.\n", input_file,
                strerror(errno));
        state = EXIT_FAILURE;
        goto end;
    }

#if LIBAVFORMAT_VERSION_MAJOR < 58
    av_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif
#if LIBAVCODEC_VERSION_MAJOR < 58
    // Register all codecs
    avcodec_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif
#if LIBAVFILTER_VERSION_MAJOR < 7
    avfilter_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif

    common = alloc_common();
    if (common == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate common data.\n");
        state = EXIT_FAILURE;
        goto end;
    }

    common->total_threads = threads;

    codec_workers = malloc(sizeof(struct codec_worker) * threads);
    if (codec_workers == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate encode workers.\n");
        state = EXIT_FAILURE;
        goto end;
    }
    memset(codec_workers, 0, sizeof(struct codec_worker) * threads);

    for (i = 0; i < threads; i++) {
        struct codec_worker *worker = &codec_workers[i];
        worker->index = i;
        worker->bitrate = bitrate;
        worker->width = video_width;
        worker->height = video_height;
        worker->fps = fps;
        worker->codec_name = codec_name;
        worker->mode = mode;
        worker->filename = input_file;
        worker->output_dir = output_dir;
        worker->logname = logfile;
        worker->dev_id = devid;
        worker->common = common;
        worker->loops = loop_count;
        worker->codec_params = xcoder_params;

        ret = stream_context_init(worker);
        if (ret) {
            av_log(NULL, AV_LOG_ERROR, "failed to init stream context %d: %s.\n",
                    i, strerror(errno));
            state = EXIT_FAILURE;
            goto end;
        }

        av_log(NULL, AV_LOG_INFO, "create thread %d, bitrate=%d, "
                "width=%u, height=%u, fps=%d, input=%s, output_dir=%s, loops=%d.\n",
                worker->index, worker->bitrate, worker->width, worker->height,
                worker->fps, worker->filename, worker->output_dir, worker->loops);

        ret = pthread_create(&worker->tid, NULL, filter_routine, worker);
        if (ret) {
            av_log(NULL, AV_LOG_ERROR, "failed to create pthread for %d: %s.\n",
                    i, strerror(errno));
            state = EXIT_FAILURE;
            goto end;
        }

        actual_threads++;
    }

    pthread_mutex_lock(&common->lock);
    while (common->ready_num < common->total_threads)
        pthread_cond_wait(&common->ready_cond, &common->lock);
    pthread_mutex_unlock(&common->lock);

    if (SIG_ERR == signal(SIGINT, sigint_handler)) {
        av_log(NULL, AV_LOG_ERROR, "failed to set signal.\n");
        ret = EXIT_FAILURE;
        goto end;
    }

    timer_start = av_gettime_relative();
    while (!global_stop) {
        if (common->exit_num == actual_threads) {
            break;
        }
        sleep(1);
        print_report(0, timer_start, av_gettime_relative());
    }
    print_report(1, timer_start, av_gettime_relative());

    av_log(NULL, AV_LOG_INFO, "main thread is going to exit.\n");

end:
    if (codec_workers) {
        for (i = 0; i < actual_threads; i++) {
            struct codec_worker *worker = &codec_workers[i];
            worker->should_stop = 1;
        }

        if (sync_mode) {
            pthread_mutex_lock(&common->sync_lock);
            pthread_cond_broadcast(&common->sync_cond);
            pthread_mutex_unlock(&common->sync_lock);
        }

        for (i = 0; i < actual_threads; i++) {
            void *result;
            struct codec_worker *worker = &codec_workers[i];

            ret = pthread_join(worker->tid, &result);
            if (ret) {
                av_log(NULL, AV_LOG_ERROR, "failed to join pthread for %d: %s.\n",
                        worker->index, strerror(errno));
                state = EXIT_FAILURE;
            }

            if ((long)result != 0) {
                av_log(NULL, AV_LOG_ERROR, "pthread %d result=0x%lx.\n",
                        worker->index, (long)result);
                state = EXIT_FAILURE;
            }

            stream_context_cleanup(worker);
        }
        free(codec_workers);
    }

    free_common(common);

    av_log(NULL, AV_LOG_INFO, "EXIT.. state=0x%x.\n", state);

    return state;
}
