/*
 * Copyright (c) 2010 Nicolas George
 * Copyright (c) 2011 Stefano Sabatini
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * @file
 * API example for ppu, ni_quadra_split and multi encoder
 * @example ni_ppu_split.c
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>

#include <libavcodec/avcodec.h>
#include <libavformat/avformat.h>
#include <libavfilter/buffersink.h>
#include <libavfilter/buffersrc.h>
#include <libswscale/swscale.h>
#include <libavutil/time.h>
#include <libavutil/opt.h>
#include <libavutil/hwcontext.h>
#include <libavutil/fifo.h>
#include <pthread.h>

#define MAX_FILTER_SINK 12

// end = 1 means demux finished
// end = 2 means decode finished
// end = 3 means decoder and encoder has finished, main thread ready to finish
int end = 0;
int64_t timer_start,timer_now;

typedef struct _decoder_worker {
    int devid;
    const char *input_file;
    const char *decoder_name;
    const char *decoder_params;
    int video_stream_index;
    AVFormatContext *ifmt_ctx;
    AVCodecContext *dec_ctx;
} decoder_worker;

typedef struct _filter_worker {
    int init_filter;
    const char *filter_desc;
    AVFilterContext *buffersrc_ctx;
    AVFilterContext *buffersink_ctx[MAX_FILTER_SINK];
    AVFilterGraph *filter_graph;
} filter_worker;

typedef struct _encoder_worker {
    pthread_t enc_tid;
    pthread_mutex_t frame_lock;
    pthread_cond_t frame_cond;
    int enc_thread_state;

    int should_exit;
    int enc_flush;
#if LIBAVCODEC_VERSION_MAJOR >= 61 //7.0
    AVFifo *enc_frame_fifo;
#else
    AVFifoBuffer *enc_frame_fifo;
#endif
    AVFrame *buffered_frame;
    AVFrame *filtered_frame;

    int devid;
    const char *output_file;
    const char *encoder_name;
    const char *encoder_params;
    int init_enc;
    int encode_num;
    AVCodecContext *enc_ctx;
    FILE *outfile;
} encoder_worker;

typedef struct _prog_args {
    int devid;
    int ppu_num;
    int split_num;
    decoder_worker *dec_work;
    filter_worker *filt_work;
    encoder_worker *enc_work[MAX_FILTER_SINK];
} prog_args;

prog_args *p_args = NULL;

#if (LIBAVCODEC_VERSION_MAJOR >= 61)
static inline int is_fifo_empty(AVFifo *fifo)
{
    return av_fifo_can_read(fifo) ? 0 : 1;
}
#else
static inline int is_fifo_empty(AVFifoBuffer *fifo)
{
    return av_fifo_size(fifo) < sizeof(AVFrame*);
}
#endif

#if (LIBAVCODEC_VERSION_MAJOR >= 61)
static inline int is_fifo_full(AVFifo *fifo)
{
    return av_fifo_can_write(fifo) ? 0 : 1;
}
#else
static inline int is_fifo_full(AVFifoBuffer *fifo)
{
    return av_fifo_space(fifo) < sizeof(AVFrame*);
}
#endif

#if (LIBAVCODEC_VERSION_MAJOR >= 61)
static inline int get_fifo_size(AVFifo *fifo) {
    return av_fifo_can_read(fifo);
}
#else
static inline int get_fifo_size(AVFifoBuffer *fifo) {
    return av_fifo_size(fifo) / sizeof(AVFrame*);
}
#endif

#if (LIBAVCODEC_VERSION_MAJOR >= 61)
static void free_fifo(AVFifo *fifo) {
    av_fifo_freep2(&fifo);
}
#else
static void free_fifo(AVFifoBuffer *fifo) {
    return av_fifo_free(fifo);
}
#endif

static int init_prog_args(prog_args *p_args) {
    int i;

    p_args->devid = 0;
    p_args->ppu_num = 1;
    p_args->split_num = 0;
    // init dec_worker
    decoder_worker *dec_work = malloc(sizeof(decoder_worker));
    if (dec_work == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate decoder_worker.\n");
        return -1;
    }
    dec_work->devid = 0;
    dec_work->input_file = NULL;
    dec_work->decoder_name = NULL;
    dec_work->decoder_params = NULL;
    dec_work->video_stream_index = -1;
    dec_work->ifmt_ctx = NULL;
    dec_work->dec_ctx = NULL;
    p_args->dec_work = dec_work;

    // init filt_worker
    filter_worker *filt_work = malloc(sizeof(filter_worker));
    if (filt_work == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate filter_worker.\n");
        return -1;
    }
    filt_work->init_filter = 0;
    filt_work->filter_desc = NULL;
    filt_work->filter_graph = NULL;
    filt_work->buffersrc_ctx = NULL;

    for (i = 0; i < MAX_FILTER_SINK; i++) {
        filt_work->buffersink_ctx[i] = NULL;
    }
    p_args->filt_work = filt_work;

    // init enc_worker
    for (i = 0; i < MAX_FILTER_SINK; i++) {
        encoder_worker *enc_work = malloc(sizeof(encoder_worker));
        if (enc_work == NULL) {
            av_log(NULL, AV_LOG_ERROR, "failed to allocate %d encoder_worker.\n", i);
            return -1;
        }
        enc_work->enc_thread_state = 0;
        enc_work->devid = 0;
        enc_work->should_exit = 0;
        enc_work->enc_flush = 0;
        enc_work->enc_frame_fifo = NULL;
        enc_work->buffered_frame = NULL;
        enc_work->filtered_frame = NULL;
        enc_work->output_file = NULL;
        enc_work->encoder_name = NULL;
        enc_work->encoder_params = NULL;
        enc_work->init_enc = 0;
        enc_work->encode_num = 0;
        enc_work->enc_ctx = NULL;
        enc_work->outfile = NULL;
        p_args->enc_work[i] = enc_work;
    }
    return 0;
}

static int uninit_prog_args(prog_args *p_args) {
    if (!p_args) {
        return 0;
    }
    int i;
    avcodec_free_context(&p_args->dec_work->dec_ctx);
    avformat_close_input(&p_args->dec_work->ifmt_ctx);
    avfilter_graph_free(&p_args->filt_work->filter_graph);
    // encoder free in cleanup_encoder_worker

    free(p_args->dec_work);
    free(p_args->filt_work);
    for (i = 0; i < MAX_FILTER_SINK; i++) {
        free(p_args->enc_work[i]);
    }
    return 0;
}

static int init_encoder_worker(encoder_worker *enc_worker) {
    int ret;

    ret = pthread_mutex_init(&enc_worker->frame_lock, NULL);
    if (ret) {
        goto fail_init_frame_lock;
    }

    ret = pthread_cond_init(&enc_worker->frame_cond, NULL);
    if (ret) {
        goto fail_init_frame_cond;
    }

    // enc cache three frame
#if LIBAVCODEC_VERSION_MAJOR >= 61 //7.0
    enc_worker->enc_frame_fifo = av_fifo_alloc2(1, sizeof(AVFrame*), AV_FIFO_FLAG_AUTO_GROW);
#else
    enc_worker->enc_frame_fifo = av_fifo_alloc_array(1, sizeof(AVFrame*));
#endif
    if (! enc_worker->enc_frame_fifo) {
        goto fail_init_fifo;
    }

    enc_worker->buffered_frame = av_frame_alloc();

    if (!enc_worker->buffered_frame) {
        goto fail_init_buf;
    }

    return 0;

fail_init_buf:
    free_fifo(enc_worker->enc_frame_fifo);
fail_init_fifo:
    pthread_cond_destroy(&enc_worker->frame_cond);
fail_init_frame_cond:
    pthread_mutex_destroy(&enc_worker->frame_lock);
fail_init_frame_lock:
    return -1;
}

static void cleanup_encoder_worker(encoder_worker *enc_worker) {
    if (enc_worker) {
        pthread_mutex_destroy(&enc_worker->frame_lock);
        pthread_cond_destroy(&enc_worker->frame_cond);
        free_fifo(enc_worker->enc_frame_fifo);
        av_frame_free(&enc_worker->buffered_frame);
        if (enc_worker->enc_ctx) {
            avcodec_free_context(&enc_worker->enc_ctx);
        }
        if (enc_worker->outfile) {
            fclose(enc_worker->outfile);
        }
    }
}

static int open_input_file(decoder_worker *dec_work) {
    int i, ret;
    AVCodec *dec;
    if ((ret = avformat_open_input(&dec_work->ifmt_ctx, dec_work->input_file, NULL, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot open input file %s\n", dec_work->input_file);
        return ret;
    }
    if ((ret = avformat_find_stream_info(dec_work->ifmt_ctx, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot find stream information\n");
        return ret;
    }

    for (i = 0; i < dec_work->ifmt_ctx->nb_streams; i++) {
        if (dec_work->ifmt_ctx->streams[i]->codecpar->codec_type == AVMEDIA_TYPE_VIDEO) {
            dec_work->video_stream_index = i;
            break;
        }
    }
    if (dec_work->video_stream_index < 0) {
        av_log(NULL, AV_LOG_ERROR, "can not find video stream\n");
        return dec_work->video_stream_index;
    }

    dec = avcodec_find_decoder_by_name(dec_work->decoder_name);
    if (dec) {
        if (dec_work->ifmt_ctx->streams[dec_work->video_stream_index]->codecpar->codec_id != dec->id) {
            av_log(NULL, AV_LOG_ERROR, "codec does not match with stream id\n");
            return AVERROR_DECODER_NOT_FOUND;
        }
    }

    /* create decoding context */
    AVCodecContext *dec_ctx = avcodec_alloc_context3(dec);
    if (!dec_ctx){
        av_log(NULL, AV_LOG_ERROR, "Failed to allocate the decoder context for stream\n");
        return AVERROR(ENOMEM);
    }
    avcodec_parameters_to_context(dec_ctx, dec_work->ifmt_ctx->streams[dec_work->video_stream_index]->codecpar);
    if(dec_work->devid >= 0)
    {
        char str_devid[4] = {0};
        snprintf(str_devid, sizeof(str_devid), "%d", dec_work->devid);
        av_opt_set(dec_ctx->priv_data, "dec", str_devid, 0);
    }
    if(dec_work->decoder_params)
        av_opt_set(dec_ctx->priv_data, "xcoder-params", dec_work->decoder_params, 0);

    dec_ctx->framerate = av_guess_frame_rate(dec_work->ifmt_ctx, dec_work->ifmt_ctx->streams[dec_work->video_stream_index], NULL);

    ret = avcodec_parameters_from_context(dec_work->ifmt_ctx->streams[dec_work->video_stream_index]->codecpar, dec_ctx);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to copy decoder parameters from codec context for stream #");
        return ret;
    }

    /* init the video decoder */
    if ((ret = avcodec_open2(dec_ctx, dec, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot open video decoder\n");
        return ret;
    }
    dec_work->dec_ctx = dec_ctx;
    return 0;
}

static int open_output_file(encoder_worker *enc_work, AVFrame *frame) {
    int ret = 0;

    enc_work->outfile = fopen(enc_work->output_file,"wb");
    if(!enc_work->outfile)
    {
        av_log(NULL, AV_LOG_ERROR,"open file fail!\n");
        return -1;
    }

    AVCodec *enc;
    // Find video encoder codec selected
    enc = avcodec_find_encoder_by_name(enc_work->encoder_name);
    if (!enc)
    {
        av_log(NULL, AV_LOG_ERROR,"Codec '%s' not found\n", enc_work->encoder_name);
        return AVERROR_ENCODER_NOT_FOUND;
    }

    // Allocate codec context for encoding
    AVCodecContext *enc_ctx = avcodec_alloc_context3(enc);
    if (!enc_ctx)
    {
        av_log(NULL, AV_LOG_ERROR, "Could not allocate video codec context\n");
        return AVERROR(ENOMEM);
    }

    // enc_ctx->bit_rate = p_args->bitrate;
    enc_ctx->height = frame->height;
    enc_ctx->width = frame->width;
    enc_ctx->sample_aspect_ratio = frame->sample_aspect_ratio;
    enc_ctx->pix_fmt = frame->format;
    if (frame->hw_frames_ctx)
    {
        AVHWFramesContext* pAVHFWCtx = (AVHWFramesContext *)frame->hw_frames_ctx->data;
        enc_ctx->sw_pix_fmt = pAVHFWCtx->sw_format;
    }

    /* video time_base can be set to whatever is handy and supported by encoder */
    enc_ctx->time_base = av_inv_q(p_args->dec_work->dec_ctx->framerate);

    if ((enc->id == AV_CODEC_ID_H264) || (enc->id == AV_CODEC_ID_H265))
    {
        if(enc_work->encoder_params)
        {
            av_opt_set(enc_ctx->priv_data, "xcoder-params", enc_work->encoder_params, 0);
        }
        if(enc_work->devid >= 0)
        {
            char str_devid[4] = {0};
            snprintf(str_devid, sizeof(str_devid), "%d", enc_work->devid);
            av_opt_set(enc_ctx->priv_data, "enc", str_devid, 0);
        }
    }
    else
    {
        av_log(NULL, AV_LOG_ERROR, "codec id %d not supported.\n", enc->id);
        ret = -1;
        return ret;
    }

    // Open encoder
    ret = avcodec_open2(enc_ctx, enc, NULL);
    if (ret < 0)
    {
        av_log(NULL, AV_LOG_ERROR, "Could not open enc\n");
        return ret;
    }
    enc_work->enc_ctx = enc_ctx;
    return ret;
}

static int init_filters(filter_worker *filt_work, decoder_worker *dec_work)
{
    char args[512] = { 0 };
    char name[32] = { 0 };
    int i, ret = 0;
    AVFilterInOut *inputs, *outputs, *cur;
    AVBufferSrcParameters *par;
    AVRational time_base = dec_work->ifmt_ctx->streams[dec_work->video_stream_index]->time_base;
    AVCodecContext *dec_ctx = dec_work->dec_ctx;

    // enum AVPixelFormat pix_fmts[] = {AV_PIX_FMT_NI_QUAD,AV_PIX_FMT_BGRA,AV_PIX_FMT_BGRP,AV_PIX_FMT_YUV420P,AV_PIX_FMT_NONE};

    filt_work->filter_graph = avfilter_graph_alloc();
    if (filt_work->filter_graph == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate filter graph\n");
        goto end;
    }

    ret = avfilter_graph_parse2(filt_work->filter_graph, filt_work->filter_desc, &inputs, &outputs);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to parse graph\n");
        goto end;
    }

    // link input
    cur = inputs, i = 0;
    snprintf(name, sizeof(name), "in_%d", i);
    snprintf(args, sizeof(args),
            "video_size=%dx%d:pix_fmt=%d:time_base=%d/%d:pixel_aspect=%d/%d:frame_rate=%d/%d",
            dec_ctx->width, dec_ctx->height, dec_ctx->pix_fmt,time_base.num, time_base.den,
            dec_ctx->sample_aspect_ratio.num, dec_ctx->sample_aspect_ratio.den,dec_ctx->framerate.num,dec_ctx->framerate.den);
    ret = avfilter_graph_create_filter(&filt_work->buffersrc_ctx, avfilter_get_by_name("buffer"), name,
                                    args, NULL, filt_work->filter_graph);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot create buffer source\n");
        goto end;
    }

    // decoder out=hw
    if (dec_ctx->hw_frames_ctx != NULL) {
        av_log(NULL, AV_LOG_INFO, "hw mode filter\n");
        // Allocate a new AVBufferSrcParameters instance when decoder out=hw
        par = av_buffersrc_parameters_alloc();
        if (!par) {
            ret = AVERROR(ENOMEM);
            goto end;
        }
        memset(par, 0, sizeof(*par));
        // set format and hw_frames_ctx to AVBufferSrcParameters when out=hw
        par->format = AV_PIX_FMT_NONE;
        par->hw_frames_ctx = dec_ctx->hw_frames_ctx;
#if LIBAVCODEC_VERSION_MAJOR  >= 61
        par->color_range = dec_ctx->color_range;
        par->color_space = dec_ctx->colorspace;
#endif
        av_log(NULL, AV_LOG_DEBUG, "hw_frames_ctx %p\n", par->hw_frames_ctx);
        // Initialize the buffersrc filter with the provided parameters
        ret = av_buffersrc_parameters_set(filt_work->buffersrc_ctx, par);
        av_freep(&par);
        if (ret < 0)
            goto end;
    } else { // decoder out=sw
        av_log(NULL, AV_LOG_ERROR, "not sw mode filter for ppu and ni_quadra_split\n");
        ret = -1;
        goto end;
    }

    ret = avfilter_link(filt_work->buffersrc_ctx, 0, cur->filter_ctx, cur->pad_idx);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to link input filter\n");
        goto end;
    }

    // link output
    for (cur = outputs, i = 0; cur && i < p_args->split_num; cur = cur->next, i++) {
        snprintf(name, sizeof(name), "out_%d", i);
        ret = avfilter_graph_create_filter(&filt_work->buffersink_ctx[i], avfilter_get_by_name("buffersink"),
                                           name, NULL, NULL, filt_work->filter_graph);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "failed to create output filter: %d\n", i);
            goto end;
        }

        // connect  dst (index i) pads to one of buffer sink (index 0) pad
        ret = avfilter_link(cur->filter_ctx, cur->pad_idx, filt_work->buffersink_ctx[i], 0);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "failed to link output filter: %d\n", i);
            goto end;
        }
    }

    // configure and validate the filter graph
    ret = avfilter_graph_config(filt_work->filter_graph, NULL);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "%s failed to config graph filter\n",
               __func__);
        goto end;
    } else {
        av_log(NULL, AV_LOG_INFO, "%s success config graph filter %s\n",
               __func__, filt_work->filter_desc);
        //av_log(NULL, AV_LOG_INFO, "%s filter graph dump:\n%s\n", __func__,
        //       avfilter_graph_dump(f->filter_graph, NULL));
    }

end:
    avfilter_inout_free(&inputs);
    avfilter_inout_free(&outputs);
    return ret;
}

// place a filtered frame into queue for encoding
static int send_encode_frame(encoder_worker *enc_worker, AVFrame *filt_frame)
{
    int ret = 0;

    if (!filt_frame) {
        av_log(NULL, AV_LOG_ERROR, "%s: filt frame is null\n", __func__);
        return -1;
    }

    pthread_mutex_lock(&enc_worker->frame_lock);
    while (is_fifo_full(enc_worker->enc_frame_fifo))
    {
        av_log(NULL, AV_LOG_INFO, "dec or enc fifo space has full, wait encoder thread process\n");
        pthread_cond_wait(&enc_worker->frame_cond, &enc_worker->frame_lock);
    }
    // call av_frame_ref to increase buffer ref count / preserve buffer
    ret = av_frame_ref(enc_worker->buffered_frame, filt_frame);
    if (ret < 0)
    {
        av_log(NULL, AV_LOG_ERROR, "%s: av_frame_ref ERROR %d!!!\n", __func__, ret);
        pthread_mutex_unlock(&enc_worker->frame_lock);
        return ret;
    }

#if LIBAVCODEC_VERSION_MAJOR >= 61 //7.0
    ret = av_fifo_write(enc_worker->enc_frame_fifo, &enc_worker->buffered_frame, 1);
#else
    ret = av_fifo_generic_write(enc_worker->enc_frame_fifo, &enc_worker->buffered_frame, sizeof(AVFrame*), NULL);
#endif
    pthread_cond_signal(&enc_worker->frame_cond);
    pthread_mutex_unlock(&enc_worker->frame_lock);

    return ret;
}

/* Send one frame to encoder, attempt to read all available encoder packets from encoder,
   writing packets to output stream. If run with with *frame=NULL, flush the encoder of
   remaining packets until encoder reports EOF */
static int encode_frame(encoder_worker *enc_work, AVFrame *frame, AVPacket *pkt)
{
    int ret;
    if (!enc_work->init_enc && frame)
    {
        if((ret = open_output_file(enc_work, frame)) < 0)
        {
            av_log(NULL, AV_LOG_ERROR, "init encoder failed ret = %d\n", ret);
            return ret;
        }
        enc_work->init_enc = 1;
    }

    int attempt_cnt = 1;
    int re_send = 0;

    // Send frame to encoder. If frame=NULL, this signals end of stream
    if (frame)
    {
        frame->pict_type = AV_PICTURE_TYPE_NONE;
        av_log(NULL, AV_LOG_DEBUG, "Send frame %3"PRId64", type=%d\n", frame->pts, frame->pict_type);
    } else
    {
        av_log(NULL, AV_LOG_DEBUG,"Send done, flush encoder\n");
    }

resend:
    ret = avcodec_send_frame(enc_work->enc_ctx, frame);
    // Encoder is too busy to receive frame, try sending frame again
    if (ret == AVERROR(EAGAIN))
    {
        av_log(NULL, AV_LOG_ERROR, "Resending frame. Try #%d.\n", ++attempt_cnt);
        ret = 0;
        re_send = 1;
    }
    // Frame send issue occured
    else if (ret < 0)
    {
        av_log(NULL, AV_LOG_ERROR, "Error sending frame for encoding. RC=%d\n", AVUNERROR(ret));
        return ret;
    }

    /* Run this loop if a frame was sucessfully sent during avcodec_send_frame() OR
        no frame was sent because this function is in flushing mode OR
        a frame was sucessfully received by avcodec_receive_packet() */
    while (ret >= 0 || ! frame )
    {
        av_log(NULL, AV_LOG_DEBUG, "Receiving packet\n");
        ret = avcodec_receive_packet(enc_work->enc_ctx, pkt);
        if (frame == NULL)
            av_log(NULL, AV_LOG_DEBUG, "Draining packet ret=%x, pkt size=%5d\n", ret, pkt->size);

        // No packets in encoder output buffer but not end of stream either
        if (ret == AVERROR(EAGAIN))
        {
            // If function not in flushing mode
            if (frame)
            {
                if (re_send)
                {
                    av_log(NULL, AV_LOG_DEBUG, "resend frame\n");
                    re_send = 0;
                    goto resend;
                }
                return 0; // Exit this function to send next packet
            }
            else
            {
                continue; // Loop in this function to flush encoder
            }
        }
        // End of stream detected
        else if (ret == AVERROR_EOF)
        {
            av_log(NULL, AV_LOG_DEBUG, "Encoder EOF reached\n");
            return 0;
        }
        // Packet receive error occured
        else if (ret < 0)
        {
            av_log(NULL, AV_LOG_ERROR, "Error during encoding\n");
            return ret;
        }
        // Packet sucessfully received
        av_log(NULL, AV_LOG_DEBUG,"Received packet, pts=%"PRId64"\n", pkt->pts);
        enc_work->encode_num++;

        fwrite(pkt->data, 1, pkt->size, enc_work->outfile);
        av_packet_unref(pkt);
    }
    return 0;
}

// encoder thread routine
static void *encoder_thread_run(void *thread_data)
{
    int ret = 0;
    encoder_worker *enc_worker = (encoder_worker *)thread_data;
    AVPacket *enc_packet = av_packet_alloc();
    while (1) {
        pthread_mutex_lock(&enc_worker->frame_lock);
        while (is_fifo_empty(enc_worker->enc_frame_fifo)) {
            // flush the encoder if filter has flushed
            if (enc_worker->enc_flush) {
                av_log(NULL, AV_LOG_INFO, "%s filter flushed, encoder ready to flush\n",
                       __func__);
                pthread_mutex_unlock(&enc_worker->frame_lock);
                goto flush;
            }

            if (! enc_worker->should_exit) {
                pthread_cond_wait(&enc_worker->frame_cond, &enc_worker->frame_lock);
            } else {
                pthread_mutex_unlock(&enc_worker->frame_lock);
                break;
            }
        }

        // read encode frame from encoder fifo buffer
#if LIBAVCODEC_VERSION_MAJOR >= 61 //7.0
        av_fifo_read(enc_worker->enc_frame_fifo, &enc_worker->filtered_frame, 1);
#else
        av_fifo_generic_read(enc_worker->enc_frame_fifo, &enc_worker->filtered_frame, sizeof(AVFrame*), NULL);
#endif
        pthread_cond_signal(&enc_worker->frame_cond);
        if (enc_worker->filtered_frame->data != enc_worker->filtered_frame->extended_data) {
            av_log(NULL, AV_LOG_DEBUG, "%s frame %p data %p != extended_data %p\n",
                   __func__, enc_worker->filtered_frame, enc_worker->filtered_frame->data,
                   enc_worker->filtered_frame->extended_data);
            enc_worker->filtered_frame->extended_data = enc_worker->filtered_frame->data;
        }

        if (!enc_worker->init_enc) {
            if ((ret = open_output_file(enc_worker, enc_worker->filtered_frame)) < 0) {
                av_log(NULL, AV_LOG_ERROR, "init encoder failed ret = %d\n", ret);
                av_frame_unref(enc_worker->filtered_frame);
                pthread_mutex_unlock(&enc_worker->frame_lock);
                goto end;
            }
            enc_worker->init_enc = 1;
        }
        if ((ret = encode_frame(enc_worker, enc_worker->filtered_frame, enc_packet)) < 0) {
            av_log(NULL, AV_LOG_ERROR, "encode one frame failed\n");
            av_frame_unref(enc_worker->filtered_frame);
            pthread_mutex_unlock(&enc_worker->frame_lock);
            goto end;
        }

        av_frame_unref(enc_worker->filtered_frame);
        av_log(NULL, AV_LOG_DEBUG, "encode one frame finish\n");
        pthread_mutex_unlock(&enc_worker->frame_lock);
    } // while
flush:
    ret = encode_frame(enc_worker, NULL, enc_packet);
end:
    av_packet_free(&enc_packet);
    return (void *)((long)ret);
}

static void print_report(int is_last_report,
        int64_t timer_start, int64_t cur_time)
{
    int i;
    unsigned long total_frame_num = 0;
    for (i = 0; i < p_args->split_num; i++) {
        total_frame_num += p_args->enc_work[i]->encode_num;
    }
    static int64_t last_time = -1;
    float t;
    char buf[1024];
    float fps;

    if (!is_last_report) {
        if (last_time == -1) {
            last_time = cur_time;
            return;
        }
        if ((cur_time - last_time) < 500000)
            return;
        last_time = cur_time;
    }

    t = (cur_time - timer_start) / 1000000.0;

    fps = t > 1 ? total_frame_num / t : 0;

    const char end = is_last_report ? '\n' : '\r';

    buf[0] = '\0';
    snprintf(buf + strlen(buf), sizeof(buf) - strlen(buf), "frame=%5lu fps=%3.*f ",
            total_frame_num, (fps < 9.95), fps);
    fprintf(stderr, "%s   %c", buf, end);
    fflush(stderr);
}

static void *printfps(void *arg) {
    while (1) {
        if (end == 3) {
            break;
        }
        usleep(100);
        print_report(0, timer_start, av_gettime_relative());
    }
    print_report(1, timer_start, av_gettime_relative());
    return NULL;
}

static void help_usage(void) {
    printf("Usage: \n"
            "-i | --input             input file path.\n"
            "-d | --decoder           decoder name [h264_ni_quadra_dec | h265_ni_quadra_dec].\n"
            "-p | --decoder_params    decoder parameters.\n"
            "-o | --output_file       output file path.\n"
            "-e | --encoder           encoder name [h264_ni_quadra_enc | h265_ni_quadra_enc].\n"
            "-x | --encode_params     encoder parameters.\n"
            "-f | --filter_desc       filter description.\n"
            "-n | --devid             device id.\n"
            "-v | --loglevel          available debug level: warning, info, debug, trace.\n"
            "-h | --help              print this help information.\n");
}

int setup_loglevel(char *loglevel) {
    int i;
    const struct { const char *name; int level; } log_levels[] =
    {
         { "quiet"  , AV_LOG_QUIET   },
         { "panic"  , AV_LOG_PANIC   },
         { "fatal"  , AV_LOG_FATAL   },
         { "error"  , AV_LOG_ERROR   },
         { "warning", AV_LOG_WARNING },
         { "info"   , AV_LOG_INFO    },
         { "verbose", AV_LOG_VERBOSE },
         { "debug"  , AV_LOG_DEBUG   },
         { "trace"  , AV_LOG_TRACE   },
    };

    for (i = 0; i < (sizeof(log_levels) / sizeof((log_levels)[0])); i++) {
        if (!strcmp(log_levels[i].name, loglevel)) {
            av_log_set_level(log_levels[i].level);
            return 0;
        }
    }

    return -1;
}

int main(int argc, char **argv) {
    int ret, i;
    int opt, opt_index;
    char *loglevel = "info";
    AVPacket *packet = NULL;
    AVFrame *frame = NULL;
    AVFrame *filt_frame = NULL;
    pthread_t time_tid;
    int fps_thread_state = 0;
    void *result = NULL;
    int saw_first_ts = 0;
    AVStream *st = NULL;
    int64_t duration_dts = 0;
    int64_t last_decoded_dts = AV_NOPTS_VALUE;
    int64_t last_decoded_pts = AV_NOPTS_VALUE;

    p_args = malloc(sizeof(prog_args));
    if (p_args == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate codec workers.\n");
        goto end;
    }

    ret = init_prog_args(p_args);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to init prog args.\n");
        goto end;
    }
    int output_file_num = 0, encoder_name_num = 0, encoder_params_num = 0;
    decoder_worker *dec_work = p_args->dec_work;
    filter_worker *filt_work = p_args->filt_work;
    const char *opt_string = "i:d:p:o:e:x:f:b:n:v:h";
    static struct option long_options[] = {
        {"input",          required_argument, NULL, 'i'},
        {"decoder",        required_argument, NULL, 'd'},
        {"decoder_params", required_argument, NULL, 'p'},
        {"output_file",    required_argument, NULL, 'o'},
        {"encoder",        required_argument, NULL, 'e'},
        {"encode_params",  required_argument, NULL, 'x'},
        {"filter_desc",    required_argument, NULL, 'f'},
        // {"bitrate",        required_argument, NULL, 'b'},
        {"devid",          required_argument, NULL, 'n'},
        {"loglevel",       required_argument, NULL, 'v'},
        {"help",           no_argument,       NULL, 'h'},
        { NULL,            0,                 NULL,  0 },
    };

    while ((opt = getopt_long(argc, argv, opt_string, long_options, &opt_index)) != -1) {
        switch (opt) {
            case 'i':
                dec_work->input_file = optarg;
                break;
            case 'd':
                dec_work->decoder_name = optarg;
                break;
            case 'p':
                dec_work->decoder_params = optarg;
                break;
            case 'o':
                if (output_file_num > MAX_FILTER_SINK) {
                    av_log(NULL, AV_LOG_ERROR, "Error, exceeding max output_file_num 12\n");
                    goto end;
                }
                p_args->enc_work[output_file_num]->output_file = optarg;
                output_file_num++;
                break;
            case 'e':
                if (encoder_name_num > MAX_FILTER_SINK) {
                    av_log(NULL, AV_LOG_ERROR, "Error, exceeding max encoder_name_num 12\n");
                    goto end;
                }
                p_args->enc_work[encoder_name_num]->encoder_name = optarg;
                encoder_name_num++;
                break;
            case 'x':
                if (encoder_params_num > MAX_FILTER_SINK) {
                    av_log(NULL, AV_LOG_ERROR, "Error, exceeding max encoder_params_num 12\n");
                    goto end;
                }
                p_args->enc_work[encoder_params_num]->encoder_params = optarg;
                encoder_params_num++;
                break;
            case 'f':
                filt_work->filter_desc = optarg;
                break;
            // case 'b':
            //     p_args->bitrate = atoi(optarg);
            //     break;
            case 'n':
                p_args->devid = atoi(optarg);
                break;
            case 'v':
                loglevel = optarg;
                break;
            case 'h':
                help_usage();
                return EXIT_SUCCESS;
            default:
                help_usage();
                return EXIT_FAILURE;
        }
    }

    if(setup_loglevel(loglevel) < 0) {
        av_log(NULL, AV_LOG_ERROR, "set log level failed.\n");
        ret = EXIT_FAILURE;
        goto end;
    }

    if (!dec_work->input_file || !dec_work->decoder_name) {
        av_log(NULL, AV_LOG_ERROR, "input file name must be specified.\n");
        ret = EXIT_FAILURE;
        goto end;
    }

    if (output_file_num != encoder_name_num) {
        av_log(NULL, AV_LOG_ERROR, "output file name num not match encoder name num.\n");
        ret = EXIT_FAILURE;
        goto end;
    }
    p_args->split_num = output_file_num;

    for (i = 0; i < output_file_num; i++) {
        if (!p_args->enc_work[i]->output_file || !p_args->enc_work[i]->encoder_name) {
            av_log(NULL, AV_LOG_ERROR, "output file name and encoder name must be specified.\n");
            ret = EXIT_FAILURE;
            goto end;
        }
    }

    if(!filt_work->filter_desc) {
        av_log(NULL, AV_LOG_ERROR, "filter description not be specified.\n");
        ret = EXIT_FAILURE;
        goto end;
    }

    // Register all formats, codecs, filters
#if LIBAVFORMAT_VERSION_MAJOR < 58
    av_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif
#if LIBAVCODEC_VERSION_MAJOR < 58
    avcodec_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif
#if LIBAVFILTER_VERSION_MAJOR < 7
    avfilter_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif

    if (p_args->devid >= 0) {
        dec_work->devid = p_args->devid;
        for (i = 0; i < p_args->split_num; i++) {
            p_args->enc_work[i]->devid = p_args->devid;
        }
    }

    // Allocate packet for encoder output
    packet = av_packet_alloc();
    frame = av_frame_alloc();
    filt_frame = av_frame_alloc();
    if (!packet || !frame || !filt_frame) {
        av_log(NULL, AV_LOG_ERROR,"Could not allocate pkt frame\n");
        ret = EXIT_FAILURE;
        goto end;
    }

    if ((ret = open_input_file(dec_work)) < 0)
        goto end;

    for (i = 0; i < p_args->split_num; i++) {
        encoder_worker *enc_worker = p_args->enc_work[i];
        if (init_encoder_worker(enc_worker)) {
            av_log(NULL, AV_LOG_ERROR, "init_encoder_worker failed !\n");
            goto end;
        }

        ret = pthread_create(&enc_worker->enc_tid, NULL, &encoder_thread_run, enc_worker);
        if (ret) {
            av_log(NULL, AV_LOG_ERROR, "failed to create codec thread %d: %s.\n", i, strerror(ret));
            cleanup_encoder_worker(enc_worker);
            goto end;
        }
        enc_worker->enc_thread_state = 1;
    }

    timer_start = av_gettime_relative();
    if (pthread_create(&time_tid, NULL, printfps, 0) != 0)
    {
        av_log(NULL, AV_LOG_ERROR,"create thread failed!\n");
        goto end;
    }
    fps_thread_state = 1;

    /* read all packets */
    while (1) {
        if ((ret = av_read_frame(dec_work->ifmt_ctx, packet)) < 0) {
            if(end == 2) {
                for (i = 0; i < p_args->split_num; i++) {
                    p_args->enc_work[i]->enc_flush = 1;
                }
                break;
            }
            end = 1;
        }
        if (packet->stream_index == dec_work->video_stream_index || end == 1) {
            if(!end) {
                if (!saw_first_ts) {
                    st = dec_work->ifmt_ctx->streams[dec_work->video_stream_index];
                    last_decoded_dts = st->avg_frame_rate.num ? - dec_work->dec_ctx->has_b_frames * AV_TIME_BASE / av_q2d(st->avg_frame_rate) : 0;
                    if (packet && packet->pts != AV_NOPTS_VALUE) {
                        last_decoded_dts += av_rescale_q(packet->pts, st->time_base, AV_TIME_BASE_Q);
                    }
                    saw_first_ts = 1;
                }

                // set pkt dts here
                if (last_decoded_dts != AV_NOPTS_VALUE) {
                    packet->dts = av_rescale_q(last_decoded_dts, AV_TIME_BASE_Q, st->time_base);
                    av_log(NULL, AV_LOG_DEBUG, "send pkt dts %ld lastdts %ld\n", packet->dts, last_decoded_dts);
                }
                ret = avcodec_send_packet(dec_work->dec_ctx, packet);

                // update the last_decoded_dts
                if (packet && packet->duration) {
                    duration_dts = av_rescale_q(packet->duration, st->time_base, AV_TIME_BASE_Q);
                } else if(dec_work->dec_ctx->framerate.num != 0 && dec_work->dec_ctx->framerate.den != 0) {
                    int ticks= av_stream_get_parser(st) ? av_stream_get_parser(st)->repeat_pict+1 : dec_work->dec_ctx->ticks_per_frame;
                    duration_dts = ((int64_t)AV_TIME_BASE *
                                    dec_work->dec_ctx->framerate.den * ticks) /
                                    dec_work->dec_ctx->framerate.num / dec_work->dec_ctx->ticks_per_frame;
                }

                if(last_decoded_dts != AV_NOPTS_VALUE && duration_dts) {
                    last_decoded_dts += duration_dts;
                }else
                    last_decoded_dts = AV_NOPTS_VALUE;
            }
            else{
                ret = avcodec_send_packet(dec_work->dec_ctx, NULL);
            }

            if (ret < 0 && ret != AVERROR_EOF) {
                av_log(NULL, AV_LOG_ERROR, "Error while sending a packet to the decoder\n");
                break;
            }
            while (ret >= 0 || end == 1) {
                ret = avcodec_receive_frame(dec_work->dec_ctx, frame);
                if (ret == AVERROR(EAGAIN)) {
                    break;
                }
                else if(ret == AVERROR_EOF) {
                    end = 2;
                    break;
                }
                else if (ret < 0) {
                    av_packet_unref(packet);
                    av_log(NULL, AV_LOG_ERROR, "Error while receiving a frame from the decoder\n");
                    goto end;
                }
                frame->pts = frame->best_effort_timestamp;
                if (frame->pts == AV_NOPTS_VALUE) {
                    if (last_decoded_pts == AV_NOPTS_VALUE) {
                        frame->pts = 0;
                    } else {
#if LIBAVCODEC_VERSION_MAJOR >= 61 //7.0
                        frame->pts = last_decoded_pts + frame->duration;
#else
                        frame->pts = last_decoded_pts + frame->pkt_duration;
#endif
                    }
                    av_log(NULL, AV_LOG_DEBUG, "%s frame->pts == AV_NOPTS_VALUE, adjust to %ld\n", __func__, frame->pts);
                }
                last_decoded_pts = frame->pts;

                if (!filt_work->init_filter) {
                    if ((ret = init_filters(filt_work, dec_work)) < 0) {
                        av_log(NULL, AV_LOG_ERROR, "init filter failed ret = %d\n", ret);
                        goto end;
                    }
                    filt_work->init_filter = 1;
                }
                if ((ret = av_buffersrc_add_frame_flags(filt_work->buffersrc_ctx, frame,
                                                        AV_BUFFERSRC_FLAG_KEEP_REF)) < 0) {
                    av_frame_unref(frame);
                    av_packet_unref(packet);
                    av_log(NULL, AV_LOG_ERROR, "Error while feeding the filtergraph\n");
                    goto end;
                }

                /* pull filtered frames from the filtergraph */
                while (1) {
                    for (i = 0; i < p_args->split_num; i++) {
                        ret = av_buffersink_get_frame(filt_work->buffersink_ctx[i], filt_frame);
                        if (ret == AVERROR(EAGAIN)) {
                            goto filter_one_frame;
                        }
                        else if(ret == AVERROR_EOF) {
                            goto filter_one_frame;
                        }
                        else if (ret < 0) {
                            av_log(NULL, AV_LOG_ERROR, "av_buffersink_get_frame failed ret = %d\n", ret);
                            goto end;
                        }
                        ret = send_encode_frame(p_args->enc_work[i], filt_frame);
                        av_frame_unref(filt_frame);
                        if (ret < 0) {
                            av_log(NULL, AV_LOG_ERROR, "encode send frame failed ret = %d\n", ret);
                            goto end;
                        }
                    }
                }
filter_one_frame:
                av_frame_unref(frame);
            }
        }
        av_packet_unref(packet);
    }
end:
    for (i = 0; i < p_args->split_num; i++) {
        if (p_args->enc_work[i]->enc_thread_state) {
            pthread_mutex_lock(&p_args->enc_work[i]->frame_lock);
            p_args->enc_work[i]->should_exit = 1;
            pthread_cond_signal(&p_args->enc_work[i]->frame_cond);
            pthread_mutex_unlock(&p_args->enc_work[i]->frame_lock);
            if (pthread_join(p_args->enc_work[i]->enc_tid, &result) == 0) {
                if ((long)result != 0) {
                    av_log(NULL, AV_LOG_INFO, "pthread_join fps thread ret %ld\n",
                        (long)result);
                    ret = EXIT_FAILURE;
                }
            }
            cleanup_encoder_worker(p_args->enc_work[i]);
        }
    }

    end = 3;
    if (fps_thread_state) {
        if (pthread_join(time_tid, &result) == 0) {
            if ((long)result != 0) {
                av_log(NULL, AV_LOG_INFO, "pthread_join fps thread ret %ld\n",
                    (long)result);
                ret = EXIT_FAILURE;
            }
        }
    }

    av_frame_free(&frame);
    av_frame_free(&filt_frame);
    av_packet_free(&packet);
    uninit_prog_args(p_args);
    free(p_args);
    if (ret < 0 && ret != AVERROR_EOF) {
        av_log(NULL, AV_LOG_ERROR, "Error occurred: %s\n", av_err2str(ret));
        return EXIT_FAILURE;
    }

    return 0;
}
