/*
 * Copyright (c) 2010 Nicolas George
 * Copyright (c) 2011 Stefano Sabatini
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * @file
 * API example for transcode and AI
 * @example ni_transcode_ai_test.c
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>

#include <libavcodec/avcodec.h>
#include <libavformat/avformat.h>
#include <libavfilter/buffersink.h>
#include <libavfilter/buffersrc.h>
#include <libswscale/swscale.h>
#include <libavutil/imgutils.h>
#include <libavutil/opt.h>
#include <libavutil/time.h>
#include <libavutil/channel_layout.h>
#include <pthread.h>

FILE *outfile = NULL;
AVFrame *frame;
AVFrame *filt_frame;
AVFrame *frame1;
static AVFormatContext *fmt_ctx;
static AVCodecContext *dec_ctx;
static AVCodecContext *enc_ctx;
AVFilterContext *buffersink_ctx;
AVFilterContext *buffersrc_ctx;
AVFilterGraph *filter_graph;
static int video_stream_index = -1;
int frame_num = 0,end = 0,encode_num = 0;
int begin_roi = 1;
int64_t timer_start;
struct codec_worker {
    const char *decoder_name;
    const char *input_file;
    const char *output_file;
    const char *decoder_params;
    const char *encoder_name;
    const char *enccoder_params;
    int devid;
    int bitrate;
    int select_frame;
};

struct codec_worker *codec_workers = NULL;

static int open_input_file()
{
    int ret;
    const AVCodec *dec;

    if ((ret = avformat_open_input(&fmt_ctx, codec_workers->input_file, NULL, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot open input file\n");
        return ret;
    }

    if ((ret = avformat_find_stream_info(fmt_ctx, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot find stream information\n");
        return ret;
    }

    dec = avcodec_find_decoder_by_name(codec_workers->decoder_name);
    if (dec) {
        if (fmt_ctx->streams[0]->codecpar->codec_id != dec->id) {
            av_log(NULL, AV_LOG_ERROR, "codec does not match with stream id\n");
            return AVERROR_DECODER_NOT_FOUND;
        }
    }

    video_stream_index = 0;

    /* create decoding context */
    dec_ctx = avcodec_alloc_context3(dec);
    if (!dec_ctx)
        return AVERROR(ENOMEM);

    avcodec_parameters_to_context(dec_ctx, fmt_ctx->streams[0]->codecpar);

    char str_devid[4] = {0};
    snprintf(str_devid, sizeof(str_devid), "%d", codec_workers->devid);
    av_opt_set(dec_ctx->priv_data, "dec", str_devid, 0);
    if(codec_workers->decoder_params)
        av_opt_set(dec_ctx->priv_data, "xcoder-params",codec_workers->decoder_params, 0);

    if (dec_ctx->codec_type == AVMEDIA_TYPE_VIDEO)
            dec_ctx->framerate = av_guess_frame_rate(fmt_ctx, fmt_ctx->streams[0], NULL);

    ret = avcodec_parameters_from_context(fmt_ctx->streams[0]->codecpar, dec_ctx);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to copy decoder parameters from codec context for stream #");
        return ret;
    }

    /* init the video decoder */
    if ((ret = avcodec_open2(dec_ctx, dec, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot open video decoder\n");
        return ret;
    }
    return 0;
}

static int open_output_file(void)
{
    int ret = 0;
    const AVCodec *enc;

#if LIBAVCODEC_VERSION_MAJOR < 58
    // Register all codecs
    avcodec_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif

    // Find video encoder codec selected
    enc = avcodec_find_encoder_by_name(codec_workers->encoder_name);
    if (!enc)
    {
        av_log(NULL, AV_LOG_ERROR,"Codec '%s' not found\n", codec_workers->encoder_name);
        return AVERROR_ENCODER_NOT_FOUND;
    }

    // Allocate codec context for encoding
    enc_ctx = avcodec_alloc_context3(enc);
    if (!enc_ctx)
    {
        av_log(NULL, AV_LOG_ERROR, "Could not allocate video codec context\n");
        return AVERROR(ENOMEM);
    }

    if (codec_workers->bitrate != -1)
        enc_ctx->bit_rate = codec_workers->bitrate;
    enc_ctx->width = dec_ctx->width; // resolution must be divisible by 2
    enc_ctx->height = dec_ctx->height;
    enc_ctx->framerate = dec_ctx->framerate;
    enc_ctx->sample_aspect_ratio = dec_ctx->sample_aspect_ratio;
    enc_ctx->pix_fmt = AV_PIX_FMT_NI_QUAD;
    enc_ctx->sw_pix_fmt = dec_ctx->sw_pix_fmt;
    /* video time_base can be set to whatever is handy and supported by encoder */
    enc_ctx->time_base = av_inv_q(dec_ctx->framerate);

    if ((enc->id == AV_CODEC_ID_H264) || (enc->id == AV_CODEC_ID_H265))
    {
        char str_devid[4] = {0};
        av_opt_set(enc_ctx->priv_data, "xcoder-params", codec_workers->enccoder_params, 0);
        snprintf(str_devid, sizeof(str_devid), "%d", codec_workers->devid);
        av_opt_set(dec_ctx->priv_data, "enc", str_devid, 0);
    }
    else
    {
        av_log(NULL, AV_LOG_ERROR, "codec id %d not supported.\n", enc->id);
        ret = -1;
        return ret;
    }

    // Open encoder
    ret = avcodec_open2(enc_ctx, enc, NULL);
    if (ret < 0)
    {
        av_log(NULL, AV_LOG_ERROR, "Could not open enc\n");
        return ret;
    }

    return ret;
}

/* Send one frame to encoder, attempt to read all available encoder packets from encoder,
   writing packets to output stream. If run with with *frame=NULL, flush the encoder of
   remaining packets until encoder reports EOF */
static void encode(AVCodecContext *enc_ctx, AVFrame *frame, AVPacket *pkt,
                   FILE *outfile)
{
    int rc;
    int attempt_cnt = 1;
    static int send_cnt = 0;
    static int recv_cnt = 0;

    // Send frame to encoder. If frame=NULL, this signals end of stream
    if (frame)
    {
        frame->pict_type = AV_PICTURE_TYPE_NONE;
        av_log(NULL, AV_LOG_DEBUG, "Send %d frame %3"PRId64", type=%d\n", ++send_cnt, frame->pts, frame->pict_type);
    } else
    {
        av_log(NULL, AV_LOG_DEBUG,"Send done, flush encoder\n");
    }
    do
    {
        rc = avcodec_send_frame(enc_ctx, frame);
        // Encoder is too busy to receive frame, try sending frame again
        if (rc == AVERROR(EAGAIN))
        {
            av_log(NULL, AV_LOG_DEBUG, "Resending frame. Try #%d.\n", ++attempt_cnt);
            continue;
        }
        // Frame send issue occured
        else if (rc < 0)
        {
            av_log(NULL, AV_LOG_ERROR, "Error sending frame for encoding. RC=%d\n", AVUNERROR(rc));
            avcodec_free_context(&enc_ctx);
            exit(1);
        }
        // Frame sent successfully
        else
            break;
    }
    while (1);

    /* Run this loop if a frame was sucessfully sent during avcodec_send_frame() OR
        no frame was sent because this function is in flushing mode OR
        a frame was sucessfully received by avcodec_receive_packet() */
    while (rc >= 0 || ! frame )
    {
        av_log(NULL, AV_LOG_DEBUG, "Receiving packet\n");
        rc = avcodec_receive_packet(enc_ctx, pkt);
        if (frame == NULL)
            av_log(NULL, AV_LOG_DEBUG, "Draining packet rc=%x, pkt size=%5d\n", rc, pkt->size);

        // No packets in encoder output buffer but not end of stream either
        if (rc == AVERROR(EAGAIN))
        {
            // If function not in flushing mode
            if (frame)
            return; // Exit this function to send next packet
            else
            continue; // Loop in this function to flush encoder
        }
        // End of stream detected
        else if (rc == AVERROR_EOF)
        {
            av_log(NULL, AV_LOG_DEBUG, "Encoder EOF reached\n");
            return;
        }
        // Packet receive error occured
        else if (rc < 0)
        {
            av_log(NULL, AV_LOG_ERROR, "Error during encoding\n");
            avcodec_free_context(&enc_ctx);
            exit(1);
        }
        // Packet sucessfully received
        recv_cnt++;
        av_log(NULL, AV_LOG_DEBUG,"Received packet #%d, pts=\033[K%3"PRId64"\n", recv_cnt, pkt->pts);

        av_log(NULL, AV_LOG_DEBUG,"Write %d packet pts %3"PRId64" (size=%5d)\n",recv_cnt,
                pkt->pts, pkt->size);
        encode_num++;

        fwrite(pkt->data, 1, pkt->size, outfile);

        av_packet_unref(pkt);
    }
}


static int init_filters(const char *filters_descr)
{
    char args[512];
    int ret = 0;
    const AVFilter *buffersrc  = avfilter_get_by_name("buffer");
    const AVFilter *buffersink = avfilter_get_by_name("buffersink");
    AVFilterInOut *outputs = avfilter_inout_alloc();
    AVFilterInOut *inputs  = avfilter_inout_alloc();
    AVRational time_base = fmt_ctx->streams[video_stream_index]->time_base;

    enum AVPixelFormat pix_fmts[] = { AV_PIX_FMT_NI_QUAD, AV_PIX_FMT_BGR24,AV_PIX_FMT_BGRP,AV_PIX_FMT_YUV420P,AV_PIX_FMT_NONE };

    filter_graph = avfilter_graph_alloc();
    if (!outputs || !inputs || !filter_graph) {
        ret = AVERROR(ENOMEM);
        goto end;
    }
    char flag[32] = "flags=bicubic";
    filter_graph->scale_sws_opts = flag;
    /* buffer video source: the decoded frames from the decoder will be inserted here. */
    snprintf(args, sizeof(args),
            "video_size=%dx%d:pix_fmt=%d:time_base=%d/%d:pixel_aspect=%d/%d:frame_rate=%d/%d",
            dec_ctx->width, dec_ctx->height, dec_ctx->pix_fmt,time_base.num, time_base.den,
            dec_ctx->sample_aspect_ratio.num, dec_ctx->sample_aspect_ratio.den,dec_ctx->framerate.num,dec_ctx->framerate.den);
    ret = avfilter_graph_create_filter(&buffersrc_ctx, buffersrc, "in",
                                       args, NULL, filter_graph);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot create buffer source\n");
        goto end;
    }

    AVBufferSrcParameters *par = av_buffersrc_parameters_alloc();
    if (!par)
        return AVERROR(ENOMEM);

    memset(par, 0, sizeof(*par));
    par->format = AV_PIX_FMT_NONE;
    par->hw_frames_ctx = dec_ctx->hw_frames_ctx;

    ret = av_buffersrc_parameters_set(buffersrc_ctx, par);
    if (ret < 0)
        goto end;
    av_freep(&par);

    /* buffer video sink: to terminate the filter chain. */
    ret = avfilter_graph_create_filter(&buffersink_ctx, buffersink, "out",
                                       NULL, NULL, filter_graph);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot create buffer sink\n");
        goto end;
    }

    ret = av_opt_set_int_list(buffersink_ctx, "pix_fmts", pix_fmts,
                              AV_PIX_FMT_NONE, AV_OPT_SEARCH_CHILDREN);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot set output pixel format\n");
        goto end;
    }

    /*
     * Set the endpoints for the filter graph. The filter_graph will
     * be linked to the graph described by filters_descr.
     */

    /*
     * The buffer source output must be connected to the input pad of
     * the first filter described by filters_descr; since the first
     * filter input label is not specified, it is set to "in" by
     * default.
     */
    outputs->name       = av_strdup("in");
    outputs->filter_ctx = buffersrc_ctx;
    outputs->pad_idx    = 0;
    outputs->next       = NULL;

    /*
     * The buffer sink input must be connected to the output pad of
     * the last filter described by filters_descr; since the last
     * filter output label is not specified, it is set to "out" by
     * default.
     */
    inputs->name       = av_strdup("out");
    inputs->filter_ctx = buffersink_ctx;
    inputs->pad_idx    = 0;
    inputs->next       = NULL;

    if ((ret = avfilter_graph_parse_ptr(filter_graph, filters_descr,
                                    &inputs, &outputs, NULL)) < 0)
        goto end;

    if ((ret = avfilter_graph_config(filter_graph, NULL)) < 0)
    {
        goto end;
    }

end:
    avfilter_inout_free(&inputs);
    avfilter_inout_free(&outputs);

    return ret;
}

static void print_report(int is_last_report, int64_t timer_start,
        int64_t cur_time, unsigned long frame_number)
{
    static int64_t last_time = -1;
    float t;
    char buf[1024];
    float fps;

    if (!is_last_report) {
        if (last_time == -1) {
            last_time = cur_time;
            return;
        }
        if ((cur_time - last_time) < 500000)
            return;
        last_time = cur_time;
    }

    t = (cur_time - timer_start) / 1000000.0;

    fps = t > 1 ? frame_number / t : 0;

    const char end = is_last_report ? '\n' : '\r';

    buf[0] = '\0';
    snprintf(buf + strlen(buf), sizeof(buf) - strlen(buf), "frame=%5ld fps=%3.*f ",
            frame_number, (fps < 9.95), fps);
    fprintf(stderr, "%s   %c", buf, end);
    fflush(stderr);
}

static void *printfps(void *arg)
{
    while (1) {
        if (end == 3) {
            break;
        }
        usleep(100);
        print_report(0, timer_start, av_gettime_relative(), encode_num);
    }
    print_report(1, timer_start, av_gettime_relative(), encode_num);
    return NULL;
}

static void *printAIframe(void *arg)
{
    int ret = 0;
    while(1)
    {
        if(end == 3)
            break;
        if(begin_roi == 0)
        {
            if ((ret = av_buffersrc_add_frame_flags(buffersrc_ctx, frame1, AV_BUFFERSRC_FLAG_KEEP_REF)) < 0) {
                av_log(NULL, AV_LOG_ERROR, "Error while feeding the filtergraph\n");
                break;
            }

            /* pull filtered frames from the filtergraph */
            while (1) {
                ret = av_buffersink_get_frame(buffersink_ctx, filt_frame);
                if (ret == AVERROR(EAGAIN))
                {
                    break;
                }
                else if(ret == AVERROR_EOF)
                {
                    return NULL;
                }
                else if (ret < 0)
                    return NULL;
                printf("format = %d,height = %d,width = %d\n",\
                        filt_frame->format,filt_frame->height,filt_frame->width);
                av_frame_unref(filt_frame);
            }
            av_frame_unref(frame1);
            begin_roi = 1;
        }
    }

    return NULL;
}
static void help_usage(void)
{
    printf("This example program takes in encoded input, decodes it with a netint HW\n"
           "decoder, then applies netint HW scaling. Select 1 frame every X frames to\n"
           "do ROI.\n"
           "\n"
           "Usage example:\n"
           "  ./ni_transcode_ai_test -d h264_ni_quadra_dec -x \"out=hw\" -i vidyo4_720p_60fps.h264 \\\n"
           "  -f \"ni_quadra_roi=nb=./network_binary.nb:qpoffset=-0.6\" -m 30 \\\n"
           "  -e h264_ni_quadra_enc -n 0 -p \"roiEnable=1:RcEnable=1:bitrate=2000000\" \\\n"
           "  -o vidyo4_720p60_roi.h264\n"
           "\n"
           "Usage:\n"
           "  -x | --decoder-params    decoder parameters.\n"
           "  -i | --input             input file path.\n"
           "  -o | --output_file       output file path.\n"
           "  -d | --decoder           decoder name [h264_ni_quadra_dec | h265_ni_quadra_dec].\n"
           "  -n | --devid             device id.\n"
           "  -v | --loglevel          available debug level: warning, info, debug, trace.\n"
           "  -f | --filter_dsc        filter set params.\n"
           "  -b | --bitrate           set bitrate.\n"
           "  -m | --select_frame      frame period to re-acquire target for ROI.\n"
           "  -e | --encode-name       encode name [h264_ni_enc | h265_ni_enc].\n"
           "  -p | --encode_params     encode params.\n"
           "  -h | --help              print this help information.\n");
}

void setup_loglevel(char *loglevel)
{
    int i;
    const struct { const char *name; int level; } log_levels[] =
    {
         { "quiet"  , AV_LOG_QUIET   },
         { "panic"  , AV_LOG_PANIC   },
         { "fatal"  , AV_LOG_FATAL   },
         { "error"  , AV_LOG_ERROR   },
         { "warning", AV_LOG_WARNING },
         { "info"   , AV_LOG_INFO    },
         { "verbose", AV_LOG_VERBOSE },
         { "debug"  , AV_LOG_DEBUG   },
         { "trace"  , AV_LOG_TRACE   },
    };

    for (i = 0; i < (sizeof(log_levels) / sizeof((log_levels)[0])); i++)
    {
        if (!strcmp(log_levels[i].name, loglevel))
        {
            av_log_set_level(log_levels[i].level);
        }
    }
}

int main(int argc, char **argv)
{
    int ret;
    AVPacket packet;
    AVPacket *enc_packet;
    char *loglevel = NULL;
    int opt;
    int opt_index;
    char *filter_dsc;
    pthread_t pthid;
    pthread_t pthtimeid;
    int saw_first_ts = 0;
    AVStream *st = NULL;
    int64_t last_decoded_dts = AV_NOPTS_VALUE;
    int64_t last_decoded_pts = AV_NOPTS_VALUE;
    int64_t duration_dts = 0;
    const char *opt_string = "n:x:i:d:o:e:p:v:b:m:f:h";
    static struct option long_options[] = {
        {"input",          required_argument, NULL, 'i'},
        {"output",         required_argument, NULL, 'o'},
        {"decoder",        required_argument, NULL, 'd'},
        {"decoder-params", required_argument, NULL, 'x'},
        {"devid",          required_argument, NULL, 'n'},
        {"loglevel",       required_argument, NULL, 'v'},
        {"filter_dsc",     required_argument, NULL, 'f'},
        {"bitrate",        required_argument, NULL, 'b'},
        {"encode_name",    required_argument, NULL, 'e'},
        {"encode_params",  required_argument, NULL, 'p'},
        {"select_frame",   required_argument, NULL, 'm'},
        {"help",           no_argument,       NULL, 'h'},
        { NULL,            0,                 NULL,  0 },
    };

    codec_workers = malloc(sizeof(struct codec_worker));
    if (codec_workers == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate codec workers.\n");
        goto end;
    }
    memset(codec_workers, 0, sizeof(struct codec_worker));
    codec_workers->devid = -1;
    codec_workers->bitrate = -1;

    while ((opt = getopt_long(argc, argv, opt_string, long_options, &opt_index)) != -1) {
        switch (opt) {

            case 'd':
                codec_workers->decoder_name = optarg;
                break;
            case 'i':
                codec_workers->input_file = optarg;
                break;
            case 'o':
                codec_workers->output_file = optarg;
                break;
            case 'n':
                codec_workers->devid = atoi(optarg);
                break;
            case 'x':
                codec_workers->decoder_params = optarg;
                break;
            case 'v':
                loglevel = optarg;
                break;
            case 'f':
                filter_dsc = optarg;
                break;
            case 'b':
                codec_workers->bitrate = atoi(optarg);
                break;
            case 'e':
                codec_workers->encoder_name = optarg;
                break;
            case 'm':
                codec_workers->select_frame = atoi(optarg);
                break;
            case 'p':
                codec_workers->enccoder_params = optarg;
                break;
            case 'h':
                help_usage();
                return EXIT_SUCCESS;
            default:
                help_usage();
                return EXIT_FAILURE;
        }
    }

    if (loglevel)
    {
        setup_loglevel(loglevel);
    }

    if (codec_workers->input_file == NULL) {
        av_log(NULL, AV_LOG_ERROR, "input file name must be specified.\n");
        return EXIT_FAILURE;
    }

    if (codec_workers->output_file == NULL) {
        av_log(NULL, AV_LOG_ERROR, "output file name must be specified.\n");
        return EXIT_FAILURE;
    }

    if (!codec_workers->decoder_name || !codec_workers->encoder_name) {
        av_log(NULL, AV_LOG_ERROR, "decoder_name name must be specified.\n");
        return EXIT_FAILURE;
    }

    if(codec_workers->select_frame <= 0)
    {
        av_log(NULL, AV_LOG_ERROR, "select frame must be greater than zero.\n");
        return EXIT_FAILURE;
    }

    outfile = fopen(codec_workers->output_file,"wb");
    if(!outfile)
    {
        perror("open file fail!\n");
        exit(1);
    }
    // Allocate packet for encoder output
    enc_packet = av_packet_alloc();
    frame = av_frame_alloc();
    filt_frame = av_frame_alloc();
    frame1 = av_frame_alloc();
    if (!enc_packet || !frame || !filt_frame || !frame1) {
        perror("Could not allocate pkt frame\n");
        exit(1);
    }

    if ((ret = open_input_file()) < 0)
        goto end;

    if((ret = open_output_file()) < 0)
        goto end;
    if(!filter_dsc)
    {
        exit(1);
    }
    if ((ret = init_filters(filter_dsc)) < 0)
        goto end;

    timer_start = av_gettime_relative();

    if (pthread_create(&pthid, NULL, printAIframe, 0) != 0)
    {
        perror("creat thread failed!\n");
        goto end;
    }

    if (pthread_create(&pthtimeid, NULL, printfps, 0) != 0)
    {
        perror("creat thread failed!\n");
        goto end;
    }

    /* read all packets */
    while (1) {
        if ((ret = av_read_frame(fmt_ctx, &packet)) < 0)
        {
            if(end == 2)
            {
                encode(enc_ctx,NULL,enc_packet,outfile);
                break;
            }

            end = 1;
        }
        if (packet.stream_index == video_stream_index || end == 1) {
            if(!end)
            {
                if (!saw_first_ts) {
                    st = fmt_ctx->streams[video_stream_index];
                    last_decoded_dts = st->avg_frame_rate.num ? - dec_ctx->has_b_frames * AV_TIME_BASE / av_q2d(st->avg_frame_rate) : 0;
                    if (packet.pts != AV_NOPTS_VALUE) {
                        last_decoded_dts += av_rescale_q(packet.pts, st->time_base, AV_TIME_BASE_Q);
                    }
                    saw_first_ts = 1;
                }
                // set pkt dts here
                if (last_decoded_dts != AV_NOPTS_VALUE) {
                    packet.dts = av_rescale_q(last_decoded_dts, AV_TIME_BASE_Q, st->time_base);
                    av_log(NULL, AV_LOG_DEBUG, "send pkt dts %ld lastdts %ld\n", packet.dts, last_decoded_dts);
                }
                ret = avcodec_send_packet(dec_ctx, &packet);
                // update the last_decoded_dts
                if ( packet.duration) {
                    duration_dts = av_rescale_q(packet.duration, st->time_base, AV_TIME_BASE_Q);
                } else if(dec_ctx->framerate.num != 0 && dec_ctx->framerate.den != 0) {
                    int ticks= av_stream_get_parser(st) ? av_stream_get_parser(st)->repeat_pict+1 : dec_ctx->ticks_per_frame;
                    duration_dts = ((int64_t)AV_TIME_BASE *
                                    dec_ctx->framerate.den * ticks) /
                                    dec_ctx->framerate.num / dec_ctx->ticks_per_frame;
                }

                if (last_decoded_dts != AV_NOPTS_VALUE && duration_dts) {
                    last_decoded_dts += duration_dts;
                } else {
                    last_decoded_dts = AV_NOPTS_VALUE;
                }
            }
            else{
                ret = avcodec_send_packet(dec_ctx, NULL);
            }

            if (ret < 0 && ret != AVERROR_EOF) {
                av_log(NULL, AV_LOG_ERROR, "Error while sending a packet to the decoder\n");
                break;
            }
            while (ret >= 0 || end == 1) {
                ret = avcodec_receive_frame(dec_ctx, frame);
                if (ret == AVERROR(EAGAIN)) {
                    break;
                }
                else if(ret == AVERROR_EOF){
                    end = 2;
                    break;
                }
                else if (ret < 0) {
                    av_log(NULL, AV_LOG_ERROR, "Error while receiving a frame from the decoder\n");
                    goto end;
                }
                frame->pts = frame->best_effort_timestamp;
                if(frame_num %codec_workers->select_frame == 0 && begin_roi == 1)
                {
                    ret = av_frame_ref(frame1, frame);
                    if (ret < 0) {
                        av_frame_free(&frame1);
                        printf("copy frame to filter failed\n");
                    }
                    begin_roi = 0;
                }
                if (frame->pts == AV_NOPTS_VALUE) {
                    if (last_decoded_pts == AV_NOPTS_VALUE) {
                        frame->pts = 0;
                    } else {
#if LIBAVCODEC_VERSION_MAJOR >= 61 //7.0
                        frame->pts = last_decoded_pts + frame->duration;
#else
                        frame->pts = last_decoded_pts + frame->pkt_duration;
#endif
                    }
                    av_log(NULL, AV_LOG_DEBUG, "%s frame->pts == AV_NOPTS_VALUE, adjust to %ld\n", __func__, frame->pts);
                }
                last_decoded_pts = frame->pts;
                if (frame->pts != AV_NOPTS_VALUE) {
                    int64_t start_time =  0;
                    AVRational filter_tb = fmt_ctx->streams[video_stream_index]->time_base;
                    frame->pts =
                        av_rescale_q(frame->pts, filter_tb, enc_ctx->time_base) -
                        av_rescale_q(start_time, AV_TIME_BASE_Q, enc_ctx->time_base);
                }
                frame_num++;
                encode(enc_ctx,frame,enc_packet,outfile);
                av_frame_unref(frame);
            }
        }
        av_packet_unref(&packet);
    }
    end = 3;
end:
    av_frame_free(&frame);
    av_frame_free(&filt_frame);
    avcodec_free_context(&dec_ctx);
    avcodec_free_context(&enc_ctx);
    avformat_close_input(&fmt_ctx);
    fclose(outfile);
    if (ret < 0 && ret != AVERROR_EOF) {
        fprintf(stderr, "Error occurred: %s\n", av_err2str(ret));
        exit(1);
    }

    return 0;
}
