/*
 * Copyright (c) 2010 Nicolas George
 * Copyright (c) 2011 Stefano Sabatini
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * @file
 * API example for transcode and filter
 * @example ni_transcode_filter.c
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>

#include <libavcodec/avcodec.h>
#include <libavformat/avformat.h>
#include <libavfilter/buffersink.h>
#include <libavfilter/buffersrc.h>
#include <libswscale/swscale.h>
#include <libavutil/time.h>
#include <libavutil/opt.h>
#include <libavutil/hwcontext.h>
#include <pthread.h>

static AVFormatContext *ifmt_ctx;
static AVFormatContext *ofmt_ctx;
static AVCodecContext *dec_ctx;
static AVCodecContext *enc_ctx;
AVFilterContext *buffersink_ctx;
AVFilterContext *buffersrc_ctx;
AVFilterGraph *filter_graph;
int video_stream_index = -1;
int encode_num = 0;
int decode_num = 0;
int need_filter = 0;
int init_filter = 0;
int init_enc = 0;

int64_t last_decoded_dts = AV_NOPTS_VALUE;
int64_t last_decoded_pts = AV_NOPTS_VALUE;
void *result = NULL;
/*
end_state can be setted the following value:
0 means all module is running
1 means av_read_frame return eof
2 means decoder return eof
3 means all modules is eof
*/
int end_state = 0;
int64_t timer_start,timer_now;
struct input_info {
    const char *input_file;
    const char *output_file;
    const char *decoder_name;
    const char *decoder_params;
    const char *encoder_name;
    const char *encoder_params;
    const char *reconfig_desc;
    int reconf_num;
    int devid;
    int bitrate;
    int reconf_ppu_num;
    int reconf_scale;
    AVNIPpuConfig ni_ppu_reconfig;
};

struct input_info *input_infos = NULL;

static int init_input_fmt_ctx() {
    int i = 0, ret = 0;
    const char *input_file = input_infos->input_file;
    if ((ret = avformat_open_input(&ifmt_ctx, input_file, NULL, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot open input file %s\n", input_file);
        return ret;
    }
    if ((ret = avformat_find_stream_info(ifmt_ctx, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot find stream information\n");
        return ret;
    }

    for (i = 0; i < ifmt_ctx->nb_streams; i++) {
        if (ifmt_ctx->streams[i]->codecpar->codec_type == AVMEDIA_TYPE_VIDEO) {
            video_stream_index = i;
            break;
        }
    }
    if (video_stream_index < 0) {
        av_log(NULL, AV_LOG_ERROR, "can not find video stream\n");
        return video_stream_index;
    }
    if (strstr(input_infos->input_file, "rtsp")) {
        av_opt_set(ifmt_ctx->priv_data, "rtsp_transport", "tcp", 0);
    }
    return 0;
}

static int open_input_file() {
    int ret;
    ret = init_input_fmt_ctx();
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR,"fail to open file\n");
        return ret;
    }

    AVCodec *dec;
    dec = avcodec_find_decoder_by_name(input_infos->decoder_name);
    if (dec) {
        if (ifmt_ctx->streams[video_stream_index]->codecpar->codec_id != dec->id) {
            av_log(NULL, AV_LOG_ERROR, "codec does not match with stream id\n");
            return AVERROR_DECODER_NOT_FOUND;
        }
    }

    /* create decoding context */
    dec_ctx = avcodec_alloc_context3(dec);
    if (!dec_ctx){
        av_log(NULL, AV_LOG_ERROR, "Failed to allocate the decoder context for stream\n");
        return AVERROR(ENOMEM);
    }
    avcodec_parameters_to_context(dec_ctx, ifmt_ctx->streams[video_stream_index]->codecpar);
    if(input_infos->devid >= 0)
    {
        char str_devid[4] = {0};
        snprintf(str_devid, sizeof(str_devid), "%d", input_infos->devid);
        av_opt_set(dec_ctx->priv_data, "dec", str_devid, 0);
    }
    if(input_infos->decoder_params)
        av_opt_set(dec_ctx->priv_data, "xcoder-params", input_infos->decoder_params, 0);

    dec_ctx->framerate = av_guess_frame_rate(ifmt_ctx, ifmt_ctx->streams[video_stream_index], NULL);

    ret = avcodec_parameters_from_context(ifmt_ctx->streams[video_stream_index]->codecpar, dec_ctx);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to copy decoder parameters from codec context for stream #");
        return ret;
    }

    /* init the video decoder */
    if ((ret = avcodec_open2(dec_ctx, dec, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot open video decoder\n");
        return ret;
    }
    return 0;
}

static int init_output_fmt_ctx()
{
    int i = 0, ret = 0;
    const char *output_file = input_infos->output_file;
    if (!strcmp(output_file, "null")) {
        avformat_alloc_output_context2(&ofmt_ctx, NULL, "null", NULL);
    } else if (strstr(output_file, "rtmp")){
        avformat_alloc_output_context2(&ofmt_ctx, NULL, "flv", output_file);
    } else if (strstr(output_file, "rtsp")){
        avformat_alloc_output_context2(&ofmt_ctx, NULL, "rtsp", output_file);
    } else {
        avformat_alloc_output_context2(&ofmt_ctx, NULL, NULL, output_file);
    }
    if (strstr(output_file, "rtsp")) {
        av_opt_set(ofmt_ctx->priv_data, "rtsp_transport", "tcp", 0);
    }

    for (i = 0; i < ifmt_ctx->nb_streams; i++)
    {
        AVStream *in_stream = ifmt_ctx->streams[i];
        AVCodecParameters *in_codecpar = in_stream->codecpar;
        // only transcode for video stream
        if (in_codecpar->codec_type != AVMEDIA_TYPE_VIDEO) {
            continue;
        }
        AVStream *avs = avformat_new_stream(ofmt_ctx, NULL);
        if (!avs) {
            av_log(NULL, AV_LOG_ERROR,"Failed allocating output stream\n");
            return -1;
        }
        ret = avcodec_parameters_from_context(avs->codecpar, enc_ctx);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR,"Failed to copy codec parameters\n");
            return ret;
        }
    }

    if (!(ofmt_ctx->oformat->flags & AVFMT_NOFILE)) {
        ret = avio_open(&ofmt_ctx->pb, output_file, AVIO_FLAG_WRITE);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "Could not open output file '%s'", output_file);
            return ret;
        }
    }

    /* init muxer, write output file header */
    ret = avformat_write_header(ofmt_ctx, NULL);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Error occurred when opening output file\n");
        return ret;
    }
    return 0;
}

static int open_output_file(AVFrame *frame) {
    int ret = 0;
    AVCodec *enc;

    // Find video encoder codec selected
    enc = avcodec_find_encoder_by_name(input_infos->encoder_name);
    if (!enc)
    {
        av_log(NULL, AV_LOG_ERROR,"Codec '%s' not found\n", input_infos->encoder_name);
        return AVERROR_ENCODER_NOT_FOUND;
    }

    // Allocate codec context for encoding
    enc_ctx = avcodec_alloc_context3(enc);
    if (!enc_ctx)
    {
        av_log(NULL, AV_LOG_ERROR, "Could not allocate video codec context\n");
        return AVERROR(ENOMEM);
    }
    if (input_infos->bitrate)
    {
        enc_ctx->bit_rate = input_infos->bitrate;
    }
    enc_ctx->height = frame->height;
    enc_ctx->width = frame->width;
    enc_ctx->sample_aspect_ratio = frame->sample_aspect_ratio;
    enc_ctx->pix_fmt = frame->format;
    if (frame->hw_frames_ctx)
    {
        AVHWFramesContext* pAVHFWCtx = (AVHWFramesContext *)frame->hw_frames_ctx->data;
        enc_ctx->sw_pix_fmt = pAVHFWCtx->sw_format;
    }

    /* video time_base can be set to whatever is handy and supported by encoder */
    enc_ctx->time_base = av_inv_q(dec_ctx->framerate);
    char enc_param[256] = {0};
    if (input_infos->encoder_params) {
        snprintf(enc_param, sizeof(enc_param), "%s", input_infos->encoder_params);
        if (strstr(input_infos->output_file, ".mkv")) {
            snprintf(enc_param + strlen(enc_param), sizeof(enc_param) - strlen(enc_param), "%s", ":GenHdrs=1");
        }
    } else if (strstr(input_infos->output_file, ".mkv")) {
        snprintf(enc_param, sizeof(enc_param), "%s", "GenHdrs=1");
    }

    if ((enc->id == AV_CODEC_ID_H264) || (enc->id == AV_CODEC_ID_H265)
#ifdef AV1_SUPPORTED
        || (enc->id == AV_CODEC_ID_AV1)
#endif
       )
    {
        if(strlen(enc_param))
        {
            av_opt_set(enc_ctx->priv_data, "xcoder-params", enc_param, 0);
        }
        if(input_infos->devid >= 0)
        {
            char str_devid[4] = {0};
            snprintf(str_devid, sizeof(str_devid), "%d", input_infos->devid);
            av_opt_set(enc_ctx->priv_data, "enc", str_devid, 0);
        }
    }
    else
    {
        av_log(NULL, AV_LOG_ERROR, "codec id %d not supported.\n", enc->id);
        ret = -1;
        return ret;
    }

    // Open encoder
    ret = avcodec_open2(enc_ctx, enc, NULL);
    if (ret < 0)
    {
        av_log(NULL, AV_LOG_ERROR, "Could not open enc\n");
        return ret;
    }
    ret = init_output_fmt_ctx();
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR,"fail to open file\n");
        return ret;
    }
    if (ofmt_ctx->oformat->flags & AVFMT_GLOBALHEADER)
        enc_ctx->flags |= AV_CODEC_FLAG_GLOBAL_HEADER;

    return ret;
}

static int init_filters(const char *filters_desc)
{
    char args[512] = { 0 };
    char name[32] = { 0 };
    int i, ret = 0;
    AVFilterInOut *inputs, *outputs, *cur;
    AVBufferSrcParameters *par;
    AVRational time_base = ifmt_ctx->streams[video_stream_index]->time_base;

    // enum AVPixelFormat pix_fmts[] = {AV_PIX_FMT_NI_QUAD,AV_PIX_FMT_BGRA,AV_PIX_FMT_BGRP,AV_PIX_FMT_YUV420P,AV_PIX_FMT_NONE};

    filter_graph = avfilter_graph_alloc();
    if (filter_graph == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate filter graph\n");
        goto end;
    }

    ret = avfilter_graph_parse2(filter_graph, filters_desc, &inputs, &outputs);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to parse graph\n");
        goto end;
    }

    // link input
    cur = inputs, i = 0;
    snprintf(name, sizeof(name), "in_%d", i);
    snprintf(args, sizeof(args),
            "video_size=%dx%d:pix_fmt=%d:time_base=%d/%d:pixel_aspect=%d/%d:frame_rate=%d/%d",
            dec_ctx->width, dec_ctx->height, dec_ctx->pix_fmt,time_base.num, time_base.den,
            dec_ctx->sample_aspect_ratio.num, dec_ctx->sample_aspect_ratio.den,dec_ctx->framerate.num,dec_ctx->framerate.den);
    ret = avfilter_graph_create_filter(&buffersrc_ctx, avfilter_get_by_name("buffer"), name,
                                    args, NULL, filter_graph);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot create buffer source\n");
        goto end;
    }

    // decoder out=hw
    if (dec_ctx->hw_frames_ctx != NULL) {
        av_log(NULL, AV_LOG_INFO, "hw mode filter\n");
        // Allocate a new AVBufferSrcParameters instance when decoder out=hw
        par = av_buffersrc_parameters_alloc();
        if (!par) {
            ret = AVERROR(ENOMEM);
            goto end;
        }
        memset(par, 0, sizeof(*par));
        // set format and hw_frames_ctx to AVBufferSrcParameters when out=hw
        par->format = AV_PIX_FMT_NONE;
        par->hw_frames_ctx = dec_ctx->hw_frames_ctx;
#if LIBAVCODEC_VERSION_MAJOR  >= 61
        par->color_range = dec_ctx->color_range;
        par->color_space = dec_ctx->colorspace;
#endif
        av_log(NULL, AV_LOG_DEBUG, "hw_frames_ctx %p\n", par->hw_frames_ctx);
        // Initialize the buffersrc filter with the provided parameters
        ret = av_buffersrc_parameters_set(buffersrc_ctx, par);
        av_freep(&par);
        if (ret < 0)
            goto end;
    } else { // decoder out=sw
        av_log(NULL, AV_LOG_INFO, "sw mode filter\n");
    }

    ret = avfilter_link(buffersrc_ctx, 0, cur->filter_ctx, cur->pad_idx);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to link input filter\n");
        goto end;
    }

    cur = outputs, i = 0;
    snprintf(name, sizeof(name), "out_%d", i);
    ret = avfilter_graph_create_filter(&buffersink_ctx, avfilter_get_by_name("buffersink"),
                                        name, NULL, NULL, filter_graph);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to create output filter: %d\n", i);
        goto end;
    }

    // connect  dst (index i) pads to one of buffer sink (index 0) pad
    ret = avfilter_link(cur->filter_ctx, cur->pad_idx, buffersink_ctx, 0);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to link output filter: %d\n", i);
        goto end;
    }

    // configure and validate the filter graph
    ret = avfilter_graph_config(filter_graph, NULL);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "%s failed to config graph filter\n",
               __func__);
        goto end;
    } else {
        av_log(NULL, AV_LOG_DEBUG, "%s success config graph filter %s\n",
               __func__, filters_desc);
    }

end:
    avfilter_inout_free(&inputs);
    avfilter_inout_free(&outputs);
    return ret;
}

int reconfig_drawtext_filter(const char *reconfig_desc) {
    int i, ret;
    AVFilterContext *drawtext_filter_ctx = NULL;
    for (i = 0; i < filter_graph->nb_filters; i++) {
        if (strstr(filter_graph->filters[i]->name, "ni_quadra_drawtext") != NULL) {
            drawtext_filter_ctx = filter_graph->filters[i];
            break;
        }
    }
    if (drawtext_filter_ctx) {
        ret = avfilter_graph_send_command(filter_graph, "ni_quadra_drawtext", "reinit", reconfig_desc, NULL, 0, 0);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "cannot send drawtext filter command %d\n", ret);
            return -1;
        }
    } else {
        av_log(NULL, AV_LOG_ERROR, "not find drawtext filter\n");
    }
    return 0;
}

/* Send one frame to encoder, attempt to read all available encoder packets from encoder,
   writing packets to output stream. If run with with *frame=NULL, flush the encoder of
   remaining packets until encoder reports EOF */
static int encode_frame(AVFrame *frame, AVPacket *pkt)
{
    int ret;
    if (!init_enc && frame)
    {
        if((ret = open_output_file(frame)) < 0)
        {
            av_log(NULL, AV_LOG_ERROR, "init encoder failed ret = %d\n", ret);
            return ret;
        }
        init_enc = 1;
    }

    int attempt_cnt = 1;
    int re_send = 0;
    static int send_cnt = 0;
    static int recv_cnt = 0;

    // Send frame to encoder. If frame=NULL, this signals end of stream
    if (frame)
    {
        frame->pict_type = AV_PICTURE_TYPE_NONE;
        av_log(NULL, AV_LOG_DEBUG, "Send %d frame %3"PRId64", type=%d\n", ++send_cnt, frame->pts, frame->pict_type);
    } else
    {
        av_log(NULL, AV_LOG_DEBUG,"Send done, flush encoder\n");
    }

resend:
    ret = avcodec_send_frame(enc_ctx, frame);
    // Encoder is too busy to receive frame, try sending frame again
    if (ret == AVERROR(EAGAIN))
    {
        av_log(NULL, AV_LOG_DEBUG, "Resending frame. Try #%d.\n", ++attempt_cnt);
        ret = 0;
        re_send = 1;
    }
    // Frame send issue occured
    else if (ret < 0)
    {
        av_log(NULL, AV_LOG_ERROR, "Error sending frame for encoding. RC=%d\n", AVUNERROR(ret));
        return ret;
    }

    /* Run this loop if a frame was sucessfully sent during avcodec_send_frame() OR
        no frame was sent because this function is in flushing mode OR
        a frame was sucessfully received by avcodec_receive_packet() */
    while (ret >= 0 || ! frame )
    {
        av_log(NULL, AV_LOG_DEBUG, "Receiving packet\n");
        ret = avcodec_receive_packet(enc_ctx, pkt);
        if (frame == NULL)
            av_log(NULL, AV_LOG_DEBUG, "Draining packet ret=%x, pkt size=%5d\n", ret, pkt->size);

        // No packets in encoder output buffer but not end of stream either
        if (ret == AVERROR(EAGAIN))
        {
            // If function not in flushing mode
            if (frame)
            {
                if (re_send)
                {
                    re_send = 0;
                    goto resend;
                }
                return 0; // Exit this function to send next packet
            }
            else
            {
                continue; // Loop in this function to flush encoder
            }
        }
        // End of stream detected
        else if (ret == AVERROR_EOF)
        {
            av_log(NULL, AV_LOG_DEBUG, "Encoder EOF reached\n");
            return 0;
        }
        // Packet receive error occured
        else if (ret < 0)
        {
            av_log(NULL, AV_LOG_ERROR, "Error during encoding\n");
            return ret;
        }
        // Packet sucessfully received
        av_log(NULL, AV_LOG_DEBUG,"mux receive packet %d, pts=%"PRId64"\n", ++recv_cnt, pkt->pts);
        encode_num++;
        pkt->pos = -1;
        av_packet_rescale_ts(pkt, ifmt_ctx->streams[video_stream_index]->time_base,
                             ofmt_ctx->streams[0]->time_base);
        av_log(NULL, AV_LOG_DEBUG,"mux write packet %d, pts=%"PRId64"\n", recv_cnt, pkt->pts);
        ret = av_interleaved_write_frame(ofmt_ctx, pkt);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "Error during mux\n");
            return ret;
        }
        av_packet_unref(pkt);
    }
    return 0;
}

static void print_report(int is_last_report,
        int64_t timer_start, int64_t cur_time,
        unsigned long frame_number)
{
    static int64_t last_time = -1;
    float t;
    char buf[1024];
    float fps;

    if (!is_last_report) {
        if (last_time == -1) {
            last_time = cur_time;
            return;
        }
        if ((cur_time - last_time) < 500000)
            return;
        last_time = cur_time;
    }

    t = (cur_time - timer_start) / 1000000.0;

    fps = t > 1 ? frame_number / t : 0;

    const char end = is_last_report ? '\n' : '\r';

    buf[0] = '\0';
    snprintf(buf + strlen(buf), sizeof(buf) - strlen(buf), "frame=%5lu fps=%3.*f ",
            frame_number, (fps < 9.95), fps);
    fprintf(stderr, "%s   %c", buf, end);
    fflush(stderr);
}


static void *printfps(void *arg) {
    while (1) {
        if (end_state == 3) {
            break;
        }
        usleep(100);
        print_report(0, timer_start, av_gettime_relative(), encode_num);
    }
    print_report(1, timer_start, av_gettime_relative(), encode_num);
    return NULL;
}

static void help_usage(void) {
    printf("Usage: \n"
            "-i | --input             input file path.\n"
            "-d | --decoder           decoder name [h264_ni_quadra_dec | h265_ni_quadra_dec].\n"
            "-p | --decoder_params    decoder parameters.\n"
            "-o | --output_file       output file path.\n"
            "-e | --encoder           encoder name [h264_ni_quadra_enc | h265_ni_quadra_enc].\n"
            "-x | --encode_params     encoder parameters.\n"
            "-f | --filter_desc       filter description.\n"
            "-m | --reconf_num        the frame num reconfig happen.\n"
            "-s | --reconfig_desc     reconfig filter description.\n"
            "-b | --bitrate           set bitrate.\n"
            "-n | --devid             device id.\n"
            "-v | --loglevel          available debug level: warning, info, debug, trace.\n"
            "-a | --reconf_ppu_num    the packet num to reconfig ppu resolution.\n"
            "-t | --ppu0_resolution   the ppu0 output resolution [width]x[height].\n"
            "-r | --ppu1_resolution   the ppu1 output resolution [width]x[height].\n"
            "-u | --ppu_reconfig_mode the ppu reconfig mode [0 | 1]\n"
            "-h | --help              print this help information.\n");
}

int setup_loglevel(char *loglevel) {
    int i;
    const struct { const char *name; int level; } log_levels[] =
    {
         { "quiet"  , AV_LOG_QUIET   },
         { "panic"  , AV_LOG_PANIC   },
         { "fatal"  , AV_LOG_FATAL   },
         { "error"  , AV_LOG_ERROR   },
         { "warning", AV_LOG_WARNING },
         { "info"   , AV_LOG_INFO    },
         { "verbose", AV_LOG_VERBOSE },
         { "debug"  , AV_LOG_DEBUG   },
         { "trace"  , AV_LOG_TRACE   },
    };

    for (i = 0; i < (sizeof(log_levels) / sizeof((log_levels)[0])); i++) {
        if (!strcmp(log_levels[i].name, loglevel)) {
            av_log_set_level(log_levels[i].level);
            return 0;
        }
    }

    return -1;
}

int main(int argc, char **argv) {
    int ret;
    AVPacket *packet;
    AVPacket *enc_packet;
    AVFrame *frame;
    AVFrame *filt_frame;
    char *loglevel = "info";
    int opt;
    int opt_index;
    char *filter_dsc = NULL;
    pthread_t pthtimeid;
    void *result = NULL;
    int fps_thread_state = 0;
    int saw_first_ts = 0;
    AVStream *st = NULL;
    int64_t duration_dts = 0;
    input_infos = malloc(sizeof(struct input_info));
    int orig_width = 0, orig_height = 0;
    char *n;
    int input_pkt_num = 0;
    int pkt_need_resend = 0;
    AVPacket *buffered_pkt = NULL;
    AVFrameSideData *side_data;
    if (input_infos == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate codec workers.\n");
        ret = EXIT_FAILURE;
        goto end;
    }

    input_infos->input_file = NULL;
    input_infos->output_file = NULL;
    input_infos->decoder_name = NULL;
    input_infos->decoder_params = NULL;
    input_infos->encoder_name = NULL;
    input_infos->encoder_params = NULL;
    input_infos->reconfig_desc = NULL;
    input_infos->reconf_num = 0;
    input_infos->bitrate = 0;
    input_infos->devid = -1;
    input_infos->reconf_ppu_num = 0;
    memset(&input_infos->ni_ppu_reconfig, 0, sizeof(AVNIPpuConfig));

    const char *opt_string = "i:d:p:o:e:x:f:c:m:s:b:n:v:a:r:t:u:h";
    static struct option long_options[] = {
        {"input",          required_argument, NULL, 'i'},
        {"decoder",        required_argument, NULL, 'd'},
        {"decoder_params", required_argument, NULL, 'p'},
        {"output_file",    required_argument, NULL, 'o'},
        {"encoder",        required_argument, NULL, 'e'},
        {"encode_params",  required_argument, NULL, 'x'},
        {"filter_desc",    required_argument, NULL, 'f'},
        {"reconf_scale",   required_argument, NULL, 'c'},
        {"reconf_num",     required_argument, NULL, 'm'},
        {"reconfig_desc",  required_argument, NULL, 's'},
        {"bitrate",        required_argument, NULL, 'b'},
        {"devid",          required_argument, NULL, 'n'},
        {"loglevel",       required_argument, NULL, 'v'},
        {"reconf_ppu_num", required_argument, NULL, 'a'},
        {"ppu0_resolution", required_argument, NULL, 't'},
        {"ppu1_resolution", required_argument, NULL, 'r'},
        {"ppu_reconfig_mode", required_argument, NULL, 'u'},
        {"help",           no_argument,       NULL, 'h'},
        { NULL,            0,                 NULL,  0 },
    };

    while ((opt = getopt_long(argc, argv, opt_string, long_options, &opt_index)) != -1) {
        switch (opt) {
            case 'i':
                input_infos->input_file = optarg;
                break;
            case 'd':
                input_infos->decoder_name = optarg;
                break;
            case 'p':
                input_infos->decoder_params = optarg;
                break;
            case 'o':
                input_infos->output_file = optarg;
                break;
            case 'e':
                input_infos->encoder_name = optarg;
                break;
            case 'x':
                input_infos->encoder_params = optarg;
                break;
            case 'f':
                filter_dsc = optarg;
                break;
            case 'c':
                input_infos->reconf_scale = atoi(optarg);
                break;
            case 'm':
                input_infos->reconf_num = atoi(optarg);
                break;
            case 's':
                input_infos->reconfig_desc = optarg;
            case 'b':
                input_infos->bitrate = atoi(optarg);
                break;
            case 'n':
                input_infos->devid = atoi(optarg);
                break;
            case 'v':
                loglevel = optarg;
                break;
            case 'a':
                input_infos->reconf_ppu_num = atoi(optarg);
                break;
            case 't':
                input_infos->ni_ppu_reconfig.ppu_w[0] = strtoul(optarg, &n, 10);
                if (*n != 'x') {
                    help_usage();
                    return -1;
                }
                input_infos->ni_ppu_reconfig.ppu_h[0] = strtoul(n + 1, NULL, 10);
                input_infos->ni_ppu_reconfig.out_enabled[0] = 1;
                break;
            case 'r':
                input_infos->ni_ppu_reconfig.ppu_w[1] = strtoul(optarg, &n, 10);
                if (*n != 'x') {
                    help_usage();
                    return -1;
                }
                input_infos->ni_ppu_reconfig.ppu_h[1] = strtoul(n + 1, NULL, 10);
                input_infos->ni_ppu_reconfig.out_enabled[1] = 1;
                break;
            case 'u':
                input_infos->ni_ppu_reconfig.ppu_reconfig_mode = atoi(optarg);
                if (input_infos->ni_ppu_reconfig.ppu_reconfig_mode != 0 &&
                    input_infos->ni_ppu_reconfig.ppu_reconfig_mode != 1)
                {
                    help_usage();
                    return -1;
                }
                break;
            case 'h':
                help_usage();
                return EXIT_SUCCESS;
            default:
                help_usage();
                return EXIT_FAILURE;
        }
    }

    if(setup_loglevel(loglevel) < 0) {
        av_log(NULL, AV_LOG_ERROR, "set log level failed.\n");
        return EXIT_FAILURE;
    }

    if (input_infos->input_file == NULL) {
        av_log(NULL, AV_LOG_ERROR, "input file name must be specified.\n");
        return EXIT_FAILURE;
    }

    if (input_infos->output_file == NULL) {
        av_log(NULL, AV_LOG_ERROR, "output file name must be specified.\n");
        return EXIT_FAILURE;
    }

    if (!input_infos->decoder_name || !input_infos->encoder_name) {
        av_log(NULL, AV_LOG_ERROR, "decoder_name name must be specified.\n");
        return EXIT_FAILURE;
    }

    if(!filter_dsc) {
        av_log(NULL, AV_LOG_INFO, "filter description not be specified.\n");
        need_filter = 0;
    } else {
        need_filter = 1;
    }

    // Register all formats, codecs, filters
#if LIBAVFORMAT_VERSION_MAJOR < 58
    av_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif
#if LIBAVCODEC_VERSION_MAJOR < 58
    avcodec_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif
#if LIBAVFILTER_VERSION_MAJOR < 7
    avfilter_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif

    // Allocate packet for encoder output
    packet = av_packet_alloc();
    buffered_pkt = av_packet_alloc();
    enc_packet = av_packet_alloc();
    frame = av_frame_alloc();
    filt_frame = av_frame_alloc();
    if (!packet || !enc_packet || !frame || !filt_frame) {
        av_log(NULL, AV_LOG_ERROR,"Could not allocate pkt frame\n");
        ret = EXIT_FAILURE;
        goto end;
    }

    if ((ret = open_input_file()) < 0)
        goto end;


    timer_start = av_gettime_relative();
    if (pthread_create(&pthtimeid, NULL, printfps, 0) != 0)
    {
        av_log(NULL, AV_LOG_ERROR,"create thread failed!\n");
        goto end;
    }
    fps_thread_state = 1;

    orig_width = dec_ctx->width;
    orig_height = dec_ctx->height;

    /* read all packets */
    while (1) {
        if (! pkt_need_resend) {
          if ((ret = av_read_frame(ifmt_ctx, packet)) < 0) {
            if(end_state == 2) {
              ret = encode_frame(NULL, enc_packet);
              break;
            }
            end_state = 1;
          }
        } else {
          av_packet_ref (packet, buffered_pkt);
          av_packet_unref (buffered_pkt);
        }

        if (packet->stream_index == video_stream_index || end_state == 1) {
            if(! end_state || pkt_need_resend) {
                input_pkt_num++;
                if (!saw_first_ts) {
                    st = ifmt_ctx->streams[video_stream_index];
                    last_decoded_dts = st->avg_frame_rate.num ? - dec_ctx->has_b_frames * AV_TIME_BASE / av_q2d(st->avg_frame_rate) : 0;
                    if (packet && packet->pts != AV_NOPTS_VALUE) {
                        last_decoded_dts += av_rescale_q(packet->pts, st->time_base, AV_TIME_BASE_Q);
                    }
                    saw_first_ts = 1;
                }

                if (input_infos->reconf_ppu_num == input_pkt_num)
                {
                    av_log(NULL, AV_LOG_INFO, "Need add sei info\n");
                    uint8_t *side_data = NULL;
                    side_data = av_packet_new_side_data(packet, AV_PKT_DATA_PPU_CONFIG, sizeof(AVNIPpuConfig));
                    if (!side_data)
                    {
                        av_log(NULL, AV_LOG_ERROR, "av_packet_new_side_data failed\n");
                        goto end;
                    }
                    memcpy(side_data, &input_infos->ni_ppu_reconfig, sizeof(AVNIPpuConfig));
                }
                // set pkt dts here
                if (last_decoded_dts != AV_NOPTS_VALUE) {
                    packet->dts = av_rescale_q(last_decoded_dts, AV_TIME_BASE_Q, st->time_base);
                    av_log(NULL, AV_LOG_DEBUG, "send pkt dts %ld lastdts %ld\n", packet->dts, last_decoded_dts);
                }
                ret = avcodec_send_packet(dec_ctx, packet);

                // update the last_decoded_dts
                if (packet && packet->duration) {
                    duration_dts = av_rescale_q(packet->duration, st->time_base, AV_TIME_BASE_Q);
                } else if(dec_ctx->framerate.num != 0 && dec_ctx->framerate.den != 0) {
                    int ticks= av_stream_get_parser(st) ? av_stream_get_parser(st)->repeat_pict+1 : dec_ctx->ticks_per_frame;
                    duration_dts = ((int64_t)AV_TIME_BASE *
                                    dec_ctx->framerate.den * ticks) /
                                    dec_ctx->framerate.num / dec_ctx->ticks_per_frame;
                }

                if (last_decoded_dts != AV_NOPTS_VALUE && duration_dts) {
                    last_decoded_dts += duration_dts;
                } else {
                    last_decoded_dts = AV_NOPTS_VALUE;
                }
            }
            else {
                av_log(NULL, AV_LOG_DEBUG, "send null pkt to flush\n");
                ret = avcodec_send_packet(dec_ctx, NULL);
            }

            if (ret < 0 && ret != AVERROR_EOF) {
                if (ret == AVERROR(EAGAIN)) {
                  pkt_need_resend = 1;
                  av_packet_ref (buffered_pkt, packet);
                } else {
                  av_log(NULL, AV_LOG_ERROR, "Error while sending a packet to the decoder\n");
                  break;
                }
            } else {
              if (pkt_need_resend) {
                pkt_need_resend = 0;
              }
            }

            int try_recv_frame = 0;
            if (pkt_need_resend) {
              try_recv_frame = 1;
            }

            while (ret >= 0 || end_state == 1 || try_recv_frame) {
                try_recv_frame = 0;
                ret = avcodec_receive_frame(dec_ctx, frame);
                if (ret == AVERROR(EAGAIN)) {
                    break;
                }
                else if(ret == AVERROR_EOF) {
                    end_state = 2;
                    break;
                }
                else if (ret < 0) {
                    av_log(NULL, AV_LOG_ERROR, "Error while receiving a frame from the decoder\n");
                    goto end;
                }
                decode_num++;
                frame->pts = frame->best_effort_timestamp;
                if (frame->pts == AV_NOPTS_VALUE) {
                    if (last_decoded_pts == AV_NOPTS_VALUE) {
                        frame->pts = 0;
                    } else {
#if LIBAVCODEC_VERSION_MAJOR >= 61 //7.0
                        frame->pts = last_decoded_pts + frame->duration;
#else
                        frame->pts = last_decoded_pts + frame->pkt_duration;
#endif
                    }
                    av_log(NULL, AV_LOG_DEBUG, "%s frame->pts == AV_NOPTS_VALUE, adjust to %ld\n", __func__, frame->pts);
                }
                last_decoded_pts = frame->pts;
                if (frame)
                {
                    side_data = av_frame_get_side_data(frame, AV_FRAME_DATA_NETINT_ERROR_RATIO);
                    if (side_data && side_data->size == sizeof(uint32_t))
                    {
                        uint32_t error_ratio = *((uint32_t *)side_data->data);
                        av_log(NULL, AV_LOG_WARNING, "Warning: decoded %d frame has error ratio %lf\% \n",
                                        (float)error_ratio/100);
                    }
                }

                if (need_filter) {
                    if (orig_width != frame->width || orig_height != frame->height)
                    {
                        av_log(NULL, AV_LOG_INFO, "Input frame resolution changed "
                                    "from %dx%d to %dx%d So reopen fiters now\n",
                                    orig_width, orig_height, frame->width, frame->height);
                        avfilter_graph_free(&filter_graph);
                        init_filter = 0;
                        orig_width = frame->width;
                        orig_height = frame->height;
                    }

                    if (input_infos->reconf_num && decode_num == input_infos->reconf_num &&
                        input_infos->reconfig_desc && input_infos->reconf_scale) {
                        av_log(NULL, AV_LOG_INFO, "reconfig scale filter\n");
                        if (init_filter) {
                          avfilter_graph_free(&filter_graph);
                          init_filter = 0;
                        }
                        filter_dsc = input_infos->reconfig_desc;
                        orig_width = frame->width;
                        orig_height = frame->height;
                    }

                    if (!init_filter) {
                        if ((ret = init_filters(filter_dsc)) < 0) {
                            av_log(NULL, AV_LOG_ERROR, "init filter failed ret = %d\n", ret);
                            goto end;
                        }
                        init_filter = 1;
                    }

                    if (! input_infos->reconf_scale && input_infos->reconf_num && decode_num == input_infos->reconf_num) {
                        if (reconfig_drawtext_filter(input_infos->reconfig_desc)) {
                            av_log(NULL, AV_LOG_ERROR, "reconfig filter failed\n");
                            goto end;
                        }
                    }
                    if ((ret = av_buffersrc_add_frame_flags(buffersrc_ctx, frame, AV_BUFFERSRC_FLAG_KEEP_REF)) < 0) {
                        av_log(NULL, AV_LOG_ERROR, "Error while feeding the filtergraph\n");
                        break;
                    }

                    /* pull filtered frames from the filtergraph */
                    while (1) {
                        ret = av_buffersink_get_frame(buffersink_ctx, filt_frame);
                        if (ret == AVERROR(EAGAIN)) {
                            break;
                        }
                        else if(ret == AVERROR_EOF) {
                            break;
                        }
                        else if (ret < 0) {
                            av_log(NULL, AV_LOG_ERROR, "av_buffersink_get_frame failed ret = %d\n", ret);
                            goto end;
                        }
                        ret = encode_frame(filt_frame, enc_packet);
                        av_frame_unref(filt_frame);
                        if (ret < 0) {
                            av_log(NULL, AV_LOG_ERROR, "encode frame failed ret = %d\n", ret);
                            goto end;
                        }
                    }
                } else {
                    ret = encode_frame(frame, enc_packet);
                    if (ret < 0) {
                        av_log(NULL, AV_LOG_ERROR, "encode frame failed ret = %d\n", ret);
                        goto end;
                    }
                }
                av_frame_unref(frame);
            }
        }
        av_packet_unref(packet);
    }
end:
    end_state = 3;
    if (fps_thread_state) {
        if (pthread_join(pthtimeid, &result) == 0) {
            if ((long)result != 0) {
                av_log(NULL, AV_LOG_INFO, "pthread_join fps thread ret %ld\n",
                    (long)result);
                ret = EXIT_FAILURE;
            }
        }
    }

    if (ofmt_ctx) {
        av_write_trailer(ofmt_ctx);
        if (!(ofmt_ctx->oformat->flags & AVFMT_NOFILE)) {
            avio_closep(&ofmt_ctx->pb);
        }
        avformat_free_context(ofmt_ctx);
    }
    avformat_close_input(&ifmt_ctx);
    avfilter_graph_free(&filter_graph);
    avcodec_free_context(&dec_ctx);
    avcodec_free_context(&enc_ctx);
    av_frame_free(&frame);
    av_frame_free(&filt_frame);
    av_packet_free(&packet);
    av_packet_free(&buffered_pkt);
    av_packet_free(&enc_packet);
    free (input_infos);
    if (ret < 0 && ret != AVERROR_EOF) {
        av_log(NULL, AV_LOG_ERROR, "Error occurred: %s\n", av_err2str(ret));
        return EXIT_FAILURE;
    }

    return 0;
}
