/*******************************************************************************
 *
 * Copyright (C) 2022 NETINT Technologies
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 ******************************************************************************/

/*!*****************************************************************************
 *  \file   ni_test_utils.c
 *
 *  \brief  Commonly used functions for performing transcoding, hw upload,
 *          hw dowload processing with libxcoder API.
 *          Its code provides examples on how to programatically use libxcoder
 *          API.
 ******************************************************************************/

#include <math.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/stat.h>

#include "ni_test_utils.h"

#if defined(LRETURN)
#undef LRETURN
#define LRETURN goto end;
#undef END
#define END                                                                    \
    end:
#else
#define LRETURN goto end;
#define END                                                                    \
    end:
#endif

#define ALIGN(x, a) (((x)+(a)-1)&~((a)-1))

typedef struct gc620_pixel_fmts {
    ni_pix_fmt_t pix_fmt_ni;
    int pix_fmt_gc620;
} gc620_pixel_fmts_t;

static struct gc620_pixel_fmts gc620_pixel_fmt_list[] = {
    {NI_PIX_FMT_NV12, GC620_NV12},       {NI_PIX_FMT_YUV420P, GC620_I420},
    {NI_PIX_FMT_P010LE, GC620_P010_MSB}, {NI_PIX_FMT_YUV420P10LE, GC620_I010},
    {NI_PIX_FMT_YUYV422, GC620_YUYV},    {NI_PIX_FMT_UYVY422, GC620_UYVY},
    {NI_PIX_FMT_NV16, GC620_NV16},       {NI_PIX_FMT_RGBA, GC620_RGBA8888},
    {NI_PIX_FMT_BGR0, GC620_BGRX8888},   {NI_PIX_FMT_BGRA, GC620_BGRA8888},
    {NI_PIX_FMT_ABGR, GC620_ABGR8888},   {NI_PIX_FMT_ARGB, GC620_ARGB8888}};

enum var_name {
    VAR_IN_W,
    VAR_IW,
    VAR_IN_H,
    VAR_IH,
    VAR_OUT_W,
    VAR_OW,
    VAR_OUT_H,
    VAR_OH,
    VAR_X,
    VAR_Y,
    VAR_VARS_NB
};
static const char *skip_whitespace(const char *expr);
static double parse_number(const char *expr, char **endPtr);
static double parse_factor(const char *expr, char **endPtr);
static double parse_term(const char *expr, char **endPtr);
static double parse_expression(const char *expr, char **endPtr);
static double ni_expr_parse_and_eval(const char *expression);

// Function to skip whitespace characters
static const char *skip_whitespace(const char *expr) {
    while (isspace(*expr))
        expr++;
    return expr;
}

// Function to parse a number from the expression
static double parse_number(const char *expr, char **endPtr) {
    double number = strtod(expr, endPtr);
    if (expr == *endPtr) {
        ni_log(NI_LOG_ERROR, "Error: Invalid number\n");
        exit(1);
    }
    return number;
}

// Function to parse a factor (number or subexpression)
static double parse_factor(const char *expr, char **endPtr) {
    expr = skip_whitespace(expr);
    double result;

    if (*expr == '(') {
        expr++;
        result = parse_expression(expr, endPtr);
        if (**endPtr != ')') {
            ni_log(NI_LOG_ERROR, "Error: Missing closing parenthesis\n");
            exit(1);
        }
        (*endPtr)++;
    } else {
        result = parse_number(expr, endPtr);
    }

    return result;
}

// Function to parse a term (multiplication or division)
static double parse_term(const char *expr, char **endPtr) {
    double result = parse_factor(expr, endPtr);
    expr = skip_whitespace(*endPtr);

    while (*expr == '*' || *expr == '/') {
        char op = *expr;
        expr++;
        double factor = parse_factor(expr, endPtr);
        expr = skip_whitespace(*endPtr);

        if (op == '*')
            result *= factor;
        else if (op == '/')
            result /= factor;
    }

    return result;
}

// Function to parse an expression (addition or subtraction)
static double parse_expression(const char *expr, char **endPtr) {
    double result = parse_term(expr, endPtr);
    expr = skip_whitespace(*endPtr);

    while (*expr == '+' || *expr == '-') {
        char op = *expr;
        expr++;
        double term = parse_term(expr, endPtr);
        expr = skip_whitespace(*endPtr);

        if (op == '+')
            result += term;
        else if (op == '-')
            result -= term;
    }

    return result;
}

// Main function to parse and evaluate an expression
static double ni_expr_parse_and_eval(const char *expression) {
    char *endPtr = NULL;
    double result = parse_expression(expression, &endPtr);
    endPtr = (char *)skip_whitespace(endPtr);

    if (*endPtr != '\0') {
        ni_log(NI_LOG_ERROR, "Error: Invalid expression\n");
        exit(1);
    }

    return result;
}

char *ni_strtok(char *s, const char *delim, char **saveptr) {
    char *tok;

    if (!s && !(s = *saveptr))
        return NULL;

    /* skip leading delimiters */
    s += strspn(s, delim);

    /* s now points to the first non delimiter char, or to the end of the string
     */
    if (!*s) {
        *saveptr = NULL;
        return NULL;
    }
    tok = s++;

    /* skip non delimiters */
    s += strcspn(s, delim);
    if (*s) {
        *s = 0;
        *saveptr = s + 1;
    } else {
        *saveptr = NULL;
    }

    return tok;
}

int get_pix_fmt_from_desc(const char *fmt_desc) {
    int pix_fmt;
    if (strcmp(fmt_desc, "yuv420p") == 0) {
        pix_fmt = NI_PIX_FMT_YUV420P;
    } else if (strcmp(fmt_desc, "yuv420p10le") == 0) {
        pix_fmt = NI_PIX_FMT_YUV420P10LE;
    } else if (strcmp(fmt_desc, "p010le") == 0) {
        pix_fmt = NI_PIX_FMT_P010LE;
    } else if (strcmp(fmt_desc, "bgr0") == 0) {
        pix_fmt = NI_PIX_FMT_BGR0;
    } else if (strcmp(fmt_desc, "rgba") == 0) {
        pix_fmt = NI_PIX_FMT_RGBA;
    } else if (strcmp(fmt_desc, "argb") == 0) {
        pix_fmt = NI_PIX_FMT_ARGB;
    } else if (strcmp(fmt_desc, "abgr") == 0) {
        pix_fmt = NI_PIX_FMT_ABGR;
    } else if (strcmp(fmt_desc, "bgra") == 0) {
        pix_fmt = NI_PIX_FMT_BGRA;
    } else if (strcmp(fmt_desc, "nv16") == 0) {
        pix_fmt = NI_PIX_FMT_NV16;
    } else if (strcmp(fmt_desc, "nv12") == 0) {
        pix_fmt = NI_PIX_FMT_NV12;
    } else if (strcmp(fmt_desc, "yuyv422") == 0) {
        pix_fmt = NI_PIX_FMT_YUYV422;
    } else if (strcmp(fmt_desc, "uyuv422") == 0) {
        pix_fmt = NI_PIX_FMT_UYVY422;
    } else {
        ni_log(NI_LOG_ERROR, "Unknown pixel format specified\n");
        return NI_RETCODE_FAILURE;
    }
    return pix_fmt;
}

/*!******************************************************************************
 *  \brief  Grab bitdepth factor from NI_PIX_FMT
 *
 *  \param[in]      pix_fmt         ni_pix_fmt_t
 *
 *  \return         1 or 2 for success, -1 for error
 *******************************************************************************/
int get_bitdepth_factor_from_pixfmt(int pix_fmt) {
    switch (pix_fmt) {
    case NI_PIX_FMT_YUV420P:
    case NI_PIX_FMT_NV12:
    case NI_PIX_FMT_8_TILED4X4:
        return 1;
    case NI_PIX_FMT_YUV420P10LE:
    case NI_PIX_FMT_P010LE:
    case NI_PIX_FMT_10_TILED4X4:
        return 2;
    default: {
        ni_log(NI_LOG_ERROR, "ERROR: %s() non applicable format %d\n", __func__,
               pix_fmt);
        break;
    }
    }
    return -1;
}

int ni_to_gc620_pix_fmt(ni_pix_fmt_t pix_fmt_ni) {
    int i, tablesz;

    tablesz = sizeof(gc620_pixel_fmt_list) / sizeof(struct gc620_pixel_fmts);

    /* linear search through table to find if the pixel format is supported */
    for (i = 0; i < tablesz; i++) {
        if (gc620_pixel_fmt_list[i].pix_fmt_ni == pix_fmt_ni) {
            return gc620_pixel_fmt_list[i].pix_fmt_gc620;
        }
    }
    return -1;
}

int gc620_to_ni_pix_fmt(int pix_fmt_gc620) {
    int i, tablesz;

    tablesz = sizeof(gc620_pixel_fmt_list) / sizeof(struct gc620_pixel_fmts);

    /* linear search through table to find if the pixel format is supported */
    for (i = 0; i < tablesz; i++) {
        if (gc620_pixel_fmt_list[i].pix_fmt_gc620 == pix_fmt_gc620) {
            return gc620_pixel_fmt_list[i].pix_fmt_ni;
        }
    }
    return -1;
}

ni_pixel_planar_format get_pixel_planar(ni_pix_fmt_t pix_fmt) {
    ni_pixel_planar_format ret = -1;
    switch (pix_fmt) {
    case NI_PIX_FMT_NV12:
    case NI_PIX_FMT_P010LE:
        ret = NI_PIXEL_PLANAR_FORMAT_SEMIPLANAR;
        break;
    case NI_PIX_FMT_8_TILED4X4:
    case NI_PIX_FMT_10_TILED4X4:
        ret = NI_PIXEL_PLANAR_FORMAT_TILED4X4;
        break;
    case NI_PIX_FMT_YUV420P:
    case NI_PIX_FMT_YUV420P10LE:
    case NI_PIX_FMT_ABGR: /* 32-bit ABGR packed        */
    case NI_PIX_FMT_ARGB:
    case NI_PIX_FMT_RGBA:
    case NI_PIX_FMT_BGRA:
        ret = NI_PIXEL_PLANAR_FORMAT_PLANAR;
        break;
    default:
        break;
    }

    return ret;
}

int load_input_file(const char *filename, unsigned long *bytes_read) {
    struct stat info;

    /* Get information on the file */
    if (stat(filename, &info) < 0) {
        ni_log(NI_LOG_ERROR, "Can't stat %s\n", filename);
        return -1;
    }

    /* Check the file size */
    if (info.st_size <= 0) {
        ni_log(NI_LOG_ERROR, "File %s is empty\n", filename);
        return -1;
    }

    *bytes_read = info.st_size;

    return 0;
}

// Note we do not need to consider padding bytes from yuv/rgba file reading
int calc_swframe_size(int w, int h, ni_pix_fmt_t pix_fmt) {
    int data_len = 0;
    switch (pix_fmt) {
    case NI_PIX_FMT_NV12:
    case NI_PIX_FMT_YUV420P:
        data_len = w * h * 3 / 2;
        break;
    case NI_PIX_FMT_P010LE:
    case NI_PIX_FMT_YUV420P10LE:
        data_len = w * h * 3;
        break;
    case NI_PIX_FMT_RGBA:
    case NI_PIX_FMT_BGRA:
    case NI_PIX_FMT_ARGB:
    case NI_PIX_FMT_ABGR:
    case NI_PIX_FMT_BGR0:
        data_len = w * 4 * h;
        break;
    case NI_PIX_FMT_NV16:
    case NI_PIX_FMT_YUYV422:
    case NI_PIX_FMT_UYVY422:
    default:
        break;
    }
    return data_len;
}

// return actual bytes copied from cache, in requested size
uint32_t read_one_swframe(int pfs, ni_file_reader_t *file_reader,
                          uint8_t *p_dst, uint32_t to_read) {
    uint8_t *tmp_dst = p_dst;
    unsigned int to_copy = to_read;
    unsigned long tmpFileSize = to_read;

    ni_log(NI_LOG_DEBUG, "%s: p_dst %p len %u totalSize %llu left %llu\n",
           __func__, tmp_dst, to_read, file_reader->total_file_size,
           file_reader->data_left_size);

    if (file_reader->data_left_size == 0) {
        if (file_reader->loop > 1) {
            file_reader->data_left_size = file_reader->total_file_size;
            file_reader->loop--;
            ni_log(NI_LOG_DEBUG, "input processed %llu left\n",
                   file_reader->loop);
            lseek(pfs, 0, SEEK_SET); // back to beginning
        } else {
            return 0;
        }
    } else if (file_reader->data_left_size < to_read) {
        tmpFileSize = file_reader->data_left_size;
        to_copy = file_reader->data_left_size;
    }

    int one_read_size = read(pfs, tmp_dst, to_copy);
    if (one_read_size == -1) {
        ni_log(NI_LOG_ERROR, "Error: reading file, quit! left-to-read %lu\n",
               tmpFileSize);
        ni_log(NI_LOG_ERROR, "Error: input file read error\n");
        return -1;
    } else if (one_read_size != to_copy) {
        ni_log(NI_LOG_ERROR, "Error: input file read bytes %d != expected %d\n",
               one_read_size, to_copy);
        return -1;
    }
    file_reader->data_left_size -= one_read_size;

    return one_read_size;
}

void set_demo_roi_map(ni_session_context_t *p_enc_ctx) {
    ni_xcoder_params_t *p_param =
        (ni_xcoder_params_t *)(p_enc_ctx->p_session_config);
    uint32_t i, j, sumQp = 0;
    uint32_t mbWidth, mbHeight, numMbs;
    // mode 1: Set QP for center 1/3 of picture to highest - lowest quality
    // the rest to lowest - highest quality;
    // mode non-1: reverse of mode 1
    int importanceLevelCentre = p_param->roi_demo_mode == 1 ? 40 : 10;
    int importanceLevelRest = p_param->roi_demo_mode == 1 ? 10 : 40;
    int32_t width, height;

    if (!p_enc_ctx->roi_map) {
        p_enc_ctx->roi_map =
            (ni_enc_quad_roi_custom_map *)calloc(1, p_enc_ctx->roi_len);
    }
    if (!p_enc_ctx->roi_map) {
        return;
    }
    uint32_t roiMapBlockUnitSize = 64; // HEVC
    uint32_t max_cu_size = 64;         // HEVC
    if (NI_CODEC_FORMAT_H264 == p_enc_ctx->codec_format) {
        max_cu_size = 16;
        roiMapBlockUnitSize = 16;
    }

    width = p_param->source_width;
    height = p_param->source_height;
    // AV1 non-8x8-aligned resolution is implicitly cropped due to Quadra HW
    // limitation
    if (NI_CODEC_FORMAT_AV1 == p_enc_ctx->codec_format) {
        width = (width / 8) * 8;
        height = (height / 8) * 8;
    }

    mbWidth = ((width + max_cu_size - 1) & (~(max_cu_size - 1))) /
              roiMapBlockUnitSize;
    mbHeight = ((height + max_cu_size - 1) & (~(max_cu_size - 1))) /
               roiMapBlockUnitSize;
    numMbs = mbWidth * mbHeight;

    // copy roi MBs QPs into custom map
    bool bIsCenter;
    // number of qp info (8x8) per mb or ctb
    uint32_t entryPerMb = (roiMapBlockUnitSize / 8) * (roiMapBlockUnitSize / 8);

    for (i = 0; i < numMbs; i++) {
        if ((i % mbWidth > mbWidth / 3) && (i % mbWidth < mbWidth * 2 / 3))
            bIsCenter = 1;
        else
            bIsCenter = 0;

        for (j = 0; j < entryPerMb; j++) {
            /*
              g_quad_roi_map[i*4+j].field.skip_flag = 0; // don't force
              skip mode g_quad_roi_map[i*4+j].field.roiAbsQp_flag = 1; //
              absolute QP g_quad_roi_map[i*4+j].field.qp_info = bIsCenter
              ? importanceLevelCentre : importanceLevelRest;
            */
            p_enc_ctx->roi_map[i * entryPerMb + j].field.ipcm_flag =
                0; // don't force skip mode
            p_enc_ctx->roi_map[i * entryPerMb + j].field.roiAbsQp_flag =
                1; // absolute QP
            p_enc_ctx->roi_map[i * entryPerMb + j].field.qp_info =
                bIsCenter ? importanceLevelCentre : importanceLevelRest;
        }
        sumQp += p_enc_ctx->roi_map[i * entryPerMb].field.qp_info;
    }
    p_enc_ctx->roi_avg_qp =
        // NOLINTNEXTLINE(clang-analyzer-core.DivideZero)
        (sumQp + (numMbs >> 1)) / numMbs; // round off
}

int parse_symbolic_decoder_param(ni_xcoder_params_t *p_src) {
    ni_decoder_input_params_t *pdec_param = &p_src->dec_input_params;
    int i;
    double res;
    double var_values[VAR_VARS_NB];

    if (pdec_param == NULL) {
        return NI_RETCODE_FAILURE;
    }

    for (i = 0; i < NI_MAX_NUM_OF_DECODER_OUTPUTS; i++) {
        /*Set output width and height*/
        var_values[VAR_IN_W] = var_values[VAR_IW] = pdec_param->crop_whxy[i][0];
        var_values[VAR_IN_H] = var_values[VAR_IH] = pdec_param->crop_whxy[i][1];
        var_values[VAR_OUT_W] = var_values[VAR_OW] =
            pdec_param->crop_whxy[i][0];
        var_values[VAR_OUT_H] = var_values[VAR_OH] =
            pdec_param->crop_whxy[i][1];
        if (pdec_param->cr_expr[i][0][0] && pdec_param->cr_expr[i][1][0]) {
            res = ni_expr_parse_and_eval(pdec_param->cr_expr[i][0]);
            var_values[VAR_OUT_W] = var_values[VAR_OW] = (double)floor(res);

            res = ni_expr_parse_and_eval(pdec_param->cr_expr[i][1]);
            var_values[VAR_OUT_H] = var_values[VAR_OH] = (double)floor(res);

            /* evaluate again ow as it may depend on oh */
            res = ni_expr_parse_and_eval(pdec_param->cr_expr[i][0]);
            var_values[VAR_OUT_W] = var_values[VAR_OW] = (double)floor(res);
            pdec_param->crop_whxy[i][0] = (int)var_values[VAR_OUT_W];
            pdec_param->crop_whxy[i][1] = (int)var_values[VAR_OUT_H];
        }
        /*Set output crop offset X,Y*/
        if (pdec_param->cr_expr[i][2][0]) {
            res = ni_expr_parse_and_eval(pdec_param->cr_expr[i][2]);
            var_values[VAR_X] = res;
            pdec_param->crop_whxy[i][2] = floor(var_values[VAR_X]);
        }
        if (pdec_param->cr_expr[i][3][0]) {
            res = ni_expr_parse_and_eval(pdec_param->cr_expr[i][3]);
            var_values[VAR_Y] = res;
            pdec_param->crop_whxy[i][3] = floor(var_values[VAR_Y]);
        }
        /*Set output Scale*/
        /*Reset OW and OH to next lower even number*/
        var_values[VAR_OUT_W] = var_values[VAR_OW] =
            (double)(pdec_param->crop_whxy[i][0] -
                     (pdec_param->crop_whxy[i][0] % 2));
        var_values[VAR_OUT_H] = var_values[VAR_OH] =
            (double)(pdec_param->crop_whxy[i][1] -
                     (pdec_param->crop_whxy[i][1] % 2));
        if (pdec_param->sc_expr[i][0][0] && pdec_param->sc_expr[i][1][0]) {
            res = ni_expr_parse_and_eval(pdec_param->sc_expr[i][0]);
            pdec_param->scale_wh[i][0] = ceil(res);

            res = ni_expr_parse_and_eval(pdec_param->sc_expr[i][1]);
            pdec_param->scale_wh[i][1] = ceil(res);
        }
    }
    return 0;
}

int read_input_file(ni_file_reader_t *file_reader, int pfs) {
    uint32_t read_chunk = 4096;
    uint64_t tmpFileSize = file_reader->total_file_size;
    uint8_t *file_cache = NULL;
    uint8_t *curr_cache_pos;

    if (tmpFileSize > 0 && !(file_cache = malloc(tmpFileSize))) {
        ni_log(NI_LOG_ERROR,
               "Error: input file size %llu exceeding max malloc, quit\n",
               (unsigned long long)tmpFileSize);
        return NI_RETCODE_ERROR_MEM_ALOC;
    }
    curr_cache_pos = file_cache;
    while (tmpFileSize) {
        if (read_chunk > tmpFileSize) {
            read_chunk = tmpFileSize;
        }
        int one_read_size = read(pfs, curr_cache_pos, read_chunk);
        if (one_read_size == -1) {
            ni_log(NI_LOG_ERROR, "Error: reading file, quit! left-to-read %llu\n",
                   (unsigned long long)tmpFileSize);
            free(file_cache);
            return NI_RETCODE_FAILURE;
        } else {
            tmpFileSize -= one_read_size;
            curr_cache_pos += one_read_size;
        }
    }

    if (file_cache)
    {
        curr_cache_pos = file_cache;
        file_reader->curr_cache_pos = curr_cache_pos;
        file_reader->file_cache = file_cache;
    }
    return NI_RETCODE_SUCCESS;
}

int probe_dec_stream_info(ni_codec_format_t codec_type, void *p_stream_info,
                          ni_file_reader_t *file_reader, int *width,
                          int *height, int *bit_depth, int *fps_num,
                          int *fps_den) {
    if (codec_type == NI_CODEC_FORMAT_H264) {
        ni_h264_sps_t *sps = (ni_h264_sps_t *)(p_stream_info);
        if (probe_h264_stream_info(sps, file_reader)) {
            ni_log(NI_LOG_ERROR,
                    "ERROR: H.264 file probing complete, source file format "
                    "not supported !\n");
            return NI_RETCODE_FAILURE;
        }
        *bit_depth = sps->bit_depth_luma;
        *width = sps->width;
        *height = sps->height;
    } else if (codec_type == NI_CODEC_FORMAT_H265) {
        ni_h265_sps_t *sps = (ni_h265_sps_t *)(p_stream_info);
        if (probe_h265_stream_info(sps, file_reader)) {
            ni_log(NI_LOG_ERROR,
                   "ERROR: H.264 file probing complete, source file format "
                   "not supported !\n");
            return NI_RETCODE_FAILURE;
        }
        *bit_depth = sps->bit_depth_chroma;
        *width = (int)(sps->width - (sps->pic_conf_win.left_offset +
                                     sps->pic_conf_win.right_offset));
        *height = (int)(sps->height - (sps->pic_conf_win.top_offset +
                                       sps->pic_conf_win.bottom_offset));
    } else if (codec_type == NI_CODEC_FORMAT_VP9) {
        ni_vp9_header_info_t *header_info =
            (ni_vp9_header_info_t *)(p_stream_info);
        if (probe_vp9_stream_info(header_info, file_reader)) {
            ni_log(NI_LOG_ERROR,
                   "ERROR: VP9 file probing complete, source file format "
                   "not supported !\n");
            return NI_RETCODE_FAILURE;
        }
        *bit_depth = header_info->profile ? 10 : 8;
        *width = header_info->width;
        *height = header_info->height;
        *fps_num = header_info->timebase.num;
        *fps_den = header_info->timebase.den;
    } else {
        ni_log(NI_LOG_ERROR, "Error: invalid codec format %u\n", codec_type);
        return NI_RETCODE_FAILURE;
    }
    return NI_RETCODE_SUCCESS;
}

int uploader_open_session(ni_session_context_t *p_upl_ctx, int width,
                          int height, ni_pix_fmt_t pix_fmt, int pool_size,
                          int devid) {
    int ret = 0;

    p_upl_ctx->session_id = NI_INVALID_SESSION_ID;
    p_upl_ctx->device_handle = NI_INVALID_DEVICE_HANDLE;
    p_upl_ctx->blk_io_handle = NI_INVALID_DEVICE_HANDLE;
    p_upl_ctx->hw_id = devid;
    // Set the input frame format of the upload session
    ni_uploader_set_frame_format(p_upl_ctx, width, height, pix_fmt, 0);

    ret = ni_device_session_open(p_upl_ctx, NI_DEVICE_TYPE_UPLOAD);
    if (ret != NI_RETCODE_SUCCESS) {
        ni_log(NI_LOG_ERROR, "Error: %s failure!\n", __func__);
        return ret;
    }
    ret = ni_device_session_init_framepool(p_upl_ctx, pool_size, 0);
    if (ret < 0) {
        ni_log(NI_LOG_ERROR, "Error: %s failure!\n", __func__);
    }
    return ret;
}

/*!*****************************************************************************
 *  \brief Read from input file, upload to encoder, retrieve HW descriptor
 *
 *  \param
 *
 *  \return
 ******************************************************************************/
int upload_send_data_get_desc(
    ni_session_context_t *p_upl_ctx,
    ni_session_data_io_t *p_swin_data, // intermediate for swf
    ni_session_data_io_t *p_in_data, int input_video_width,
    int input_video_height, void *yuv_buf) {
    int retval, is_semiplanar;
    ni_frame_t *p_in_frame = &p_in_data->data.frame;     // hwframe
    ni_frame_t *p_swin_frame = &p_swin_data->data.frame; // swframe
    niFrameSurface1_t *dst_surf = NULL;

    ni_log(NI_LOG_DEBUG, "===> upload_send_data <===\n");

    // reset encoder change data buffer
    memset(p_upl_ctx->enc_change_params, 0, sizeof(ni_encoder_change_params_t));
    // reset various aux data size
    p_in_frame->roi_len = 0;
    p_in_frame->reconf_len = 0;
    p_in_frame->sei_total_len = 0;
    p_in_frame->start_of_stream = 0;
    p_in_frame->end_of_stream = yuv_buf == NULL;
    p_in_frame->force_key_frame = 0;
    p_in_frame->video_width = p_swin_frame->video_width = input_video_width;
    p_in_frame->video_height = p_swin_frame->video_height = input_video_height;
    // only metadata header for now
    p_in_frame->extra_data_len = NI_APP_ENC_FRAME_META_DATA_SIZE;
    p_swin_frame->extra_data_len = NI_APP_ENC_FRAME_META_DATA_SIZE;

    int dst_stride[NI_MAX_NUM_DATA_POINTERS] = {0};
    int dst_height_aligned[NI_MAX_NUM_DATA_POINTERS] = {0};
    ni_get_min_frame_dim(
            input_video_width, input_video_height,
            p_upl_ctx->pixel_format, dst_stride,
            dst_height_aligned);
    is_semiplanar = (
        get_pixel_planar(p_upl_ctx->pixel_format) == NI_PIXEL_PLANAR_FORMAT_SEMIPLANAR);
    ni_encoder_sw_frame_buffer_alloc(
            !is_semiplanar, p_swin_frame, input_video_width,
            dst_height_aligned[0], dst_stride, 0,
            (int)p_swin_frame->extra_data_len, false);

    if (!p_swin_frame->p_data[0]) {
        ni_log(NI_LOG_ERROR, "Error: could not allocate YUV frame buffer!");
        return NI_RETCODE_FAILURE;
    }

    // can also be ni_frame_buffer_alloc()
    ni_frame_buffer_alloc_hwenc(p_in_frame, input_video_width,
                                input_video_height,
                                (int)p_in_frame->extra_data_len);
    if (!p_in_frame->p_data[3]) {
        ni_log(NI_LOG_ERROR, "Error: could not allocate hw frame buffer!");
        return NI_RETCODE_FAILURE;
    }

    dst_surf = (niFrameSurface1_t *)p_in_frame->p_data[3];

    ni_log(NI_LOG_DEBUG,
           "p_dst alloc linesize = %d/%d/%d  src height=%d  "
           "dst height aligned = %d/%d/%d\n",
           dst_stride[0], dst_stride[1], dst_stride[2], input_video_height,
           dst_height_aligned[0], dst_height_aligned[1], dst_height_aligned[2]);

    if (p_in_frame->end_of_stream) {
        goto hwupload;
    }

    uint8_t *p_src[NI_MAX_NUM_DATA_POINTERS] = {NULL};
    int src_stride[NI_MAX_NUM_DATA_POINTERS] = {0};
    int src_height[NI_MAX_NUM_DATA_POINTERS] = {0};
    src_height[0] = input_video_height;
    src_height[1] = input_video_height / 2;
    src_height[2] = (is_semiplanar) ? 0 : (input_video_height / 2);
    uint32_t conf_win_right = 0;

    switch (p_upl_ctx->pixel_format) {
    case NI_PIX_FMT_RGBA:
    case NI_PIX_FMT_BGRA:
    case NI_PIX_FMT_ABGR:
    case NI_PIX_FMT_ARGB:
    case NI_PIX_FMT_BGR0:
        src_stride[0] = input_video_width * p_upl_ctx->bit_depth_factor;
        src_height[0] = input_video_height;
        src_height[1] = 0;
        src_height[2] = 0;
        p_src[0] = yuv_buf;
        break;
    case NI_PIX_FMT_NV12:
    case NI_PIX_FMT_P010LE:
    case NI_PIX_FMT_YUV420P:
    case NI_PIX_FMT_YUV420P10LE:
        src_stride[0] = input_video_width * p_upl_ctx->bit_depth_factor;
        src_stride[1] = is_semiplanar ? src_stride[0] : src_stride[0] / 2;
        src_stride[2] = is_semiplanar ? 0 : src_stride[0] / 2;

        p_src[0] = yuv_buf;
        p_src[1] = p_src[0] + src_stride[0] * src_height[0];
        p_src[2] = p_src[1] + src_stride[1] * src_height[1];
        if (input_video_width < NI_MIN_WIDTH)
        {
            conf_win_right += (NI_MIN_WIDTH - input_video_width) / 2 * 2;
        } else
        {
            conf_win_right += (NI_VPU_CEIL(input_video_width, 2) - input_video_width) / 2 * 2;
        }
        break;
    default:
        ni_log(NI_LOG_ERROR, "%s: Error Invalid pixel format %d\n", __func__,
               p_upl_ctx->pixel_format);
        return NI_RETCODE_FAILURE;
    }

    ni_copy_frame_data(
        (uint8_t **)(p_swin_frame->p_data),
        p_src, input_video_width,
        input_video_height, p_upl_ctx->bit_depth_factor,
        p_upl_ctx->pixel_format, conf_win_right, dst_stride,
        dst_height_aligned, src_stride, src_height);

hwupload:
    retval = ni_device_session_hwup(p_upl_ctx, p_swin_data, dst_surf);
    if (retval < 0) {
        ni_log(NI_LOG_ERROR, "Error: ni_device_session_hwup():%d\n", retval);
        return NI_RETCODE_FAILURE;
    } else {
        dst_surf->ui16width = input_video_width;
        dst_surf->ui16height = input_video_height;
        dst_surf->ui32nodeAddress = 0; // always 0 offset for upload
        dst_surf->encoding_type = is_semiplanar ?
                                  NI_PIXEL_PLANAR_FORMAT_SEMIPLANAR :
                                  NI_PIXEL_PLANAR_FORMAT_PLANAR;
    }
    return 0;
}

/*!*****************************************************************************
 *  \brief Read from input file, send to uploader for hw scaling,
 *         retrieve HW descriptor
 *
 *  \param
 *
 *  \return
 ******************************************************************************/
int upload_send_data_get_desc2(
    ni_session_context_t *p_upl_ctx,
    ni_session_data_io_t *p_swin_data, // intermediate for swf
    ni_session_data_io_t *p_in_data, int input_video_width,
    int input_video_height, void *yuv_buf)
{
    int retval, is_semiplanar;
    ni_frame_t *p_in_frame = &p_in_data->data.frame;     // hwframe
    ni_frame_t *p_swin_frame = &p_swin_data->data.frame; // swframe
    niFrameSurface1_t *dst_surf = NULL;

    ni_log(NI_LOG_DEBUG, "===> upload_send_data <===\n");

    // reset encoder change data buffer
    memset(p_upl_ctx->enc_change_params, 0, sizeof(ni_encoder_change_params_t));
    // reset various aux data size
    p_in_frame->roi_len = 0;
    p_in_frame->reconf_len = 0;
    p_in_frame->sei_total_len = 0;
    p_in_frame->start_of_stream = 0;
    p_in_frame->end_of_stream = yuv_buf == NULL;
    p_in_frame->force_key_frame = 0;
    p_in_frame->video_width = p_swin_frame->video_width = input_video_width;
    p_in_frame->video_height = p_swin_frame->video_height = input_video_height;
    // only metadata header for now
    p_in_frame->extra_data_len = NI_APP_ENC_FRAME_META_DATA_SIZE;
    p_swin_frame->extra_data_len = NI_APP_ENC_FRAME_META_DATA_SIZE;

    int dst_stride[NI_MAX_NUM_DATA_POINTERS] = {0};
    int dst_height_aligned[NI_MAX_NUM_DATA_POINTERS] = {0};
    dst_height_aligned[0] = NI_VPU_CEIL(input_video_height, 2);
    dst_height_aligned[2] = dst_height_aligned[1] = dst_height_aligned[0] / 2;
    is_semiplanar = (
        get_pixel_planar(p_upl_ctx->pixel_format) == NI_PIXEL_PLANAR_FORMAT_SEMIPLANAR);
    switch (p_upl_ctx->pixel_format) {
    case NI_PIX_FMT_RGBA:
    case NI_PIX_FMT_BGRA:
    case NI_PIX_FMT_ABGR:
    case NI_PIX_FMT_ARGB:
    case NI_PIX_FMT_BGR0:
        dst_stride[0] = NI_VPU_CEIL(input_video_width, 16) * p_upl_ctx->bit_depth_factor;
        dst_height_aligned[2] = dst_height_aligned[1] = 0;
        break;
    case NI_PIX_FMT_NV12:
        dst_stride[0] = NI_VPU_CEIL(input_video_width, 128);
        dst_stride[1] = dst_stride[0];
        dst_height_aligned[2] = 0;
        break;
    case NI_PIX_FMT_P010LE:
        dst_stride[0] = NI_VPU_CEIL(input_video_width * 2, 128);
        dst_stride[1] = dst_stride[0];
        dst_height_aligned[2] = 0;
        break;
    case NI_PIX_FMT_YUV420P:
        dst_stride[0] = NI_VPU_CEIL(input_video_width, 128);
        dst_stride[1] = NI_VPU_CEIL((input_video_width / 2), 128);
        dst_stride[2] = dst_stride[1];
        break;
    case NI_PIX_FMT_YUV420P10LE:
        dst_stride[0] = NI_VPU_CEIL(input_video_width * 2, 128);
        dst_stride[1] = NI_VPU_CEIL(input_video_width, 128);
        dst_stride[2] = dst_stride[1];
        break;
    case NI_PIX_FMT_NV16:
    case NI_PIX_FMT_YUYV422:
    case NI_PIX_FMT_UYVY422:
    default:
        ni_log(NI_LOG_ERROR, "%s: Error Invalid pixel format %d\n", __func__,
               p_upl_ctx->pixel_format);
        return NI_RETCODE_FAILURE;
    }

    ni_frame_buffer_alloc_pixfmt(
        p_swin_frame, p_upl_ctx->pixel_format, input_video_width,
        input_video_height, dst_stride,
        p_upl_ctx->codec_format == NI_CODEC_FORMAT_H264,
        (int)p_swin_frame->extra_data_len);
    if (!p_swin_frame->p_data[0]) {
        ni_log(NI_LOG_ERROR, "Error: could not allocate YUV frame buffer!");
        return NI_RETCODE_FAILURE;
    }

    // alloc dest avframe buff
    // can also be ni_frame_buffer_alloc()
    ni_frame_buffer_alloc_hwenc(p_in_frame, input_video_width,
                                input_video_height,
                                (int)p_in_frame->extra_data_len);
    if (!p_in_frame->p_data[3]) {
        ni_log(NI_LOG_ERROR, "Error: could not allocate hw frame buffer!");
        return NI_RETCODE_FAILURE;
    }

    dst_surf = (niFrameSurface1_t *)p_in_frame->p_data[3];

    ni_log(NI_LOG_DEBUG,
           "p_dst alloc linesize = %d/%d/%d  src height=%d  "
           "dst height aligned = %d/%d/%d\n",
           dst_stride[0], dst_stride[1], dst_stride[2], input_video_height,
           dst_height_aligned[0], dst_height_aligned[1], dst_height_aligned[2]);

    if (p_in_frame->end_of_stream) {
        goto hwupload;
    }

    uint8_t *p_src[NI_MAX_NUM_DATA_POINTERS] = {NULL};
    int src_stride[NI_MAX_NUM_DATA_POINTERS] = {0};
    int src_height[NI_MAX_NUM_DATA_POINTERS] = {0};

    switch (p_upl_ctx->pixel_format) {
    case NI_PIX_FMT_RGBA:
    case NI_PIX_FMT_BGRA:
    case NI_PIX_FMT_ABGR:
    case NI_PIX_FMT_ARGB:
    case NI_PIX_FMT_BGR0:
        src_stride[0] = input_video_width * p_upl_ctx->bit_depth_factor;
        src_height[0] = input_video_height;
        p_src[0] = yuv_buf;
        break;
    case NI_PIX_FMT_NV12:
    case NI_PIX_FMT_P010LE:
    case NI_PIX_FMT_YUV420P:
    case NI_PIX_FMT_YUV420P10LE:
        src_stride[0] = input_video_width * p_upl_ctx->bit_depth_factor;
        src_stride[1] = is_semiplanar ? src_stride[0] : src_stride[0] / 2;
        src_stride[2] = is_semiplanar ? 0 : src_stride[0] / 2;

        src_height[0] = input_video_height;
        src_height[1] = src_height[0] / 2;
        src_height[2] = is_semiplanar ? 0 : src_height[1];

        p_src[0] = yuv_buf;
        p_src[1] = p_src[0] + src_stride[0] * src_height[0];
        p_src[2] = p_src[1] + src_stride[1] * src_height[1];
        break;
    default:
        ni_log(NI_LOG_ERROR, "%s: Error Invalid pixel format %d\n", __func__,
               p_upl_ctx->pixel_format);
        return NI_RETCODE_FAILURE;
    }

    ni_copy_frame_data(
        (uint8_t **)(p_swin_frame->p_data),
        p_src, input_video_width,
        input_video_height, p_upl_ctx->bit_depth_factor,
        p_upl_ctx->pixel_format, 0, dst_stride,
        dst_height_aligned, src_stride, src_height);

hwupload:
    retval = ni_device_session_hwup(p_upl_ctx, p_swin_data, dst_surf);
    if (retval < 0) {
        ni_log(NI_LOG_ERROR, "Error: ni_device_session_hwup():%d\n", retval);
        return NI_RETCODE_FAILURE;
    } else {
        dst_surf->ui16width = input_video_width;
        dst_surf->ui16height = input_video_height;
        dst_surf->ui32nodeAddress = 0; // always 0 offset for upload
        dst_surf->encoding_type = is_semiplanar
                                      ? NI_PIXEL_PLANAR_FORMAT_SEMIPLANAR
                                      : NI_PIXEL_PLANAR_FORMAT_PLANAR;
    }
    return 0;
}

// convert various reconfig and demo modes (stored in encoder configuration) to
// aux data and store them in frame
void prep_reconf_demo_data(ni_session_context_t *p_enc_ctx, ni_frame_t *frame) {
    ni_xcoder_params_t *api_param =
        (ni_xcoder_params_t *)p_enc_ctx->p_session_config;
    ni_aux_data_t *aux_data = NULL;

    switch (api_param->reconf_demo_mode) {
    case XCODER_TEST_RECONF_BR:
        if (p_enc_ctx->frame_num ==
            api_param->reconf_hash[p_enc_ctx->reconfigCount][0]) {
            aux_data = ni_frame_new_aux_data(frame, NI_FRAME_AUX_DATA_BITRATE,
                                             sizeof(int32_t));
            if (!aux_data) {
                ni_log(NI_LOG_ERROR,
                       "Error %s(): no mem for reconf BR aux_data\n", __func__);
                return;
            }
            *((int32_t *)aux_data->data) =
                api_param->reconf_hash[p_enc_ctx->reconfigCount][1];
            ni_log(NI_LOG_DEBUG, "%s(): frame #%lu reconfig BR by frame %d\n",
                   __func__, p_enc_ctx->frame_num,
                   api_param->reconf_hash[p_enc_ctx->reconfigCount][1]);

            p_enc_ctx->reconfigCount++;
        }
        break;
    /*
    case XCODER_TEST_RECONF_INTRAPRD:
        if (p_enc_ctx->frame_num ==
            api_param->reconf_hash[p_enc_ctx->reconfigCount][0])
        {
            p_enc_ctx->enc_change_params->enable_option |=
                NI_SET_CHANGE_PARAM_INTRA_PARAM;
            p_enc_ctx->enc_change_params->intraQP =
                api_param->reconf_hash[p_enc_ctx->reconfigCount][1];
            p_enc_ctx->enc_change_params->intraPeriod =
                api_param->reconf_hash[p_enc_ctx->reconfigCount][2];
            p_enc_ctx->enc_change_params->repeatHeaders =
                api_param->reconf_hash[p_enc_ctx->reconfigCount][3];
            ni_log(NI_LOG_DEBUG,
                "%s(): frame #%lu reconf intraQP %d intraPeriod %d "
                "repeatHeaders %d\n",
                __func__, p_enc_ctx->frame_num,
                p_enc_ctx->enc_change_params->intraQP,
                p_enc_ctx->enc_change_params->intraPeriod,
                p_enc_ctx->enc_change_params->repeatHeaders);

            // frame reconf_len needs to be set here
            frame->reconf_len = sizeof(ni_encoder_change_params_t);
            p_enc_ctx->reconfigCount++;
        }
        break;
    */
    case XCODER_TEST_RECONF_VUI_HRD:
        if (p_enc_ctx->frame_num ==
            api_param->reconf_hash[p_enc_ctx->reconfigCount][0]) {
            p_enc_ctx->enc_change_params->enable_option |=
                NI_SET_CHANGE_PARAM_VUI_HRD_PARAM;
            p_enc_ctx->enc_change_params->colorDescPresent =
                api_param->reconf_hash[p_enc_ctx->reconfigCount][1];
            p_enc_ctx->enc_change_params->colorPrimaries =
                api_param->reconf_hash[p_enc_ctx->reconfigCount][2];
            p_enc_ctx->enc_change_params->colorTrc =
                api_param->reconf_hash[p_enc_ctx->reconfigCount][3];
            p_enc_ctx->enc_change_params->colorSpace =
                api_param->reconf_hash[p_enc_ctx->reconfigCount][4];
            p_enc_ctx->enc_change_params->aspectRatioWidth =
                api_param->reconf_hash[p_enc_ctx->reconfigCount][5];
            p_enc_ctx->enc_change_params->aspectRatioHeight =
                api_param->reconf_hash[p_enc_ctx->reconfigCount][6];
            p_enc_ctx->enc_change_params->videoFullRange =
                api_param->reconf_hash[p_enc_ctx->reconfigCount][7];

            // frame reconf_len needs to be set here
            frame->reconf_len = sizeof(ni_encoder_change_params_t);
            p_enc_ctx->reconfigCount++;
        }
        break;
    case XCODER_TEST_RECONF_LONG_TERM_REF:
        // the reconf file data line format for this is:
        // <frame-number>:useCurSrcAsLongtermPic,useLongtermRef where
        // values will stay the same on every frame until changed.
        if (p_enc_ctx->frame_num ==
            api_param->reconf_hash[p_enc_ctx->reconfigCount][0]) {
            aux_data =
                ni_frame_new_aux_data(frame, NI_FRAME_AUX_DATA_LONG_TERM_REF,
                                      sizeof(ni_long_term_ref_t));
            if (!aux_data) {
                ni_log(NI_LOG_ERROR,
                       "Error %s(): no mem for reconf LTR aux_data\n",
                       __func__);
                return;
            }
            ni_long_term_ref_t *ltr = (ni_long_term_ref_t *)aux_data->data;
            ltr->use_cur_src_as_long_term_pic =
                (uint8_t)api_param->reconf_hash[p_enc_ctx->reconfigCount][1];
            ltr->use_long_term_ref =
                (uint8_t)api_param->reconf_hash[p_enc_ctx->reconfigCount][2];

            ni_log(NI_LOG_DEBUG,
                   "%s(): frame #%lu reconf LTR "
                   "use_cur_src_as_long_term_pic %u use_long_term_ref "
                   "%u\n",
                   __func__, p_enc_ctx->frame_num,
                   ltr->use_cur_src_as_long_term_pic, ltr->use_long_term_ref);

            p_enc_ctx->reconfigCount++;
        }
        break;
    case XCODER_TEST_RECONF_LTR_INTERVAL:
        if (p_enc_ctx->frame_num ==
            api_param->reconf_hash[p_enc_ctx->reconfigCount][0]) {
            aux_data = ni_frame_new_aux_data(
                frame, NI_FRAME_AUX_DATA_LTR_INTERVAL, sizeof(int32_t));
            if (!aux_data) {
                ni_log(NI_LOG_ERROR,
                       "Error %s(): no mem for reconf LTR interval "
                       "aux_data\n",
                       __func__);
                return;
            }
            *((int32_t *)aux_data->data) =
                api_param->reconf_hash[p_enc_ctx->reconfigCount][1];
            ni_log(NI_LOG_DEBUG, "%s(): frame #%lu reconf LTR interval %d\n",
                   __func__, p_enc_ctx->frame_num,
                   *((int32_t *)aux_data->data));

            p_enc_ctx->reconfigCount++;
        }
        break;
    case XCODER_TEST_INVALID_REF_FRAME:
        if (p_enc_ctx->frame_num ==
            api_param->reconf_hash[p_enc_ctx->reconfigCount][0]) {
            aux_data = ni_frame_new_aux_data(
                frame, NI_FRAME_AUX_DATA_INVALID_REF_FRAME, sizeof(int32_t));
            if (!aux_data) {
                ni_log(NI_LOG_ERROR,
                       "Error %s(): no mem for reconf invalid ref "
                       "frame aux_data\n",
                       __func__);
                return;
            }
            *((int32_t *)aux_data->data) =
                api_param->reconf_hash[p_enc_ctx->reconfigCount][1];
            ni_log(NI_LOG_DEBUG,
                   "%s(): frame #%lu reconf invalid ref frame %d\n", __func__,
                   p_enc_ctx->frame_num, *((int32_t *)aux_data->data));

            p_enc_ctx->reconfigCount++;
        }
        break;
    case XCODER_TEST_RECONF_FRAMERATE:
        if (p_enc_ctx->frame_num ==
            api_param->reconf_hash[p_enc_ctx->reconfigCount][0]) {
            aux_data = ni_frame_new_aux_data(frame, NI_FRAME_AUX_DATA_FRAMERATE,
                                             sizeof(ni_framerate_t));
            if (!aux_data) {
                ni_log(NI_LOG_ERROR,
                       "Error %s(): no mem for reconf framerate aux_data\n",
                       __func__);
                return;
            }
            ni_framerate_t *framerate = (ni_framerate_t *)aux_data->data;
            framerate->framerate_num =
                (int32_t)api_param->reconf_hash[p_enc_ctx->reconfigCount][1];
            framerate->framerate_denom =
                (int32_t)api_param->reconf_hash[p_enc_ctx->reconfigCount][2];

            ni_log(NI_LOG_DEBUG,
                   "%s(): frame #%lu reconfig framerate by frame (%d/%d)\n",
                   __func__, p_enc_ctx->frame_num, framerate->framerate_num,
                   framerate->framerate_denom);

            p_enc_ctx->reconfigCount++;
        }
        break;
    case XCODER_TEST_RECONF_MAX_FRAME_SIZE:
        if (p_enc_ctx->frame_num ==
            api_param->reconf_hash[p_enc_ctx->reconfigCount][0]) {
            aux_data = ni_frame_new_aux_data(
                frame, NI_FRAME_AUX_DATA_MAX_FRAME_SIZE, sizeof(int32_t));
            if (!aux_data) {
                ni_log(NI_LOG_ERROR,
                       "Error %s(): no mem for reconf maxFrameSize aux_data\n",
                       __func__);
                return;
            }
            *((int32_t *)aux_data->data) =
                api_param->reconf_hash[p_enc_ctx->reconfigCount][1];
            ni_log(NI_LOG_DEBUG,
                   "%s(): frame #%lu reconfig maxFrameSize by frame %d\n",
                   __func__, p_enc_ctx->frame_num,
                   api_param->reconf_hash[p_enc_ctx->reconfigCount][1]);

            p_enc_ctx->reconfigCount++;
        }
        break;
    case XCODER_TEST_RECONF_RC_MIN_MAX_QP:
    case XCODER_TEST_RECONF_RC_MIN_MAX_QP_REDUNDANT:
        if (p_enc_ctx->frame_num ==
            api_param->reconf_hash[p_enc_ctx->reconfigCount][0]) {
            aux_data = ni_frame_new_aux_data(
                frame, NI_FRAME_AUX_DATA_MAX_MIN_QP, sizeof(ni_rc_min_max_qp));
            if (!aux_data) {
                ni_log(NI_LOG_ERROR,
                       "Error %s(): no mem for reconf max&min QP aux_data\n",
                       __func__);
                return;
            }
            ni_rc_min_max_qp *qp_info = (ni_rc_min_max_qp *)aux_data->data;
            qp_info->minQpI = api_param->reconf_hash[p_enc_ctx->reconfigCount][1];
            qp_info->maxQpI = api_param->reconf_hash[p_enc_ctx->reconfigCount][2];
            qp_info->maxDeltaQp = api_param->reconf_hash[p_enc_ctx->reconfigCount][3];
            qp_info->minQpPB = api_param->reconf_hash[p_enc_ctx->reconfigCount][4];
            qp_info->maxQpPB = api_param->reconf_hash[p_enc_ctx->reconfigCount][5];

            p_enc_ctx->reconfigCount++;
        }
        break;
    case XCODER_TEST_RECONF_CRF:
        if (p_enc_ctx->frame_num ==
            api_param->reconf_hash[p_enc_ctx->reconfigCount][0]) {
            aux_data = ni_frame_new_aux_data(
                frame, NI_FRAME_AUX_DATA_CRF, sizeof(int32_t));
            if (!aux_data) {
                ni_log(NI_LOG_ERROR,
                       "Error %s(): no mem for reconf crf aux_data\n",
                       __func__);
                return;
            }
            *((int32_t *)aux_data->data) =
                api_param->reconf_hash[p_enc_ctx->reconfigCount][1];
            ni_log(NI_LOG_DEBUG, "%s(): frame #%lu reconfig crf %d by frame aux data\n",
                   __func__, p_enc_ctx->frame_num,
                   api_param->reconf_hash[p_enc_ctx->reconfigCount][1]);

            p_enc_ctx->reconfigCount++;
        }
        break;
    case XCODER_TEST_RECONF_CRF_FLOAT:
        if (p_enc_ctx->frame_num ==
            api_param->reconf_hash[p_enc_ctx->reconfigCount][0]) {
            aux_data = ni_frame_new_aux_data(
                frame, NI_FRAME_AUX_DATA_CRF_FLOAT, sizeof(float));
            if (!aux_data) {
                ni_log(NI_LOG_ERROR,
                       "Error %s(): no mem for reconf crf aux_data\n",
                       __func__);
                return;
            }
            float crf = (float)(api_param->reconf_hash[p_enc_ctx->reconfigCount][1] +
                (float)api_param->reconf_hash[p_enc_ctx->reconfigCount][2] / 100.0);
            *((float *)aux_data->data) = crf;
            ni_log(NI_LOG_DEBUG,
                   "%s(): frame #%lu reconfig float type crf %f by frame "
                   "aux data\n", __func__, p_enc_ctx->frame_num, crf);
            p_enc_ctx->reconfigCount++;
        }
        break;

    case XCODER_TEST_RECONF_VBV:
        if (p_enc_ctx->frame_num ==
            api_param->reconf_hash[p_enc_ctx->reconfigCount][0]) {
            aux_data = ni_frame_new_aux_data(
                frame, NI_FRAME_AUX_DATA_VBV_MAX_RATE, sizeof(int32_t));
            if (!aux_data) {
                ni_log(NI_LOG_ERROR,
                       "Error %s(): no mem for reconf vbfMaxRate aux_data\n",
                       __func__);
                return;
            }
            *((int32_t *)aux_data->data) =
                api_param->reconf_hash[p_enc_ctx->reconfigCount][1];
            aux_data = ni_frame_new_aux_data(
                frame, NI_FRAME_AUX_DATA_VBV_BUFFER_SIZE, sizeof(int32_t));
            if (!aux_data) {
                ni_log(NI_LOG_ERROR,
                       "Error %s(): no mem for reconf vbvBufferSize aux_data\n",
                       __func__);
                return;
            }
            *((int32_t *)aux_data->data) =
                api_param->reconf_hash[p_enc_ctx->reconfigCount][2];
            ni_log(NI_LOG_DEBUG,
                   "%s(): frame #%lu reconfig vbfMaxRate %d vbvBufferSize "
                   "%d by frame aux data\n",
                   __func__, p_enc_ctx->frame_num,
                   api_param->reconf_hash[p_enc_ctx->reconfigCount][1],
                   api_param->reconf_hash[p_enc_ctx->reconfigCount][2]);

            p_enc_ctx->reconfigCount++;
        }
        break;
    case XCODER_TEST_RECONF_MAX_FRAME_SIZE_RATIO:
        if (p_enc_ctx->frame_num ==
            api_param->reconf_hash[p_enc_ctx->reconfigCount][0]) {
            int maxFrameSizeRatio = api_param->reconf_hash[p_enc_ctx->reconfigCount][1];
            if (maxFrameSizeRatio < 1) {
                ni_log(NI_LOG_ERROR, "maxFrameSizeRatio %d cannot < 1\n",
                       maxFrameSizeRatio);
                return;
            }
            aux_data = ni_frame_new_aux_data(
                frame, NI_FRAME_AUX_DATA_MAX_FRAME_SIZE, sizeof(int32_t));
            if (!aux_data) {
                ni_log(NI_LOG_ERROR,
                       "Error %s(): no mem for reconf maxFrameSizeRatio aux_data\n",
                       __func__);
                return;
            }

            int32_t bitrate, framerate_num, framerate_denom;
            uint32_t min_maxFrameSize, maxFrameSize;
            bitrate = (p_enc_ctx->target_bitrate > 0) ?  p_enc_ctx->target_bitrate : api_param->bitrate;

            if ((p_enc_ctx->framerate.framerate_num > 0) && (p_enc_ctx->framerate.framerate_denom > 0)) {
                framerate_num = p_enc_ctx->framerate.framerate_num;
                framerate_denom = p_enc_ctx->framerate.framerate_denom;
            } else {
                framerate_num = (int32_t) api_param->fps_number;
                framerate_denom = (int32_t) api_param->fps_denominator;
            }

            min_maxFrameSize = ((uint32_t)bitrate / framerate_num * framerate_denom) / 8;
            maxFrameSize = min_maxFrameSize * maxFrameSizeRatio > NI_MAX_FRAME_SIZE ?
                            NI_MAX_FRAME_SIZE : min_maxFrameSize * maxFrameSizeRatio;
            *((int32_t *)aux_data->data) = maxFrameSize;
            ni_log(NI_LOG_DEBUG,
                   "xcoder_send_frame: frame #%lu reconf "
                   "maxFrameSizeRatio %d maxFrameSize %d\n",
                   p_enc_ctx->frame_num, maxFrameSizeRatio, maxFrameSize);

            p_enc_ctx->reconfigCount++;
        }
        break;

    case XCODER_TEST_FORCE_IDR_FRAME:
        if (p_enc_ctx->frame_num ==
            api_param->reconf_hash[p_enc_ctx->reconfigCount][0]) {
            ni_force_idr_frame_type(p_enc_ctx);
            ni_log(NI_LOG_DEBUG, "%s(): frame #%lu force IDR frame\n", __func__,
                   p_enc_ctx->frame_num);
            p_enc_ctx->reconfigCount++;
        }
        break;
    case XCODER_TEST_RECONF_BR_API:
        if (p_enc_ctx->frame_num ==
            api_param->reconf_hash[p_enc_ctx->reconfigCount][0]) {
            ni_reconfig_bitrate(p_enc_ctx,
                                api_param->reconf_hash[p_enc_ctx->reconfigCount][1]);
            ni_log(NI_LOG_DEBUG, "%s(): frame #%lu API reconfig BR %d\n",
                   __func__, p_enc_ctx->frame_num,
                   api_param->reconf_hash[p_enc_ctx->reconfigCount][1]);
            p_enc_ctx->reconfigCount++;
        }
        break;
    case XCODER_TEST_RECONF_VUI_HRD_API:
        if (p_enc_ctx->frame_num ==
            api_param->reconf_hash[p_enc_ctx->reconfigCount][0]) {
            ni_vui_hrd_t vui;
            vui.colorDescPresent =
                (uint8_t)api_param->reconf_hash[p_enc_ctx->reconfigCount][1];
            vui.colorPrimaries =
                (uint8_t)api_param->reconf_hash[p_enc_ctx->reconfigCount][2];
            vui.colorTrc =
                (uint8_t)api_param->reconf_hash[p_enc_ctx->reconfigCount][3];
            vui.colorSpace =
                (uint8_t)api_param->reconf_hash[p_enc_ctx->reconfigCount][4];
            vui.aspectRatioWidth =
                (uint8_t)api_param->reconf_hash[p_enc_ctx->reconfigCount][5];
            vui.aspectRatioHeight =
                (uint8_t)api_param->reconf_hash[p_enc_ctx->reconfigCount][6];
            vui.videoFullRange =
                (uint8_t)api_param->reconf_hash[p_enc_ctx->reconfigCount][7];

            ni_reconfig_vui(p_enc_ctx, &vui);
            ni_log(NI_LOG_DEBUG,
                   "%s(): frame #%lu API reconfig VUI HRD "
                   "colorDescPresent %d colorPrimaries %d "
                   "colorTrc %d colorSpace %d aspectRatioWidth %d "
                   "aspectRatioHeight %d videoFullRange %d\n",
                   __func__, p_enc_ctx->frame_num,
                   api_param->reconf_hash[p_enc_ctx->reconfigCount][1],
                   api_param->reconf_hash[p_enc_ctx->reconfigCount][2],
                   api_param->reconf_hash[p_enc_ctx->reconfigCount][3],
                   api_param->reconf_hash[p_enc_ctx->reconfigCount][4],
                   api_param->reconf_hash[p_enc_ctx->reconfigCount][5],
                   api_param->reconf_hash[p_enc_ctx->reconfigCount][6],
                   api_param->reconf_hash[p_enc_ctx->reconfigCount][7]);
            p_enc_ctx->reconfigCount++;
        }
        break;
    case XCODER_TEST_RECONF_LTR_API:
        if (p_enc_ctx->frame_num ==
            api_param->reconf_hash[p_enc_ctx->reconfigCount][0]) {
            ni_long_term_ref_t ltr;
            ltr.use_cur_src_as_long_term_pic =
                (uint8_t)api_param->reconf_hash[p_enc_ctx->reconfigCount][1];
            ltr.use_long_term_ref =
                (uint8_t)api_param->reconf_hash[p_enc_ctx->reconfigCount][2];

            ni_set_ltr(p_enc_ctx, &ltr);
            ni_log(NI_LOG_DEBUG, "%s(): frame #%lu API set LTR\n", __func__,
                   p_enc_ctx->frame_num);
            p_enc_ctx->reconfigCount++;
        }
        break;
    case XCODER_TEST_RECONF_LTR_INTERVAL_API:
        if (p_enc_ctx->frame_num ==
            api_param->reconf_hash[p_enc_ctx->reconfigCount][0]) {
            ni_set_ltr_interval(p_enc_ctx,
                                api_param->reconf_hash[p_enc_ctx->reconfigCount][1]);
            ni_log(NI_LOG_DEBUG, "%s(): frame #%lu API set LTR interval %d\n",
                   __func__, p_enc_ctx->frame_num,
                   api_param->reconf_hash[p_enc_ctx->reconfigCount][1]);
            p_enc_ctx->reconfigCount++;
        }
        break;
    case XCODER_TEST_INVALID_REF_FRAME_API:
        if (p_enc_ctx->frame_num ==
            api_param->reconf_hash[p_enc_ctx->reconfigCount][0]) {
            ni_set_frame_ref_invalid(
                p_enc_ctx, api_param->reconf_hash[p_enc_ctx->reconfigCount][1]);
            ni_log(NI_LOG_DEBUG,
                   "%s(): frame #%lu API set frame ref invalid "
                   "%d\n",
                   __func__, p_enc_ctx->frame_num,
                   api_param->reconf_hash[p_enc_ctx->reconfigCount][1]);
            p_enc_ctx->reconfigCount++;
        }
        break;
    case XCODER_TEST_RECONF_FRAMERATE_API:
        if (p_enc_ctx->frame_num ==
            api_param->reconf_hash[p_enc_ctx->reconfigCount][0]) {
            ni_framerate_t framerate;
            framerate.framerate_num =
                (int32_t)api_param->reconf_hash[p_enc_ctx->reconfigCount][1];
            framerate.framerate_denom =
                (int32_t)api_param->reconf_hash[p_enc_ctx->reconfigCount][2];
            ni_reconfig_framerate(p_enc_ctx, &framerate);
            ni_log(NI_LOG_DEBUG,
                   "%s(): frame #%lu API reconfig framerate (%d/%d)\n",
                   __func__, p_enc_ctx->frame_num,
                   api_param->reconf_hash[p_enc_ctx->reconfigCount][1],
                   api_param->reconf_hash[p_enc_ctx->reconfigCount][2]);
            p_enc_ctx->reconfigCount++;
        }
        break;
    case XCODER_TEST_RECONF_MAX_FRAME_SIZE_API:
        if (p_enc_ctx->frame_num ==
            api_param->reconf_hash[p_enc_ctx->reconfigCount][0]) {
            ni_reconfig_max_frame_size(
                p_enc_ctx, api_param->reconf_hash[p_enc_ctx->reconfigCount][1]);
            ni_log(NI_LOG_DEBUG,
                   "%s(): frame #%lu API reconfig maxFrameSize %d\n", __func__,
                   p_enc_ctx->frame_num,
                   api_param->reconf_hash[p_enc_ctx->reconfigCount][1]);
            p_enc_ctx->reconfigCount++;
        }
        break;
    case XCODER_TEST_RECONF_RC_MIN_MAX_QP_API:
    case XCODER_TEST_RECONF_RC_MIN_MAX_QP_API_REDUNDANT:
        if (p_enc_ctx->frame_num ==
            api_param->reconf_hash[p_enc_ctx->reconfigCount][0]) {
            ni_rc_min_max_qp qp_info;
            qp_info.minQpI =
                (int32_t)api_param->reconf_hash[p_enc_ctx->reconfigCount][1];
            qp_info.maxQpI =
                (int32_t)api_param->reconf_hash[p_enc_ctx->reconfigCount][2];
            qp_info.maxDeltaQp =
                (int32_t)api_param->reconf_hash[p_enc_ctx->reconfigCount][3];
            qp_info.minQpPB =
                (int32_t)api_param->reconf_hash[p_enc_ctx->reconfigCount][4];
            qp_info.maxQpPB =
                (int32_t)api_param->reconf_hash[p_enc_ctx->reconfigCount][5];
            ni_reconfig_min_max_qp(p_enc_ctx, &qp_info);
            ni_log(NI_LOG_DEBUG,
                   "%s(): frame %d minQpI %d maxQpI %d maxDeltaQp %d minQpPB "
                   "%d maxQpPB %d\n",
                   __func__, p_enc_ctx->frame_num, qp_info.minQpI,
                   qp_info.maxQpI, qp_info.maxDeltaQp, qp_info.minQpPB,
                   qp_info.maxQpPB);
            p_enc_ctx->reconfigCount++;
        }
        break;
    case XCODER_TEST_CRF_API:
        if (p_enc_ctx->frame_num ==
            api_param->reconf_hash[p_enc_ctx->reconfigCount][0]) {
            ni_reconfig_crf(p_enc_ctx,
                            api_param->reconf_hash[p_enc_ctx->reconfigCount][1]);
            ni_log(NI_LOG_DEBUG, "%s(): frame #%lu API reconfig crf %d\n",
                   __func__, p_enc_ctx->frame_num,
                   api_param->reconf_hash[p_enc_ctx->reconfigCount][1]);
            p_enc_ctx->reconfigCount++;
        }
        break;
    case XCODER_TEST_CRF_FLOAT_API:
        if (p_enc_ctx->frame_num ==
            api_param->reconf_hash[p_enc_ctx->reconfigCount][0]) {
            float crf = (float)(api_param->reconf_hash[p_enc_ctx->reconfigCount][1] +
                        (float)api_param->reconf_hash[p_enc_ctx->reconfigCount][2] / 100.0);
            ni_reconfig_crf2(p_enc_ctx, crf);
            ni_log(NI_LOG_DEBUG, "%s(): frame #%lu API reconfig crf %f\n",
                   __func__, p_enc_ctx->frame_num, crf);
            p_enc_ctx->reconfigCount++;
        }
        break;
    case XCODER_TEST_RECONF_VBV_API:
        if (p_enc_ctx->frame_num ==
            api_param->reconf_hash[p_enc_ctx->reconfigCount][0]) {
            ni_reconfig_vbv_value(
                p_enc_ctx, api_param->reconf_hash[p_enc_ctx->reconfigCount][1],
                api_param->reconf_hash[p_enc_ctx->reconfigCount][2]);
            ni_log(NI_LOG_DEBUG, "%s(): frame #%lu API reconfig vbvMaxRate %d vbvBufferSize %d\n",
                   __func__, p_enc_ctx->frame_num,
                   api_param->reconf_hash[p_enc_ctx->reconfigCount][1],
                   api_param->reconf_hash[p_enc_ctx->reconfigCount][2]);
            p_enc_ctx->reconfigCount++;
        }
        break;
    case XCODER_TEST_RECONF_MAX_FRAME_SIZE_RATIO_API:
        if (p_enc_ctx->frame_num ==
            api_param->reconf_hash[p_enc_ctx->reconfigCount][0]) {
            ni_reconfig_max_frame_size_ratio(
                p_enc_ctx, api_param->reconf_hash[p_enc_ctx->reconfigCount][1]);
            ni_log(NI_LOG_DEBUG,
                   "xcoder_send_frame: frame #%lu reconf maxFrameSizeRatio %d\n",
                   p_enc_ctx->frame_num, api_param->reconf_hash[p_enc_ctx->reconfigCount][1]);

            p_enc_ctx->reconfigCount++;
        }
        break;
    case XCODER_TEST_RECONF_OFF:
    default:;
    }
}

int ni_frame_copy_aux_data(ni_frame_t *p_dst_frame,
                           const ni_frame_t *p_src_frame) {
    int i;
    ni_aux_data_t *aux;

    for (i = 0; i < p_src_frame->nb_aux_data; i++) {
        aux = p_src_frame->aux_data[i];
        if (p_dst_frame->aux_data[i]->size != aux->size) {
            if (p_dst_frame->aux_data[i]->size)
                free(p_dst_frame->aux_data[i]->data);
            p_dst_frame->aux_data[i]->data = malloc(aux->size);
            if (!p_dst_frame->aux_data[i]->data)
                return NI_RETCODE_ERROR_MEM_ALOC;

            p_dst_frame->aux_data[i]->size = aux->size;
            p_dst_frame->aux_data[i]->type = aux->type;
        }
        memcpy(p_dst_frame->aux_data[i]->data, aux->data, aux->size);
    }
    return NI_RETCODE_SUCCESS;
}

int frame_copy_props(ni_session_data_io_t *dst, const ni_session_data_io_t *src) {
    int ret = NI_RETCODE_SUCCESS;
    ni_frame_t *p_dst_frame = &dst->data.frame;
    const ni_frame_t *p_src_frame = &src->data.frame;
    p_dst_frame->src_codec = p_src_frame->src_codec;
    p_dst_frame->dts = p_src_frame->dts;
    p_dst_frame->pts = p_src_frame->pts;
    p_dst_frame->end_of_stream = p_src_frame->end_of_stream;
    p_dst_frame->start_of_stream = p_src_frame->start_of_stream;

    p_dst_frame->ni_pict_type = p_src_frame->ni_pict_type;

    p_dst_frame->sei_total_len = p_src_frame->sei_total_len;
    p_dst_frame->sei_cc_offset = p_src_frame->sei_cc_offset;
    p_dst_frame->sei_cc_len = p_src_frame->sei_cc_len;
    p_dst_frame->sei_hdr_mastering_display_color_vol_offset =
        p_src_frame->sei_hdr_mastering_display_color_vol_offset;
    p_dst_frame->sei_hdr_mastering_display_color_vol_len =
        p_src_frame->sei_hdr_mastering_display_color_vol_len;
    p_dst_frame->sei_hdr_content_light_level_info_offset =
        p_src_frame->sei_hdr_content_light_level_info_offset;
    p_dst_frame->sei_hdr_content_light_level_info_len =
        p_src_frame->sei_hdr_content_light_level_info_len;
    p_dst_frame->sei_hdr_plus_offset = p_src_frame->sei_hdr_plus_offset;
    p_dst_frame->sei_hdr_plus_len = p_src_frame->sei_hdr_plus_len;
    p_dst_frame->sei_user_data_unreg_offset =
        p_src_frame->sei_user_data_unreg_offset;
    p_dst_frame->sei_user_data_unreg_len = p_src_frame->sei_user_data_unreg_len;
    p_dst_frame->sei_alt_transfer_characteristics_offset =
        p_src_frame->sei_alt_transfer_characteristics_offset;
    p_dst_frame->sei_alt_transfer_characteristics_len =
        p_src_frame->sei_alt_transfer_characteristics_len;
    p_dst_frame->vui_offset = p_src_frame->vui_offset;
    p_dst_frame->vui_len = p_src_frame->vui_len;

    p_dst_frame->pixel_format = p_src_frame->pixel_format;
    p_dst_frame->nb_aux_data = p_src_frame->nb_aux_data;

    p_dst_frame->color_primaries = p_src_frame->color_primaries;
    p_dst_frame->color_trc = p_src_frame->color_trc;
    p_dst_frame->color_space = p_src_frame->color_space;
    p_dst_frame->video_full_range_flag = p_src_frame->video_full_range_flag;
    p_dst_frame->aspect_ratio_idc = p_src_frame->aspect_ratio_idc;
    p_dst_frame->sar_width = p_src_frame->sar_width;
    p_dst_frame->sar_height = p_src_frame->sar_height;
    p_dst_frame->vui_num_units_in_tick = p_src_frame->vui_num_units_in_tick;
    p_dst_frame->vui_time_scale = p_src_frame->vui_time_scale;
    // copy aux data
    ret = ni_frame_copy_aux_data(p_dst_frame, p_src_frame);
    if (ret < 0) {
        ni_log(NI_LOG_ERROR, "Error: %s copy aux data failed\n", __func__);
        return ret;
    }
    // TBD: support custom sei

    return ret;
}

int frame_copy(ni_session_data_io_t *dst, const ni_session_data_io_t *src,
               bool force_copy) {
    int ret = NI_RETCODE_SUCCESS;
    ret = frame_copy_props(dst, src);
    if (ret < 0) {
        return ret;
    }

    ni_frame_t *p_dst_frame = &dst->data.frame;
    const ni_frame_t *p_src_frame = &src->data.frame;
    bool is_hw = false;
    niFrameSurface1_t *p_src_surface =
        (niFrameSurface1_t *)(p_src_frame->p_data[3]);
    if (p_src_surface && p_src_surface->ui16FrameIdx)
        is_hw = true;
    niFrameSurface1_t *p_dst_surface =
        (niFrameSurface1_t *)(p_dst_frame->p_data[3]);
    if (p_dst_surface && p_dst_surface->ui16FrameIdx) {
        if (force_copy) {
            ni_log(NI_LOG_INFO,
                   "Warning: %s dst frame already has hw surface, recycle then "
                   "force copy\n",
                   __func__);
            ni_hwframe_buffer_recycle2(p_dst_surface);
        } else {
            ni_log(
                NI_LOG_DEBUG,
                "Warning: %s dst frame already has hw surface, cannot copy\n",
                __func__);
            return NI_RETCODE_SUCCESS;
        }
    }
    if (is_hw) {
        ret = ni_frame_buffer_alloc_hwenc(p_dst_frame, p_src_frame->video_width,
                                          p_src_frame->video_height, 0);
        if (ret) {
            ni_log(NI_LOG_ERROR,
                   "Error: %s failed to allocate hw surface for dst frame, ret "
                   "= %d\n",
                   __func__, ret);
            return ret;
        }
        memcpy(p_dst_frame->p_data[3], p_src_frame->p_data[3],
               sizeof(niFrameSurface1_t));
    } else {
        // currently only points the dst frame's buffer to the src frame's
        // buffer. TBD: implement some mechanism like reference counting to
        // avoid double free
        p_dst_frame->dec_buf = p_src_frame->dec_buf;
    }
    return ret;
}

int prepare_enc_input_frame(ni_session_context_t *p_enc_ctx,
                            ni_session_data_io_t *p_dec_out_data,
                            ni_session_data_io_t *p_enc_in_data,
                            ni_pix_fmt_t src_fmt, int input_video_width,
                            int input_video_height) {
    // pointer to data struct to be sent
    ni_session_data_io_t *p_to_send = NULL;
    // frame pointer to data frame struct to be sent
    ni_frame_t *p_in_frame = NULL;
    ni_xcoder_params_t *api_params =
        (ni_xcoder_params_t *)p_enc_ctx->p_session_config;
    int is_semiplanar = get_pixel_planar(p_enc_ctx->pixel_format) ==
                        NI_PIXEL_PLANAR_FORMAT_SEMIPLANAR;
    int is_hwframe = p_enc_ctx->hw_action != NI_CODEC_HW_NONE;

    p_to_send = p_enc_in_data;
    p_in_frame = &(p_to_send->data.frame);
    p_in_frame->end_of_stream = p_dec_out_data->data.frame.end_of_stream;
    p_in_frame->ni_pict_type = 0;

    // reset encoder change data buffer
    memset(p_enc_ctx->enc_change_params, 0, sizeof(ni_encoder_change_params_t));

    // extra data starts with metadata header, and reset various aux data
    // size
    p_in_frame->extra_data_len = NI_APP_ENC_FRAME_META_DATA_SIZE;
    p_in_frame->roi_len = 0;
    p_in_frame->reconf_len = 0;
    p_in_frame->sei_total_len = 0;
    p_in_frame->force_pic_qp = 0;

    // collect encode reconfig and demo info and save them in the decode out
    // frame, to be used in the aux data prep and copy later
    prep_reconf_demo_data(p_enc_ctx, &(p_dec_out_data->data.frame));

    int dst_stride[NI_MAX_NUM_DATA_POINTERS] = {0};
    int dst_height_aligned[NI_MAX_NUM_DATA_POINTERS] = {0};
    bool alignment_2pass_wa =
         ((api_params->cfg_enc_params.lookAheadDepth ||
           api_params->cfg_enc_params.crf >= 0) &&
         ((p_enc_ctx->codec_format == NI_CODEC_FORMAT_H265) ||
          (p_enc_ctx->codec_format == NI_CODEC_FORMAT_AV1)));
    ni_get_hw_yuv420p_dim(input_video_width, input_video_height,
                          p_enc_ctx->bit_depth_factor, is_semiplanar,
                          dst_stride, dst_height_aligned);

    // ROI demo mode takes higher priority over aux data
    // Note: when ROI demo modes enabled, supply ROI map for the specified
    //       range frames, and 0 map for others
    if (api_params->roi_demo_mode && api_params->cfg_enc_params.roi_enable) {
        if (p_enc_ctx->frame_num > 90 && p_enc_ctx->frame_num < 300) {
            p_in_frame->roi_len = p_enc_ctx->roi_len;
        } else {
            p_in_frame->roi_len = 0;
        }
        // when ROI enabled, always have a data buffer for ROI
        // Note: this is handled separately from ROI through side/aux data
        p_in_frame->extra_data_len += p_enc_ctx->roi_len;
    }

    int should_send_sei_with_frame = ni_should_send_sei_with_frame(
        p_enc_ctx, p_in_frame->ni_pict_type, api_params);

    // data buffer for various SEI: HDR mastering display color volume, HDR
    // content light level, close caption, User data unregistered, HDR10+
    // etc.
    uint8_t mdcv_data[NI_MAX_SEI_DATA];
    uint8_t cll_data[NI_MAX_SEI_DATA];
    uint8_t cc_data[NI_MAX_SEI_DATA];
    uint8_t udu_data[NI_MAX_SEI_DATA];
    uint8_t hdrp_data[NI_MAX_SEI_DATA];

    // prep for auxiliary data (various SEI, ROI) in p_in_frame, based on
    // the data returned in decoded frame and also reconfig and demo modes
    // collected in prep_reconf_demo_data
    ni_enc_prep_aux_data(p_enc_ctx, p_in_frame, &(p_dec_out_data->data.frame),
                         p_enc_ctx->codec_format, should_send_sei_with_frame,
                         mdcv_data, cll_data, cc_data, udu_data, hdrp_data);

    p_in_frame->extra_data_len += p_in_frame->sei_total_len;

    // data layout requirement: leave space for reconfig data if at least
    // one of reconfig, SEI or ROI is present
    // Note: ROI is present when enabled, so use encode config flag instead
    //       of frame's roi_len as it can be 0 indicating a 0'd ROI map
    //       setting !
    if (p_in_frame->reconf_len || p_in_frame->sei_total_len ||
        (api_params->roi_demo_mode && api_params->cfg_enc_params.roi_enable)) {
        p_in_frame->extra_data_len += sizeof(ni_encoder_change_params_t);
    }

    if (!is_hwframe) {
        ni_encoder_sw_frame_buffer_alloc(
            api_params->cfg_enc_params.planar, p_in_frame, input_video_width,
            input_video_height, dst_stride,
            p_enc_ctx->codec_format == NI_CODEC_FORMAT_H264,
            (int)(p_in_frame->extra_data_len), alignment_2pass_wa);
        if (!p_in_frame->p_data[0]) {
            ni_log(NI_LOG_ERROR, "Error: cannot allocate YUV frame buffer!\n");
            return -1;
        }
    } else {
        ni_frame_buffer_alloc_hwenc(p_in_frame, input_video_width,
                                    input_video_height,
                                    (int)(p_in_frame->extra_data_len));
        if (!p_in_frame->p_data[3]) {
            ni_log(NI_LOG_ERROR, "Error: cannot allocate YUV frame buffer!\n");
            return -1;
        }
    }

    ni_log(NI_LOG_DEBUG,
           "p_dst alloc linesize = %d/%d/%d  src height=%d  "
           "dst height aligned = %d/%d/%d force_key_frame=%d, "
           "extra_data_len=%u"
           " sei_size=%u (hdr_content_light_level %u hdr_mastering_display_"
           "color_vol %u hdr10+ %u hrd %d) reconf_size=%u roi_size=%u "
           "force_pic_qp=%u udu_sei_size=%u "
           "use_cur_src_as_long_term_pic %u use_long_term_ref %u\n",
           dst_stride[0], dst_stride[1], dst_stride[2], input_video_height,
           dst_height_aligned[0], dst_height_aligned[1], dst_height_aligned[2],
           p_in_frame->force_key_frame, p_in_frame->extra_data_len,
           p_in_frame->sei_total_len,
           p_in_frame->sei_hdr_content_light_level_info_len,
           p_in_frame->sei_hdr_mastering_display_color_vol_len,
           p_in_frame->sei_hdr_plus_len, 0, /* hrd is 0 size for now */
           p_in_frame->reconf_len, p_in_frame->roi_len,
           p_in_frame->force_pic_qp, p_in_frame->sei_user_data_unreg_len,
           p_in_frame->use_cur_src_as_long_term_pic,
           p_in_frame->use_long_term_ref);

    uint8_t *p_src[NI_MAX_NUM_DATA_POINTERS];
    int src_stride[NI_MAX_NUM_DATA_POINTERS];
    int src_height[NI_MAX_NUM_DATA_POINTERS];

    src_height[0] = p_dec_out_data->data.frame.video_height;
    src_height[1] = src_height[2] = src_height[0] / 2;
    src_height[3] = 0;

    src_stride[0] =
        (int)(p_dec_out_data->data.frame.data_len[0]) / src_height[0];
    src_stride[1] =
        (int)(p_dec_out_data->data.frame.data_len[1]) / src_height[1];
    src_stride[2] = src_stride[1];
    if (is_semiplanar) {
        src_height[2] = 0;
        src_stride[2] = 0;
    }
    src_stride[3] = 0;

    p_src[0] = p_dec_out_data->data.frame.p_data[0];
    p_src[1] = p_dec_out_data->data.frame.p_data[1];
    p_src[2] = p_dec_out_data->data.frame.p_data[2];
    p_src[3] = p_dec_out_data->data.frame.p_data[3];

    if (!is_hwframe) { // YUV part of the encoder input data layout
        ni_copy_hw_yuv420p(
            (uint8_t **)(p_in_frame->p_data), p_src, input_video_width,
            input_video_height, p_enc_ctx->bit_depth_factor, is_semiplanar,
            ((ni_xcoder_params_t *)p_enc_ctx->p_session_config)
                ->cfg_enc_params.conf_win_right,
            dst_stride, dst_height_aligned, src_stride, src_height);
    } else {
        ni_copy_hw_descriptors((uint8_t **)(p_in_frame->p_data), p_src);
    }
    // auxiliary data part of the encoder input data layout
    ni_enc_copy_aux_data(p_enc_ctx, p_in_frame, &(p_dec_out_data->data.frame),
                         p_enc_ctx->codec_format, mdcv_data, cll_data, cc_data,
                         udu_data, hdrp_data, is_hwframe, is_semiplanar);
    p_in_frame->video_width = input_video_width;
    p_in_frame->video_height = input_video_height;

    p_in_frame->start_of_stream = 0;

    return 0;
}

/*!*****************************************************************************
 *  \brief  Send encoder input data, read from xstack output hwframe
 *
 *  \param
 *
 *  \return
 ******************************************************************************/
int encoder_send_data(ni_session_context_t *p_enc_ctx,
                      ni_session_data_io_t *p_in_data, int input_video_width,
                      int input_video_height, device_state_t *p_device_state,
                      int eos) {
    int oneSent;
    // frame is prepared in prepare_enc_input_frame
    ni_frame_t *p_in_frame = &(p_in_data->data.frame);

    ni_log(NI_LOG_DEBUG, "===> encoder_send_data <===\n");

    if (p_device_state->enc_eos_sent == 1) {
        ni_log(NI_LOG_DEBUG, "encoder_send_data: ALL data (incl. eos) sent "
                             "already!\n");
        return 0;
    }

    if (input_video_width < NI_MIN_WIDTH ||
        input_video_height < NI_MIN_HEIGHT) {
        ni_log(NI_LOG_ERROR, "Error %s: input resolution (%dx%d) is too small!\n",
               __func__, input_video_width, input_video_height);
        return NI_RETCODE_FAILURE;
    }

    if (p_device_state->enc_resend) {
        goto send_frame;
    }

    p_in_frame->start_of_stream = 0;
    if (!p_device_state->enc_sos_sent) {
        p_device_state->enc_sos_sent = 1;
        p_in_frame->start_of_stream = 1;
    }
    p_in_frame->end_of_stream = eos;
    p_in_frame->video_width = input_video_width;
    p_in_frame->video_height = input_video_height;
    if (eos) {
        ni_log(NI_LOG_DEBUG, "encoder_send_data: no new frame, eos!\n");
    }

send_frame:
    oneSent = (int)(p_in_frame->data_len[0] + p_in_frame->data_len[1] +
                    p_in_frame->data_len[2] + p_in_frame->data_len[3]);
    if (oneSent > 0 || p_in_frame->end_of_stream) {
        oneSent = ni_device_session_write(p_enc_ctx, p_in_data,
                                          NI_DEVICE_TYPE_ENCODER);
        p_in_frame->end_of_stream = 0;
    } else {
        return NI_RETCODE_SUCCESS;
    }

    if (oneSent < 0) {
        ni_log(NI_LOG_ERROR, "Error: encoder_send_data ret %d\n", oneSent);
        p_device_state->enc_resend = 1;
        return NI_RETCODE_FAILURE;
    } else if (oneSent == 0) {
        if (p_device_state->enc_eos_sent == 0 && p_enc_ctx->ready_to_close) {
            p_device_state->enc_resend = 0;
            p_device_state->enc_eos_sent = 1;
        } else {
            p_device_state->enc_resend = 1;
            ni_log(NI_LOG_DEBUG, "encoder_send_data needs to resend\n");
            return NI_TEST_RETCODE_EAGAIN;
        }
    } else {
        p_device_state->enc_resend = 0;

        ni_log(NI_LOG_DEBUG, "encoder_send_data: total sent data size=%u\n",
               p_in_frame->data_len[3]);

        ni_log(NI_LOG_DEBUG, "encoder_send_data: success\n");

        if (p_enc_ctx->ready_to_close) {
            p_device_state->enc_eos_sent = 1;
        }
    }

    return 0;
}

int encoder_send_data2(ni_session_context_t *p_enc_ctx,
                       ni_session_data_io_t *p_dec_out_data,
                       ni_session_data_io_t *p_enc_in_data,
                       int input_video_width, int input_video_height,
                       device_state_t *p_device_state)
{
    int oneSent;
    // pointer to data struct to be sent
    ni_session_data_io_t *p_to_send = NULL;
    // frame pointer to data frame struct to be sent
    ni_frame_t *p_in_frame = NULL;
    ni_xcoder_params_t *api_params =
        (ni_xcoder_params_t *)p_enc_ctx->p_session_config;
    int is_semiplanar = get_pixel_planar(p_enc_ctx->pixel_format) == NI_PIXEL_PLANAR_FORMAT_SEMIPLANAR;
    int is_hwframe = p_enc_ctx->hw_action != NI_CODEC_HW_NONE;

    ni_log(NI_LOG_DEBUG, "===> encoder_send_data2 <===\n");

    if (p_device_state->enc_eos_sent == 1) {
        ni_log(NI_LOG_DEBUG, "encoder_send_data2: ALL data (incl. eos) sent "
               "already!\n");
        return 0;
    }

    if (input_video_width < NI_MIN_WIDTH ||
        input_video_height < NI_MIN_HEIGHT) {
        ni_log(NI_LOG_ERROR, "Error %s: input resolution (%dx%d) is too small!\n",
               __func__, input_video_width, input_video_height);
        return NI_RETCODE_FAILURE;
    }

    // frame resend
    if (p_device_state->enc_resend) {
        p_to_send = p_enc_in_data;
        p_in_frame = &(p_to_send->data.frame);
        goto send_frame;
    }

    // otherwise have to pad/crop the source and copy to a new frame struct
    // and prep for the SEI aux data
    p_to_send = p_enc_in_data;
    p_in_frame = &(p_to_send->data.frame);
    p_in_frame->end_of_stream = p_dec_out_data->data.frame.end_of_stream;
    p_in_frame->ni_pict_type = 0;

    // reset encoder change data buffer
    memset(p_enc_ctx->enc_change_params, 0,
            sizeof(ni_encoder_change_params_t));

    // extra data starts with metadata header, and reset various aux data
    // size
    p_in_frame->extra_data_len = NI_APP_ENC_FRAME_META_DATA_SIZE;
    p_in_frame->roi_len = 0;
    p_in_frame->reconf_len = 0;
    p_in_frame->sei_total_len = 0;
    p_in_frame->force_pic_qp = 0;

    // collect encode reconfig and demo info and save them in the decode out
    // frame, to be used in the aux data prep and copy later
    prep_reconf_demo_data(p_enc_ctx, &(p_dec_out_data->data.frame));

    int dst_stride[NI_MAX_NUM_DATA_POINTERS] = {0};
    int dst_height_aligned[NI_MAX_NUM_DATA_POINTERS] = {0};
    bool alignment_2pass_wa = (
                                    (api_params->cfg_enc_params.lookAheadDepth ||
                                    api_params->cfg_enc_params.crf >= 0) &&
                                    (p_enc_ctx->codec_format == NI_CODEC_FORMAT_H265 ||
                                    p_enc_ctx->codec_format == NI_CODEC_FORMAT_AV1));
    ni_get_hw_yuv420p_dim(input_video_width, input_video_height,
                            p_enc_ctx->bit_depth_factor, is_semiplanar,
                            dst_stride, dst_height_aligned);

    // ROI demo mode takes higher priority over aux data
    // Note: when ROI demo modes enabled, supply ROI map for the specified
    //       range frames, and 0 map for others
    if (api_params->roi_demo_mode && api_params->cfg_enc_params.roi_enable) {
        if (p_enc_ctx->frame_num > 90 && p_enc_ctx->frame_num < 300) {
            p_in_frame->roi_len = p_enc_ctx->roi_len;
        } else {
            p_in_frame->roi_len = 0;
        }
        // when ROI enabled, always have a data buffer for ROI
        // Note: this is handled separately from ROI through side/aux data
        p_in_frame->extra_data_len += p_enc_ctx->roi_len;
    }

    int should_send_sei_with_frame = ni_should_send_sei_with_frame(
                                            p_enc_ctx, p_in_frame->ni_pict_type, api_params);

    // data buffer for various SEI: HDR mastering display color volume, HDR
    // content light level, close caption, User data unregistered, HDR10+
    // etc.
    uint8_t mdcv_data[NI_MAX_SEI_DATA];
    uint8_t cll_data[NI_MAX_SEI_DATA];
    uint8_t cc_data[NI_MAX_SEI_DATA];
    uint8_t udu_data[NI_MAX_SEI_DATA];
    uint8_t hdrp_data[NI_MAX_SEI_DATA];

    // prep for auxiliary data (various SEI, ROI) in p_in_frame, based on
    // the data returned in decoded frame and also reconfig and demo modes
    // collected in prep_reconf_demo_data
    ni_enc_prep_aux_data(
        p_enc_ctx, p_in_frame, &(p_dec_out_data->data.frame),
        p_enc_ctx->codec_format, should_send_sei_with_frame, mdcv_data,
        cll_data, cc_data, udu_data, hdrp_data);

    p_in_frame->extra_data_len += p_in_frame->sei_total_len;

    // data layout requirement: leave space for reconfig data if at least
    // one of reconfig, SEI or ROI is present
    // Note: ROI is present when enabled, so use encode config flag instead
    //       of frame's roi_len as it can be 0 indicating a 0'd ROI map
    //       setting !
    if (p_in_frame->reconf_len || p_in_frame->sei_total_len ||
        (api_params->roi_demo_mode &&
            api_params->cfg_enc_params.roi_enable)) {
        p_in_frame->extra_data_len += sizeof(ni_encoder_change_params_t);
    }

    if (!is_hwframe) {
        ni_encoder_sw_frame_buffer_alloc(api_params->cfg_enc_params.planar,
                                            p_in_frame, input_video_width, input_video_height, dst_stride,
                                            p_enc_ctx->codec_format == NI_CODEC_FORMAT_H264,
                                            (int)(p_in_frame->extra_data_len), alignment_2pass_wa);
        if (!p_in_frame->p_data[0]) {
            ni_log(NI_LOG_ERROR, "Error: cannot allocate YUV frame buffer!");
            return -1;
        }
    } else {
        ni_frame_buffer_alloc_hwenc(p_in_frame, input_video_width,
                                    input_video_height,
                                    (int)(p_in_frame->extra_data_len));
        if (!p_in_frame->p_data[3]) {
            ni_log(NI_LOG_ERROR, "Error: cannot allocate YUV frame buffer!");
            return -1;
        }
    }

    ni_log(NI_LOG_DEBUG,
            "p_dst alloc linesize = %d/%d/%d  src height=%d  "
            "dst height aligned = %d/%d/%d force_key_frame=%d, "
            "extra_data_len=%u"
            " sei_size=%u (hdr_content_light_level %u hdr_mastering_display_"
            "color_vol %u hdr10+ %u hrd %d) reconf_size=%u roi_size=%u "
            "force_pic_qp=%u udu_sei_size=%u "
            "use_cur_src_as_long_term_pic %u use_long_term_ref %u\n",
            dst_stride[0], dst_stride[1], dst_stride[2], input_video_height,
            dst_height_aligned[0], dst_height_aligned[1], dst_height_aligned[2],
            p_in_frame->force_key_frame, p_in_frame->extra_data_len,
            p_in_frame->sei_total_len,
            p_in_frame->sei_hdr_content_light_level_info_len,
            p_in_frame->sei_hdr_mastering_display_color_vol_len,
            p_in_frame->sei_hdr_plus_len, 0, /* hrd is 0 size for now */
            p_in_frame->reconf_len, p_in_frame->roi_len,
            p_in_frame->force_pic_qp, p_in_frame->sei_user_data_unreg_len,
            p_in_frame->use_cur_src_as_long_term_pic,
            p_in_frame->use_long_term_ref);

    uint8_t *p_src[NI_MAX_NUM_DATA_POINTERS];
    int src_stride[NI_MAX_NUM_DATA_POINTERS];
    int src_height[NI_MAX_NUM_DATA_POINTERS];

    src_height[0] = p_dec_out_data->data.frame.video_height;
    src_height[1] = src_height[2] = src_height[0] / 2;
    src_height[3] = 0;

    src_stride[0] =
        (int)(p_dec_out_data->data.frame.data_len[0]) / src_height[0];
    src_stride[1] =
        (int)(p_dec_out_data->data.frame.data_len[1]) / src_height[1];
    src_stride[2] = src_stride[1];
    if (is_semiplanar) {
        src_height[2] = 0;
        src_stride[2] = 0;
    }
    src_stride[3] = 0;

    p_src[0] = p_dec_out_data->data.frame.p_data[0];
    p_src[1] = p_dec_out_data->data.frame.p_data[1];
    p_src[2] = p_dec_out_data->data.frame.p_data[2];
    p_src[3] = p_dec_out_data->data.frame.p_data[3];

    if (!is_hwframe) {
        // YUV part of the encoder input data layout
        ni_copy_hw_yuv420p(
            (uint8_t **)(p_in_frame->p_data), p_src, input_video_width,
            input_video_height, p_enc_ctx->bit_depth_factor, is_semiplanar,
            ((ni_xcoder_params_t *)p_enc_ctx->p_session_config)
            ->cfg_enc_params.conf_win_right,
            dst_stride, dst_height_aligned, src_stride, src_height);
    } else {
        ni_copy_hw_descriptors((uint8_t **)(p_in_frame->p_data), p_src);
    }
    // auxiliary data part of the encoder input data layout
    ni_enc_copy_aux_data(p_enc_ctx, p_in_frame,
                            &(p_dec_out_data->data.frame),
                            p_enc_ctx->codec_format, mdcv_data, cll_data,
                            cc_data, udu_data, hdrp_data, is_hwframe,
                            is_semiplanar);


    p_in_frame->video_width = input_video_width;
    p_in_frame->video_height = input_video_height;

    p_in_frame->start_of_stream = 0;
    if (!p_device_state->enc_sos_sent) {
        p_device_state->enc_sos_sent = 1;
        p_in_frame->start_of_stream = 1;
    }
    // p_in_frame->end_of_stream = 0;

send_frame:
    oneSent = (int)(p_in_frame->data_len[0] + p_in_frame->data_len[1] +
                    p_in_frame->data_len[2] + p_in_frame->data_len[3]);

    if (oneSent > 0 || p_in_frame->end_of_stream) {
        oneSent = ni_device_session_write(p_enc_ctx, p_to_send,
                                          NI_DEVICE_TYPE_ENCODER);
        p_in_frame->end_of_stream = 0;
    } else {
        return 0;
    }

    if (oneSent < 0) {
        ni_log(NI_LOG_ERROR, "Error: encoder_send_data2\n");
        p_device_state->enc_resend = 1;
        return -1;
    } else if (oneSent == 0) {
        if (p_device_state->enc_eos_sent == 0 && p_enc_ctx->ready_to_close) {
            p_device_state->enc_resend = 0;
            p_device_state->enc_eos_sent = 1;
        } else {
            p_device_state->enc_resend = 1;
            return NI_TEST_RETCODE_EAGAIN;
        }
    } else {
        p_device_state->enc_resend = 0;

        if (p_enc_ctx->ready_to_close) {
            p_device_state->enc_eos_sent = 1;
        }
        ni_log(NI_LOG_DEBUG, "encoder_send_data2: success\n");
    }

    return 0;
}

/*!*****************************************************************************
 *  \brief  Receive output data from encoder
 *
 *  \param  p_in_data is passed in to specify new frame resolution upon sequence
 *          change
 *
 *  \return 0 - success got packet
 *          1 - received eos
 *          2 - got nothing, need retry
 *          -1 - failure
 ******************************************************************************/
int encoder_receive_data(ni_session_context_t *p_enc_ctx,
                         ni_session_data_io_t *p_out_data, FILE *p_file,
                         volatile unsigned int *nb_pkts_received) {
    int packet_size = NI_MAX_TX_SZ;
    int rc = 0;
    int end_flag = 0;
    int rx_size = 0;
    ni_packet_t *p_out_pkt = &(p_out_data->data.packet);
    int meta_size = p_enc_ctx->meta_size;
    ni_xcoder_params_t *p_api_param =
        (ni_xcoder_params_t *)p_enc_ctx->p_session_config;

    ni_log(NI_LOG_TRACE, "===> encoder_receive_data <===\n");

    if (NI_INVALID_SESSION_ID == p_enc_ctx->session_id ||
        NI_INVALID_DEVICE_HANDLE == p_enc_ctx->blk_io_handle) {
        ni_log(NI_LOG_DEBUG, "encode session not opened yet, return\n");
        return NI_RETCODE_SUCCESS;
    }

receive_data:
    rc = ni_packet_buffer_alloc(p_out_pkt, packet_size);
    if (rc != NI_RETCODE_SUCCESS) {
        ni_log(NI_LOG_ERROR, "Error: malloc packet failed, ret = %d!\n", rc);
        return NI_RETCODE_FAILURE;
    }

    rc = ni_device_session_read(p_enc_ctx, p_out_data, NI_DEVICE_TYPE_ENCODER);

    end_flag = p_out_pkt->end_of_stream;
    rx_size = rc;

    ni_log(NI_LOG_TRACE, "encoder_receive_data: received data size=%d\n",
           rx_size);

    if (rx_size > meta_size) {
        if (p_file &&
            (fwrite((uint8_t *)p_out_pkt->p_data + meta_size,
                    p_out_pkt->data_len - meta_size, 1, p_file) != 1)) {
            ni_log(NI_LOG_ERROR, "Error: writing data %u bytes error!\n",
                    p_out_pkt->data_len - meta_size);
            ni_log(NI_LOG_ERROR, "Error: ferror rc = %d\n", ferror(p_file));
        }

        if (0 == p_enc_ctx->pkt_num) {
            p_enc_ctx->pkt_num = 1;
            ni_log(NI_LOG_DEBUG,
                   "got encoded stream header, keep reading ..\n");
            goto receive_data;
        }
        (*nb_pkts_received)++;

        ni_log(NI_LOG_DEBUG, "Got:   Packets= %u\n", *nb_pkts_received);
    } else if (rx_size != 0) {
        ni_log(NI_LOG_ERROR, "Error: received %d bytes, <= metadata size %d!\n",
                rx_size, meta_size);
        return NI_RETCODE_FAILURE;
    } else if (!end_flag && p_api_param->low_delay_mode) {
        ni_log(NI_LOG_DEBUG, "low delay mode and NO pkt, keep reading ..\n");
        if (p_enc_ctx->frame_num <= p_enc_ctx->pkt_num) {
            // sleep to switch into encoder send thread
            ni_usleep(100);
        }
        goto receive_data;
    }

    if (end_flag) {
        ni_log(NI_LOG_DEBUG, "Encoder Receiving done.\n");
        return NI_TEST_RETCODE_END_OF_STREAM;
    } else if (0 == rx_size) {
        return NI_TEST_RETCODE_EAGAIN;
    }
    ni_log(NI_LOG_DEBUG, "encoder_receive_data: success\n");

    return NI_RETCODE_SUCCESS;
}

/*!*****************************************************************************
 *  \brief  Send decoder input data
 *
 *  \param
 *
 *  \return
 ******************************************************************************/
int decoder_send_data(ni_session_context_t *p_dec_ctx,
                      ni_session_data_io_t *p_in_data,
                      device_state_t *p_device_state,
                      ni_file_reader_t *file_reader, void *stream_info,
                      int width, int height, uint8_t *tmp_buf) {

    if (p_device_state->dec_eos_sent) {
        ni_log(NI_LOG_DEBUG, "decoder_send_data: ALL data (incl. eos) sent "
                             "already!\n");
        return NI_RETCODE_SUCCESS;
    }

    uint8_t *tmp_buf_ptr = tmp_buf;
    int packet_size = 0;
    uint32_t frame_pkt_size = 0, nal_size;
    int nal_type = -1;
    int tx_size = 0;
    uint32_t send_size = 0;
    int new_packet = 0;
    int saved_prev_size = 0;
    int32_t frame_num = -1, curr_frame_num;
    unsigned int first_mb_in_slice = 0;
    ni_packet_t *p_in_pkt = &(p_in_data->data.packet);
    int retval = NI_RETCODE_SUCCESS;

    ni_log(NI_LOG_DEBUG, "===> decoder_send_data <===\n");
    if (p_device_state->dec_eos_sent) {
        ni_log(NI_LOG_DEBUG, "decoder_send_data: ALL data (incl. eos) sent "
               "already!\n");
        LRETURN;
    }

    if (0 == p_in_pkt->data_len) {
        memset(p_in_pkt, 0, sizeof(ni_packet_t));

        if (NI_CODEC_FORMAT_H264 == p_dec_ctx->codec_format) {
            ni_h264_sps_t *sps;
            sps = (ni_h264_sps_t *)stream_info;
            // send whole encoded packet which ends with a slice NAL
            while ((nal_size = find_h264_next_nalu(tmp_buf_ptr, &nal_type,
                                                   file_reader)) > 0) {
                frame_pkt_size += nal_size;
                tmp_buf_ptr += nal_size;
                ni_log(NI_LOG_DEBUG, "%s nal %d  nal_size %d\n", __func__,
                       nal_type, nal_size);

                // save parsed out sps/pps as stream headers in the decode
                // session
                if (H264_NAL_PPS == nal_type) {
                    if (NI_RETCODE_SUCCESS !=
                        ni_device_dec_session_save_hdrs(p_dec_ctx, tmp_buf,
                                                        frame_pkt_size)) {
                        ni_log(NI_LOG_ERROR,
                               "decoder_send_data: save_hdr failed!\n");
                    }
                }

                if (H264_NAL_SLICE == nal_type ||
                    H264_NAL_IDR_SLICE == nal_type) {
                    if (!parse_h264_slice_header(tmp_buf_ptr - nal_size,
                                                 nal_size, sps, &curr_frame_num,
                                                 &first_mb_in_slice)) {
                        if (-1 == frame_num) {
                            // first slice, continue to check
                            frame_num = curr_frame_num;
                        } else if (curr_frame_num != frame_num ||
                                   0 == first_mb_in_slice) {
                            // this slice has diff. frame_num or
                            // first_mb_in_slice addr is 0: not the same frame
                            // and return
                            rewind_data_buf_pos_by(file_reader, nal_size);
                            frame_pkt_size -= nal_size;
                            break;
                        }
                        // this slice is in the same frame, so continue to check
                        // and see if there is more
                    } else {
                        ni_log(NI_LOG_ERROR,
                               "decoder_send_data: parse_slice_header error "
                               "NAL type %d size %u, continue\n",
                               nal_type, nal_size);
                    }
                } else if (-1 != frame_num) {
                    // already got a slice and this is non-slice NAL: return
                    rewind_data_buf_pos_by(file_reader, nal_size);
                    frame_pkt_size -= nal_size;
                    break;
                }
                // otherwise continue until a slice is found
            } // while there is still NAL
        } else if (NI_CODEC_FORMAT_H265 == p_dec_ctx->codec_format) {
            while ((nal_size = find_h265_next_nalu(tmp_buf_ptr, &nal_type,
                                                   file_reader)) > 0) {
                frame_pkt_size += nal_size;
                tmp_buf_ptr += nal_size;
                ni_log(NI_LOG_DEBUG, "%s nal_type %d nal_size %d\n", __func__,
                       nal_type, nal_size);

                if (nal_type == HEVC_NAL_VPS || nal_type == HEVC_NAL_SPS ||
                    nal_type == HEVC_NAL_PPS) { // save vps, sps, pps
                    if (nal_type == 32 && p_dec_ctx->prev_size > 0) {
                        // sequence change situation, replace previous
                        // p_leftover with new headers
                        p_dec_ctx->prev_size = 0;
                    }
                    // copy the nal to the end of p_leftover
                    memcpy(p_dec_ctx->p_leftover + p_dec_ctx->prev_size,
                           tmp_buf_ptr - nal_size, nal_size);
                    p_dec_ctx->prev_size += nal_size;
                }

                if (nal_type >= 0 && nal_type <= 23) { // vcl units
                    ni_log(NI_LOG_DEBUG, "%s send vcl_nal %d nal_size %d\n",
                           __func__, nal_type, nal_size);
                    break;
                }
            }
        } else if (NI_CODEC_FORMAT_VP9 == p_dec_ctx->codec_format) {
            while ((packet_size = find_vp9_next_packet(tmp_buf_ptr, stream_info,
                                                       file_reader)) > 0) {
                frame_pkt_size += packet_size;
                ni_log(NI_LOG_DEBUG, "%s vp9 packet_size %d\n", __func__,
                       packet_size);
                break;
            }
        }
        ni_log(NI_LOG_DEBUG, "decoder_send_data * frame_pkt_size %d\n",
               frame_pkt_size);

        p_in_pkt->p_data = NULL;
        p_in_pkt->data_len = frame_pkt_size;

        if (frame_pkt_size + p_dec_ctx->prev_size > 0) {
            ni_packet_buffer_alloc(p_in_pkt,
                                   (int)frame_pkt_size + p_dec_ctx->prev_size);
        }

        new_packet = 1;
        send_size = frame_pkt_size + p_dec_ctx->prev_size;
        saved_prev_size = p_dec_ctx->prev_size;
    } else {
        send_size = p_in_pkt->data_len;
    }

    p_in_pkt->start_of_stream = 0;
    if (!p_device_state->dec_sos_sent) {
        p_in_pkt->start_of_stream = 1;
        p_device_state->dec_sos_sent = 1;
    }
    p_in_pkt->end_of_stream = 0;
    p_in_pkt->video_width = width;
    p_in_pkt->video_height = height;

    if (send_size == 0) {
        if (new_packet) {
            send_size =
                ni_packet_copy(p_in_pkt->p_data, tmp_buf, 0,
                               p_dec_ctx->p_leftover, &p_dec_ctx->prev_size);
        }
        p_in_pkt->data_len = send_size;

        if (file_reader->curr_found_pos) {
            p_in_pkt->end_of_stream = 1;
            ni_log(NI_LOG_DEBUG, "Sending p_last packet (size %u) + eos\n",
                   p_in_pkt->data_len);
        }
    } else {
        if (new_packet) {
            send_size =
                ni_packet_copy(p_in_pkt->p_data, tmp_buf, frame_pkt_size,
                               p_dec_ctx->p_leftover, &p_dec_ctx->prev_size);
            // p_in_pkt->data_len is the actual packet size to be sent to
            // decoder
            p_in_pkt->data_len += saved_prev_size;
        }
    }

    tx_size =
        ni_device_session_write(p_dec_ctx, p_in_data, NI_DEVICE_TYPE_DECODER);

    if (tx_size < 0) {
        // Error
        ni_log(NI_LOG_ERROR, "Error: sending data error. rc:%d\n", tx_size);
        retval = NI_RETCODE_FAILURE;
        LRETURN;
    } else if (tx_size == 0) {
        ni_log(NI_LOG_DEBUG, "0 byte sent this time, sleep and will re-try.\n");
        ni_usleep(200);
    }

    if (p_dec_ctx->ready_to_close) {
        p_device_state->dec_eos_sent = 1;
    }

    if (tx_size > 0) {
        ni_log(NI_LOG_DEBUG, "decoder_send_data: reset packet_buffer.\n");
        ni_packet_buffer_free(p_in_pkt);
    }

    retval = NI_RETCODE_SUCCESS;

end:
    return retval;
}

/*!*****************************************************************************
 *  \brief  Receive decoded output data from decoder
 *
 *  \param
 *
 *  \return 0: got YUV frame;  1: end-of-stream;  2: got nothing
 ******************************************************************************/
int decoder_receive_data(ni_session_context_t *p_dec_ctx,
                         ni_session_data_io_t *p_out_data,
                         device_state_t *p_device_state, int output_video_width,
                         int output_video_height, FILE *p_file) {
    int rc = NI_RETCODE_FAILURE;
    int end_flag = 0;
    int rx_size = 0;
    bool b_is_hwframe = p_dec_ctx->hw_action;
    ni_frame_t *p_out_frame = &(p_out_data->data.frame);
    int width, height;
    // In decoder session read function it will allocate the actual YUV
    // transfer size for the very first read. And the pixel format of session
    // context would be set as well. So it does not matter with the planar
    // format for the first call of this function.
    int is_planar = get_pixel_planar(p_dec_ctx->pixel_format) ==
                    NI_PIXEL_PLANAR_FORMAT_PLANAR;

    ni_log(NI_LOG_DEBUG,
           "===> decoder_receive_data hwframe %d pixel_format %d <===\n",
           b_is_hwframe, p_dec_ctx->pixel_format);

    if (p_device_state->dec_eos_received) {
        ni_log(NI_LOG_DEBUG,
               "decoder_receive_data eos received already, Done!\n");
        rc = 2;
        LRETURN;
    }

    // prepare memory buffer for receiving decoded frame
    width = p_dec_ctx->actual_video_width > 0
                ? (int)(p_dec_ctx->actual_video_width)
                : output_video_width;
    height = p_dec_ctx->active_video_height > 0
                 ? (int)(p_dec_ctx->active_video_height)
                 : output_video_height;

    // allocate memory only after resolution is known (for buffer pool set up)
    int alloc_mem =
        (p_dec_ctx->active_video_width > 0 && p_dec_ctx->active_video_height > 0
             ? 1
             : 0);
    if (!b_is_hwframe) {
        rc = ni_decoder_frame_buffer_alloc(
            p_dec_ctx->dec_fme_buf_pool, &(p_out_data->data.frame), alloc_mem,
            width, height, p_dec_ctx->codec_format == NI_CODEC_FORMAT_H264,
            p_dec_ctx->bit_depth_factor, is_planar);
        if (NI_RETCODE_SUCCESS != rc) {
            LRETURN;
        }
        rx_size = ni_device_session_read(p_dec_ctx, p_out_data,
                                         NI_DEVICE_TYPE_DECODER);
    } else {
        rc = ni_frame_buffer_alloc(
            &(p_out_data->data.frame), width, height,
            p_dec_ctx->codec_format == NI_CODEC_FORMAT_H264, 1,
            p_dec_ctx->bit_depth_factor,
            3 /*3 is max supported hwframe output count per frame*/, is_planar);
        if (NI_RETCODE_SUCCESS != rc) {
            LRETURN;
        }
        rx_size = ni_device_session_read_hwdesc(p_dec_ctx, p_out_data,
                                                NI_DEVICE_TYPE_DECODER);
    }

    // the actual pix_fmt is known and updated in ctx only after the first
    // frame is decoded, so check/update it here again to be used below
    is_planar = get_pixel_planar(p_dec_ctx->pixel_format) ==
                NI_PIXEL_PLANAR_FORMAT_PLANAR;

    end_flag = p_out_frame->end_of_stream;

    if (rx_size < 0) {
        ni_log(NI_LOG_ERROR, "Error: receiving data error. rc:%d\n", rx_size);
        if (!b_is_hwframe) {
            ni_decoder_frame_buffer_free(&(p_out_data->data.frame));
        } else {
            ni_frame_buffer_free(&(p_out_data->data.frame));
        }
        rc = NI_RETCODE_FAILURE;
        LRETURN;
    } else if (rx_size > 0) {
        ni_log(NI_LOG_DEBUG, "Got frame # %" PRIu64 " bytes %d\n",
               p_dec_ctx->frame_num, rx_size);

        ni_dec_retrieve_aux_data(p_out_frame);
    }
    // rx_size == 0 means no decoded frame is available now
    if (rx_size > 0 && p_file) {
        write_rawvideo_data(p_file, p_dec_ctx->active_video_width,
                            p_dec_ctx->active_video_height,
                            p_dec_ctx->pixel_format, p_out_frame);
    }

    if (end_flag) {
        ni_log(NI_LOG_DEBUG, "Decoder Receiving done.\n");
        p_device_state->dec_eos_received = 1;
        rc = 1;
    } else if (0 == rx_size) {
        rc = 2;
    }

    ni_log(NI_LOG_DEBUG, "decoder_receive_data: success\n");

end:
    return rc;
}

int hwdl_frame(ni_session_context_t *p_ctx,
               ni_session_data_io_t *p_session_data, ni_frame_t *p_src_frame,
               ni_pix_fmt_t output_format) {
    niFrameSurface1_t *src_surf = (niFrameSurface1_t *)(p_src_frame->p_data[3]);
    int ret = 0;

    ret = ni_frame_buffer_alloc_dl(&(p_session_data->data.frame),
                                   src_surf->ui16width, src_surf->ui16height,
                                   output_format);

    if (ret != NI_RETCODE_SUCCESS) {
        return NI_RETCODE_ERROR_MEM_ALOC;
    }

    p_ctx->is_auto_dl = false;
    ret = ni_device_session_hwdl(p_ctx, p_session_data, src_surf);
    if (ret <= 0) {
        ni_log(NI_LOG_ERROR, "ni_device_session_hwdl failed, ret = %d\n", ret);
        ni_frame_buffer_free(&p_session_data->data.frame);
        return ret;
    }
    return ret;
}

int hwup_frame(ni_session_context_t *p_upl_ctx, ni_session_data_io_t *p_sw_data,
               ni_session_data_io_t *p_hw_data, int pfs, int frame_offset,
               ni_file_reader_t *file_reader, int *eos, void *p_yuv_buf) {
    int ret, chunk_size, frame_size;
    frame_size = calc_swframe_size(p_upl_ctx->active_video_width,
                                   p_upl_ctx->active_video_height,
                                   p_upl_ctx->pixel_format);

    chunk_size = read_one_swframe(pfs, file_reader, p_yuv_buf, frame_size);
    ni_log(NI_LOG_INFO, "%s: read frame_size %d chunk size %d\n", __func__, frame_size, chunk_size);
    if (chunk_size < 0) {
        ni_log(NI_LOG_ERROR, "Error: read yuv file error\n");
        return NI_RETCODE_FAILURE;
    } else if (chunk_size == 0) {
        *eos = 1;
        ni_log(NI_LOG_ERROR, "%s: read chunk size 0, eos!\n", __func__);
    } else {
        *eos = 0;
    }

    // need to have the hwframe before open encoder
    ret = upload_send_data_get_desc2(p_upl_ctx, p_sw_data, p_hw_data,
                                     p_upl_ctx->active_video_width,
                                     p_upl_ctx->active_video_height,
                                     *eos ? NULL : p_yuv_buf);
    if (p_upl_ctx->status == NI_RETCODE_NVME_SC_WRITE_BUFFER_FULL) {
        ni_log(NI_LOG_DEBUG, "No space to write to, try to read a packet\n");
        //file was read so reset read pointer and try again
        file_reader->data_left_size += chunk_size;
        lseek(pfs, frame_size * frame_offset, SEEK_SET);
    } else if (ret) {
        ni_log(NI_LOG_ERROR, "Error: upload frame error\n");
    }

    return ret;
}

/*!*****************************************************************************
 *  \brief  Write hwdl data to files.
 *
 *  \param
 *
 *  \return 0 if successful, < 0 otherwise
 *
 ******************************************************************************/
int write_rawvideo_data(FILE *p_file, int width, int height,
                        ni_pix_fmt_t format, ni_frame_t *p_out_frame) {
    int src_linesize[4], src_height[4], dst_linesize[4];
    int i, j;
    uint8_t *src;
    if (p_file && p_out_frame) {
        switch (format) {
        case NI_PIX_FMT_YUV420P: {
            src_linesize[0] = ALIGN(width, 128);
            src_linesize[1] = ALIGN(width / 2, 128);
            src_linesize[2] = src_linesize[1];
            src_linesize[3] = 0;

            dst_linesize[0] = ALIGN(width, 2);
            dst_linesize[1] = ALIGN(width / 2, 2);
            dst_linesize[2] = dst_linesize[1];
            dst_linesize[3] = 0;

            src_height[0] = height;
            src_height[1] = ALIGN(height, 2) / 2;
            src_height[2] = src_height[1];
            src_height[3] = 0;
            break;
        }
        case NI_PIX_FMT_RGBA:
        case NI_PIX_FMT_BGRA:
        case NI_PIX_FMT_ARGB:
        case NI_PIX_FMT_ABGR:
        case NI_PIX_FMT_BGR0: {
            src_linesize[0] = ALIGN(width, 16) * 4;
            src_linesize[1] = 0;
            src_linesize[2] = 0;
            src_linesize[3] = 0;

            dst_linesize[0] = ALIGN(width, 2) * 4;
            dst_linesize[1] = 0;
            dst_linesize[2] = 0;
            dst_linesize[3] = 0;

            src_height[0] = height;
            src_height[1] = 0;
            src_height[2] = 0;
            src_height[3] = 0;
            break;
        }
        case NI_PIX_FMT_BGRP: {
            src_linesize[0] = ALIGN(width, 32);
            src_linesize[1] = ALIGN(width, 32);
            src_linesize[2] = ALIGN(width, 32);
            src_linesize[3] = 0;

            dst_linesize[0] = ALIGN(width, 2);
            dst_linesize[1] = ALIGN(width, 2);
            dst_linesize[2] = ALIGN(width, 2);
            dst_linesize[3] = 0;

            src_height[0] = height;
            src_height[1] = height;
            src_height[2] = height;
            src_height[3] = 0;
            break;
        }
        default:
            ni_log(NI_LOG_ERROR, "Unsupported format %d\n", format);
            exit(EXIT_FAILURE);
        }
        for (i = 0; i < 4; i++) {
            if (!src_linesize[i]) {
                break;
            }
            src = p_out_frame->p_data[i];
            for (j = 0; j < src_height[i]; j++) {
                if (fwrite(src, dst_linesize[i], 1, p_file) != 1) {
                    ni_log(NI_LOG_ERROR, "Error:  writing data plane %d: height %d ferror rc = %d\n",
                           i, src_height[i], ferror(p_file));
                }
                src += src_linesize[i];
            }
        }
        if (fflush(p_file)) {
            ni_log(NI_LOG_ERROR,
                    "Error: writing data frame flush failed! errno %d\n",
                    errno);
        }
    }
    return 0;
}

uint64_t ni_get_utime(void)
{
    struct timeval tv;
    (void)ni_gettimeofday(&tv, NULL);
    return (tv.tv_sec * 1000000LL + tv.tv_usec);
}


// reference HW frame
void ni_hw_frame_ref(ni_hwframe_ref_t *g_hwframe_pool,
                     const niFrameSurface1_t *p_surface)
{
    uint16_t hwframe_index;

    if (!p_surface) {
        return;
    }

    hwframe_index = p_surface->ui16FrameIdx;
    g_hwframe_pool[hwframe_index].ref_cnt++;
    if (1 == g_hwframe_pool[hwframe_index].ref_cnt) {
        memcpy(&g_hwframe_pool[hwframe_index].surface, p_surface,
               sizeof(niFrameSurface1_t));
    }
    ni_log(NI_LOG_TRACE, "%s frame idx %u ref_cnt %d ..\n", __func__,
           hwframe_index, g_hwframe_pool[hwframe_index].ref_cnt);
}

// unref HW frame
void ni_hw_frame_unref(ni_hwframe_ref_t *g_hwframe_pool,
                       uint16_t hwframe_index)
{
    if (g_hwframe_pool[hwframe_index].ref_cnt > 0) {
        g_hwframe_pool[hwframe_index].ref_cnt--;
        if (0 == g_hwframe_pool[hwframe_index].ref_cnt &&
            g_hwframe_pool[hwframe_index].surface.ui16FrameIdx) {
            ni_log(NI_LOG_TRACE, "%s frame idx recycing frame idx %u\n", __func__,
                   g_hwframe_pool[hwframe_index].surface.ui16FrameIdx);

            ni_hwframe_buffer_recycle2(&g_hwframe_pool[hwframe_index].surface);
        }
        ni_log(NI_LOG_TRACE, "%s frame idx %u ref_cnt now: %d\n", __func__,
               hwframe_index, g_hwframe_pool[hwframe_index].ref_cnt);
    } else {
        ni_log(NI_LOG_ERROR, "%s error frame idx %u ref_cnt %d <= 0\n",
               __func__, hwframe_index,
               g_hwframe_pool[hwframe_index].ref_cnt);
    }
}

void arg_error_exit(char *arg_name, char *param)
{
    ni_log(NI_LOG_ERROR, "Error: unrecognized argument for %s, \"%s\"\n", arg_name,
            param);
    return;
}
