/*******************************************************************************
 *
 * Copyright (C) 2022 NETINT Technologies
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 ******************************************************************************/

/*!*****************************************************************************
 *  \file   ni_watermark_test.c
 *
 *  \brief  Application for applying watermark using libxcoder API overlay.
 *          Its code provides examples on how to programatically use
 *          libxcoder API.
 ******************************************************************************/

#define _POSIX_C_SOURCE 200809L
#include <getopt.h>
#include <unistd.h>
#include <sys/types.h>
#include <fcntl.h>
#include <sys/time.h>
#include <time.h>

#include "ni_test_utils.h"

#define POOL_SIZE 1
#define MAX_INPUT_FILES 2
#define NI_MAX_PPU_OUTPUTS 3
#define NI_MAX_HW_FRAME 3

static uint8_t tmp_buf[NI_MAX_TX_SZ] = {0};

// simplistic ref counted HW frame
static ni_hwframe_ref_t g_hwframe_pool[NI_MAX_DR_HWDESC_FRAME_INDEX];

int end_of_all_streams = 0;
struct timeval start_time;
struct timeval previous_time;
struct timeval current_time;
uint32_t time_diff;
int print_time = 0;

typedef struct _frame_list
{
    ni_session_data_io_t frames[NI_MAX_HW_FRAME];
    int head;
    int tail;
    int size;
} frame_list_t;

typedef enum
{
    UNINIT = 0,
    INIT,
    RUNING,
    EOS,
    EXIT
} thread_state_t;

typedef struct _ni_filter_worker
{
    ni_scaler_input_params_t *upload_params;
    ni_scaler_input_params_t *scaler_params;
    ni_scaler_multi_watermark_params_t scaler_watermark_params;
    ni_session_context_t watermark_ctx;
    niFrameSurface1_t *p_up_hwframe;

    int iXcoderGUID;
} ni_filter_worker_t;

typedef struct _ni_encoder_worker
{
    ni_xcoder_params_t enc_api_param;
    ni_session_context_t enc_ctx;
    ni_session_data_io_t enc_in_frame;
    ni_session_data_io_t out_packet;
    device_state_t enc_xcoder_state;

    int iXcoderGUID;
    int pix_fmt;

    FILE *p_file;
    uint32_t number_of_packets_list;
} ni_encoder_worker_t;

typedef struct _ni_filter_encoder_worker
{
    ni_pthread_t pid;
    thread_state_t thread_state;
    int index;
    pthread_mutex_t frm_lock;
    pthread_cond_t  frm_cond;
    frame_list_t frame_list;
    ni_filter_worker_t filter_worker;
    ni_session_data_io_t filter_output;
    ni_encoder_worker_t encoder_worker;
} ni_filter_encoder_worker_t;

static inline bool frame_list_is_empty(frame_list_t *list) {
    return (list->head == list->tail);
}

static inline bool frame_list_is_full(frame_list_t *list) {
    return (list->head == ((list->tail + 1) % (list->size)));
}

static inline int frame_list_enqueue(frame_list_t *list) {
    if (frame_list_is_full(list)) {
        ni_log(NI_LOG_ERROR, "ERROR: frame_list is full\n");
        return -1;
    }
    list->tail = (list->tail + 1) % (list->size);
    return 0;
}

static inline int frame_list_dequeue(frame_list_t *list) {
    if (frame_list_is_empty(list)) {
        ni_log(NI_LOG_ERROR, "ERROR: frame_list is empty\n");
        return -1;
    }
    list->head = (list->head + 1) % (list->size);
    return 0;
}

static inline int frame_list_length(frame_list_t *list) {
    return ((list->tail - list->head + list->size) % (list->size));
}

void print_list(frame_list_t *frame_list) {
    for (int i = 0; i < NI_MAX_HW_FRAME; i++) {
        niFrameSurface1_t *surface = (niFrameSurface1_t *)frame_list->frames[i].data.frame.p_data[3];
        if (surface) {
            ni_log(NI_LOG_ERROR, "print index %d surface id %d\n", i, surface->ui16FrameIdx);
        }
    }
}

int send_data(ni_filter_encoder_worker_t *worker, ni_session_data_io_t *in_data, int index) {
    frame_list_t *frame_list = &(worker->frame_list);
    ni_session_data_io_t *split_data = &frame_list->frames[frame_list->tail];

    memset(split_data, 0, sizeof(ni_session_data_io_t));
    // width height does not matter//factor does not matter//no metadata
    ni_frame_buffer_alloc(&(split_data->data.frame), 1, 1, 1, 0, 1, 1, 0);
    frame_copy_props(split_data, in_data);

    niFrameSurface1_t *dst_surface = (niFrameSurface1_t *) split_data->data.frame.p_data[3];
    uint8_t *src_surface = in_data->data.frame.p_data[3];
    memcpy(dst_surface, src_surface + index * sizeof(niFrameSurface1_t), sizeof(niFrameSurface1_t));

    split_data->data.frame.video_width = dst_surface->ui16width;
    split_data->data.frame.video_height = dst_surface->ui16height;

    ni_pthread_mutex_lock(&worker->frm_lock);
    while (frame_list_is_full(frame_list) && worker->thread_state != EXIT) {
        ni_pthread_cond_wait(&worker->frm_cond, &worker->frm_lock);
    }
    if (worker->thread_state == EXIT) {
        ni_pthread_mutex_unlock(&worker->frm_lock);
        return -1;
    }
    if (frame_list_is_empty(frame_list)) {
        frame_list_enqueue(frame_list);
        ni_pthread_cond_signal(&worker->frm_cond);
    } else {
        frame_list_enqueue(frame_list);
    }
    // print_list(frame_list);
    ni_pthread_mutex_unlock(&worker->frm_lock);
    return 0;
}

int get_data(ni_filter_encoder_worker_t *worker, ni_session_data_io_t *data) {
    frame_list_t *frame_list = &worker->frame_list;
    ni_pthread_mutex_lock(&worker->frm_lock);
    if (worker->thread_state >= EOS && frame_list_is_empty(frame_list)) {
        ni_pthread_mutex_unlock(&worker->frm_lock);
        return -1;
    }
    if (frame_list_is_empty(frame_list)) {
        ni_pthread_cond_wait(&worker->frm_cond, &worker->frm_lock);
    }
    if (frame_list_is_full(frame_list)) {
        memcpy(data, &frame_list->frames[frame_list->head], sizeof(ni_session_data_io_t));
        frame_list_dequeue(frame_list);
        ni_pthread_cond_signal(&worker->frm_cond);
    } else {
        memcpy(data, &frame_list->frames[frame_list->head], sizeof(ni_session_data_io_t));
        frame_list_dequeue(frame_list);
    }
    ni_pthread_mutex_unlock(&worker->frm_lock);
    return 0;
}

void send_eos(ni_filter_encoder_worker_t *worker) {
    ni_pthread_mutex_lock(&worker->frm_lock);
    worker->thread_state = EOS;
    if (frame_list_is_empty(&worker->frame_list)) {
        ni_pthread_cond_signal(&worker->frm_cond);
    }
    ni_pthread_mutex_unlock(&worker->frm_lock);
}

void exit_signal(ni_filter_encoder_worker_t *worker) {
    ni_pthread_mutex_lock(&worker->frm_lock);
    worker->thread_state = EXIT;
    ni_pthread_cond_signal(&worker->frm_cond);
    ni_pthread_mutex_unlock(&worker->frm_lock);
}

int filter_frame(ni_session_context_t *filter_ctx, niFrameSurface1_t *p_up_hwframe,
                 ni_session_data_io_t *in_data, ni_session_data_io_t *out_data,
                 ni_scaler_input_params_t *main_params, ni_scaler_input_params_t *upload_params)
{
    int ret = 0;
    frame_copy_props(out_data, in_data);
    // allocate a ni_frame_t structure on the host PC
    ret = ni_frame_buffer_alloc_hwenc(&(out_data->data.frame),
                                      main_params->output_width,
                                      main_params->output_height, 0);
    if (ret != 0) {
        ni_log(NI_LOG_ERROR, "Failed to alloc hwenc\n");
        return -1;
    }

    // Allocate scaler input frame
    ret = ni_scaler_input_frame_alloc(filter_ctx, *upload_params, p_up_hwframe);
    if (ret != 0) {
        ni_log(NI_LOG_ERROR, "Failed to alloc input frame\n");
        return -1;
    }

    // Allocate scaler destination frame.
    ret = ni_scaler_dest_frame_alloc(filter_ctx, *main_params, (niFrameSurface1_t *)in_data->data.frame.p_data[3]);
    if (ret != 0) {
        ni_log(NI_LOG_ERROR, "Failed to alloc dest frame\n");
        return -1;
    }

    // Retrieve hardware frame info from 2D engine and put it in the ni_frame_t structure.
    ret = ni_device_session_read_hwdesc(filter_ctx, out_data, NI_DEVICE_TYPE_SCALER);
    if (ret != 0)
    {
        ni_log(NI_LOG_ERROR, "Failed to lauch scaler operation %d\n", main_params->op);
        return -1;
    }

    ni_hwframe_buffer_recycle2((niFrameSurface1_t *)in_data->data.frame.p_data[3]);
    ni_frame_buffer_free(&(in_data->data.frame));
    return 0;
}

/*!*****************************************************************************
 *  \brief  decoder session open
 *
 *  \param
 *
 *  \return 0 if successful, < 0 otherwise
 ******************************************************************************/
int decoder_open_session(ni_session_context_t *p_dec_ctx,
                         ni_xcoder_params_t *p_dec_params,
                         ni_codec_format_t codec_type, int devid,
                         int bit_depth)
{
    int ret = 0;
    p_dec_ctx->session_id = NI_INVALID_SESSION_ID;
    p_dec_ctx->device_handle = NI_INVALID_DEVICE_HANDLE;
    p_dec_ctx->blk_io_handle = NI_INVALID_DEVICE_HANDLE;
    p_dec_ctx->p_session_config = p_dec_params;
    p_dec_ctx->codec_format = codec_type;

    // assign the card GUID in the encoder context and let session open
    // take care of the rest
    p_dec_ctx->hw_id = devid;

    // default: little endian
    p_dec_ctx->src_bit_depth = bit_depth;
    p_dec_ctx->src_endian = NI_FRAME_LITTLE_ENDIAN;
    p_dec_ctx->bit_depth_factor = (bit_depth == 8) ? 1 : 2;

    if (p_dec_params->dec_input_params.hwframes) {
        p_dec_ctx->hw_action = NI_CODEC_HW_ENABLE;
    } else {
        p_dec_ctx->hw_action = NI_CODEC_HW_NONE;
    }

    ret = ni_device_session_open(p_dec_ctx, NI_DEVICE_TYPE_DECODER);

    if (ret != NI_RETCODE_SUCCESS) {
        ni_log(NI_LOG_ERROR, "Error: ni_decoder_session_open() failure!\n");
    }
    return ret;
}

/*!****************************************************************************
 *  \brief  Open an encoder session to Quadra
 *
 *  \param  [out] p_enc_ctx         pointer to an encoder session context
 *          [in]  dst_codec_format  AVC or HEVC
 *          [in]  iXcoderGUID       id to identify the Quadra device
 *          [in]  p_enc_params      sets the encoder parameters
 *          [in]  width             width of frames to encode
 *          [in]  height            height of frames to encode
 *          [in]  p_frame           first frame to encode
 *          [in]  pix_fmt           pixel format
 *
 *  \return 0 if successful, < 0 otherwise
 ******************************************************************************/
int encoder_open_session(ni_session_context_t *p_enc_ctx, int dst_codec_format,
                         int iXcoderGUID, ni_xcoder_params_t *p_enc_params,
                         int width, int height, ni_frame_t *p_frame, int pix_fmt)
{
    int ret = 0;
    int planar = get_pixel_planar(pix_fmt);

    // Enable hardware frame encoding
    p_enc_ctx->hw_action = NI_CODEC_HW_ENABLE;
    p_enc_params->hwframes = 1;

    // Provide the first frame to the Quadra encoder
    p_enc_params->p_first_frame = p_frame;

    // Specify codec, AVC vs HEVC
    p_enc_ctx->codec_format = dst_codec_format;

    p_enc_ctx->p_session_config = p_enc_params;
    p_enc_ctx->session_id = NI_INVALID_SESSION_ID;

    // Assign the card GUID in the encoder context to open a session
    // to that specific Quadra device
    p_enc_ctx->device_handle = NI_INVALID_DEVICE_HANDLE;
    p_enc_ctx->blk_io_handle = NI_INVALID_DEVICE_HANDLE;
    p_enc_ctx->hw_id = iXcoderGUID;

    ni_encoder_set_input_frame_format(p_enc_ctx, p_enc_params, width, height, 8,
                                      NI_FRAME_LITTLE_ENDIAN, planar);

    ret = ni_device_session_open(p_enc_ctx, NI_DEVICE_TYPE_ENCODER);
    if (ret != NI_RETCODE_SUCCESS) {
        ni_log(NI_LOG_ERROR, "Error: encoder open session failure\n");
    }

    return ret;
}

/*!*****************************************************************************
*  \brief  open scaler session
*
*  \param
*
*  \return 0 if successful, < 0 otherwise
******************************************************************************/
static int scaler_open_session(ni_filter_worker_t *filter_worker)
{
    ni_session_context_t *p_scaler_ctx = &filter_worker->watermark_ctx;
    p_scaler_ctx->session_id = NI_INVALID_SESSION_ID;
    p_scaler_ctx->device_handle = NI_INVALID_DEVICE_HANDLE;
    p_scaler_ctx->blk_io_handle = NI_INVALID_DEVICE_HANDLE;
    p_scaler_ctx->hw_id = filter_worker->iXcoderGUID;
    p_scaler_ctx->device_type = NI_DEVICE_TYPE_SCALER;
    p_scaler_ctx->scaler_operation = NI_SCALER_OPCODE_WATERMARK;
    p_scaler_ctx->keep_alive_timeout = NI_DEFAULT_KEEP_ALIVE_TIMEOUT;

    if (ni_device_session_open(p_scaler_ctx, NI_DEVICE_TYPE_SCALER) != NI_RETCODE_SUCCESS) {
        ni_log(NI_LOG_ERROR, "Error: ni_scaler_session_open() failure!\n");
        return -1;
    }

    for(int watermark_idx = 0; watermark_idx < NI_MAX_SUPPORT_WATERMARK_NUM; watermark_idx++)
    {
        filter_worker->scaler_watermark_params.multi_watermark_params[watermark_idx].ui32StartX=200 * watermark_idx;
        filter_worker->scaler_watermark_params.multi_watermark_params[watermark_idx].ui32StartY=100 * watermark_idx;
        filter_worker->scaler_watermark_params.multi_watermark_params[watermark_idx].ui32Width=200;
        filter_worker->scaler_watermark_params.multi_watermark_params[watermark_idx].ui32Height=100;
        filter_worker->scaler_watermark_params.multi_watermark_params[watermark_idx].ui32Valid=1;
    }
    if(ni_scaler_set_watermark_params(&filter_worker->watermark_ctx, &filter_worker->scaler_watermark_params.multi_watermark_params[0])) {
        ni_log(NI_LOG_ERROR, "Error: set watermark params\n");
        ni_device_session_close(&filter_worker->watermark_ctx, 1, NI_DEVICE_TYPE_SCALER);
        return -1;
    }

    if (0 != ni_scaler_frame_pool_alloc(&filter_worker->watermark_ctx, *(filter_worker->scaler_params))) {
        ni_log(NI_LOG_ERROR, "Error: init filter hwframe pool\n");
        ni_device_session_close(&filter_worker->watermark_ctx, 1, NI_DEVICE_TYPE_SCALER);
        return -1;
    }
    return 0;
}

/*!****************************************************************************
 *  \brief    Print usage information
 *
 *  \param    none
 *
 *  \return   none
 ******************************************************************************/
void print_usage(void)
{
    printf("Application for applying watermark using libxcoder API overlay\n"
           "Libxcoder release v%s\n"
           "Usage: ni_watermark_test [options]\n"
           "\n"
           "options:\n"
           "  -h | --help        Show help.\n"
           "  -v | --version     Print version info.\n"
           "  -l | --loglevel    Set loglevel of libxcoder API.\n"
           "                     [none, fatal, error, info, debug, trace]\n"
           "                     Default: info\n"
           "  -c | --card        Set card index to use.\n"
           "                     See `ni_rsrc_mon` for cards on system.\n"
           "                     (Default: 0)\n"
           "  -i | --input       Input file path.\n"
           "  -r | --repeat      (Positive integer) to Repeat input X times "
           "for performance \n"
           "                     test. (Default: 1)\n"
           "  -s | --size        Resolution of input file in format "
           "WIDTHxHEIGHT.\n"
           "                     (eg. '1920x1080')\n"
           "  -o | --output      Output file path.\n"
           "  -d | --decoder_params Decode parameters.\n",
           NI_XCODER_REVISION);
}

/*!****************************************************************************
 *  \brief  Parse user command line arguments
 *
 *  \param [in] argc                argument count
 *         [in] argv                argument vector
 *         [out] input_filename     input filename
 *         [out] output_filename    output filename
 *         [out] iXcoderGUID        Quadra device
 *         [out] repeat             input read repeat times
 *         [out] arg_width          resolution width
 *         [out] arg_height         resolution height
 *         [out] arg_pic_width      pic resolution width
 *         [out] arg_pic_ehight     pic resolution height
 *         [out] decConfXcoderParams dec param
 *         [out] output_num         output number
 *
 *  \return nothing                 program exit on error
 ******************************************************************************/
int parse_arguments(int argc, char *argv[], char (*input_filename)[FILE_NAME_LEN],
                    char (*output_filename)[FILE_NAME_LEN], int *iXcoderGUID, int *repeat, int *arg_width,
                    int *arg_height, int *arg_pic_width, int *arg_pic_height, char *decConfXcoderParams,
                    int *output_num)
{
    char xcoderGUID[32];
    char *n;   // used for parsing width and height from --size
    int opt;
    int opt_index;
    ni_log_level_t log_level;
    static int input_index = 0;
    static int output_index = 0;
    static int input_resolution_index = 0;

    static const char *opt_string = "hvl:c:i:s:o:r:d:";
    static const struct option long_options[] = {
        {"help", no_argument, NULL, 'h'},
        {"version", no_argument, NULL, 'v'},
        {"loglevel", no_argument, NULL, 'l'},
        {"card", required_argument, NULL, 'c'},
        {"input", required_argument, NULL, 'i'},
        {"size", required_argument, NULL, 's'},
        {"output", required_argument, NULL, 'o'},
        {"repeat", required_argument, NULL, 'r'},
        {"decoder_params", required_argument, NULL, 'd'},
        {NULL, 0, NULL, 0},
    };

    while ((opt = getopt_long(argc, argv, opt_string, long_options,
                              &opt_index)) != -1) {
        switch (opt) {
        case 'h':
            print_usage();
            return 1;
        case 'v':
            printf("Release ver: %s\n"
                   "API ver:     %s\n"
                   "Date:        %s\n"
                   "ID:          %s\n",
                   NI_XCODER_REVISION, LIBXCODER_API_VERSION,
                   NI_SW_RELEASE_TIME, NI_SW_RELEASE_ID);
            return 1;
        case 'l':
            log_level = arg_to_ni_log_level(optarg);
            if (log_level != NI_LOG_INVALID) {
                ni_log_set_level(log_level);
            } else {
                arg_error_exit("-l | --loglevel", optarg);
            }
            break;
        case 'c':
            strcpy(xcoderGUID, optarg);
            *iXcoderGUID = (int)strtol(optarg, &n, 10);
            // No numeric characters found in left side of optarg
            if (n == xcoderGUID)
                arg_error_exit("-c | --card", optarg);
            break;
        case 'i':
            if (input_index >= MAX_INPUT_FILES) {
                ni_log(NI_LOG_ERROR,
                       "Error: number of input files cannot exceed %d\n", MAX_INPUT_FILES);
                return -1;
            }
            strcpy(input_filename[input_index], optarg);
            input_index++;
            break;
        case 's':
            if(input_resolution_index == 0) {
                *arg_width = (int)strtol(optarg, &n, 10);
                *arg_height = atoi(n + 1);
                if ((*n != 'x') || (!*arg_width || !*arg_height))
                    arg_error_exit("-s | --size", optarg);
                input_resolution_index++;
            } else {
                *arg_pic_width = (int)strtol(optarg, &n, 10);
                *arg_pic_height = atoi(n + 1);
                if ((*n != 'x') || (!*arg_pic_width || !*arg_pic_height))
                    arg_error_exit("-s | --size", optarg);
            }
            break;
        case 'o':
            if (output_index >= NI_MAX_PPU_OUTPUTS) {
                ni_log(NI_LOG_ERROR,
                       "Error: number of output files cannot exceed %d\n", NI_MAX_PPU_OUTPUTS);
                return -1;
            }
            strcpy(output_filename[output_index], optarg);
            output_index++;
            break;
        case 'r':
            if (!(atoi(optarg) >= 1))
                arg_error_exit("-r | --repeat", optarg);
            *repeat = atoi(optarg);
            break;
        case 'd':
            strcpy(decConfXcoderParams, optarg);
            break;
        default:
            print_usage();
            return -1;
        }
    }

    // Check required args are present
    if (!input_filename[0]) {
        ni_log(NI_LOG_ERROR, "Error: missing argument for -i | --input\n");
        return -1;
    }

    if (!output_filename[0]) {
        ni_log(NI_LOG_ERROR, "Error: missing argument for -o | --output\n");
        return -1;
    }
    *output_num = output_index;
    return 0;
}

static int encoder_receive(ni_session_context_t *enc_ctx_list,
                           ni_session_data_io_t *pkt,
                           uint32_t *number_of_packets_list,
                           int output_total, FILE **pfs_list,
                           device_state_t *xcoder_state)
{
    int i, recycle_index;
    int recv_fin_flag = NI_TEST_RETCODE_SUCCESS;
    uint32_t prev_num_pkt[1] = {0};

    for (i = 0; i < output_total; i++) {
        pkt->data.packet.end_of_stream = 0;
        prev_num_pkt[i] = number_of_packets_list[i];

        recv_fin_flag = encoder_receive_data(&enc_ctx_list[i], pkt, pfs_list[i],
                                             &number_of_packets_list[i]);

        recycle_index = pkt->data.packet.recycle_index;
        if (prev_num_pkt[i] < number_of_packets_list[i] &&
            enc_ctx_list[i].hw_action && recycle_index > 0 &&
            recycle_index < NI_GET_MAX_HWDESC_FRAME_INDEX(enc_ctx_list[i].ddr_config)) {
            //encoder only returns valid recycle index
            //when there's something to recycle.
            //This range is suitable for all memory bins
            ni_hw_frame_unref(g_hwframe_pool, recycle_index);
        } else {
            ni_log(NI_LOG_DEBUG, "enc %d recv, prev_num_pkt %u "
                   "number_of_packets_list %u recycle_index %u\n", i,
                   prev_num_pkt[i], number_of_packets_list[i], recycle_index);
        }

        xcoder_state[i].enc_eos_received = pkt->data.packet.end_of_stream;

        if (recv_fin_flag < 0) {
            ni_log(NI_LOG_ERROR, "enc %d error, quit !\n", i);
            break;
        } else if (recv_fin_flag == NI_TEST_RETCODE_EAGAIN) {
            ni_usleep(100);
        }
    }

    return recv_fin_flag;
}

static void free_filter_encoder_worker(ni_filter_encoder_worker_t *worker) {
    ni_filter_worker_t *filter_worker = &worker->filter_worker;
    ni_encoder_worker_t *encoder_worker = &worker->encoder_worker;
    ni_device_session_close(&filter_worker->watermark_ctx, 1, NI_DEVICE_TYPE_SCALER);
    ni_device_session_context_clear(&filter_worker->watermark_ctx);
    ni_device_session_close(&encoder_worker->enc_ctx, 1, NI_DEVICE_TYPE_ENCODER);
    ni_device_session_context_clear(&encoder_worker->enc_ctx);
    ni_frame_buffer_free(&(worker->filter_output.data.frame));
    ni_frame_buffer_free(&(encoder_worker->enc_in_frame.data.frame));
    ni_packet_buffer_free(&(encoder_worker->out_packet.data.packet));
}

void *filter_encoder_thread(void *args)
{
    ni_filter_encoder_worker_t *worker = args;
    ni_filter_worker_t *filter_worker = &worker->filter_worker;
    ni_encoder_worker_t *encoder_worker = &worker->encoder_worker;
    ni_xcoder_params_t *p_enc_param = &encoder_worker->enc_api_param;
    int ret = 0;
    int dst_codec_format = NI_CODEC_FORMAT_H265;
    ni_session_data_io_t dec_out_frame = {0};
    niFrameSurface1_t *p_hwframe = NULL;
    int send_fin_flag = 0;
    int receive_fin_flag = 0;
    int end_of_cur_streams = 0;
    while (!end_of_all_streams && !end_of_cur_streams && (send_fin_flag == 0 || receive_fin_flag == 0)) {
        if (filter_worker->watermark_ctx.session_id == NI_INVALID_SESSION_ID) {
            ret = scaler_open_session(filter_worker);
            if (ret < 0) {
                ni_log(NI_LOG_ERROR, "Error: scaler_open_session\n");
                goto thread_end;
            }
        }

        if (get_data(worker, &dec_out_frame) == 0) {
            ret = filter_frame(&filter_worker->watermark_ctx, filter_worker->p_up_hwframe, &dec_out_frame, &worker->filter_output,
                            filter_worker->scaler_params, filter_worker->upload_params);
            if (ret) {
                goto thread_end;
            }
        } else {
            worker->filter_output.data.frame.end_of_stream = 1;
        }
        if(encoder_worker->enc_ctx.session_id == NI_INVALID_SESSION_ID) {
            // Open the encoder session with given parameters
            ret = encoder_open_session(&encoder_worker->enc_ctx, dst_codec_format, encoder_worker->iXcoderGUID,
                                       p_enc_param, p_enc_param->source_width, p_enc_param->source_height,
                                       &worker->filter_output.data.frame, encoder_worker->pix_fmt);
            if (ret < 0) {
                goto thread_end;
            }
        }
        // YUV Sending
        send_fin_flag = encoder_send_data2(&encoder_worker->enc_ctx, &worker->filter_output, &encoder_worker->enc_in_frame,
                                 p_enc_param->source_width, p_enc_param->source_height, &encoder_worker->enc_xcoder_state);
        if (send_fin_flag < 0) { //Error
            goto thread_end;
        }
        p_hwframe = (niFrameSurface1_t *)worker->filter_output.data.frame.p_data[3];
        ni_hw_frame_ref(g_hwframe_pool, p_hwframe);
        ni_frame_wipe_aux_data(&worker->filter_output.data.frame);

        while (1) {
            receive_fin_flag = encoder_receive(&encoder_worker->enc_ctx, &encoder_worker->out_packet,
                                  &encoder_worker->number_of_packets_list, 1,
                                  &encoder_worker->p_file, &encoder_worker->enc_xcoder_state);
            if (receive_fin_flag < 0) {
                goto thread_end;
            } else if (receive_fin_flag == NI_TEST_RETCODE_EAGAIN) {
                receive_fin_flag = 0;
                break;
            }
            if (!encoder_worker->enc_xcoder_state.enc_eos_received) {
                ni_log(NI_LOG_DEBUG, "enc continues to read!\n");
            } else {
                ni_log(NI_LOG_INFO, "enc %d eos !\n", worker->index);
                end_of_cur_streams = 1;
                break;
            }
        }

        if (print_time) {
            time_diff = (uint32_t)(current_time.tv_sec - start_time.tv_sec);
            if (time_diff == 0) {
                time_diff = 1;
            }
            ni_log(NI_LOG_INFO, "[ENC %d] Packets= %u fps=%f\n",
                    worker->index, encoder_worker->enc_ctx.frame_num,
                    (float)encoder_worker->enc_ctx.frame_num / (float)time_diff);
        }
    }
thread_end:
    // Broadcast all codec threads to quit on exception such as NVMe IO.
    exit_signal(worker);
    if (ret < 0)
    {
        end_of_all_streams = 1;
    }
    ni_log(NI_LOG_INFO, "enc thread %d exit\n", worker->index);
    return (void *)(long)ret;
}

int main(int argc, char *argv[])
{
    static char input_filename[MAX_INPUT_FILES][FILE_NAME_LEN];
    static char output_filename[NI_MAX_PPU_OUTPUTS][FILE_NAME_LEN];
    int iXcoderGUID = NI_INVALID_HWID;
    int repeat = 1;
    int arg_width = 0;
    int arg_height = 0;
    int arg_pic_width;
    int arg_pic_height;
    char decConfXcoderParams[2048] = {0};
    int pix_fmt = NI_PIX_FMT_YUV420P;
    int ret, i;
    int send_fin_flag = 0;
    int receive_fin_flag = 0;

    int input_file_fd[MAX_INPUT_FILES] = {-1};
    ni_file_reader_t *temp_reader = NULL;
    ni_file_reader_t file_reader[MAX_INPUT_FILES] = {0};


    ni_xcoder_params_t p_dec_api_param = {0};
    int dec_input_width;
    int dec_input_height;
    int dec_output_width[NI_MAX_NUM_OF_DECODER_OUTPUTS];
    int dec_output_height[NI_MAX_NUM_OF_DECODER_OUTPUTS];
    uint32_t number_of_frames = 0;
    ni_session_context_t dec_ctx = {0};
    ni_session_data_io_t dec_in_pkt = {0};
    ni_session_data_io_t dec_out_frame = {0};
    ni_h264_sps_t SPS = {0};   // input header SPS
    device_state_t dec_xcoder_state = {0};

    int input_exhausted = 0;
    ni_session_context_t upl_ctx = {0};
    ni_session_data_io_t swin_frame = {0};
    ni_session_data_io_t upload_frame = {0};
    niFrameSurface1_t *p_up_hwframe = NULL;
    ni_scaler_input_params_t upload_params = {0};


    int output_num = 0;
    ni_scaler_input_params_t watermark_params[NI_MAX_NUM_OF_DECODER_OUTPUTS] = {0};
    ni_filter_encoder_worker_t *filter_enc_worker[NI_MAX_PPU_OUTPUTS] = {0};

    ret = parse_arguments(argc, argv, input_filename, output_filename, &iXcoderGUID, &repeat,
                          &arg_width, &arg_height, &arg_pic_width, &arg_pic_height, decConfXcoderParams,
                          &output_num);
    if (ret < 0) {
        ni_log(NI_LOG_ERROR, "parse_arguments\n");
        return ret;
    } else if (ret > 0) {
        return 0;
    }

    dec_input_width = arg_width;
    dec_input_height = arg_height;

    ni_log(NI_LOG_INFO, "User video resolution: %dx%d\n", dec_input_width, dec_input_height);

    if (ni_decoder_init_default_params(&p_dec_api_param, 30, 1, 200000, arg_width,
                                       arg_height) < 0) {
        ni_log(NI_LOG_ERROR, "Error: decoder p_config set up error\n");
        return -1;
    }
    // check and set ni_decoder_params from --xcoder-params
    if (ni_retrieve_decoder_params(decConfXcoderParams, &p_dec_api_param,
                                   &dec_ctx)) {
        ni_log(NI_LOG_ERROR, "Error: encoder p_config parsing error\n");
        return -1;
    }
    pix_fmt = p_dec_api_param.dec_input_params.semi_planar[0] ? NI_PIX_FMT_NV12 : NI_PIX_FMT_YUV420P;
    if (parse_symbolic_decoder_param(&p_dec_api_param)) {
        ni_log(NI_LOG_ERROR," Error: decoder p_config parsing error\n");
        return -1;
    }

    for (i = 0; i < NI_MAX_NUM_OF_DECODER_OUTPUTS; i++){
        dec_output_width[i] = p_dec_api_param.dec_input_params.scale_wh[i][0] ?
                        p_dec_api_param.dec_input_params.scale_wh[i][0] : dec_input_width;
        dec_output_height[i] = p_dec_api_param.dec_input_params.scale_wh[i][1] ?
                        p_dec_api_param.dec_input_params.scale_wh[i][1] : dec_input_height;
    }

    if (ni_device_session_context_init(&dec_ctx) < 0) {
        ni_log(NI_LOG_ERROR, "Error: init decoder context error\n");
        return -1;
    }


    upload_params.input_format = GC620_RGBA8888;
    upload_params.in_rec_width = upload_params.input_width = arg_pic_width;
    upload_params.in_rec_height = upload_params.input_height = arg_pic_height;

    if (ni_device_session_context_init(&upl_ctx) < 0) {
        ni_log(NI_LOG_ERROR, "Error: init uploader context error\n");
        return -1;
    }

    for(i = 0; i < MAX_INPUT_FILES; i++) {
        // Load input file into memory
        temp_reader = &file_reader[i];
        if (load_input_file(input_filename[i], &temp_reader->total_file_size) < 0) {
            return -1;
        }
        temp_reader->loop = repeat;
        temp_reader->data_left_size = temp_reader->total_file_size;
#ifdef _WIN32
        input_file_fd[i] = open(input_filename[i], O_RDONLY | O_BINARY);
#else
        input_file_fd[i] = open(input_filename[i], O_RDONLY);
#endif

        if (input_file_fd[i] < 0) {
            ni_log(NI_LOG_ERROR, "Error: can not open input file %s\n", input_filename[i]);
            goto end;
        }
    }

    //load decode file
    ret = read_input_file(&file_reader[0], input_file_fd[0]);
    if (ret) {
        ni_log(NI_LOG_ERROR, "Error: input file %s read error\n", input_filename[0]);
        goto end;
    }

    for (i = 0; i < output_num; i++) {
        filter_enc_worker[i] = calloc(1, sizeof(ni_filter_encoder_worker_t));
        ni_filter_encoder_worker_t *worker = filter_enc_worker[i];
        if (ni_device_session_context_init(&worker->filter_worker.watermark_ctx) < 0) {
            ni_log(NI_LOG_ERROR, "Error: init scaler context error\n");
            goto end;
        }
        // ni_scaler_input_params_t watermark_params = {0};
        watermark_params[i].input_format = ni_to_gc620_pix_fmt(pix_fmt);
        watermark_params[i].in_rec_width = watermark_params[i].input_width = dec_output_width[i];
        watermark_params[i].in_rec_height = watermark_params[i].input_height = dec_output_height[i];
        watermark_params[i].output_format = ni_to_gc620_pix_fmt(pix_fmt);
        watermark_params[i].out_rec_width = watermark_params[i].output_width = dec_output_width[i];
        watermark_params[i].out_rec_height = watermark_params[i].output_height = dec_output_height[i];
        watermark_params[i].op = NI_SCALER_OPCODE_WATERMARK;
        watermark_params[i].out_rec_x = 0;
        watermark_params[i].out_rec_y = 0;
        watermark_params[i].rgba_color = 0;
        // params set by user
        watermark_params[i].in_rec_x = 0;
        watermark_params[i].in_rec_y = 0;
        worker->filter_worker.iXcoderGUID = iXcoderGUID;
        worker->filter_worker.upload_params = &upload_params;
        worker->filter_worker.scaler_params = &watermark_params[i];

        if (ni_device_session_context_init(&worker->encoder_worker.enc_ctx) < 0) {
            ni_log(NI_LOG_ERROR, "Error: init encoder context error\n");
            goto end;
        }

        // Configure the encoder parameter structure. We'll use some basic
        // defaults: 30 fps, 200000 bps CBR encoding, AVC or HEVC encoding
        if (ni_encoder_init_default_params(&worker->encoder_worker.enc_api_param, 30, 1, 200000, dec_output_width[i],
                                           dec_output_height[i], worker->encoder_worker.enc_ctx.codec_format) < 0) {
            ni_log(NI_LOG_ERROR, "Error: encoder init default set up error\n");
            goto end;
        }
        worker->encoder_worker.iXcoderGUID = iXcoderGUID;
        worker->encoder_worker.pix_fmt = pix_fmt;
        worker->index = i;
        if (ni_pthread_mutex_init(&worker->frm_lock)) {
            ni_log(NI_LOG_ERROR,
                "Failed to initialize frm_lock mutex\n");
            goto end;
        }
        if (ni_pthread_cond_init(&worker->frm_cond, NULL)) {
            pthread_mutex_destroy(&worker->frm_lock);
            ni_log(NI_LOG_ERROR,
                "Failed to initialize frm_cond cond\n");
            goto end;
        }
        worker->frame_list.size = NI_MAX_HW_FRAME;
        filter_enc_worker[i]->thread_state = INIT;
    }

    // open output file
    for (i = 0; i < output_num; i++) {
        ni_filter_encoder_worker_t *worker = filter_enc_worker[i];
        if (strcmp(output_filename[i], "null") != 0) {
            worker->encoder_worker.p_file = fopen(output_filename[i], "wb");
            if (worker->encoder_worker.p_file == NULL) {
                ni_log(NI_LOG_ERROR, "Error: cannot open %s\n", output_filename[i]);
                goto end;
            }
        }
        ni_log(NI_LOG_INFO, "SUCCESS: Opened output file: %s\n", output_filename[i]);
    }

    ni_gettimeofday(&start_time, NULL);
    ni_gettimeofday(&previous_time, NULL);
    ni_gettimeofday(&current_time, NULL);

    // Decode, set default param for dec_ctx
    ni_codec_format_t codec_type = NI_CODEC_FORMAT_H264;
    int bit_depth = 8;
    ret = decoder_open_session(&dec_ctx, &p_dec_api_param, codec_type,
                               iXcoderGUID, bit_depth);
    if (ret != 0) {
        goto end;
    }
    iXcoderGUID = dec_ctx.hw_id;
    // keep filter encoder iXcoderGUID same as decoder
    for (i = 0; i < output_num; i++) {
        filter_enc_worker[i]->filter_worker.iXcoderGUID = iXcoderGUID;
        filter_enc_worker[i]->encoder_worker.iXcoderGUID = iXcoderGUID;
    }

    // Open an uploader session to Quadra
    if (uploader_open_session(&upl_ctx, arg_pic_width, arg_pic_height,
                              NI_PIX_FMT_ABGR, POOL_SIZE, iXcoderGUID)) {
        ni_device_session_close(&dec_ctx, 1, NI_DEVICE_TYPE_DECODER);
        goto end;
    }
    void *yuv_buf = malloc(MAX_RGBA_FRAME_SIZE);
    if(!yuv_buf)
    {
        ni_log(NI_LOG_ERROR, "Error: allocate yuv buf fail\n");
        goto end;
    }

    ret = hwup_frame(&upl_ctx, &swin_frame, &upload_frame,
                     input_file_fd[1], 0, &file_reader[1],
                     &input_exhausted, yuv_buf);
    p_up_hwframe = (niFrameSurface1_t *)(upload_frame.data.frame.p_data[3]);
    ni_frame_buffer_free(&(swin_frame.data.frame));
    if (yuv_buf) {
        free(yuv_buf);
    }
    // upload an rgba frame to quadra
    if (ret) {
        ni_log(NI_LOG_ERROR, "Error: upload frame error\n");
        ni_device_session_close(&dec_ctx, 1, NI_DEVICE_TYPE_DECODER);
        ni_device_session_close(&upl_ctx, 1, NI_DEVICE_TYPE_UPLOAD);
        goto end;
    }

    for (i = 0; i < output_num; i++) {
        filter_enc_worker[i]->filter_worker.p_up_hwframe = p_up_hwframe;
        if (ni_pthread_create(&filter_enc_worker[i]->pid, NULL, filter_encoder_thread, filter_enc_worker[i])) {
            ni_log(NI_LOG_ERROR, "Error: create filter encoder %d thread failed\n", i);
            goto end;
        }
        filter_enc_worker[i]->thread_state = RUNING;
    }
    int end_of_cur_streams = 0;
    while (!end_of_all_streams && !end_of_cur_streams && (send_fin_flag == 0 || receive_fin_flag == 0)) {
        (void)ni_gettimeofday(&current_time, NULL);
        print_time = current_time.tv_sec - previous_time.tv_sec >= 1;
        if (print_time) {
            previous_time = current_time;
        }

        send_fin_flag = decoder_send_data(&dec_ctx, &dec_in_pkt, &dec_xcoder_state,
                                          &file_reader[0], (void *)(&SPS),
                                          dec_input_width,
                                          dec_input_height, tmp_buf);
        if (send_fin_flag < 0) {
            end_of_all_streams = 1;
            ni_log(NI_LOG_ERROR, "decoder send packet failed!\n");
            break;
        }
        while (!end_of_all_streams && !end_of_cur_streams) {
            // YUV Receiving: not writing to file
            receive_fin_flag = decoder_receive_data(
                                &dec_ctx, &dec_out_frame, &dec_xcoder_state,
                                dec_output_width[0],
                                dec_output_height[0], NULL);

            if (receive_fin_flag < 0) {
                end_of_all_streams = 1;
                ni_log(NI_LOG_ERROR, "decoder receive packet failed!\n");
                break;
            }
            if (receive_fin_flag == NI_TEST_RETCODE_EAGAIN) {
                ni_log(NI_LOG_DEBUG,
                    "decoder get eagain, need to send new packet!\n");
                if (!dec_ctx.hw_action) {
                    ni_decoder_frame_buffer_free(&(dec_out_frame.data.frame));
                } else {
                    ni_frame_buffer_free(&(dec_out_frame.data.frame));
                }
                receive_fin_flag = NI_TEST_RETCODE_SUCCESS;
                break;
            } else if (receive_fin_flag == NI_TEST_RETCODE_SUCCESS) {
                number_of_frames++;
            }

            if (print_time) {
                time_diff = (uint32_t)(current_time.tv_sec - start_time.tv_sec);
                if (time_diff == 0) {
                    time_diff = 1;
                }
                ni_log(NI_LOG_INFO, "[DEC] Frames= %u  fps=%f \n",
                    number_of_frames, (float)number_of_frames / (float)time_diff);
            }

            if (receive_fin_flag == NI_TEST_RETCODE_SUCCESS) {
                for (i = 0; i < output_num; i++) {
                    if (send_data(filter_enc_worker[i], &dec_out_frame, i)) {
                        end_of_all_streams = 1;
                        ni_log(NI_LOG_ERROR, "get enc exit, break!\n");
                        break;
                    }
                }
            } else {
                for (i = 0; i < output_num; i++) {
                    send_eos(filter_enc_worker[i]);
                }
                ni_log(NI_LOG_INFO, "decoder eof\n");
                end_of_cur_streams = 1;
            }
        }
    }

    void *filter_encoder_result;
    for (i = 0; i < output_num; i++) {
        if (filter_enc_worker[i]->thread_state > INIT) {
            ret = pthread_join(filter_enc_worker[i]->pid, &filter_encoder_result);
            if (ret != 0) {
                ni_log(NI_LOG_ERROR, "Failed to join filter encoder thread\n");
            }
            if ((int)((long)filter_encoder_result)) {
                ni_log(NI_LOG_ERROR, "Bad filter encoder thread result: %d\n", (int)((long)filter_encoder_result));
            }
        }
    }

    time_diff = (uint32_t)(current_time.tv_sec - start_time.tv_sec);
    if (time_diff == 0) {
        time_diff = 1;
    }
    ni_log(NI_LOG_INFO, "[DEC] Frames= %u  fps=%f \n",
           number_of_frames, (float)number_of_frames / (float)time_diff);
    for (i = 0; i < output_num; i++) {
        ni_log(NI_LOG_INFO, "[ENC %d] Packets= %u fps=%f\n",
               i, filter_enc_worker[i]->encoder_worker.enc_ctx.frame_num,
               (float)filter_enc_worker[i]->encoder_worker.enc_ctx.frame_num / (float)time_diff);
    }

    if(p_up_hwframe)
        ni_hwframe_buffer_recycle2(p_up_hwframe);

    ni_device_session_close(&dec_ctx, 1, NI_DEVICE_TYPE_DECODER);
    ni_device_session_context_clear(&dec_ctx);
    ni_frame_buffer_free(&(dec_out_frame.data.frame));

    ni_device_session_close(&upl_ctx, 1, NI_DEVICE_TYPE_UPLOAD);
    ni_device_session_context_clear(&upl_ctx);
    ni_frame_buffer_free(&(upload_frame.data.frame));


    for (i = 0; i < output_num; i++) {
        free_filter_encoder_worker(filter_enc_worker[i]);
    }

end:
    for(i = 0; i < MAX_INPUT_FILES; i++) {
        if (file_reader[i].file_cache) {
            free(file_reader[i].file_cache);
        }
        if (input_file_fd[i] > 0) {
            close(input_file_fd[i]);
        }
    }

    for(i = 0; i < output_num; i++) {
        if (filter_enc_worker[i]) {
            if (filter_enc_worker[i]->encoder_worker.p_file) {
                fclose(filter_enc_worker[i]->encoder_worker.p_file);
            }
            if (filter_enc_worker[i]->thread_state > UNINIT) {
                pthread_mutex_destroy(&filter_enc_worker[i]->frm_lock);
                pthread_cond_destroy(&filter_enc_worker[i]->frm_cond);
            }

            free(filter_enc_worker[i]);
            filter_enc_worker[i] = NULL;
        }
    }
    ni_log(NI_LOG_INFO, "All done\n");

    return 0;
}
