// Copyright (c) "2022" Advanced Micro Devices, Inc. All rights reserved.

#pragma once
#include "encoding_device.h"

#include <vulkan/vulkan.hpp>

#include <options.h>
#include <renderer/surface_render_context.h>
#include <vk_helper/vk_swapchain_rsc.h>

// Instead of doing encoding, this fake device would render the received, to-be-encoded image
// onto a WindowSurface and present it.
// It is used merely to ensure the transfering between rendering device and encoding device is sane.
class FakeGpuEncoding : public IEncodeDevice, public ISurfaceRenderContext {
public:
    typedef struct {
        vk::CommandBuffer cmd;
        vk::Framebuffer frame_buffer;
        vk::DescriptorSet descriptor_set;
        vk::Image exported_image;
        vk::ImageView exported_image_view;
        vk::DeviceMemory exported_image_memory;
    } ShowExportedImageRscs;

    FakeGpuEncoding();
    ~FakeGpuEncoding() override {}

    void device_init(const Option &) override;
    void device_cleanup() override;

    FrameIndex get_max_number_of_inflight_commands() override;

    void import_semaphore_encoding_wait_semaphore(SyncFileFd syncFd, FrameIndex index) override;

    SyncFileFd export_semaphore_encoding_signal_semaphore(FrameIndex index) override;

    ResourceIndex get_max_number_of_shared_images() override {
        return ResourceIndex{swapchain_rsc->get_swapchain_image_count()};
    }

    DmaBufFd allocate_shared_image_by_index(ResourceIndex index,
                                            int32_t width,
                                            int32_t height) override;

    size_t get_shared_image_size_by_index(ResourceIndex index) override;

    void get_shared_image_plane_range_by_index(ResourceIndex index,
                                               ImagePlaneAspect image_aspect,
                                               size_t *offset,
                                               size_t *size) override;

    ResourceIndex acquire_shared_image(FrameIndex frame_index) override;

    void encoding_on_memory(uint32_t cur_frame,
                            TimestampRecord *p_timestamp_record,
                            FrameIndex frame_index,
                            ResourceIndex rsc_index) override;

    void set_window_surface(WindowSurfaceContext *window_surface) {
        swapchain_rsc = std::make_unique<VkSwapChainResources>(window_surface);
    }

    void draw(uint32_t cur_frame, TimestampRecord *p_timestamp_record) override {
        assert(false && "never called!");
    }
    void resize() override;

    int32_t width() {
        return swapchain_rsc->width();
    }
    int32_t height() {
        return swapchain_rsc->height();
    }
    int32_t get_physical_device_index() const {
        return m_selected_gpu_index;
    }

private:
    void init_vk_device();
    void init_ext_api();
    void init_device_memory_properties();

    void init_shared_images();
    void init_descriptor_layout();
    void init_render_pass();
    void init_pipeline();
    void init_descriptor_set();
    void init_framebuffer();
    void init_sync_primitives();
    void build_cmds();

    vk::ShaderModule prepare_vs();
    vk::ShaderModule prepare_fs();

    Option m_option;

    std::unique_ptr<VkSwapChainResources> swapchain_rsc;

    uint32_t m_enabled_device_extension_count;
    const char *m_device_extension_names[64];
    uint32_t m_enabled_layer_count;
    const char *m_layer_names[64];

    vk::Instance m_instance;
    PFN_vkGetMemoryFdKHR pfnGetMemoryFdKHR;
    PFN_vkGetSemaphoreFdKHR pfnGetSemaphoreFdKHR;
    PFN_vkImportSemaphoreFdKHR pfnImportSemaphoreFdKHR;
    PFN_vkCreateSamplerYcbcrConversionKHR pfnCreateSamplerYcbcrConversionKHR;
    PFN_vkDestroySamplerYcbcrConversionKHR pfnDestroySamplerYcbcrConversionKHR;

    int32_t m_selected_gpu_index;
    vk::PhysicalDevice m_gpu;
    vk::PhysicalDeviceProperties m_gpu_props;
    vk::PhysicalDeviceMemoryProperties m_gpu_memory_props;

    uint32_t m_queue_family_count;
    std::unique_ptr<vk::QueueFamilyProperties[]> m_queue_props;

    uint32_t m_graphics_queue_family_index = UINT32_MAX;
    vk::Queue m_graphics_queue;
    vk::Device m_device;

    vk::MemoryRequirements m_exported_image_memory_requirements;
    uint32_t m_exported_image_memory_type_index;

    std::unique_ptr<vk::Semaphore[]> m_swapchain_image_acquired_semaphores;
    std::unique_ptr<vk::Semaphore[]> m_show_exported_image_complete_semaphores;
    std::unique_ptr<vk::Semaphore[]> m_acquire_exported_image_semaphores;
    std::unique_ptr<vk::Fence[]> m_fences;
    std::unique_ptr<vk::Fence[]> m_swp_chain_acquire_image_fences;

    vk::CommandPool m_cmd_pool;
    std::unique_ptr<ShowExportedImageRscs[]> m_show_exported_image_rscs;

    vk::PipelineCache m_pipeline_cache;

    vk::PipelineLayout m_pipeline_layout;
    vk::DescriptorSetLayout m_desc_layout;
    vk::RenderPass m_render_pass;
    vk::Pipeline m_pipeline;

    vk::DescriptorPool m_desc_pool;
    vk::DescriptorSet m_desc_set;

    vk::Sampler m_texture_sampler;
    VkSamplerYcbcrConversion m_ycbcr_conversion;
};
