// Copyright (c) "2022" Advanced Micro Devices, Inc. All rights reserved.

#include <signal.h>
#include <sys/time.h>
#include <unistd.h>
#include <chrono>
#include <condition_variable>
#include <fstream>
#include <mutex>
#include <thread>

#include <encoder/fake_gpu_encoding.h>
#include <options.h>
#include <renderer/cube.h>
#include <util/gettime.h>
#include <util/timestamp_record.h>
#include <vk_helper/window_surface.h>

#if BUILD_NETINT_ENCODING_SUPPORT
#include <encoder/netint_encoding.h>
#endif

static bool is_offscreen_running = true;

void sig_handler(int signo) {
    if (signo == SIGINT) {
        is_offscreen_running = false;
    }
}

std::mutex mtx[2];
std::condition_variable cv[2];
volatile bool ready[2] = {false, false};

void encode_thread(TimestampRecord *p_timestamp_record,
                   IEncodeDevice *encoder,
                   uint32_t frame_count) {
    if (signal(SIGINT, sig_handler) == SIG_ERR) {
        fprintf(stderr, "Not able to register SIGINT handler!\n");
        return;
    }

    for (uint32_t i = 0; is_offscreen_running && i < frame_count; ++i) {
        uint32_t idx = i % 2;
        std::unique_lock<std::mutex> lck(mtx[idx]);
        while (!ready[idx])
            cv[idx].wait(lck);
        encoder->encoding_on_memory(i, p_timestamp_record, idx, idx);
        ready[idx] = false;
    }
}

void offscreen_draw(TimestampRecord *p_timestamp_record,
                    uint32_t frame_count,
                    const std::unique_ptr<ISurfaceRenderContext> &renderer,
                    const std::unique_ptr<IEncodeDevice> &encoder) {
    // If you are using offscreen with NetInt, change FPS needs to change NetInt
    // config as well.
    constexpr int frame_per_seconds = 30;

    if (signal(SIGINT, sig_handler) == SIG_ERR) {
        fprintf(stderr, "Not able to register SIGINT handler!\n");
        return;
    }

    std::chrono::steady_clock::duration second = std::chrono::seconds{1};
    std::chrono::steady_clock::duration frame_time = second / frame_per_seconds;
    std::chrono::steady_clock::time_point target;
    std::thread th(encode_thread, p_timestamp_record, encoder.get(), frame_count);
    for (uint32_t i = 0; is_offscreen_running && i < frame_count; ++i) {
        target = std::chrono::steady_clock::now() + frame_time;
        uint32_t idx = i % 2;
        while (ready[idx])
            std::this_thread::yield();
        {
            std::unique_lock<std::mutex> lck(mtx[idx]);
            renderer->draw(i, p_timestamp_record);
            ready[idx] = true;
            cv[idx].notify_all();
        }
        std::this_thread::sleep_until(target);
    }
    th.join();
}

int main(int argc, char **argv) {
    Option option = parse_arguments_to_options(argc, argv);
    if (option.reconfBR_and_FKF_test && option.encXcoderParams &&
        !strstr(option.encXcoderParams, "RcEnable=1")) {
        fprintf(stderr, "Cannot use --reconfBR_and_FKF_test without enabling "
                        "rate control\n");
        exit(1);
    }
#if defined(VK_USE_PLATFORM_XLIB_KHR)
    option.use_xlib = true;
#endif

    WindowSurfaceContext window(option);

#if !defined(OFFSCREEN_RENDER)
    window.init_connection();
#endif

#if defined(VK_USE_PLATFORM_XCB_KHR)
    window.create_xcb_window();
#elif defined(VK_USE_PLATFORM_XLIB_KHR)
    window.create_xlib_window();
#elif defined(VK_USE_PLATFORM_WAYLAND_KHR)
    window.create_window();
#elif defined(VK_USE_PLATFORM_DIRECTFB_EXT)
    window.create_directfb_window();
#endif

    std::unique_ptr<ISurfaceRenderContext> rendering_context = {};
    std::unique_ptr<IEncodeDevice> encoding_device = {};

    switch (option.encodingMode) {
    case EncodingMode::NoEncoding: {
        Cube *cube = new Cube(option);
        cube->set_window_surface(&window);
        rendering_context.reset(static_cast<ISurfaceRenderContext *>(cube));
        rendering_context->init();

        window.set_draw_callback(rendering_context.get());
        window.set_resize_callback(rendering_context.get());
    } break;

    case EncodingMode::FakeGpuEncoding_RGBA:
    case EncodingMode::FakeGpuEncoding_NV12:
    case EncodingMode::FakeGpuEncoding_YUV420p: {
        // Fake GPU encoding device does not encode, instead it displays its
        // received, to-be-encoded image to a window surface. So that, it can be
        // used to verify the transfer between rendering and encoding is
        // functionally working.
        FakeGpuEncoding *fake_gpu_encoding = new FakeGpuEncoding();
        fake_gpu_encoding->set_window_surface(&window);
        encoding_device.reset(static_cast<IEncodeDevice *>(fake_gpu_encoding));
        encoding_device->device_init(option);

        Cube *cube = new Cube(option);
        cube->set_encoding_device(encoding_device.get());
        rendering_context.reset(static_cast<ISurfaceRenderContext *>(cube));
        rendering_context->init();

        // draw issues to the rendering device.
        window.set_draw_callback(rendering_context.get());

        // resize goes to fake GPU encoding, because it is tied to window
        // surface and swapchain. The render context in fake encoding mode would
        // keep rendering to the initial width and height as specified in
        // option.
        window.set_resize_callback(static_cast<ISurfaceRenderContext *>(fake_gpu_encoding));
    } break;
#if BUILD_NETINT_ENCODING_SUPPORT
    case EncodingMode::NetintEncoding_YUV420p: {
        NetintEncoding *netint_encoding = new NetintEncoding();
        encoding_device.reset(static_cast<IEncodeDevice *>(netint_encoding));
        encoding_device->device_init(option);

        Cube *cube = new Cube(option);
        cube->set_encoding_device(encoding_device.get());
        rendering_context.reset(static_cast<ISurfaceRenderContext *>(cube));
        rendering_context->init();
    } break;
#endif

    default:
        assert(false && "encoding mode not supported");
        break;
    }

    TimestampRecord timestamp_record(option);

#if defined(VK_USE_PLATFORM_XCB_KHR)
    window.run_xcb(&timestamp_record);
#elif defined(VK_USE_PLATFORM_XLIB_KHR)
    window.run_xlib(&timestamp_record);
#elif defined(VK_USE_PLATFORM_WAYLAND_KHR)
    window.run(&timestamp_record);
#elif defined(VK_USE_PLATFORM_DIRECTFB_EXT)
    window.run_directfb(&timestamp_record);
#elif defined(VK_USE_PLATFORM_DISPLAY_KHR)
    window.run_display(&timestamp_record);
#elif defined(OFFSCREEN_RENDER)
    offscreen_draw(&timestamp_record, option.frameCount, rendering_context, encoding_device);
#endif

    if (encoding_device.get() != nullptr) {
        encoding_device->device_cleanup();
    }
    rendering_context->cleanup();

    window.cleanup();

    timestamp_record.report();

    return 0;
}
