// Modifications Copyright (C)2022 Advanced Micro Devices, Inc. All rights reserved.
// Notified per clause 4(b) of the license

/*
 * Copyright (c) 2015-2019 The Khronos Group Inc.
 * Copyright (c) 2015-2019 Valve Corporation
 * Copyright (c) 2015-2019 LunarG, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Authors: Jeremy Hayes <jeremy@lunarg.com>
 */

#include "options.h"

#include <sys/stat.h>
#include <sstream>

Option parse_arguments_to_options(int argc, char **argv) {
    Option option = {};
    char *path_buffer = nullptr;

    for (int i = 1; i < argc; i++) {
        if (strcmp(argv[i], "--use_staging") == 0) {
            option.use_staging_buffer = true;
            continue;
        }
        if ((strcmp(argv[i], "--present_mode") == 0) && (i < argc - 1)) {
            option.presentMode = (vk::PresentModeKHR)atoi(argv[i + 1]);
            i++;
            continue;
        }
        if ((strcmp(argv[i], "--encoding_mode") == 0) && (i < argc - 1)) {
            option.encodingMode = (EncodingMode)atoi(argv[i + 1]);
            i++;
            continue;
        }
        if (strcmp(argv[i], "--break") == 0) {
            option.use_break = true;
            continue;
        }
        if (strcmp(argv[i], "--validate") == 0) {
            option.validate = true;
            continue;
        }
        if (strcmp(argv[i], "--xlib") == 0) {
            fprintf(stderr, "--xlib is deprecated and no longer does anything");
            continue;
        }
        if (strcmp(argv[i], "--c") == 0 && option.frameCount == UINT32_MAX && i < argc - 1 &&
            sscanf(argv[i + 1], "%" SCNu32, &option.frameCount) == 1) {
            i++;
            continue;
        }
        if (strcmp(argv[i], "--width") == 0 && i < argc - 1 &&
            sscanf(argv[i + 1], "%" SCNi32, &option.width) == 1 && option.width > 0) {
            i++;
            continue;
        }
        if (strcmp(argv[i], "--height") == 0 && i < argc - 1 &&
            sscanf(argv[i + 1], "%" SCNi32, &option.height) == 1 && option.height > 0) {
            i++;
            continue;
        }
        if (strcmp(argv[i], "--suppress_popups") == 0) {
            option.suppress_popups = true;
            continue;
        }
        if ((strcmp(argv[i], "--gpu_number") == 0) && (i < argc - 1)) {
            option.gpu_number = atoi(argv[i + 1]);
            assert(option.gpu_number >= 0);
            i++;
            continue;
        }
        if (strcmp(argv[i], "--output_directory") == 0 && i < argc - 1 &&
            sscanf(argv[i + 1], "%ms", &path_buffer) == 1) {
            const size_t path_len = strlen(path_buffer);
            int ret = mkdir(path_buffer, 0755);
            if (ret != 0 && errno != EEXIST) {
                option.output_dir_path = nullptr;
            } else if (path_len > 0 && path_buffer[path_len - 1] != '/') {
                // We don't bother to free it, let OS do the cleanup
                option.output_dir_path = (char *)malloc(path_len + 2);
                option.output_dir_path[0] = '\0';
                strncat(option.output_dir_path, path_buffer, path_len);
                option.output_dir_path[path_len] = '/';
                option.output_dir_path[path_len + 1] = '\0';
            } else {
                option.output_dir_path = path_buffer;
            }
            i++;
            continue;
        }
        if (strcmp(argv[i], "--debug_always_cpu_wait") == 0) {
            option.debug_always_cpu_wait = true;
            continue;
        }
        if (strcmp(argv[i], "--debug_fake_encoding_same_gpu") == 0) {
            option.debug_fake_encoding_same_gpu = true;
            continue;
        }
        if (strcmp(argv[i], "--video_codec") == 0) {
            option.videoCodec = (VideoCodec)atoi(argv[i + 1]);
            i++;
            assert(option.videoCodec >= VideoCodec::H265_Codec &&
                   option.videoCodec <= VideoCodec::AV1_Codec);
            continue;
        }
        if (strcmp(argv[i], "--log_level") == 0) {
            option.log_level = atoi(argv[i + 1]);
            i++;

            assert(option.log_level >= 0 && option.log_level <= 5);
            continue;
        }
        if (strcmp(argv[i], "--xcoder-params") == 0) {
            strcpy(option.encXcoderParams, argv[i + 1]);
            i++;
            continue;
        }
        if (strcmp(argv[i], "--reconfBR_and_FKF_test") == 0) {
            option.reconfBR_and_FKF_test = 1;
            unsigned int rand_seed = (unsigned int) time(NULL);
            printf("--reconfBR_and_FKF_test using random seed: %u\n", rand_seed);
            srand(rand_seed);
            i++;
            continue;
        }
        std::stringstream usage;
        usage << "Usage:\n  " << APP_SHORT_NAME << "\t[--use_staging] [--validate]\n"
              << "\t[--break] [--c <framecount>] [--suppress_popups]\n"
              << "\t[--gpu_number <index of physical device>]\n"
              << "\t[--present_mode <present mode enum>]\n"
              << "\t[--encoding_mode <encoding mode enum>]\n"
              << "\t[--width <width>] [--height <height>]\n"
              << "\t[--debug_always_cpu_wait] [--debug_fake_encoding_same_gpu]\n"
              << "\t[--video_codec <codec enum>]\n"
              << "\t[--log_level <log_level>]\n"
              << "\t[--xcoder-params <encoder params>]\n"
              << "\t[--reconfBR_and_FKF_test]\n"
              << "\t<present_mode_enum>\n"
              << "\t\tVK_PRESENT_MODE_IMMEDIATE_KHR = " << VK_PRESENT_MODE_IMMEDIATE_KHR << "\n"
              << "\t\tVK_PRESENT_MODE_MAILBOX_KHR = " << VK_PRESENT_MODE_MAILBOX_KHR << "\n"
              << "\t\tVK_PRESENT_MODE_FIFO_KHR = " << VK_PRESENT_MODE_FIFO_KHR << "\n"
              << "\t\tVK_PRESENT_MODE_FIFO_RELAXED_KHR = " << VK_PRESENT_MODE_FIFO_RELAXED_KHR
              << "\n"
              << "\t<encoding_mode_enum>\n"
              << "\t\tNo Encoding(requires WSI) = " << (int)EncodingMode::NoEncoding << "\n"
              << "\t\tFake GPU Encoding RGBA(requires WSI) = "
              << (int)EncodingMode::FakeGpuEncoding_RGBA << "\n"
              << "\t\tFake GPU Encoding NV12(requires WSI) = "
              << (int)EncodingMode::FakeGpuEncoding_NV12 << "\n"
              << "\t\tFake Gpu Encoding YUV420p(requires WSI)= "
              << (int)EncodingMode::FakeGpuEncoding_YUV420p << "\n"
              << "\t\tNetint Encoding YUV420p = " << (int)EncodingMode::NetintEncoding_YUV420p
              << "\n"
              << "\t<codec enum>\n"
              << "\t\tH265 = " << (int)VideoCodec::H265_Codec << "\n"
              << "\t\tH264 = " << (int)VideoCodec::H264_Codec << "\n"
              << "\t\tAV1  = " << (int)VideoCodec::AV1_Codec << "\n"
              << "\t<log_level>\n"
              << "\t\t0 = none\n"
              << "\t\t1 = fatal\n"
              << "\t\t2 = error\n"
              << "\t\t3 = info\n"
              << "\t\t4 = debug\n"
              << "\t\t5 = trace\n"
              << "\t\t<xcoder_params> Note: lowDelay=1, gopPresetIdx=9 are always forced";

#if defined(_WIN32)
        if (!suppress_popups)
            MessageBox(NULL, usage.str().c_str(), "Usage Error", MB_OK);
#else
        std::cerr << usage.str();
        std::cerr.flush();
#endif
        exit(1);
    }

    return option;
}
