// Modifications Copyright (C)2022 Advanced Micro Devices, Inc. All rights reserved.
// Notified per clause 4(b) of the license

/*
 * Copyright (c) 2015-2019 The Khronos Group Inc.
 * Copyright (c) 2015-2019 Valve Corporation
 * Copyright (c) 2015-2019 LunarG, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Authors: Jeremy Hayes <jeremy@lunarg.com>
 */

#pragma once

#include <vulkan/vulkan.hpp>

#include "window_surface.h"

typedef struct {
    vk::Image image;
    vk::ImageView view;
} SwapchainImageResources;

class VkSwapChainResources {
public:
    explicit VkSwapChainResources(WindowSurfaceContext *window_surface);
    ~VkSwapChainResources() {}

    VkSwapChainResources(const VkSwapChainResources &) = delete;
    VkSwapChainResources &operator=(const VkSwapChainResources &) = delete;

    void
    init_surface_and_select_queue(vk::Instance inst,
                                  vk::PhysicalDevice gpu,
                                  uint32_t queue_family_count,
                                  const std::unique_ptr<vk::QueueFamilyProperties[]> &queue_props,
                                  uint32_t *out_present_queue_family_index_with_graphics);
    void cleanup(vk::Instance inst, vk::Device device);

    void set_present_queue(vk::Device device) {
        device.getQueue(present_queue_family_index, 0, &present_queue);
    }

    void prepare_images(vk::Device device,
                        vk::PhysicalDevice gpu,
                        const vk::PresentModeKHR presentMode);
    void destroy_images(vk::Device device);

    uint32_t acquire_next_image(vk::Instance inst,
                                vk::Device device,
                                vk::Fence image_acquire_fence,
                                vk::Semaphore acquire_semaphore);

    void present_image(vk::Instance inst,
                       vk::PhysicalDevice gpu,
                       vk::Device device,
                       vk::Semaphore draw_complete_semaphore,
                       uint32_t active_image_index);

    int32_t width() const {
        return m_window_surface->get_width();
    }

    int32_t height() const {
        return m_window_surface->get_height();
    }

    void set_width(int32_t w) {
        m_window_surface->set_width(w);
    }
    void set_height(int32_t h) {
        m_window_surface->set_height(h);
    }

    uint32_t get_swapchain_image_count() const {
        assert(swapchain_image_count > 0);
        return swapchain_image_count;
    }

    vk::ImageView get_image_view(uint32_t index) const {
        return swapchain_image_resources[index].view;
    }
    vk::Format get_image_format() const {
        return format;
    }

private:
    WindowSurfaceContext *m_window_surface;

    vk::Queue present_queue;
    uint32_t present_queue_family_index;

    vk::Format format;
    vk::ColorSpaceKHR color_space;

    vk::SwapchainKHR swapchain;
    vk::SurfaceKHR surface;

    uint32_t swapchain_image_count;
    std::unique_ptr<SwapchainImageResources[]> swapchain_image_resources;
};
