// Modifications Copyright (C)2022 Advanced Micro Devices, Inc. All rights reserved.
// Notified per clause 4(b) of the license

/*
 * Copyright (c) 2015-2019 The Khronos Group Inc.
 * Copyright (c) 2015-2019 Valve Corporation
 * Copyright (c) 2015-2019 LunarG, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Authors: Jeremy Hayes <jeremy@lunarg.com>
 */

#include "window_surface.h"

#include <renderer/surface_render_context.h>
#include <util/timestamp_record.h>

WindowSurfaceContext::WindowSurfaceContext(const Option &option)
    :
#if defined(VK_USE_PLATFORM_XLIB_KHR)
      xlib_window{0}, xlib_wm_delete_window{0}, display{nullptr},
#elif defined(VK_USE_PLATFORM_XCB_KHR)
      xcb_window{0}, screen{nullptr}, connection{nullptr},
#elif defined(VK_USE_PLATFORM_WAYLAND_KHR)
      display{nullptr}, registry{nullptr}, compositor{nullptr}, window{nullptr}, wm_base{nullptr},
      xdg_decoration_mgr{nullptr}, toplevel_decoration{nullptr}, window_surface{nullptr},
      xdg_surface_has_been_configured{false},
      window_toplevel{nullptr}, seat{nullptr}, pointer{nullptr}, keyboard{nullptr},
#elif defined(VK_USE_PLATFORM_DIRECTFB_EXT)
      dfb{nullptr}, window{nullptr}, event_buffer{nullptr},
#endif
      p_draw_render_context(nullptr), p_resize_render_context(nullptr),
      kMaxFrameCount(option.frameCount), width{option.width}, height{option.height}, pause{false},
      quit(false), curFrame{0} {
}

void WindowSurfaceContext::cleanup() {
#if defined(VK_USE_PLATFORM_XLIB_KHR)
    XDestroyWindow(display, xlib_window);
    XCloseDisplay(display);
#elif defined(VK_USE_PLATFORM_XCB_KHR)
    xcb_destroy_window(connection, xcb_window);
    xcb_disconnect(connection);
    free(atom_wm_delete_window);
#elif defined(VK_USE_PLATFORM_WAYLAND_KHR)
    wl_keyboard_destroy(keyboard);
    wl_pointer_destroy(pointer);
    wl_seat_destroy(seat);
    xdg_toplevel_destroy(window_toplevel);
    xdg_surface_destroy(window_surface);
    wl_surface_destroy(window);
    xdg_wm_base_destroy(wm_base);
    if (xdg_decoration_mgr) {
        zxdg_toplevel_decoration_v1_destroy(toplevel_decoration);
        zxdg_decoration_manager_v1_destroy(xdg_decoration_mgr);
    }
    wl_compositor_destroy(compositor);
    wl_registry_destroy(registry);
    wl_display_disconnect(display);
#elif defined(VK_USE_PLATFORM_DIRECTFB_EXT)
    event_buffer->Release(event_buffer);
    window->Release(window);
    dfb->Release(dfb);
#endif
}

void WindowSurfaceContext::draw_callback(uint32_t cur_frame, TimestampRecord *p_timestamp_record) {
    p_draw_render_context->draw(cur_frame, p_timestamp_record);
}

void WindowSurfaceContext::resize_callback() {
    p_resize_render_context->resize();
}

void WindowSurfaceContext::init_connection() {
#if defined(VK_USE_PLATFORM_XCB_KHR)
    const xcb_setup_t *setup;
    xcb_screen_iterator_t iter;
    int scr;

    const char *display_envar = getenv("DISPLAY");
    if (display_envar == nullptr || display_envar[0] == '\0') {
        printf("Environment variable DISPLAY requires a valid value.\nExiting ...\n");
        fflush(stdout);
        exit(1);
    }

    connection = xcb_connect(nullptr, &scr);
    if (xcb_connection_has_error(connection) > 0) {
        printf("Cannot find a compatible Vulkan installable client driver "
               "(ICD).\nExiting ...\n");
        fflush(stdout);
        exit(1);
    }

    setup = xcb_get_setup(connection);
    iter = xcb_setup_roots_iterator(setup);
    while (scr-- > 0)
        xcb_screen_next(&iter);

    screen = iter.data;
#elif defined(VK_USE_PLATFORM_WAYLAND_KHR)
    display = wl_display_connect(nullptr);

    if (display == nullptr) {
        printf("Cannot find a compatible Vulkan installable client driver (ICD).\nExiting ...\n");
        fflush(stdout);
        exit(1);
    }

    registry = wl_display_get_registry(display);
    wl_registry_add_listener(registry, &registry_listener, this);
    wl_display_dispatch(display);
#endif
}

#if defined(VK_USE_PLATFORM_XLIB_KHR)

void WindowSurfaceContext::create_xlib_window() {
    const char *display_envar = getenv("DISPLAY");
    if (display_envar == nullptr || display_envar[0] == '\0') {
        printf("Environment variable DISPLAY requires a valid value.\nExiting ...\n");
        fflush(stdout);
        exit(1);
    }

    XInitThreads();
    display = XOpenDisplay(nullptr);
    long visualMask = VisualScreenMask;
    int numberOfVisuals;
    XVisualInfo vInfoTemplate = {};
    vInfoTemplate.screen = DefaultScreen(display);
    XVisualInfo *visualInfo = XGetVisualInfo(display, visualMask, &vInfoTemplate, &numberOfVisuals);

    Colormap colormap = XCreateColormap(
        display, RootWindow(display, vInfoTemplate.screen), visualInfo->visual, AllocNone);

    XSetWindowAttributes windowAttributes = {};
    windowAttributes.colormap = colormap;
    windowAttributes.background_pixel = 0xFFFFFFFF;
    windowAttributes.border_pixel = 0;
    windowAttributes.event_mask =
        KeyPressMask | KeyReleaseMask | StructureNotifyMask | ExposureMask;

    xlib_window = XCreateWindow(display,
                                RootWindow(display, vInfoTemplate.screen),
                                0,
                                0,
                                width,
                                height,
                                0,
                                visualInfo->depth,
                                InputOutput,
                                visualInfo->visual,
                                CWBackPixel | CWBorderPixel | CWEventMask | CWColormap,
                                &windowAttributes);

    XSelectInput(display, xlib_window, ExposureMask | KeyPressMask);
    XMapWindow(display, xlib_window);
    XFlush(display);
    xlib_wm_delete_window = XInternAtom(display, "WM_DELETE_WINDOW", False);
}

void WindowSurfaceContext::handle_xlib_event(const XEvent *event) {
    switch (event->type) {
    case ClientMessage:
        if ((Atom)event->xclient.data.l[0] == xlib_wm_delete_window) {
            quit = true;
        }
        break;
    case KeyPress:
        switch (event->xkey.keycode) {
        case 0x9: // Escape
            quit = true;
            break;
        case 0x71: // left arrow key
                   //    spin_angle -= spin_increment;
            break;
        case 0x72: // right arrow key
                   //    spin_angle += spin_increment;
            break;
        case 0x41: // space bar
            pause = !pause;
            break;
        }
        break;
    case ConfigureNotify:
        if (((int32_t)width != event->xconfigure.width) ||
            ((int32_t)height != event->xconfigure.height)) {
            width = event->xconfigure.width;
            height = event->xconfigure.height;
            resize_callback();
        }
        break;
    default:
        break;
    }
}

void WindowSurfaceContext::run_xlib(TimestampRecord *p_timestamp_record) {
    while (!quit) {
        XEvent event;

        if (pause) {
            XNextEvent(display, &event);
            handle_xlib_event(&event);
        }
        while (XPending(display) > 0) {
            XNextEvent(display, &event);
            handle_xlib_event(&event);
        }

        draw_callback(curFrame, p_timestamp_record);
        curFrame++;

        if (kMaxFrameCount != UINT32_MAX && curFrame == kMaxFrameCount) {
            quit = true;
        }
    }
}
#elif defined(VK_USE_PLATFORM_XCB_KHR)

void WindowSurfaceContext::handle_xcb_event(const xcb_generic_event_t *event) {
    uint8_t event_code = event->response_type & 0x7f;
    switch (event_code) {
    case XCB_EXPOSE:
        // TODO: Resize window
        break;
    case XCB_CLIENT_MESSAGE:
        if ((*(xcb_client_message_event_t *)event).data.data32[0] ==
            (*atom_wm_delete_window).atom) {
            quit = true;
        }
        break;
    case XCB_KEY_RELEASE: {
        const xcb_key_release_event_t *key = (const xcb_key_release_event_t *)event;

        switch (key->detail) {
        case 0x9: // Escape
            quit = true;
            break;
        case 0x71: // left arrow key
                   //    spin_angle -= spin_increment;
            break;
        case 0x72: // right arrow key
                   //    spin_angle += spin_increment;
            break;
        case 0x41: // space bar
            pause = !pause;
            break;
        }
    } break;
    case XCB_CONFIGURE_NOTIFY: {
        const xcb_configure_notify_event_t *cfg = (const xcb_configure_notify_event_t *)event;
        if ((width != cfg->width) || (height != cfg->height)) {
            width = cfg->width;
            height = cfg->height;
            resize_callback();
        }
    } break;
    default:
        break;
    }
}

void WindowSurfaceContext::run_xcb(TimestampRecord *p_timestamp_record) {
    xcb_flush(connection);

    while (!quit) {
        xcb_generic_event_t *event;

        if (pause) {
            event = xcb_wait_for_event(connection);
        } else {
            event = xcb_poll_for_event(connection);
        }
        while (event) {
            handle_xcb_event(event);
            free(event);
            event = xcb_poll_for_event(connection);
        }

        draw_callback(curFrame, p_timestamp_record);
        curFrame++;
        if (kMaxFrameCount != UINT32_MAX && curFrame == kMaxFrameCount) {
            quit = true;
        }
    }
}

void WindowSurfaceContext::create_xcb_window() {
    uint32_t value_mask, value_list[32];

    xcb_window = xcb_generate_id(connection);

    value_mask = XCB_CW_BACK_PIXEL | XCB_CW_EVENT_MASK;
    value_list[0] = screen->black_pixel;
    value_list[1] =
        XCB_EVENT_MASK_KEY_RELEASE | XCB_EVENT_MASK_EXPOSURE | XCB_EVENT_MASK_STRUCTURE_NOTIFY;

    xcb_create_window(connection,
                      XCB_COPY_FROM_PARENT,
                      xcb_window,
                      screen->root,
                      0,
                      0,
                      width,
                      height,
                      0,
                      XCB_WINDOW_CLASS_INPUT_OUTPUT,
                      screen->root_visual,
                      value_mask,
                      value_list);

    /* Magic code that will send notification when window is destroyed */
    xcb_intern_atom_cookie_t cookie = xcb_intern_atom(connection, 1, 12, "WM_PROTOCOLS");
    xcb_intern_atom_reply_t *reply = xcb_intern_atom_reply(connection, cookie, 0);

    xcb_intern_atom_cookie_t cookie2 = xcb_intern_atom(connection, 0, 16, "WM_DELETE_WINDOW");
    atom_wm_delete_window = xcb_intern_atom_reply(connection, cookie2, 0);

    xcb_change_property(connection,
                        XCB_PROP_MODE_REPLACE,
                        xcb_window,
                        (*reply).atom,
                        4,
                        32,
                        1,
                        &(*atom_wm_delete_window).atom);

    free(reply);

    xcb_map_window(connection, xcb_window);

    // Force the x/y coordinates to 100,100 results are identical in
    // consecutive
    // runs
    const uint32_t coords[] = {100, 100};
    xcb_configure_window(connection, xcb_window, XCB_CONFIG_WINDOW_X | XCB_CONFIG_WINDOW_Y, coords);
}
#elif defined(VK_USE_PLATFORM_WAYLAND_KHR)

void WindowSurfaceContext::run(TimestampRecord *p_timestamp_record) {
    while (!quit) {
        if (pause) {
            wl_display_dispatch(display);
        } else {
            wl_display_dispatch_pending(display);
            update_data_buffer();
            draw_callback(curFrame, p_timestamp_record);
            curFrame++;
            if (kMaxFrameCount != UINT32_MAX && curFrame == kMaxFrameCount) {
                quit = true;
            }
        }
    }
}

static void handle_surface_configure(void *data, xdg_surface *xdg_surface, uint32_t serial) {
    WindowSurfaceContext *demo = (WindowSurfaceContext *)data;
    xdg_surface_ack_configure(xdg_surface, serial);
    if (demo->xdg_surface_has_been_configured) {
        demo->resize_callback();
    }
    demo->xdg_surface_has_been_configured = true;
}

static const xdg_surface_listener surface_listener = {handle_surface_configure};

static void handle_toplevel_configure(void *data,
                                      xdg_toplevel *xdg_toplevel,
                                      int32_t width,
                                      int32_t height,
                                      struct wl_array *states) {
    WindowSurfaceContext *demo = (WindowSurfaceContext *)data;
    /* zero values imply the program may choose its own size, so in that case
     * stay with the existing value (which on startup is the default) */
    if (width > 0) {
        demo->width = width;
    }
    if (height > 0) {
        demo->height = height;
    }
    // This will be followed by a surface configure
}

static void handle_toplevel_close(void *data, xdg_toplevel *xdg_toplevel) {
    WindowSurfaceContext *demo = (WindowSurfaceContext *)data;
    demo->quit = true;
}

static const xdg_toplevel_listener toplevel_listener = {handle_toplevel_configure,
                                                        handle_toplevel_close};

void WindowSurfaceContext::create_window() {
    if (!wm_base) {
        printf("Compositor did not provide the standard protocol xdg-wm-base\n");
        fflush(stdout);
        exit(1);
    }

    window = wl_compositor_create_surface(compositor);
    if (!window) {
        printf("Can not create wayland_surface from compositor!\n");
        fflush(stdout);
        exit(1);
    }

    window_surface = xdg_wm_base_get_xdg_surface(wm_base, window);
    if (!window_surface) {
        printf("Can not get xdg_surface from wayland_surface!\n");
        fflush(stdout);
        exit(1);
    }
    window_toplevel = xdg_surface_get_toplevel(window_surface);
    if (!window_toplevel) {
        printf("Can not allocate xdg_toplevel for xdg_surface!\n");
        fflush(stdout);
        exit(1);
    }
    xdg_surface_add_listener(window_surface, &surface_listener, this);
    xdg_toplevel_add_listener(window_toplevel, &toplevel_listener, this);
    xdg_toplevel_set_title(window_toplevel, APP_SHORT_NAME);
    if (xdg_decoration_mgr) {
        // if supported, let the compositor render titlebars for us
        toplevel_decoration =
            zxdg_decoration_manager_v1_get_toplevel_decoration(xdg_decoration_mgr, window_toplevel);
        zxdg_toplevel_decoration_v1_set_mode(toplevel_decoration,
                                             ZXDG_TOPLEVEL_DECORATION_V1_MODE_SERVER_SIDE);
    }

    wl_surface_commit(window);
}

#elif defined(VK_USE_PLATFORM_DISPLAY_KHR)

vk::Result WindowSurfaceContext::create_display_surface() {
    vk::Result result;
    uint32_t display_count;
    uint32_t mode_count;
    uint32_t plane_count;
    vk::DisplayPropertiesKHR display_props;
    vk::DisplayKHR display;
    vk::DisplayModePropertiesKHR mode_props;
    vk::DisplayPlanePropertiesKHR *plane_props;
    vk::Bool32 found_plane = VK_FALSE;
    uint32_t plane_index;
    vk::Extent2D image_extent;

    display_count = 1;
    result = gpu.getDisplayPropertiesKHR(&display_count, &display_props);
    VERIFY((result == vk::Result::eSuccess) || (result == vk::Result::eIncomplete));

    display = display_props.display;

    // Get the first mode of the display
    result = gpu.getDisplayModePropertiesKHR(display, &mode_count, nullptr);
    VERIFY(result == vk::Result::eSuccess);

    if (mode_count == 0) {
        printf("Cannot find any mode for the display!\n");
        fflush(stdout);
        exit(1);
    }

    mode_count = 1;
    result = gpu.getDisplayModePropertiesKHR(display, &mode_count, &mode_props);
    VERIFY((result == vk::Result::eSuccess) || (result == vk::Result::eIncomplete));

    // Get the list of planes
    result = gpu.getDisplayPlanePropertiesKHR(&plane_count, nullptr);
    VERIFY(result == vk::Result::eSuccess);

    if (plane_count == 0) {
        printf("Cannot find any plane!\n");
        fflush(stdout);
        exit(1);
    }

    plane_props = (vk::DisplayPlanePropertiesKHR *)malloc(sizeof(vk::DisplayPlanePropertiesKHR) *
                                                          plane_count);
    VERIFY(plane_props != nullptr);

    result = gpu.getDisplayPlanePropertiesKHR(&plane_count, plane_props);
    VERIFY(result == vk::Result::eSuccess);

    // Find a plane compatible with the display
    for (plane_index = 0; plane_index < plane_count; plane_index++) {
        uint32_t supported_count;
        vk::DisplayKHR *supported_displays;

        // Disqualify planes that are bound to a different display
        if (plane_props[plane_index].currentDisplay &&
            (plane_props[plane_index].currentDisplay != display)) {
            continue;
        }

        result = gpu.getDisplayPlaneSupportedDisplaysKHR(plane_index, &supported_count, nullptr);
        VERIFY(result == vk::Result::eSuccess);

        if (supported_count == 0) {
            continue;
        }

        supported_displays = (vk::DisplayKHR *)malloc(sizeof(vk::DisplayKHR) * supported_count);
        VERIFY(supported_displays != nullptr);

        result = gpu.getDisplayPlaneSupportedDisplaysKHR(
            plane_index, &supported_count, supported_displays);
        VERIFY(result == vk::Result::eSuccess);

        for (uint32_t i = 0; i < supported_count; i++) {
            if (supported_displays[i] == display) {
                found_plane = VK_TRUE;
                break;
            }
        }

        free(supported_displays);

        if (found_plane) {
            break;
        }
    }

    if (!found_plane) {
        printf("Cannot find a plane compatible with the display!\n");
        fflush(stdout);
        exit(1);
    }

    free(plane_props);

    vk::DisplayPlaneCapabilitiesKHR planeCaps;
    gpu.getDisplayPlaneCapabilitiesKHR(mode_props.displayMode, plane_index, &planeCaps);
    // Find a supported alpha mode
    vk::DisplayPlaneAlphaFlagBitsKHR alphaMode = vk::DisplayPlaneAlphaFlagBitsKHR::eOpaque;
    vk::DisplayPlaneAlphaFlagBitsKHR alphaModes[4] = {
        vk::DisplayPlaneAlphaFlagBitsKHR::eOpaque,
        vk::DisplayPlaneAlphaFlagBitsKHR::eGlobal,
        vk::DisplayPlaneAlphaFlagBitsKHR::ePerPixel,
        vk::DisplayPlaneAlphaFlagBitsKHR::ePerPixelPremultiplied,
    };
    for (uint32_t i = 0; i < sizeof(alphaModes); i++) {
        if (planeCaps.supportedAlpha & alphaModes[i]) {
            alphaMode = alphaModes[i];
            break;
        }
    }

    image_extent.setWidth(mode_props.parameters.visibleRegion.width);
    image_extent.setHeight(mode_props.parameters.visibleRegion.height);

    auto const createInfo = vk::DisplaySurfaceCreateInfoKHR()
                                .setDisplayMode(mode_props.displayMode)
                                .setPlaneIndex(plane_index)
                                .setPlaneStackIndex(plane_props[plane_index].currentStackIndex)
                                .setGlobalAlpha(1.0f)
                                .setAlphaMode(alphaMode)
                                .setImageExtent(image_extent);

    return inst.createDisplayPlaneSurfaceKHR(&createInfo, nullptr, &surface);
}

void WindowSurfaceContext::run_display(TimestampRecord *p_timestamp_record) {
    while (!quit) {
        draw_callback(curFrame, p_timestamp_record);
        curFrame++;

        if (frameCount != UINT32_MAX && curFrame == frameCount) {
            quit = true;
        }
    }
}
#endif
