// Modifications Copyright (C)2022 Advanced Micro Devices, Inc. All rights reserved.
// Notified per clause 4(b) of the license

/*
 * Copyright (c) 2015-2019 The Khronos Group Inc.
 * Copyright (c) 2015-2019 Valve Corporation
 * Copyright (c) 2015-2019 LunarG, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Authors: Jeremy Hayes <jeremy@lunarg.com>
 */

#pragma once

#if defined(VK_USE_PLATFORM_XLIB_KHR) || defined(VK_USE_PLATFORM_XCB_KHR)
#include <X11/Xutil.h>
#elif defined(VK_USE_PLATFORM_WAYLAND_KHR)
#include "xdg-decoration-client-header.h"
#include "xdg-shell-client-header.h"
#include <linux/input.h>
#endif

#include <stdint.h>

#include "vk_common.h"
#include <options.h>

struct ISurfaceRenderContext;
class TimestampRecord;

struct WindowSurfaceContext {
    explicit WindowSurfaceContext(const Option &option);
    ~WindowSurfaceContext() {}

    void init_connection();

    void cleanup();

    int32_t get_width() {
        return width;
    }
    int32_t get_height() {
        return height;
    }

    void set_width(int32_t w) {
        width = w;
    }
    void set_height(int32_t h) {
        height = h;
    }

    vk::SurfaceKHR create_vk_surface(vk::Instance inst) {
        // Create a WSI surface for the window:
        vk::SurfaceKHR surface = {};
#if defined(VK_USE_PLATFORM_WIN32_KHR)
        {
            auto const createInfo =
                vk::Win32SurfaceCreateInfoKHR().setHinstance(connection).setHwnd(window);

            auto result = inst.createWin32SurfaceKHR(&createInfo, nullptr, &surface);
            VERIFY(result == vk::Result::eSuccess);
        }
#elif defined(VK_USE_PLATFORM_WAYLAND_KHR)
        {
            auto const createInfo =
                vk::WaylandSurfaceCreateInfoKHR().setDisplay(display).setSurface(window);

            auto result = inst.createWaylandSurfaceKHR(&createInfo, nullptr, &surface);
            VERIFY(result == vk::Result::eSuccess);
        }
#elif defined(VK_USE_PLATFORM_XLIB_KHR)
        {
            auto const createInfo =
                vk::XlibSurfaceCreateInfoKHR().setDpy(display).setWindow(xlib_window);

            auto result = inst.createXlibSurfaceKHR(&createInfo, nullptr, &surface);
            VERIFY(result == vk::Result::eSuccess);
        }
#elif defined(VK_USE_PLATFORM_XCB_KHR)
        {
            auto const createInfo =
                vk::XcbSurfaceCreateInfoKHR().setConnection(connection).setWindow(xcb_window);

            auto result = inst.createXcbSurfaceKHR(&createInfo, nullptr, &surface);
            VERIFY(result == vk::Result::eSuccess);
        }
#elif defined(VK_USE_PLATFORM_DIRECTFB_EXT)
        {
            auto const createInfo =
                vk::DirectFBSurfaceCreateInfoEXT().setDfb(dfb).setSurface(window);

            auto result = inst.createDirectFBSurfaceEXT(&createInfo, nullptr, &surface);
            VERIFY(result == vk::Result::eSuccess);
        }
#elif defined(VK_USE_PLATFORM_METAL_EXT)
        {
            auto const createInfo = vk::MetalSurfaceCreateInfoEXT().setPLayer(
                static_cast<CAMetalLayer *>(caMetalLayer));

            auto result = inst.createMetalSurfaceEXT(&createInfo, nullptr, &surface);
            VERIFY(result == vk::Result::eSuccess);
        }
#elif defined(VK_USE_PLATFORM_DISPLAY_KHR)
        {
            auto result = create_display_surface();
            VERIFY(result == vk::Result::eSuccess);
        }
#else
        assert(false && "No supported WSI platform!");
#endif
        return surface;
    }

#if defined(VK_USE_PLATFORM_XLIB_KHR)
    void create_xlib_window();
    void handle_xlib_event(const XEvent *);
    void run_xlib(TimestampRecord *p_timestamp_record);
#elif defined(VK_USE_PLATFORM_XCB_KHR)
    void handle_xcb_event(const xcb_generic_event_t *);
    void run_xcb(TimestampRecord *p_timestamp_record);
    void create_xcb_window();
#elif defined(VK_USE_PLATFORM_WAYLAND_KHR)
    void run();
    void create_window();
#elif defined(VK_USE_PLATFORM_DIRECTFB_EXT)
    void handle_directfb_event(const DFBInputEvent *);
    void run_directfb(TimestampRecord *p_timestamp_record);
    void create_directfb_window();
#elif defined(VK_USE_PLATFORM_DISPLAY_KHR)
    vk::Result create_display_surface();
    void run_display(TimestampRecord *p_timestamp_record);
#endif

#if defined(VK_USE_PLATFORM_XLIB_KHR)
    Window xlib_window;
    Atom xlib_wm_delete_window;
    Display *display;
#elif defined(VK_USE_PLATFORM_XCB_KHR)
    xcb_window_t xcb_window;
    xcb_screen_t *screen;
    xcb_connection_t *connection;
    xcb_intern_atom_reply_t *atom_wm_delete_window;
#elif defined(VK_USE_PLATFORM_WAYLAND_KHR)
    wl_display *display;
    wl_registry *registry;
    wl_compositor *compositor;
    wl_surface *window;
    xdg_wm_base *wm_base;
    zxdg_decoration_manager_v1 *xdg_decoration_mgr;
    zxdg_toplevel_decoration_v1 *toplevel_decoration;
    xdg_surface *window_surface;
    bool xdg_surface_has_been_configured;
    xdg_toplevel *window_toplevel;
    wl_seat *seat;
    wl_pointer *pointer;
    wl_keyboard *keyboard;
#elif defined(VK_USE_PLATFORM_DIRECTFB_EXT)
    IDirectFB *dfb;
    IDirectFBSurface *window;
    IDirectFBEventBuffer *event_buffer;
#endif

    void resize_callback();
    void draw_callback(uint32_t cur_frame, TimestampRecord *p_timestamp_record);

    void set_draw_callback(ISurfaceRenderContext *context) {
        p_draw_render_context = context;
    }
    void set_resize_callback(ISurfaceRenderContext *context) {
        p_resize_render_context = context;
    }

private:
    ISurfaceRenderContext *p_draw_render_context;
    ISurfaceRenderContext *p_resize_render_context;

    const uint32_t kMaxFrameCount;

    int32_t width;
    int32_t height;

    bool pause;
    bool quit;
    uint32_t curFrame;
};
