/* Modifications Copyright (C) 2022-2023 Advanced Micro Devices, Inc. All rights reserved. */

/*******************************************************************************
 *
 * Copyright (C) 2022 NETINT Technologies
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 ******************************************************************************/

#include <errno.h>
#include <fcntl.h>
#include <linux/limits.h>
#include <cassert>

#include <ni_device_api.h>
#include <ni_util.h>

#include "netint_encoding.h"
#include "timestamp_record.h"

static uint32_t number_of_frames = 0;
static uint32_t number_of_packets = 0;
static uint32_t data_left_size = 0;

static struct timeval start_time;
static struct timeval previous_time;
static struct timeval current_time;

static constexpr int is_nv12 = 0;

/*!****************************************************************************
 *  \brief  Recycle hw frames back to Quadra
 *
 *  \param [in] p2p_frame - array of p2p hw frames to recycle
 *
 *  \return  Returns the number of hw frames that have been recycled
 *******************************************************************************/
static int recycle_frames(ni_frame_t p2p_frame[]) {
    int i;
    int cnt = 0;
    ni_retcode_t rc;

    for (i = 0; i < NetintEncoding::kPoolSize; i++) {
        rc = ni_hwframe_p2p_buffer_recycle(&p2p_frame[i]);

        if (rc != NI_RETCODE_SUCCESS) {
            fprintf(stderr, "Recycle failed\n");
        }

        cnt += (rc == NI_RETCODE_SUCCESS) ? 1 : 0;
    }

    return cnt;
}

/*!****************************************************************************
 *  \brief  Prepare frames to simulate P2P transfers
 *
 *  \param [in] p_upl_ctx           pointer to caller allocated uploader
 *                                  session context
 *         [in] input_video_width   video width
 *         [in] input_video_height  video height
 *         [out] p2p_frame          array of hw frames
 *
 *  \return  0  on success
 *          -1  on error
 ******************************************************************************/
static int p2p_prepare_frames(ni_session_context_t *p_upl_ctx,
                              int input_video_width,
                              int input_video_height,
                              ni_frame_t p2p_frame[]) {
    int i;
    int ret = 0;
    ni_frame_t *p_in_frame;

    // Allocate memory for two hardware frames
    for (i = 0; i < NetintEncoding::kPoolSize; i++) {
        p_in_frame = &p2p_frame[i];

        p_in_frame->start_of_stream = 0;
        p_in_frame->end_of_stream = 0;
        p_in_frame->force_key_frame = 0;
        p_in_frame->extra_data_len = 0;

        // Allocate a hardware ni_frame structure for the encoder
        if (ni_frame_buffer_alloc_hwenc(p_in_frame,
                                        input_video_width,
                                        input_video_height,
                                        (int)p_in_frame->extra_data_len) != NI_RETCODE_SUCCESS) {
            fprintf(stderr, "Error: could not allocate hw frame buffer!");
            ret = -1;
            goto fail_out;
        }

        // Acquire a hw frame from the upload session. This obtains a handle
        // to Quadra memory from the previously created frame pool.
        if (ni_device_session_acquire(p_upl_ctx, p_in_frame)) {
            fprintf(stderr, "Error: failed ni_device_session_acquire()\n");
            ret = -1;
            goto fail_out;
        }
    }

    return ret;

fail_out:
    for (i = 0; i < NetintEncoding::kPoolSize; i++) {
        ni_frame_buffer_free(&(p2p_frame[i]));
    }

    return ret;
}

/*!****************************************************************************
 *  \brief  Send the Quadra encoder a hardware frame which triggers
 *          Quadra to encode the frame
 *
 *  \param  [in] p_enc_ctx              pointer to encoder context
 *          [in] p_in_frame             pointer to hw frame
 *          [in] input_exhausted        flag indicating this is the last frame
 *          [in/out] need_to_resend     flag indicating need to re-send
 *
 *  \return  0 on success
 *          -1 on failure
 ******************************************************************************/
int encoder_encode_frame(ni_session_context_t *p_enc_ctx,
                         ni_frame_t *p_in_frame,
                         int input_exhausted,
                         int *need_to_resend,
                         int *enc_eos_sent) {
    static int started = 0;
    int oneSent;
    ni_session_data_io_t in_data;

    ni_log(NI_LOG_DEBUG, "===> encoder_encode_frame <===\n");

    if (*enc_eos_sent == 1) {
        ni_log(NI_LOG_DEBUG,
               "encoder_encode_frame: ALL data (incl. eos) sent "
               "already!\n");
        return 0;
    }

    if (*need_to_resend) {
        goto send_frame;
    }

    p_in_frame->start_of_stream = 0;

    // If this is the first frame, mark the frame as start-of-stream
    if (!started) {
        started = 1;
        p_in_frame->start_of_stream = 1;
        ni_gettimeofday(&start_time, NULL);
    }

    // If this is the last frame, mark the frame as end-of-stream
    p_in_frame->end_of_stream = input_exhausted ? 1 : 0;
    p_in_frame->force_key_frame = 0;

send_frame:

    in_data.data.frame = *p_in_frame;
    oneSent = ni_device_session_write(p_enc_ctx, &in_data, NI_DEVICE_TYPE_ENCODER);

    if (oneSent < 0) {
        fprintf(stderr, "Error: failed ni_device_session_write() for encoder\n");
        *need_to_resend = 1;
        return -1;
    } else if (oneSent == 0 && !p_enc_ctx->ready_to_close) {
        *need_to_resend = 1;
        ni_log(NI_LOG_DEBUG, "NEEDED TO RESEND");
    } else {
        *need_to_resend = 0;

        ni_log(NI_LOG_DEBUG,
               "encoder_encode_frame: total sent data size=%u\n",
               p_in_frame->data_len[3]);
        ni_log(NI_LOG_DEBUG, "encoder_encode_frame: success\n");

        if (p_enc_ctx->ready_to_close) {
            *enc_eos_sent = 1;
        }
    }

    return 0;
}

/*!****************************************************************************
 *  \brief  Receive output packet data from the Quadra encoder
 *
 *  \param  [in] p_enc_ctx              pointer to encoder session context
 *          [in] p_out_data             pointer to output data session
 *          [in] p_file                 pointer to file to write the packet
 *          [out] total_bytes_received  running counter of bytes read
 *          [in] print_time             1 = print the time
 *
 *  \return 0 - success got packet
 *          1 - received eos
 *          2 - got nothing, need retry
 *         -1 - failure
 ******************************************************************************/
static int encoder_receive_data(ni_session_context_t *p_enc_ctx,
                                ni_session_data_io_t *p_out_data,
                                FILE *p_file,
                                unsigned long long *total_bytes_received,
                                int print_time) {
    int packet_size = NI_MAX_TX_SZ;
    int rc = 0;
    int end_flag = 0;
    int rx_size = 0;
    int meta_size = p_enc_ctx->meta_size;
    ni_packet_t *p_out_pkt = &(p_out_data->data.packet);
    static int received_stream_header = 0;

    ni_log(NI_LOG_DEBUG, "===> encoder_receive_data <===\n");

    if (NI_INVALID_SESSION_ID == p_enc_ctx->session_id ||
        NI_INVALID_DEVICE_HANDLE == p_enc_ctx->blk_io_handle) {
        ni_log(NI_LOG_DEBUG, "encode session not opened yet, return\n");
        return 0;
    }

    if (p_file == NULL) {
        ni_log(NI_LOG_ERROR, "Bad file pointer, return\n");
        return -1;
    }

    rc = ni_packet_buffer_alloc(p_out_pkt, packet_size);
    if (rc != NI_RETCODE_SUCCESS) {
        fprintf(stderr, "Error: malloc packet failed, ret = %d!\n", rc);
        return -1;
    }

    /*
     * The first data read from the encoder session context
     * is a stream header read.
     */
    if (!received_stream_header) {
        /* Read the encoded stream header */
        rc = ni_encoder_session_read_stream_header(p_enc_ctx, p_out_data);

        if (rc > 0) {
            /* Write out the stream header */
            if (fwrite((uint8_t *)p_out_pkt->p_data + meta_size,
                       p_out_pkt->data_len - meta_size,
                       1,
                       p_file) != 1) {
                fprintf(stderr,
                        "Error: writing data %u bytes error!\n",
                        p_out_pkt->data_len - meta_size);
                fprintf(stderr, "Error: ferror rc = %d\n", ferror(p_file));
            }

            *total_bytes_received += (rx_size - meta_size);
            number_of_packets++;
            received_stream_header = 1;
        } else if (rc != 0) {
            fprintf(stderr, "Error: reading header %d\n", rc);
            return -1;
        }

        if (print_time) {
            int timeDiff = (int)(current_time.tv_sec - start_time.tv_sec);
            if (timeDiff == 0) {
                timeDiff = 1;
            }
            printf("[R] Got:%d   Packets= %u fps=%u  Total bytes %llu\n",
                   rx_size,
                   number_of_packets,
                   number_of_packets / timeDiff,
                   *total_bytes_received);
        }

        /* This shouldn't happen */
        if (p_out_pkt->end_of_stream) {
            return 1;
        } else if (rc == 0) {
            return 2;
        }
    }

receive_data:
    rc = ni_device_session_read(p_enc_ctx, p_out_data, NI_DEVICE_TYPE_ENCODER);

    end_flag = p_out_pkt->end_of_stream;
    rx_size = rc;

    ni_log(NI_LOG_DEBUG, "encoder_receive_data: received data size=%d\n", rx_size);

    if (rx_size > meta_size) {
        if (fwrite((uint8_t *)p_out_pkt->p_data + meta_size,
                   p_out_pkt->data_len - meta_size,
                   1,
                   p_file) != 1) {
            fprintf(
                stderr, "Error: writing data %u bytes error!\n", p_out_pkt->data_len - meta_size);
            fprintf(stderr, "Error: ferror rc = %d\n", ferror(p_file));
        }

        *total_bytes_received += rx_size - meta_size;
        number_of_packets++;

        ni_log(NI_LOG_DEBUG, "Got:   Packets= %u\n", number_of_packets);
    } else if (rx_size != 0) {
        fprintf(stderr, "Error: received %d bytes, <= metadata size %d!\n", rx_size, meta_size);
        return -1;
    } else if (!end_flag && ((ni_xcoder_params_t *)(p_enc_ctx->p_session_config))->low_delay_mode) {
        ni_log(NI_LOG_DEBUG, "low delay mode and NO pkt, keep reading...\n");
        goto receive_data;
    }

    if (print_time) {
        int timeDiff = (int)(current_time.tv_sec - start_time.tv_sec);
        if (timeDiff == 0) {
            timeDiff = 1;
        }
        printf("[R] Got:%d   Packets= %u fps=%u  Total bytes %llu\n",
               rx_size,
               number_of_packets,
               number_of_packets / timeDiff,
               *total_bytes_received);
    }

    if (end_flag) {
        printf("Encoder Receiving done\n");
        return 1;
    } else if (0 == rx_size) {
        return 2;
    }

    ni_log(NI_LOG_DEBUG, "encoder_receive_data: success\n");

    return 0;
}

/*!****************************************************************************
 *  \brief  Open an encoder session to Quadra
 *
 *  \param  [out] p_enc_ctx         pointer to an encoder session context
 *          [in]  dst_codec_format  AVC or HEVC
 *          [in]  iXcoderGUID       id to identify the Quadra device
 *          [in]  p_enc_params      sets the encoder parameters
 *          [in]  width             width of frames to encode
 *          [in]  height            height of frames to encode
 *
 *  \return 0 if successful, < 0 otherwise
 ******************************************************************************/
static int encoder_open_session(ni_session_context_t *p_enc_ctx,
                                int dst_codec_format,
                                int iXcoderGUID,
                                ni_xcoder_params_t *p_enc_params,
                                int width,
                                int height,
                                ni_frame_t *p_frame,
                                const Option &option) {
    int ret = 0;

    // Enable hardware frame encoding
    p_enc_ctx->hw_action = NI_CODEC_HW_ENABLE;
    p_enc_params->hwframes = 1;

    // Provide the first frame to the Quadra encoder
    p_enc_params->p_first_frame = p_frame;

    // Specify codec, AVC vs HEVC
    p_enc_ctx->codec_format = dst_codec_format;

    p_enc_ctx->p_session_config = p_enc_params;
    p_enc_ctx->session_id = NI_INVALID_SESSION_ID;

    // Assign the card GUID in the encoder context to open a session
    // to that specific Quadra device
    p_enc_ctx->device_handle = NI_INVALID_DEVICE_HANDLE;
    p_enc_ctx->blk_io_handle = NI_INVALID_DEVICE_HANDLE;
    p_enc_ctx->hw_id = iXcoderGUID;

    if (option.encodingMode == EncodingMode::RGBA) {
        p_enc_ctx->pixel_format = NI_PIX_FMT_ABGR;
    }

    ni_encoder_set_input_frame_format(
        p_enc_ctx, p_enc_params, width, height, 8, NI_FRAME_LITTLE_ENDIAN, 1);

    // Encoder will operate in P2P mode
    ret = ni_device_session_open(p_enc_ctx, NI_DEVICE_TYPE_ENCODER);
    if (ret != NI_RETCODE_SUCCESS) {
        fprintf(stderr, "Error: encoder open session failure\n");
        ni_device_session_close(p_enc_ctx, 1, NI_DEVICE_TYPE_ENCODER);
        ni_device_session_context_clear(p_enc_ctx);
    } else {
        printf("Encoder device %d session open successful\n", iXcoderGUID);
    }

    return ret;
}

/*!****************************************************************************
 *  \brief  Open an upload session to Quadra
 *
 *  \param  [out] p_upl_ctx   pointer to an upload context of the open session
 *          [in]  iXcoderGUID pointer to  Quadra card hw id
 *          [in]  width       width of the frames
 *          [in]  height      height of the frames
 *          [in]  format      pixel format of the frames (RGBA or YUV420p)
 *
 *  \return 0 if successful, < 0 otherwise
 ******************************************************************************/
static int uploader_open_session(
    ni_session_context_t *p_upl_ctx, int *iXcoderGUID, int width, int height, ni_pix_fmt_t format) {
    int ret = 0;

    p_upl_ctx->session_id = NI_INVALID_SESSION_ID;

    // Assign the card GUID in the encoder context
    p_upl_ctx->device_handle = NI_INVALID_DEVICE_HANDLE;
    p_upl_ctx->blk_io_handle = NI_INVALID_DEVICE_HANDLE;

    // Assign the card id to specify the specific Quadra device
    p_upl_ctx->hw_id = *iXcoderGUID;

    // Set the input frame format of the upload session
    ni_uploader_set_frame_format(p_upl_ctx, width, height, format, 1);

    ret = ni_device_session_open(p_upl_ctx, NI_DEVICE_TYPE_UPLOAD);
    if (ret != NI_RETCODE_SUCCESS) {
        fprintf(stderr, "Error: uploader_open_session failure!\n");
        ni_device_session_close(p_upl_ctx, 1, NI_DEVICE_TYPE_UPLOAD);
        ni_device_session_context_clear(p_upl_ctx);
        return ret;
    } else {
        printf("Uploader device %d session opened successfully\n", *iXcoderGUID);
        *iXcoderGUID = p_upl_ctx->hw_id;
    }

    // Create a P2P frame pool for the uploader sesson of pool size 2
    ret = ni_device_session_init_framepool(p_upl_ctx, NetintEncoding::kPoolSize, 1);
    if (ret < 0) {
        fprintf(stderr, "Error: Can't create frame pool\n");
        ni_device_session_close(p_upl_ctx, 1, NI_DEVICE_TYPE_UPLOAD);
        ni_device_session_context_clear(p_upl_ctx);
    } else {
        printf("Uploader device %d configured successfully\n", *iXcoderGUID);
    }

    return ret;
}

void NetintEncoding::device_init(const Option &option) {
    int ret;
    this->option = option;

    printf("Ver:  %s\n"
           "Date: %s\n"
           "ID:   %s\n",
           NI_XCODER_REVISION,
           NI_SW_RELEASE_TIME,
           NI_SW_RELEASE_ID);

    //    ni_log_level_t log_level = arg_to_ni_log_level("debug");
    ni_log_level_t log_level = arg_to_ni_log_level("none");
    if (log_level != NI_LOG_INVALID) {
        ni_log_set_level(log_level);
    } else {
        fprintf(stderr, "wrong log level");
        exit(EXIT_FAILURE);
    }

    input_video_width = option.width;
    input_video_height = option.height;

    if (ni_device_session_context_init(&enc_ctx) < 0) {
        fprintf(stderr, "Error: init encoder context error\n");
        goto fail_cleanup_output_file;
    }

    if (ni_device_session_context_init(&upl_ctx) < 0) {
        fprintf(stderr, "Error: init uploader context error\n");
        goto fail_cleanup_output_file;
    }

    do {
        ni_gettimeofday(&start_time, NULL);
        ni_gettimeofday(&previous_time, NULL);
        ni_gettimeofday(&current_time, NULL);

        char output_filepath[PATH_MAX] = {};
        if (option.output_dir_path) {
            snprintf(output_filepath,
                     PATH_MAX,
                     "%scube_%lds%ldus.265",
                     option.output_dir_path,
                     start_time.tv_sec,
                     start_time.tv_usec);
        } else {
            snprintf(output_filepath,
                     PATH_MAX,
                     "cube_%lds%ldus.265",
                     start_time.tv_sec,
                     start_time.tv_usec);
        }

        // atomically check file existance and create file
        int fd = open(output_filepath, O_CREAT | O_WRONLY | O_EXCL, S_IRUSR | S_IWUSR);
        if (fd >= 0) {
            p_output_file = fdopen(fd, "wb");
            printf("SUCCESS: Opened output file: %s\n", output_filepath);
            break;
        }

        if (errno == EEXIST) {
            printf("file:'%s' exists, retry..\n", output_filepath);
        } else {
            fprintf(stderr, "Error: cannot open %s\n", output_filepath);
            exit(EXIT_FAILURE);
        }
    } while (1);

    printf("P2P Encoding resolution: %dx%d\n", input_video_width, input_video_height);

    // Open an uploader session to Quadra
    if (uploader_open_session(&upl_ctx,
                              &iXcoderGUID,
                              input_video_width,
                              input_video_height,
                              (option.encodingMode == EncodingMode::RGBA) ? NI_PIX_FMT_RGBA
                                                                          : NI_PIX_FMT_YUV420P)) {
        fprintf(stderr, "Error: uploader open session error\n");
        goto fail_cleanup_output_file;
    }

    // Configure the encoder parameter structure. We'll use some basic
    // defaults: 30 fps, 200000 bps CBR encoding, AVC or HEVC encoding
    if (ni_encoder_init_default_params(&api_param,
                                       30,
                                       1,
                                       200000,
                                       input_video_width,
                                       input_video_height,
                                       (ni_codec_format_t)enc_ctx.codec_format) < 0) {
        fprintf(stderr, "Error: encoder init default set up error\n");
        goto fail_close_upload_session;
    }

    // For P2P demo, change some of the encoding parameters from
    // the default. Enable low delay encoding.
    if ((ret = ni_encoder_params_set_value(&api_param, "lowDelay", "1")) != NI_RETCODE_SUCCESS) {
        fprintf(stderr, "Error: can't set low delay mode %d\n", ret);
        goto fail_close_upload_session;
    }

    // Use a GOP preset of 9 which represents a GOP pattern of
    // IPPPPPPP....This will be low latency.
    if ((ret = ni_encoder_params_set_value(&api_param, "gopPresetIdx", "9")) !=
        NI_RETCODE_SUCCESS) {
        fprintf(stderr, "Error: can't set gop preset %d\n", ret);
        goto fail_close_upload_session;
    }

    // Prepare two frames for double buffering
    ret = p2p_prepare_frames(&upl_ctx, input_video_width, input_video_height, p2p_frame);
    if (ret < 0) {
        fprintf(stderr, "Error: prepare dma-buf frame failed!\n");
        goto fail_close_upload_session;
    }

    // Open the encoder session with given parameters
    if (ret = encoder_open_session(&enc_ctx,
                                   dst_codec_format,
                                   iXcoderGUID,
                                   &api_param,
                                   input_video_width,
                                   input_video_height,
                                   p2p_frame,
                                   option)) {
        goto fail_close_upload_session;
    }

    return;

fail_close_upload_session:
    ni_device_session_close(&upl_ctx, 1, NI_DEVICE_TYPE_UPLOAD);
    ni_device_session_context_clear(&upl_ctx);

fail_cleanup_output_file:
    if (p_output_file) {
        fclose(p_output_file);
    }

    exit(-1);
}

void NetintEncoding::get_shared_image_plane_range_by_index(ResourceIndex index,
                                                           ImagePlaneAspect image_aspect,
                                                           size_t *offset,
                                                           size_t *size) {
    assert(input_video_width == 1280);
    assert(input_video_height == 720);
    assert(!is_nv12);

    if (option.encodingMode == EncodingMode::RGBA) {
        assert(image_aspect == ImagePlaneAspect::Plane0);
        *offset = 0;
        *size = input_video_width * input_video_height * 4;
    } else {
        int dst_stride[NI_MAX_NUM_DATA_POINTERS];
        int dst_height[NI_MAX_NUM_DATA_POINTERS];
        ni_get_hw_yuv420p_dim(input_video_width,
                              input_video_height,
                              upl_ctx.bit_depth_factor,
                              is_nv12,
                              dst_stride,
                              dst_height);

        size_t offsets[4] = {};
        offsets[0] = 0;
        for (uint32_t i = 1; i < 4; i++) {
            offsets[i] =
                offsets[i - 1] + dst_stride[i - 1] * dst_height[i - 1] * upl_ctx.bit_depth_factor;
        }
        // Note, buffer pitch should be 128 columns,
        // buffer height should be multiple of 2
        switch (image_aspect) {
        case ImagePlaneAspect::Plane0:
            *offset = offsets[0];
            *size = offsets[1] - offsets[0];
            break;
        case ImagePlaneAspect::Plane1:
            *offset = offsets[1];
            *size = offsets[2] - offsets[1];
            break;
        case ImagePlaneAspect::Plane2:
            *offset = offsets[2];
            *size = offsets[3] - offsets[2];
            break;
        }
    }
}

DmaBufFd NetintEncoding::allocate_shared_image_by_index(ResourceIndex index,
                                                        int32_t width,
                                                        int32_t height) {
    assert(index < get_max_number_of_shared_images());
    (void)width;
    (void)height;

    return DmaBufFd{dup(((niFrameSurface1_t *)p2p_frame[index].p_data[3])->dma_buf_fd)};
}

size_t NetintEncoding::get_shared_image_size_by_index(ResourceIndex index) {
    assert(index < get_max_number_of_shared_images());

    if (option.encodingMode == EncodingMode::RGBA) {
        return input_video_width * input_video_height * 4;
    } else {
        int dst_stride[NI_MAX_NUM_DATA_POINTERS];
        int dst_height[NI_MAX_NUM_DATA_POINTERS];
        ni_get_hw_yuv420p_dim(input_video_width,
                              input_video_height,
                              upl_ctx.bit_depth_factor,
                              is_nv12,
                              dst_stride,
                              dst_height);
        size_t byte_count = 0;
        for (uint32_t i = 0; i < 3; i++) {
            byte_count += dst_stride[i] * dst_height[i] * upl_ctx.bit_depth_factor;
        }
        return byte_count;
    }
}

ResourceIndex NetintEncoding::acquire_shared_image(FrameIndex frame_index) {
    assert(frame_index < get_max_number_of_inflight_commands() &&
           "Netint encoding, too many frames opened!");

    return ResourceIndex{frame_index};
}

void NetintEncoding::encoding_on_memory(uint32_t cur_frame,
                                        TimestampRecord *p_timestamp_record,
                                        FrameIndex frame_index,
                                        ResourceIndex rsc_index) {
    assert(frame_index == rsc_index && "Netint encoding, frame and resource index not much!");

    // Encode the frame
    int send_fin_flag = 0;
    int need_to_resend = 0;
    const int input_exhausted = 0; // input is a realtime rendering, never

    do {
        ni_gettimeofday(&current_time, NULL);

        // Print the time if >= 1 second has passed
        print_time = ((current_time.tv_sec - previous_time.tv_sec) > 1);

        send_fin_flag =
            encoder_encode_frame(&enc_ctx, &p2p_frame[rsc_index], input_exhausted, &need_to_resend, &enc_eos_sent);

        // Error, exit
        if (send_fin_flag == 2) {
            fprintf(stderr, "Failed on encoding buffer!");
        }

        /* TODO: Move receive_data into a separate thread to not blocking rendering loop */
        // Receive encoded packet data from the encoder
        int receive_fin_flag = encoder_receive_data(
            &enc_ctx, &out_packet, p_output_file, &total_bytes_received, print_time);

        p_timestamp_record->mark_end_encoding(cur_frame);

        if (print_time) {
            previous_time = current_time;
        }

        // Error or eos
        if (receive_fin_flag < 0 || out_packet.data.packet.end_of_stream) {
            fprintf(stderr, "Failed on receiving buffer\n");
        }

    } while (need_to_resend);
}

void NetintEncoding::device_cleanup() {
    timeDiff = (int)(current_time.tv_sec - start_time.tv_sec);
    timeDiff = (timeDiff > 0) ? timeDiff : 1; // avoid division by zero

    printf("[R] Got:  Packets= %u fps=%f  Total bytes %llu\n",
           number_of_packets,
           (double)number_of_packets / timeDiff,
           total_bytes_received);

    // Recycle the hardware frames back to the pool prior
    // to closing the uploader session.
    int num_post_recycled = recycle_frames(p2p_frame);

    ni_log(NI_LOG_DEBUG, "Cleanup recycled %d internal buffers\n", num_post_recycled);

    ni_device_session_close(&enc_ctx, 1, NI_DEVICE_TYPE_ENCODER);
    ni_device_session_close(&upl_ctx, 1, NI_DEVICE_TYPE_UPLOAD);

    ni_device_session_context_clear(&enc_ctx);
    ni_device_session_context_clear(&upl_ctx);

    for (int i = 0; i < NetintEncoding::kPoolSize; i++) {
        ni_frame_buffer_free(&(p2p_frame[i]));
    }

    ni_packet_buffer_free(&(out_packet.data.packet));
}
