// Modifications Copyright (C) 2022-2023 Advanced Micro Devices, Inc. All rights reserved.
// Notified per clause 4(b) of the license

/*
 * Copyright (c) 2015-2019 The Khronos Group Inc.
 * Copyright (c) 2015-2019 Valve Corporation
 * Copyright (c) 2015-2019 LunarG, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Authors: Jeremy Hayes <jeremy@lunarg.com>
 */

#pragma once

#include <stdint.h>

// Allow a maximum of two outstanding presentation operations.

#define VULKAN_HPP_NO_EXCEPTIONS
#define VULKAN_HPP_TYPESAFE_CONVERSION 1
#include <vulkan/vulkan.hpp>

#include "linmath.h"

#include "encoding_device.h"
#include "options.h"
#include "surface_render_context.h"

struct texture_object {
    vk::Sampler sampler;

    vk::Image image;
    vk::Buffer buffer;
    vk::ImageLayout imageLayout{vk::ImageLayout::eUndefined};

    vk::MemoryAllocateInfo mem_alloc;
    vk::DeviceMemory mem;
    vk::ImageView view;

    int32_t tex_width{0};
    int32_t tex_height{0};
};

struct Cube : public ISurfaceRenderContext {
    static constexpr uint32_t FRAME_LAG = 2;

    typedef struct {
        vk::CommandBuffer cmd;
        vk::Buffer yuv_buffer;
        vk::DeviceMemory yuv_buffer_memory;
        vk::BufferView plane0_texel_buffer_view;
        vk::BufferView plane1_texel_buffer_view;
        vk::BufferView plane2_texel_buffer_view;
        vk::DescriptorSet descriptor_set;

        DmaBufFd exportedFd;
    } ColorConversionResource;

    typedef struct {
        vk::CommandBuffer cmd;
        vk::Buffer ni_buffer;
        vk::DeviceMemory ni_buffer_memory;
    } TransferResources;

    typedef struct {
        vk::CommandBuffer cmd;
        vk::Buffer uniform_buffer;
        vk::DeviceMemory uniform_memory;
        void *uniform_memory_ptr;
        vk::Framebuffer framebuffer;
        vk::DescriptorSet descriptor_set;

        // Only used for offscreen rendering
        vk::Image color_attachment_image;
        vk::ImageView color_attachment_image_view;
        vk::DeviceMemory color_attachment_image_memory;

        // used if no YUV420p conv.
        DmaBufFd exportedFd;
    } DrawResources;

    Cube(const Option &);
    ~Cube() override {}

    void draw(uint32_t cur_frame, TimestampRecord *p_timestamp_record) override;
    void cleanup() override;

    void set_encoding_device(IEncodeDevice *encoding_device) {
        assert(prepared == false);
        this->encoding_device = encoding_device;
    }

    vk::Bool32 check_layers(uint32_t, const char *const *, uint32_t, vk::LayerProperties *);
    void destroy_texture(texture_object *);
    void draw_build_cmd(vk::CommandBuffer cmd, ResourceIndex current_buffer);
    void flush_init_cmd();

    void init();
    void init_vk();
    void select_queue_family();
    void create_device();
    void prepare();
    void prepare_cube_data_buffers();
    void prepare_depth();
    void prepare_descriptor_layout();
    void prepare_descriptor_pool();
    void prepare_descriptor_set();
    void prepare_framebuffers();
    vk::ShaderModule prepare_shader_module(const uint32_t *, size_t);
    vk::ShaderModule prepare_vs();
    vk::ShaderModule prepare_fs();
    void prepare_pipeline();
    void prepare_render_pass();
    void prepare_texture_image(const char *,
                               texture_object *,
                               vk::ImageTiling,
                               vk::ImageUsageFlags,
                               vk::MemoryPropertyFlags);
    void prepare_texture_buffer(const char *, texture_object *);
    void prepare_textures();

    void set_image_layout(vk::Image,
                          vk::ImageAspectFlags,
                          vk::ImageLayout,
                          vk::ImageLayout,
                          vk::AccessFlags,
                          vk::PipelineStageFlags,
                          vk::PipelineStageFlags);
    void update_data_buffer(ResourceIndex active_index);
    bool loadTexture(const char *, uint8_t *, vk::SubresourceLayout *, int32_t *, int32_t *);

    void create_offscreen_render_targets();
    void cleanup_offscreen_render_targets();

    void prepare_for_color_conversion_and_transfer();
    void cleanup_for_color_conversion_and_transfer();

    void prepare_for_color_conversion();
    void cleanup_for_color_conversion();

    void create_transfer_external_image();
    void build_transfer_cmds();
    void build_transfer_syncs();
    void build_convert_syncs();
    void transfer_framebuffer_to_encoding(uint32_t cur_frame,
                                          TimestampRecord *p_timestamp_record,
                                          ResourceIndex transfer_rsc_index);
    void dma_framebuffer_to_encoding(const uint32_t cur_frame,
                                            TimestampRecord *p_timestamp_record,
                                            ResourceIndex transfer_rsc_index);
    void create_rgb2yuv_convert_images();
    void create_rgb2yuv_descriptor_layout();
    void create_rgb2yuv_descriptors();
    void create_rgb2yuv_pipeline();
    vk::ShaderModule create_rgb2yuv_cs();
    void update_rgb2yuv_descriptors();
    void build_rgb2yuv_convert_cmds();
    void convert_rgb2yuv(ResourceIndex rsc_index);
    void color_conversion_and_transfer_to_encoding(uint32_t cur_frame,
                                                   TimestampRecord *p_timestamp_record,
                                                   ResourceIndex transfer_rsc_index);
    void color_conversion(uint32_t cur_frame,
                                                   TimestampRecord *p_timestamp_record,
                                                   ResourceIndex transfer_rsc_index);

    int32_t width() {
        return option.width;
    }
    int32_t height() {
        return option.height;
    }

    FrameIndex get_max_frame_number() {
        return encoding_device->get_max_number_of_inflight_commands();
    }

    ResourceIndex get_image_count() {
        return encoding_device->get_max_number_of_shared_images();
    }

    vk::ImageView get_render_target_image_view(uint32_t idx) {
        return draw_resources[idx].color_attachment_image_view;
    }

    vk::Format get_render_target_format() {
        // return vk::Format::eR8G8B8A8Unorm;
        return vk::Format::eR8G8B8A8Srgb;
    }

    void prepare_render_target_resources() {
        draw_resources.reset(new DrawResources[get_image_count()]);
        create_offscreen_render_targets();
    }

    void cleanup_render_target_resources() {
        cleanup_offscreen_render_targets();
    }

    void prepare_dma();
    void clean_dma();

    Option option;

    IEncodeDevice *encoding_device;

    vk::Queue compute_queue;
    uint32_t compute_queue_family_index;
    vk::Pipeline color_conversion_pipeline;
    vk::PipelineLayout color_conversion_pipeline_layout;
    vk::DescriptorSetLayout color_conversion_desc_layout;
    vk::DescriptorPool color_conversion_desc_pool;
    std::unique_ptr<ColorConversionResource[]> color_conversion_resources;
    std::unique_ptr<vk::Semaphore[]> color_conversion_acquire_semaphores;
    std::unique_ptr<vk::Semaphore[]> color_conversion_complete_semaphores;
    std::unique_ptr<vk::Fence[]> color_conversion_cmd_fences;
    vk::CommandPool compute_cmd_pool;

    vk::Queue transfer_queue;
    uint32_t transfer_queue_family_index;
    std::unique_ptr<TransferResources[]> transfer_resources;
    std::unique_ptr<vk::Semaphore[]> transfer_acquire_semaphores;
    std::unique_ptr<vk::Semaphore[]> transfer_complete_semaphores;
    std::unique_ptr<vk::Fence[]> transfer_cmd_fences;
    vk::CommandPool transfer_cmd_pool;

    bool prepared;
    int32_t selected_gpu_number;

    vk::Instance inst;
    vk::PhysicalDevice gpu;
    vk::Device device;
    vk::Queue graphics_queue;
    uint32_t graphics_queue_family_index;
    vk::PhysicalDeviceProperties gpu_props;
    std::unique_ptr<vk::QueueFamilyProperties[]> queue_props;
    uint32_t queue_family_count;
    vk::PhysicalDeviceMemoryProperties memory_properties;

    PFN_vkGetSemaphoreFdKHR pfnGetSemaphoreFdKHR;
    PFN_vkImportSemaphoreFdKHR pfnImportSemaphoreFdKHR;
    PFN_vkGetMemoryFdPropertiesKHR pfnGetMemoryFdPropertiesKHR;
    PFN_vkGetMemoryFdKHR pfnGetMemoryFdKHR;

    uint32_t enabled_extension_count;
    uint32_t enabled_layer_count;
    char const *extension_names[64];
    char const *layer_names[64];

    std::unique_ptr<DrawResources[]> draw_resources;
    std::unique_ptr<vk::Semaphore[]> image_acquire_semaphores;
    std::unique_ptr<vk::Semaphore[]> draw_complete_semaphores;
    std::unique_ptr<vk::Fence[]> fences;
    FrameIndex frame_index;

    vk::CommandPool cmd_pool;

    struct {
        vk::Format format;
        vk::Image image;
        vk::MemoryAllocateInfo mem_alloc;
        vk::DeviceMemory mem;
        vk::ImageView view;
    } depth;

    static int32_t const texture_count = 1;
    texture_object textures[texture_count];
    texture_object staging_texture;

    struct {
        vk::Buffer buf;
        vk::MemoryAllocateInfo mem_alloc;
        vk::DeviceMemory mem;
        vk::DescriptorBufferInfo buffer_info;
    } uniform_data;

    vk::CommandBuffer cmd; // Buffer for initialization commands
    vk::PipelineLayout pipeline_layout;
    vk::DescriptorSetLayout desc_layout;
    vk::PipelineCache pipelineCache;
    vk::RenderPass render_pass;
    vk::Pipeline pipeline;

    mat4x4 projection_matrix;
    mat4x4 view_matrix;
    mat4x4 model_matrix;

    float spin_angle;
    float spin_increment;

    vk::ShaderModule vert_shader_module;
    vk::ShaderModule frag_shader_module;

    vk::DescriptorPool desc_pool;
    vk::DescriptorSet desc_set;
};
