// Modifications Copyright (C) 2022-2023 Advanced Micro Devices, Inc. All rights reserved.
// Notified per clause 4(b) of the license

/*
 * Copyright (c) 2015-2019 The Khronos Group Inc.
 * Copyright (c) 2015-2019 Valve Corporation
 * Copyright (c) 2015-2019 LunarG, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Authors: Jeremy Hayes <jeremy@lunarg.com>
 */

#include <sys/stat.h>
#include <sstream>

#include "app.h"
#include "options.h"

Option parse_arguments_to_options(int argc, char **argv) {
    Option option = {};
    char *path_buffer = nullptr;

    for (int i = 1; i < argc; i++) {
        if (strcmp(argv[i], "--use_staging") == 0) {
            option.use_staging_buffer = true;
            continue;
        }
        if ((strcmp(argv[i], "--present_mode") == 0) && (i < argc - 1)) {
            option.presentMode = (vk::PresentModeKHR)atoi(argv[i + 1]);
            i++;
            continue;
        }
        if ((strcmp(argv[i], "--encoding_mode") == 0) && (i < argc - 1)) {
            option.encodingMode = (EncodingMode)atoi(argv[i + 1]);
            i++;
            continue;
        }
        if (strcmp(argv[i], "--validate") == 0) {
            option.validate = true;
            continue;
        }
        if (strcmp(argv[i], "--c") == 0 && option.frameCount == UINT32_MAX && i < argc - 1 &&
            sscanf(argv[i + 1], "%" SCNu32, &option.frameCount) == 1) {
            i++;
            continue;
        }
        if (strcmp(argv[i], "--width") == 0 && i < argc - 1 &&
            sscanf(argv[i + 1], "%" SCNi32, &option.width) == 1 && option.width > 0) {
            i++;
            continue;
        }
        if (strcmp(argv[i], "--height") == 0 && i < argc - 1 &&
            sscanf(argv[i + 1], "%" SCNi32, &option.height) == 1 && option.height > 0) {
            i++;
            continue;
        }
        if ((strcmp(argv[i], "--gpu_number") == 0) && (i < argc - 1)) {
            option.gpu_number = atoi(argv[i + 1]);
            assert(option.gpu_number >= 0);
            i++;
            continue;
        }
        if ((strcmp(argv[i], "--enc_number") == 0) && (i < argc - 1)) {
            option.enc_number = atoi(argv[i + 1]);
            assert(option.enc_number >= 0);
            i++;
            continue;
        }
        if (strcmp(argv[i], "--output_directory") == 0 && i < argc - 1 &&
            sscanf(argv[i + 1], "%ms", &path_buffer) == 1) {
            const size_t path_len = strlen(path_buffer);
            int ret = mkdir(path_buffer, 0755);
            if (ret != 0 && errno != EEXIST) {
                option.output_dir_path = nullptr;
            } else if (path_len > 0 && path_buffer[path_len - 1] != '/') {
                // We don't bother to free it, let OS do the cleanup
                option.output_dir_path = (char *)malloc(path_len + 2);
                option.output_dir_path[0] = '\0';
                strncat(option.output_dir_path, path_buffer, path_len);
                option.output_dir_path[path_len] = '/';
                option.output_dir_path[path_len + 1] = '\0';
            } else {
                option.output_dir_path = path_buffer;
            }
            i++;
            continue;
        }
        if (strcmp(argv[i], "--debug_always_cpu_wait") == 0) {
            option.debug_always_cpu_wait = true;
            continue;
        }
        std::stringstream usage;
        usage << "Usage:\n  " << APP_SHORT_NAME << "\t[--use_staging] [--validate]\n"
              << "\t[--c <framecount>]\n"
              << "\t[--gpu_number <index of physical device>]\n"
              << "\t[--present_mode <present mode enum>]\n"
              << "\t[--encoding_mode <encoding mode enum>]\n"
              << "\t[--width <width>] [--height <height>]\n"
              << "\t[--debug_always_cpu_wait]\n"
              << "\t<present_mode_enum>\n"
              << "\t\tVK_PRESENT_MODE_IMMEDIATE_KHR = " << VK_PRESENT_MODE_IMMEDIATE_KHR << "\n"
              << "\t\tVK_PRESENT_MODE_MAILBOX_KHR = " << VK_PRESENT_MODE_MAILBOX_KHR << "\n"
              << "\t\tVK_PRESENT_MODE_FIFO_KHR = " << VK_PRESENT_MODE_FIFO_KHR << "\n"
              << "\t\tVK_PRESENT_MODE_FIFO_RELAXED_KHR = " << VK_PRESENT_MODE_FIFO_RELAXED_KHR
              << "\n"
              << "\t<encoding_mode_enum>\n"
              << "\t\tNetint Encoding RGBA = " << (int)EncodingMode::RGBA
              << "\t\tNetint Encoding YUV420p = " << (int)EncodingMode::YUV420p << "\n";

        std::cerr << usage.str();
        std::cerr.flush();
        exit(1);
    }

    return option;
}
