#include <libdrm/drm_fourcc.h>

#include <unistd.h>

#include <iostream>

#include "p2p_gl_egl.h"

static const char *eglErrorString(EGLint error)
{
    switch (error)
    {
    case EGL_SUCCESS:
        return "No error";
    case EGL_NOT_INITIALIZED:
        return "EGL not initialized or failed to initialize";
    case EGL_BAD_ACCESS:
        return "Resource inaccessible";
    case EGL_BAD_ALLOC:
        return "Cannot allocate resources";
    case EGL_BAD_ATTRIBUTE:
        return "Unrecognized attribute or attribute value";
    case EGL_BAD_CONTEXT:
        return "Invalid EGL context";
    case EGL_BAD_CONFIG:
        return "Invalid EGL frame buffer configuration";
    case EGL_BAD_CURRENT_SURFACE:
        return "Current surface is no longer valid";
    case EGL_BAD_DISPLAY:
        return "Invalid EGL display";
    case EGL_BAD_SURFACE:
        return "Invalid surface";
    case EGL_BAD_MATCH:
        return "Inconsistent arguments";
    case EGL_BAD_PARAMETER:
        return "Invalid argument";
    case EGL_BAD_NATIVE_PIXMAP:
        return "Native pixmap handle is no longer valid";
    case EGL_BAD_NATIVE_WINDOW:
        return "Native window handle is no longer valid";
    case EGL_CONTEXT_LOST:
        return "Context lost due to a power management event";
    default:
        return "Unknown EGL error";
    }
}

GL_egl_display::GL_egl_display() : init_(false)
{
    EGLBoolean init;
    eglDisplay_ = eglGetDisplay(EGL_DEFAULT_DISPLAY);
    if(eglDisplay_ == EGL_NO_DISPLAY)
    {
        std::cerr << "Error: Failed to get EGL display\n";
        return;
    }
    // EGLint major, minor;
    init = eglInitialize(eglDisplay_, nullptr, nullptr);
    if (init == EGL_FALSE)
    {
        EGLint error = eglGetError();
        std::cerr << "Unable to initialize EGL. ERROR:" << eglErrorString(error) << std::endl;
        return;
    }

    const EGLint configAttributes[] = {
        EGL_SURFACE_TYPE, EGL_PBUFFER_BIT,
        EGL_RENDERABLE_TYPE, EGL_OPENGL_ES2_BIT,
        EGL_RED_SIZE, 8,
        EGL_GREEN_SIZE, 8,
        EGL_BLUE_SIZE, 8,
        EGL_ALPHA_SIZE, 8,
        EGL_DEPTH_SIZE, 16,
        EGL_STENCIL_SIZE, 8,
        EGL_NONE};

    EGLConfig eglConfig;
    EGLint numConfigs;
    eglChooseConfig(eglDisplay_, configAttributes, &eglConfig, 1, &numConfigs);
    if (numConfigs <= 0)
    {
        std::cerr << "No suitable EGLConfig found.\n";
        eglTerminate(eglDisplay_);
        return;
    }

    const EGLint contextAttributes[] = {
        EGL_CONTEXT_CLIENT_VERSION, 2,
        EGL_NONE};

    eglContext_ = eglCreateContext(eglDisplay_, eglConfig, EGL_NO_CONTEXT, contextAttributes);
    if (eglContext_ == EGL_NO_CONTEXT)
    {
        std::cerr << "Failed to create EGL context.\n";
        eglTerminate(eglDisplay_);
        return;
    }

    if (!make_current_display())
    {
        std::cerr << "Failed to make the EGLContext current.\n";
        eglTerminate(eglDisplay_);
        return;
    }

    init_ = true;
}

PFNEGLCREATEIMAGEKHRPROC GL_rgba_egl::get_eglCreateImageKHR()
{
    static PFNEGLCREATEIMAGEKHRPROC p_func = (PFNEGLCREATEIMAGEKHRPROC)eglGetProcAddress("eglCreateImageKHR");
    if (!p_func)
    {
        std::cerr << "Failed to load eglCreateImageKHR\n";
    }
    return p_func;
}

PFNEGLDESTROYIMAGEKHRPROC GL_rgba_egl::get_eglDestroyImageKHR()
{
    static PFNEGLDESTROYIMAGEKHRPROC p_func = (PFNEGLDESTROYIMAGEKHRPROC)eglGetProcAddress("eglDestroyImageKHR");
    if (!p_func)
    {
        std::cerr << "Failed to load eglDestroyImageKHR\n";
    }
    return p_func;
}

PFNGLEGLIMAGETARGETTEXTURE2DOESPROC GL_rgba_egl::get_glEGLImageTargetTexture2DOES()
{
    static PFNGLEGLIMAGETARGETTEXTURE2DOESPROC p_func = (PFNGLEGLIMAGETARGETTEXTURE2DOESPROC)eglGetProcAddress("glEGLImageTargetTexture2DOES");
    if (!p_func)
    {
        std::cerr << "Failed to load glEGLImageTargetTexture2DOES\n";
    }
    return p_func;
}

GL_rgba_egl::GL_rgba_egl(int fd, int w, int h, std::shared_ptr<GL_egl_display> display) : GL_rgba_base(),
                                                                                          fd_(fd),
                                                                                          w_(w), h_(h), display_(display),
                                                                                          texture_(0), eglImage_(EGL_NO_IMAGE_KHR),
                                                                                          fbo_(0), initialized_(false)

{
    if (!display_->make_current_display())
    {
        std::cerr << "Invalid dispaly\n";
        return;
    }

    PFNEGLCREATEIMAGEKHRPROC eglCreateImageKHR = get_eglCreateImageKHR();
    PFNGLEGLIMAGETARGETTEXTURE2DOESPROC glEGLImageTargetTexture2DOES = get_glEGLImageTargetTexture2DOES();

    if (!eglCreateImageKHR || !glEGLImageTargetTexture2DOES)
    {
        std::cerr << "Failed to load external functions\n";
        return;
    }

    EGLint img_attrs[] = {
        EGL_WIDTH, w /*width of dma buffer*/,
        EGL_HEIGHT, h /*height of dma buffer */,
        EGL_LINUX_DRM_FOURCC_EXT, DRM_FORMAT_ARGB8888,// set as actually format
        EGL_DMA_BUF_PLANE0_FD_EXT, fd,
        EGL_DMA_BUF_PLANE0_OFFSET_EXT, 0,
        EGL_DMA_BUF_PLANE0_PITCH_EXT, w * (4) /*stride of dma buf*/,
        EGL_NONE};

    eglImage_ = eglCreateImageKHR(display_->get_display(), EGL_NO_CONTEXT, EGL_LINUX_DMA_BUF_EXT, nullptr, img_attrs);

    if (eglImage_ == EGL_NO_IMAGE_KHR)
    {
        EGLint error = eglGetError();
        std::cerr << "Failed to create EGLImageKHR. ERROR:" << eglErrorString(error) << std::endl;
        return;
    }

    glGenTextures(1, &texture_);
    glBindTexture(GL_TEXTURE_2D, texture_);

    glEGLImageTargetTexture2DOES(GL_TEXTURE_2D, (GLeglImageOES)eglImage_);

    glGenFramebuffers(1, &fbo_);

    glBindFramebuffer(GL_FRAMEBUFFER, fbo_);

    glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, texture_, 0);

    GLenum check_value = glCheckFramebufferStatus(GL_FRAMEBUFFER);

    if (check_value != GL_FRAMEBUFFER_COMPLETE)
    {
        std::cerr << "Framebuffer is not complete" << std::endl;

        return;
    }

    initialized_ = true;
}

GL_rgba_egl::~GL_rgba_egl()
{
    if (!display_->make_current_display())//remove this if no dispaly and context changing
    {
        // std::cerr << "Invaild Display\n";
        return;
    }

    if (eglImage_ != EGL_NO_IMAGE_KHR)
    {
        PFNEGLDESTROYIMAGEKHRPROC eglDestroyImageKHR = get_eglDestroyImageKHR();
        if(eglDestroyImageKHR)
        {
            eglDestroyImageKHR(display_->get_display(), eglImage_);
        }
    }

    if (texture_ != 0)
    {
        glDeleteTextures(1, &texture_);
    }

    if (fbo_)
    {
        glDeleteFramebuffers(1, &fbo_);
    }
}

GL_rgba_egl::GL_rgba_egl(GL_rgba_egl &&other) : GL_rgba_base(std::move(other)), display_(std::move(other.display_))
{
    fd_ = other.fd_;
    w_ = other.w_;
    h_ = other.h_;

    texture_ = other.texture_;
    eglImage_ = other.eglImage_;
    fbo_ = other.fbo_;
    initialized_ = other.initialized_;

    other.fd_ = -1;
    other.w_ = 0;
    other.h_ = 0;
    
    other.texture_ = 0;
    other.eglImage_ = EGL_NO_IMAGE_KHR;
    other.fbo_ = 0;
    other.initialized_ = false;
}

GL_rgba_egl &GL_rgba_egl::operator=(GL_rgba_egl &&other)
{
    if (this == &other)
    {
        return *this;
    }

    GL_rgba_base::operator=(std::move(other));

    // check if need to close fd

    if (display_->make_current_display())//remove this if no dispaly and context changing
    {  
        if (eglImage_ != EGL_NO_IMAGE_KHR)
        {
            PFNEGLDESTROYIMAGEKHRPROC eglDestroyImageKHR = get_eglDestroyImageKHR();
            if(eglDestroyImageKHR)
            {
                eglDestroyImageKHR(display_->get_display(), eglImage_);
            }
        }

        if (texture_ != 0)
        {
            glDeleteTextures(1, &texture_);
        }

        if (fbo_ != 0)
        {
            glDeleteFramebuffers(1, &fbo_);
        }
    }

    fd_ = other.fd_;
    w_ = other.w_;
    h_ = other.h_;
    display_ = std::move(other.display_);
    texture_ = other.texture_;
    eglImage_ = other.eglImage_;
    fbo_ = other.fbo_;
    initialized_ = other.initialized_;

    other.fd_ = -1;
    other.w_ = 0;
    other.h_ = 0;
    // other.display_ = nullptr;
    other.texture_ = 0;
    other.eglImage_ = EGL_NO_IMAGE_KHR;
    other.fbo_ = 0;
    other.initialized_ = false;

    return *this;
}

bool GL_rgba_egl::initialized() const
{
    return initialized_;
}
bool GL_rgba_egl::make_current() const
{
    if(!initialized())
    {
        return false;
    }

    if (!display_->make_current_display())//remove this if no dispaly and context changing
    {
        return false;
    }
    
    glBindTexture(GL_TEXTURE_2D, texture_);
    glBindFramebuffer(GL_FRAMEBUFFER, fbo_);

    return true;

}