#include <fcntl.h>
#include <linux/dma-buf.h>
#include <sys/ioctl.h>
#include <sys/mman.h>

#include <GL/glew.h>
#include <GLFW/glfw3.h>

#include <iostream>
#include <vector>

#include "p2p_gl_gl.h"

GLFW_GLEW_EXT_INIT::GLFW_GLEW_EXT_INIT() : init_(false), inside_window_(nullptr)
{
    if (!glfwInit())
    {
        std::cerr << "Failed to initialize GLFW\n";
        return;
    }

    // this window is only for glewInit() and is unused in the rest of the demo
    glfwWindowHint(GLFW_VISIBLE, GLFW_FALSE);
    GLFWwindow *window = glfwCreateWindow(1920, 1080, "Off-screen Rendering", nullptr, nullptr);
    if (!window)
    {
        std::cerr << "Failed to create GLFW window\n";
        glfwTerminate();
        return;
    }

    inside_window_ = window;

    glfwMakeContextCurrent(window);

    if (glewInit() != GLEW_OK)
    {
        std::cerr << "Failed to initialize GLEW\n";
        glfwDestroyWindow(window);
        glfwTerminate();
        return;
    }

    if (!glewIsSupported("GL_EXT_memory_object"))
    {
        std::cerr << "GL_EXT_memory_object is not supported\n";
        glfwDestroyWindow(window);
        glfwTerminate();
        return;
    }

    if (!glewIsSupported("GL_EXT_memory_object_fd"))
    {
        std::cerr << "GL_EXT_memory_object_fd is not supported\n";
        glfwDestroyWindow(window);
        glfwTerminate();
        return;
    }

    // glfwDestroyWindow(window);//Destory here?

    init_ = true;
}

GLFW_GLEW_EXT_INIT &GLFW_GLEW_EXT_INIT::init_gwfw_glew_ext()
{
    static GLFW_GLEW_EXT_INIT init;
    return init;
}

GLFW_window::GLFW_window(int width, int height)
{
    glfwWindowHint(GLFW_VISIBLE, GLFW_FALSE);
    window_ = glfwCreateWindow(width, height, "Off-screen Rendering", nullptr, nullptr);
    if (!window_)
    {
        std::cerr << "Failed to create GLFW window\n";
    }
}

GL_rgba_glew::GL_rgba_glew(int fd, int w, int h, std::shared_ptr<GLFW_window> window) : GL_rgba_base(),
                                                                                        fd_{fd},
                                                                                        w_{w},
                                                                                        h_{h},
                                                                                        window_{window},
                                                                                        texture_{0},
                                                                                        memobj_{0},
                                                                                        fbo_{0},
                                                                                        initialized_(false)
{
    if(!window_->initialized())
    {
        std::cerr<<"Invaild window\n";
        return;
    }

    glfwMakeContextCurrent(window->window());

    glGenTextures(1, &texture_);

    // create OpenGL memory object
    glCreateMemoryObjectsEXT(1, &memobj_);

    // import DMA-BUF as OpenGL memory object
    glImportMemoryFdEXT(memobj_, w * h * 4, GL_HANDLE_TYPE_OPAQUE_FD_EXT, fd);

    glBindTexture(GL_TEXTURE_2D, texture_);

    // attach the memory object to texture
    glTexStorageMem2DEXT(GL_TEXTURE_2D, 1, GL_RGBA8, w, h, memobj_, 0);

    glGenFramebuffers(1, &fbo_);

    glBindFramebuffer(GL_FRAMEBUFFER, fbo_);

    glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, texture_, 0);

    GLenum status = glCheckFramebufferStatus(GL_FRAMEBUFFER);
    if (status != GL_FRAMEBUFFER_COMPLETE)
    {
        std::cerr << "ERROR: check for FBO status: " << status << std::endl;
        return;
    }

    initialized_ = true;
}

GL_rgba_glew::GL_rgba_glew(GL_rgba_glew &&other) : GL_rgba_base(std::move(other)), window_(std::move(other.window_))
{
    
    texture_ = other.texture_;
    memobj_ = other.memobj_;
    fbo_ = other.fbo_;
    w_ = other.w_;
    h_ = other.h_;
    initialized_ = other.initialized_;

    other.texture_ = 0;
    other.memobj_ = 0;
    other.fbo_ = 0;
    other.w_ = -1;
    other.h_ = -1;
    other.initialized_ = false;
}

GL_rgba_glew &GL_rgba_glew::operator=(GL_rgba_glew &&other)
{
    if (this == &other)
    {
        return *this;
    }

    GL_rgba_base::operator=(std::move(other));

    if(window_->initialized())
    {
        glfwMakeContextCurrent(window_->window());

        if (texture_ != 0)
        {
            glDeleteTextures(1, &texture_);
        }

        if (memobj_ != 0)
        {
            glDeleteMemoryObjectsEXT(1, &memobj_); // fd is also closed here
        }

        if (fbo_)
        {
            glDeleteFramebuffers(1, &fbo_);
        }
    }

    window_ = std::move(other.window_);
    texture_ = other.texture_;
    memobj_ = other.memobj_;
    fbo_ = other.fbo_;
    w_ = other.w_;
    h_ = other.h_;
    initialized_ = other.initialized_;

    other.texture_ = 0;
    other.memobj_ = 0;
    other.fbo_ = 0;
    other.w_ = -1;
    other.h_ = -1;
    other.initialized_ = false;

    return *this;
}

GL_rgba_glew::~GL_rgba_glew()
{
    if(!window_->initialized())
    {
        return;
    }

    glfwMakeContextCurrent(window_->window());

    if (texture_ != 0)
    {
        glDeleteTextures(1, &texture_);
    }

    if (memobj_ != 0)
    {
        glDeleteMemoryObjectsEXT(1, &memobj_); // fd is also closed here
    }

    if (fbo_)
    {
        glDeleteFramebuffers(1, &fbo_);
    }
}

bool GL_rgba_glew::make_current() const
{
    if(!initialized())
    {
        return false;
    }

    glfwMakeContextCurrent(window_->window());
    glBindTexture(GL_TEXTURE_2D, texture_);
    glBindFramebuffer(GL_FRAMEBUFFER, fbo_);

    return true;
}
