#pragma once

#include <condition_variable>
#include <iostream>
#include <memory>
#include <mutex>
#include <set>
#include <thread>
#include <tuple>
#include <string>

#include <ni_device_api.h>


//blocking queue sorted by display order
class ni_block_frame_set
{
public:

    struct ni_frame_stream_t
    {
        //Currently one p2p frame is allocated within an uploader instance
        ni_session_context_t *p_upl_ctx = nullptr;

        // frame associated with p_upl_ctx
        ni_frame_t *frame = nullptr;

        // flag indicating the end of input, used to control the program's termination
        bool input_end = false;

        // picture order (display order) count
        int poc = -1;

        // information returned by encoder indicating if a resend is needed
        bool need_to_resend = false;
        
        // DMA address to read from
        ni_p2p_sgl_t *p_dma_address = nullptr;
    };

private:
    static bool ni_frame_stream_compare_by_poc(const ni_frame_stream_t&lhs, const ni_frame_stream_t&rhs)
    {
        return lhs.poc < rhs.poc;
    }

public:
    using ni_frame_set = std::set<ni_frame_stream_t, decltype(&ni_frame_stream_compare_by_poc)>;

    ni_block_frame_set();

    ni_block_frame_set(const ni_block_frame_set &) = delete;
    ni_block_frame_set &operator=(const ni_block_frame_set) = delete;

    ni_block_frame_set(ni_block_frame_set &&) = default;
    ni_block_frame_set &operator=(ni_block_frame_set &&) = default;

    /*!****************************************************************************
    *  \brief  get the first item(the frame with the smallest poc)
    *          and pop it from the set
    *          This function is blocking and thread-safe
    *
    *  \return  the ni_frame_set item
    *******************************************************************************/
    ni_frame_stream_t get_and_pop();

    /*!****************************************************************************
    *  \brief  get and pop 
    *          when the smallest poc of the item in the set equals to the parameter poc
    *          This function is blocking and thread-safe
    * 
    *  \param [in] poc - the poc which is expected to be the smallest poc
    * 
    *  \return  the ni_frame_set item
    *******************************************************************************/
    ni_frame_stream_t get_and_pop_if_begin_poc(int poc);

    /*!****************************************************************************
    *  \brief  push an item to the set 
    *          This function is blocking and thread-safe
    * 
    *  \param [in] poc - the item to be pushed
    * 
    *******************************************************************************/
    void push(ni_frame_stream_t item);

    /*!****************************************************************************
    *  \brief  push an item to the set but without mutex 
    *          This function is not thread-safe
    * 
    *  \param [in] poc - the item to be pushed
    * 
    *******************************************************************************/
    void push_with_no_mutex(ni_frame_stream_t item);

    ~ni_block_frame_set();

private:

    std::mutex m_mtx;

    std::condition_variable m_cv;

    std::shared_ptr<ni_frame_set> m_frame_buffer;
};