/*******************************************************************************
 *
 * Copyright (C) 2023 NETINT Technologies
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 ******************************************************************************/

/*!*****************************************************************************
 *  \file   ni_ai_transcode.c
 *
 *  \brief  Application for performing video processing using libxcoder API.
 *          Its code provides examples on how to programatically use
 *          libxcoder API.
 ******************************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdint.h>

#define _POSIX_C_SOURCE 200809L
#include <getopt.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/time.h>
#include <time.h>

#include "ni_device_api.h"
#include "ni_util.h"
#include "ni_av_codec.h"
#include "ni_bitstream.h"
// max YUV frame size
#define MAX_YUV_FRAME_SIZE (7680 * 4320 * 3 / 2)
#define MAX_ABGR_FRAME_SIZE (7680 * 4320 * 4)
#define POOL_SIZE 3
#define FILE_NAME_LEN 256
#define MAX_INPUT_FILES 1

#define NI_TEST_RETCODE_FAILURE -1
#define NI_TEST_RETCODE_SUCCESS 0
#define NI_TEST_RETCODE_END_OF_STREAM 1
#define NI_TEST_RETCODE_EAGAIN 2
#define NI_TEST_RETCODE_NEXT_INPUT 3
#define NI_TEST_RETCODE_SEQ_CHANGE_DONE 4

int enc_eos_sent = 0;
uint32_t number_of_frames = 0;
uint32_t number_of_packets = 0;
uint64_t data_left_size[MAX_INPUT_FILES] = {0};
int g_repeat = 1;

struct timeval start_time;
struct timeval previous_time;
struct timeval current_time;

time_t start_timestamp = 0;
time_t previous_timestamp = 0;
time_t current_timestamp = 0;

unsigned long curr_found_pos = {0};
uint8_t *g_file_cache[MAX_INPUT_FILES] = {NULL};
static int g_reconfigCount = 0;

unsigned long total_file_size[MAX_INPUT_FILES] = {0};

typedef struct _ni_roi_network_layer {
    int32_t width;
    int32_t height;
    int32_t channel;
    int32_t classes;
    int32_t component;
    int32_t mask[3];
    float biases[12];
    int32_t output_number;
    float *output;
} ni_roi_network_layer_t;

typedef struct _ni_roi_network {
    int32_t netw;
    int32_t neth;
    ni_network_data_t raw;
    ni_roi_network_layer_t *layers;
} ni_roi_network_t;

// simplistic ref counted HW frame
typedef struct _ni_hwframe_ref_t {
    int ref_cnt;
    niFrameSurface1_t surface;
} ni_hwframe_ref_t;

static ni_hwframe_ref_t g_hwframe_pool[NI_MAX_DR_HWDESC_FRAME_INDEX];

uint8_t *g_curr_cache_pos = NULL;

typedef enum {
    NI_SW_PIX_FMT_NONE = -1,       /* invalid format       */
    NI_SW_PIX_FMT_YUV444P,         /* 8-bit YUV444 planar  */
    NI_SW_PIX_FMT_YUV444P10LE,     /* 10-bit YUV444 planar */
} ni_sw_pix_fmt_t;

typedef struct _device_state {
    int dec_sos_sent;
    int dec_eos_sent;
    int dec_eos_received;
    int enc_resend;
    int enc_sos_sent;
    int enc_eos_sent;
    int enc_eos_received;
} device_state_t;

typedef enum _ni_nalu_type {
    H264_NAL_UNSPECIFIED = 0,
    H264_NAL_SLICE = 1,
    H264_NAL_DPA = 2,
    H264_NAL_DPB = 3,
    H264_NAL_DPC = 4,
    H264_NAL_IDR_SLICE = 5,
    H264_NAL_SEI = 6,
    H264_NAL_SPS = 7,
    H264_NAL_PPS = 8,
    H264_NAL_AUD = 9,
    H264_NAL_END_SEQUENCE = 10,
    H264_NAL_END_STREAM = 11,
    H264_NAL_FILLER_DATA = 12,
    H264_NAL_SPS_EXT = 13,
    H264_NAL_PREFIX = 14,
    H264_NAL_SUB_SPS = 15,
    H264_NAL_DPS = 16,
    H264_NAL_AUXILIARY_SLICE = 19,
} ni_nalu_type_t;

typedef enum _ni_hevc_nalu_type {
    HEVC_NAL_TRAIL_N = 0,
    HEVC_NAL_TRAIL_R = 1,
    HEVC_NAL_TSA_N = 2,
    HEVC_NAL_TSA_R = 3,
    HEVC_NAL_STSA_N = 4,
    HEVC_NAL_STSA_R = 5,
    HEVC_NAL_RADL_N = 6,
    HEVC_NAL_RADL_R = 7,
    HEVC_NAL_RASL_N = 8,
    HEVC_NAL_RASL_R = 9,
    HEVC_NAL_IDR_W_RADL = 19,
    HEVC_NAL_IDR_N_LP = 20,
    HEVC_NAL_CRA_NUT = 21,
    HEVC_NAL_VPS = 32,
    HEVC_NAL_SPS = 33,
    HEVC_NAL_PPS = 34,
    HEVC_NAL_AUD = 35,
    HEVC_NAL_EOS_NUT = 36,
    HEVC_NAL_EOB_NUT = 37,
    HEVC_NAL_FD_NUT = 38,
    HEVC_NAL_SEI_PREFIX = 39,
    HEVC_NAL_SEI_SUFFIX = 40,
} ni_hevc_nalu_type;

typedef struct _ni_h264_sps_t {
    int width;
    int height;

    unsigned int sps_id;
    int profile_idc;
    int level_idc;
    int chroma_format_idc;
    int transform_bypass;     ///< qpprime_y_zero_transform_bypass_flag
    int log2_max_frame_num;   ///< log2_max_frame_num_minus4 + 4
    int poc_type;             ///< pic_order_cnt_type
    int log2_max_poc_lsb;     ///< log2_max_pic_order_cnt_lsb_minus4
    int delta_pic_order_always_zero_flag;
    int offset_for_non_ref_pic;
    int offset_for_top_to_bottom_field;
    int poc_cycle_length;   ///< num_ref_frames_in_pic_order_cnt_cycle
    int ref_frame_count;    ///< num_ref_frames
    int gaps_in_frame_num_allowed_flag;
    int mb_width;   ///< pic_width_in_mbs_minus1 + 1
    ///< (pic_height_in_map_units_minus1 + 1) * (2 - frame_mbs_only_flag)
    int mb_height;
    int frame_mbs_only_flag;
    int mb_aff;   ///< mb_adaptive_frame_field_flag
    int direct_8x8_inference_flag;
    int crop;   ///< frame_cropping_flag

    unsigned int crop_left;     ///< frame_cropping_rect_left_offset
    unsigned int crop_right;    ///< frame_cropping_rect_right_offset
    unsigned int crop_top;      ///< frame_cropping_rect_top_offset
    unsigned int crop_bottom;   ///< frame_cropping_rect_bottom_offset
    int vui_parameters_present_flag;
    ni_rational_t sar;
    int video_signal_type_present_flag;
    int full_range;
    int colour_description_present_flag;
    ni_color_primaries_t color_primaries;
    ni_color_transfer_characteristic_t color_trc;
    ni_color_space_t colorspace;
    int timing_info_present_flag;
    uint32_t num_units_in_tick;
    uint32_t time_scale;
    int fixed_frame_rate_flag;
    short offset_for_ref_frame[256];
    int bitstream_restriction_flag;
    int num_reorder_frames;
    unsigned int max_dec_frame_buffering;
    int scaling_matrix_present;
    uint8_t scaling_matrix4[6][16];
    uint8_t scaling_matrix8[6][64];
    int nal_hrd_parameters_present_flag;
    int vcl_hrd_parameters_present_flag;
    int pic_struct_present_flag;
    int time_offset_length;
    int cpb_cnt;                            ///< See H.264 E.1.2
    int initial_cpb_removal_delay_length;   ///< initial_cpb_removal_delay_length_minus1 + 1
    int cpb_removal_delay_length;   ///< cpb_removal_delay_length_minus1 + 1
    int dpb_output_delay_length;    ///< dpb_output_delay_length_minus1 + 1
    int bit_depth_luma;             ///< bit_depth_luma_minus8 + 8
    int bit_depth_chroma;           ///< bit_depth_chroma_minus8 + 8
    int residual_color_transform_flag;   ///< residual_colour_transform_flag
    int constraint_set_flags;            ///< constraint_set[0-3]_flag
    uint8_t data[4096];
    size_t data_size;
} ni_h264_sps_t;

#define HEVC_MAX_SUB_LAYERS 7
#define HEVC_MAX_SHORT_TERM_REF_PIC_SETS 64
#define HEVC_MAX_LONG_TERM_REF_PICS 32

typedef struct _ni_h265_window_t {
    unsigned int left_offset;
    unsigned int right_offset;
    unsigned int top_offset;
    unsigned int bottom_offset;
} ni_h265_window_t;

typedef struct VUI {
    ni_rational_t sar;

    int overscan_info_present_flag;
    int overscan_appropriate_flag;

    int video_signal_type_present_flag;
    int video_format;
    int video_full_range_flag;
    int colour_description_present_flag;
    uint8_t colour_primaries;
    uint8_t transfer_characteristic;
    uint8_t matrix_coeffs;

    int chroma_loc_info_present_flag;
    int chroma_sample_loc_type_top_field;
    int chroma_sample_loc_type_bottom_field;
    int neutra_chroma_indication_flag;

    int field_seq_flag;
    int frame_field_info_present_flag;

    int default_display_window_flag;
    ni_h265_window_t def_disp_win;

    int vui_timing_info_present_flag;
    uint32_t vui_num_units_in_tick;
    uint32_t vui_time_scale;
    int vui_poc_proportional_to_timing_flag;
    int vui_num_ticks_poc_diff_one_minus1;
    int vui_hrd_parameters_present_flag;

    int bitstream_restriction_flag;
    int tiles_fixed_structure_flag;
    int motion_vectors_over_pic_boundaries_flag;
    int restricted_ref_pic_lists_flag;
    int min_spatial_segmentation_idc;
    int max_bytes_per_pic_denom;
    int max_bits_per_min_cu_denom;
    int log2_max_mv_length_horizontal;
    int log2_max_mv_length_vertical;
} VUI;

typedef struct PTLCommon {
    uint8_t profile_space;
    uint8_t tier_flag;
    uint8_t profile_idc;
    uint8_t profile_compatibility_flag[32];
    uint8_t progressive_source_flag;
    uint8_t interlaced_source_flag;
    uint8_t non_packed_constraint_flag;
    uint8_t frame_only_constraint_flag;
    uint8_t max_12bit_constraint_flag;
    uint8_t max_10bit_constraint_flag;
    uint8_t max_8bit_constraint_flag;
    uint8_t max_422chroma_constraint_flag;
    uint8_t max_420chroma_constraint_flag;
    uint8_t max_monochrome_constraint_flag;
    uint8_t intra_constraint_flag;
    uint8_t one_picture_only_constraint_flag;
    uint8_t lower_bit_rate_constraint_flag;
    uint8_t max_14bit_constraint_flag;
    uint8_t inbld_flag;
    uint8_t level_idc;
} PTLCommon;

typedef struct PTL {
    PTLCommon general_ptl;
    PTLCommon sub_layer_ptl[HEVC_MAX_SUB_LAYERS];

    uint8_t sub_layer_profile_present_flag[HEVC_MAX_SUB_LAYERS];
    uint8_t sub_layer_level_present_flag[HEVC_MAX_SUB_LAYERS];
} PTL;

typedef struct ScalingList {
    /* This is a little wasteful, since sizeID 0 only needs 8 coeffs,
     * and size ID 3 only has 2 arrays, not 6. */
    uint8_t sl[4][6][64];
    uint8_t sl_dc[2][6];
} ScalingList;

typedef struct ShortTermRPS {
    unsigned int num_negative_pics;
    int num_delta_pocs;
    int rps_idx_num_delta_pocs;
    int32_t delta_poc[32];
    uint8_t used[32];
} ShortTermRPS;

typedef struct _ni_h265_sps_t {
    unsigned vps_id;
    int chroma_format_idc;
    uint8_t separate_colour_plane_flag;

    ni_h265_window_t output_window;
    ni_h265_window_t pic_conf_win;

    int bit_depth;
    int bit_depth_chroma;
    int pixel_shift;
    int pix_fmt;

    unsigned int log2_max_poc_lsb;
    int pcm_enabled_flag;

    int max_sub_layers;
    struct {
        int max_dec_pic_buffering;
        int num_reorder_pics;
        int max_latency_increase;
    } temporal_layer[HEVC_MAX_SUB_LAYERS];
    uint8_t temporal_id_nesting_flag;

    VUI vui;
    PTL ptl;

    uint8_t scaling_list_enable_flag;
    ScalingList scaling_list;

    unsigned int nb_st_rps;
    ShortTermRPS st_rps[HEVC_MAX_SHORT_TERM_REF_PIC_SETS];

    uint8_t amp_enabled_flag;
    uint8_t sao_enabled;

    uint8_t long_term_ref_pics_present_flag;
    uint16_t lt_ref_pic_poc_lsb_sps[HEVC_MAX_LONG_TERM_REF_PICS];
    uint8_t used_by_curr_pic_lt_sps_flag[HEVC_MAX_LONG_TERM_REF_PICS];
    uint8_t num_long_term_ref_pics_sps;

    struct {
        uint8_t bit_depth;
        uint8_t bit_depth_chroma;
        unsigned int log2_min_pcm_cb_size;
        unsigned int log2_max_pcm_cb_size;
        uint8_t loop_filter_disable_flag;
    } pcm;
    uint8_t sps_temporal_mvp_enabled_flag;
    uint8_t sps_strong_intra_smoothing_enable_flag;

    unsigned int log2_min_cb_size;
    unsigned int log2_diff_max_min_coding_block_size;
    unsigned int log2_min_tb_size;
    unsigned int log2_max_trafo_size;
    unsigned int log2_ctb_size;
    unsigned int log2_min_pu_size;

    int max_transform_hierarchy_depth_inter;
    int max_transform_hierarchy_depth_intra;

    int sps_range_extension_flag;
    int transform_skip_rotation_enabled_flag;
    int transform_skip_context_enabled_flag;
    int implicit_rdpcm_enabled_flag;
    int explicit_rdpcm_enabled_flag;
    int extended_precision_processing_flag;
    int intra_smoothing_disabled_flag;
    int high_precision_offsets_enabled_flag;
    int persistent_rice_adaptation_enabled_flag;
    int cabac_bypass_alignment_enabled_flag;

    ///< coded frame dimension in various units
    int width;
    int height;
    int ctb_width;
    int ctb_height;
    int ctb_size;
    int min_cb_width;
    int min_cb_height;
    int min_tb_width;
    int min_tb_height;
    int min_pu_width;
    int min_pu_height;
    int tb_mask;

    int hshift[3];
    int vshift[3];

    int qp_bd_offset;

    uint8_t data[4096];
    int data_size;
} ni_h265_sps_t;

static inline int need_time_print(struct timeval *curr, struct timeval *prev)
{
#ifdef _WIN32
    return curr->tv_sec - prev->tv_sec > 3;
#else
    return curr->tv_sec - prev->tv_sec > 1;
#endif
}

/*!****************************************************************************
 *  \brief  Exit on argument error
 *
 *  \param[in]  arg_name    pointer to argument name
 *        [in]  param       pointer to provided parameter
 *
 *  \return     None        program exit
 ******************************************************************************/
void arg_error_exit(char *arg_name, char *param)
{
    fprintf(stderr, "Error: unrecognized argument for %s, \"%s\"\n", arg_name,
            param);
    exit(-1);
}

/*!****************************************************************************
 *  \brief  Read the next frame
 *
 *  \param[in]  fd          file descriptor of input file
 *        [out] p_dst       pointer to place the frame
 *        [in]  to_read     number of bytes to copy to the pointer
 *        [in]  input_index input index
 *
 *  \return     bytes copied
 ******************************************************************************/
int read_next_chunk_from_file(int fd, uint8_t *p_dst, uint32_t to_read, int input_index)
{
    uint8_t *tmp_dst = p_dst;
    ni_log(NI_LOG_DEBUG,
           "read_next_chunk_from_file:p_dst %p len %u totalSize %llu left %llu\n",
           tmp_dst, to_read, (unsigned long long)total_file_size[input_index], (unsigned long long)data_left_size[input_index]);
    int to_copy = to_read;
    unsigned long tmpFileSize = to_read;
    if (data_left_size[input_index] == 0) {
        if (g_repeat > 1) {
            data_left_size[input_index] = total_file_size[input_index];
            g_repeat--;
            ni_log(NI_LOG_DEBUG, "input processed %d left\n", g_repeat);
            lseek(fd, 0, SEEK_SET);   //back to beginning
        } else {
            return 0;
        }
    } else if (data_left_size[input_index] < to_read) {
        tmpFileSize = data_left_size[input_index];
        to_copy = data_left_size[input_index];
    }

    int one_read_size = read(fd, tmp_dst, to_copy);
    if (one_read_size == -1) {
        fprintf(stderr, "Error: reading file, quit! left-to-read %lu\n",
                tmpFileSize);
        fprintf(stderr, "Error: input file read error\n");
        return -1;
    }
    else if (one_read_size != to_copy)
    {
        fprintf(stderr,
                "Error: read unexpected bytes from file, actual %d != expected "
                "%u\n",
                one_read_size, to_copy);
        return -1;
    }
    data_left_size[input_index] -= one_read_size;

    return one_read_size;
}

/*!****************************************************************************
 *  \brief  Load the input file into memory
 *
 *  \param [in] filename        name of input file
 *         [out] bytes_read     number of bytes read from file
 *
 *  \return     0 on success
 *              < 0 on error
 ******************************************************************************/
int load_input_file(const char *filename, unsigned long *bytes_read)
{
    struct stat info;

    /* Get information on the file */
    if (stat(filename, &info) < 0) {
        fprintf(stderr, "Can't stat %s\n", filename);
        return -1;
    }

    /* Check the file size */
    if (info.st_size <= 0) {
        fprintf(stderr, "File %s is empty\n", filename);
        return -1;
    }

    *bytes_read = info.st_size;

    return 0;
}

static ni_pixel_planar_format get_pixel_planar(ni_pix_fmt_t pix_fmt)
{
    ni_pixel_planar_format ret = -1;
    switch (pix_fmt) {
        case NI_PIX_FMT_NV12:
        case NI_PIX_FMT_P010LE:
            ret = NI_PIXEL_PLANAR_FORMAT_SEMIPLANAR;
            break;
        case NI_PIX_FMT_8_TILED4X4:
        case NI_PIX_FMT_10_TILED4X4:
            ret = NI_PIXEL_PLANAR_FORMAT_TILED4X4;
            break;
        case NI_PIX_FMT_YUV420P:
        case NI_PIX_FMT_YUV420P10LE:
        case NI_PIX_FMT_ABGR: /* 32-bit ABGR packed        */
        case NI_PIX_FMT_ARGB:
        case NI_PIX_FMT_RGBA:
        case NI_PIX_FMT_BGRA:
            ret = NI_PIXEL_PLANAR_FORMAT_PLANAR;
            break;
        default:
            break;
    }

    return ret;
}

int ni_frame_copy_aux_data(ni_frame_t *p_dst_frame,
                           const ni_frame_t *p_src_frame) {
    int i;
    ni_aux_data_t *aux;

    for (i = 0; i < p_src_frame->nb_aux_data; i++) {
        aux = p_src_frame->aux_data[i];
        if (p_dst_frame->aux_data[i]->size != aux->size) {
            if (p_dst_frame->aux_data[i]->size)
                free(p_dst_frame->aux_data[i]->data);
            p_dst_frame->aux_data[i]->data = malloc(aux->size);
            if (!p_dst_frame->aux_data[i]->data)
                return NI_RETCODE_ERROR_MEM_ALOC;
            p_dst_frame->aux_data[i]->size = aux->size;
            p_dst_frame->aux_data[i]->type = aux->type;
        }
        memcpy(p_dst_frame->aux_data[i]->data, aux->data, aux->size);
    }
    return NI_RETCODE_SUCCESS;
}

int frame_copy_props(ni_session_data_io_t *dst, const ni_session_data_io_t *src) {
    int ret = NI_RETCODE_SUCCESS;
    ni_frame_t *p_dst_frame = &dst->data.frame;
    const ni_frame_t *p_src_frame = &src->data.frame;
    p_dst_frame->src_codec = p_src_frame->src_codec;
    p_dst_frame->dts = p_src_frame->dts;
    p_dst_frame->pts = p_src_frame->pts;
    p_dst_frame->end_of_stream = p_src_frame->end_of_stream;
    p_dst_frame->start_of_stream = p_src_frame->start_of_stream;

    p_dst_frame->ni_pict_type = p_src_frame->ni_pict_type;

    p_dst_frame->sei_cc_len = p_src_frame->sei_cc_len;
    p_dst_frame->sei_hdr_mastering_display_color_vol_offset =
            p_src_frame->sei_hdr_mastering_display_color_vol_offset;
    p_dst_frame->sei_hdr_mastering_display_color_vol_len =
            p_src_frame->sei_hdr_mastering_display_color_vol_len;
    p_dst_frame->sei_hdr_content_light_level_info_offset =
            p_src_frame->sei_hdr_content_light_level_info_offset;
    p_dst_frame->sei_hdr_content_light_level_info_len =
            p_src_frame->sei_hdr_content_light_level_info_len;
    p_dst_frame->sei_hdr_plus_offset = p_src_frame->sei_hdr_plus_offset;
    p_dst_frame->sei_hdr_plus_len = p_src_frame->sei_hdr_plus_len;
    p_dst_frame->sei_user_data_unreg_offset =
            p_src_frame->sei_user_data_unreg_offset;
    p_dst_frame->sei_user_data_unreg_len = p_src_frame->sei_user_data_unreg_len;
    p_dst_frame->sei_alt_transfer_characteristics_offset =
            p_src_frame->sei_alt_transfer_characteristics_offset;
    p_dst_frame->sei_alt_transfer_characteristics_len =
            p_src_frame->sei_alt_transfer_characteristics_len;
    p_dst_frame->vui_offset = p_src_frame->vui_offset;
    p_dst_frame->vui_len = p_src_frame->vui_len;

    p_dst_frame->pixel_format = p_src_frame->pixel_format;
    p_dst_frame->nb_aux_data = p_src_frame->nb_aux_data;

    p_dst_frame->color_primaries = p_src_frame->color_primaries;
    p_dst_frame->color_trc = p_src_frame->color_trc;
    p_dst_frame->color_space = p_src_frame->color_space;
    p_dst_frame->video_full_range_flag = p_src_frame->video_full_range_flag;
    p_dst_frame->aspect_ratio_idc = p_src_frame->aspect_ratio_idc;
    p_dst_frame->sar_width = p_src_frame->sar_width;
    p_dst_frame->sar_height = p_src_frame->sar_height;
    p_dst_frame->vui_num_units_in_tick = p_src_frame->vui_num_units_in_tick;
    p_dst_frame->vui_time_scale = p_src_frame->vui_time_scale;
    // copy aux data
    ret = ni_frame_copy_aux_data(p_dst_frame, p_src_frame);
    if (ret < 0) {
        return ret;
    }
    // TBD: support custom sei

    return ret;
}

// Note we do not need to consider padding bytes from yuv/rgba file reading
static inline int frame_read_buffer_size(int w, int h, ni_pix_fmt_t pix_fmt,
                                         ni_sw_pix_fmt_t sw_pix_fmt)
{
    int data_len = 0;

    if (sw_pix_fmt == NI_SW_PIX_FMT_YUV444P) {
        data_len = w * h * 3;
    } else if (sw_pix_fmt == NI_SW_PIX_FMT_YUV444P10LE) {
        data_len = w * h * 6;
    } else {
        switch (pix_fmt) {
            case NI_PIX_FMT_NV12:
            case NI_PIX_FMT_YUV420P:
                data_len = w * h * 3 / 2;
                break;
            case NI_PIX_FMT_P010LE:
            case NI_PIX_FMT_YUV420P10LE:
                data_len = w * h * 3;
                break;
            case NI_PIX_FMT_RGBA:
            case NI_PIX_FMT_BGRA:
            case NI_PIX_FMT_ARGB:
            case NI_PIX_FMT_ABGR:
            case NI_PIX_FMT_BGR0:
            case NI_PIX_FMT_BGRP:
                data_len = w * 4 * h;
                break;
            default:
                break;
        }
    }

    return data_len;
}

// find/copy next H.264 NAL unit (including start code) and its type;
// return NAL data size if found, 0 otherwise
uint64_t find_h264_next_nalu(uint8_t *p_dst, int *nal_type)
{
    uint64_t data_size;
    uint64_t i = curr_found_pos;

    if (i + 3 >= total_file_size[0]) {
        ni_log(NI_LOG_DEBUG,
               "%s reaching end, curr_pos %llu "
               "total input size %llu\n",
               __func__, (unsigned long long)curr_found_pos, (unsigned long long)total_file_size[0]);

        if (g_repeat > 1) {
            g_repeat--;
            ni_log(NI_LOG_DEBUG, "input processed, %d left\n", g_repeat);
            i = curr_found_pos = 0;
        } else {
            return 0;
        }
    }

    // search for start code 0x000001 or 0x00000001
    while ((g_curr_cache_pos[i] != 0x00 || g_curr_cache_pos[i + 1] != 0x00 ||
            g_curr_cache_pos[i + 2] != 0x01) &&
           (g_curr_cache_pos[i] != 0x00 || g_curr_cache_pos[i + 1] != 0x00 ||
            g_curr_cache_pos[i + 2] != 0x00 || g_curr_cache_pos[i + 3] != 0x01)) {
        i++;
        if (i + 3 > total_file_size[0]) {
            return 0;
        }
    }

    // found start code, advance to NAL unit start depends on actual start code
    if (g_curr_cache_pos[i] != 0x00 || g_curr_cache_pos[i + 1] != 0x00 ||
        g_curr_cache_pos[i + 2] != 0x01) {
        i++;
    }

    i += 3;

    // get the NAL type
    *nal_type = (g_curr_cache_pos[i] & 0x1f);

    // advance to the end of NAL, or stream
    while ((g_curr_cache_pos[i] != 0x00 || g_curr_cache_pos[i + 1] != 0x00 ||
            g_curr_cache_pos[i + 2] != 0x00) &&
           (g_curr_cache_pos[i] != 0x00 || g_curr_cache_pos[i + 1] != 0x00 ||
            g_curr_cache_pos[i + 2] != 0x01)) {
        i++;
        // if reaching the stream end
        if (i + 3 > total_file_size[0]) {
            data_size = total_file_size[0] - curr_found_pos;
            memcpy(p_dst, &g_curr_cache_pos[curr_found_pos], data_size);
            curr_found_pos = total_file_size[0];
            return data_size;
        }
    }

    data_size = i - curr_found_pos;
    memcpy(p_dst, &g_curr_cache_pos[curr_found_pos], data_size);
    curr_found_pos = i;
    return data_size;
}

// parse H.264 slice header to get frame_num; return 0 if success, -1 otherwise
int parse_h264_slice_header(uint8_t *buf, int size_bytes, ni_h264_sps_t *sps,
                            int32_t *frame_num, unsigned int *first_mb_in_slice)
{
    ni_bitstream_reader_t br;
    uint8_t *p_buf = buf;
    unsigned int slice_type, pps_id;

    // skip the start code
    while (!(p_buf[0] == 0x00 && p_buf[1] == 0x00 && p_buf[2] == 0x01) &&
           size_bytes > 3) {
        p_buf++;
        size_bytes--;
    }
    if (size_bytes <= 3) {
        ni_log(NI_LOG_ERROR, "Error parse_h264_slice_header slice has no header\n");
        return -1;
    }

    p_buf += 3;
    size_bytes -= 3;

    ni_bitstream_reader_init(&br, p_buf, 8 * size_bytes);

    // skip NAL header
    ni_bs_reader_skip_bits(&br, 8);

    *first_mb_in_slice = ni_bs_reader_get_ue(&br);
    slice_type = ni_bs_reader_get_ue(&br);
    if (slice_type > 9) {
        ni_log(NI_LOG_ERROR, "parse_h264_slice_header error: slice type %u too "
                             "large at %u\n",
               slice_type, *first_mb_in_slice);
        return -1;
    }
    pps_id = ni_bs_reader_get_ue(&br);
    *frame_num = ni_bs_reader_get_bits(&br, sps->log2_max_frame_num);

    ni_log(NI_LOG_DEBUG, "parse_h264_slice_header slice type %u frame_num %d "
                         "pps_id %u size %d first_mb %u\n",
           slice_type, *frame_num, pps_id, size_bytes,
           *first_mb_in_slice);

    return 0;
}

// rewind input data buffer position by a number of bytes, if possible
void rewind_data_buf_pos_by(uint64_t nb_bytes)
{
    if (curr_found_pos >= nb_bytes) {
        curr_found_pos -= nb_bytes;
    } else {
        ni_log(NI_LOG_ERROR, "Error %s %d bytes!\n", __func__, nb_bytes);
    }
}

/**
 * find/copy next H.265 NAL unit (including start code) and its type;
 * return NAL data size if found, 0 otherwise
*/
uint64_t find_h265_next_nalu(uint8_t *p_dst, int *nal_type)
{
    uint64_t data_size;
    uint64_t i = curr_found_pos;

    if (i + 3 >= total_file_size[0]) {
        ni_log(NI_LOG_DEBUG,
               "%s reaching end, curr_pos %llu "
               "total input size %llu\n",
               __func__, (unsigned long long)curr_found_pos, (unsigned long long)total_file_size[0]);

        if (g_repeat > 1) {
            g_repeat--;
            ni_log(NI_LOG_DEBUG, "input processed, %d left\n", g_repeat);
            i = curr_found_pos = 0;
        } else {
            return 0;
        }
    }

    // search for start code 0x000001 or 0x00000001
    while ((g_curr_cache_pos[i] != 0x00 || g_curr_cache_pos[i + 1] != 0x00 ||
            g_curr_cache_pos[i + 2] != 0x01) &&
           (g_curr_cache_pos[i] != 0x00 || g_curr_cache_pos[i + 1] != 0x00 ||
            g_curr_cache_pos[i + 2] != 0x00 || g_curr_cache_pos[i + 3] != 0x01)) {
        i++;
        if (i + 3 > total_file_size[0]) {
            return 0;
        }
    }

    // found start code, advance to NAL unit start depends on actual start code
    if (g_curr_cache_pos[i] != 0x00 || g_curr_cache_pos[i + 1] != 0x00 ||
        g_curr_cache_pos[i + 2] != 0x01) {
        i++;
    }

    i += 3;

    // get the NAL type
    *nal_type = (g_curr_cache_pos[i] & 0x7E) >> 1;

    // advance to the end of NAL, or stream
    while ((g_curr_cache_pos[i] != 0x00 || g_curr_cache_pos[i + 1] != 0x00 ||
            g_curr_cache_pos[i + 2] != 0x00) &&
           (g_curr_cache_pos[i] != 0x00 || g_curr_cache_pos[i + 1] != 0x00 ||
            g_curr_cache_pos[i + 2] != 0x01)) {
        i++;
        // if reaching the stream end
        if (i + 3 > total_file_size[0]) {
            data_size = total_file_size[0] - curr_found_pos;
            memcpy(p_dst, &g_curr_cache_pos[curr_found_pos], data_size);
            curr_found_pos = total_file_size[0];
            return data_size;
        }
    }

    data_size = i - curr_found_pos;
    memcpy(p_dst, &g_curr_cache_pos[curr_found_pos], data_size);
    curr_found_pos = i;
    return data_size;
}

// return actual bytes copied from cache, in requested size
uint32_t read_next_chunk(uint8_t *p_dst, uint32_t to_read)
{
    unsigned int to_copy = to_read;

    if (data_left_size[0] == 0) {
        if (g_repeat > 1) {
            data_left_size[0] = total_file_size[0];
            g_repeat--;
            ni_log(NI_LOG_DEBUG, "input processed %d left\n", g_repeat);
            to_copy = to_read;
            g_curr_cache_pos = g_file_cache[0];
        } else {
            return 0;
        }
    } else if (data_left_size[0] < to_read) {
        to_copy = data_left_size[0];
    }

    memcpy(p_dst, g_curr_cache_pos, to_copy);
    g_curr_cache_pos += to_copy;
    data_left_size[0] -= to_copy;

    return to_copy;
}

/*!*****************************************************************************
 *  \brief  Send decoder input data
 *
 *  \param
 *
 *  \return
 ******************************************************************************/
ni_retcode_t decoder_send_data(ni_session_context_t *p_dec_ctx,
                               ni_session_data_io_t *p_in_data,
                               int input_video_width, int input_video_height,
                               int pkt_size, unsigned long *total_bytes_sent,
                               int print_time, device_state_t *p_device_state,
                               void *stream_info)
{
    static uint8_t tmp_buf[NI_MAX_TX_SZ] = {0};
    uint8_t *tmp_buf_ptr = tmp_buf;
    int packet_size = pkt_size;
    // int chunk_size = 0;
    uint32_t frame_pkt_size = 0, nal_size;
    int nal_type = -1;
    int tx_size = 0;
    uint32_t send_size = 0;
    int new_packet = 0;
    int saved_prev_size = 0;
    int32_t frame_num = -1, curr_frame_num;
    unsigned int first_mb_in_slice = 0;
    ni_packet_t *p_in_pkt = &(p_in_data->data.packet);
    ni_retcode_t retval = NI_RETCODE_SUCCESS;

    ni_log(NI_LOG_DEBUG, "===> decoder_send_data <===\n");

    if (p_device_state->dec_eos_sent) {
        ni_log(NI_LOG_DEBUG, "decoder_send_data: ALL data (incl. eos) sent "
                             "already!\n");
        LRETURN;
    }

    if (0 == p_in_pkt->data_len) {
        memset(p_in_pkt, 0, sizeof(ni_packet_t));

        if (NI_CODEC_FORMAT_H264 == p_dec_ctx->codec_format) {
            ni_h264_sps_t *sps;
            sps = (ni_h264_sps_t *)stream_info;
            // send whole encoded packet which ends with a slice NAL
            while ((nal_size = find_h264_next_nalu(tmp_buf_ptr, &nal_type)) > 0) {
                frame_pkt_size += nal_size;
                tmp_buf_ptr += nal_size;
                ni_log(NI_LOG_DEBUG, "%s nal %d  nal_size %d\n", __func__,
                       nal_type, nal_size);

                // save parsed out sps/pps as stream headers in the decode session
                if (H264_NAL_PPS == nal_type) {
                    if (NI_RETCODE_SUCCESS !=
                        ni_device_dec_session_save_hdrs(p_dec_ctx, tmp_buf,
                                                        frame_pkt_size)) {
                        ni_log(NI_LOG_ERROR, "decoder_send_data: save_hdr failed!\n");
                    }
                }

                if (H264_NAL_SLICE == nal_type ||
                    H264_NAL_IDR_SLICE == nal_type) {
                    if (!parse_h264_slice_header(tmp_buf_ptr - nal_size,
                                                 nal_size, sps, &curr_frame_num,
                                                 &first_mb_in_slice)) {
                        if (-1 == frame_num) {
                            // first slice, continue to check
                            frame_num = curr_frame_num;
                        } else if (curr_frame_num != frame_num ||
                                   0 == first_mb_in_slice) {
                            // this slice has diff. frame_num or first_mb_in_slice addr is
                            // 0: not the same frame and return
                            rewind_data_buf_pos_by(nal_size);
                            frame_pkt_size -= nal_size;
                            break;
                        }
                        // this slice is in the same frame, so continue to check and see
                        // if there is more
                    } else {
                        ni_log(NI_LOG_ERROR,
                               "decoder_send_data: parse_slice_header error "
                               "NAL type %d size %u, continue\n",
                               nal_type, nal_size);
                    }
                } else if (-1 != frame_num) {
                    // already got a slice and this is non-slice NAL: return
                    rewind_data_buf_pos_by(nal_size);
                    frame_pkt_size -= nal_size;
                    break;
                }
                // otherwise continue until a slice is found
            }   // while there is still NAL
        } else if (NI_CODEC_FORMAT_H265 == p_dec_ctx->codec_format) {
            while ((nal_size = find_h265_next_nalu(tmp_buf_ptr, &nal_type)) > 0) {
                frame_pkt_size += nal_size;
                tmp_buf_ptr += nal_size;
                ni_log(NI_LOG_DEBUG, "%s nal_type %d nal_size %d\n", __func__,
                       nal_type, nal_size);

                if (nal_type == HEVC_NAL_VPS || nal_type == HEVC_NAL_SPS ||
                    nal_type == HEVC_NAL_PPS) { // save vps, sps, pps
                    if (nal_type == 32 && p_dec_ctx->prev_size > 0) {
                        // sequence change situation, replace previous p_leftover with new headers
                        p_dec_ctx->prev_size = 0;
                    }
                    // copy the nal to the end of p_leftover
                    memcpy(p_dec_ctx->p_leftover + p_dec_ctx->prev_size,
                           tmp_buf_ptr - nal_size, nal_size);
                    p_dec_ctx->prev_size += nal_size;
                }

                if (nal_type >= 0 && nal_type <= 23) { // vcl units
                    ni_log(NI_LOG_DEBUG, "%s send vcl_nal %d nal_size %d\n",
                           __func__, nal_type, nal_size);
                    break;
                }
            }
        } else {
            frame_pkt_size = read_next_chunk(tmp_buf, packet_size);
            // chunk_size = frame_pkt_size;
        }
        ni_log(NI_LOG_DEBUG, "decoder_send_data * frame_pkt_size %d\n",
               frame_pkt_size);

        p_in_pkt->p_data = NULL;
        p_in_pkt->data_len = frame_pkt_size;

        if (frame_pkt_size + p_dec_ctx->prev_size > 0) {
            ni_packet_buffer_alloc(p_in_pkt,
                                   (int)frame_pkt_size + p_dec_ctx->prev_size);
        }

        new_packet = 1;
        send_size = frame_pkt_size + p_dec_ctx->prev_size;
        saved_prev_size = p_dec_ctx->prev_size;
    } else {
        send_size = p_in_pkt->data_len;
    }

    p_in_pkt->start_of_stream = 0;
    if (!p_device_state->dec_sos_sent) {
        p_in_pkt->start_of_stream = 1;
        p_device_state->dec_sos_sent = 1;
    }
    p_in_pkt->end_of_stream = 0;
    p_in_pkt->video_width = input_video_width;
    p_in_pkt->video_height = input_video_height;

    if (send_size == 0) {
        if (new_packet) {
            send_size =
                    ni_packet_copy(p_in_pkt->p_data, tmp_buf, 0,
                                   p_dec_ctx->p_leftover, &p_dec_ctx->prev_size);
        }
        p_in_pkt->data_len = send_size;

        if (curr_found_pos) {
            p_in_pkt->end_of_stream = 1;
            printf("Sending p_last packet (size %u) + eos\n", p_in_pkt->data_len);
        }
    } else {
        if (new_packet) {
            send_size =
                    ni_packet_copy(p_in_pkt->p_data, tmp_buf, frame_pkt_size,
                                   p_dec_ctx->p_leftover, &p_dec_ctx->prev_size);
            // p_in_pkt->data_len is the actual packet size to be sent to decoder
            p_in_pkt->data_len += saved_prev_size;
        }
    }

    tx_size =
            ni_device_session_write(p_dec_ctx, p_in_data, NI_DEVICE_TYPE_DECODER);

    if (tx_size < 0) {
        // Error
        fprintf(stderr, "Error: sending data error. rc:%d\n", tx_size);
        retval = NI_RETCODE_FAILURE;
        LRETURN;
    } else if (tx_size == 0) {
        ni_log(NI_LOG_DEBUG, "0 byte sent this time, sleep and will re-try.\n");
        ni_usleep(10000);
    } else if ((uint32_t)tx_size < send_size) {
        if (print_time) {
            //printf("Sent %d < %d , re-try next time ?\n", tx_size, send_size);
        }
    }

    *total_bytes_sent += tx_size;

    if (p_dec_ctx->ready_to_close) {
        p_device_state->dec_eos_sent = 1;
    }

    if (print_time) {
        printf("decoder_send_data: success, total sent: %lu\n",
               *total_bytes_sent);
    }

    if (tx_size > 0) {
        ni_log(NI_LOG_DEBUG, "decoder_send_data: reset packet_buffer.\n");
        ni_packet_buffer_free(p_in_pkt);
    }

    retval = NI_RETCODE_SUCCESS;

    END:

    return retval;
}

/*!*****************************************************************************
 *  \brief  Receive decoded output data from decoder
 *
 *  \param
 *
 *  \return 0: got YUV frame;  1: end-of-stream;  2: got nothing
 ******************************************************************************/
int decoder_receive_data(ni_session_context_t *p_dec_ctx,
                         ni_session_data_io_t *p_out_data,
                         int output_video_width, int output_video_height,
                         FILE *p_file, unsigned long long *total_bytes_received,
                         int print_time, int write_to_file,
                         device_state_t *p_device_state,
                         int * p_rx_size)
{
    int rc = NI_RETCODE_FAILURE;
    int end_flag = 0;
    int rx_size = 0;
    bool b_is_hwframe = p_dec_ctx->hw_action;
    ni_frame_t *p_out_frame = &(p_out_data->data.frame);
    int width, height;
    // In decoder session read function it will allocate the actual YUV
    // transfer size for the very first read. And the pixel format of session
    // context would be set as well. So it does not matter with the planar
    // format for the first call of this function.
    int is_planar = get_pixel_planar(p_dec_ctx->pixel_format) == NI_PIXEL_PLANAR_FORMAT_PLANAR;

    ni_log(NI_LOG_DEBUG,
           "===> decoder_receive_data hwframe %d pixel_format %d <===\n",
           b_is_hwframe, p_dec_ctx->pixel_format);

    if (p_device_state->dec_eos_received) {
        ni_log(NI_LOG_DEBUG, "decoder_receive_data eos received already, Done!\n");
        rc = 2;
        LRETURN;
    }

    // prepare memory buffer for receiving decoded frame
    width = p_dec_ctx->actual_video_width > 0 ?
            (int)(p_dec_ctx->actual_video_width) :
            output_video_width;
    height = p_dec_ctx->active_video_height > 0 ?
             (int)(p_dec_ctx->active_video_height) :
             output_video_height;

    // allocate memory only after resolution is known (for buffer pool set up)
    int alloc_mem = (p_dec_ctx->active_video_width > 0 &&
                     p_dec_ctx->active_video_height > 0 ?
                     1 :
                     0);
    if (!b_is_hwframe) {
        rc = ni_decoder_frame_buffer_alloc(
                p_dec_ctx->dec_fme_buf_pool, &(p_out_data->data.frame), alloc_mem,
                width, height, p_dec_ctx->codec_format == NI_CODEC_FORMAT_H264,
                p_dec_ctx->bit_depth_factor, is_planar);
        if (NI_RETCODE_SUCCESS != rc) {
            LRETURN;
        }
        rx_size = ni_device_session_read(p_dec_ctx, p_out_data,
                                         NI_DEVICE_TYPE_DECODER);
    } else {
        rc = ni_frame_buffer_alloc(
                &(p_out_data->data.frame), width, height,
                p_dec_ctx->codec_format == NI_CODEC_FORMAT_H264, 1,
                p_dec_ctx->bit_depth_factor,
                3 /*3 is max supported hwframe output count per frame*/, is_planar);
        if (NI_RETCODE_SUCCESS != rc) {
            LRETURN;
        }
        rx_size = ni_device_session_read_hwdesc(p_dec_ctx, p_out_data,
                                                NI_DEVICE_TYPE_DECODER);
    }

    // the actual pix_fmt is known and updated in ctx only after the first
    // frame is decoded, so check/update it here again to be used below
    is_planar = get_pixel_planar(p_dec_ctx->pixel_format) ==
                NI_PIXEL_PLANAR_FORMAT_PLANAR;

    end_flag = p_out_frame->end_of_stream;

    if (rx_size < 0) {
        fprintf(stderr, "Error: receiving data error. rc:%d\n", rx_size);
        if (!b_is_hwframe) {
            ni_decoder_frame_buffer_free(&(p_out_data->data.frame));
        } else {
            ni_frame_buffer_free(&(p_out_data->data.frame));
        }
        rc = NI_RETCODE_FAILURE;
        LRETURN;
    } else if (rx_size > 0) {
        number_of_frames++;
        ni_log(NI_LOG_DEBUG, "Got frame # %" PRIu64 " bytes %d\n",
               p_dec_ctx->frame_num, rx_size);

        ni_dec_retrieve_aux_data(p_out_frame);
    }
    // rx_size == 0 means no decoded frame is available now
    if (rx_size > 0 && p_file && write_to_file) {
        int i, j;
        uint8_t *src;
        for (i = 0; i < 3; i++) {
            src = p_out_frame->p_data[i];

            int plane_height = p_dec_ctx->active_video_height;
            int plane_width = p_dec_ctx->active_video_width;
            int write_height = output_video_height;
            int write_width = output_video_width;

            // support for 8/10 bit depth
            // plane_width is the actual Y stride size
            write_width *= p_dec_ctx->bit_depth_factor;

            if (i == 1 || i == 2) {
                plane_height /= 2;
                // U/V stride size is multiple of 128, following the calculation
                // in ni_decoder_frame_buffer_alloc
                plane_width = (((int)(p_dec_ctx->actual_video_width) / 2 *
                                p_dec_ctx->bit_depth_factor +
                                127) /
                               128) *
                              128;

                if (!is_planar) {
                    plane_width = ((((int)(p_dec_ctx->actual_video_width) *
                                     p_dec_ctx->bit_depth_factor +
                                     127) /
                                    128) *
                                   128) /
                                  p_dec_ctx->bit_depth_factor;
                    // for semi-planar format, output UV at same time (data[1])
                    // and skip data[2]
                    if (i == 1) {
                        write_width *= 2;
                    }
                    if (i == 2) {
                        plane_height = 0;
                    }
                }

                write_height /= 2;
                write_width /= 2;
            }

            // apply the cropping windown in writing out the YUV frame
            // for now the windown is usually crop-left = crop-top = 0, and we
            // use this to simplify the cropping logic
            for (j = 0; j < plane_height; j++) {
                if (j < write_height &&
                    fwrite(src, write_width, 1, p_file) != 1) {
                    fprintf(stderr,
                            "Error: writing data plane %d: height %d error!\n",
                            i, plane_height);
                    fprintf(stderr, "Error: ferror rc = %d\n", ferror(p_file));
                }
                src += plane_width;
            }
        }

        if (fflush(p_file)) {
            fprintf(stderr,
                    "Error: writing data frame flush failed! errno %d\n",
                    errno);
        }
    }

    *total_bytes_received += rx_size;
    *p_rx_size = rx_size;

    if (print_time) {
        printf("[R] Got:%d  Frames= %u  fps=%f  Total bytes %llu\n", rx_size,
               number_of_frames,
               ((float)number_of_frames /
                (float)(current_time.tv_sec - start_time.tv_sec)),
               (unsigned long long)*total_bytes_received);
    }

    if (end_flag) {
        printf("Decoder Receiving done.\n");
        p_device_state->dec_eos_received = 1;
        rc = 1;
    } else if (0 == rx_size) {
        rc = 2;
    }

    ni_log(NI_LOG_DEBUG, "decoder_receive_data: success\n");

    END:

    return rc;
}

/*!*****************************************************************************
 *  \brief  Receive output data from encoder
 *
 *  \param  p_in_data is passed in to specify new frame resolution upon sequence
 *          change
 *
 *  \return 0 - success got packet
 *          1 - received eos
 *          2 - got nothing, need retry
 *          -1 - failure
 ******************************************************************************/
int encoder_receive_data(
        ni_session_context_t *p_enc_ctx, ni_session_data_io_t *p_out_data,
        int output_video_width, int output_video_height, FILE *p_file,
        unsigned long long *total_bytes_received, int print_time,
        volatile unsigned int *nb_pkts_received,
        ni_session_data_io_t * p_in_data)
{
    int packet_size = NI_MAX_TX_SZ;
    int rc = 0;
    int end_flag = 0;
    int rx_size = 0;
    ni_packet_t *p_out_pkt = &(p_out_data->data.packet);
    int meta_size = p_enc_ctx->meta_size;
    ni_xcoder_params_t *p_api_param = (ni_xcoder_params_t *)p_enc_ctx->p_session_config;

    ni_log(NI_LOG_DEBUG, "===> encoder_receive_data <===\n");
    if (NI_INVALID_SESSION_ID == p_enc_ctx->session_id) {
        // keep-alive-thread timeout will set session_id to invalid, should exit
        ni_log(NI_LOG_ERROR,
               "encode session id invalid, the session should be closed\n");
        return NI_TEST_RETCODE_FAILURE;
    }

    receive_data:
    rc = ni_packet_buffer_alloc(p_out_pkt, packet_size);
    if (rc != NI_RETCODE_SUCCESS) {
        fprintf(stderr, "Error: malloc packet failed, ret = %d!\n", rc);
        return NI_TEST_RETCODE_FAILURE;
    }

    rc = ni_device_session_read(p_enc_ctx, p_out_data, NI_DEVICE_TYPE_ENCODER);

    end_flag = p_out_pkt->end_of_stream;
    rx_size = rc;

    ni_log(NI_LOG_DEBUG, "encoder_receive_data: received data size=%d\n", rx_size);

    if (rx_size > meta_size) {

        if (p_file &&
            (fwrite((uint8_t *)p_out_pkt->p_data + meta_size,
                    p_out_pkt->data_len - meta_size, 1, p_file) != 1)) {
            fprintf(stderr, "Error: writing data %u bytes error!\n",
                    p_out_pkt->data_len - meta_size);
            fprintf(stderr, "Error: ferror rc = %d\n", ferror(p_file));
        }
        *total_bytes_received += rx_size - meta_size;

        if (0 == p_enc_ctx->pkt_num) {
            p_enc_ctx->pkt_num = 1;
            ni_log(NI_LOG_DEBUG, "got encoded stream header, keep reading ..\n");
            goto receive_data;
        }
        (*nb_pkts_received)++;

        ni_log(NI_LOG_DEBUG, "Got:   Packets= %u\n", *nb_pkts_received);
    } else if (rx_size != 0) {
        fprintf(stderr, "Error: received %d bytes, <= metadata size %d!\n",
                rx_size, meta_size);
        return NI_TEST_RETCODE_FAILURE;
    } else if (!end_flag && p_api_param->low_delay_mode) {
        ni_log(NI_LOG_DEBUG, "low delay mode and NO pkt, keep reading ..\n");
        ni_usleep(200);
        goto receive_data;
    }

    if (print_time) {
        uint32_t time_diff = (uint32_t)(current_time.tv_sec - start_time.tv_sec);
        if (time_diff == 0) {
            time_diff = 1;
        }
        printf("[R] Got:%d   Packets= %u fps=%f  Total bytes %llu\n", rx_size,
               *nb_pkts_received, (float)p_enc_ctx->frame_num / (float)time_diff,
               *total_bytes_received);
    }

    if (end_flag) {
        printf("Encoder Receiving done.\n");
        return NI_TEST_RETCODE_END_OF_STREAM;
    } else if (0 == rx_size) {
        return NI_TEST_RETCODE_EAGAIN;
    }
    ni_log(NI_LOG_DEBUG, "encoder_receive_data: success\n");

    return NI_TEST_RETCODE_SUCCESS;
}


/*!*****************************************************************************
 *  \brief  decoder session open
 *
 *  \param
 *
 *  \return 0 if successful, < 0 otherwise
 ******************************************************************************/
int decoder_open_session(ni_session_context_t *p_dec_ctx, int iXcoderGUID,
                         ni_xcoder_params_t *p_dec_params)
{
    int ret = 0;

    p_dec_ctx->p_session_config = p_dec_params;
    p_dec_ctx->session_id = NI_INVALID_SESSION_ID;

    // assign the card GUID in the encoder context and let session open
    // take care of the rest
    p_dec_ctx->device_handle = NI_INVALID_DEVICE_HANDLE;
    p_dec_ctx->blk_io_handle = NI_INVALID_DEVICE_HANDLE;
    p_dec_ctx->hw_id = iXcoderGUID;

    if (p_dec_params->dec_input_params.hwframes) {
        p_dec_ctx->hw_action = NI_CODEC_HW_ENABLE;
    } else {
        p_dec_ctx->hw_action = NI_CODEC_HW_NONE;
    }

    ret = ni_device_session_open(p_dec_ctx, NI_DEVICE_TYPE_DECODER);

    if (ret != NI_RETCODE_SUCCESS) {
        fprintf(stderr, "Error: ni_decoder_session_open() failure!\n");
        return -1;
    } else {
        printf("Decoder device %d session open successful.\n", iXcoderGUID);
        return 0;
    }
}

/*!****************************************************************************
 *  \brief  Open an encoder session to Quadra
 *
 *  \param  [out] p_enc_ctx         pointer to an encoder session context
 *          [in]  dst_codec_format  AVC or HEVC
 *          [in]  iXcoderGUID       id to identify the Quadra device
 *          [in]  p_enc_params      sets the encoder parameters
 *          [in]  width             width of frames to encode
 *          [in]  height            height of frames to encode
 *          [in]  p_frame           first frame to encode
 *
 *  \return 0 if successful, < 0 otherwise
 ******************************************************************************/
int encoder_open_session(ni_session_context_t *p_enc_ctx, int dst_codec_format,
                         int iXcoderGUID, ni_xcoder_params_t *p_enc_params,
                         int width, int height, ni_frame_t *p_frame)
{
    int ret = 0;

    // Enable hardware frame encoding
    p_enc_ctx->hw_action = NI_CODEC_HW_ENABLE;
    p_enc_params->hwframes = 1;

    // Provide the first frame to the Quadra encoder
    p_enc_params->p_first_frame = p_frame;

    // Specify codec, AVC vs HEVC
    p_enc_ctx->codec_format = dst_codec_format;

    p_enc_ctx->p_session_config = p_enc_params;
    p_enc_ctx->session_id = NI_INVALID_SESSION_ID;

    // Assign the card GUID in the encoder context to open a session
    // to that specific Quadra device
    p_enc_ctx->device_handle = NI_INVALID_DEVICE_HANDLE;
    p_enc_ctx->blk_io_handle = NI_INVALID_DEVICE_HANDLE;
    p_enc_ctx->hw_id = iXcoderGUID;

    ni_encoder_set_input_frame_format(p_enc_ctx, p_enc_params, width, height, 8,
                                      NI_FRAME_LITTLE_ENDIAN, 1);

    // Encoder will operate in P2P mode
    ret = ni_device_session_open(p_enc_ctx, NI_DEVICE_TYPE_ENCODER);
    if (ret != NI_RETCODE_SUCCESS) {
        fprintf(stderr, "Error: encoder open session failure\n");
    } else {
        printf("Encoder device %d session open successful\n", iXcoderGUID);
    }

    return ret;
}

/*!*****************************************************************************
*  \brief  open scaler session
*
*  \param
*
*  \return 0 if successful, < 0 otherwise
******************************************************************************/
static int ai_open_session(ni_session_context_t *p_ai_ctx,
                           int iXcoderGUID, int width, int height,
                           int pool_size, ni_network_data_t* p_raw, const char* nb_file_path)
{
    int ret = 0;

    p_ai_ctx->session_id = NI_INVALID_SESSION_ID;

    p_ai_ctx->device_handle = NI_INVALID_DEVICE_HANDLE;
    p_ai_ctx->blk_io_handle = NI_INVALID_DEVICE_HANDLE;
    p_ai_ctx->hw_id = iXcoderGUID;
    p_ai_ctx->hw_action = NI_CODEC_HW_ENABLE;
    p_ai_ctx->device_type = NI_DEVICE_TYPE_AI;
    p_ai_ctx->keep_alive_timeout = NI_DEFAULT_KEEP_ALIVE_TIMEOUT;

    ret = ni_device_session_open(p_ai_ctx, NI_DEVICE_TYPE_AI);

    if (ret != NI_RETCODE_SUCCESS) {
        fprintf(stderr, "Error: ni_ai_session_open() failure!\n");
        return -1;
    } else {
#ifdef _WIN32
        printf("Scaler session open: device_handle %p, session_id %u.\n",
               p_scaler_ctx->device_handle, p_scaler_ctx->session_id);
#else
        printf("AI session open: device_handle %d, session_id %u.\n",
               p_ai_ctx->device_handle, p_ai_ctx->session_id);
#endif
    }
    printf("nb=%s\n",nb_file_path);
    ret = ni_ai_config_network_binary(p_ai_ctx, p_raw, nb_file_path);
    if(ret != NI_RETCODE_SUCCESS){
        fprintf(stderr, "Ai Config error, ret=%d\n", ret);
        return ret;
    }

    int options;

    options = NI_AI_FLAG_IO |  NI_AI_FLAG_PC;
    /* Allocate a pool of frames by the scaler */
    ret = ni_device_alloc_frame( p_ai_ctx,
                                 width,
                                 height,
                                 GC620_I420,
                                 options,
                                 0, // rec width
                                 0, // rec height
                                 0, // rec X pos
                                 0, // rec Y pos
                                 pool_size, // rgba color/pool size
                                 0, // frame index
                                 NI_DEVICE_TYPE_AI);

    printf("Ai open session finish\n");

    return 0;
}

/*!****************************************************************************
 *  \brief    Print usage information
 *
 *  \param    none
 *
 *  \return   none
 ******************************************************************************/
void print_usage(void)
{
    printf("Video encoder/P2P application directly using Netint "
           "Libxcoder release v%s\n"
           "Usage: xcoderp2p [options]\n"
           "\n"
           "options:\n"
           "--------------------------------------------------------------------------------\n"
           "  -h | --help        Show help.\n"
           "  -v | --version     Print version info.\n"
           "  -l | --loglevel    Set loglevel of libxcoder API.\n"
           "                     [none, fatal, error, info, debug, trace]\n"
           "                     Default: info\n"
           "  -c | --card        Set card index to use.\n"
           "                     See `ni_rsrc_mon` for cards on system.\n"
           "                     (Default: 0)\n"
           "  -i | --input       Input file path.\n"
           "  -r | --repeat      (Positive integer) to Repeat input X times "
           "for performance \n"
           "                     test. (Default: 1)\n"
           "  -s | --size        Resolution of input file in format "
           "WIDTHxHEIGHT.\n"
           "                     (eg. '1920x1080')\n"
           "  -o | --output      Output file path.\n",
           "  -d | --decoder-params Decode parameters.\n"
           "  -n | --nbfile      AI model for overlay.\n"
           NI_XCODER_REVISION);
}

/*!****************************************************************************
 *  \brief  Parse user command line arguments
 *
 *  \param [in] argc                argument count
 *         [in] argv                argument vector
 *         [out] input_filename     input filename
 *         [out] output_filename    output filename
 *         [out] nb_filename        model file
 *         [out] iXcoderGUID        Quadra device
 *         [out] arg_width          resolution width
 *         [out] arg_height         resolution height
 *         [out] arg_pic_width      pic resolution width
 *         [out] arg_pic_ehight     pic resolution height
 *         [out] decConfXcoderParams dec param
 *
 *  \return nothing                 program exit on error
 ******************************************************************************/
void parse_arguments(int argc, char *argv[], char (*input_filename)[FILE_NAME_LEN],
                     char *output_filename, char* nb_filename, int *iXcoderGUID, int *arg_width,
                     int *arg_height, int *arg_pic_width, int *arg_pic_height, char *decConfXcoderParams)
{
    char xcoderGUID[32];
    char *n;   // used for parsing width and height from --size
    int opt;
    int opt_index;
    ni_log_level_t log_level;
    static int input_index = 0;
    static int input_resolution_index = 0;

    static const char *opt_string = "hvl:c:i:s:o:r:d:n:";
    static const struct option long_options[] = {
            {"help", no_argument, NULL, 'h'},
            {"version", no_argument, NULL, 'v'},
            {"loglevel", no_argument, NULL, 'l'},
            {"card", required_argument, NULL, 'c'},
            {"input", required_argument, NULL, 'i'},
            {"size", required_argument, NULL, 's'},
            {"output", required_argument, NULL, 'o'},
            {"repeat", required_argument, NULL, 'r'},
            {"decoder-params", required_argument, NULL, 'd'},
            {"nbfile", required_argument, NULL, 'm'},
            {NULL, 0, NULL, 0},
    };

    while ((opt = getopt_long(argc, argv, opt_string, long_options,
                              &opt_index)) != -1) {
        switch (opt) {
            case 'h':
                print_usage();
                exit(0);
            case 'v':
                printf("Release ver: %s\n"
                       "API ver:     %s\n"
                       "Date:        %s\n"
                       "ID:          %s\n",
                       NI_XCODER_REVISION, LIBXCODER_API_VERSION,
                       NI_SW_RELEASE_TIME, NI_SW_RELEASE_ID);
                exit(0);
            case 'l':
                log_level = arg_to_ni_log_level(optarg);
                if (log_level != NI_LOG_INVALID) {
                    ni_log_set_level(log_level);
                } else {
                    arg_error_exit("-l | --loglevel", optarg);
                }
                break;
            case 'c':
                strcpy(xcoderGUID, optarg);
                *iXcoderGUID = (int)strtol(optarg, &n, 10);
                // No numeric characters found in left side of optarg
                if (n == xcoderGUID)
                    arg_error_exit("-c | --card", optarg);
                break;
            case 'i':
                if (input_index == MAX_INPUT_FILES) {
                    printf("Error: number of input files cannot exceed %d\n", MAX_INPUT_FILES);
                    exit(-1);
                }
                strcpy(input_filename[input_index], optarg);
                input_index++;
                break;
            case 's':
                if(input_resolution_index == 0) {
                    *arg_width = (int)strtol(optarg, &n, 10);
                    *arg_height = atoi(n + 1);
                    if ((*n != 'x') || (!*arg_width || !*arg_height))
                        arg_error_exit("-s | --size", optarg);
                    input_resolution_index++;
                } else {
                    *arg_pic_width = (int)strtol(optarg, &n, 10);
                    *arg_pic_height = atoi(n + 1);
                    if ((*n != 'x') || (!*arg_pic_width || !*arg_pic_height))
                        arg_error_exit("-s | --size", optarg);
                }
                break;
            case 'o':
                strcpy(output_filename, optarg);
                break;
            case 'r':
                if (!(atoi(optarg) >= 1))
                    arg_error_exit("-r | --repeat", optarg);
                g_repeat = atoi(optarg);
                break;
            case 'd':
                strcpy(decConfXcoderParams, optarg);
                break;
            case 'n':
                strcpy(nb_filename, optarg);
                break;
            default:
                print_usage();
                exit(1);
        }
    }

    // Check required args are present
    if (!input_filename[0]) {
        printf("Error: missing argument for -i | --input\n");
        exit(-1);
    }

    if (!output_filename[0]) {
        printf("Error: missing argument for -o | --output\n");
        exit(-1);
    }
}

// reference HW frame
static void ni_hw_frame_ref(const niFrameSurface1_t *p_surface)
{
    uint16_t hwframe_index;

    if (!p_surface) {
        return;
    }

    hwframe_index = p_surface->ui16FrameIdx;
    g_hwframe_pool[hwframe_index].ref_cnt++;
    if (1 == g_hwframe_pool[hwframe_index].ref_cnt) {
        memcpy(&g_hwframe_pool[hwframe_index].surface, p_surface,
               sizeof(niFrameSurface1_t));
    }
    ni_log(NI_LOG_TRACE, "%s frame idx %u ref_cnt %d ..\n", __func__,
           hwframe_index, g_hwframe_pool[hwframe_index].ref_cnt);
}

// unref HW frame
static void ni_hw_frame_unref(uint16_t hwframe_index)
{
    if (g_hwframe_pool[hwframe_index].ref_cnt > 0) {
        g_hwframe_pool[hwframe_index].ref_cnt--;
        if (0 == g_hwframe_pool[hwframe_index].ref_cnt &&
            g_hwframe_pool[hwframe_index].surface.ui16FrameIdx) {
            ni_log(NI_LOG_TRACE, "%s frame idx recycing frame idx %u\n", __func__,
                   g_hwframe_pool[hwframe_index].surface.ui16FrameIdx);

            ni_hwframe_buffer_recycle2(&g_hwframe_pool[hwframe_index].surface);
        }
        ni_log(NI_LOG_TRACE, "%s frame idx %u ref_cnt now: %d\n", __func__,
               hwframe_index, g_hwframe_pool[hwframe_index].ref_cnt);
    } else {
        ni_log(NI_LOG_ERROR, "%s error frame idx %u ref_cnt %d <= 0\n",
               __func__, hwframe_index,
               g_hwframe_pool[hwframe_index].ref_cnt);
    }
}
static int encoder_receive(ni_session_context_t *enc_ctx_list,
                           ni_session_data_io_t *in_frame,
                           ni_session_data_io_t *pkt, int width, int height,
                           uint32_t *number_of_packets_list,
                           int output_total, FILE **pfs_list,
                           unsigned long long *total_bytes_received_list,
                           int print_time, device_state_t *xcoder_state)
{
    int i, recycle_index;
    int recv_fin_flag = NI_TEST_RETCODE_SUCCESS;
    uint32_t prev_num_pkt[1] = {0};

    for (i = 0; i < output_total; i++) {
        pkt->data.packet.end_of_stream = 0;
        prev_num_pkt[i] = number_of_packets_list[i];

        recv_fin_flag = encoder_receive_data(&enc_ctx_list[i], pkt, width,
                                             height, pfs_list[i],
                                             &total_bytes_received_list[i],
                                             print_time,
                                             &number_of_packets_list[i],
                                             in_frame);

        recycle_index = pkt->data.packet.recycle_index;
        if (prev_num_pkt[i] < number_of_packets_list[i] &&
            enc_ctx_list[i].hw_action && recycle_index > 0 &&
            recycle_index < NI_GET_MAX_HWDESC_FRAME_INDEX(enc_ctx_list[i].ddr_config)) {
            //encoder only returns valid recycle index
            //when there's something to recycle.
            //This range is suitable for all memory bins
            ni_hw_frame_unref(recycle_index);
        } else {
            ni_log(NI_LOG_DEBUG, "enc %d recv, prev_num_pkt %u "
                                 "number_of_packets_list %u recycle_index %u\n", i,
                   prev_num_pkt[i], number_of_packets_list[i], recycle_index);
        }

        xcoder_state[i].enc_eos_received = pkt->data.packet.end_of_stream;

        if (recv_fin_flag < 0) {
            ni_log(NI_LOG_DEBUG, "enc %d error, quit !\n", i);
            break;
        } else if (recv_fin_flag == NI_TEST_RETCODE_EAGAIN) {
            ni_usleep(100);
        }
    }

    return recv_fin_flag;
}
// convert various reconfig and demo modes (stored in encoder configuration) to
// aux data and store them in frame
void prep_reconf_demo_data(ni_session_context_t *p_enc_ctx, ni_frame_t *frame)
{
    ni_xcoder_params_t *api_param =
            (ni_xcoder_params_t *)p_enc_ctx->p_session_config;
    ni_aux_data_t *aux_data = NULL;

    switch (api_param->reconf_demo_mode) {
        case XCODER_TEST_RECONF_BR:
            if (p_enc_ctx->frame_num ==
                api_param->reconf_hash[g_reconfigCount][0]) {
                aux_data = ni_frame_new_aux_data(
                        frame, NI_FRAME_AUX_DATA_BITRATE, sizeof(int32_t));
                if (!aux_data) {
                    ni_log(NI_LOG_ERROR,
                           "Error %s(): no mem for reconf BR aux_data\n",
                           __func__);
                    return;
                }
                *((int32_t *)aux_data->data) =
                        api_param->reconf_hash[g_reconfigCount][1];
                ni_log(NI_LOG_DEBUG, "%s(): frame #%lu reconfig BR by frame %d\n",
                       __func__, p_enc_ctx->frame_num,
                       api_param->reconf_hash[g_reconfigCount][1]);

                g_reconfigCount++;
            }
            break;
            /*
            case XCODER_TEST_RECONF_INTRAPRD:
                if (p_enc_ctx->frame_num ==
                    api_param->reconf_hash[g_reconfigCount][0])
                {
                    p_enc_ctx->enc_change_params->enable_option |=
                        NI_SET_CHANGE_PARAM_INTRA_PARAM;
                    p_enc_ctx->enc_change_params->intraQP =
                        api_param->reconf_hash[g_reconfigCount][1];
                    p_enc_ctx->enc_change_params->intraPeriod =
                        api_param->reconf_hash[g_reconfigCount][2];
                    p_enc_ctx->enc_change_params->repeatHeaders =
                        api_param->reconf_hash[g_reconfigCount][3];
                    ni_log(NI_LOG_DEBUG,
                        "%s(): frame #%lu reconf intraQP %d intraPeriod %d "
                        "repeatHeaders %d\n",
                        __func__, p_enc_ctx->frame_num,
                        p_enc_ctx->enc_change_params->intraQP,
                        p_enc_ctx->enc_change_params->intraPeriod,
                        p_enc_ctx->enc_change_params->repeatHeaders);

                    // frame reconf_len needs to be set here
                    frame->reconf_len = sizeof(ni_encoder_change_params_t);
                    g_reconfigCount++;
                }
                break;
            */
        case XCODER_TEST_RECONF_VUI_HRD:
            if (p_enc_ctx->frame_num ==
                api_param->reconf_hash[g_reconfigCount][0]) {
                p_enc_ctx->enc_change_params->enable_option |=
                        NI_SET_CHANGE_PARAM_VUI_HRD_PARAM;
                p_enc_ctx->enc_change_params->colorDescPresent =
                        api_param->reconf_hash[g_reconfigCount][1];
                p_enc_ctx->enc_change_params->colorPrimaries =
                        api_param->reconf_hash[g_reconfigCount][2];
                p_enc_ctx->enc_change_params->colorTrc =
                        api_param->reconf_hash[g_reconfigCount][3];
                p_enc_ctx->enc_change_params->colorSpace =
                        api_param->reconf_hash[g_reconfigCount][4];
                p_enc_ctx->enc_change_params->aspectRatioWidth =
                        api_param->reconf_hash[g_reconfigCount][5];
                p_enc_ctx->enc_change_params->aspectRatioHeight =
                        api_param->reconf_hash[g_reconfigCount][6];
                p_enc_ctx->enc_change_params->videoFullRange =
                        api_param->reconf_hash[g_reconfigCount][7];

                // frame reconf_len needs to be set here
                frame->reconf_len = sizeof(ni_encoder_change_params_t);
                g_reconfigCount++;
            }
            break;
        case XCODER_TEST_RECONF_LONG_TERM_REF:
            // the reconf file data line format for this is:
            // <frame-number>:useCurSrcAsLongtermPic,useLongtermRef where
            // values will stay the same on every frame until changed.
            if (p_enc_ctx->frame_num ==
                api_param->reconf_hash[g_reconfigCount][0]) {
                aux_data = ni_frame_new_aux_data(
                        frame, NI_FRAME_AUX_DATA_LONG_TERM_REF,
                        sizeof(ni_long_term_ref_t));
                if (!aux_data) {
                    ni_log(NI_LOG_ERROR,
                           "Error %s(): no mem for reconf LTR aux_data\n",
                           __func__);
                    return;
                }
                ni_long_term_ref_t *ltr = (ni_long_term_ref_t *)aux_data->data;
                ltr->use_cur_src_as_long_term_pic =
                        (uint8_t)api_param->reconf_hash[g_reconfigCount][1];
                ltr->use_long_term_ref =
                        (uint8_t)api_param->reconf_hash[g_reconfigCount][2];

                ni_log(NI_LOG_DEBUG,
                       "%s(): frame #%lu reconf LTR "
                       "use_cur_src_as_long_term_pic %u use_long_term_ref "
                       "%u\n",
                       __func__, p_enc_ctx->frame_num,
                       ltr->use_cur_src_as_long_term_pic, ltr->use_long_term_ref);

                g_reconfigCount++;
            }
            break;
        case XCODER_TEST_RECONF_LTR_INTERVAL:
            if (p_enc_ctx->frame_num ==
                api_param->reconf_hash[g_reconfigCount][0]) {
                aux_data = ni_frame_new_aux_data(
                        frame, NI_FRAME_AUX_DATA_LTR_INTERVAL, sizeof(int32_t));
                if (!aux_data) {
                    ni_log(NI_LOG_ERROR, "Error %s(): no mem for reconf LTR interval "
                                         "aux_data\n",
                           __func__);
                    return;
                }
                *((int32_t *)aux_data->data) =
                        api_param->reconf_hash[g_reconfigCount][1];
                ni_log(NI_LOG_DEBUG, "%s(): frame #%lu reconf LTR interval %d\n",
                       __func__, p_enc_ctx->frame_num,
                       *((int32_t *)aux_data->data));

                g_reconfigCount++;
            }
            break;
        case XCODER_TEST_INVALID_REF_FRAME:
            if (p_enc_ctx->frame_num ==
                api_param->reconf_hash[g_reconfigCount][0]) {
                aux_data = ni_frame_new_aux_data(
                        frame, NI_FRAME_AUX_DATA_INVALID_REF_FRAME,
                        sizeof(int32_t));
                if (!aux_data) {
                    ni_log(NI_LOG_ERROR, "Error %s(): no mem for reconf invalid ref "
                                         "frame aux_data\n",
                           __func__);
                    return;
                }
                *((int32_t *)aux_data->data) =
                        api_param->reconf_hash[g_reconfigCount][1];
                ni_log(NI_LOG_DEBUG, "%s(): frame #%lu reconf invalid ref frame %d\n",
                       __func__, p_enc_ctx->frame_num,
                       *((int32_t *)aux_data->data));

                g_reconfigCount++;
            }
            break;
        case XCODER_TEST_RECONF_FRAMERATE:
            if (p_enc_ctx->frame_num ==
                api_param->reconf_hash[g_reconfigCount][0]) {
                aux_data = ni_frame_new_aux_data(
                        frame, NI_FRAME_AUX_DATA_FRAMERATE, sizeof(ni_framerate_t));
                if (!aux_data) {
                    ni_log(NI_LOG_ERROR,
                           "Error %s(): no mem for reconf framerate aux_data\n",
                           __func__);
                    return;
                }
                ni_framerate_t *framerate = (ni_framerate_t *)aux_data->data;
                framerate->framerate_num =
                        (int32_t)api_param->reconf_hash[g_reconfigCount][1];
                framerate->framerate_denom =
                        (int32_t)api_param->reconf_hash[g_reconfigCount][2];

                ni_log(NI_LOG_DEBUG,
                       "%s(): frame #%lu reconfig framerate by frame (%d/%d)\n",
                       __func__, p_enc_ctx->frame_num, framerate->framerate_num,
                       framerate->framerate_denom);

                g_reconfigCount++;
            }
            break;
        case XCODER_TEST_RECONF_MAX_FRAME_SIZE:
            if (p_enc_ctx->frame_num ==
                api_param->reconf_hash[g_reconfigCount][0]) {
                aux_data = ni_frame_new_aux_data(
                        frame, NI_FRAME_AUX_DATA_MAX_FRAME_SIZE, sizeof(int32_t));
                if (!aux_data) {
                    ni_log(NI_LOG_ERROR,
                           "Error %s(): no mem for reconf maxFrameSize aux_data\n",
                           __func__);
                    return;
                }
                *((int32_t *)aux_data->data) =
                        api_param->reconf_hash[g_reconfigCount][1];
                ni_log(NI_LOG_DEBUG, "%s(): frame #%lu reconfig maxFrameSize by frame %d\n",
                       __func__, p_enc_ctx->frame_num,
                       api_param->reconf_hash[g_reconfigCount][1]);

                g_reconfigCount++;
            }
            break;
        case XCODER_TEST_RECONF_RC_MIN_MAX_QP:
            if (p_enc_ctx->frame_num ==
                api_param->reconf_hash[g_reconfigCount][0]) {
                aux_data = ni_frame_new_aux_data(
                        frame, NI_FRAME_AUX_DATA_MAX_MIN_QP, sizeof(ni_rc_min_max_qp));
                if (!aux_data) {
                    ni_log(NI_LOG_ERROR,
                           "Error %s(): no mem for reconf max&min QP aux_data\n",
                           __func__);
                    return;
                }
                ni_rc_min_max_qp *qp_info = (ni_rc_min_max_qp *)aux_data->data;
                qp_info->minQpI     = api_param->reconf_hash[g_reconfigCount][1];
                qp_info->maxQpI     = api_param->reconf_hash[g_reconfigCount][2];
                qp_info->maxDeltaQp = api_param->reconf_hash[g_reconfigCount][3];
                qp_info->minQpPB    = api_param->reconf_hash[g_reconfigCount][4];
                qp_info->maxQpPB    = api_param->reconf_hash[g_reconfigCount][5];

                g_reconfigCount++;
            }
            break;
        case XCODER_TEST_FORCE_IDR_FRAME:
            if (p_enc_ctx->frame_num ==
                api_param->reconf_hash[g_reconfigCount][0]) {
                ni_force_idr_frame_type(p_enc_ctx);
                ni_log(NI_LOG_DEBUG, "%s(): frame #%lu force IDR frame\n", __func__,
                       p_enc_ctx->frame_num);
                g_reconfigCount++;
            }
            break;
        case XCODER_TEST_RECONF_BR_API:
            if (p_enc_ctx->frame_num ==
                api_param->reconf_hash[g_reconfigCount][0]) {
                ni_reconfig_bitrate(p_enc_ctx,
                                    api_param->reconf_hash[g_reconfigCount][1]);
                ni_log(NI_LOG_DEBUG, "%s(): frame #%lu API reconfig BR %d\n",
                       __func__, p_enc_ctx->frame_num,
                       api_param->reconf_hash[g_reconfigCount][1]);
                g_reconfigCount++;
            }
            break;
        case XCODER_TEST_RECONF_VUI_HRD_API:
            if (p_enc_ctx->frame_num ==
                api_param->reconf_hash[g_reconfigCount][0]) {
                ni_vui_hrd_t vui;
                vui.colorDescPresent =
                        (uint8_t)api_param->reconf_hash[g_reconfigCount][1];
                vui.colorPrimaries =
                        (uint8_t)api_param->reconf_hash[g_reconfigCount][2];
                vui.colorTrc =
                        (uint8_t)api_param->reconf_hash[g_reconfigCount][3];
                vui.colorSpace =
                        (uint8_t)api_param->reconf_hash[g_reconfigCount][4];
                vui.aspectRatioWidth =
                        (uint8_t)api_param->reconf_hash[g_reconfigCount][5];
                vui.aspectRatioHeight =
                        (uint8_t)api_param->reconf_hash[g_reconfigCount][6];
                vui.videoFullRange =
                        (uint8_t)api_param->reconf_hash[g_reconfigCount][7];

                ni_reconfig_vui(p_enc_ctx, &vui);
                ni_log(NI_LOG_DEBUG, "%s(): frame #%lu API reconfig VUI HRD "
                                     "colorDescPresent %d colorPrimaries %d "
                                     "colorTrc %d colorSpace %d aspectRatioWidth %d "
                                     "aspectRatioHeight %d videoFullRange %d\n",
                       __func__, p_enc_ctx->frame_num,
                       api_param->reconf_hash[g_reconfigCount][1],
                       api_param->reconf_hash[g_reconfigCount][2],
                       api_param->reconf_hash[g_reconfigCount][3],
                       api_param->reconf_hash[g_reconfigCount][4],
                       api_param->reconf_hash[g_reconfigCount][5],
                       api_param->reconf_hash[g_reconfigCount][6],
                       api_param->reconf_hash[g_reconfigCount][7]);
                g_reconfigCount++;
            }
            break;
        case XCODER_TEST_RECONF_LTR_API:
            if (p_enc_ctx->frame_num ==
                api_param->reconf_hash[g_reconfigCount][0]) {
                ni_long_term_ref_t ltr;
                ltr.use_cur_src_as_long_term_pic =
                        (uint8_t)api_param->reconf_hash[g_reconfigCount][1];
                ltr.use_long_term_ref =
                        (uint8_t)api_param->reconf_hash[g_reconfigCount][2];

                ni_set_ltr(p_enc_ctx, &ltr);
                ni_log(NI_LOG_DEBUG, "%s(): frame #%lu API set LTR\n", __func__,
                       p_enc_ctx->frame_num);
                g_reconfigCount++;
            }
            break;
        case XCODER_TEST_RECONF_LTR_INTERVAL_API:
            if (p_enc_ctx->frame_num ==
                api_param->reconf_hash[g_reconfigCount][0]) {
                ni_set_ltr_interval(p_enc_ctx,
                                    api_param->reconf_hash[g_reconfigCount][1]);
                ni_log(NI_LOG_DEBUG, "%s(): frame #%lu API set LTR interval %d\n",
                       __func__, p_enc_ctx->frame_num,
                       api_param->reconf_hash[g_reconfigCount][1]);
                g_reconfigCount++;
            }
            break;
        case XCODER_TEST_INVALID_REF_FRAME_API:
            if (p_enc_ctx->frame_num ==
                api_param->reconf_hash[g_reconfigCount][0]) {
                ni_set_frame_ref_invalid(
                        p_enc_ctx, api_param->reconf_hash[g_reconfigCount][1]);
                ni_log(NI_LOG_DEBUG, "%s(): frame #%lu API set frame ref invalid "
                                     "%d\n",
                       __func__, p_enc_ctx->frame_num,
                       api_param->reconf_hash[g_reconfigCount][1]);
                g_reconfigCount++;
            }
            break;
        case XCODER_TEST_RECONF_FRAMERATE_API:
            if (p_enc_ctx->frame_num ==
                api_param->reconf_hash[g_reconfigCount][0]) {
                ni_framerate_t framerate;
                framerate.framerate_num =
                        (int32_t)api_param->reconf_hash[g_reconfigCount][1];
                framerate.framerate_denom =
                        (int32_t)api_param->reconf_hash[g_reconfigCount][2];
                ni_reconfig_framerate(p_enc_ctx, &framerate);
                ni_log(NI_LOG_DEBUG,
                       "%s(): frame #%lu API reconfig framerate (%d/%d)\n",
                       __func__, p_enc_ctx->frame_num,
                       api_param->reconf_hash[g_reconfigCount][1],
                       api_param->reconf_hash[g_reconfigCount][2]);
                g_reconfigCount++;
            }
            break;
        case XCODER_TEST_RECONF_MAX_FRAME_SIZE_API:
            if (p_enc_ctx->frame_num ==
                api_param->reconf_hash[g_reconfigCount][0]) {
                ni_reconfig_max_frame_size(p_enc_ctx,
                                           api_param->reconf_hash[g_reconfigCount][1]);
                ni_log(NI_LOG_DEBUG, "%s(): frame #%lu API reconfig maxFrameSize %d\n",
                       __func__, p_enc_ctx->frame_num,
                       api_param->reconf_hash[g_reconfigCount][1]);
                g_reconfigCount++;
            }
            break;
        case XCODER_TEST_RECONF_RC_MIN_MAX_QP_API:
            if (p_enc_ctx->frame_num ==
                api_param->reconf_hash[g_reconfigCount][0]) {
                ni_rc_min_max_qp qp_info;
                qp_info.minQpI = (int32_t)api_param->reconf_hash[g_reconfigCount][1];
                qp_info.maxQpI = (int32_t)api_param->reconf_hash[g_reconfigCount][2];
                qp_info.maxDeltaQp = (int32_t)api_param->reconf_hash[g_reconfigCount][3];
                qp_info.minQpPB = (int32_t)api_param->reconf_hash[g_reconfigCount][4];
                qp_info.maxQpPB = (int32_t)api_param->reconf_hash[g_reconfigCount][5];
                ni_reconfig_min_max_qp(p_enc_ctx, &qp_info);
                ni_log(NI_LOG_DEBUG,
                       "%s(): frame %d minQpI %d maxQpI %d maxDeltaQp %d minQpPB %d maxQpPB %d\n",
                       __func__, p_enc_ctx->frame_num,
                       qp_info.minQpI, qp_info.maxQpI, qp_info.maxDeltaQp, qp_info.minQpPB, qp_info.maxQpPB);
                g_reconfigCount++;
            }
            break;
        case XCODER_TEST_RECONF_OFF:
        default:
            ;
    }
}
/*******************************************************************************
 *  @brief  Send encoder input data, directly after receiving from decoder
 *
 *  @param  p_enc_ctx encoder context
 *          p_dec_ctx decoder context
 *          p_dec_out_data frame returned by decoder
 *          p_enc_in_data  frame to be sent to encoder
 *
 *  @return
 ******************************************************************************/
int encoder_send_data2(ni_session_context_t *p_enc_ctx,
                       ni_session_data_io_t *p_dec_out_data,
                       ni_session_data_io_t *p_enc_in_data,
                       int input_video_width, int input_video_height,
                       unsigned long *bytes_sent, device_state_t *p_device_state)
{
    int oneSent;
    // pointer to data struct to be sent
    ni_session_data_io_t *p_to_send = NULL;
    // frame pointer to data frame struct to be sent
    ni_frame_t *p_in_frame = NULL;
    ni_xcoder_params_t *api_params =
            (ni_xcoder_params_t *)p_enc_ctx->p_session_config;
    int is_semiplanar = get_pixel_planar(p_enc_ctx->pixel_format) == NI_PIXEL_PLANAR_FORMAT_SEMIPLANAR;
    int is_hwframe = p_enc_ctx->hw_action != NI_CODEC_HW_NONE;

    ni_log(NI_LOG_DEBUG, "===> encoder_send_data2 <===\n");

    if (p_device_state->enc_eos_sent == 1) {
        ni_log(NI_LOG_DEBUG, "encoder_send_data2: ALL data (incl. eos) sent "
                             "already!\n");
        return 1;
    }

    // frame resend
    if (p_device_state->enc_resend) {
        p_to_send = p_enc_in_data;
        p_in_frame = &(p_to_send->data.frame);
        goto send_frame;
    }

    // if the source and target are of the same codec type, AND there is no
    // other aux data such as close caption, HDR10 etc, AND no padding required
    // (e.g. in the case of 32x32 transcoding that needs padding to 256x128),
    // then reuse the YUV frame data layout passed in because it's already in
    // the format required by VPU
    // Note: for now disable the reuse of decode frame !
    if (0/*p_enc_ctx->codec_format == dec_codec_format &&
                                                        input_video_width >= NI_MIN_WIDTH &&
                                                        input_video_height >= NI_MIN_HEIGHT &&
                                                        !p_dec_out_data->data.frame.sei_hdr_content_light_level_info_len &&
                                                        !p_dec_out_data->data.frame.sei_hdr_mastering_display_color_vol_len &&
                                                        !p_dec_out_data->data.frame.sei_hdr_plus_len &&
                                                        !p_dec_out_data->data.frame.sei_cc_len &&
                                                        !p_dec_out_data->data.frame.sei_user_data_unreg_len &&
                                                        !p_dec_out_data->data.frame.roi_len*/) {
        ni_log(NI_LOG_DEBUG,
               "encoder_send_data2: encoding to the same codec "
               "format as the source: %u, NO SEI, reusing the frame struct!\n",
               p_enc_ctx->codec_format);
        p_to_send = p_dec_out_data;
        p_in_frame = &(p_to_send->data.frame);

        p_in_frame->force_key_frame = 0;

        p_in_frame->sei_total_len = p_in_frame->sei_cc_offset =
        p_in_frame->sei_cc_len =
        p_in_frame->sei_hdr_mastering_display_color_vol_offset =
        p_in_frame->sei_hdr_mastering_display_color_vol_len =
        p_in_frame->sei_hdr_content_light_level_info_offset =
        p_in_frame->sei_hdr_content_light_level_info_len =
        p_in_frame->sei_hdr_plus_offset =
        p_in_frame->sei_hdr_plus_len = 0;

        p_in_frame->roi_len = 0;
        p_in_frame->reconf_len = 0;
        p_in_frame->force_pic_qp = 0;
        p_in_frame->extra_data_len = NI_APP_ENC_FRAME_META_DATA_SIZE;
        p_in_frame->ni_pict_type = 0;
    } else {
        // otherwise have to pad/crop the source and copy to a new frame struct
        // and prep for the SEI aux data
        p_to_send = p_enc_in_data;
        p_in_frame = &(p_to_send->data.frame);
        p_in_frame->end_of_stream = p_dec_out_data->data.frame.end_of_stream;
        p_in_frame->ni_pict_type = 0;

        // reset encoder change data buffer
        memset(p_enc_ctx->enc_change_params, 0,
               sizeof(ni_encoder_change_params_t));

        // extra data starts with metadata header, and reset various aux data
        // size
        p_in_frame->extra_data_len = NI_APP_ENC_FRAME_META_DATA_SIZE;
        p_in_frame->roi_len = 0;
        p_in_frame->reconf_len = 0;
        p_in_frame->sei_total_len = 0;
        p_in_frame->force_pic_qp = 0;

        // collect encode reconfig and demo info and save them in the decode out
        // frame, to be used in the aux data prep and copy later
        prep_reconf_demo_data(p_enc_ctx, &(p_dec_out_data->data.frame));

        int dst_stride[NI_MAX_NUM_DATA_POINTERS] = {0};
        int dst_height_aligned[NI_MAX_NUM_DATA_POINTERS] = {0};
        bool alignment_2pass_wa = (
                (api_params->cfg_enc_params.lookAheadDepth ||
                 api_params->cfg_enc_params.crf >= 0) &&
                (p_enc_ctx->codec_format == NI_CODEC_FORMAT_H265 ||
                 p_enc_ctx->codec_format == NI_CODEC_FORMAT_AV1));
        ni_get_hw_yuv420p_dim(input_video_width, input_video_height,
                              p_enc_ctx->bit_depth_factor, is_semiplanar,
                              dst_stride, dst_height_aligned);

        // ROI demo mode takes higher priority over aux data
        // Note: when ROI demo modes enabled, supply ROI map for the specified
        //       range frames, and 0 map for others
        if (api_params->roi_demo_mode && api_params->cfg_enc_params.roi_enable) {
            if (p_enc_ctx->frame_num > 90 && p_enc_ctx->frame_num < 300) {
                p_in_frame->roi_len = p_enc_ctx->roi_len;
            } else {
                p_in_frame->roi_len = 0;
            }
            // when ROI enabled, always have a data buffer for ROI
            // Note: this is handled separately from ROI through side/aux data
            p_in_frame->extra_data_len += p_enc_ctx->roi_len;
        }

        int should_send_sei_with_frame = ni_should_send_sei_with_frame(
                p_enc_ctx, p_in_frame->ni_pict_type, api_params);

        // data buffer for various SEI: HDR mastering display color volume, HDR
        // content light level, close caption, User data unregistered, HDR10+
        // etc.
        uint8_t mdcv_data[NI_MAX_SEI_DATA];
        uint8_t cll_data[NI_MAX_SEI_DATA];
        uint8_t cc_data[NI_MAX_SEI_DATA];
        uint8_t udu_data[NI_MAX_SEI_DATA];
        uint8_t hdrp_data[NI_MAX_SEI_DATA];

        // prep for auxiliary data (various SEI, ROI) in p_in_frame, based on
        // the data returned in decoded frame and also reconfig and demo modes
        // collected in prep_reconf_demo_data
        ni_enc_prep_aux_data(
                p_enc_ctx, p_in_frame, &(p_dec_out_data->data.frame),
                p_enc_ctx->codec_format, should_send_sei_with_frame, mdcv_data,
                cll_data, cc_data, udu_data, hdrp_data);

        p_in_frame->extra_data_len += p_in_frame->sei_total_len;

        // data layout requirement: leave space for reconfig data if at least
        // one of reconfig, SEI or ROI is present
        // Note: ROI is present when enabled, so use encode config flag instead
        //       of frame's roi_len as it can be 0 indicating a 0'd ROI map
        //       setting !
        if (p_in_frame->reconf_len || p_in_frame->sei_total_len ||
            (api_params->roi_demo_mode &&
             api_params->cfg_enc_params.roi_enable)) {
            p_in_frame->extra_data_len += sizeof(ni_encoder_change_params_t);
        }

        if (!is_hwframe) {
            ni_encoder_sw_frame_buffer_alloc(api_params->cfg_enc_params.planar,
                                             p_in_frame, input_video_width, input_video_height, dst_stride,
                                             p_enc_ctx->codec_format == NI_CODEC_FORMAT_H264,
                                             (int)(p_in_frame->extra_data_len), alignment_2pass_wa);
            if (!p_in_frame->p_data[0]) {
                fprintf(stderr, "Error: cannot allocate YUV frame buffer!");
                return -1;
            }
        } else {
            ni_frame_buffer_alloc_hwenc(p_in_frame, input_video_width,
                                        input_video_height,
                                        (int)(p_in_frame->extra_data_len));
            if (!p_in_frame->p_data[3]) {
                fprintf(stderr, "Error: cannot allocate YUV frame buffer!");
                return -1;
            }
        }

        ni_log(NI_LOG_DEBUG,
               "p_dst alloc linesize = %d/%d/%d  src height=%d  "
               "dst height aligned = %d/%d/%d force_key_frame=%d, "
               "extra_data_len=%u"
               " sei_size=%u (hdr_content_light_level %u hdr_mastering_display_"
               "color_vol %u hdr10+ %u hrd %d) reconf_size=%u roi_size=%u "
               "force_pic_qp=%u udu_sei_size=%u "
               "use_cur_src_as_long_term_pic %u use_long_term_ref %u\n",
               dst_stride[0], dst_stride[1], dst_stride[2], input_video_height,
               dst_height_aligned[0], dst_height_aligned[1], dst_height_aligned[2],
               p_in_frame->force_key_frame, p_in_frame->extra_data_len,
               p_in_frame->sei_total_len,
               p_in_frame->sei_hdr_content_light_level_info_len,
               p_in_frame->sei_hdr_mastering_display_color_vol_len,
               p_in_frame->sei_hdr_plus_len, 0, /* hrd is 0 size for now */
               p_in_frame->reconf_len, p_in_frame->roi_len,
               p_in_frame->force_pic_qp, p_in_frame->sei_user_data_unreg_len,
               p_in_frame->use_cur_src_as_long_term_pic,
               p_in_frame->use_long_term_ref);

        uint8_t *p_src[NI_MAX_NUM_DATA_POINTERS];
        int src_stride[NI_MAX_NUM_DATA_POINTERS];
        int src_height[NI_MAX_NUM_DATA_POINTERS];

        src_height[0] = p_dec_out_data->data.frame.video_height;
        src_height[1] = src_height[2] = src_height[0] / 2;
        src_height[3] = 0;

        src_stride[0] =
                (int)(p_dec_out_data->data.frame.data_len[0]) / src_height[0];
        src_stride[1] =
                (int)(p_dec_out_data->data.frame.data_len[1]) / src_height[1];
        src_stride[2] = src_stride[1];
        if (is_semiplanar) {
            src_height[2] = 0;
            src_stride[2] = 0;
        }
        src_stride[3] = 0;

        p_src[0] = p_dec_out_data->data.frame.p_data[0];
        p_src[1] = p_dec_out_data->data.frame.p_data[1];
        p_src[2] = p_dec_out_data->data.frame.p_data[2];
        p_src[3] = p_dec_out_data->data.frame.p_data[3];

        if (!is_hwframe) {
            // YUV part of the encoder input data layout
            ni_copy_hw_yuv420p(
                    (uint8_t **)(p_in_frame->p_data), p_src, input_video_width,
                    input_video_height, p_enc_ctx->bit_depth_factor, is_semiplanar,
                    ((ni_xcoder_params_t *)p_enc_ctx->p_session_config)
                            ->cfg_enc_params.conf_win_right,
                    dst_stride, dst_height_aligned, src_stride, src_height);
        } else {
            ni_copy_hw_descriptors((uint8_t **)(p_in_frame->p_data), p_src);
        }
        // auxiliary data part of the encoder input data layout
        ni_enc_copy_aux_data(p_enc_ctx, p_in_frame,
                             &(p_dec_out_data->data.frame),
                             p_enc_ctx->codec_format, mdcv_data, cll_data,
                             cc_data, udu_data, hdrp_data, is_hwframe,
                             is_semiplanar);
    }

    p_in_frame->video_width = input_video_width;
    p_in_frame->video_height = input_video_height;

    p_in_frame->start_of_stream = 0;
    if (!p_device_state->enc_sos_sent) {
        p_device_state->enc_sos_sent = 1;
        p_in_frame->start_of_stream = 1;
    }
    // p_in_frame->end_of_stream = 0;

    send_frame:
    oneSent = (int)(p_in_frame->data_len[0] + p_in_frame->data_len[1] +
                    p_in_frame->data_len[2] + p_in_frame->data_len[3]);

    if (oneSent > 0 || p_in_frame->end_of_stream) {
        oneSent = ni_device_session_write(p_enc_ctx, p_to_send,
                                          NI_DEVICE_TYPE_ENCODER);
        p_in_frame->end_of_stream = 0;
    } else {
        goto end_encoder_send_data2;
    }

    if (oneSent < 0) {
        fprintf(stderr, "Error: encoder_send_data2\n");
        p_device_state->enc_resend = 1;
        return -1;
    } else if (oneSent == 0) {
        if (p_device_state->enc_eos_sent == 0 && p_enc_ctx->ready_to_close) {
            p_device_state->enc_resend = 0;
            p_device_state->enc_eos_sent = 1;
        } else {
            p_device_state->enc_resend = 1;
            return NI_TEST_RETCODE_EAGAIN;
        }
    } else {
        p_device_state->enc_resend = 0;

        if (p_enc_ctx->ready_to_close) {
            p_device_state->enc_eos_sent = 1;
        }
        ni_log(NI_LOG_DEBUG, "encoder_send_data2: success\n");
    }

    end_encoder_send_data2:
    return 0;
}

int main(int argc, char *argv[])
{
    static char input_filename[MAX_INPUT_FILES][FILE_NAME_LEN];
    static char output_filename[FILE_NAME_LEN];
    static char nb_filename[FILE_NAME_LEN];
    unsigned long total_bytes_sent = 0;
    unsigned long long total_bytes_received = 0;
    uint32_t number_of_packets_list = 0;
    int input_video_width;
    int input_video_height;
    int iXcoderGUID = 0;
    int arg_width = 0;
    int arg_height = 0;
    int arg_pic_width;
    int arg_pic_height;
    int dst_codec_format = NI_CODEC_FORMAT_H265;
    int send_fin_flag = 0;
    int receive_fin_flag = 0;
    int end_of_all_streams = 0;
    int ret;
    int print_time = 0;
    int bit_depth = 8;
    int src_codec_format = NI_CODEC_FORMAT_H264;
    int input_index = 0;
    int user_data_sei_passthru = 0;
    int rx_size;
    int pkt_size = 131040;
    ni_xcoder_params_t *p_dec_api_param;
    char decConfXcoderParams[2048] = {0};
    FILE *p_file = NULL;
    ni_xcoder_params_t api_param;
    ni_session_context_t dec_ctx = {0};
    ni_session_context_t enc_ctx = {0};
    ni_session_context_t ai_pre_ctx = {0};
    ni_session_data_io_t out_packet = {0};
    int input_file_fd[MAX_INPUT_FILES] = {-1};
    ni_session_data_io_t in_pkt = {0};
    ni_session_data_io_t out_frame = {0};
    ni_session_data_io_t ai_result_data = {0};
    ni_session_data_io_t enc_in_frame = {0};
    niFrameSurface1_t *p_hwframe = NULL;

    ni_h264_sps_t SPS = {0};   // input header SPS
    ni_h265_sps_t HEVC_SPS = {0};

    void *p_stream_info = NULL;
    device_state_t xcoder_state = {0};

    parse_arguments(argc, argv, input_filename, output_filename, nb_filename, &iXcoderGUID,
                    &arg_width, &arg_height, &arg_pic_width, &arg_pic_height, decConfXcoderParams);


    // Load input file into memory
    for(input_index = 0; input_index < MAX_INPUT_FILES; input_index++) {
        if (load_input_file(input_filename[input_index], &total_file_size[input_index]) < 0) {
            exit(-1);
        }
    }

    if (arg_width == 0 || arg_height == 0) {
        input_video_width = 1280;
        input_video_height = 720;
    } else {
        input_video_width = arg_width;
        input_video_height = arg_height;
    }

    p_dec_api_param = malloc(sizeof(*p_dec_api_param));
    if (!p_dec_api_param) {
        printf("Error: failed to allocate p_dec_api_param\n");
        exit(-1);
    }


    for(input_index = 0; input_index < MAX_INPUT_FILES; input_index++) {
        data_left_size[input_index] = total_file_size[input_index];
    }

    for(input_index = 0; input_index < MAX_INPUT_FILES; input_index++) {

#ifdef _WIN32
        input_file_fd[input_index] = open(input_filename[input_index], O_RDONLY | O_BINARY);
#else
        input_file_fd[input_index] = open(input_filename[input_index], O_RDONLY);
#endif

        if (input_file_fd[input_index] < 0) {
            fprintf(stderr, "Error: can not open input file %s\n", input_filename[input_index]);
            goto end;
        }
    }

    if (!(g_file_cache[0] = malloc(total_file_size[0]))) {
        fprintf(stderr,
                "Error: input file size %llu exceeding max malloc, quit\n",
                (unsigned long long)total_file_size[0]);
        goto end;
    }

    //load decode file
    int read_size = read_next_chunk_from_file(input_file_fd[0], g_file_cache[0], total_file_size[0], 0);
    if (read_size != total_file_size[0]) {
        fprintf(stderr, "Error: reading file, quit! left-to-read %llu\n",
                (unsigned long long)total_file_size[0] - read_size);
        fprintf(stderr, "Error: input file read error\n");
        goto end;
    }
    g_curr_cache_pos = g_file_cache[0];

    // Create output file
    if (strcmp(output_filename, "null") != 0) {
        p_file = fopen(output_filename, "wb");
        if (p_file == NULL) {
            fprintf(stderr, "Error: cannot open %s\n", output_filename);
            goto end;
        }
    }

    printf("SUCCESS: Opened output file: %s\n", output_filename);

    if (ni_decoder_init_default_params(p_dec_api_param, 30, 1, 200000, arg_width,
                                       arg_height) < 0) {
        fprintf(stderr, "Error: decoder p_config set up error\n");
        return -1;
    }

    if (ni_device_session_context_init(&dec_ctx) < 0) {
        fprintf(stderr, "Error: init decoder context error\n");
        return -1;
    }

    if (ni_device_session_context_init(&enc_ctx) < 0) {
        fprintf(stderr, "Error: init encoder context error\n");
        return -1;
    }
    enc_ctx.codec_format = dst_codec_format;

    if (ni_device_session_context_init(&ai_pre_ctx) < 0) {
        fprintf(stderr, "Error: init ai context error\n");
        return -1;
    }

    ni_gettimeofday(&start_time, NULL);
    ni_gettimeofday(&previous_time, NULL);
    ni_gettimeofday(&current_time, NULL);
    start_timestamp = previous_timestamp = current_timestamp = time(NULL);

    printf("P2P Encoding resolution: %dx%d\n", input_video_width,
           input_video_height);


    dec_ctx.p_session_config = NULL;
    dec_ctx.session_id = NI_INVALID_SESSION_ID;
    dec_ctx.codec_format = src_codec_format;

    // user_data_sei_passthru disabled by default
    if ((NI_CODEC_FORMAT_H264 == src_codec_format) || (NI_CODEC_FORMAT_H265 == src_codec_format))
        dec_ctx.enable_user_data_sei_passthru = user_data_sei_passthru;

    // assign the card GUID in the decoder context and let session open
    // take care of the rest
    dec_ctx.device_handle = dec_ctx.blk_io_handle =
            NI_INVALID_DEVICE_HANDLE;
    dec_ctx.hw_id = iXcoderGUID;

    dec_ctx.p_session_config = p_dec_api_param;
    // default: little endian
    dec_ctx.src_bit_depth = bit_depth;
    dec_ctx.src_endian = NI_FRAME_LITTLE_ENDIAN;
    dec_ctx.bit_depth_factor = 1;
    if (10 == dec_ctx.src_bit_depth) {
        dec_ctx.bit_depth_factor = 2;
    }

    // check and set ni_decoder_params from --xcoder-params
    if (ni_retrieve_decoder_params(decConfXcoderParams, p_dec_api_param,
                                   &dec_ctx)) {
        fprintf(stderr, "Error: encoder p_config parsing error\n");
        return -1;
    }

    // Configure the encoder parameter structure. We'll use some basic
    // defaults: 30 fps, 200000 bps CBR encoding, AVC or HEVC encoding
    if (ni_encoder_init_default_params(&api_param, 30, 1, 200000, arg_width,
                                       arg_height, enc_ctx.codec_format) < 0) {
        fprintf(stderr, "Error: encoder init default set up error\n");
        return -1;
    }

    // Decode, use all the parameters specified by user
    ret = decoder_open_session(&dec_ctx, iXcoderGUID, p_dec_api_param);
    if (ret != 0) {
        goto end;
    }

    ni_roi_network_t roi_network;
    if(ai_open_session(&ai_pre_ctx, iXcoderGUID, input_video_width, input_video_height, 5,
                       &roi_network.raw, nb_filename)) {
        ni_device_session_close(&dec_ctx, 1, NI_DEVICE_TYPE_DECODER);
        goto end;
    }

    if (src_codec_format == NI_CODEC_FORMAT_H264)
        p_stream_info = &SPS;
    else if (src_codec_format == NI_CODEC_FORMAT_H265)
        p_stream_info = &HEVC_SPS;

    while (!end_of_all_streams && (send_fin_flag == 0 || receive_fin_flag == 0)) {
        (void)ni_gettimeofday(&current_time, NULL);
        print_time = need_time_print(&current_time, &previous_time);

        decode_send:
        send_fin_flag = decoder_send_data(&dec_ctx, &in_pkt,
                                          input_video_width,
                                          input_video_height, pkt_size,
                                          &total_bytes_sent, print_time,
                                          &xcoder_state, p_stream_info);
        if (send_fin_flag < 0) {
            break;
        }

        // YUV Receiving: not writing to file
        receive_fin_flag = decoder_receive_data(
                &dec_ctx, &out_frame, input_video_width,
                input_video_height, NULL, &total_bytes_received,
                print_time, 0, &xcoder_state, &rx_size);
        if (print_time)
        {
            previous_time = current_time;
        }

        if (receive_fin_flag == NI_TEST_RETCODE_EAGAIN) {
            ni_log(NI_LOG_DEBUG,
                   "no decoder output, jump to encoder receive!\n");
            if (!dec_ctx.hw_action) {
                ni_decoder_frame_buffer_free(&(out_frame.data.frame));
            } else {
                ni_frame_buffer_free(&(out_frame.data.frame));
            }

            // use first encode config low delay flag for call flow
            if (enc_ctx.session_id != NI_INVALID_SESSION_ID) {
                ni_log(NI_LOG_DEBUG,
                       "no decoder output, jump to encoder receive!\n");
                goto encode_recv;
            } else {
                ni_log(NI_LOG_DEBUG,
                       "no decoder output, encode low_delay, jump to "
                       "decoder send!\n");
                goto decode_send;
            }
        }

        if (receive_fin_flag != NI_TEST_RETCODE_END_OF_STREAM) {
            frame_copy_props(&ai_result_data, &out_frame);
            // allocate a ni_frame_t structure on the host PC
            ret = ni_frame_buffer_alloc_hwenc(&ai_result_data.data.frame,
                                              input_video_width,
                                              input_video_height, 0);
            if (ret != 0) {
                return -1;
            }

            //Allocate output frame for AI engine.
            ret = ni_device_alloc_frame(
                    &ai_pre_ctx, input_video_width, input_video_height,
                    GC620_I420, NI_SCALER_FLAG_IO, 0, 0,
                    0, 0, 0, -1, NI_DEVICE_TYPE_AI);

            niFrameSurface1_t* dec_frame_surface = (niFrameSurface1_t *)out_frame.data.frame.p_data[3];
            /* set input buffer */
            ret = ni_device_alloc_frame(&ai_pre_ctx, 0, 0, 0, 0, 0, 0, 0, 0,
                                           dec_frame_surface->ui32nodeAddress,
                                           dec_frame_surface->ui16FrameIdx,
                                           NI_DEVICE_TYPE_AI);
            // Retrieve hardware frame info from AI engine and put it in the ni_frame_t structure.
            ret = ni_device_session_read_hwdesc(&ai_pre_ctx, &ai_result_data, NI_DEVICE_TYPE_AI);
            if(ret != 0){
                printf("Failed to get ai process frame %d\n", ret);
                return -1;
            }
            ni_hwframe_buffer_recycle2((niFrameSurface1_t *)out_frame.data.frame.p_data[3]);
        } else {
            ai_result_data.data.frame.end_of_stream = 1;
        }
        if(enc_ctx.session_id == NI_INVALID_SESSION_ID) {
            // Open the encoder session with given parameters
            ret = encoder_open_session(&enc_ctx, dst_codec_format, iXcoderGUID,
                                       &api_param, arg_width, arg_height,
                                       &ai_result_data.data.frame);
        }
        // YUV Sending
        send_fin_flag = encoder_send_data2(&enc_ctx,
                                           &ai_result_data,
                                           &enc_in_frame,
                                           input_video_width,
                                           input_video_height,
                                           &total_bytes_sent,
                                           &xcoder_state);
        if (send_fin_flag < 0) { //Error
            break;
        }
        p_hwframe = (niFrameSurface1_t *)ai_result_data.data.frame.p_data[3];
        ni_hw_frame_ref(p_hwframe);
        ni_frame_wipe_aux_data(&ai_result_data.data.frame);

        encode_recv:
        receive_fin_flag = encoder_receive(&enc_ctx,
                                           &out_frame, &out_packet,
                                           input_video_width,
                                           input_video_height,
                                           &number_of_packets_list,
                                           1, &p_file,
                                           &total_bytes_received,
                                           print_time,
                                           &xcoder_state);
        if (!xcoder_state.enc_eos_received) {
            ni_log(NI_LOG_DEBUG, "enc continues to read!\n");
            end_of_all_streams = 0;
        } else {
            ni_log(NI_LOG_INFO, "enc eos !\n");
            end_of_all_streams = 1;
        }
    }

    ni_device_session_close(&dec_ctx, 1, NI_DEVICE_TYPE_DECODER);
    ni_device_session_close(&ai_pre_ctx, 1, NI_DEVICE_TYPE_AI);
    ni_device_session_close(&enc_ctx, 1, NI_DEVICE_TYPE_ENCODER);

    ni_device_session_context_clear(&enc_ctx);
    ni_device_session_context_clear(&dec_ctx);
    ni_device_session_context_clear(&ai_pre_ctx);

    ni_packet_buffer_free(&(out_packet.data.packet));

    end:
    for(input_index = 0; input_index < MAX_INPUT_FILES; input_index++) {
        close(input_file_fd[input_index]);
    }

    if (p_file) {
        fclose(p_file);
    }

    printf("All done\n");

    return 0;
}
