/*
 * Copyright (c) 2010 Nicolas George
 * Copyright (c) 2011 Stefano Sabatini
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * @file
 * API example for defog
 * @example ni_defog.c
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>

#include <sys/time.h>

#include <libavcodec/avcodec.h>
#include <libavformat/avformat.h>
#include <libavfilter/buffersink.h>
#include <libavfilter/buffersrc.h>
#include <libswscale/swscale.h>
#include <libavutil/imgutils.h>
#include <libavutil/opt.h>
#include <libavutil/time.h>
#include <pthread.h>
#include <stdint.h>
#include <float.h>

#include "ni_list_head.h"
#include "netint_network.h"
#include "defog_network.h"

#include "ni_log.h"

#define NI_TRANSCODE_FRAME_NUM 3
#define NI_SAME_CENTER_THRESH 2
#define NI_SAME_BORDER_THRESH 8
#define IS_FFMPEG_43_AND_ABOVE                                                \
    ((LIBAVFILTER_VERSION_MAJOR > 7) ||                                       \
     (LIBAVFILTER_VERSION_MAJOR == 7 && LIBAVFILTER_VERSION_MINOR >= 85))

bool use_hwframe = false;
FILE *outfile = NULL;
AVFrame *frame;
static AVFormatContext *fmt_ctx;
static AVCodecContext *dec_ctx;
static AVCodecContext *enc_ctx;
static int video_stream_index = -1;
int frame_num = 0, end = 0, encode_num = 0;
int64_t timer_start;
bool stop_fps = false;
int init_filters = 0;
int init_encoder = 0;

typedef struct _filter_worker
{
    AVFilterContext *buffersink_ctx;
    AVFilterContext *buffersrc_ctx;
    AVFilterGraph *filter_graph;
}filter_worker;

struct frame_req {
    struct ni_list_head list;
    AVFrame avframe;
    int eof;
};

struct frame_req *pframes = NULL;

// Only count previous 1000 frames time statistics
#define MAX_TIMES_NUMBER 1000

struct codec_worker {
    const char *decoder_name;
    const char *input_file;
    const char *output_file;
    const char *decoder_params;
    const char *encoder_name;
    const char *encoder_params;
    int devid;
    int bitrate;

    pthread_t pthid;
    pthread_t pthtimeid;

    int infer_exit;

    struct ni_list_head frm_list;
    pthread_mutex_t frm_lock;
    pthread_cond_t  frm_cond;

    struct ni_list_head free_list;
    pthread_mutex_t free_lock;
    pthread_cond_t  free_cond;
};

struct codec_worker *codec_workers = NULL;

struct defog_network {
    NiNetworkContext *dark_chnl_network;
    NiNetworkContext *dehaze_network;
    DarkChnlModelCtx *dark_chnl_model;
    DehazeModelCtx   *dehaze_model;
};

struct defog_network *defog_network;

NI_LIST_HEAD(network_frame_free_list);
//NI_LIST_HEAD(network_frame_busy_list);

struct network_frame {
    NiNetworkFrame dark_frame;
    NiNetworkFrame dehaze_frame;
    struct ni_list_head list;
    struct frame_req *frame_req;
    float atm;
};

struct network_frame *defog_network_frames;

static int init_network_frames(struct defog_network *defog_network)
{
    int i, ret;

    ni_init_list_head(&network_frame_free_list);
//    ni_init_list_head(&network_frame_busy_list);

    defog_network_frames = (struct network_frame *)calloc(NUM_NETWORK_FRAME,
            sizeof(struct network_frame));
    if (defog_network_frames == NULL) {
        return NIERROR(ENOMEM);
    }

    for (i = 0; i < NUM_NETWORK_FRAME; i++) {
        //dark frame
        NiNetworkFrame *frame = &defog_network_frames[i].dark_frame;

        ret = ni_ai_frame_buffer_alloc(&frame->api_frame.data.frame,
                &defog_network->dark_chnl_network->network_data);
        if (ret != NI_RETCODE_SUCCESS) {
            av_log(NULL, AV_LOG_ERROR, "failed to allocate dark frame\n");
            return NIERROR(ENOMEM);
        }

        ret = ni_ai_packet_buffer_alloc(&frame->api_packet.data.packet,
                &defog_network->dark_chnl_network->network_data);
        if (ret != NI_RETCODE_SUCCESS) {
            av_log(NULL, AV_LOG_ERROR, "failed to allocate dark packet\n");
            return NIERROR(ENOMEM);
        }

        //dehaze frame
        frame = &defog_network_frames[i].dehaze_frame;

        ret = ni_ai_frame_buffer_alloc(&frame->api_frame.data.frame,
                &defog_network->dehaze_network->network_data);
        if (ret != NI_RETCODE_SUCCESS) {
            av_log(NULL, AV_LOG_ERROR, "failed to allocate dehaze frame\n");
            return NIERROR(ENOMEM);
        }

        ret = ni_ai_packet_buffer_alloc(&frame->api_packet.data.packet,
                &defog_network->dehaze_network->network_data);
        if (ret != NI_RETCODE_SUCCESS) {
            av_log(NULL, AV_LOG_ERROR, "failed to allocate dehaze packet\n");
            return NIERROR(ENOMEM);
        }

        ni_init_list_head(&defog_network_frames[i].list);
        ni_list_add_tail(&defog_network_frames[i].list, &network_frame_free_list);
    }

    return 0;
}

static void free_network_frames()
{
    int i;

//    if (!ni_list_empty(&network_frame_busy_list)) {
//        av_log(NULL, AV_LOG_WARNING, "Network frames are still in use\n");
//    }

    for (i = 0; i < NUM_NETWORK_FRAME; i++) {
        //dark frame
        NiNetworkFrame *frame = &defog_network_frames[i].dark_frame;
        ni_frame_buffer_free(&frame->api_frame.data.frame);
        ni_packet_buffer_free(&frame->api_packet.data.packet);

        //dehaze frame
        frame = &defog_network_frames[i].dehaze_frame;
        ni_frame_buffer_free(&frame->api_frame.data.frame);
        ni_packet_buffer_free(&frame->api_packet.data.packet);
    }

    if (defog_network_frames != NULL) {
        free(defog_network_frames);
        defog_network_frames = NULL;
    }
}

static void destroy_dark_chnl_network(struct defog_network *network)
{
    if (network) {
        if (network->dehaze_model) {
            dehaze_model.destroy_model(network->dehaze_model);
            free(network->dehaze_model);
            network->dehaze_model = NULL;
        }
    }
}

static void destroy_dehaze_network(struct defog_network *network)
{
    if (network) {
        if (network->dark_chnl_model) {
            dark_chnl_model.destroy_model(network->dark_chnl_model);
            free(network->dark_chnl_model);
            network->dark_chnl_model = NULL;
        }
    }
}

static void destroy_defog_network(struct defog_network *defog_network)
{
    destroy_dark_chnl_network(defog_network);
    destroy_dehaze_network(defog_network);
}

static int create_dark_chnl_network(struct defog_network *defog_network, const char *nbg_file)
{
    int ret = 0;
    DarkChnlModelCtx *model_ctx = NULL;

    ret = ni_alloc_network_context(&defog_network->dark_chnl_network, use_hwframe,
            0, 30, 0, 0, 0, nbg_file);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate network context\n");
        goto out;
    }

    ni_network_data_t *network_data = &defog_network->dark_chnl_network->network_data;

    model_ctx = (DarkChnlModelCtx *)calloc(1, sizeof(DarkChnlModelCtx));
    if (model_ctx == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate dark chnl model\n");
        ret = NIERROR(ENOMEM);
        goto out;
    }
    defog_network->dark_chnl_model = model_ctx;

    ret = dark_chnl_model.create_model(model_ctx, network_data);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to initialize dark chnl model\n");
        goto out;
    }

    return ret;

out:
    if (model_ctx) {
      free(model_ctx);
      defog_network->dark_chnl_model = NULL;
    }
    ni_cleanup_network_context(defog_network->dark_chnl_network, use_hwframe);

    return ret;
}

static int create_dehaze_network(struct defog_network *defog_network, const char *nbg_file)
{
    int ret = 0;
    DehazeModelCtx *model_ctx = NULL;

    ret = ni_alloc_network_context(&defog_network->dehaze_network, use_hwframe,
            0, 30, 0, 0, 0, nbg_file);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate network context\n");
        goto out;
    }

    ni_network_data_t *network_data = &defog_network->dehaze_network->network_data;

    model_ctx = (DehazeModelCtx *)calloc(1, sizeof(DehazeModelCtx));
    if (model_ctx == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate dehaze model\n");
        ret = NIERROR(ENOMEM);
        goto out;
    }
    defog_network->dehaze_model = model_ctx;

    ret = dehaze_model.create_model(model_ctx, network_data);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to initialize dehaze model\n");
        goto out;
    }

    return ret;

out:
    if (model_ctx) {
      free(model_ctx);
      defog_network->dehaze_model = NULL;
    }
    ni_cleanup_network_context(defog_network->dehaze_network, use_hwframe);

    return ret;
}

static int init_network_struct(
        const char *dark_chnl_nbg_file,
        const char *dehaze_nbg_file)
{
    int ret;
    defog_network = (struct defog_network *)calloc(1, sizeof(struct defog_network));
    if (defog_network == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate defog network structure\n");
        return -1;
    }

    ret = create_dark_chnl_network(defog_network, dark_chnl_nbg_file);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to create dark chnl network\n");
        goto fail_out;
    }

    ret = create_dehaze_network(defog_network, dehaze_nbg_file);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to create dehaze network\n");
        goto fail_out;
    }

    ret = init_network_frames(defog_network);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to initialize network frames\n");
        goto fail_out;
    }

    return 0;
fail_out:
    destroy_defog_network(defog_network);
    free_network_frames();
    return -1;
}

static void free_network_struct()
{
    if (defog_network) {
        destroy_defog_network(defog_network);
        ni_cleanup_network_context(defog_network->dark_chnl_network, use_hwframe);
        ni_cleanup_network_context(defog_network->dehaze_network, use_hwframe);
        free(defog_network);
        defog_network = NULL;
    }

    free_network_frames();
}

static int open_input_file()
{
    int ret;
    AVCodec *dec;
    if ((ret = avformat_open_input(&fmt_ctx, codec_workers->input_file, NULL, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot open input file\n");
        return ret;
    }

    if ((ret = avformat_find_stream_info(fmt_ctx, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot find stream information\n");
        return ret;
    }

    dec = avcodec_find_decoder_by_name(codec_workers->decoder_name);
    if (dec) {
                if (fmt_ctx->streams[0]->codecpar->codec_id != dec->id) {
                    av_log(NULL, AV_LOG_ERROR, "codec does not match with stream id\n");
                    return AVERROR_DECODER_NOT_FOUND;
                }
            }

    video_stream_index = 0;

    /* create decoding context */
    dec_ctx = avcodec_alloc_context3(dec);
    if (!dec_ctx)
        return AVERROR(ENOMEM);

    avcodec_parameters_to_context(dec_ctx, fmt_ctx->streams[0]->codecpar);

    char str_devid[4] = {0};
    snprintf(str_devid, sizeof(str_devid), "%d", codec_workers->devid);
    av_opt_set(dec_ctx->priv_data, "dec", str_devid, 0);
    if(codec_workers->decoder_params)
        av_opt_set(dec_ctx->priv_data, "xcoder-params",codec_workers->decoder_params, 0);

    if (dec_ctx->codec_type == AVMEDIA_TYPE_VIDEO)
            dec_ctx->framerate = av_guess_frame_rate(fmt_ctx, fmt_ctx->streams[0], NULL);

    ret = avcodec_parameters_from_context(fmt_ctx->streams[0]->codecpar, dec_ctx);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to copy decoder parameters from codec context for stream #");
        return ret;
    }

    /* init the video decoder */
    if ((ret = avcodec_open2(dec_ctx, dec, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot open video decoder\n");
        return ret;
    }
    return 0;
}

static int open_output_file(AVFrame *frame)
{
    int ret = 0;
    AVCodec *enc;

    // Find video encoder codec selected
    enc = avcodec_find_encoder_by_name(codec_workers->encoder_name);
    if (!enc)
    {
        av_log(NULL, AV_LOG_ERROR,"Codec '%s' not found\n", codec_workers->encoder_name);
        return AVERROR_ENCODER_NOT_FOUND;
    }

    // Allocate codec context for encoding
    enc_ctx = avcodec_alloc_context3(enc);
    if (!enc_ctx)
    {
        av_log(NULL, AV_LOG_ERROR, "Could not allocate video codec context\n");
        return AVERROR(ENOMEM);
    }

    if (codec_workers->bitrate)
    {
        enc_ctx->bit_rate = codec_workers->bitrate;
    }
    enc_ctx->height = frame->height;
    enc_ctx->width = frame->width;
    enc_ctx->framerate = dec_ctx->framerate;
    enc_ctx->sample_aspect_ratio = dec_ctx->sample_aspect_ratio;
    enc_ctx->pix_fmt = frame->format;
    if (frame->hw_frames_ctx)
    {
        AVHWFramesContext* pAVHFWCtx = (AVHWFramesContext *)frame->hw_frames_ctx->data;
        enc_ctx->sw_pix_fmt = pAVHFWCtx->sw_format;
    }

    /* video time_base can be set to whatever is handy and supported by encoder */
    enc_ctx->time_base = av_inv_q(dec_ctx->framerate);

    av_log(NULL, AV_LOG_INFO, "dec_ctx->sw_pix_fmt %d, pix_fmt %d\n",
            dec_ctx->sw_pix_fmt, dec_ctx->pix_fmt);

    if ((enc->id == AV_CODEC_ID_H264) || (enc->id == AV_CODEC_ID_H265))
    {
        if(codec_workers->encoder_params)
        {
            av_opt_set(enc_ctx->priv_data, "xcoder-params", codec_workers->encoder_params, 0);
        }
        if(codec_workers->devid >= 0)
        {
            char str_devid[4] = {0};
            snprintf(str_devid, sizeof(str_devid), "%d", codec_workers->devid);
            av_opt_set(enc_ctx->priv_data, "enc", str_devid, 0);
        }
    }
    else
    {
        av_log(NULL, AV_LOG_ERROR, "codec id %d not supported.\n", enc->id);
        ret = -1;
        return ret;
    }

    // Open encoder
    ret = avcodec_open2(enc_ctx, enc, NULL);
    if (ret < 0)
    {
        av_log(NULL, AV_LOG_ERROR, "Could not open enc\n");
        return ret;
    }
    return 0;
}

static int encode_frame(AVFrame *frame, AVPacket *pkt, FILE *outfile)
{
    int ret;
    if (!init_encoder && frame)
    {
        if((ret = open_output_file(frame)) < 0)
        {
            av_log(NULL, AV_LOG_ERROR, "init encoder failed ret = %d\n", ret);
            return ret;
        }
        init_encoder = 1;
    }

    int attempt_cnt = 1;
    int re_send = 0;
    static int send_cnt = 0;
    static int recv_cnt = 0;

    // Send frame to encoder. If frame=NULL, this signals end of stream
    if (frame)
    {
        frame->pict_type = AV_PICTURE_TYPE_NONE;
        av_log(NULL, AV_LOG_DEBUG, "Send %d frame %3"PRId64", type=%d\n", ++send_cnt, frame->pts, frame->pict_type);
    } else
    {
        av_log(NULL, AV_LOG_DEBUG,"Send done, flush encoder\n");
    }

resend:
    ret = avcodec_send_frame(enc_ctx, frame);
    // Encoder is too busy to receive frame, try sending frame again
    if (ret == AVERROR(EAGAIN))
    {
        av_log(NULL, AV_LOG_DEBUG, "Resending frame. Try #%d.\n", ++attempt_cnt);
        ret = 0;
        re_send = 1;
    }
    // Frame send issue occured
    else if (ret < 0)
    {
        av_log(NULL, AV_LOG_ERROR, "Error sending frame for encoding. RC=%d\n", AVUNERROR(ret));
        return ret;
    }

    /* Run this loop if a frame was sucessfully sent during avcodec_send_frame() OR
        no frame was sent because this function is in flushing mode OR
        a frame was sucessfully received by avcodec_receive_packet() */
    while (ret >= 0 || ! frame )
    {
        av_log(NULL, AV_LOG_DEBUG, "Receiving packet\n");
        ret = avcodec_receive_packet(enc_ctx, pkt);
        if (frame == NULL)
            av_log(NULL, AV_LOG_DEBUG, "Draining packet ret=%x, pkt size=%5d\n", ret, pkt->size);

        // No packets in encoder output buffer but not end of stream either
        if (ret == AVERROR(EAGAIN))
        {
            // If function not in flushing mode
            if (frame)
            {
                if (re_send)
                {
                    re_send = 0;
                    goto resend;
                }
                return 0; // Exit this function to send next packet
            }
            else
            {
                continue; // Loop in this function to flush encoder
            }
        }
        // End of stream detected
        else if (ret == AVERROR_EOF)
        {
            av_log(NULL, AV_LOG_DEBUG, "Encoder EOF reached\n");
            return 0;
        }
        // Packet receive error occured
        else if (ret < 0)
        {
            av_log(NULL, AV_LOG_ERROR, "Error during encoding\n");
            return ret;
        }
        // Packet sucessfully received
        av_log(NULL, AV_LOG_DEBUG,"Received packet %d, pts=%"PRId64"\n", ++recv_cnt, pkt->pts);
        encode_num++;

        fwrite(pkt->data, 1, pkt->size, outfile);
        av_packet_unref(pkt);
    }
    return 0;
}

static int init_filter(const char *filters_desc, filter_worker *f, bool hwmode)
{
    char args[512] = { 0 };
    char name[32] = { 0 };
    int i, ret = 0;
    AVFilterInOut *inputs, *outputs, *cur;
    AVBufferSrcParameters *par;
    AVRational time_base = fmt_ctx->streams[video_stream_index]->time_base;
    enum AVPixelFormat input_fmt;
    if (hwmode) {
        input_fmt = dec_ctx->pix_fmt;
    } else {
        input_fmt = dec_ctx->sw_pix_fmt;
    }

    f->filter_graph = avfilter_graph_alloc();
    if (!f->filter_graph) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate filter graph\n");
        ret = AVERROR(ENOMEM);
        goto end;
    }

    ret = avfilter_graph_parse2(f->filter_graph, filters_desc, &inputs, &outputs);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to parse graph\n");
        goto end;
    }

    // link input
    cur = inputs, i = 0;
    snprintf(name, sizeof(name), "in_%d", i);
    snprintf(args, sizeof(args),
            "video_size=%dx%d:pix_fmt=%d:time_base=%d/%d:pixel_aspect=%d/%d:frame_rate=%d/%d",
            dec_ctx->width, dec_ctx->height, input_fmt, time_base.num, time_base.den,
            dec_ctx->sample_aspect_ratio.num, dec_ctx->sample_aspect_ratio.den,dec_ctx->framerate.num,dec_ctx->framerate.den);
    ret = avfilter_graph_create_filter(&f->buffersrc_ctx, avfilter_get_by_name("buffer"), name,
                                    args, NULL, f->filter_graph);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot create buffer source\n");
        goto end;
    }

    // decoder out=hw
    if (hwmode && dec_ctx->hw_frames_ctx != NULL) {
        av_log(NULL, AV_LOG_DEBUG, "hw mode filter\n");
        // Allocate a new AVBufferSrcParameters instance when decoder out=hw
        par = av_buffersrc_parameters_alloc();
        if (!par) {
            av_log(NULL, AV_LOG_ERROR, "cannot allocate hwdl buffersrc parameters\n");
            ret = AVERROR(ENOMEM);
            goto end;
        }
        memset(par, 0, sizeof(*par));
        // set format and hw_frames_ctx to AVBufferSrcParameters when out=hw
        par->format = AV_PIX_FMT_NONE;
        par->hw_frames_ctx = dec_ctx->hw_frames_ctx;
        av_log(NULL, AV_LOG_DEBUG, "hw_frames_ctx %p\n", par->hw_frames_ctx);
        // Initialize the buffersrc filter with the provided parameters
        ret = av_buffersrc_parameters_set(f->buffersrc_ctx, par);
        av_freep(&par);
        if (ret < 0)
            goto end;
    } else { // decoder out=sw
        av_log(NULL, AV_LOG_DEBUG, "sw mode filter\n");
    }

    ret = avfilter_link(f->buffersrc_ctx, 0, cur->filter_ctx, cur->pad_idx);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to link input filter\n");
        goto end;
    }

    cur = outputs, i = 0;
    snprintf(name, sizeof(name), "out_%d", i);
    ret = avfilter_graph_create_filter(&f->buffersink_ctx, avfilter_get_by_name("buffersink"),
                                        name, NULL, NULL, f->filter_graph);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to create output filter: %d\n", i);
        goto end;
    }

    // connect  dst (index i) pads to one of buffer sink (index 0) pad
    ret = avfilter_link(cur->filter_ctx, cur->pad_idx, f->buffersink_ctx, 0);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to link output filter: %d\n", i);
        goto end;
    }

    // configure and validate the filter graph
    ret = avfilter_graph_config(f->filter_graph, NULL);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "%s failed to config graph filter\n",
               __func__);
        goto end;
    } else {
        av_log(NULL, AV_LOG_DEBUG, "%s success config graph filter %s\n",
               __func__, filters_desc);
    }

end:
    avfilter_inout_free(&inputs);
    avfilter_inout_free(&outputs);
    return ret;
}

static void print_report(int is_last_report, int64_t timer_start,
        int64_t cur_time, unsigned long frame_number)
{
    static int64_t last_time = -1;
    float t;
    char buf[1024];
    float fps;

    if (!is_last_report) {
        if (last_time == -1) {
            last_time = cur_time;
            return;
        }
        if ((cur_time - last_time) < 500000)
            return;
        last_time = cur_time;
    }

    t = (cur_time - timer_start) / 1000000.0;

    fps = t > 1 ? frame_number / t : 0;

    const char end = is_last_report ? '\n' : '\r';

    buf[0] = '\0';
    snprintf(buf + strlen(buf), sizeof(buf) - strlen(buf), "frame=%5ld fps=%3.*f ",
            frame_number, (fps < 9.95), fps);
    pr_err("%s   %c", buf, end);
    fflush(stderr);
}

static void *printfps(void *data)
{
    while (1) {
        if (stop_fps == true) {
            break;
        }
        usleep(1000);
        print_report(0, timer_start, av_gettime_relative(), encode_num);
    }
    print_report(1, timer_start, av_gettime_relative(), encode_num);
    return (void *)0;
}

static int ni_recreate_ai_frame(ni_frame_t *ni_frame, AVFrame *frame)
{
    av_log(NULL, AV_LOG_DEBUG,
           "linesize %d/%d/%d, data %p/%p/%p, pixel %dx%d\n",
           frame->linesize[0], frame->linesize[1], frame->linesize[2],
           frame->data[0], frame->data[1], frame->data[2], frame->width,
           frame->height);
    memcpy(ni_frame->p_data[0], frame->data[0], frame->linesize[0] * frame->height);
    return 0;
}

static int generate_dark_model_frame(struct defog_network *defog_network, ni_session_data_io_t *ai_frame,
        AVFrame *avframe, bool hwframe)
{
    int ret = 0;
    NiNetworkContext *dark_chnl_network = defog_network->dark_chnl_network;
    ni_retcode_t retval;

    if (hwframe == false) {
        retval = ni_ai_frame_buffer_alloc(&ai_frame->data.frame, &dark_chnl_network->network_data);
        if (retval != NI_RETCODE_SUCCESS) {
            av_log(NULL, AV_LOG_ERROR, "cannot allocate sw ai frame buffer\n");
            return NIERROR(ENOMEM);
        }
        ret = ni_recreate_ai_frame(&ai_frame->data.frame, avframe);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "cannot recreate sw ai frame\n");
            return ret;
        }
    } else {
        ai_frame->data.frame.p_data[3] = avframe->data[3];
    }

    return ret;
}

static void free_frame_req(struct frame_req *frame_req)
{
    if (!frame_req) {
        return;
    }
    pthread_mutex_lock(&codec_workers->free_lock);
    if (ni_list_empty(&codec_workers->free_list)) {
        ni_list_add_tail(&frame_req->list, &codec_workers->free_list);
        pthread_cond_signal(&codec_workers->free_cond);
    } else {
        ni_list_add_tail(&frame_req->list, &codec_workers->free_list);
    }
    pthread_mutex_unlock(&codec_workers->free_lock);
}

static int infer_dark_chnl_frame(struct defog_network *defog_network,
        struct network_frame *network_frame)
{
    AVFrame *avframe = &network_frame->frame_req->avframe;
    return ni_set_network_input(defog_network->dark_chnl_network, use_hwframe,
            &network_frame->dark_frame.api_frame, NULL,
            avframe->width, avframe->height, &network_frame->dark_frame, true);
}

static float process_histogram(float *ds_dark, int *histo, int length)
{
    int i;
    float min = FLT_MAX;
    float max = FLT_MIN;
    for (i = 0; i < length; i++) {
        max = fmax(ds_dark[i], max);
        min = fmin(ds_dark[i], min);
    }

    float dis = (max - min) / 2000;

    for (i = 0; i < length; i++) {
        int index = (int)((ds_dark[i] - min) / dis);
        histo[index]++;
    }
    histo[1999] += histo[2000];

    int sum = 0;
    int threshold = (int)(length*(1 - 0.999));
    for (i = 1999; i >= 0; i--) {
        if (sum >= threshold) {
            break;
        }
        sum += histo[i];
    }
    float atm = min + (max - min) * i / 2000;
    if (atm < 0) {
        atm = 0.0;
    } else if (atm > 0.86) {
        atm = 0.86;
    }
    atm = atm - 16/255.0;

    return atm;
}

static void patch_dark_tensor(float *ds_dark, float omega, int length)
{
    int i = 0;
    for (i = 0; i < length; i++)
        ds_dark[i] = 1 - omega * ds_dark[i];
}

static int recv_dark_chnl_pkt(struct defog_network *defog_network,
        struct network_frame *network_frame)
{
    DarkChnlModelCtx *dark_chnl_model = defog_network->dark_chnl_model;
    NiNetworkContext *dark_chnl_network = defog_network->dark_chnl_network;
    int ret = ni_get_network_output(dark_chnl_network, use_hwframe, &network_frame->dark_frame,
            false, false, NULL);
    if (ret != 0) {
        return ret;
    }

    ret = ni_convert_to_tensor(dark_chnl_network, &network_frame->dark_frame,
            dark_chnl_model->ds_dark, 1);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to convert data\n");
        return ret;
    }

    int length = dark_chnl_model->r0 * dark_chnl_model->r1;
    memset(dark_chnl_model->histo, 0, dark_chnl_model->histo_size * sizeof(int));
    float atm = process_histogram(dark_chnl_model->ds_dark, dark_chnl_model->histo, length);
    network_frame->atm = 0.859 * 255 * atm;
    patch_dark_tensor(dark_chnl_model->ds_dark, 0.75 / atm, length);
    return 0;
}

static int generate_dehaze_frame(struct defog_network *defog_network,
        struct network_frame *network_frame)
{
    NiNetworkContext *dehaze_network = defog_network->dehaze_network;
    DehazeModelCtx *dehaze_model = defog_network->dehaze_model;
    DarkChnlModelCtx *dark_chnl_model = defog_network->dark_chnl_model;
    ni_network_data_t *network_data = &dehaze_network->network_data;

    ni_frame_t *frame = &network_frame->dehaze_frame.api_frame.data.frame;

    ni_packet_t *packet = &network_frame->dark_frame.api_packet.data.packet;
    memcpy(frame->p_data[0] + network_data->inset[0].offset, packet->p_buffer,
            dehaze_model->r0 * dehaze_model->r1);

    if (ni_network_convert_tensor_to_data(frame->p_data[0] + network_data->inset[1].offset,
            dehaze_model->r0 * dehaze_model->r1, dark_chnl_model->ds_dark,
            dark_chnl_model->ds_dark_bytes, &network_data->linfo.in_param[1])
            != NI_RETCODE_SUCCESS) {
        return -1;
    }

    return 0;
}

static int infer_dehaze_frame(struct defog_network *defog_network,
        struct network_frame *network_frame)
{
    DehazeModelCtx *ctx = defog_network->dehaze_model;
    return ni_set_network_input(defog_network->dehaze_network, use_hwframe,
            &network_frame->dehaze_frame.api_frame, NULL,
            ctx->r0, ctx->r1, &network_frame->dehaze_frame, true);
}

static int do_clamp(int value, int min, int max)
{
    if (value < min) {
        return min;
    }
    if (value > max) {
        return max;
    }
    return value;
}

static void process_dehaze(struct defog_network *defog_network,
        struct network_frame *network_frame)
{
    DehazeModelCtx *dehaze_model = defog_network->dehaze_model;

    int i, j;
    struct frame_req *frame_req = network_frame->frame_req;
    AVFrame *avframe = &frame_req->avframe;
    uint8_t *y_data = avframe->data[0];
    uint8_t *u_data = avframe->data[1];
    uint8_t *v_data = avframe->data[2];
    float *dehaze = dehaze_model->dehaze;
    long length = avframe->width * avframe->height;
    float atm_sum = 16 + network_frame->atm;

    //Recover Y
    for (i = 0; i < length; i++) {
        float value = atm_sum + (y_data[i] - atm_sum) / dehaze[i];
        y_data[i] = do_clamp((int)value, 0, 255);
    }

    //Recover UV
    int uv_width = avframe->width >> 1;
    int uv_height = avframe->height >> 1;
    int main_prefix = 0;
    int sub_prefix = 0;
    int sub_stride = avframe->linesize[1];
    int main_stride = avframe->width << 1;

    for (i = 0; i < uv_height; i++) {
        for (j = 0; j < uv_width; j++) {
            int index = main_prefix + (j << 1);
            int sub_index = sub_prefix + j;
            float f_u = 128 + (u_data[sub_index] - 128) / dehaze[index];
            float f_v = 128 + (v_data[sub_index] - 128) / dehaze[index];
            u_data[sub_index] = do_clamp((int)f_u, 0, 255);
            v_data[sub_index] = do_clamp((int)f_v, 0, 255);
        }
        sub_prefix += sub_stride;
        main_prefix += main_stride;
    }
}

static int recv_dehaze_pkt(struct defog_network *defog_network,
        struct network_frame *network_frame)
{
    NiNetworkContext *dehaze_network = defog_network->dehaze_network;
    DehazeModelCtx *dehaze_model = defog_network->dehaze_model;
    float *tensors[1];
    tensors[0] = dehaze_model->dehaze;
    int ret = ni_get_network_output(dehaze_network, use_hwframe, &network_frame->dehaze_frame,
            false, true, (uint8_t **)tensors);
    if (ret != 0) {
        return ret;
    } else {
        process_dehaze(defog_network, network_frame);
    }
    return 0;
}

static int do_encode(struct defog_network *defog_network, struct frame_req *frame_req)
{
    AVPacket enc_packet = {0};
    AVFrame *avframe = &frame_req->avframe;
    static int encode_frame_number = 0;
    int ret = encode_frame(frame_req->eof ? NULL : avframe, &enc_packet, outfile);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "error when encode frame: %d\n", ret);
        return ret;
    }
    encode_num++;
    av_frame_unref(avframe);
    free_frame_req(frame_req);
    encode_frame_number++;
    return 0;
}

static void put_back_free_network_frame(struct network_frame *network_frame)
{
    network_frame->frame_req = NULL;
    ni_list_add_tail(&network_frame->list, &network_frame_free_list);
}

static void *infer_frames(void *data)
{
    int ret = 0;
    struct frame_req *frame_req = NULL;
    struct network_frame *network_frame = NULL;
    int should_exit = 0;

    struct network_frame *pending_dark_chnl_frame = NULL;
    struct network_frame *pending_dehaze_frame = NULL;

    NI_LIST_HEAD(dark_chnl_frame_list);
    NI_LIST_HEAD(interval_frame_list);
    NI_LIST_HEAD(dehaze_frame_list);

    ni_init_list_head(&dark_chnl_frame_list);
    ni_init_list_head(&interval_frame_list);
    ni_init_list_head(&dehaze_frame_list);

    while(!should_exit)
    {
        bool idle = false;
        if (ni_list_empty(&dark_chnl_frame_list) &&
                ni_list_empty(&interval_frame_list) &&
                ni_list_empty(&dehaze_frame_list) &&
                !pending_dark_chnl_frame &&
                !pending_dehaze_frame) {
            idle = true;
        }
        if (end == 3 && ni_list_empty(&codec_workers->frm_list) && idle) {
            break;
        }

        if (pending_dark_chnl_frame) {
            ret = infer_dark_chnl_frame(defog_network, pending_dark_chnl_frame);
            if (ret == 0) {
                ni_list_add_tail(&pending_dark_chnl_frame->list, &dark_chnl_frame_list);
                pending_dark_chnl_frame = NULL;
            } else if (ret != NIERROR(EAGAIN)) {
                av_log(NULL, AV_LOG_ERROR, "cannot infer dark chnl frame\n");
                should_exit = 1;
                break;
            }
        } else {
            pthread_mutex_lock(&codec_workers->frm_lock);
            if (ni_list_empty(&codec_workers->frm_list) && idle) {
                pthread_cond_wait(&codec_workers->frm_cond,
                        &codec_workers->frm_lock);
            }

            // send frame to ai
            if ((!ni_list_empty(&codec_workers->frm_list)) &&
                    !ni_list_empty(&network_frame_free_list)) {
                frame_req = ni_list_first_entry(&codec_workers->frm_list, struct frame_req, list);
                ni_list_del(&frame_req->list);
                pthread_mutex_unlock(&codec_workers->frm_lock);

                network_frame = ni_list_first_entry(&network_frame_free_list,
                        struct network_frame, list);
                ni_list_del(&network_frame->list);

                if (!frame_req->eof) {
                    ret = generate_dark_model_frame(defog_network,
                            &network_frame->dark_frame.api_frame,
                            &frame_req->avframe, use_hwframe);
                    if (ret < 0) {
                        av_log(NULL, AV_LOG_ERROR, "cannot generate ai frame\n");
                        should_exit = 1;
                        break;
                    }
                    network_frame->frame_req = frame_req;

                    ret = infer_dark_chnl_frame(defog_network, network_frame);
                    if (ret == 0) {
                        ni_list_add_tail(&network_frame->list, &dark_chnl_frame_list);
                    } else if (ret == NIERROR(EAGAIN)) {
                        pending_dark_chnl_frame = network_frame;
                    } else {
                        av_log(NULL, AV_LOG_ERROR, "cannot infer dark chnl frame\n");
                        should_exit = 1;
                        break;
                    }
                } else {
                    network_frame->frame_req = frame_req;
                    ni_list_add_tail(&network_frame->list, &dark_chnl_frame_list);
                }
            } else {
                pthread_mutex_unlock(&codec_workers->frm_lock);
            }
        }

        // get packet from ai
        if (!ni_list_empty(&dark_chnl_frame_list)) {
            network_frame = ni_list_first_entry(&dark_chnl_frame_list, struct network_frame, list);
            if (network_frame->frame_req->eof) {
                ni_list_del(&network_frame->list);
                ni_list_add_tail(&network_frame->list, &interval_frame_list);
            } else {
                ret = recv_dark_chnl_pkt(defog_network, network_frame);
                if (ret != 0 && ret != NIERROR(EAGAIN)) {
                    should_exit = 1;
                    break;
                } else if (ret == 0) {
                    ni_list_del(&network_frame->list);
                    ni_list_add_tail(&network_frame->list, &interval_frame_list);
                }
            }
        }

        if (pending_dehaze_frame) {
            ret = infer_dehaze_frame(defog_network, network_frame);
            if (ret != 0) {
                if (!NIERROR(EAGAIN)) {
                    should_exit = 1;
                    break;
                }
            } else if (ret == 0) {
                ni_list_del(&pending_dehaze_frame->list);
                ni_list_add_tail(&pending_dehaze_frame->list, &dehaze_frame_list);
                pending_dehaze_frame = NULL;
            }
        } else {
            if (!ni_list_empty(&interval_frame_list)) {
                network_frame = ni_list_first_entry(&interval_frame_list, struct network_frame, list);
                if (network_frame->frame_req->eof) {
                    ni_list_del(&network_frame->list);
                    ni_list_add_tail(&network_frame->list, &dehaze_frame_list);
                } else {
                    int ret = generate_dehaze_frame(defog_network, network_frame);
                    if (ret != 0) {
                        should_exit = 1;
                        break;
                    }
                    ni_list_del(&network_frame->list);
                    ret = infer_dehaze_frame(defog_network, network_frame);
                    if (ret != 0) {
                        if (!NIERROR(EAGAIN)) {
                            should_exit = 1;
                            break;
                        }
                        pending_dehaze_frame = network_frame;
                    } else {
                        ni_list_add_tail(&network_frame->list, &dehaze_frame_list);
                    }
                }
            }
        }

        if (!ni_list_empty(&dehaze_frame_list)) {
            bool need_encode = false;
            network_frame = ni_list_first_entry(&dehaze_frame_list, struct network_frame, list);
            if (network_frame->frame_req->eof) {
                need_encode = true;
            } else {
                ret = recv_dehaze_pkt(defog_network, network_frame);
                if (ret != 0 && ret != NIERROR(EAGAIN)) {
                    should_exit = 1;
                    break;
                } else if (ret == 0) {
                    need_encode = true;
                }
            }

            if (need_encode) {
                struct frame_req *frame_req = network_frame->frame_req;
                ni_list_del(&network_frame->list);
                put_back_free_network_frame(network_frame);

                ret = do_encode(defog_network, frame_req);
                if (ret != 0) {
                    should_exit = 1;
                    break;
                }
            }
        }
    }

    // avoid main thread block
    while (!ni_list_empty(&codec_workers->frm_list)) {
        pthread_mutex_lock(&codec_workers->frm_lock);
        frame_req = ni_list_first_entry(&codec_workers->frm_list, struct frame_req, list);
        ni_list_del(&frame_req->list);
        pthread_mutex_unlock(&codec_workers->frm_lock);
        free_frame_req(frame_req);
    }
    codec_workers->infer_exit = 1;
    return (void *)0;
}

static void help_usage(void)
{
    printf("Usage: \n"
            "-i | --input             input file path.\n"
            "-o | --output_file       output file path.\n"
            "-d | --decoder           decoder name.\n"
            "-n | --devid             device id.\n"
            "-v | --loglevel          available debug level: warning, info, debug, trace.\n"
            "-b | --bitrate           set bitrate.\n"
            "-e | --encode_name       encode name [h264_ni_dec | h265_ni_dec].\n"
            "-p | --encode_params     encode params.\n"
            "-a | --dark_chnl_model   dark channel model file path.\n"
            "-z | --dehaze_model      dehaze model file path.\n"
            "-h | --help              print this help information.\n");
}

void setup_loglevel(char *loglevel)
{
    if (loglevel) {
        if (!strcmp(loglevel, "error")) {
            av_log_set_level(AV_LOG_ERROR);
        } else if (!strcmp(loglevel, "warning")) {
            av_log_set_level(AV_LOG_WARNING);
        } else if (!strcmp(loglevel, "info")) {
            av_log_set_level(AV_LOG_INFO);
        } else if (!strcmp(loglevel, "debug")) {
            av_log_set_level(AV_LOG_DEBUG);
        } else if (!strcmp(loglevel, "trace")) {
            av_log_set_level(AV_LOG_TRACE);
        } else {
            av_log_set_level(AV_LOG_INFO);
        }
    } else {
        av_log_set_level(AV_LOG_INFO);
    }
}

static int init_shared_thread_data(struct codec_worker *worker)
{
    int ret;
    int i;

    /* available list notification */
    ret = pthread_mutex_init(&worker->frm_lock, NULL);
    if (ret) {
        return ret;
    }

    ret = pthread_cond_init(&worker->frm_cond, NULL);
    if (ret) {
        goto fail_init_frm_cond;
    }

    ni_init_list_head(&worker->frm_list);

    /* free list notification */
    ret = pthread_mutex_init(&worker->free_lock, NULL);
    if (ret) {
        goto fail_init_free_lock;
    }

    ret = pthread_cond_init(&worker->free_cond, NULL);
    if (ret) {
        goto fail_init_free_cond;
    }

    ni_init_list_head(&worker->free_list);

    pframes = (struct frame_req *)calloc(NI_TRANSCODE_FRAME_NUM, sizeof(struct frame_req));
    if (!pframes) {
        goto fail_alloc_frames;
    }

    for (i = 0; i < NI_TRANSCODE_FRAME_NUM; i++) {
        //pr_log("init frame_req %p\n", &pframes[i]);
        ni_list_add_tail(&pframes[i].list, &worker->free_list);
    }

    return 0;
fail_alloc_frames:
    pthread_cond_destroy(&worker->free_cond);
fail_init_free_cond:
    pthread_mutex_destroy(&worker->free_lock);
fail_init_free_lock:
    pthread_cond_destroy(&worker->frm_cond);
fail_init_frm_cond:
    pthread_mutex_destroy(&worker->frm_lock);
    return ret;
}

static void cleanup_shared_thread_data(struct codec_worker *worker)
{
    if (!ni_list_empty(&worker->frm_list)) {
        av_log(NULL, AV_LOG_ERROR, "frame list is not empty\n");
    }

    free(pframes);

    pthread_mutex_destroy(&worker->frm_lock);
    pthread_cond_destroy(&worker->frm_cond);

    pthread_mutex_destroy(&worker->free_lock);
    pthread_cond_destroy(&worker->free_cond);
}

static int create_ai_thread(struct codec_worker *worker)
{
    int ret;
    pthread_attr_t attr;

    ret = pthread_attr_init(&attr);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "fail to initialize attr: %d.\n", ret);
        return ret;
    }

    ret = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "fail to set attr detachstate: %d.\n", ret);
        goto end;
    }

    ret = pthread_create(&worker->pthid, NULL, infer_frames, 0);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "creat thread failed!, %d\n", ret);
        goto end;
    }
end:
    if (0 != pthread_attr_destroy(&attr)) {
        av_log(NULL, AV_LOG_ERROR, "failed to destroy pthread attr\n");
    }
    return ret;
}

static int destroy_ai_thread(struct codec_worker *worker)
{
    int ret;
    void *result;
    av_log(NULL, AV_LOG_INFO, "try to stop ai thread\n");
    pthread_cond_signal(&worker->frm_cond);
    ret = pthread_join(worker->pthid, &result);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to join ai thread\n");
    }
    if ((int)((long)result)) {
        av_log(NULL, AV_LOG_ERROR, "Bad ai thread result: %d\n", (int)((long)result));
    }

    return 0;
}

static int create_fps_thread(struct codec_worker *worker)
{
    int ret;
    pthread_attr_t attr;

    ret = pthread_attr_init(&attr);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "fail to initialize attr: %d.\n", ret);
        return ret;
    }

    ret = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "fail to set attr detachstate: %d.\n", ret);
        goto end;
    }

    ret = pthread_create(&worker->pthtimeid,NULL,printfps,0);
    if (ret !=0) {
        av_log(NULL, AV_LOG_ERROR, "creat thread failed!\n");
    }
end:
    if (0 != pthread_attr_destroy(&attr)) {
        av_log(NULL, AV_LOG_ERROR, "failed to destroy pthread attr\n");
    }
    return ret;
}

static int destroy_fps_thread(struct codec_worker *worker)
{
    int ret;
    void *result;
    av_log(NULL, AV_LOG_INFO, "try to stop fps thread\n");
    ret = pthread_join(worker->pthtimeid, &result);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to join fps thread\n");
    }
    if ((int)((long)result)) {
        av_log(NULL, AV_LOG_ERROR, "Bad fps thread result: %d\n", (int)((long)result));
    }

    return ret;
}

static struct frame_req *get_free_frame_req()
{
    struct frame_req *frame_req;
wait_free_frame:
    pthread_mutex_lock(&codec_workers->free_lock);
    if (ni_list_empty(&codec_workers->free_list)) {
        pthread_cond_wait(&codec_workers->free_cond, &codec_workers->free_lock);
    }
    if (ni_list_empty(&codec_workers->free_list)) {
        pthread_mutex_unlock(&codec_workers->free_lock);
        goto wait_free_frame;
    }
    frame_req = ni_list_first_entry(&codec_workers->free_list, struct frame_req, list);
    ni_list_del(&frame_req->list);
    pthread_mutex_unlock(&codec_workers->free_lock);

    return frame_req;
}

static void put_busy_frame_req(struct frame_req *frame_req)
{
    pthread_mutex_lock(&codec_workers->frm_lock);
    if (ni_list_empty(&codec_workers->frm_list)) {
        ni_list_add_tail(&frame_req->list, &codec_workers->frm_list);
        pthread_cond_signal(&codec_workers->frm_cond);
    } else {
        ni_list_add_tail(&frame_req->list, &codec_workers->frm_list);
    }
    pthread_mutex_unlock(&codec_workers->frm_lock);
}

int main(int argc, char **argv)
{
    int ret;
    AVPacket packet;

    const char *dark_chnl_model_file = NULL;
    const char *dehaze_model_file = NULL;

    struct frame_req *frame_req;

    char *loglevel = NULL;
    int opt;
    int opt_index;
    codec_workers = (struct codec_worker *)calloc(1, sizeof(struct codec_worker));
    if (codec_workers == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate codec workers.\n");
        return EXIT_FAILURE;
    }
    codec_workers->bitrate = 0;
    const char *opt_string = "n:i:d:o:e:p:v:b:f:l:a:z:h";
    static struct option long_options[] = {
        {"input",          required_argument, NULL, 'i'},
        {"output",         required_argument, NULL, 'o'},
        {"decoder",        required_argument, NULL, 'd'},
        {"devid",          required_argument, NULL, 'n'},
        {"loglevel",       required_argument, NULL, 'v'},
        {"bitrate",        required_argument, NULL, 'b'},
        {"encode_name",    required_argument, NULL, 'e'},
        {"encode_params",  required_argument, NULL, 'p'},
        {"dark_chnl_model", required_argument, NULL, 'a'},
        {"dehaze_model",   required_argument, NULL, 'z'},
        {"help",           no_argument,       NULL, 'h'},
        { NULL,            0,                 NULL,  0 },
    };

    while ((opt = getopt_long(argc, argv, opt_string, long_options, &opt_index)) != -1) {
        switch (opt) {

            case 'd':
                codec_workers->decoder_name = optarg;
                break;
            case 'i':
                codec_workers->input_file = optarg;
                break;
            case 'o':
                codec_workers->output_file = optarg;
                break;
            case 'n':
                codec_workers->devid = atoi(optarg);
                break;
            case 'v':
                loglevel = optarg;
                break;
            case 'b':
                codec_workers->bitrate = atoi(optarg);
                break;
            case 'e':
                codec_workers->encoder_name = optarg;
                break;
            case 'p':
                codec_workers->encoder_params = optarg;
                break;
            case 'a':
                dark_chnl_model_file = optarg;
                break;
            case 'z':
                dehaze_model_file = optarg;
                break;
            case 'h':
                help_usage();
                return EXIT_SUCCESS;
            default:
                help_usage();
                return EXIT_FAILURE;
        }
    }
    setup_loglevel(loglevel);

    // Register all formats, codecs, filters
#if LIBAVFORMAT_VERSION_MAJOR < 58
    av_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif
#if LIBAVCODEC_VERSION_MAJOR < 58
    avcodec_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif
#if LIBAVFILTER_VERSION_MAJOR < 7
    avfilter_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif

    if (access(dark_chnl_model_file, R_OK) != 0) {
        av_log(NULL, AV_LOG_ERROR, "cannot find model: %s\n", dark_chnl_model_file);
        return EXIT_FAILURE;
    }
    av_log(NULL, AV_LOG_INFO, "dark_chnl_model %s\n", dark_chnl_model_file);

    if (access(dehaze_model_file, R_OK) != 0) {
        av_log(NULL, AV_LOG_ERROR, "cannot find model: %s\n", dehaze_model_file);
        return EXIT_FAILURE;
    }
    av_log(NULL, AV_LOG_INFO, "dehaze_model %s\n", dehaze_model_file);

    if (codec_workers->input_file == NULL) {
        av_log(NULL, AV_LOG_ERROR, "input file name must be specified.\n");
        return EXIT_FAILURE;
    }

    if (codec_workers->output_file == NULL) {
        av_log(NULL, AV_LOG_ERROR, "output file name must be specified.\n");
        return EXIT_FAILURE;
    }

    if (!codec_workers->decoder_name || !codec_workers->encoder_name) {
        av_log(NULL, AV_LOG_ERROR, "decoder_name name must be specified.\n");
        return EXIT_FAILURE;
    }

    if (init_shared_thread_data(codec_workers) != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to initialize common data\n");
        return EXIT_FAILURE;
    }

    outfile = fopen(codec_workers->output_file,"wb");
    if(!outfile)
    {
        av_log(NULL, AV_LOG_ERROR, "open file fail!\n");
        exit(1);
    }
    // Allocate packet for encoder output
    frame = av_frame_alloc();
    if (!frame) {
        av_log(NULL, AV_LOG_ERROR, "Could not allocate pkt frame");
        exit(1);
    }

    ret = init_network_struct(dark_chnl_model_file, dehaze_model_file);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to initialize network struct\n");
        goto end;
    }

    if ((ret = open_input_file()) < 0)
        goto end;

    timer_start = av_gettime_relative();

    if ((ret = create_fps_thread(codec_workers)) != 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to create fps thread\n");
        goto end;
    }

    if ((ret = create_ai_thread(codec_workers)) != 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to create ai thread\n");
        goto stop_fpst;
    }

    /* read all packets */
    while (1) {
        if (codec_workers->infer_exit) {
            break;
        }
        if ((ret = av_read_frame(fmt_ctx, &packet)) < 0)
        {
            if(end == 2)
            {
                frame_req = get_free_frame_req();
                frame_req->eof = 1;
                put_busy_frame_req(frame_req);
                break;
            }
            end = 1;
        }
        if (packet.stream_index == video_stream_index || end == 1) {
            if(!end)
            {
                ret = avcodec_send_packet(dec_ctx, &packet);
            }
            else{
                ret = avcodec_send_packet(dec_ctx, NULL);
            }

            if (ret < 0 && ret != AVERROR_EOF) {
                av_log(NULL, AV_LOG_ERROR, "Error while sending a packet to the decoder\n");
                break;
            }
            while (ret >= 0 || end == 1) {
                ret = avcodec_receive_frame(dec_ctx, frame);
                if (ret == AVERROR(EAGAIN)) {
                    break;
                }
                else if(ret == AVERROR_EOF){
                    end = 2;
                    break;
                }
                else if (ret < 0) {
                    av_log(NULL, AV_LOG_ERROR, "Error while receiving a frame from the decoder\n");
                    goto stop_ait;
                }

                frame->pts = frame->best_effort_timestamp;
                frame_num++;

                av_log(NULL, AV_LOG_DEBUG, "decoded frame_num %d\n", frame_num);

                frame_req = get_free_frame_req();
                frame_req->eof = 0;
                ret = av_frame_ref(&frame_req->avframe, frame);
                if (ret < 0) {
                    av_log(NULL, AV_LOG_ERROR, "copy frame to filter failed\n");
                    goto stop_ait;
                }
                //av_log(NULL, AV_LOG_DEBUG, "generate frame %d\n",
                //        ((niFrameSurface1_t *)frame->data[3])->ui16FrameIdx);
                put_busy_frame_req(frame_req);
                av_frame_unref(frame);
            }
        }
        av_packet_unref(&packet);
    }

stop_ait:
    av_log(NULL, AV_LOG_INFO, "main thread end\n");
    end = 3;
    destroy_ai_thread(codec_workers);
stop_fpst:
    stop_fps = true;
    destroy_fps_thread(codec_workers);
end:
    avcodec_free_context(&dec_ctx);
    avcodec_free_context(&enc_ctx);
    avformat_close_input(&fmt_ctx);
    av_frame_free(&frame);
    fclose(outfile);

    cleanup_shared_thread_data(codec_workers);
    free(codec_workers);
    free_network_struct();

    if (ret < 0 && ret != AVERROR_EOF) {
        av_log(NULL, AV_LOG_ERROR, "Error occurred: %s\n", av_err2str(ret));
        exit(1);
    }
    exit(0);
}
