/*
 * Copyright (c) 2010 Nicolas George
 * Copyright (c) 2011 Stefano Sabatini
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * @file
 * API example for lpr
 * @example ni_lpr.c
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>

#include <locale.h>

#include <sys/time.h>

#include <libavcodec/avcodec.h>
#include <libavformat/avformat.h>
#include <libavfilter/buffersink.h>
#include <libavfilter/buffersrc.h>
#include <libswscale/swscale.h>
#include <libavutil/imgutils.h>
#include <libavutil/opt.h>
#include <libavutil/time.h>
#include <pthread.h>
#include <stdint.h>
#include <float.h>

#include "ni_list_head.h"
#include "netint_network.h"
#include "lpr_network.h"

#include "ni_log.h"

#define NI_TRANSCODE_FRAME_NUM 3
#define NI_SAME_CENTER_THRESH 2
#define NI_SAME_BORDER_THRESH 8
#define IS_FFMPEG_43_AND_ABOVE                                                \
    ((LIBAVFILTER_VERSION_MAJOR > 7) ||                                       \
     (LIBAVFILTER_VERSION_MAJOR == 7 && LIBAVFILTER_VERSION_MINOR >= 85))

bool use_hwframe = false;
FILE *outfile = NULL;
AVFrame *frame;
static AVFormatContext *fmt_ctx;
static AVCodecContext *dec_ctx;
static AVCodecContext *enc_ctx;
static int video_stream_index = -1;
unsigned long decode_frame_number = 0;
int end = 0;
unsigned long e2e_frame_number = 0;
int64_t timer_start;
bool stop_fps = false;

struct frame_req {
    struct ni_list_head list;
    AVFrame avframe;
    long frame_index;
};

static inline bool is_eof(struct frame_req *frame_req)
{
    return frame_req->frame_index == -1;
}

static inline void set_eof(struct frame_req *frame_req)
{
    frame_req->frame_index = -1;
}

struct frame_req *pframes = NULL;

// Only count previous 1000 frames time statistics
#define MAX_TIMES_NUMBER 1000

struct codec_worker {
    const char *decoder_name;
    const char *input_file;
    const char *output_file;
    const char *decoder_params;
    const char *encoder_name;
    const char *encoder_params;
    int devid;
    int bitrate;

    pthread_t pthid;
    pthread_t pthtimeid;

    int infer_exit;

    struct ni_list_head frm_list;
    pthread_mutex_t frm_lock;
    pthread_cond_t  frm_cond;

    struct ni_list_head free_list;
    pthread_mutex_t free_lock;
    pthread_cond_t  free_cond;
};

struct codec_worker *codec_workers = NULL;

struct lpr_network {
    NiNetworkContext *lic_det_network;
    NiNetworkContext *lic_rec_network;
    LicDetModelCtx *lic_det_model;
    LicRecModelCtx *lic_rec_model;

    //
    struct plate_box *plate_box;
    int box_num;
    int running_box;
    long frame_index;

    AVFrame input_ref;
};

struct lpr_network *lpr_network;

NI_LIST_HEAD(network_frame_free_list);
//NI_LIST_HEAD(network_frame_busy_list);

NI_LIST_HEAD(lic_rec_frame_free_list);
NI_LIST_HEAD(lic_rec_frame_busy_list);

struct location {
    int top;
    int left;
    int bottom;
    int right;
    long frame_index;
};

struct network_frame {
    NiNetworkFrame frame;
    struct ni_list_head list;
    void *priv_data;
};

struct network_frame *lic_det_network_frames;
struct network_frame *lic_rec_network_frames;

static int init_network_frames(struct lpr_network *lpr_network)
{
    int i, ret;

    lic_det_network_frames = (struct network_frame *)calloc(NUM_NETWORK_FRAME,
            sizeof(struct network_frame));
    if (lic_det_network_frames == NULL) {
        return NIERROR(ENOMEM);
    }

    //lic det frame
    for (i = 0; i < NUM_NETWORK_FRAME; i++) {
        struct network_frame *network_frame = &lic_det_network_frames[i];
        NiNetworkFrame *frame = &network_frame->frame;

        ret = ni_ai_frame_buffer_alloc(&frame->api_frame.data.frame,
                &lpr_network->lic_det_network->network_data);
        if (ret != NI_RETCODE_SUCCESS) {
            av_log(NULL, AV_LOG_ERROR, "failed to allocate licence detection frame\n");
            return NIERROR(ENOMEM);
        }

        ret = ni_ai_packet_buffer_alloc(&frame->api_packet.data.packet,
                &lpr_network->lic_det_network->network_data);
        if (ret != NI_RETCODE_SUCCESS) {
            av_log(NULL, AV_LOG_ERROR, "failed to allocate licence detection packet\n");
            return NIERROR(ENOMEM);
        }

        ni_list_add_tail(&network_frame->list, &network_frame_free_list);
    }

    lic_rec_network_frames = (struct network_frame *)calloc(NUM_NETWORK_FRAME,
            sizeof(struct network_frame));
    if (lic_rec_network_frames == NULL) {
        return NIERROR(ENOMEM);
    }

    //lic rec frame
    for (i = 0; i < NUM_NETWORK_FRAME; i++) {
        struct network_frame *network_frame = &lic_rec_network_frames[i];
        NiNetworkFrame *frame = &network_frame->frame;

        ret = ni_ai_frame_buffer_alloc(&frame->api_frame.data.frame,
                &lpr_network->lic_rec_network->network_data);
        if (ret != NI_RETCODE_SUCCESS) {
            av_log(NULL, AV_LOG_ERROR, "failed to allocate licence recognition frame\n");
            return NIERROR(ENOMEM);
        }

        ret = ni_ai_packet_buffer_alloc(&frame->api_packet.data.packet,
                &lpr_network->lic_rec_network->network_data);
        if (ret != NI_RETCODE_SUCCESS) {
            av_log(NULL, AV_LOG_ERROR, "failed to allocate licence recgonition packet\n");
            return NIERROR(ENOMEM);
        }

        network_frame->priv_data = malloc(sizeof(struct location));
        if (network_frame->priv_data == NULL) {
            av_log(NULL, AV_LOG_ERROR, "failed to allocate location\n");
            return NIERROR(ENOMEM);
        }

        ni_list_add_tail(&network_frame->list, &lic_rec_frame_free_list);
    }

    return 0;
}

static void free_network_frames()
{

//    if (!ni_list_empty(&network_frame_busy_list)) {
//        av_log(NULL, AV_LOG_WARNING, "Network frames are still in use\n");
//    }

    if (lic_det_network_frames != NULL) {
        int i;
        for (i = 0; i < NUM_NETWORK_FRAME; i++) {
            //lic det frame
            struct network_frame *network_frame = &lic_det_network_frames[i];
            NiNetworkFrame *frame = &network_frame->frame;
            ni_frame_buffer_free(&frame->api_frame.data.frame);
            ni_packet_buffer_free(&frame->api_packet.data.packet);
        }
        free(lic_det_network_frames);
        lic_det_network_frames = NULL;
    }

    if (lic_rec_network_frames != NULL) {
        int i;
        for (i = 0; i < NUM_NETWORK_FRAME; i++) {
            //lic rec frame
            struct network_frame *network_frame = &lic_rec_network_frames[i];
            NiNetworkFrame *frame = &network_frame->frame;
            ni_frame_buffer_free(&frame->api_frame.data.frame);
            ni_packet_buffer_free(&frame->api_packet.data.packet);
            free(network_frame->priv_data);
        }

        free(lic_rec_network_frames);
        lic_rec_network_frames = NULL;
    }
}

static void destroy_lic_det_network(struct lpr_network *network)
{
    if (network) {
        if (network->lic_det_model) {
            destroy_lic_detect_model(network->lic_det_model);
            free(network->lic_det_model);
            network->lic_det_model = NULL;
        }
    }
}

static void destroy_lic_rec_network(struct lpr_network *network)
{
    if (network) {
        if (network->lic_rec_model) {
            destroy_lic_rec_model(network->lic_rec_model);
            free(network->lic_rec_model);
            network->lic_rec_model = NULL;
        }
    }
}

static void destroy_lpr_network(struct lpr_network *lpr_network)
{
    destroy_lic_det_network(lpr_network);
    destroy_lic_rec_network(lpr_network);
}

static int create_lic_det_network(struct lpr_network *lpr_network, const char *nbg_file,
        int pic_width, int pic_height, float obj_thresh, float nms_thresh)
{
    int ret = 0;
    LicDetModelCtx *model_ctx = NULL;

    ret = ni_alloc_network_context(&lpr_network->lic_det_network, use_hwframe,
            0, 10, 0, 0, 0, nbg_file);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate network context\n");
        goto out;
    }

    ni_network_data_t *network_data = &lpr_network->lic_det_network->network_data;

    model_ctx = (LicDetModelCtx *)calloc(1, sizeof(LicDetModelCtx));
    if (model_ctx == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate licence detection model\n");
        ret = NIERROR(ENOMEM);
        goto out;
    }
    lpr_network->lic_det_model = model_ctx;

    ret = create_lic_detect_model(model_ctx, network_data, pic_width, pic_height,
            obj_thresh, nms_thresh);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to initialize licence detection model\n");
        goto out;
    }

    return ret;

out:
    if (lpr_network->lic_det_model) {
      free(lpr_network->lic_det_model);
      lpr_network->lic_det_model = NULL;
    }
    ni_cleanup_network_context(lpr_network->lic_det_network, use_hwframe);

    return ret;
}

static int create_lic_rec_network(struct lpr_network *lpr_network, const char *nbg_file)
{
    int ret = 0;
    LicRecModelCtx *model_ctx = NULL;

    ret = ni_alloc_network_context(&lpr_network->lic_rec_network, use_hwframe,
            0, 30, 0, 0, 0, nbg_file);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate network context\n");
        goto out;
    }

    ni_network_data_t *network_data = &lpr_network->lic_rec_network->network_data;

    model_ctx = (LicRecModelCtx *)calloc(1, sizeof(LicRecModelCtx));
    if (model_ctx == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate licence recognition model\n");
        ret = NIERROR(ENOMEM);
        goto out;
    }
    lpr_network->lic_rec_model = model_ctx;

    ret = create_lic_rec_model(model_ctx, network_data);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to initialize licence recognition model\n");
        goto out;
    }

    return ret;

out:
    if (model_ctx) {
      free(model_ctx);
      lpr_network->lic_rec_model = NULL;
    }
    ni_cleanup_network_context(lpr_network->lic_rec_network, use_hwframe);

    return ret;
}

static int init_network_struct(const char *lic_det_nbg_file, const char *lic_rec_nbg_file,
        int pic_width, int pic_height, float obj_thresh, float nms_thresh)
{
    int ret;
    lpr_network = (struct lpr_network *)calloc(1, sizeof(struct lpr_network));
    if (lpr_network == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate lpr network structure\n");
        return -1;
    }

    ret = create_lic_det_network(lpr_network, lic_det_nbg_file, pic_width, pic_height,
            obj_thresh, nms_thresh);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to create lic det network\n");
        goto fail_out;
    }

    ret = create_lic_rec_network(lpr_network, lic_rec_nbg_file);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to create lic rec network\n");
        goto fail_out;
    }

    ret = init_network_frames(lpr_network);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to initialize network frames\n");
        goto fail_out;
    }

    return 0;
fail_out:
    destroy_lpr_network(lpr_network);
    free_network_frames();
    return -1;
}

static void free_network_struct()
{
    if (lpr_network) {
        destroy_lpr_network(lpr_network);
        ni_cleanup_network_context(lpr_network->lic_det_network, use_hwframe);
        ni_cleanup_network_context(lpr_network->lic_rec_network, use_hwframe);
        free(lpr_network);
        lpr_network = NULL;
    }

    free_network_frames();
}

static int open_input_file()
{
    int ret;
    AVCodec *dec;
    if ((ret = avformat_open_input(&fmt_ctx, codec_workers->input_file, NULL, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot open input file\n");
        return ret;
    }

    if ((ret = avformat_find_stream_info(fmt_ctx, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot find stream information\n");
        return ret;
    }

    dec = avcodec_find_decoder_by_name(codec_workers->decoder_name);
    if (dec) {
        if (fmt_ctx->streams[0]->codecpar->codec_id != dec->id) {
            av_log(NULL, AV_LOG_ERROR, "codec does not match with stream id\n");
            return AVERROR_DECODER_NOT_FOUND;
        }
    }

    video_stream_index = 0;

    /* create decoding context */
    dec_ctx = avcodec_alloc_context3(dec);
    if (!dec_ctx)
        return AVERROR(ENOMEM);

    avcodec_parameters_to_context(dec_ctx, fmt_ctx->streams[0]->codecpar);

    char str_devid[4] = {0};
    snprintf(str_devid, sizeof(str_devid), "%d", codec_workers->devid);
    av_opt_set(dec_ctx->priv_data, "dec", str_devid, 0);
    if(codec_workers->decoder_params)
        av_opt_set(dec_ctx->priv_data, "xcoder-params",codec_workers->decoder_params, 0);

    if (dec_ctx->codec_type == AVMEDIA_TYPE_VIDEO)
            dec_ctx->framerate = av_guess_frame_rate(fmt_ctx, fmt_ctx->streams[0], NULL);

    ret = avcodec_parameters_from_context(fmt_ctx->streams[0]->codecpar, dec_ctx);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to copy decoder parameters from codec context for stream #");
        return ret;
    }

    /* init the video decoder */
    if ((ret = avcodec_open2(dec_ctx, dec, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot open video decoder\n");
        return ret;
    }
    return 0;
}

static void print_report(int is_last_report, int64_t timer_start,
        int64_t cur_time, unsigned long frame_number)
{
    static int64_t last_time = -1;
    float t;
    char buf[1024];
    float fps;

    if (!is_last_report) {
        if (last_time == -1) {
            last_time = cur_time;
            return;
        }
        if ((cur_time - last_time) < 500000)
            return;
        last_time = cur_time;
    }

    t = (cur_time - timer_start) / 1000000.0;

    fps = t > 1 ? frame_number / t : 0;

    const char end = is_last_report ? '\n' : '\r';

    buf[0] = '\0';
    snprintf(buf + strlen(buf), sizeof(buf) - strlen(buf), "frame=%5ld fps=%3.*f ",
            frame_number, (fps < 9.95), fps);
    pr_err("%s   %c", buf, end);
    fflush(stderr);
}

static void *printfps(void *data)
{
    while (1) {
        if (stop_fps == true) {
            break;
        }
        usleep(1000);
        print_report(0, timer_start, av_gettime_relative(), e2e_frame_number);
    }
    print_report(1, timer_start, av_gettime_relative(), e2e_frame_number);
    return (void *)0;
}

static int ni_recreate_lic_det_frame(LicDetModelCtx *ctx, ni_frame_t *ni_frame, AVFrame *in_frame)
{
    int ret;
    uint8_t *scale_rgb_data[3];
    int dst_stride[3];
    YoloModelCtx *yolo_model_ctx = &ctx->yolo_model_ctx;

    av_log(NULL, AV_LOG_DEBUG,
           "linesize %d/%d/%d, data %p/%p/%p, pixel %dx%d\n",
           in_frame->linesize[0], in_frame->linesize[1], in_frame->linesize[2],
           in_frame->data[0], in_frame->data[1], in_frame->data[2], in_frame->width,
           in_frame->height);

    //GBRPlanar -> BGRPlanar
    scale_rgb_data[0] = ni_frame->p_data[0] + yolo_model_ctx->input_width *
            yolo_model_ctx->input_height;
    scale_rgb_data[1] = ni_frame->p_data[0];
    scale_rgb_data[2] = ni_frame->p_data[0] + yolo_model_ctx->input_width *
            yolo_model_ctx->input_height * 2;

    dst_stride[0] = yolo_model_ctx->input_width;
    dst_stride[1] = yolo_model_ctx->input_width;
    dst_stride[2] = yolo_model_ctx->input_width;

    ret = sws_scale(ctx->scale_ctx, (const uint8_t * const *)in_frame->data,
            in_frame->linesize, 0, in_frame->height, scale_rgb_data,
            dst_stride);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "cannot scale lic det frame\n");
        return ret;
    }

    return 0;
}

static int generate_lic_det_model_frame(struct lpr_network *lpr_network,
        ni_session_data_io_t *ai_frame, AVFrame *avframe, bool hwframe)
{
    int ret = 0;
    NiNetworkContext *lic_det_network = lpr_network->lic_det_network;
    ni_retcode_t retval;

    //if (hwframe == false) {
        retval = ni_ai_frame_buffer_alloc(&ai_frame->data.frame, &lic_det_network->network_data);
        if (retval != NI_RETCODE_SUCCESS) {
            av_log(NULL, AV_LOG_ERROR, "cannot allocate sw ai frame buffer\n");
            return NIERROR(ENOMEM);
        }
        ret = ni_recreate_lic_det_frame(lpr_network->lic_det_model, &ai_frame->data.frame, avframe);
        if (ret < 0) {
            av_log(NULL, AV_LOG_ERROR, "cannot recreate sw ai frame\n");
            return ret;
        }
    //} else {
    //    ai_frame->data.frame.p_data[3] = avframe->data[3];
    //}

    return ret;
}

static void free_frame_req(struct frame_req *frame_req)
{
    if (!frame_req) {
        return;
    }
    pthread_mutex_lock(&codec_workers->free_lock);
    if (ni_list_empty(&codec_workers->free_list)) {
        ni_list_add_tail(&frame_req->list, &codec_workers->free_list);
        pthread_cond_signal(&codec_workers->free_cond);
    } else {
        ni_list_add_tail(&frame_req->list, &codec_workers->free_list);
    }
    pthread_mutex_unlock(&codec_workers->free_lock);
}

static int infer_lic_det_frame(struct lpr_network *lpr_network,
        struct network_frame *network_frame)
{
    AVFrame *avframe = &((struct frame_req *)network_frame->priv_data)->avframe;

    int ret = ni_set_network_input(lpr_network->lic_det_network, use_hwframe,
            &network_frame->frame.api_frame, NULL,
            avframe->width, avframe->height, &network_frame->frame, true);
    return ret;
}

static int recv_lic_det_pkt(struct lpr_network *lpr_network,
        struct network_frame *network_frame)
{
    NiNetworkContext *lic_det_network = lpr_network->lic_det_network;
    int ret = ni_get_network_output(lic_det_network, use_hwframe, &network_frame->frame,
            false, false, NULL);
    if (ret != 0) {
        return ret;
    }

    return 0;
}

static uint8_t do_linear_interpolate(uint8_t *data, int x, int y, int width, int height,
        int linesize)
{
    int subU, subV, sum, index, subUI, subVI;

    subU = x & 255;
    subV = y & 255;
    x  >>= 8;
    y  >>= 8;

    index = x + y * linesize;
    subUI = 256 - subU;
    subVI = 256 - subV;

    if ((unsigned)x < (unsigned)(width - 1)){
        if((unsigned)y < (unsigned)(height - 1)){
            sum = subVI * (subUI * data[index] + subU * data[index + 1]) +
                subV  * (subUI * data[index + linesize] + subU * data[index + linesize + 1]);
            sum = (sum + (1 << 15))>> 16;
        } else {
            if (y < 0)
                y = 0;
            else
                y = height - 1;
            index = x + y * linesize;
            sum   = subUI * data[index] + subU * data[index + 1];
            sum   = (sum + (1 << 7)) >> 8;
        }
    } else {
        if (x < 0)
            x = 0;
        else
            x = width - 1;
        if ((unsigned)y < (unsigned)(height - 1)){
            index = x + y * linesize;
            sum   = subVI * data[index] + subV * data[index + linesize];
            sum   = (sum + (1 << 7)) >> 8;
        } else {
            if (y < 0)
                y = 0;
            else
                y = height - 1;
            index = x + y * linesize;
            sum   = data[index];
        }
    }

    if (sum > 255) {
        return 255;
    } else if (sum < 0) {
        return 0;
    } else {
        return sum;
    }
}

static int do_linear_sample(uint8_t *dst, uint8_t *src, int dst_linesize, int src_linesize,
        int hsub, int vsub, int dst_w, int dst_h, int src_w, int src_h,
        double f[9])
{
    int x, y;
    for (y = 0; y < dst_h; y++){
        int sy = y << vsub;
        for (x = 0; x < dst_w; x++){
            int u, v, sum, sx;
            sx = x << hsub;
            u = (lrint(256 * (f[0] * sx + f[1] * sy + f[2]) /
                (f[6] * sx + f[7] * sy + f[8]))) >> hsub;
            v = (lrint(256 * (f[3] * sx + f[4] * sy + f[5]) /
                (f[6] * sx + f[7] * sy + f[8]))) >> vsub;
            sum = do_linear_interpolate(src, u, v, src_w, src_h, src_linesize);
            dst[x + y * dst_linesize] = sum;
        }
    }
    return 0;
}

/*
 * For reference:
 * https://dev.aao.org.au/rds/aaomathutils/-/blob/master/doc/Vorlesung_3.2_SpatialTransformations.pdf
 * Section 2.4.1 Inverse
 *
 * P0(x0,y0)      P1(x1,y1)       P0'(x0',y0')    P1'(x1',y1')
 * ----------------\               -------------------
 * \                \              |                 |
 *  \                \      ->     |                 |
 *   \                \            |                 |
 *    -----------------            -------------------
 *    P3(x3,y3)     P2(x2,y2)     P3'(x3',y3')    P2'(x2',y2')
 *
 * dx1 = x1 - x2
 * dx2 = x3 - x2
 * dx3 = x0 - x1 + x2 - x3
 * dy1 = y1 - y2
 * dy2 = y3 - y2
 * dy3 = y0 - y1 + y2 - y3
 *
 * a13 = (dx3 * dy2 - dy3 * dx2) / (dx1 * dy2 - dy1 * dx2)
 * a23 = (dx1 * dy3 - dy1 * dx3) / (dx1 * dy2 - dy1 * dx2)
 * a11 = x1 - x0 + a13 * x1
 * a21 = x3 - x0 + a23 * x3
 * a31 = x0
 * a12 = y1 - y0 + a13 * y1
 * a22 = y3 - y0 + a23 * y3
 * a32 = y0
 *
 *                        |a22*a33-a23*a32 a13*a32-a12*a33 a12*a23-a13*a22|
 * [u v 1] = [x' y' w'] * |a23*a31-a21*a33 a11*a33-a13*a31 a13*a21-a11*a23|
 *                        |a21*a32-a22*a31 a12*a31-a11*a32 a11*a22-a12*a21|
 *
 * (u,v) is the point (x,y) in the original image, while (x',y') is the point normalized to (0,1) in the transformed image.
 *
 * i copied the implement from the ffmpeg vf_perspective filter only for convenient.
 */

static void sample_perspective(AVFrame *in, AVFrame *out, double landmark[4][2])
{
    double f[9], q;

    f[6] = ((landmark[0][0] - landmark[1][0] - landmark[2][0] + landmark[3][0]) *
            (landmark[2][1] - landmark[3][1]) -
            (landmark[0][1] - landmark[1][1] - landmark[2][1] + landmark[3][1]) *
            (landmark[2][0] - landmark[3][0])) * out->height;
    f[7] = ((landmark[0][1] - landmark[1][1] - landmark[2][1] + landmark[3][1]) *
            (landmark[1][0] - landmark[3][0]) -
            (landmark[0][0] - landmark[1][0] - landmark[2][0] + landmark[3][0]) *
            (landmark[1][1] - landmark[3][1])) * out->width;
    q =     (landmark[1][0] - landmark[3][0]) * (landmark[2][1] - landmark[3][1]) -
            (landmark[2][0] - landmark[3][0]) * (landmark[1][1] - landmark[3][1]);

    f[0] = q * (landmark[1][0] - landmark[0][0]) * out->height + f[6] * landmark[1][0];
    f[1] = q * (landmark[2][0] - landmark[0][0]) * out->width  + f[7] * landmark[2][0];
    f[2] = q *  landmark[0][0] * out->width * out->height;
    f[3] = q * (landmark[1][1] - landmark[0][1]) * out->height + f[6] * landmark[1][1];
    f[4] = q * (landmark[2][1] - landmark[0][1]) * out->width  + f[7] * landmark[2][1];
    f[5] = q *  landmark[0][1] * out->width * out->height;
    f[8] = q * out->width * out->height;

    do_linear_sample(out->data[0], in->data[0], out->linesize[0],
            in->linesize[0], 0, 0, out->width, out->height, in->width, in->height, f);
    do_linear_sample(out->data[1], in->data[1], out->linesize[1],
            in->linesize[1], 1, 1, out->width / 2, out->height / 2, in->width, in->height, f);
    do_linear_sample(out->data[2], in->data[2], out->linesize[2],
            in->linesize[2], 1, 1, out->width / 2, out->height / 2, in->width, in->height, f);
}

static void to_fp(AVFrame *frame, float *buffer)
{
    int i, j, k = 0;
    uint8_t *bdata = frame->data[1];
    uint8_t *gdata = frame->data[0];
    uint8_t *rdata = frame->data[2];
    float *bp = &buffer[0];
    float *gp = &buffer[frame->width * frame->height];
    float *rp = &buffer[frame->width * frame->height * 2];
    int b_stride = 0;
    int g_stride = 0;
    int r_stride = 0;

    //printf("frame width %d, height %d, linesize %d/%d/%d\n", frame->width, frame->height,
    //        frame->linesize[0], frame->linesize[1], frame->linesize[2]);

    for (i = 0; i < frame->height; i++) {
        for (j = 0; j < frame->width; j++, k++) {
            //B
            bp[k] = ((float)bdata[b_stride + j] / 255 - 0.588) / 0.193;
            //G
            gp[k] = ((float)gdata[g_stride + j] / 255 - 0.588) / 0.193;
            //R
            rp[k] = ((float)rdata[r_stride + j] / 255 - 0.588) / 0.193;
        }
        b_stride += frame->linesize[1];
        g_stride += frame->linesize[0];
        r_stride += frame->linesize[2];
    }
}

static int preprocess_licence_plate_rec(LicRecModelCtx *ctx,
        ni_network_data_t *network_data,
        struct network_frame *network_frame, AVFrame *in)
{
    int ret;
    sample_perspective(in, &ctx->persp_frame, ctx->landmark);
    ret = sws_scale(ctx->rgb_sws_ctx, (const uint8_t *const*)ctx->persp_frame.data,
            ctx->persp_frame.linesize, 0, ctx->persp_frame.height, ctx->rgb_frame.data,
            ctx->rgb_frame.linesize);
    if (ret < 0) {
        pr_err("cannot convert frame\n");
        return ret;
    }
    to_fp(&ctx->rgb_frame, ctx->rgb_planar_data);
    ni_frame_t *ni_frame = &network_frame->frame.api_frame.data.frame;
    ni_network_convert_tensor_to_data(ni_frame->p_data[0],
            ni_ai_network_layer_size(&network_data->linfo.in_param[0]),
            ctx->rgb_planar_data, ctx->input_width * ctx->input_height * sizeof(float) * 3,
            &network_data->linfo.in_param[0]);
    return 0;
}

static int do_box(struct lpr_network *lpr_network, struct plate_box *pbox,
        struct network_frame *network_frame)
{
    LicRecModelCtx *lic_rec_model = lpr_network->lic_rec_model;

    struct location *location = (struct location *)network_frame->priv_data;

    int tl_idx = 0;
    int tr_idx = 0;
    int bl_idx = 0;
    int br_idx = 0;
    int j;
    for (j = 1; j < 4; j++) {
        if (pbox->landmark[tl_idx][0] + pbox->landmark[tl_idx][1] >
                pbox->landmark[j][0] + pbox->landmark[j][1]) {
            tl_idx = j;
        }
        if (pbox->landmark[br_idx][0] + pbox->landmark[br_idx][1] <
                pbox->landmark[j][0] + pbox->landmark[j][1]) {
            br_idx = j;
        }
        if (pbox->landmark[tr_idx][1] - pbox->landmark[tr_idx][0] >
                pbox->landmark[j][1] - pbox->landmark[j][0]) {
            tr_idx = j;
        }
        if (pbox->landmark[bl_idx][1] - pbox->landmark[bl_idx][0] <
                pbox->landmark[j][1] - pbox->landmark[j][0]) {
            bl_idx = j;
        }
    }
    ni_trace("roi: (%d,%d,%d,%d), landmark: (%f,%f,%f,%f,%f,%f,%f,%f)\n",
            pbox->roi_box.left, pbox->roi_box.top, pbox->roi_box.right, pbox->roi_box.bottom,
            pbox->landmark[0][0], pbox->landmark[0][1],
            pbox->landmark[1][0], pbox->landmark[1][1],
            pbox->landmark[2][0], pbox->landmark[2][1],
            pbox->landmark[3][0], pbox->landmark[3][1]);

    ni_trace("landmark ordered: %d,%d,%d,%d\n", tl_idx, tr_idx, br_idx, bl_idx);
    lic_rec_model->landmark[0][0] = pbox->landmark[tl_idx][0];
    lic_rec_model->landmark[0][1] = pbox->landmark[tl_idx][1];
    lic_rec_model->landmark[1][0] = pbox->landmark[tr_idx][0];
    lic_rec_model->landmark[1][1] = pbox->landmark[tr_idx][1];
    lic_rec_model->landmark[2][0] = pbox->landmark[bl_idx][0];
    lic_rec_model->landmark[2][1] = pbox->landmark[bl_idx][1];
    lic_rec_model->landmark[3][0] = pbox->landmark[br_idx][0];
    lic_rec_model->landmark[3][1] = pbox->landmark[br_idx][1];

    location->top = pbox->roi_box.top;
    location->left = pbox->roi_box.left;
    location->bottom = pbox->roi_box.bottom;
    location->right = pbox->roi_box.right;
    location->frame_index = lpr_network->frame_index;

    return preprocess_licence_plate_rec(lic_rec_model,
            &lpr_network->lic_rec_network->network_data, network_frame,
            &lpr_network->input_ref);
}

static int process_lic_det_result(struct lpr_network *lpr_network,
        struct network_frame *network_frame, AVFrame *in)
{
    LicDetModelCtx *lic_det_model = lpr_network->lic_det_model;
    struct plate_box *plate_box = NULL;
    int roi_num = 0;
    int ret;

    ret = ni_convert_to_tensors(lpr_network->lic_det_network, &network_frame->frame,
            (void **)lic_det_model->yolo_model_ctx.out_tensor);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "cannot convert tensors\n");
        return ret;
    }

    ret = get_licence_plate_boxes(lic_det_model, in->width, in->height, &plate_box, &roi_num);
    if (ret != 0) {
        return ret;
    }

    if (roi_num == 0) {
        return 0;
    }

    int i;
    for (i = 0; i < roi_num; i++) {
        struct plate_box *pbox = &plate_box[i];
        ni_dbg("frame %ld, %d, (%d,%d,%d,%d), lm (%f,%f,%f,%f,%f,%f,%f,%f)\n",
                ((struct frame_req *)network_frame->priv_data)->frame_index, i,
                pbox->roi_box.left, pbox->roi_box.top, pbox->roi_box.right, pbox->roi_box.bottom,
                pbox->landmark[0][0], pbox->landmark[0][1],
                pbox->landmark[1][0], pbox->landmark[1][1],
                pbox->landmark[2][0], pbox->landmark[2][1],
                pbox->landmark[3][0], pbox->landmark[3][1]
              );
    }

    lpr_network->plate_box = plate_box;
    lpr_network->box_num = roi_num;
    lpr_network->running_box = 0;
    lpr_network->frame_index = ((struct frame_req *)network_frame->priv_data)->frame_index;
    av_frame_ref(&lpr_network->input_ref, in);
    return 0;
}

static int infer_lic_rec_frame(struct lpr_network *lpr_network,
        struct network_frame *network_frame)
{
    LicRecModelCtx *ctx = lpr_network->lic_rec_model;
    return ni_set_network_input(lpr_network->lic_rec_network, use_hwframe,
            &network_frame->frame.api_frame, NULL,
            ctx->input_width, ctx->input_height, &network_frame->frame, true);
}

static int recv_lic_rec_pkt(struct lpr_network *lpr_network,
        struct network_frame *network_frame)
{
    NiNetworkContext *lic_rec_network = lpr_network->lic_rec_network;
    LicRecModelCtx *lic_rec_model = lpr_network->lic_rec_model;
    int ret = ni_get_network_output(lic_rec_network, use_hwframe, &network_frame->frame,
            false, true, (uint8_t **)lic_rec_model->out_tensor);
    if (ret != 0) {
        return ret;
    } else {
        int chars_number = get_plate(lic_rec_model, lic_rec_model->plate_result);
        if (chars_number > 0) {
            char result[128];
            int n;
            struct location *location = (struct location *)network_frame->priv_data;
            lic_rec_model->plate_result[chars_number] = '\0';
            n = wcstombs(lic_rec_model->plate_string, lic_rec_model->plate_result,
                    lic_rec_model->plate_str_len - 1);
            if (n > 0) {
                lic_rec_model->plate_string[n] = '\0';
                n = snprintf(result, sizeof(result), "plate result: %ld (%d,%d,%d,%d): %s\n",
                        location->frame_index,
                        location->top, location->left, location->bottom, location->right,
                        lic_rec_model->plate_string);
                result[n] = '\0';
                if (fwrite(result, n, 1, outfile) != 1) {
                    av_log(NULL, AV_LOG_ERROR, "failed to write plate result: %s\n", strerror(errno));
                }
            } else {
                av_log(NULL, AV_LOG_ERROR, "failed to wcstrombs to chars\n");
            }
        }
    }
    return 0;
}

static void put_back_free_network_frame(struct network_frame *network_frame)
{
    network_frame->priv_data = NULL;
    ni_list_add(&network_frame->list, &network_frame_free_list);
}

static void *infer_frames(void *data)
{
    int ret = 0;
    struct frame_req *frame_req = NULL;
    struct network_frame *network_frame = NULL;
    int should_exit = 0;
    int eof = 0;

    struct network_frame *pending_lic_det_frame = NULL;
    struct network_frame *pending_lic_rec_frame = NULL;

    NI_LIST_HEAD(lic_det_frame_list);
    NI_LIST_HEAD(lic_rec_frame_list);
    NI_LIST_HEAD(interval_frame_list);

    while(!should_exit)
    {
        bool idle = false;
        if (ni_list_empty(&lic_det_frame_list) &&
                ni_list_empty(&lic_rec_frame_list) &&
                ni_list_empty(&lic_rec_frame_busy_list) &&
                !pending_lic_det_frame &&
                !pending_lic_rec_frame) {
            idle = true;
        }
        if (end == 3 && ni_list_empty(&codec_workers->frm_list) && idle) {
            break;
        }

        if (pending_lic_det_frame) {
            ret = infer_lic_det_frame(lpr_network, pending_lic_det_frame);
            if (ret == 0) {
                ni_list_add_tail(&pending_lic_det_frame->list, &lic_det_frame_list);
                pending_lic_det_frame = NULL;
            } else if (ret != NIERROR(EAGAIN)) {
                av_log(NULL, AV_LOG_ERROR, "cannot infer lic det frame\n");
                should_exit = 1;
                break;
            }
        } else {
            pthread_mutex_lock(&codec_workers->frm_lock);
            if (ni_list_empty(&codec_workers->frm_list) && idle) {
                pthread_cond_wait(&codec_workers->frm_cond,
                        &codec_workers->frm_lock);
            }

            // send frame to ai
            if ((ni_list_empty(&codec_workers->frm_list)) ||
                    ni_list_empty(&network_frame_free_list)) {
                pthread_mutex_unlock(&codec_workers->frm_lock);
            } else {
                frame_req = ni_list_first_entry(&codec_workers->frm_list, struct frame_req, list);
                ni_list_del(&frame_req->list);
                pthread_mutex_unlock(&codec_workers->frm_lock);

                network_frame = ni_list_first_entry(&network_frame_free_list,
                        struct network_frame, list);
                ni_list_del(&network_frame->list);

                if (!is_eof(frame_req)) {
                    ret = generate_lic_det_model_frame(lpr_network,
                            &network_frame->frame.api_frame,
                            &frame_req->avframe, use_hwframe);
                    if (ret < 0) {
                        av_log(NULL, AV_LOG_ERROR, "cannot generate ai frame\n");
                        should_exit = 1;
                        break;
                    }
                    network_frame->priv_data = frame_req;

                    ret = infer_lic_det_frame(lpr_network, network_frame);
                    if (ret == 0) {
                        ni_list_add_tail(&network_frame->list, &lic_det_frame_list);
                    } else if (ret == NIERROR(EAGAIN)) {
                        pending_lic_det_frame = network_frame;
                    } else {
                        av_log(NULL, AV_LOG_ERROR, "cannot infer lic det frame\n");
                        should_exit = 1;
                        break;
                    }
                } else {
                    network_frame->priv_data = frame_req;
                    ni_list_add_tail(&network_frame->list, &lic_det_frame_list);
                }
            }
        }

        // get packet from ai
        if (!ni_list_empty(&lic_det_frame_list)) {
            network_frame = ni_list_first_entry(&lic_det_frame_list, struct network_frame, list);
            if (is_eof((struct frame_req *)network_frame->priv_data)) {
                ni_list_del(&network_frame->list);
                ni_list_add_tail(&network_frame->list, &interval_frame_list);
            } else {
                ret = recv_lic_det_pkt(lpr_network, network_frame);
                if (ret != 0 && ret != NIERROR(EAGAIN)) {
                    should_exit = 1;
                    break;
                } else if (ret == 0) {
                    ni_list_del(&network_frame->list);
                    ni_list_add_tail(&network_frame->list, &interval_frame_list);
                }
            }
        }

        if (!lpr_network->plate_box && !ni_list_empty(&interval_frame_list)) {
            network_frame = ni_list_first_entry(&interval_frame_list, struct network_frame, list);
            struct frame_req *frame_req = (struct frame_req *)network_frame->priv_data;
            if (is_eof(frame_req)) {
                eof = 1;
            } else {
                ret = process_lic_det_result(lpr_network, network_frame,
                        &frame_req->avframe);
                if (ret != 0) {
                    should_exit = 1;
                    break;
                }

                av_frame_unref(&frame_req->avframe);
            }

            ni_list_del(&network_frame->list);
            free_frame_req(frame_req);
            put_back_free_network_frame(network_frame);
        }

        if (lpr_network->plate_box && !ni_list_empty(&lic_rec_frame_free_list)) {
            network_frame = ni_list_first_entry(&lic_rec_frame_free_list, struct network_frame,
                    list);
            ret = do_box(lpr_network, &lpr_network->plate_box[lpr_network->running_box],
                    network_frame);
            if (ret != 0) {
                should_exit = 1;
                break;
            }
            if (++lpr_network->running_box == lpr_network->box_num) {
                free(lpr_network->plate_box);
                lpr_network->plate_box = NULL;
                lpr_network->box_num = 0;
                av_frame_unref(&lpr_network->input_ref);
            }
            ni_list_del(&network_frame->list);
            ni_list_add_tail(&network_frame->list, &lic_rec_frame_busy_list);
        }

        if (!ni_list_empty(&lic_rec_frame_busy_list)) {
            network_frame = ni_list_first_entry(&lic_rec_frame_busy_list, struct network_frame,
                    list);
            ret = infer_lic_rec_frame(lpr_network, network_frame);
            if (ret != 0) {
                if (!NIERROR(EAGAIN)) {
                    should_exit = 1;
                    break;
                }
            } else {
                ni_list_del(&network_frame->list);
                ni_list_add_tail(&network_frame->list, &lic_rec_frame_list);
            }
        }

        if (!ni_list_empty(&lic_rec_frame_list)) {
            network_frame = ni_list_first_entry(&lic_rec_frame_list, struct network_frame, list);
            ret = recv_lic_rec_pkt(lpr_network, network_frame);
            if (ret != 0 && ret != NIERROR(EAGAIN)) {
                should_exit = 1;
                break;
            } else if (ret == 0) {
                e2e_frame_number++;
                ni_list_del(&network_frame->list);
                ni_list_add(&network_frame->list, &lic_rec_frame_free_list);
            }
        }

        if (ni_list_empty(&lic_rec_frame_list) && ni_list_empty(&lic_rec_frame_busy_list) && eof) {
            should_exit = 1;
            break;
        }
    }

    // avoid main thread block
    while (!ni_list_empty(&codec_workers->frm_list)) {
        pthread_mutex_lock(&codec_workers->frm_lock);
        frame_req = ni_list_first_entry(&codec_workers->frm_list, struct frame_req, list);
        ni_list_del(&frame_req->list);
        pthread_mutex_unlock(&codec_workers->frm_lock);
        free_frame_req(frame_req);
    }
    codec_workers->infer_exit = 1;
    return (void *)0;
}

static void help_usage(void)
{
    printf("Usage: \n"
            "-i | --input             input file path.\n"
            "-o | --output_file       output file path.\n"
            "-d | --decoder           decoder name.\n"
            "-n | --devid             device id.\n"
            "-v | --loglevel          available debug level: warning, info, debug, trace.\n"
            "-b | --bitrate           set bitrate.\n"
            "-t | --lic_det_model     licence detect model file path.\n"
            "-c | --lic_rec_model     licence recognize model file path.\n"
            "-O | --obj_thresh        detection objectness threshold: (0.0,1.0).\n"
            "-N | --nms_thresh        detection nms threshold: (0.0,1.0).\n"
            "-h | --help              print this help information.\n");
}

void setup_loglevel(char *loglevel)
{
    if (loglevel) {
        if (!strcmp(loglevel, "error")) {
            av_log_set_level(AV_LOG_ERROR);
            ni_dev_set_log_level(NI_LVL_ERROR);
        } else if (!strcmp(loglevel, "warning")) {
            av_log_set_level(AV_LOG_WARNING);
            ni_dev_set_log_level(NI_LVL_WARNING);
        } else if (!strcmp(loglevel, "info")) {
            av_log_set_level(AV_LOG_INFO);
            ni_dev_set_log_level(NI_LVL_INFO);
        } else if (!strcmp(loglevel, "debug")) {
            av_log_set_level(AV_LOG_DEBUG);
            ni_dev_set_log_level(NI_LVL_DEBUG);
        } else if (!strcmp(loglevel, "trace")) {
            av_log_set_level(AV_LOG_TRACE);
            ni_dev_set_log_level(NI_LVL_TRACE);
        } else {
            av_log_set_level(AV_LOG_INFO);
            ni_dev_set_log_level(NI_LVL_INFO);
        }
    } else {
        av_log_set_level(AV_LOG_INFO);
        ni_dev_set_log_level(NI_LVL_INFO);
    }
}

static int init_shared_thread_data(struct codec_worker *worker)
{
    int ret;
    int i;

    /* available list notification */
    ret = pthread_mutex_init(&worker->frm_lock, NULL);
    if (ret) {
        return ret;
    }

    ret = pthread_cond_init(&worker->frm_cond, NULL);
    if (ret) {
        goto fail_init_frm_cond;
    }

    ni_init_list_head(&worker->frm_list);

    /* free list notification */
    ret = pthread_mutex_init(&worker->free_lock, NULL);
    if (ret) {
        goto fail_init_free_lock;
    }

    ret = pthread_cond_init(&worker->free_cond, NULL);
    if (ret) {
        goto fail_init_free_cond;
    }

    ni_init_list_head(&worker->free_list);

    pframes = (struct frame_req *)calloc(NI_TRANSCODE_FRAME_NUM, sizeof(struct frame_req));
    if (!pframes) {
        goto fail_alloc_frames;
    }

    for (i = 0; i < NI_TRANSCODE_FRAME_NUM; i++) {
        //pr_log("init frame_req %p\n", &pframes[i]);
        ni_list_add_tail(&pframes[i].list, &worker->free_list);
    }

    return 0;
fail_alloc_frames:
    pthread_cond_destroy(&worker->free_cond);
fail_init_free_cond:
    pthread_mutex_destroy(&worker->free_lock);
fail_init_free_lock:
    pthread_cond_destroy(&worker->frm_cond);
fail_init_frm_cond:
    pthread_mutex_destroy(&worker->frm_lock);
    return ret;
}

static void cleanup_shared_thread_data(struct codec_worker *worker)
{
    if (!ni_list_empty(&worker->frm_list)) {
        av_log(NULL, AV_LOG_ERROR, "frame list is not empty\n");
    }

    free(pframes);

    pthread_mutex_destroy(&worker->frm_lock);
    pthread_cond_destroy(&worker->frm_cond);

    pthread_mutex_destroy(&worker->free_lock);
    pthread_cond_destroy(&worker->free_cond);
}

static int create_ai_thread(struct codec_worker *worker)
{
    int ret;
    pthread_attr_t attr;

    ret = pthread_attr_init(&attr);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "fail to initialize attr: %d.\n", ret);
        return ret;
    }

    ret = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "fail to set attr detachstate: %d.\n", ret);
        goto end;
    }

    ret = pthread_create(&worker->pthid, NULL, infer_frames, 0);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "creat thread failed!, %d\n", ret);
        goto end;
    }
end:
    if (0 != pthread_attr_destroy(&attr)) {
        av_log(NULL, AV_LOG_ERROR, "failed to destroy pthread attr\n");
    }
    return ret;
}

static int destroy_ai_thread(struct codec_worker *worker)
{
    int ret;
    void *result;
    av_log(NULL, AV_LOG_INFO, "try to stop ai thread\n");
    pthread_cond_signal(&worker->frm_cond);
    ret = pthread_join(worker->pthid, &result);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to join ai thread\n");
    }
    if ((int)((long)result)) {
        av_log(NULL, AV_LOG_ERROR, "Bad ai thread result: %d\n", (int)((long)result));
    }

    return 0;
}

static int create_fps_thread(struct codec_worker *worker)
{
    int ret;
    pthread_attr_t attr;

    ret = pthread_attr_init(&attr);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "fail to initialize attr: %d.\n", ret);
        return ret;
    }

    ret = pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_JOINABLE);
    if (ret) {
        av_log(NULL, AV_LOG_ERROR, "fail to set attr detachstate: %d.\n", ret);
        goto end;
    }

    ret = pthread_create(&worker->pthtimeid,NULL,printfps,0);
    if (ret !=0) {
        av_log(NULL, AV_LOG_ERROR, "creat thread failed!\n");
    }
end:
    if (0 != pthread_attr_destroy(&attr)) {
        av_log(NULL, AV_LOG_ERROR, "failed to destroy pthread attr\n");
    }
    return ret;
}

static int destroy_fps_thread(struct codec_worker *worker)
{
    int ret;
    void *result;
    av_log(NULL, AV_LOG_INFO, "try to stop fps thread\n");
    ret = pthread_join(worker->pthtimeid, &result);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to join fps thread\n");
    }
    if ((int)((long)result)) {
        av_log(NULL, AV_LOG_ERROR, "Bad fps thread result: %d\n", (int)((long)result));
    }

    return ret;
}

static struct frame_req *get_free_frame_req()
{
    struct frame_req *frame_req;
wait_free_frame:
    pthread_mutex_lock(&codec_workers->free_lock);
    if (ni_list_empty(&codec_workers->free_list)) {
        pthread_cond_wait(&codec_workers->free_cond, &codec_workers->free_lock);
    }
    if (ni_list_empty(&codec_workers->free_list)) {
        pthread_mutex_unlock(&codec_workers->free_lock);
        goto wait_free_frame;
    }
    frame_req = ni_list_first_entry(&codec_workers->free_list, struct frame_req, list);
    ni_list_del(&frame_req->list);
    pthread_mutex_unlock(&codec_workers->free_lock);

    return frame_req;
}

static void put_busy_frame_req(struct frame_req *frame_req)
{
    pthread_mutex_lock(&codec_workers->frm_lock);
    if (ni_list_empty(&codec_workers->frm_list)) {
        ni_list_add_tail(&frame_req->list, &codec_workers->frm_list);
        pthread_cond_signal(&codec_workers->frm_cond);
    } else {
        ni_list_add_tail(&frame_req->list, &codec_workers->frm_list);
    }
    pthread_mutex_unlock(&codec_workers->frm_lock);
}

int main(int argc, char **argv)
{
    int ret;
    AVPacket packet;

    const char *lic_det_model_file = NULL;
    const char *lic_rec_model_file = NULL;

    float detect_obj_thresh = 0.5;
    float detect_nms_thresh = 0.45;

    struct frame_req *frame_req;

    char *loglevel = NULL;
    int opt;
    int opt_index;
    codec_workers = (struct codec_worker *)calloc(1, sizeof(struct codec_worker));
    if (codec_workers == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate codec workers.\n");
        return EXIT_FAILURE;
    }
    codec_workers->bitrate = 0;
    const char *opt_string = "n:i:d:o:v:t:c:O:N:h";
    static struct option long_options[] = {
        {"input",          required_argument, NULL, 'i'},
        {"output",         required_argument, NULL, 'o'},
        {"decoder",        required_argument, NULL, 'd'},
        {"devid",          required_argument, NULL, 'n'},
        {"loglevel",       required_argument, NULL, 'v'},
        {"lic_det_model",  required_argument, NULL, 't'},
        {"lic_rec_model",  required_argument, NULL, 'c'},
        {"obj_thresh",     required_argument, NULL, 'O'},
        {"nms_thresh",     required_argument, NULL, 'N'},
        {"help",           no_argument,       NULL, 'h'},
        { NULL,            0,                 NULL,  0 },
    };

    while ((opt = getopt_long(argc, argv, opt_string, long_options, &opt_index)) != -1) {
        switch (opt) {

            case 'd':
                codec_workers->decoder_name = optarg;
                break;
            case 'i':
                codec_workers->input_file = optarg;
                break;
            case 'o':
                codec_workers->output_file = optarg;
                break;
            case 'n':
                codec_workers->devid = atoi(optarg);
                break;
            case 'v':
                loglevel = optarg;
                break;
            case 't':
                lic_det_model_file = optarg;
                break;
            case 'c':
                lic_rec_model_file = optarg;
                break;
            case 'O':
                detect_obj_thresh = atof(optarg);
                if (detect_obj_thresh < 0 || detect_obj_thresh >= 1) {
                    av_log(NULL, AV_LOG_ERROR, "invalid detect object thresh\n");
                    return EXIT_FAILURE;
                }
                break;
            case 'N':
                detect_nms_thresh = atof(optarg);
                if (detect_nms_thresh < 0 || detect_nms_thresh >= 1) {
                    av_log(NULL, AV_LOG_ERROR, "invalid detect nms thresh\n");
                    return EXIT_FAILURE;
                }
                break;
            case 'h':
                help_usage();
                return EXIT_SUCCESS;
            default:
                help_usage();
                return EXIT_FAILURE;
        }
    }
    setup_loglevel(loglevel);

    setlocale(LC_ALL, "");

    // Register all formats, codecs, filters
#if LIBAVFORMAT_VERSION_MAJOR < 58
    av_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif
#if LIBAVCODEC_VERSION_MAJOR < 58
    avcodec_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif
#if LIBAVFILTER_VERSION_MAJOR < 7
    avfilter_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif

    if (access(lic_det_model_file, R_OK) != 0) {
        av_log(NULL, AV_LOG_ERROR, "cannot find model: %s\n", lic_det_model_file);
        return EXIT_FAILURE;
    }
    av_log(NULL, AV_LOG_INFO, "lic_det_model %s\n", lic_det_model_file);

    if (access(lic_rec_model_file, R_OK) != 0) {
        av_log(NULL, AV_LOG_ERROR, "cannot find model: %s\n", lic_rec_model_file);
        return EXIT_FAILURE;
    }
    av_log(NULL, AV_LOG_INFO, "lic_rec_model %s\n", lic_rec_model_file);

    if (codec_workers->input_file == NULL) {
        av_log(NULL, AV_LOG_ERROR, "input file name must be specified.\n");
        return EXIT_FAILURE;
    }

    if (codec_workers->output_file == NULL) {
        av_log(NULL, AV_LOG_ERROR, "output file name must be specified.\n");
        return EXIT_FAILURE;
    }

    if (!codec_workers->decoder_name) {
        av_log(NULL, AV_LOG_ERROR, "decoder_name name must be specified.\n");
        return EXIT_FAILURE;
    }

    if (init_shared_thread_data(codec_workers) != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to initialize common data\n");
        return EXIT_FAILURE;
    }

    outfile = fopen(codec_workers->output_file,"w");
    if(!outfile)
    {
        av_log(NULL, AV_LOG_ERROR, "open file fail!\n");
        exit(1);
    }
    // Allocate packet for encoder output
    frame = av_frame_alloc();
    if (!frame) {
        av_log(NULL, AV_LOG_ERROR, "Could not allocate pkt frame");
        exit(1);
    }

    if ((ret = open_input_file()) < 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to initialize decoder session\n");
        goto end;
    }

    av_log(NULL, AV_LOG_INFO, "decode frame %dx%d\n", dec_ctx->width, dec_ctx->height);

    ret = init_network_struct(lic_det_model_file, lic_rec_model_file,
            dec_ctx->width, dec_ctx->height, detect_obj_thresh, detect_nms_thresh);
    if (ret != 0) {
        av_log(NULL, AV_LOG_ERROR, "failed to initialize network struct\n");
        goto end;
    }

    timer_start = av_gettime_relative();

    if ((ret = create_fps_thread(codec_workers)) != 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to create fps thread\n");
        goto end;
    }

    if ((ret = create_ai_thread(codec_workers)) != 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to create ai thread\n");
        goto stop_fpst;
    }

    /* read all packets */
    while (1) {
        if (codec_workers->infer_exit) {
            break;
        }
        if ((ret = av_read_frame(fmt_ctx, &packet)) < 0)
        {
            if(end == 2)
            {
                frame_req = get_free_frame_req();
                set_eof(frame_req);
                put_busy_frame_req(frame_req);
                break;
            }
            end = 1;
        }
        if (packet.stream_index == video_stream_index || end == 1) {
            if(!end)
            {
                ret = avcodec_send_packet(dec_ctx, &packet);
            }
            else{
                ret = avcodec_send_packet(dec_ctx, NULL);
            }

            if (ret < 0 && ret != AVERROR_EOF) {
                av_log(NULL, AV_LOG_ERROR, "Error while sending a packet to the decoder\n");
                break;
            }
            while (ret >= 0 || end == 1) {
                ret = avcodec_receive_frame(dec_ctx, frame);
                if (ret == AVERROR(EAGAIN)) {
                    break;
                }
                else if(ret == AVERROR_EOF){
                    end = 2;
                    break;
                }
                else if (ret < 0) {
                    av_log(NULL, AV_LOG_ERROR, "Error while receiving a frame from the decoder\n");
                    goto stop_ait;
                }

                frame->pts = frame->best_effort_timestamp;

                av_log(NULL, AV_LOG_DEBUG, "decoded frame_number %lu\n", decode_frame_number);

                frame_req = get_free_frame_req();
                frame_req->frame_index = decode_frame_number++;
                ret = av_frame_ref(&frame_req->avframe, frame);
                if (ret < 0) {
                    av_log(NULL, AV_LOG_ERROR, "copy frame to filter failed\n");
                    goto stop_ait;
                }
                //av_log(NULL, AV_LOG_DEBUG, "generate frame %d\n",
                //        ((niFrameSurface1_t *)frame->data[3])->ui16FrameIdx);
                put_busy_frame_req(frame_req);
                av_frame_unref(frame);
            }
        }
        av_packet_unref(&packet);
    }

stop_ait:
    av_log(NULL, AV_LOG_INFO, "main thread end\n");
    end = 3;
    destroy_ai_thread(codec_workers);
stop_fpst:
    stop_fps = true;
    destroy_fps_thread(codec_workers);
end:
    avcodec_free_context(&dec_ctx);
    avcodec_free_context(&enc_ctx);
    avformat_close_input(&fmt_ctx);
    av_frame_free(&frame);
    fclose(outfile);

    cleanup_shared_thread_data(codec_workers);
    free(codec_workers);
    free_network_struct();

    if (ret < 0 && ret != AVERROR_EOF) {
        av_log(NULL, AV_LOG_ERROR, "Error occurred: %s\n", av_err2str(ret));
        exit(1);
    }
    exit(0);
}
