/*
 * Copyright (c) 2010 Nicolas George
 * Copyright (c) 2011 Stefano Sabatini
 * Copyright (c) 2014 Andrey Utkin
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * @file
 * API for ai to create sc model, get output from sc model
 * @example sc_postprocess.c
 *
 * @added by cube.sun@netint.ca
 * use network api to create sc model, process sc model
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "nierrno.h"
#include "scene_classify_postprocess.h"
#include "ni_log.h"


static int get_sc_detections(ni_sc_network_layer_t *l, sc_detection* det_res) {
    float *predictions = l->output;
    float max_args = predictions[0];
    int max_index = 0;
    int i;
    for (i = 0; i < l->classes; i++) {
        if (predictions[i] > max_args) {
            max_args = predictions[i];
            max_index = i;
        }
    }
    float sum = 0;
    for (i = 0; i < l->classes; i++) {
        sum += (exp(predictions[i] - max_args));
    }
    det_res->prob = 1 / sum;
    det_res->index = max_index;
    return 0;
}

static int ni_sc_get_detections(SCModelCtx *ctx) {
    int i, ret;
    //sc output_number = 1
    for (i = 0; i < ctx->output_number; i++) {
        ret = get_sc_detections(&ctx->layers[i], ctx->det_res);
        if (ret != 0) {
            pr_err("failed to get sc detection at layer %d\n", i);
            return ret;
        }
    }
    pr_log("sc detections classes: %d, prob %f\n", ctx->det_res->index, ctx->det_res->prob);
    return 0;
}

static void destroy_sc_model(SCModelCtx *ctx) {
    if (ctx->out_tensor) {
        int i;
        for (i = 0; i < ctx->output_number; i++) {
            if (ctx->out_tensor[i]) {
                free(ctx->out_tensor[i]);
            }
        }
        free(ctx->out_tensor);
        ctx->out_tensor = NULL;
    }
    if (ctx->det_res) {
        free(ctx->det_res);
        ctx->det_res = NULL;
    }
    if (ctx->layers) {
        free(ctx->layers);
        ctx->layers = NULL;
    }
}

static int create_sc_model(SCModelCtx *ctx, ni_network_data_t *network_data,
                           int model_width, int model_height){
    int i, ret = 0;

    ctx->input_width  = network_data->linfo.in_param[0].sizes[0];
    ctx->input_height = network_data->linfo.in_param[0].sizes[1];

    ctx->output_number = network_data->output_num;
    pr_err("input w %d h %d\n", ctx->input_width, ctx->input_height);
    pr_err("output size %d \n", network_data->linfo.out_param[i].sizes[0]);

    ctx->det_res = malloc(sizeof(sc_detection));
    if (ctx->det_res == NULL) {
        pr_err("failed to allocate detection memory\n");
        ret = NIERROR(ENOMEM);
        goto fail;
    }
    ctx->det_res->prob = 0;

    ctx->out_tensor = (uint8_t **)calloc(network_data->output_num,
            sizeof(uint8_t **));
    if (ctx->out_tensor == NULL) {
        pr_err("failed to allocate output tensor bufptr\n");
        ret = NIERROR(ENOMEM);
        goto fail;
    }

    for (i = 0; i < network_data->output_num; i++) {
        ni_network_layer_params_t *p_param = &network_data->linfo.out_param[i];
        ctx->out_tensor[i] =
                (uint8_t *)malloc(ni_ai_network_layer_dims(p_param) * sizeof(float));
        if (ctx->out_tensor[i] == NULL) {
            pr_err("failed to allocate output tensor buffer\n");
            ret = NIERROR(ENOMEM);
            goto fail;
        }
    }

    ctx->layers =
        malloc(sizeof(ni_sc_network_layer_t) * network_data->output_num);
    if (!ctx->layers) {
        pr_err("cannot allocate network layer memory\n");
        ret = NIERROR(ENOMEM);
        goto fail;
    }

    for (i = 0; i < network_data->output_num; i++) {
        ctx->layers[i].classes = network_data->linfo.out_param[i].sizes[0];
        ctx->layers[i].output_number =
            ni_ai_network_layer_dims(&network_data->linfo.out_param[i]);

        ctx->layers[i].output = (float *)ctx->out_tensor[i];

        pr_log("network layer %d: classes %d output_number %d\n", i,
                ctx->layers[i].classes, ctx->layers[i].output_number);
    }
    return ret;
fail:
    destroy_sc_model(ctx);
    return ret;
}

SCModel sc = {
    .create_model       = create_sc_model,
    .destroy_model      = destroy_sc_model,
    .ni_get_detections  = ni_sc_get_detections,
};
