#!/usr/bin/env bash

path="./";
symlink_to_file=false;
file_to_symlink=false;

# on a windows msys2 environment, detect linux symlinks for netint files and replace them with the target of symlink
# this function is primarily used for Netint internal git project where there are symlinks. Netint external releases will not have symlinks.
# this function may only be called when working_directory is top level of FFmpeg folder
# this function only scans for symlinks on this folder level and one level down
# this function only scans for symlinks that begin with '../' or '../../'
replace_symlinks_with_files_on_windows() {
    file_list=( $(find . -type f \( -iname \*ni\*.c -o -iname \*ni\*.h -o -iname \*ni\*.cpp -o -iname \*ni\*.hpp -o -iname \*.sh \)) )
    for file in ${file_list[@]}; do
        if [[ $(head -q -c 6 $file) == "../../" ]]; then
            if stat $(tail -q -c +4 $file) &> /dev/null; then
                # echo "Replaced working symlink: ${file}"
                cp $file $file.symlink
                cp $(tail -q -c +4 $file) $file
            fi
        elif [[ $(head -q -c 3 $file) == "../" ]]; then
            if stat $(cat $file) &> /dev/null; then
                # echo "Replaced working symlink: ${file}"
                cp $file $file.symlink
                cp $(cat $file) $file
            fi
        fi
    done
}

# undo the actions of replace_symlinks_with_files_on_windows() function
# removes ".symlink" suffix from files and replaces pre-existing files with them
restore_symlinks_on_windows() {
    file_list=( $(find . -type f -name \*.symlink ) )
    for file in ${file_list[@]}; do
        mv $file ${file%.*}
    done
}

# parse a flag with an arg in or after it
# $1 flag pattern, $2 entire flag arg, $3 arg after flag arg
# return 1 if path is in second arg (separated by space), else return 0. Store path in $extract_arg_ret
extract_arg () {
    unset extract_arg_ret
    # check valid arg flag
    if [ -n "$(printf "%s" ${2} | grep -Eoh "${1}")" ]; then
        # check if path string is connected by '=' or is in following arg
        if [ -n "$(echo "${2}" | grep -Eoh "${1}=")" ]; then
            arg_str=`printf "%s" "${2}" | grep -Poh "${1}=\K.+"`;
            # trim out leading and trailing quotation marks
            extract_arg_ret=`echo "${arg_str}" | sed -e 's/^\(["'\'']\)//' -e 's/\(["'\'']\)$//'`;
            return 0;
        elif [ -n "$(printf "%s" ${2} | grep -Eoh "^${1}$")" ]; then
            arg_str="${3}";
            # trim out leading and trailing quotation marks
            extract_arg_ret=`printf "%s" "${arg_str}" | sed -e 's/^\(["'\'']\)//' -e 's/\(["'\'']\)$//'`;
            return 1;
        else
            echo "Unknown option '$2', exiting";
            exit 1;
        fi
    else
        echo "Target flag '$1' not found in '$2', exiting"; exit 1;
    fi
}

while [ "$1" != "" ]; do
    case $1 in
        -h | --help)     echo "Usage: ./win_symlink_tool.sh [OPTION]";
                         echo "On windows environment, replace linux symlinks pointing outside of FFmpeg "
                         echo "folder with the files they point to or vice-versa.";
                         echo "This script is primarily used for Netint internal git project where there "
                         echo "are symlinks. Netint external releases will not have symlinks."
                         echo "This script may only be called when working directory is FFmpeg base folder.";
                         echo "Example: ./win_symlink_tool.sh --symlink_to_file";
                         echo;
                         echo "Options:";
                         echo "-h, --help           display this help and exit";
                         echo "--path               absolute path to directory from which to run this script";
                         echo "                     (default: ./)";
                         echo "--symlink_to_file    back-up symlinks with '.symlink' suffix, copy in real"
                         echo "                     files pointed to by symlinks";
                         echo "--file_to_symlink    remove '.symlink' suffix from files with it";
                         exit 0
        ;;
        --symlink_to_file)    symlink_to_file=true
        ;;
        --file_to_symlink)    file_to_symlink=true
        ;;
        --path | --path=*)    extract_arg "\-\-path" "$1" "$2"; prev_rc=$?;
                                              if [ "$prev_rc" -eq 1 ]; then shift; fi
                                              path=$extract_arg_ret
        ;;
        *)               echo "Usage: ./win_symlink_tool.sh [OPTION]...";
                         echo "Try './win_symlink_tool.sh --help' for more information";
                         exit 1
        ;;
    esac
    shift
done

cd $path
if [[ $? != 0 ]]; then
    exit 1
fi

if $symlink_to_file; then
    replace_symlinks_with_files_on_windows
elif $file_to_symlink; then
    restore_symlinks_on_windows
else
    echo "Must select either --symlink_to_file or --file_to_symlink"
    exit 1
fi
