/*
 * Copyright (c) 2010 Nicolas George
 * Copyright (c) 2011 Stefano Sabatini
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * @file
 * API example for decoding and filtering and p2p read
 * @example ni_2D_engine_p2p_test.c
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>

#include <sys/ioctl.h>
#include <sys/mman.h>
#include <sys/poll.h>

#include <libavcodec/avcodec.h>
#include <libavformat/avformat.h>
#include <libavfilter/buffersink.h>
#include <libavfilter/buffersrc.h>
#include <libswscale/swscale.h>
#include <libavutil/imgutils.h>
#include <libavutil/opt.h>
#include <libavutil/time.h>
#include <libavutil/hwcontext_ni_quad.h>
#include <ni_p2p_ioctl.h>

FILE *outfile = NULL;
static AVFormatContext *fmt_ctx;
static AVCodecContext *dec_ctx;
AVFilterContext *buffersink_ctx;
AVFilterContext *buffersrc_ctx;
AVFilterGraph *filter_graph;
static int video_stream_index = -1;

struct codec_worker {
    const char *decoder_name;
    const char *input_file;
    const char *output_file;
    const char *decoder_params;
    const char *filter_dsc;
    const char *input_txt;
    int devid;
};

struct codec_worker *codec_workers = NULL;

static int ni_to_avframe_copy(AVHWFramesContext *hwfc, AVFrame *dst,
                              const ni_frame_t *src) {
    int src_linesize[4], src_height[4];
    int i, h, nb_planes;
    uint8_t *src_line, *dst_line;

    nb_planes = av_pix_fmt_count_planes(hwfc->sw_format);

    switch (hwfc->sw_format) {
    case AV_PIX_FMT_YUV420P:
        src_linesize[0] = FFALIGN(dst->width, 128);
        src_linesize[1] = FFALIGN(dst->width / 2, 128);
        src_linesize[2] = src_linesize[1];
        src_linesize[3] = 0;

        src_height[0] = dst->height;
        src_height[1] = FFALIGN(dst->height, 2) / 2;
        src_height[2] = src_height[1];
        src_height[3] = 0;
        break;

    case AV_PIX_FMT_YUV420P10LE:
        src_linesize[0] = FFALIGN(dst->width * 2, 128);
        src_linesize[1] = FFALIGN(dst->width, 128);
        src_linesize[2] = src_linesize[1];
        src_linesize[3] = 0;

        src_height[0] = dst->height;
        src_height[1] = FFALIGN(dst->height, 2) / 2;
        src_height[2] = src_height[1];
        src_height[3] = 0;
        break;

    case AV_PIX_FMT_NV12:
        src_linesize[0] = FFALIGN(dst->width, 128);
        src_linesize[1] = FFALIGN(dst->width, 128);
        src_linesize[2] = 0;
        src_linesize[3] = 0;

        src_height[0] = dst->height;
        src_height[1] = FFALIGN(dst->height, 2) / 2;
        src_height[2] = 0;
        src_height[3] = 0;
        break;

    case AV_PIX_FMT_NV16:
        src_linesize[0] = FFALIGN(dst->width, 64);
        src_linesize[1] = FFALIGN(dst->width, 64);
        src_linesize[2] = 0;
        src_linesize[3] = 0;

        src_height[0] = dst->height;
        src_height[1] = dst->height;
        src_height[2] = 0;
        src_height[3] = 0;
        break;

    case AV_PIX_FMT_YUYV422:
    case AV_PIX_FMT_UYVY422:
        src_linesize[0] = FFALIGN(dst->width, 16) * 2;
        src_linesize[1] = 0;
        src_linesize[2] = 0;
        src_linesize[3] = 0;

        src_height[0] = dst->height;
        src_height[1] = 0;
        src_height[2] = 0;
        src_height[3] = 0;
        break;

    case AV_PIX_FMT_P010LE:
        src_linesize[0] = FFALIGN(dst->width * 2, 128);
        src_linesize[1] = FFALIGN(dst->width * 2, 128);
        src_linesize[2] = 0;
        src_linesize[3] = 0;

        src_height[0] = dst->height;
        src_height[1] = FFALIGN(dst->height, 2) / 2;
        src_height[2] = 0;
        src_height[3] = 0;
        break;

    case AV_PIX_FMT_RGBA:
    case AV_PIX_FMT_BGRA:
    case AV_PIX_FMT_ABGR:
    case AV_PIX_FMT_ARGB:
    case AV_PIX_FMT_BGR0:
        src_linesize[0] = FFALIGN(dst->width, 16) * 4;
        src_linesize[1] = 0;
        src_linesize[2] = 0;
        src_linesize[3] = 0;

        src_height[0] = dst->height;
        src_height[1] = 0;
        src_height[2] = 0;
        src_height[3] = 0;
        break;

    case AV_PIX_FMT_BGRP:
        src_linesize[0] = FFALIGN(dst->width, 32);
        src_linesize[1] = FFALIGN(dst->width, 32);
        src_linesize[2] = FFALIGN(dst->width, 32);
        src_linesize[3] = 0;

        src_height[0] = dst->height;
        src_height[1] = dst->height;
        src_height[2] = dst->height;
        src_height[3] = 0;
        break;

    default:
        av_log(hwfc, AV_LOG_ERROR, "Unsupported pixel format %s\n",
               av_get_pix_fmt_name(hwfc->sw_format));
        return AVERROR(EINVAL);
    }

    for (i = 0; i < nb_planes; i++) {
        dst_line = dst->data[i];
        src_line = src->p_data[i];

        for (h = 0; h < src_height[i]; h++) {
            memcpy(dst_line, src_line,
                   FFMIN(src_linesize[i], dst->linesize[i]));
            dst_line += dst->linesize[i];
            src_line += src_linesize[i];
        }
    }

    return 0;
}

static int open_input_file()
{
    int i, ret, stream_index = 0;
    const AVCodec *dec;

    if ((ret = avformat_open_input(&fmt_ctx, codec_workers->input_file, NULL, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot open input file\n");
        return ret;
    }

    if ((ret = avformat_find_stream_info(fmt_ctx, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot find stream information\n");
        return ret;
    }

    for (i = 0; i < fmt_ctx->nb_streams; i++) {
        if (fmt_ctx->streams[i]->codecpar->codec_type == AVMEDIA_TYPE_VIDEO) {
            av_log(NULL, AV_LOG_DEBUG, "found first video stream at index %d\n", i);
            stream_index = i;
            break;
        }
    }

    dec = avcodec_find_decoder_by_name(codec_workers->decoder_name);
    if (dec) {
        if (fmt_ctx->streams[stream_index]->codecpar->codec_id != dec->id) {
            av_log(NULL, AV_LOG_ERROR, "codec does not match with stream id\n");
            return AVERROR_DECODER_NOT_FOUND;
        }
    }

    video_stream_index = stream_index;

    /* create decoding context */
    dec_ctx = avcodec_alloc_context3(dec);
    if (!dec_ctx)
        return AVERROR(ENOMEM);

    avcodec_parameters_to_context(dec_ctx, fmt_ctx->streams[video_stream_index]->codecpar);

    char str_devid[4] = {0};
    snprintf(str_devid, sizeof(str_devid), "%d", codec_workers->devid);
    av_opt_set(dec_ctx->priv_data, "dec", str_devid, 0);
    if(codec_workers->decoder_params)
        av_opt_set(dec_ctx->priv_data, "xcoder-params",codec_workers->decoder_params, 0);

    if (dec_ctx->codec_type == AVMEDIA_TYPE_VIDEO)
            dec_ctx->framerate = av_guess_frame_rate(fmt_ctx, fmt_ctx->streams[video_stream_index], NULL);

    //avcodec_parameters_to_context(dec_ctx, fmt_ctx->streams[video_stream_index]->codecpar);
    ret = avcodec_parameters_from_context(fmt_ctx->streams[video_stream_index]->codecpar, dec_ctx);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Failed to copy decoder parameters from codec context for stream #");
        return ret;
    }

    /* init the video decoder */
    if ((ret = avcodec_open2(dec_ctx, dec, NULL)) < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot open video decoder\n");
        return ret;
    }

    av_log(NULL,AV_LOG_INFO,"open input file success!!\n");
    return 0;
}

static int init_filters(const char *filters_descr)
{
    char args[512];
    int ret = 0;
    const AVFilter *buffersrc  = avfilter_get_by_name("buffer");
    const AVFilter *buffersink = avfilter_get_by_name("buffersink");
    AVFilterInOut *outputs = avfilter_inout_alloc();
    AVFilterInOut *inputs  = avfilter_inout_alloc();
    AVRational time_base = fmt_ctx->streams[video_stream_index]->time_base;

    enum AVPixelFormat pix_fmts[] = { AV_PIX_FMT_RGBA, AV_PIX_FMT_BGRP, AV_PIX_FMT_YUV420P, AV_PIX_FMT_NI_QUAD, AV_PIX_FMT_NONE };

    filter_graph = avfilter_graph_alloc();
    if (!outputs || !inputs || !filter_graph) {
        ret = AVERROR(ENOMEM);
        goto end;
    }

    /* buffer video source: the decoded frames from the decoder will be inserted here. */
    snprintf(args, sizeof(args),
             "video_size=%dx%d:pix_fmt=%d:time_base=%d/%d:pixel_aspect=%d/%d",
             dec_ctx->width, dec_ctx->height, dec_ctx->pix_fmt,
             time_base.num, time_base.den,
             dec_ctx->sample_aspect_ratio.num, dec_ctx->sample_aspect_ratio.den);

    ret = avfilter_graph_create_filter(&buffersrc_ctx, buffersrc, "in",
                                       args, NULL, filter_graph);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot create buffer source\n");
        goto end;
    }

    AVBufferSrcParameters *par = av_buffersrc_parameters_alloc();
    if (!par)
        return AVERROR(ENOMEM);
    memset(par, 0, sizeof(*par));
    par->format = AV_PIX_FMT_NONE;
    par->hw_frames_ctx = dec_ctx->hw_frames_ctx;

    ret = av_buffersrc_parameters_set(buffersrc_ctx, par);
    if (ret < 0)
        goto end;
    av_freep(&par);


    /* buffer video sink: to terminate the filter chain. */
    ret = avfilter_graph_create_filter(&buffersink_ctx, buffersink, "out",
                                       NULL, NULL, filter_graph);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot create buffer sink\n");
        goto end;
    }

    ret = av_opt_set_int_list(buffersink_ctx, "pix_fmts", pix_fmts,
                              AV_PIX_FMT_NONE, AV_OPT_SEARCH_CHILDREN);
    if (ret < 0) {
        av_log(NULL, AV_LOG_ERROR, "Cannot set output pixel format\n");
        goto end;
    }

    /*
     * Set the endpoints for the filter graph. The filter_graph will
     * be linked to the graph described by filters_descr.
     */

    /*
     * The buffer source output must be connected to the input pad of
     * the first filter described by filters_descr; since the first
     * filter input label is not specified, it is set to "in" by
     * default.
     */
    outputs->name       = av_strdup("in");
    outputs->filter_ctx = buffersrc_ctx;
    outputs->pad_idx    = 0;
    outputs->next       = NULL;

    /*
     * The buffer sink input must be connected to the output pad of
     * the last filter described by filters_descr; since the last
     * filter output label is not specified, it is set to "out" by
     * default.
     */
    inputs->name       = av_strdup("out");
    inputs->filter_ctx = buffersink_ctx;
    inputs->pad_idx    = 0;
    inputs->next       = NULL;

    if ((ret = avfilter_graph_parse_ptr(filter_graph, filters_descr,
                                    &inputs, &outputs, NULL)) < 0)
        goto end;

    if ((ret = avfilter_graph_config(filter_graph, NULL)) < 0)
    {
        printf("ret = %d\n",ret);
        goto end;
    }

end:
    avfilter_inout_free(&inputs);
    avfilter_inout_free(&outputs);

    return ret;
}

static void help_usage(void)
{
    printf("This example program takes in encoded input, decodes it with a netint HW \n"
           "decoder, then applies netint HW scaling. Raw frame is outputed by hwdownload or p2p transfer.\n"
           "Usage example:\n"
           "  ./ni_2D_engine_p2p_test -i Dinner_1920x1080p30_300.h264 -o outbgrp.rgb"
           "  -d h264_ni_quadra_dec -x out=hw -n 0"
           "  -s ni_quadra_scale=640:360:format=bgrp:is_p2p=0 \n"
           "\n"
           "Params:\n"
           "  -x | --decoder-params    decoder parameters\n"
           "  -i | --input             input file path\n"
           "  -o | --output_file       output file path\n"
           "  -d | --decoder           decoder name\n"
           "  -n | --devid             device id\n"
           "  -v | --loglevel          ffmpeg log level. [quiet, panic, fatal, error,\n"
           "                           warning, info, verbose, debug, trace]\n"
           "  -s | --filter_dsc        filter params,like FFmpeg -vf. Note: do not use hwdownload if p2p transfer is expected\n"
           "  -t | --filetxt           input filetext,the file write many picture path.\n"
           "  -h | --help              print this help information\n");
}

void setup_loglevel(char *loglevel)
{
    int i;
    const struct { const char *name; int level; } log_levels[] =
    {
         { "quiet"  , AV_LOG_QUIET   },
         { "panic"  , AV_LOG_PANIC   },
         { "fatal"  , AV_LOG_FATAL   },
         { "error"  , AV_LOG_ERROR   },
         { "warning", AV_LOG_WARNING },
         { "info"   , AV_LOG_INFO    },
         { "verbose", AV_LOG_VERBOSE },
         { "debug"  , AV_LOG_DEBUG   },
         { "trace"  , AV_LOG_TRACE   },
    };

    for (i = 0; i < (sizeof(log_levels) / sizeof((log_levels)[0])); i++)
    {
        if (!strcmp(log_levels[i].name, loglevel))
        {
            av_log_set_level(log_levels[i].level);
        }
    }
}

static void print_report(int is_last_report, int64_t timer_start,
        int64_t cur_time, unsigned long frame_number)
{
    static int64_t last_time = -1;
    float t;
    char buf[1024];
    float fps;

    if (!is_last_report) {
        if (last_time == -1) {
            last_time = cur_time;
            return;
        }
        if ((cur_time - last_time) < 500000)
            return;
        last_time = cur_time;
    }

    t = (cur_time - timer_start) / 1000000.0;

    fps = t > 0.5 ? frame_number / t : 0;

    const char end = is_last_report ? '\n' : '\r';

    buf[0] = '\0';
    snprintf(buf + strlen(buf), sizeof(buf) - strlen(buf), "frame=%5ld fps=%3.*f ",
            frame_number, (fps < 9.95), fps);
    fprintf(stderr, "%s   %c", buf, end);
    fflush(stderr);
}

void write_frame(AVFrame *frame)
{
    if(!frame)
    {
        av_log(NULL,AV_LOG_ERROR,"frame is NULL!!\n");
        exit(1);
    }
    int j;
    switch (frame->format)
    {
    case AV_PIX_FMT_BGRP:
        //save BGRP
        fwrite(frame->data[0],1,frame->width*frame->height,outfile);
        fwrite(frame->data[1],1,frame->width*frame->height,outfile);
        fwrite(frame->data[2],1,frame->width*frame->height,outfile);
        break;
    case AV_PIX_FMT_YUV420P:
        // write YUV420P
        for (j = 0; j < frame->height; j++)
        {
            fwrite(frame->data[0] + j * frame->linesize[0], frame->width, 1, outfile);
        }
        for (j = 0; j < frame->height/2; j++)
        {
            fwrite(frame->data[1] + j * frame->linesize[1], frame->width/2, 1, outfile);
        }
        for (j = 0; j < frame->height/2; j++)
        {
            fwrite(frame->data[2] + j * frame->linesize[2], frame->width/2, 1, outfile);
        }
        break;
    case AV_PIX_FMT_RGBA:
        //write RGBA
        fwrite(frame->data[0], 1, frame->width*frame->height * 4,outfile);
        break;
    default:
        break;
    }
}

static int create_buffer(AVFrame *frame, int width, int height, int format)
{
    if (frame->data[0] != NULL)
        av_frame_free(&frame);
    frame->width = width;
    frame->height = height;
    frame->format = format;
    int size = av_image_get_buffer_size(frame->format, frame->width, frame->height, 1);
    uint8_t *buffer = (uint8_t *)av_malloc(size);
    if (!buffer) {
        av_log(NULL, AV_LOG_ERROR, "%s Failed to allocate AVFrame\n", __func__);
        return AVERROR(ENOMEM);
    }
    av_log(NULL, AV_LOG_DEBUG, "%s: Allocated %u to receive output frame\n", __func__, size);
    av_image_fill_arrays(frame->data, frame->linesize, buffer, frame->format, frame->width, frame->height, 1);
    return 0;
}

static int retrieve_filter_frame(AVFrame *dst, AVFrame *src)
{
    AVHWFramesContext *hwfc = (AVHWFramesContext *)src->hw_frames_ctx->data;
    AVNIFramesContext *ctx = hwfc->hwctx;
    ni_session_data_io_t session_io_data;
    ni_session_data_io_t *p_session_data = &session_io_data;
    niFrameSurface1_t *src_surf          = (niFrameSurface1_t *)src->data[3];
    int ret;
    int pixel_format;

    memset(&session_io_data, 0, sizeof(ni_session_data_io_t));
    av_log(hwfc, AV_LOG_VERBOSE,
           "%s handle %d trace ui16FrameIdx = [%d] SID %d\n", __func__,
           src_surf->device_handle, src_surf->ui16FrameIdx,
           src_surf->ui16session_ID);

    if (!src_surf->ui16FrameIdx) {
        av_log(hwfc, AV_LOG_ERROR, "ERROR: Unexpected to get sw frame, exit. This may caused by hwdownload, remove it and retry\n");
        ni_frame_buffer_free(&p_session_data->data.frame);
        return -1;
    }

    av_log(hwfc, AV_LOG_DEBUG, "%s hwdl processed h/w = %d/%d\n", __func__,
           src->height, src->width);

    switch (hwfc->sw_format) {
    case AV_PIX_FMT_YUV420P:
        pixel_format = NI_PIX_FMT_YUV420P;
        break;
    case AV_PIX_FMT_YUV420P10LE:
        pixel_format = NI_PIX_FMT_YUV420P10LE;
        break;
    case AV_PIX_FMT_NV12:
        pixel_format = NI_PIX_FMT_NV12;
        break;
    case AV_PIX_FMT_NV16:
        pixel_format = NI_PIX_FMT_NV16;
        break;
    case AV_PIX_FMT_YUYV422:
        pixel_format = NI_PIX_FMT_YUYV422;
        break;
    case AV_PIX_FMT_UYVY422:
        pixel_format = NI_PIX_FMT_UYVY422;
        break;
    case AV_PIX_FMT_P010LE:
        pixel_format = NI_PIX_FMT_P010LE;
        break;
    case AV_PIX_FMT_RGBA:
        pixel_format = NI_PIX_FMT_RGBA;
        break;
    case AV_PIX_FMT_BGRA:
        pixel_format = NI_PIX_FMT_BGRA;
        break;
    case AV_PIX_FMT_ABGR:
        pixel_format = NI_PIX_FMT_ABGR;
        break;
    case AV_PIX_FMT_ARGB:
        pixel_format = NI_PIX_FMT_ARGB;
        break;
    case AV_PIX_FMT_BGR0:
        pixel_format = NI_PIX_FMT_BGR0;
        break;
    case AV_PIX_FMT_BGRP:
        pixel_format = NI_PIX_FMT_BGRP;
        break;
    default:
        av_log(hwfc, AV_LOG_ERROR, "Pixel format not supported.\n");
        return AVERROR(EINVAL);
    }

    ret = ni_frame_buffer_alloc_dl(&(p_session_data->data.frame), src->width,
                                   src->height, pixel_format);
    if (ret != NI_RETCODE_SUCCESS) {
        av_log(hwfc, AV_LOG_ERROR, "%s Cannot allocate ni_frame\n", __func__);
        return AVERROR(ENOMEM);
    }

    if (ctx->api_ctx.isP2P)
    {
        ret = ni_scaler_p2p_frame_acquire(&ctx->api_ctx, src_surf, p_session_data->data.frame.buffer_size);
        if (ret)
        {
            av_log(hwfc, AV_LOG_ERROR, "%s failed to acquire p2p frame\n", __func__);
            ni_frame_buffer_free(&p_session_data->data.frame);
            return AVERROR_EXTERNAL;
        }

        uint8_t *mmap_data = mmap(0, p_session_data->data.frame.buffer_size, PROT_READ | PROT_WRITE, MAP_SHARED, src_surf->dma_buf_fd, 0);
        if (mmap_data == MAP_FAILED)
        {
            av_log(hwfc, AV_LOG_ERROR, "%s failed to mmap dmabuf : %s\n", __func__, strerror(errno));
            ni_frame_buffer_free(&p_session_data->data.frame);
            return AVERROR_EXTERNAL;
        }

        struct netint_iocmd_issue_request uis;
        memset(&uis, 0, sizeof(uis));
        uis.fd = src_surf->dma_buf_fd;
        uis.data = p_session_data->data.frame.p_buffer;
        uis.len = p_session_data->data.frame.buffer_size;
        uis.dir = NI_DMABUF_READ_FROM_DEVICE;
        memset(uis.data, 0, uis.len);
        ret = ioctl(ctx->api_ctx.netint_fd, NETINT_IOCTL_ISSUE_REQ, &uis);
        if (ret < 0)
        {
            av_log(hwfc, AV_LOG_ERROR, "Failed to send req: %s\n", strerror(errno));
            ni_frame_buffer_free(&p_session_data->data.frame);
            munmap((void *)mmap_data,p_session_data->data.frame.buffer_size);
            return AVERROR(ENOMEM);
        }

        // poll to check if read complete
        struct pollfd pfds[1];
        pfds[0].fd = src_surf->dma_buf_fd;
        pfds[0].events = POLLOUT;
        pfds[0].revents = 0;
        ret = poll(pfds, 1, -1);
        if (ret < 0)
        {
            av_log(hwfc, AV_LOG_ERROR, "Failed to poll\n");
            ni_frame_buffer_free(&p_session_data->data.frame);
            munmap((void *)mmap_data,p_session_data->data.frame.buffer_size);
            return AVERROR_EXTERNAL;
        }

        munmap((void *)mmap_data,p_session_data->data.frame.buffer_size);
    }
    else
    {
        ctx->api_ctx.is_auto_dl = false;
        ret = ni_device_session_hwdl(&ctx->api_ctx, p_session_data, src_surf);
        if (ret <= 0) {
            av_log(hwfc, AV_LOG_DEBUG, "%s failed to retrieve frame\n", __func__);
            ni_frame_buffer_free(&p_session_data->data.frame);
            return AVERROR_EXTERNAL;
        }
    }

    if ((dst->width != src->width) || (dst->height != src->height) || (dst->format != hwfc->sw_format))
    {
        // need to reallocate output buffer
        ret = create_buffer(dst, src->width, src->height, hwfc->sw_format);
        if (ret < 0) {
            av_log(hwfc, AV_LOG_ERROR, "Can't realloc output frame %d\n", ret);
            return ret;
        }
    }
    ret = ni_to_avframe_copy(hwfc, dst, &p_session_data->data.frame);
    if (ret < 0) {
        av_log(hwfc, AV_LOG_ERROR, "Can't copy frame %d\n", ret);
        ni_frame_buffer_free(&p_session_data->data.frame);
        return ret;
    }

    av_frame_copy_props(dst, src);
    ni_frame_buffer_free(&p_session_data->data.frame);

    return 0;
}

int main(int argc, char **argv)
{
    int ret;
    AVPacket packet;
    AVFrame *frame;
    AVFrame *filt_frame;
    AVFrame *out_frame;
    char *loglevel = NULL;
    char tmpfile[256] = {0};
    char filename[256] = {0};
    FILE *fopentxt;
    int opt;
    int opt_index;
    int end = 0;
    int decode_num = 0;
    int64_t timer_start;
    int input=0;
    int intext=0;
    int resend_packet = 0;

    codec_workers = malloc(sizeof(struct codec_worker));
    if (codec_workers == NULL) {
        av_log(NULL, AV_LOG_ERROR, "failed to allocate codec workers.\n");
        goto the_end;
    }
    memset(codec_workers, 0, sizeof(struct codec_worker));
    codec_workers->devid = -1;

    const char *opt_string = "n:x:i:d:o:v:s:t:h";
    static struct option long_options[] = {
        {"input",          required_argument, NULL, 'i'},
        {"input_txt",      required_argument, NULL, 't'},
        {"output",         required_argument, NULL, 'o'},
        {"decoder",        required_argument, NULL, 'd'},
        {"decoder-params", required_argument, NULL, 'x'},
        {"devid",          required_argument, NULL, 'n'},
        {"loglevel",       required_argument, NULL, 'v'},
        {"filter_dsc",     required_argument, NULL, 's'},
        {"help",           no_argument,       NULL, 'h'},
        { NULL,            0,                 NULL,  0 },
    };

    while ((opt = getopt_long(argc, argv, opt_string, long_options, &opt_index)) != -1) {
        switch (opt) {

            case 'd':
                codec_workers->decoder_name = optarg;
                break;
            case 'i':
                codec_workers->input_file = optarg;
                input=1;
                break;
            case 't':
                codec_workers->input_txt = optarg;
                intext=1;
                break;
            case 'o':
                codec_workers->output_file = optarg;
                break;
            case 'n':
                codec_workers->devid = atoi(optarg);
                break;
            case 'x':
                codec_workers->decoder_params = optarg;
                break;
            case 'v':
                loglevel = optarg;
                break;
            case 's':
                codec_workers->filter_dsc = optarg;
                break;
            case 'h':
                help_usage();
                return EXIT_SUCCESS;
            default:
                help_usage();
                return EXIT_FAILURE;
        }
    }
    if (loglevel)
    {
        setup_loglevel(loglevel);
    }

    if(input == intext)
    {
    if(input == 0)
        av_log(NULL, AV_LOG_ERROR, "input file name must be specified.\n");
    else
        av_log(NULL, AV_LOG_ERROR, "input -i and -t cannot use together.input = %d,intext=%d\n",input,intext);
        return EXIT_FAILURE;
    }

    if (codec_workers->output_file == NULL)
    {
        av_log(NULL, AV_LOG_ERROR, "output file name must be specified.\n");
        return EXIT_FAILURE;
    }

    if (!codec_workers->decoder_name)
    {
        av_log(NULL, AV_LOG_ERROR, "decoder_name name must be specified.\n");
        return EXIT_FAILURE;
    }

    // Register all formats, codecs, filters
#if LIBAVFORMAT_VERSION_MAJOR < 58
    av_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif
#if LIBAVCODEC_VERSION_MAJOR < 58
    avcodec_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif
#if LIBAVFILTER_VERSION_MAJOR < 7
    avfilter_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif

    if(intext)
    {
        fopentxt = fopen(codec_workers->input_txt,"r");
        if(!fopentxt)
        {
            printf("open file %s fail!\n",codec_workers->input_txt);
            exit(1);
        }
        if(fgets(filename,sizeof(filename),fopentxt) == NULL)
        {
            av_log(NULL,AV_LOG_ERROR,"The input filetext is NULL\n");
        }
        memcpy(tmpfile,filename,strlen(filename)-1);
        //char fil[56] = "1.jpeg";
        codec_workers->input_file = tmpfile;
    }
    outfile = fopen(codec_workers->output_file,"wb");
    if(!outfile)
    {
        av_log(NULL,AV_LOG_ERROR,"open file %s fail!\n",codec_workers->output_file);
        exit(1);
    }
    filt_frame = av_frame_alloc();
    out_frame = av_frame_alloc();
    if (!filt_frame || !out_frame)
    {
        av_log(NULL,AV_LOG_ERROR,"Could not allocate frame");
        exit(1);
    }
    if ((ret = open_input_file()) < 0)
        goto the_end;

    timer_start = av_gettime_relative();
    while(1)
    {
        frame = av_frame_alloc();
        if (!frame)
        {
            av_log(NULL,AV_LOG_ERROR,"Could not allocate frame");
            exit(1);
        }
        /* read all packets */
        while (1) {
            if (!resend_packet && ((ret = av_read_frame(fmt_ctx, &packet)) < 0))
            {
                if(end == 2) {
                    av_log(NULL, AV_LOG_DEBUG, "Received eos, exit!\n");
                    break;
                }
                end = 1;
            }
            if (end == 1 || packet.stream_index == video_stream_index) {
                if(!end)
                    ret = avcodec_send_packet(dec_ctx, &packet);
                else
                    ret = avcodec_send_packet(dec_ctx, NULL);

                if (ret < 0 && ret != AVERROR_EOF && ret != AVERROR(EAGAIN)) {
                    av_log(NULL, AV_LOG_ERROR, "Error while sending a packet to the decoder. Error: %s\n", av_err2str(ret));
                    break;
                }
                if (ret == AVERROR(EAGAIN))
                {
                    resend_packet = 1;
                }
                else
                {
                    resend_packet = 0;
                }
                while (ret >= 0 || end == 1 || resend_packet) {
                    ret = avcodec_receive_frame(dec_ctx, frame);
                    if (ret == AVERROR(EAGAIN)) {
                        break;
                    }
                    else if(ret == AVERROR_EOF){
                        end = 2;
                        break;
                    }
                    else if (ret < 0) {
                        av_log(NULL, AV_LOG_ERROR, "Error while receiving a frame from the decoder\n");
                        goto the_end;
                    }
                    frame->pts = frame->best_effort_timestamp;
                    decode_num++;
                    print_report(0, timer_start, av_gettime_relative(), decode_num);

                    if (decode_num == 1) {
                        // create filter graph once we have the input frame
                        if ((ret = init_filters(codec_workers->filter_dsc)) < 0)
                            goto the_end;
                    }

                    /* push the decoded frame into the filtergraph */
                    if (av_buffersrc_add_frame_flags(buffersrc_ctx, frame, AV_BUFFERSRC_FLAG_KEEP_REF) < 0) {
                        av_log(NULL, AV_LOG_ERROR, "Error while feeding the filtergraph\n");
                        break;
                    }

                    /* pull filtered frames from the filtergraph */
                    while (1) {
                        ret = av_buffersink_get_frame(buffersink_ctx, filt_frame);
                        if (ret == AVERROR(EAGAIN))
                        {
                            break;
                        }
                        else if(ret == AVERROR_EOF)
                        {
                            end = 2;
                            break;
                        }
                        if (ret < 0)
                            goto the_end;

                        ret = retrieve_filter_frame(out_frame, filt_frame);
                        if (ret)
                        {
                            av_log(NULL, AV_LOG_ERROR, "Error while retrieve_filter_frame\n");
                            goto the_end;
                        }
                        else
                            av_log(NULL, AV_LOG_DEBUG, "retrieve_filter frame success\n");
                        write_frame(out_frame);
                        av_frame_unref(filt_frame);
                    }
                    av_frame_unref(frame);
                }
            }
            if (!resend_packet)
            {
                av_packet_unref(&packet);
            }
        }

        av_frame_free(&frame);
        avcodec_free_context(&dec_ctx);
        avformat_close_input(&fmt_ctx);
        if(intext)
        {
            memset(filename,0,sizeof(filename));
            if(fgets(filename,sizeof(filename),fopentxt) == NULL)
            {
                break;
            }
            memset(tmpfile,0,sizeof(tmpfile));
            memcpy(tmpfile,filename,strlen(filename)-1);
            codec_workers->input_file = tmpfile;
            if ((ret = open_input_file()) < 0)
            {
                av_log(NULL,AV_LOG_ERROR,"seccond init dec failed!!!!!!\n");
            }
            end=0;
        }
        else
        {
            break;
        }
    }
    print_report(1, timer_start, av_gettime_relative(), decode_num);

the_end:
    avfilter_graph_free(&filter_graph);
    av_frame_free(&frame);
    av_frame_free(&filt_frame);
    av_frame_free(&out_frame);
    fclose(outfile);
    if(intext)
        fclose(fopentxt);
    if (ret < 0 && ret != AVERROR_EOF) {
        fprintf(stderr, "Error occurred: %s\n", av_err2str(ret));
        exit(1);
    }
    av_log(NULL,AV_LOG_INFO,"Done.\n");
    return 0;
}
