/*******************************************************************************
 *
 * Copyright (C) 2022 NETINT Technologies
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 ******************************************************************************/

/**
 * @file
 * video decoding with libavcodec API example
 *
 * @example ni_demuxing_decoding.c
 *
 * @added by tom.han@netint.ca
 * Video decoding with Netint codec through libavcodec
 */

#include <libavcodec/avcodec.h>
#include <libavformat/avformat.h>
#include <ni_device_api.h>
#include <libavutil/time.h>

/* Write an AV_PIX_FMT_YUV420P AVframe to file as YUV420P */
static void write_save(AVFrame *frame, int width, int height, FILE* f)
{
  int i;

  if (frame->format == AV_PIX_FMT_YUV420P10LE ||
      frame->format == AV_PIX_FMT_YUV420P10BE ||
      frame->format == AV_PIX_FMT_YUV420P10)
  {
    width *= 2;
  }

  for (i = 0; i < height; i++)
    fwrite(frame->data[0] + i * frame->linesize[0], 1, width, f);
  for (i = 0; i < height / 2; i++)
    fwrite(frame->data[1] + i * frame->linesize[1], 1, width / 2, f);
  for (i = 0; i < height / 2; i++)
    fwrite(frame->data[2] + i * frame->linesize[2], 1, width / 2, f);
}

/* Send one packet to decoder, attempt to read all available decoded frames from decoder,
   saving YUV to file. If run with with *pkt=NULL, flush the decoder of remaining frames
   until decoder reports EOF */
static void decode(AVCodecContext *dec_ctx, AVFrame *frame, AVPacket *pkt,
                   const char *filename, FILE* output_f, int64_t stime)
{
  int ret;
  float fps, etime;

  // Send a frame packet to decoder
  ret = avcodec_send_packet(dec_ctx, pkt);
  if (ret < 0)
  {
    fprintf(stderr, "Error sending a packet for decoding\n");
    exit(1);
  }

  /* Run this loop if a packet was sucessfully sent during avcodec_send_packet() OR
     no packet was sent because this function is in flushing mode OR
     a frame was sucessfully received by avcodec_receive_frame() */
  while (ret >= 0 || ! pkt)
  {
    // Attempt to retreive a decoded frame
    ret = avcodec_receive_frame(dec_ctx, frame);
    // Failed to get a frame because decoder output buffer is empty
    if (ret == AVERROR(EAGAIN))
    {
      // If function not in flushing mode
      if (pkt)
        return; // Exit this function to send next packet
      else
        continue; // Loop in this function to flush decoder
    }
    // Decoder reports EOF, flushing is complete
    else if (ret == AVERROR_EOF)
    {
      fprintf(stderr, "\ndecoder EOF reached\n");
      return;
    }
    // Decoding error occured, possibly bad packet in input
    else if (ret < 0)
    {
      fprintf(stderr, "Error during decoding. RC=%d\n", ret);
      exit(1);
    }

    // Report statistics regarding processing framerate and frame resolution/size
    etime = (av_gettime_relative() - stime) / 1000000.0;

#if LIBAVCODEC_VERSION_MAJOR >= 61 //7.0
    fps = etime > 0.5 ? dec_ctx->frame_num / etime : 0;
    printf("Frame= %3ld, frame resolution=%dx%d, size=%d, fps=%3.*f\r",
            dec_ctx->frame_num, frame->width, frame->height, frame->linesize[0], (fps < 9.95), fps);
#else
    fps = etime > 0.5 ? dec_ctx->frame_number / etime : 0;
    printf("Frame= %3d, frame resolution=%dx%d, size=%d, fps=%3.*f\r",
           dec_ctx->frame_number, frame->width, frame->height, frame->linesize[0], (fps < 9.95), fps);
#endif
    fflush(stdout);

    // Write raw frame to file
    write_save(frame, frame->width, frame->height, output_f);
  }
}

/* For an AVFormatContext, select first stream of AVMediaType and setup its decoder
   AVCodecContext */
static int open_codec_context(int *stream_idx, AVCodecContext **dec_ctx,
                              AVFormatContext *fmt_ctx, enum AVMediaType type)
{
  int i, ret, stream_index;
  AVStream *st;
  const AVCodec *dec = NULL;
  const AVCodecDescriptor *desc = NULL;

  printf("open_codec_context()\n");
  ret = avformat_find_stream_info(fmt_ctx, NULL);
  if (ret < 0)
  {
    fprintf(stderr, "Could not find %s stream in input file \n",
            av_get_media_type_string(type));
    return ret;
  }
  else
  {
    // Find lowest indexed video stream
    for (i = 0; i < fmt_ctx->nb_streams; i++)
    {
      if (fmt_ctx->streams[i]->codecpar->codec_type == type)
      {
        printf("found first %s stream at index %d\n", av_get_media_type_string(type), i);
        stream_index = i;
        break;
      }
    }
    st = fmt_ctx->streams[stream_index];

    // Choose appropriate Netint HW decoder for video stream
    switch(st->codecpar->codec_id)
    {
      case AV_CODEC_ID_H264:
        dec = avcodec_find_decoder_by_name("h264_ni_quadra_dec");
        break;
      case AV_CODEC_ID_HEVC:
        dec = avcodec_find_decoder_by_name("h265_ni_quadra_dec");
        break;
      case AV_CODEC_ID_MJPEG:
        dec = avcodec_find_decoder_by_name("jpeg_ni_quadra_dec");
        break;
      case AV_CODEC_ID_VP9:
        dec = avcodec_find_decoder_by_name("vp9_ni_quadra_dec");
        break;
      default:
        fprintf(stderr, "Netint decoder does not support this codec. ");
        desc = avcodec_descriptor_get(st->codecpar->codec_id);
        desc ? fprintf(stderr, "Name: %s\n", desc->name) : fprintf(stderr, "ID: %d\n", st->codecpar->codec_id);
        return AVERROR(EINVAL);
    }
    if (!dec)
    {
      fprintf(stderr, "Failed to find %s codec\n", av_get_media_type_string(type));
      return AVERROR(EINVAL);
    }

    // Allocate a codec context for the decoder
    *dec_ctx = avcodec_alloc_context3(dec);
    if (!*dec_ctx)
    {
      fprintf(stderr, "Failed to allocate the %s codec context\n",
              av_get_media_type_string(type));
      return AVERROR(ENOMEM);
    }

    // Copy codec parameters from input stream to output codec context
    if ((ret = avcodec_parameters_to_context(*dec_ctx, st->codecpar)) < 0)
    {
      fprintf(stderr, "Failed to copy %s codec parameters to decoder context\n",
              av_get_media_type_string(type));
      return ret;
    }

    if ((ret = avcodec_open2(*dec_ctx, dec, NULL)) < 0)
    {
      fprintf(stderr, "Failed to open %s codec\n", av_get_media_type_string(type));
      return ret;
    }
    *stream_idx = stream_index;
  }
  return 0;
}

int main(int argc, char **argv)
{
  const char *filename, *outfilename;
  AVCodecContext *codec_ctx= NULL;
  AVFrame *frame;
  AVPacket *pkt;
  FILE *output_f;
  int64_t stime;
  AVFormatContext *fmt_ctx = NULL;
  int video_stream_idx = -1;

  av_log_set_level(AV_LOG_INFO);

  // Process input args
  if (argc != 3)
  {
    fprintf(stderr, "Usage: %s <input_file> <output_file>\n", argv[0]);
    fprintf(stderr, "Ensure input_file contains an h264/h265 stream whose first "
                    "I-frame is near the beginning of the binary.\n");
    exit(0);
  }
  filename    = argv[1];
  outfilename = argv[2];

  // Allocate packet buffer
  pkt = av_packet_alloc();
  if (!pkt)
  {
    fprintf(stderr, "Could not av_packet_alloc()\n");
    exit(1);
  }
  // Allocate frame buffer
  frame = av_frame_alloc();
  if (!frame)
  {
    fprintf(stderr, "Could not allocate video frame\n");
    exit(1);
  }

  // Check output file is writeable
  output_f = fopen(outfilename, "w");
  if (!output_f)
  {
    fprintf(stderr, "Could not open %s\n", outfilename);
    exit(1);
  }

#if LIBAVFORMAT_VERSION_MAJOR < 58
  // Register all formats and codecs
  av_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif

  // Open input file and allocate format context
  if (avformat_open_input(&fmt_ctx, filename, NULL, NULL) < 0)
  {
    fprintf(stderr, "Could not open source file %s\n", filename);
    exit(1);
  }

  // Setup decoder context for video stream from input_file
  if (open_codec_context(&video_stream_idx, &codec_ctx, fmt_ctx, AVMEDIA_TYPE_VIDEO) < 0)
  {
    fprintf(stderr, "Could not open codec context\n");
    exit(1);
  }
  // Print informating regarding input file to stderr
  av_dump_format(fmt_ctx, 0, filename, 0);

  // read frames from file and decode them
  stime = av_gettime_relative();
  while (av_read_frame(fmt_ctx, pkt) >= 0)
  {
    AVPacket orig_pkt = *pkt;
    if (pkt->stream_index == video_stream_idx)
      decode(codec_ctx, frame, pkt, outfilename, output_f, stime);
    av_packet_unref(&orig_pkt);
  }
  printf("Stopped reading stream\n");

  // flush the decoder
  printf("Flushing decoder\n");
  decode(codec_ctx, frame, NULL, outfilename, output_f, stime);

  // teardown
  avformat_close_input(&fmt_ctx);
  fclose(output_f);
  avcodec_free_context(&codec_ctx);
  av_frame_free(&frame);
  av_packet_free(&pkt);

  return 0;
}
