/*
 * Copyright (c) 2001 Fabrice Bellard
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */

/**
 * @file
 * video encoding with libavcodec API example
 *
 * @example ni_encode_video.c
 *
 * @added by zheng.lv@netint.ca
 * Video encoding with Netint codec through libavcodec
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef _WIN32
#include <windows.h>
#include <process.h>
#include <tchar.h>
#include <time.h>
#else
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#endif
#include <getopt.h>
#include <libavcodec/avcodec.h>
#include <libavutil/opt.h>
#include <libavutil/imgutils.h>
#include <libavutil/time.h>
#include <libavutil/frame.h>
#include <libswscale/swscale.h>
#include <libavfilter/buffersink.h>
#include <libavfilter/buffersrc.h>
#include <ni_device_api.h>

#define LOG_NONE  -8
#define LOG_FATAL 8
#define LOG_ERROR 16
#define LOG_INFO  32
#define LOG_DEBUG 48
#define LOG_TRACE 56

// define the mode of interval
#define NI_INTERVAL_INVALID       0
#define NI_FIXED_INTERVAL_MODE    1
#define NI_RANDOM_INTERVAL_MODE   2

/*!*************************************/
// logging utility
/*!******************************************************************************
 *  \brief  Print log message depending on verbosity level.
 *
 *  \param  level   log level (LOG_NONE, LOG_FATAL, LOG_ERROR, LOG_INFO, LOG_DEBUG, LOG_TRACE)
 *  \param  format  fprintf format specifier
 *  \param  ...     additional arguments
 *
 *  \return
 *******************************************************************************/
# define pr_log(level, format, ...)                   \
({                                                    \
  if (level <= log_level)                             \
  {                                                   \
    if (level == LOG_TRACE)                           \
    {                                                 \
      fprintf(stdout, "[%" PRId64 "] ", get_utime()); \
    }                                                 \
    else if (level < LOG_INFO)                        \
    {                                                 \
      fprintf(stderr, format, ##__VA_ARGS__);         \
    }                                                 \
    else                                              \
    {                                                 \
      fprintf(stdout, format, ##__VA_ARGS__);         \
    }                                                 \
  }                                                   \
})

enum ROI_MODES
{
  ROI_MODE_NONE,
  ROI_MODE_BORDER,
  ROI_MODE_CENTER,
  ROI_MODE_BORDER_THEN_CENTER,
  ROI_MODE_BORDER_THEN_NONE,
  ROI_MODE_MAX
};

int32_t log_level = LOG_INFO;
int enc_idx = -1;
char xcoder_params[512] = { 0 };
int frame_number = 0;
// int rate_limit = 0;
int force_key_frame = 0;

/* Get current time in microseconds */
#ifdef _WIN32
LARGE_INTEGER Frequency;
static uint64_t get_utime(void)
{
    LARGE_INTEGER Count;
    QueryPerformanceCounter(&Count);
    return (Count.QuadPart/(Frequency.QuadPart/1000000));
}
#else
static uint64_t get_utime(void)
{
  struct timeval tv;
  (void) gettimeofday(&tv, NULL);
  return (tv.tv_sec * 1000000LL + tv.tv_usec);
}
#endif

/* Convert string of log_level to log_level integer */
static int32_t log_str_to_level(const char *log_str)
{
  int32_t converted_log_level;
  if (strcmp(log_str, "NONE") == 0)
    converted_log_level = LOG_NONE;
  else if (strcmp(log_str, "FATAL") == 0)
    converted_log_level = LOG_FATAL;
  else if (strcmp(log_str, "ERROR") == 0)
    converted_log_level = LOG_ERROR;
  else if (strcmp(log_str, "INFO") == 0)
    converted_log_level = LOG_INFO;
  else if (strcmp(log_str, "DEBUG") == 0)
    converted_log_level = LOG_DEBUG;
  else if (strcmp(log_str, "TRACE") == 0)
    converted_log_level = LOG_TRACE;
  else
    converted_log_level = -16;
  return converted_log_level;
}

/* Report processed frame count and FPS */
static void print_report(int is_last_report, int64_t start_time, int64_t cur_time)
{
  static int64_t last_time = -1;
  float t;
  char buf[1024];
  float fps;

  if (!is_last_report)
  {
    if (last_time == -1)
    {
      last_time = cur_time;
      return;
    }
    if ((cur_time - last_time) < 500000) // do not update performance more than once every 0.5s
      return;
    last_time = cur_time;
  }

  t = (cur_time - start_time) / 1000000.0;

  fps = t > 0 ? frame_number / t : 0;
  const char end = is_last_report ? '\n' : '\r';

  buf[0] = '\0';
  snprintf(buf + strlen(buf), sizeof(buf) - strlen(buf), "frame=%5d fps=%3.*f ",
          frame_number, (fps < 9.95), fps);
  pr_log(LOG_INFO, "%s   %c", buf, end);
}

/* Send one frame to encoder, attempt to read all available encoder packets from encoder,
   writing packets to output stream. If run with with *frame=NULL, flush the encoder of
   remaining packets until encoder reports EOF */
static int encode(AVCodecContext *enc_ctx, AVFrame *frame, AVPacket *pkt,
                   FILE *outfile)
{
  int rc;
  uint64_t t0, t1, t2;
  int attempt_cnt = 1;
  static int send_cnt = 0;
  static int recv_cnt = 0;

  // Send frame to encoder. If frame=NULL, this signals end of stream
  if (frame)
    pr_log(LOG_DEBUG, "Send %d frame pts %3"PRId64"\n",++send_cnt, frame->pts);
  else
    pr_log(LOG_INFO, "Send done, flush encoder\n");

  do
  {
    t0 = get_utime();
    rc = avcodec_send_frame(enc_ctx, frame);
    // Encoder is too busy to receive frame, try sending frame again
    if (rc == AVERROR(EAGAIN))
    {
      uint64_t tmp = get_utime();
      pr_log(LOG_ERROR, "Resending frame. Try #%d. Elapsed: %luus.\n", ++attempt_cnt, (tmp - t0));
      continue;
    }
    // Frame send issue occured
    else if (rc < 0)
    {
      pr_log(LOG_FATAL, "Error sending frame for encoding. RC=%d\n", AVUNERROR(rc));
      // avcodec_free_context(&enc_ctx);
      return rc;
    }
    // Frame sent successfully
    else
      break;
  }
  while (1);

  t1 = get_utime();
  /* Run this loop if a frame was sucessfully sent during avcodec_send_frame() OR
     no frame was sent because this function is in flushing mode OR
     a frame was sucessfully received by avcodec_receive_packet() */
  while (rc >= 0 || ! frame )
  {
    pr_log(LOG_DEBUG, "Receiving packet\n");
    rc = avcodec_receive_packet(enc_ctx, pkt);
    if (frame == NULL)
      pr_log(LOG_INFO, "Draining packet rc=%x, pkt size=%5d\n", rc, pkt->size);

    // No packets in encoder output buffer but not end of stream either
    if (rc == AVERROR(EAGAIN))
    {
      // If function not in flushing mode
      if (frame)
        return 0; // Exit this function to send next packet
      else
        continue; // Loop in this function to flush encoder
    }
    // End of stream detected
    else if (rc == AVERROR_EOF)
    {
      pr_log(LOG_ERROR, "Encoder EOF reached\n");
      return 0;
    }
    // Packet receive error occured
    else if (rc < 0)
    {
      pr_log(LOG_FATAL, "Error during encoding\n");
      // avcodec_free_context(&enc_ctx);
      return rc;
    }
    // Packet sucessfully received
    recv_cnt++;
    t2 = get_utime();
    pr_log(LOG_INFO, "Received packet #%d, pts=\033[K%3"PRId64", total:%luus, "
           "send:%luus, encode:%luus\n", recv_cnt, pkt->pts, (t2 - t0), (t1 - t0),
           (t2 - t1));

    pr_log(LOG_DEBUG, "Write %d packet pts %3"PRId64" (size=%5d)\n",recv_cnt,
           pkt->pts, pkt->size);
    fwrite(pkt->data, 1, pkt->size, outfile);

    av_packet_unref(pkt);
  }
  return 0;
}

#ifdef ROI_SUPPORTED
/* Called from roi_demo to initalize frame side data buffer.
   Purpose is to reduce repeated code for testing cache features. */
static void init_roi_ref (AVFrame *frame, AVCodecContext *codec_ctx,
                          AVBufferRef **roi_ref, AVRegionOfInterest **roi_data)
{
  // Section describing a rectangle in middle of frame
  (*roi_data)[0].self_size = sizeof(AVRegionOfInterest);
  (*roi_data)[0].top = (codec_ctx->height / 3);
  (*roi_data)[0].bottom = (codec_ctx->height / 3) * 2;
  (*roi_data)[0].left = (codec_ctx->width / 4);
  (*roi_data)[0].right = (codec_ctx->width / 4) * 3;
  // qoffset is an AVrational between -1 and 1
  (*roi_data)[0].qoffset.num = 0;
  (*roi_data)[0].qoffset.den = 1;

  // Section describing everything in frame
  (*roi_data)[1].self_size = sizeof(AVRegionOfInterest);
  (*roi_data)[1].top = 0;
  (*roi_data)[1].bottom = codec_ctx->height;
  (*roi_data)[1].left = 0;
  (*roi_data)[1].right = codec_ctx->width;
  (*roi_data)[1].qoffset.num = 0;
  (*roi_data)[1].qoffset.den = 1;

  // put roi_data into AVFrame
    if (!av_frame_new_side_data_from_buf(frame, AV_FRAME_DATA_REGIONS_OF_INTEREST, *roi_ref))
        pr_log(LOG_ERROR, "ROI failed to apply\n");
}

/* Call before every frame to run selected ROI demo mode.
   ROI side data must be in every AVFrame for which ROI is desired. */
static void roi_demo(AVCodecContext *codec_ctx,  AVFrame *frame, int frame_num,
                     int frame_cnt, int roi_mode, int roi_cache_mode)
{
  static AVBufferRef *roi_ref = NULL;
  static AVRegionOfInterest *roi_data = NULL;
  static int first_time = 1;
  static int applied_roi_cnt = 0;

  // Initialize roi_data
  if (first_time && (!applied_roi_cnt))
  {
    first_time = 0;

    // initalize the roi_ref buffer
    roi_ref = av_buffer_alloc(sizeof(AVRegionOfInterest) * 2);
    roi_data = (AVRegionOfInterest*)roi_ref->data;
    init_roi_ref(frame, codec_ctx, &roi_ref, &roi_data);
  }
  // Switch ROI map depending on roi_mode and frame_num
  switch (roi_mode)
  {
    case ROI_MODE_BORDER:  // Decrease QP in border (better quality), increase QP in middle (worse quality)
      if (applied_roi_cnt < 1) {
        pr_log(LOG_INFO, "ROI demo mode %d increasing quality of border, decreasing quality of center\n", roi_mode);
        roi_data[0].qoffset.num = 1;
        roi_data[1].qoffset.num = -1;
        applied_roi_cnt++;
      }
      else if (applied_roi_cnt == 1 && roi_cache_mode){
        pr_log(LOG_INFO, "ROI cache mode, removing side data.\n");
        av_frame_remove_side_data(frame, AV_FRAME_DATA_REGIONS_OF_INTEREST);
        applied_roi_cnt++;
      }
      break;
    case ROI_MODE_CENTER:  // Decrease QP in middle (better quality), increase QP in border (worse quality)
      if (applied_roi_cnt < 1) {
        pr_log(LOG_INFO, "ROI demo mode %d increasing quality of center, decreasing quality of border\n", roi_mode);
        roi_data[0].qoffset.num = -1;
        roi_data[1].qoffset.num = 1;
        applied_roi_cnt++;
      }
      else if (applied_roi_cnt == 1 && roi_cache_mode) {
        pr_log(LOG_INFO, "ROI cache mode, removing side data.\n");
        av_frame_remove_side_data(frame, AV_FRAME_DATA_REGIONS_OF_INTEREST);
        applied_roi_cnt++;
      }
      break;
    case ROI_MODE_BORDER_THEN_CENTER:  // Increase QP in middle, decrease QP in border, switch ROI halfway through clip
      if ( frame_num < 0.5*frame_cnt ) {
        if (applied_roi_cnt < 1) {
          pr_log(LOG_INFO, "ROI demo mode %d increasing quality of border, decreasing quality of center\n", roi_mode);
          roi_data[0].qoffset.num = 1;
          roi_data[1].qoffset.num = -1;
          applied_roi_cnt++;
        }
        else if (applied_roi_cnt == 1 && roi_cache_mode) {
          pr_log(LOG_INFO, "ROI cache mode, removing side data.\n");
          av_frame_remove_side_data(frame, AV_FRAME_DATA_REGIONS_OF_INTEREST);
          applied_roi_cnt++;
        }
      } else {
          if ((applied_roi_cnt < 2 && ! roi_cache_mode) || (applied_roi_cnt == 2 && roi_cache_mode)) {
            pr_log(LOG_INFO, "ROI demo mode %d increasing quality of center, decreasing quality of border\n", roi_mode);

            if (roi_cache_mode) {
              roi_ref = av_buffer_alloc(sizeof(AVRegionOfInterest) * 2);
              roi_data = (AVRegionOfInterest*)roi_ref->data;
              init_roi_ref(frame, codec_ctx, &roi_ref, &roi_data);
            }

            roi_data[0].qoffset.num = -1;
            roi_data[1].qoffset.num = 1;
            applied_roi_cnt++;
          }
          else if (applied_roi_cnt == 3 && roi_cache_mode) {
            pr_log(LOG_INFO, "ROI cache mode, removing side data.\n");
            av_frame_remove_side_data(frame, AV_FRAME_DATA_REGIONS_OF_INTEREST);
            applied_roi_cnt++;
          }
      }
      break;
    case ROI_MODE_BORDER_THEN_NONE:  // Increase QP in middle, decrease QP in border, stop ROI halfway through clip
      if ( frame_num < 0.5*frame_cnt ) {
        if (applied_roi_cnt < 1) {
          pr_log(LOG_INFO, "ROI demo mode %d increasing quality of border\n", roi_mode);
          roi_data[0].qoffset.num = 1;
          roi_data[1].qoffset.num = -1;
          applied_roi_cnt++;
        }
        else if (applied_roi_cnt == 1 && roi_cache_mode) {
          pr_log(LOG_INFO, "ROI cache mode, removing side data.\n");
          av_frame_remove_side_data(frame, AV_FRAME_DATA_REGIONS_OF_INTEREST);
          applied_roi_cnt++;
        }
      } else {
        if ((applied_roi_cnt < 2 && ! roi_cache_mode) || (applied_roi_cnt == 2 && roi_cache_mode)) {
          pr_log(LOG_INFO, "ROI demo mode %d stopping ROI insertion\n", roi_mode);

          if (roi_cache_mode) {
            roi_ref = av_buffer_alloc(sizeof(AVRegionOfInterest) * 2);
            roi_data = (AVRegionOfInterest*)roi_ref->data;
            init_roi_ref(frame, codec_ctx, &roi_ref, &roi_data);
            roi_data[0].qoffset.num = 0;
            roi_data[1].qoffset.num = 0;
          }
          else {
            if (av_frame_get_side_data(frame, AV_FRAME_DATA_REGIONS_OF_INTEREST))
              av_frame_remove_side_data(frame, AV_FRAME_DATA_REGIONS_OF_INTEREST);
          }

          applied_roi_cnt++;
        }
        else if (applied_roi_cnt == 3 && roi_cache_mode){
          pr_log(LOG_INFO, "ROI cache mode, removing side data.\n");
          av_frame_remove_side_data(frame, AV_FRAME_DATA_REGIONS_OF_INTEREST);
          applied_roi_cnt++;
        }
      }
      break;
  }
}
#endif

static int alloc_frame(AVFrame **frame, enum AVPixelFormat pix_fmt, int width, int height)
{
    *frame = av_frame_alloc();
    if (*frame == NULL)
    {
      pr_log(LOG_FATAL, "Could not allocate AVFrame\n");
      return -1;
    }
    (*frame)->format = pix_fmt;
    (*frame)->width  = width;
    (*frame)->height = height;

    // Create frame buffer with 32B alignment
    if (av_frame_get_buffer(*frame, 32) < 0)
    {
      pr_log(LOG_FATAL, "Could not allocate the AVFrame buffers\n");
      return -1;
    }
    return 0;
}

static int read_yuv_frame(FILE *input_fp, AVFrame* frame, AVCodecContext *codec_ctx, int video_width, enum AVPixelFormat pix_fmt)
{
  int read_size, j;
  // Read data for Y into frame buffer
  for (j = 0; j < codec_ctx->height; j++)
  {
    read_size = fread(frame->data[0] + j * frame->linesize[0], av_image_get_linesize(pix_fmt, video_width, 0), 1, input_fp);
    if (read_size != 1)
    {
      pr_log(LOG_FATAL, "Failed to read Y. read_size=%u.\n", read_size);
      return -1;
    }
  }
  // Read data for U into frame buffer
  for (j = 0; j < codec_ctx->height / 2; j++)
  {
    read_size = fread(frame->data[1] + j * frame->linesize[1], av_image_get_linesize(pix_fmt, video_width, 1), 1, input_fp);
    if (read_size != 1)
    {
      pr_log(LOG_FATAL, "Failed to read U. read_size=%u.\n", read_size);
      return -1;
    }
  }
  // Read data for V into frame buffer
  for (j = 0; j < codec_ctx->height / 2; j++)
  {
    read_size = fread(frame->data[2] + j * frame->linesize[2], av_image_get_linesize(pix_fmt, video_width, 2), 1, input_fp);
    if (read_size != 1)
    {
      pr_log(LOG_FATAL, "Failed to read V. read_size=%u.\n", read_size);
      return -1;
    }
  }
  return 0;
}

static int read_scale_rgba_frame(FILE *input_fp, AVFrame* frame, AVFrame* rgba_frame, AVCodecContext *codec_ctx,
                                 int video_width, enum AVPixelFormat pix_fmt, struct SwsContext *sws_ctx)
{
  int read_size, j;
  for (j = 0; j < codec_ctx->height; j++)
  {
    read_size = fread(rgba_frame->data[0] + j * rgba_frame->linesize[0], av_image_get_linesize(pix_fmt, video_width, 0), 1, input_fp);
    if (read_size != 1)
    {
      pr_log(LOG_FATAL, "Failed to read RGBA read_size=%u.\n", read_size);
      return -1;
    }
  }
  if (sws_ctx && frame) {
    sws_scale(sws_ctx, (const uint8_t * const *)rgba_frame->data, rgba_frame->linesize,
              0, codec_ctx->height, (uint8_t * const *)frame->data, frame->linesize);
  }
  return 0;
}

static int init_filters(AVFilterContext **buffersink_ctx, AVFilterContext **buffersrc_ctx,
                        AVFilterGraph **filter_graph, int src_width, int src_height,
                        enum AVPixelFormat pix_fmt, int fps)
{
  char args[128] = { 0 };
  char name[32] = { 0 };
  char filters_desc[128] = { 0 };
  int i, ret = 0;
  AVFilterInOut *inputs, *outputs;

  snprintf(filters_desc, sizeof(filters_desc), "ni_quadra_hwupload=0");

  *filter_graph = avfilter_graph_alloc();
  if (filter_graph == NULL) {
    av_log(NULL, AV_LOG_ERROR, "failed to allocate filter graph\n");
    goto end;
  }

  ret = avfilter_graph_parse2(*filter_graph, filters_desc, &inputs, &outputs);
  if (ret < 0) {
    av_log(NULL, AV_LOG_ERROR, "failed to parse graph\n");
    goto end;
  }

  // link input
  snprintf(name, sizeof(name), "in_0");
  snprintf(args, sizeof(args),
          "video_size=%dx%d:pix_fmt=%d:time_base=1/%d:pixel_aspect=0/1",
          src_width, src_height, pix_fmt, fps);
  ret = avfilter_graph_create_filter(buffersrc_ctx, avfilter_get_by_name("buffer"), name,
                                     args, NULL, *filter_graph);
  if (ret < 0) {
    av_log(NULL, AV_LOG_ERROR, "Cannot create buffer source\n");
    goto end;
  }

  ret = avfilter_link(*buffersrc_ctx, 0, inputs->filter_ctx, inputs->pad_idx);
  if (ret < 0) {
    av_log(NULL, AV_LOG_ERROR, "failed to link input filter\n");
    goto end;
  }

  // link output
  snprintf(name, sizeof(name), "out_0");
  ret = avfilter_graph_create_filter(buffersink_ctx, avfilter_get_by_name("buffersink"),
                                      name, NULL, NULL, *filter_graph);
  if (ret < 0) {
    av_log(NULL, AV_LOG_ERROR, "failed to create output filter: %d\n", i);
    goto end;
  }

  ret = avfilter_link(outputs->filter_ctx, outputs->pad_idx, *buffersink_ctx, 0);
  if (ret < 0) {
    av_log(NULL, AV_LOG_ERROR, "failed to link output filter: %d\n", i);
    goto end;
  }

  // configure and validate the filter graph
  ret = avfilter_graph_config(*filter_graph, NULL);
  if (ret < 0) {
    av_log(NULL, AV_LOG_ERROR, "%s failed to config graph filter\n",
            __func__);
    goto end;
  } else {
    av_log(NULL, AV_LOG_DEBUG, "%s success config graph filter %s\n",
            __func__, filters_desc);
  }

end:
  avfilter_inout_free(&inputs);
  avfilter_inout_free(&outputs);
  return ret;
}

static void help_usage(void)
{
  printf("Usage: \n"
         "-s | --size                  Size of input yuv in [width]x[height].\n"
         "-x | --xcoder_params         Xcoder parameters. See Integration and Programing Guide for details.\n"
         "-i | --input                 Input yuv file.\n"
         "-o | --output                Output video file.\n"
         "-c | --codec                 Codec name. [h264_ni_quadra_enc, h265_ni_quadra_enc, av1_ni_quadra_enc].\n"
         "-p | --pix_fmt               Pixel format for encoding.\n"
         "-r | --fps                   FPS of output.\n"
         "-b | --bitrate               Bitrate in bits/second.\n"
         "-e | --enc                   Encoder device index to use. See ni_rsrc_mon.\n"
         "-l | --loop                  Loop read input count.\n"
         "-g | --log_level             Log level for messages. [NONE, FATAL, ERROR, INFO, DEBUG, TRACE]\n"
         "-u | --hwupload              Use hwupload first and then encode HW frames.\n"
         "                             Performance for RGBA input is improved in this path.\n"
         "-h | --help                  Help info.\n"
         "\nTesting intermittent idle of input:\n"
         "-w | --inst_restart_count    Number of times to restart encoder instance and re-read input.\n"
         "-a | --interval_frame        Number of frames between intervals of test activity.\n"
         "                             (if 0, use random interval).\n"
         "-v | --interval_time         Number of seconds to sleep at each interval_frame.\n"
         "                             (if 0, random between 1s - 20s).\n"
         "-k | --force_key_frame       Flag to force key frame at each interval (must set interval_frame).\n"
         "\nTesting Region of Interest:\n"
         "-z | --roi                   Apply demo ROI mode. Options: 1 -> Border ROI, 2 -> Center ROI, 3 -> Switch ROI halfway , 4 -> Stop ROI halfway\n"
         "-y | --roi_cache             Use ROI Cache mode. Options: 1 -> enable caching, 0 -> disable caching\n"
         "\nTesting Bitrate reconfiguration:\n"
         "-m | --bitrate_reconfig      Use side data to reconfigure bitrate target throughout video.\n"
         "                             The --bitrate parameter must be set. Use the format \n"
         "                             bitrate_min:bitrate_max:bitrate_step:frame_interval to cause target\n"
         "                             bitrate to sweep in a saw-tooth pattern between bitrate_min and\n"
         "                             bitrate_max in increments of bitrate_step every frame_interval frames.\n");
}

int main(int argc, char **argv)
{
  enum AVPixelFormat pix_fmt = AV_PIX_FMT_YUV420P;
  const char *inputfile_str, *outputfile_str, *codec_str;
  FILE *input_fp, *output_fp;
  const AVCodec *codec;
  AVCodecContext *codec_ctx = NULL;
  int i, j, loop_idx, inst_restart_idx, rc;
  AVFrame *frame = NULL;
  AVFrame *rgba_frame = NULL;
  AVFrame *upload_frame = NULL;
  AVPacket *pkt = NULL;
  unsigned int frame_cnt;
  unsigned int read_size;
  unsigned long inputfile_size;
  int64_t start_time;
  int video_width = -1;
  int video_height = -1;
  int bitrate = 0;
  int fps = 30;
  char *n;      // used for parsing width and height from --size
  int loop_count = 1;
  int64_t ts_step;
  int roi_mode = 0;
  int roi_cache_mode = 0;
  int inst_restart_count = 0;
  int interval_mode = NI_INTERVAL_INVALID;
  int interval_time = 0;
  int interval_frame_mode = NI_INTERVAL_INVALID;
  int interval_frame_count = 0; // must set default value 0.
  int interval_frame_num = 0;
  int bitrate_reconf_direction_up = 1;
  int bitrate_reconf_interval = 0;
  int bitrate_reconf_target = 0;
  int bitrate_reconf_min = 0;
  int bitrate_reconf_max = 0;
  int bitrate_reconf_step = 0;
  AVFrameSideData *sd;
  int general_side_data_reconf = 0;
  AVNetintGeneralSideData *ni_general;
  void *cur_data;

  int is_rgba = 0;
  struct SwsContext *sws_ctx = NULL;
  int hwupload = 0;
  AVFilterContext *buffersink_ctx;
  AVFilterContext *buffersrc_ctx;
  AVFilterGraph *filter_graph;

#ifdef _WIN32
  // Get frequency firstly
  QueryPerformanceFrequency(&Frequency);
#endif

  // Input arg handling
  int opt;
  int opt_index;
  const char *opt_string = "s:x:i:o:c:r:b:e:l:p:w:v:a:g:z:m:M:ykuh";
  static struct option long_options[] =
  {
    {"size",               required_argument, NULL, 's'},
    {"xcoder_params",      required_argument, NULL, 'x'},
    {"inputfile",          required_argument, NULL, 'i'},
    {"outputfile",         required_argument, NULL, 'o'},
    {"codec",              required_argument, NULL, 'c'},
    {"fps",                required_argument, NULL, 'r'},
    {"bitrate",            required_argument, NULL, 'b'},
    {"enc",                required_argument, NULL, 'e'},
    {"loop",               required_argument, NULL, 'l'},
    {"pix_fmt",            required_argument, NULL, 'p'},
    {"inst_restart_count", required_argument, NULL, 'w'},
    {"interval_time",      required_argument, NULL, 'v'},
    {"interval_frame",     required_argument, NULL, 'a'},
    {"log_level",          required_argument, NULL, 'g'},
    {"roi",                required_argument, NULL, 'z'},
    {"bitrate_reconfig",   required_argument, NULL, 'm'},
    {"general_reconfig",   required_argument, NULL, 'M'},
    {"roi_cache",          no_argument,       NULL, 'y'},
    // {"re",                 no_argument,       NULL, 't'},
    {"force_key_frame",    no_argument,       NULL, 'k'},
    {"hwupload",           no_argument,       NULL, 'u'},
    {"help",               no_argument,       NULL, 'h'},
    {NULL,                 0,                 NULL,   0},
  };

  while ((opt = getopt_long(argc, argv, opt_string, long_options, &opt_index)) != -1)
  {
    switch (opt)
    {
      case 's':
        video_width = (int) strtol(optarg, &n, 10);
        if (*n != 'x')
        {
          pr_log(LOG_FATAL, "input resolution/size arg not of format [width]x[height]: %s\n", optarg);
          exit(1);
        }
        video_height = atoi(n + 1);
        break;
      case 'x':
        strcpy(xcoder_params, optarg);
        break;
      case 'i':
        inputfile_str = optarg;
        break;
      case 'o':
        outputfile_str = optarg;
        break;
      case 'c':
        codec_str = optarg;
        break;
      case 'e':
        enc_idx = atoi(optarg);
        break;
      case 'b':
        bitrate = atoi(optarg);
        break;
      case 'r':
        fps = atoi(optarg);
        break;
      case 'l':
        loop_count = atoi(optarg);
        break;
      case 'p':
        pix_fmt = av_get_pix_fmt(optarg);
        break;
      case 'w':
        inst_restart_count = atoi(optarg);
        break;
      case 'v':
        interval_time = atoi(optarg);
        if (!interval_time) // if interval_time equals to 0, generate interval time randomly.
        {
          interval_mode = NI_RANDOM_INTERVAL_MODE;
        }
        else if ((interval_time > 20) || (interval_time < 0))
        {
          pr_log(LOG_FATAL, "interval_time not between 0 and 20. Invalid interval_time value: %d\n", interval_time);
          exit(1);
        }
        else
        {
          interval_mode = NI_FIXED_INTERVAL_MODE;
        }
        break;
      case 'a':
        interval_frame_count = atoi(optarg);
        if (!interval_frame_count) // if interval_frame_num equals to 0, generate interval_frame_num randomly.
        {
          interval_frame_mode = NI_RANDOM_INTERVAL_MODE;
        }
        else
        {
          interval_frame_mode = NI_FIXED_INTERVAL_MODE;
        }
        break;
      case 'g':
        if (log_str_to_level(optarg) == -16)
        {
          pr_log(LOG_FATAL, "invalid log_level selected: %s\n", optarg);
          exit(1);
        }
        log_level = log_str_to_level(optarg);
        break;
      case 'z':
#ifdef ROI_SUPPORTED
        roi_mode = atoi(optarg);
#else
        (void)roi_mode;
        pr_log(LOG_FATAL, "ROI not supported for this version of FFmpeg!\n");
        exit(1);
#endif
        break;
      case 'y':
#ifdef ROI_SUPPORTED
        roi_cache_mode = 1;
#else
        (void)roi_cache_mode;
        pr_log(LOG_FATAL, "ROI not supported for this version of FFmpeg!\n");
        exit(1);
#endif
        break;
      case 'm':
        bitrate_reconf_min = (int) strtol(optarg, &n, 10);
        if (*n != ':')
        {
          pr_log(LOG_FATAL, "bitrate_reconfig not of format min:max:step:frame_interval: %s\n", optarg);
          exit(1);
        }
        bitrate_reconf_max = (int) strtol(n + 1, &n, 10);
        if (*n != ':')
        {
          pr_log(LOG_FATAL, "bitrate_reconfig not of format min:max:step:frame_interval: %s\n", optarg);
          exit(1);
        }
        bitrate_reconf_step = (int) strtol(n + 1, &n, 10);
        if (*n != ':')
        {
          pr_log(LOG_FATAL, "bitrate_reconfig not of format min:max:step:frame_interval: %s\n", optarg);
          exit(1);
        }
        bitrate_reconf_interval = (int) strtol(n + 1, &n, 10);
        if (*n != '\0')
        {
          pr_log(LOG_FATAL, "bitrate_reconfig not of format min:max:step:frame_interval: %s\n", optarg);
          exit(1);
        }
        break;
      case 'M':
        general_side_data_reconf = atoi(optarg);
        break;
      // case 't':
        // rate_limit = 1;
        // break;
      case 'k':
        force_key_frame = 1;
        break;
      case 'u':
        hwupload = 1;
        break;
      case 'h':
        help_usage();
        exit(0);
      default:
        help_usage();
        exit(1);
    }
  }

  // Input arg checking
  if ((inputfile_str == NULL) || (outputfile_str == NULL) || (codec_str == NULL) || \
      (video_width == -1) || (video_height == -1))
  {
    pr_log(LOG_FATAL, "Required args (inputfile, outputfile, codec, size) not set.\n");
    exit(1);
  }
  if (bitrate_reconf_interval && \
      ((bitrate_reconf_interval < 1) || (!bitrate) || (bitrate_reconf_min >= bitrate_reconf_max) || \
      (bitrate_reconf_step > (bitrate_reconf_max - bitrate_reconf_min)) || \
      (bitrate_reconf_min > bitrate) || (bitrate_reconf_max < bitrate)))
  {
    pr_log(LOG_FATAL, "bitrate_reconfig args not valid. bitrate >= 0; frame_interval>=1,\n"
                      "bitrate_reconf_min < bitrate_reconf_max, bitrate_step <= (bitrate_max - bitrate_min),\n"
                      "bitrate_reconf_min <= bitrate <= bitrate_reconf_max\n");
    exit(1);
  }
  if ((roi_mode < ROI_MODE_NONE) || (roi_mode >= ROI_MODE_MAX))
  {
    pr_log(LOG_FATAL, "Invalid roi mode selected: %d\n", roi_mode);
    exit(1);
  }

  pr_log(LOG_INFO, "inputfile:%s, outputfile:%s, codec:%s, width:%u, height:%u, enc_idx=%d, "
         "loop_count=%d, inst_restart_count=%d.\n", inputfile_str, outputfile_str, codec_str,
         video_width, video_height, enc_idx, loop_count, inst_restart_count);
  pr_log(LOG_INFO, "xcoder_params: %s.\n", xcoder_params);

  av_log_set_level(log_level);
  if (pix_fmt == AV_PIX_FMT_ARGB || pix_fmt == AV_PIX_FMT_ABGR || pix_fmt == AV_PIX_FMT_RGBA || pix_fmt == AV_PIX_FMT_BGRA)
  {
    is_rgba = 1;
  }
  for(inst_restart_idx = 0; inst_restart_idx <= inst_restart_count; inst_restart_idx++)
  {
    // Open input file
    input_fp = fopen(inputfile_str, "rb");
    if (input_fp == NULL)
    {
      pr_log(LOG_FATAL, "Could not to open inputfile: %s\n", inputfile_str);
      exit (1);
    }

    // Check resolution of input file
    fseek(input_fp, 0, SEEK_END);
    inputfile_size = ftell(input_fp);
    const int frame_size = av_image_get_buffer_size(pix_fmt, video_width, video_height, 1);
    if (frame_size <= 0)
    {
      pr_log(LOG_FATAL, "Unable to calculate frame size for the given pixel format: %d (%s)", pix_fmt, av_get_pix_fmt_name(pix_fmt));
      goto end;
    }
    if (inputfile_size % frame_size != 0)
    {
      pr_log(LOG_FATAL, "Size of inputfile is not integer multiple of resolution. "
             "Either input file has partial frames, or input resolution is wrong.");
      goto end;
    }
    frame_cnt = inputfile_size / frame_size;
    pr_log(LOG_INFO, "inputfile size=%lu, number of frames = %u.\n", inputfile_size, frame_cnt);
    fseek(input_fp, 0, SEEK_SET);

    // Open outputfile
    output_fp = fopen(outputfile_str, "wb");
    if (!output_fp)
    {
      pr_log(LOG_FATAL, "Could not open outputfile: %s\n", outputfile_str);
      goto end;
    }

    // Set interval time and frame number
    if (NI_RANDOM_INTERVAL_MODE == interval_mode)
    {
      interval_time = rand() % 20 + 1; //Generate 1 to 20 random Numbers
    }
    if (NI_RANDOM_INTERVAL_MODE == interval_frame_mode)
    {
      interval_frame_count = rand() % frame_cnt + 1;
      interval_frame_num = interval_frame_count;
    }
    else
    {
      interval_frame_num = interval_frame_count;
    }

    // Check interval parameters, must have interval frame for interval sleep
    if ((NI_INTERVAL_INVALID != interval_mode) && (NI_INTERVAL_INVALID == interval_frame_mode))
    {
      pr_log(LOG_FATAL, "interval sleep but no interval frame num\n");
      goto end;
    }

    // Check force_key_frame parameters, must have interval frame in force key frame mode
    if ((force_key_frame) && (NI_INTERVAL_INVALID == interval_frame_mode))
    {
      pr_log(LOG_FATAL, "fore key frame but no interval frame num\n");
      goto end;
    }

#if LIBAVCODEC_VERSION_MAJOR < 58
    // Register all codecs
    avcodec_register_all(); // This is deprecated after FFmpeg-n4.0.0
#endif

    // Find video encoder codec selected
    codec = avcodec_find_encoder_by_name(codec_str);
    if (!codec)
    {
      pr_log(LOG_FATAL, "Codec '%s' not found\n", codec_str);
      goto end;
    }

    // Allocate codec context for encoding
    codec_ctx = avcodec_alloc_context3(codec);
    if (!codec_ctx)
    {
      pr_log(LOG_FATAL, "Could not allocate video codec context\n");
      goto end;
    }

    // Allocate packet for encoder output
    pkt = av_packet_alloc();
    if (!pkt)
    {
      pr_log(LOG_FATAL, "Could not allocate AVPacket\n");
      goto end;
    }

    // Apply encoding parameters to codec context
    if (bitrate)
    {
      codec_ctx->bit_rate = bitrate;
      bitrate_reconf_target = bitrate;
    }
    codec_ctx->width = video_width; // resolution must be divisible by 2
    codec_ctx->height = video_height;
    codec_ctx->time_base = (AVRational){1, fps};
    codec_ctx->framerate = (AVRational){fps, 1};
    if (hwupload)
    {
      codec_ctx->pix_fmt = AV_PIX_FMT_NI_QUAD;
      codec_ctx->sw_pix_fmt = pix_fmt;
    }
    else
    {
      codec_ctx->pix_fmt = is_rgba ? AV_PIX_FMT_YUV420P : pix_fmt;
    }

    // Apply private codec context data to be interpreted by libxcoder
    if ((codec->id == AV_CODEC_ID_H264) || (codec->id == AV_CODEC_ID_H265)
#ifdef AV1_SUPPORTED // AV1 not supported in 3.1.1
        || (codec->id == AV_CODEC_ID_AV1)
#endif
    )
    {
      av_opt_set(codec_ctx->priv_data, "xcoder-params", xcoder_params, 0);
      if (enc_idx >= 0)
        av_opt_set_int(codec_ctx->priv_data, "enc", enc_idx, 0);
    }
    else
    {
      pr_log(LOG_FATAL, "codec id %d not supported.\n", codec->id);
      goto end;
    }

    // Open encoder
    rc = avcodec_open2(codec_ctx, codec, NULL);
    if (rc < 0)
    {
      pr_log(LOG_FATAL, "Could not open codec: %s\n", av_err2str(rc));
      goto end;
    }

    // Allocate frame object
    if (hwupload)
    {
      if (alloc_frame(&upload_frame, pix_fmt, codec_ctx->width, codec_ctx->height) < 0)
      {
        goto end;
      }
      frame = av_frame_alloc();
      if (!frame)
      {
        av_log(NULL,AV_LOG_ERROR, "Could not allocate HW AVFrame\n");
        goto end;
      }
    }
    else
    {
      if (alloc_frame(&frame, codec_ctx->pix_fmt, codec_ctx->width, codec_ctx->height) < 0)
      {
        goto end;
      }
      if (is_rgba)
      {
        if (alloc_frame(&rgba_frame, AV_PIX_FMT_RGBA, codec_ctx->width, codec_ctx->height) < 0)
        {
          goto end;
        }
        sws_ctx = sws_getContext(codec_ctx->width, codec_ctx->height, pix_fmt,
                                codec_ctx->width, codec_ctx->height, AV_PIX_FMT_YUV420P,
                                SWS_BILINEAR, NULL, NULL, NULL);
        if (!sws_ctx) {
          pr_log(LOG_ERROR, "Failed to create sws context for conversion\n");
          goto end;
        }
      }
    }

    pr_log(LOG_INFO, "Input line sizes: Y=%d, U=%d, V=%d.\n", frame->linesize[0],
           frame->linesize[1], frame->linesize[2]);

    if (hwupload)
    {
      if (init_filters(&buffersink_ctx, &buffersrc_ctx, &filter_graph, video_width, video_height, pix_fmt, fps) < 0)
      {
        av_log(NULL, AV_LOG_ERROR, "Failed to initialize filters\n");
        goto end;
      }
    }

    ts_step = 90000 / fps;
    start_time = av_gettime_relative();
    // Repeatedly loop input if selected by user
    for (loop_idx = 0; loop_idx < loop_count; loop_idx++)
    {
      fseek(input_fp, 0, SEEK_SET);
      // Loop over all frames in inputfile
      for (i = 0; i < frame_cnt; i++)
      {
        /*
        if (rate_limit)
        {
          while (1)
          {
            if ((dts + 20000) > (av_gettime_relative() - start_time))
            {
              av_usleep(10000);
              continue;
            } else
              break;
          }
        }
        */
#ifdef ROI_SUPPORTED
        /* Code for testing ROI */
        if (roi_mode)
            roi_demo(codec_ctx, frame, i, frame_cnt, roi_mode, roi_cache_mode); // ROI sidedata has to be attached for each AVFrame
#endif
        // Make sure the frame data is writable
        rc = av_frame_make_writable(hwupload ? upload_frame : frame);
        if (rc < 0)
        {
          pr_log(LOG_FATAL, "av_frame_make_writable() error.\n");
          goto end;
        }

        if (!is_rgba)
        {
          if(read_yuv_frame(input_fp, hwupload ? upload_frame : frame, codec_ctx, video_width, pix_fmt))
          {
            pr_log(LOG_FATAL, "Failed to read one frame.\n");
            goto end;
          }
        }
        else
        {
          if (hwupload)
          {
            rc = read_scale_rgba_frame(input_fp, NULL, upload_frame, codec_ctx, video_width, pix_fmt, NULL);
          }
          else
          {
            rc = read_scale_rgba_frame(input_fp, frame, rgba_frame, codec_ctx, video_width, pix_fmt, sws_ctx);
          }
          if (rc)
          {
            pr_log(LOG_FATAL, "Failed to read one frame.\n");
            goto end;
          }
        }

        if (hwupload)
        {
          if (av_buffersrc_add_frame_flags(buffersrc_ctx, upload_frame, AV_BUFFERSRC_FLAG_KEEP_REF) < 0)
          {
            av_log(NULL, AV_LOG_ERROR, "Error while feeding frame to the filtergraph\n");
            goto end;
          }
          if (av_buffersink_get_frame(buffersink_ctx, frame) < 0)
          {
            av_log(NULL, AV_LOG_ERROR, "Error while retriving frame from filtergraph\n");
            goto end;
          }
        }

        frame->pts = (loop_idx * frame_cnt + i) * ts_step;
        frame_number = loop_idx * frame_cnt + i + 1; // begin frame count at 1

        /* Code for testing bitrate reconfiguration */
        if (bitrate_reconf_interval && i && (i % bitrate_reconf_interval == 0))
        {
          sd = av_frame_new_side_data(frame, AV_FRAME_DATA_NETINT_BITRATE, sizeof(int32_t));
          if (sd == NULL)
          {
            av_log(NULL, AV_LOG_ERROR, "Fail to allocate side data\n");
          }
          else
          {
            if (bitrate_reconf_direction_up)
              bitrate_reconf_target += bitrate_reconf_step;
            else
              bitrate_reconf_target -= bitrate_reconf_step;

            if (bitrate_reconf_target >= bitrate_reconf_max)
            {
              bitrate_reconf_direction_up = 0;
              bitrate_reconf_target = bitrate_reconf_max;
            }
            else if (bitrate_reconf_target <= bitrate_reconf_min)
            {
              bitrate_reconf_direction_up = 1;
              bitrate_reconf_target = bitrate_reconf_min;
            }

            *((int32_t*)sd->data) = bitrate_reconf_target;
            pr_log(LOG_INFO, "bitrate reconfiguration set new bitrate to %d\n", bitrate_reconf_target);
          }
        }

        if (general_side_data_reconf && i == 100) {
            sd = av_frame_new_side_data(frame, AV_FRAME_DATA_NETINT_GENERAL_SIDE_DATA,
                                        sizeof(AVNetintGeneralSideData));
            if (!sd) {
                av_log(NULL, AV_LOG_ERROR, "failed to allocate general sidedata\n");
            }
            else {
              ni_general = (AVNetintGeneralSideData *)sd->data;
              ni_general->count = 0;
              ni_general->type[ni_general->count] = NI_FRAME_AUX_DATA_VBV_BUFFER_SIZE;
              ni_general->size[ni_general->count] = sizeof(int32_t);
              cur_data = (uint32_t *)&(ni_general->data[ni_general->count]);
              ((uint32_t *)cur_data)[0] = 2500;
              ni_general->count++;
              ni_general->type[ni_general->count] = NI_FRAME_AUX_DATA_CRF_FLOAT;
              ni_general->size[ni_general->count] = sizeof(float);
              cur_data = (float *)&(ni_general->data[ni_general->count]);
              ((float *)cur_data)[0] = 23;
              ni_general->count++;
            }
        }

        /* Code for testing intermittent idle of input */
        // If the interval_frame_num equals to frame_number, it will sleep "interval_time"
        // seconds before continuing encode.
        // Also, check if need to force I frame
        if (frame_number == interval_frame_num)
        {
          pr_log(LOG_INFO, "frame_number:%d, interval_time:%d, interval_frame_num:%d\n", frame_number, interval_time, interval_frame_num);
          if (NI_INTERVAL_INVALID != interval_mode)
          {
            av_usleep(interval_time * 1000000);
            if (NI_RANDOM_INTERVAL_MODE == interval_mode)
            {
              interval_time = rand() % 20 + 1; // Sleep a random length between 1s-20s
            }
          }
          if (NI_RANDOM_INTERVAL_MODE == interval_frame_mode)
          {
            // For length of next interval, use a random count
            interval_frame_count = rand() % frame_cnt + 1;
          }
          // set frame number of next interval
          interval_frame_num += interval_frame_count;

          if (force_key_frame)
          {
            frame->pict_type = AV_PICTURE_TYPE_I;
          }
        }

        // Encode frame
        if (encode(codec_ctx, frame, pkt, output_fp) < 0)
        {
          goto end;
        }

        /* Code for testing bitrate reconfiguration */
        sd = av_frame_get_side_data(frame, AV_FRAME_DATA_NETINT_BITRATE);
        if (sd)
          av_frame_remove_side_data(frame, AV_FRAME_DATA_NETINT_BITRATE);
        sd = av_frame_get_side_data(frame, AV_FRAME_DATA_NETINT_GENERAL_SIDE_DATA);
        if (sd)
          av_frame_remove_side_data(frame, AV_FRAME_DATA_NETINT_GENERAL_SIDE_DATA);
        print_report(0, start_time, av_gettime_relative());
        // Undo key frame forcing (this is idempotent)
        frame->pict_type = AV_PICTURE_TYPE_NONE;

        if (hwupload)
        {
          av_frame_unref(frame);
        }
      }
    }

    // Flush encoder
    if (encode(codec_ctx, NULL, pkt, output_fp) < 0)
    {
      goto end;
    }
    print_report(1, start_time, av_gettime_relative());
end:
    fclose(output_fp);
    fclose(input_fp);
    avcodec_free_context(&codec_ctx);
    if (hwupload)
    {
      av_frame_free(&upload_frame);
      avfilter_graph_free(&filter_graph);
    }
    else if (is_rgba)
    {
      av_frame_free(&rgba_frame);
      sws_freeContext(sws_ctx);
    }
    av_frame_free(&frame);
    av_packet_free(&pkt);
  }


  pr_log(LOG_INFO, "Encode done\n");
  return 0;
}
